/* glpmip/mip1_driver.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <float.h>
#include <math.h>
#include <stddef.h>
#include <time.h>
#include "glpmip.h"
#include "glprsm.h"

/*----------------------------------------------------------------------
-- mip1_driver - driver for the branch-and-bound procedure.
--
-- *Synopsis*
--
-- #include "glpmip.h"
-- int mip1_driver(LP *lp, LPSOL *sol, struct mip1_cp *cp);
--
-- *Description*
--
-- The mip1_driver routine is a driver to the routines which implement
-- components of the branch-and-bound procedure (based on the revised
-- simplex method) for mixed integer linear programming.
--
-- The parameter lp points to the LP problem data block. This block
-- specifies the MIP problem which should be solved. It is not changed
-- on exit. Note that bounds of integer variables MUST BE integer.
--
-- The parameter sol points to the LP problem basis solution block. On
-- entry this block should contain optimal relaxed solution of the MIP
-- problem (i.e. solution where all integer variables are considered as
-- continuous). This initial relaxed solution is used as a root node of
-- branch-and-bound tree. On exit this block contains solution obtained
-- by the mip1_driver routine.
--
-- The parameter cp points to the block of control parameters which
-- affect on the behavior of the mip1_driver routine.
--
-- Since many MIP problems may take a long time, this driver routine
-- reports some visual information about current status of the search.
-- The information is sent to stdout approximately once per second and
-- has the following format:
--
--    +nnn: mip = xxx; lp = yyy (aaa; sss)
--
-- where nnn is total number of simplex iteration, xxx is a value of
-- the objective function that corresponds to the best MIP solution
-- (if this solution has not been found yet, xxx is the text "not found
-- yet"), yyy is a value of the objective function that corresponds to
-- the initial relaxed optimal solution (it is not changed during all
-- the optimization process), aaa is number of subroblems in the active
-- list, sss is number of subproblems which have been solved yet.
--
-- *Returns*
--
-- The mip1_driver routine returns one of the following codes:
--
-- 0 - no errors. This case means that the solver has successfully
--     finished solving the problem;
-- 1 - iteration limit exceeded. In this case the solver reports the
--     recently obtained basis solution;
-- 2 - numerical unstability or problems with the basis matrix. This
--     case means that the solver is not able to solve the problem. */

static LP *lp;
/* LP problem data block */

static LPSOL *sol;
/* LP problem solution block */

static struct mip1_cp *cp;
/* control parameters block */

static int m;
/* number of rows = number of auxiliary variables */

static int n;
/* number of columns = number of structural variables */

static RSM *rsm;
/* simplex method common block */

static double *bbar; /* double bbar[1:m]; */
/* current values of basic variables */

static int *lp_indb; /* int lp_indb[1:m]; */
static int *lp_indn; /* int lp_indn[1:n]; */
static int *lp_tagn; /* int lp_tagn[1:n]; */
static EFI *lp_efi;
static RFI *lp_rfi;
/* copies of the corresponding arrays from RSM for optimal basis of the
   root LP problem; this basis being primal and dual feasible is used as
   initial basis for solving node subproblems ("warm" start) */

static double lp_objval;
/* value of the objective function for the root LP problem */

static int found;
/* if the solver has found at least one integer feasible solution, this
   flag is set; otherwise this flag is clear */

static double best_objval;
/* value of the objective function for the best integer feasible basis
   solution (if found is clear, this value is undefined) */

struct BBNODE
{     /* node of branch-and-bound tree */
      struct BBNODE *up;
      /* pointer to the parent problem (or NULL, if this problem is the
         main LP problem, i.e. root of the tree) */
      int solved;
      /* this problem status:
         0 - problem has not been solved yet (active problem)
         1 - problem has been solved and splitted on two subproblems */
      double objval;
      /* value of the objective function for optimal solution (if the
         problem has not been solved, this value is undefined) */
      double infsum;
      /* sum of integer infeasibilites (if the problem has not been
         solved, this sum is undefined) */
      int j;
      /* number of structural integer variable xS[j] (1 <= j <= n) that
         was chosen to create this problem (not used for the root) */
      int type;
      /* type of new bound of the variable xS[j]:
         'L' - xS[j] >= new lower bound
         'U' - xS[j] <= new upper bound */
      double bound;
      /* new lower (if type = 'L') or upper (if type = 'U') bound for
         the variable xS[j]; new lower bound is always set to ceil(beta)
         and new upper bound is always set to floor(beta), where beta is
         value of the variable xS[j] in optimal solution of the parent
         problem */
      struct BBNODE *left, *right;
      /* if this problem is active (solved = 0), left and right point
         respectively to the previous and to the next active problems;
         if this problem is solved (solved = 1), left and right point
         respectively to the first and to the second subproblems, on
         which this problem has been splitted; in the latter case one of
         these pointers may be NULL, that means that the corresponding
         subproblem is fathomed; both these pointers can't be NULL at
         the same time, because if this happens, this problem is also
         considered as fathomed and automatically removed from the tree
         by the solver */
      struct BBNODE *temp;
      /* temporary pointer (used for different purposes) */
};

static POOL *bb_pool;
/* memory pool that holds all branch-and-bound nodes */

static struct BBNODE *root_node;
/* pointer to the root node */

static struct BBNODE *first_node, *last_node;
/* pointers to the first and to the last active nodes (new nodes are
   always added to the end of this list) */

static int active;
/* number of problems in the active list */

static int solved;
/* number of problems which have been solved */

static clock_t t_last;
/* most recent time at which visual information was displayed */

/*----------------------------------------------------------------------
-- initialize - initialize branch-and-bound environment.
--
-- This routine creates and initializes data structures which are used
-- by the branch-and-bound solver. */

static void initialize(LP *_lp, LPSOL *_sol, struct mip1_cp *_cp)
{     int i, j, k;
      lp = _lp;
      sol = _sol;
      cp = _cp;
      m = lp->m;
      n = lp->n;
      if (lp->kind == NULL)
         fault("mip1_driver: problem has no integer variables");
      if (!(sol->m == m && sol->n == n))
         fault("mip1_driver: inconsistent dimension");
      /* check that all integer variables have integer bounds */
      for (j = 1; j <= n; j++)
      {  if (!lp->kind[j]) continue;
         k = m + j; /* x[k] = xS[j] */
         if (lp->lb[k] != floor(lp->lb[k] + 0.5))
            fault("mip1_driver: lower bound of some integer structural "
               "variable is not integer");
         if (lp->ub[k] != floor(lp->ub[k] + 0.5))
            fault("mip1_driver: upper bound of some integer structural "
               "variable is not integer");
      }
      /* create common block for the revised simplex method */
      rsm = umalloc(sizeof(RSM));
      rsm->m = m;
      rsm->n = n;
      rsm->type = ucalloc(1+m+n, sizeof(int));
      rsm->lb = ucalloc(1+m+n, sizeof(double));
      rsm->ub = ucalloc(1+m+n, sizeof(double));
      rsm->A = create_mat(m, m+n);
      rsm->posx = ucalloc(1+m+n, sizeof(int));
      rsm->indb = ucalloc(1+m, sizeof(int));
      rsm->indn = ucalloc(1+n, sizeof(int));
      rsm->tagn = ucalloc(1+n, sizeof(int));
      switch (cp->form)
      {  case 0:
            /* use EFI */
            rsm->efi = create_efi(m);
            rsm->rfi = NULL;
            break;
         case 1:
            /* use RFI + Bartels & Golub updating technique */
            rsm->efi = NULL;
            rsm->rfi = create_rfi(m);
            rsm->rfi->tech = RFI_BG;
            break;
         case 2:
            /* use RFI + Forrest & Tomlin updating technique */
            rsm->efi = NULL;
            rsm->rfi = create_rfi(m);
            rsm->rfi->tech = RFI_FT;
            break;
         default:
            insist(cp->form != cp->form);
      }
      rsm->iter = 0;
      /* copy types and bounds of variables */
      for (k = 1; k <= m+n; k++)
      {  rsm->type[k] = lp->type[k];
         rsm->lb[k] = lp->lb[k];
         rsm->ub[k] = lp->ub[k];
      }
      /* build the expanded matrix A = (I | -A'), where I is the unity
         matrix, A' is the original matrix of constraint coefficients */
      for (i = 1; i <= m; i++)
         new_elem(rsm->A, i, i, +1.0);
      for (j = 1; j <= n; j++)
      {  ELEM *e;
         for (e = lp->A->col[j]; e != NULL; e = e->col)
            new_elem(rsm->A, e->i, m+j, - e->val);
      }
      /* construct initial basis which is optimal basis of the root LP
         problem */
      if (sol->status != 'O')
err:     fault("mip1_driver: initial basis is invalid");
      i = j = 0;
      for (k = 1; k <= m+n; k++)
      {  if (sol->tagx[k] == 'B')
         {  /* x[k] = xB[i] */
            i++;
            if (i > m) goto err;
            rsm->posx[k] = +i;
            rsm->indb[i] =  k;
         }
         else
         {  /* x[k] = xN[j] */
            j++;
            if (j > n) goto err;
            rsm->posx[k] = -j;
            rsm->indn[j] =  k;
            rsm->tagn[j] = sol->tagx[k];
         }
      }
      if (!(i == m && j == n)) goto err;
      /* check RSM for correctness */
      check_rsm(rsm);
      /* reinvert the initial basis matrix */
      if (invert_b(rsm) != 0) goto err;
      /* compute initial values of basic variables */
      bbar = ucalloc(1+m, sizeof(double));
      eval_bbar(rsm, bbar);
      /* save the initial basis (for "warm" start) */
      lp_indb = ucalloc(1+m, sizeof(int));
      lp_indn = ucalloc(1+n, sizeof(int));
      lp_tagn = ucalloc(1+n, sizeof(int));
      for (i = 1; i <= m; i++)
         lp_indb[i] = rsm->indb[i];
      for (j = 1; j <= n; j++)
      {  lp_indn[j] = rsm->indn[j];
         lp_tagn[j] = rsm->tagn[j];
      }
      lp_efi = NULL;
      lp_rfi = NULL;
      if (rsm->efi != NULL)
      {  lp_efi = create_efi(m);
         copy_per(lp_efi->lu->P, rsm->efi->lu->P);
         copy_mat(lp_efi->lu->L, rsm->efi->lu->L);
         copy_mat(lp_efi->lu->U, rsm->efi->lu->U);
         copy_per(lp_efi->lu->Q, rsm->efi->lu->Q);
         insist(rsm->efi->eta->head == NULL);
      }
      if (rsm->rfi != NULL)
      {  lp_rfi = create_rfi(m);
         {  HTERM *h;
            reset_eta(lp_rfi->H);
            for (h = rsm->rfi->H->head; h != NULL; h = h->next)
               app_term(lp_rfi->H, h->i, h->j, h->val);
         }
         copy_mat(lp_rfi->V, rsm->rfi->V);
         copy_per(lp_rfi->P, rsm->rfi->P);
         copy_per(lp_rfi->Q, rsm->rfi->Q);
         lp_rfi->nzH0 = rsm->rfi->nzH0;
         lp_rfi->nzV0 = rsm->rfi->nzV0;
         lp_rfi->reid = rsm->rfi->reid;
         lp_rfi->tech = rsm->rfi->tech;
         lp_rfi->flag = 0;
      }
      /* set value of the objective function of the root problem */
      lp_objval = (lp->dir == '-' ? + sol->objval : - sol->objval);
      /* no integer feasible solution has been found yet */
      found = 0;
      best_objval = 0.0;
      /* create empty branch-and-bound tree */
      bb_pool = create_pool(sizeof(struct BBNODE));
      root_node = first_node = last_node = NULL;
      active = solved = 0;
      /* reset clock */
      t_last = 0;
      return;
}

/*----------------------------------------------------------------------
-- new_bound - set new bound for a branching variable.
--
-- This routine sets new lower (if type = 'L') or upper (if type = 'U')
-- bound for the structural variable xS[j] of integer type.
--
-- It is assumed that RSM common block corresponds to an optimal basis
-- solution of some relaxed LP problem for which the variable xS[j] was
-- chosen as a branching variable. May note that new bound of xS[j] is
-- tighter that the current one. */

static void new_bound(int j, int type, double bound)
{     int k;
      insist(1 <= j && j <= n);
      k = m + j; /* x[k] = xS[j] */
      switch (type)
      {  case 'L':
            /* xS[j] >= new lower bound */
            switch (rsm->type[k])
            {  case 'F':
                  rsm->type[k] = 'L';
                  rsm->lb[k] = bound;
                  break;
               case 'L':
                  rsm->type[k] = 'L';
                  insist(rsm->lb[k] < bound);
                  rsm->lb[k] = bound;
                  break;
               case 'U':
                  rsm->type[k] = 'D';
                  rsm->lb[k] = bound;
                  break;
               case 'D':
                  rsm->type[k] = 'D';
                  insist(rsm->lb[k] < bound);
                  rsm->lb[k] = bound;
                  break;
               case 'S':
                  /* branching variable can't be of fixed type */
               default:
                  insist(rsm->type[k] != rsm->type[k]);
            }
            break;
         case 'U':
            /* xS[j] <= new upper bound */
            switch (rsm->type[k])
            {  case 'F':
                  rsm->type[k] = 'U';
                  rsm->ub[k] = bound;
                  break;
               case 'L':
                  rsm->type[k] = 'D';
                  rsm->ub[k] = bound;
                  break;
               case 'U':
                  rsm->type[k] = 'U';
                  insist(rsm->ub[k] > bound);
                  rsm->ub[k] = bound;
                  break;
               case 'D':
                  rsm->type[k] = 'D';
                  insist(rsm->ub[k] > bound);
                  rsm->ub[k] = bound;
                  break;
               case 'S':
                  /* branching variable can't be of fixed type */
               default:
                  insist(rsm->type[k] != rsm->type[k]);
            }
            break;
         default:
            insist(type != type);
      }
      /* it may happen that bounds of xS[j] become equal to each other;
         in this case the type of xS[j] should be changed to 'S' */
      if (rsm->type[k] == 'D')
      {  insist(rsm->lb[k] <= rsm->ub[k]);
         if (rsm->lb[k] == rsm->ub[k]) rsm->type[k] = 'S';
      }
      /* if xS[j] is non-basic in the specified basis solution, its tag
         also should be corrected */
      if (rsm->posx[k] < 0)
      {  j = - rsm->posx[k]; /* xN[j] = x[k] */
         switch (rsm->type[k])
         {  case 'L':
               rsm->tagn[j] = 'L';
               break;
            case 'U':
               rsm->tagn[j] = 'U';
               break;
            case 'D':
               insist(rsm->tagn[j] == 'L' || rsm->tagn[j] == 'U');
               break;
            case 'S':
               rsm->tagn[j] = 'S';
               break;
            default:
               insist(rsm->type[k] != rsm->type[k]);
         }
      }
      return;
}

/*----------------------------------------------------------------------
-- reset_rsm - reset basic solution.
--
-- This routine restores RSM common block in order that it corresponds
-- to the initial (optimal) basic solution of the root problem. */

static void reset_rsm(void)
{     int i, j, k;
      /* restore types and bounds of variables */
      for (k = 1; k <= m+n; k++)
      {  rsm->type[k] = lp->type[k];
         rsm->lb[k] = lp->lb[k];
         rsm->ub[k] = lp->ub[k];
      }
      /* restore basic variables */
      for (i = 1; i <= m; i++)
      {  rsm->posx[lp_indb[i]] = +i;
         rsm->indb[i] = lp_indb[i];
      }
      /* restore non-basic variables */
      for (j = 1; j <= n; j++)
      {  rsm->posx[lp_indn[j]] = -j;
         rsm->indn[j] = lp_indn[j];
         rsm->tagn[j] = lp_tagn[j];
      }
      /* restore EFI (if used) */
      if (rsm->efi != NULL)
      {  copy_per(rsm->efi->lu->P, lp_efi->lu->P);
         copy_mat(rsm->efi->lu->L, lp_efi->lu->L);
         copy_mat(rsm->efi->lu->U, lp_efi->lu->U);
         copy_per(rsm->efi->lu->Q, lp_efi->lu->Q);
         reset_eta(rsm->efi->eta);
         rsm->efi->flag = 0;
      }
      /* restore RFI (if used) */
      if (rsm->rfi != NULL)
      {  {  HTERM *h;
            reset_eta(rsm->rfi->H);
            for (h = lp_rfi->H->head; h != NULL; h = h->next)
               app_term(rsm->rfi->H, h->i, h->j, h->val);
         }
         copy_mat(rsm->rfi->V, lp_rfi->V);
         copy_per(rsm->rfi->P, lp_rfi->P);
         copy_per(rsm->rfi->Q, lp_rfi->Q);
         rsm->rfi->nzH0 = lp_rfi->nzH0;
         rsm->rfi->nzV0 = lp_rfi->nzV0;
         rsm->rfi->reid = lp_rfi->reid;
         rsm->rfi->tech = lp_rfi->tech;
         rsm->rfi->flag = 0;
      }
      /* check RSM for correctness */
      check_rsm(rsm);
      return;
}

/*----------------------------------------------------------------------
-- set_bounds - set new bounds for the specified subproblem.
--
-- This routine set new bounds of variables in order to construct the
-- LP subproblem specified by the node of branch-and-bound tree. Bounds
-- are determined by a path from the root node to the specified node.
--
-- On entry RSM common block should correspond to the initial (optimal)
-- basic solution of the root problem. */

static void set_bounds(struct BBNODE *node)
{     /* climb from the specified node to the root node and remember
         the corresponding path (using the temp field of BBNODE) */
      node->temp = NULL;
      for (node = node; node != NULL; node = node->up)
         if (node->up != NULL) node->up->temp = node;
      /* come down from the root node to the specified node and assign
         new (tighter) bounds to branching variables as defined by each
         visited node in the path (the root node is skipped) */
      for (node = root_node->temp; node != NULL; node = node->temp)
         new_bound(node->j, node->type, node->bound);
      /* check RSM for correctness */
      check_rsm(rsm);
      return;
}

/*----------------------------------------------------------------------
-- get_value - get current value of variable.
--
-- This routine determines and returns current value of the variable
-- x[k] (1 <= k <= m+n) using the current basis solution. */

static double get_value(int k)
{     double val;
      insist(1 <= k && k <= m+n);
      if (rsm->posx[k] > 0)
      {  int i = +rsm->posx[k]; /* x[k] = xB[i] */
         insist(1 <= i && i <= m);
         val = bbar[i];
      }
      else
      {  int j = -rsm->posx[k]; /* x[k] = xN[j] */
         insist(1 <= j && j <= n);
         val = eval_xn(rsm, j);
      }
      return val;
}

/*----------------------------------------------------------------------
-- eval_objfun - compute current value of the objective function.
--
-- This routine computes and returns current value of the objective
-- function (which includes the constant term) using the current basis
-- solution. In case of maximization the returned value has opposite
-- sign. */

static double eval_objfun(void)
{     int j;
      double sum = lp->c[0];
      for (j = 1; j <= n; j++) sum += lp->c[j] * get_value(m+j);
      return lp->dir == '-' ? + sum : - sum;
}

/*----------------------------------------------------------------------
-- display - display visual information.
--
-- This routine displays visual information which includes total number
-- of simplex iteration, value of the objective function for the best
-- integer feasible solution, value of the objective function for the
-- root LP problem, number of subproblems in the active list, and number
-- of subproblems which have been solved. */

static void display(void)
{     double s = (lp->dir == '-' ? +1.0 : -1.0);
      if (!found)
         print("+%6d: mip = %17s; lp = %17.9e (%d; %d)",
            rsm->iter, "not found yet", s * lp_objval, active, solved);
      else
         print("+%6d: mip = %17.9e; lp = %17.9e (%d; %d)",
            rsm->iter, s * best_objval, s * lp_objval, active, solved);
      t_last = clock();
      return;
}

/*----------------------------------------------------------------------
-- monit - dual simplex method monitoring routine.
--
-- This routine is called by the rsm_dual routine (which implements the
-- dual revised simplex method) each time before the next iteration.
--
-- This routine is intended for three purposes.
--
-- Firstly, it calls the display routine approximately once per second
-- in order to display visual information about optimization progress.
--
-- Secondly, it checks the current value of the objective function. If
-- this value becomes worse than for the best integer feasible solution,
-- this routine asks the solver to terminate the search, because further
-- optimization can't improve the objective function. In most cases this
-- allows save a time.
--
-- Thirdly, if number of simplex iterations exceeds allowed maximum, it
-- also signals the solver to terminate the search. */

static int monit_flag;
/* why the monitoring routine asks the solver to terminate:
   0 - further optimization is useless
   1 - simplex iteration limit exceeded */

static int monit(void)
{     int ret = 0;
      /* display visual information (once per second) */
      if (t_last == 0 || clock() - t_last > CLOCKS_PER_SEC) display();
      /* if current value of the objective function is not better than
         for the best integer feasible solution, further optimization is
         useless and may be terminated */
      if (found)
      {  eval_bbar(rsm, bbar);
         if (check_rr(eval_objfun(), best_objval, cp->tol_obj) >= -1)
         {  monit_flag = 0;
            ret = 1;
         }
      }
      /* if total number of simplex iteration exceeds allowed maximum,
         optimization should be terminated */
      if (cp->iter_max != 0 && rsm->iter > cp->iter_max)
      {  monit_flag = 1;
         ret = 1;
      }
      return ret;
}

/*----------------------------------------------------------------------
-- remove_node - remove the problem from the active list.
--
-- This routine removes the specified problem (which has been solved)
-- from the active list. Further this problem can be either splitted on
-- two subproblems or removed from the branch-and-bound tree. */

static void remove_node(struct BBNODE *node)
{     insist(!node->solved);
      node->solved = 1;
      if (node->left == NULL)
         first_node = node->right;
      else
         node->left->right = node->right;
      if (node->right == NULL)
         last_node = node->left;
      else
         node->right->left = node->left;
      node->left = node->right = NULL;
      active--, solved++;
      return;
}

/*----------------------------------------------------------------------
-- round_sol - round off values of integer structural variables.
--
-- This routine rounds off the current values of integer structural
-- variables which are close to the corresponding integer values or to
-- their (integer) bounds. The routine assumes that the current basis
-- solution is primal feasible.
--
-- This operation is extremely important for correct branching. */

static void round_sol(void)
{     int i, j, k;
      double tol = cp->tol_int, ival;
      /* look up the list of basic variables (because only values of
         basic integer structural variables may be not integer) */
      for (i = 1; i <= m; i++)
      {  k = rsm->indb[i]; /* x[k] = xB[i] */
         if (k <= m) continue; /* skip auxiliary variable */
         j = k - m; /* x[k] = xS[j] */
         insist(1 <= j && j <= n);
         if (!lp->kind[j]) continue; /* skip continuous variable */
         /* if xS[j] violates its bound (slightly, because the basis is
            optimal and hence primal feasible), it should be set exactly
            on its bound; this is extremely important, since if we would
            not do so, we would make a false branching */
         if (rsm->type[k] == 'L' || rsm->type[k] == 'D' ||
             rsm->type[k] == 'S')
         {  /* xS[j] has lower bound */
            if (bbar[i] < rsm->lb[k]) bbar[i] = rsm->lb[k];
         }
         if (rsm->type[k] == 'U' || rsm->type[k] == 'D' ||
             rsm->type[k] == 'S')
         {  /* xS[j] has upper bound */
            if (bbar[i] > rsm->ub[k]) bbar[i] = rsm->ub[k];
         }
         /* if xS[j] is close to the corresponding integer point, we
            may suppose that actually its value is exact integer, and
            a small difference appears due to round-off errors */
         ival = floor(bbar[i] + 0.5); /* the nearest integer */
         if (ival - tol <= bbar[i] && bbar[i] <= ival + tol)
            bbar[i] = ival;
      }
      return;
}

/*----------------------------------------------------------------------
-- eval_infsum - compute sum of integer infeasibilities.
--
-- This routine returns the sum of integer infeasibilities computed for
-- the current basis solution (if the solution is integer feasible, the
-- sum is exactly equal to zero). */

static double eval_infsum(void)
{     int j;
      double sum = 0.0, val, t1, t2;
      for (j = 1; j <= n; j++)
      {  if (lp->kind[j])
         {  val = get_value(m+j);
            t1 = val - floor(val);
            t2 = ceil(val) - val;
            insist(t1 >= 0.0 && t2 >= 0.0);
            sum += (t1 < t2 ? t1 : t2);
         }
      }
      return sum;
}

/*----------------------------------------------------------------------
-- is_int - check if integer structural variable is integer feasible.
--
-- If the structural variable xS[j] (1 <= j <= n) either continuous or
-- integer whose value is integer feasible, the routine returns true.
-- Otherwise, the routine returns false. (It is assumed that values of
-- all integer variables were adjusted by the round_sol routine.) */

static int is_int(int j)
{     double val;
      insist(1 <= j && j <= n);
      if (!lp->kind[j]) return 1; /* continuous */
      val = get_value(m+j);
      return val == floor(val + 0.5);
}

/*----------------------------------------------------------------------
-- store_sol - store basis solution.
--
-- This routine stores the basis solution specified by RSM common block
-- into LP solution block. */

static void store_sol(int status, double objval)
{     int i, j, k;
      /* set mip problem flag */
      sol->mipsol = 1;
      /* set solution status */
      sol->status = status;
      /* store value of the objective function */
      sol->objval = (lp->dir == '-' ? +objval : -objval);
      /* store values of variables */
      for (k = 1; k <= m+n; k++)
      {  if (rsm->posx[k] > 0)
         {  i = +rsm->posx[k]; /* x[k] = xB[i] */
            sol->tagx[k] = 'B';
            if (cp->round && fabs(bbar[i]) < cp->tol_bnd) bbar[i] = 0.0;
            sol->valx[k] = bbar[i];
            sol->dx[k] = 0.0;
         }
         else
         {  j = -rsm->posx[k]; /* x[k] = xN[j] */
            sol->tagx[k] = rsm->tagn[j];
            sol->valx[k] = eval_xn(rsm, j);
            sol->dx[k] = 0.0;
         }
      }
      return;
}

/*----------------------------------------------------------------------
-- choose_branch - choose a branch using Driebeek-Tomlin heuristic.
--
-- This routine chooses a branching variable using an easy heuristic
-- proposed by Driebeek and developed by Tomlin.
--
-- The routine returns number of a chosen integer structural variable
-- xS[j], 1 <= j <= n, which should be used for constructing two child
-- subproblems. It also sets the parameter tag which specifies what
-- subproblem should be solved first: tag = 'L' means the subproblem
-- where xS[j] has new lower bound, and tag = 'U' means the subproblem
-- where xS[j] has new upper bound. If the current solution is integer
-- feasible, the routine returns zero. */

static int choose_branch(int *tag)
{     int i, j, j_max = 0;
      double dz_max = -1.0, *c, *pi, *cbar, *zeta, *ap;
      /* allocate working arrays */
      c = ucalloc(1+m+n, sizeof(double));
      pi = ucalloc(1+m, sizeof(double));
      cbar = ucalloc(1+n, sizeof(double));
      zeta = ucalloc(1+m, sizeof(double));
      ap = ucalloc(1+n, sizeof(double));
      /* build the expanded vector of coefficients of the objective
         function */
      for (i = 1; i <= m; i++) c[i] = 0.0;
      for (j = 1; j <= n; j++)
         c[m+j] = (lp->dir == '-' ? +1.0 : -1.0) /* S[j]*/ * lp->c[j];
      /* compute simplex multipliers */
      eval_pi(rsm, c, pi);
      /* compute reduced costs ob non-basic variables */
      eval_cbar(rsm, c, pi, cbar);
      /* look up the list of structural variables */
      for (j = 1; j <= n; j++)
      {  int p, q;
         double dx, dz_lo, dz_up;
         /* if xS[j] is continuous or if xS[j] is of integer type and
            its current value is integer feasible, skip this variable */
         if (is_int(j)) continue;
         /* xS[j] should be basic variable, because non-basic variables
            of integer type are always integer feasible */
         p = rsm->posx[m+j]; /* xB[p] = xS[j] */
         insist(1 <= p && p <= m);
         /* we need know a degradation of the objective function if
            xS[j] leave the basis, where degradation is minimal value
            by which the objective function worsens (increases) in the
            adjacent basis; to determine this quantity we simulate one
            iteration of the dual simplex method */
         /* compute p-th row of inv(B) */
         eval_zeta(rsm, p, zeta);
         /* compute p-th (pivot) row of the simplex table */
         eval_row(rsm, zeta, ap);
         /* choose non-basic variable xN[q] which would enter the basis
            (in order to keep the basis dual feasible) if basic variable
            xB[p] would be set on new LOWER bound leaving the basis */
         if (!cp->relax)
         {  /* use standard ratio test */
            q = dual_col(rsm, 'L', ap, cbar, cp->tol_piv);
         }
         else
         {  /* use two-pass ratio test */
            q = harris_col(rsm, 'L', ap, c, cbar, cp->tol_piv,
               0.15 * cp->tol_dj);
         }
         /* determine degradation of the objective function for the
            adjacent basis */
         if (q == 0)
         {  /* if the basis would be changed, the problem would have no
               primal feasible solution (infinite degradation) */
            dz_lo = DBL_MAX;
         }
         else
         {  /* compute increment of the variable xB[p] in the adjacent
               basis solution */
            dx = (ceil(get_value(m+j)) - get_value(m+j)) /* S[j]*/;
            insist(dx > 0.0);
            /* compute increment of the objective function */
            dz_lo = (cbar[q] / ap[q]) * dx;
            /* the increment can't be negative (because the current
               basis is optimal); it may happen only due to round-off
               errors */
            if (dz_lo < 0.0) dz_lo = 0.0;
         }
         /* choose non-basic variable xN[q] which would enter the basis
            (in order to keep the basis dual feasible) if basic variable
            xB[p] would be set on new UPPER bound leaving the basis */
         if (!cp->relax)
         {  /* use standard ratio test */
            q = dual_col(rsm, 'U', ap, cbar, cp->tol_piv);
         }
         else
         {  /* use two-pass ratio test */
            q = harris_col(rsm, 'U', ap, c, cbar, cp->tol_piv,
               0.30 * cp->tol_dj);
         }
         /* determine degradation of the objective function for the
            adjacent basis */
         if (q == 0)
         {  /* if the basis would be changed, the problem would have no
               primal feasible solution (infinite degradation) */
            dz_up = DBL_MAX;
         }
         else
         {  /* compute increment of the variable xB[p] in the adjacent
               basis solution */
            dx = (floor(get_value(m+j)) - get_value(m+j)) /* S[j]*/;
            insist(dx < 0.0);
            /* compute increment of the objective function */
            dz_up = (cbar[q] / ap[q]) * dx;
            /* the increment can't be negative (because the current
               basis is optimal); it may happen only due to round-off
               errors */
            if (dz_up < 0.0) dz_up = 0.0;
         }
         /* we choose that xS[j] which involves greatest degradation
            of the objective function; the subproblem where degradation
            is less should be solved first as a child of the current
            problem; other subproblem should be add to the active list
            to be solved later */
         if (dz_max < dz_lo)
         {  j_max = j, dz_max = dz_lo;
            *tag = (dz_lo < dz_up ? 'L' : 'U');
         }
         if (dz_max < dz_up)
         {  j_max = j, dz_max = dz_up;
            *tag = (dz_lo < dz_up ? 'L' : 'U');
         }
      }
      /* free working arrays */
      ufree(c);
      ufree(pi);
      ufree(cbar);
      ufree(zeta);
      ufree(ap);
      /* return to the calling routine */
      return j_max;
}

/*----------------------------------------------------------------------
-- split_node - split problem on two child subproblems.
--
-- This routine splits the specified problem on two child subproblems
-- using branching variable xS[j], 1 <= j <= n. In the first subproblem
-- the variable xS[j] gets new lower bound, and in the second one the
-- variable xS[j] gets new upper bound. */

static void split_node(struct BBNODE *node, int j)
{     struct BBNODE *child;
      insist(node->solved);
      insist(1 <= j && j <= n);
      /* create the first child subproblem */
      node->left = child = get_atom(bb_pool);
      child->up = node;
      child->solved = 0;
      child->objval = child->infsum = 0.0;
      child->j = j;
      child->type = 'L';
      child->bound = ceil(get_value(m+j));
      child->left = last_node;
      child->right = NULL;
      child->temp = NULL;
      /* add the first subproblem to the active list */
      if (first_node == NULL)
         first_node = child;
      else
         last_node->right = child;
      last_node = child;
      active++;
      /* create the second child subproblem */
      node->right = child = get_atom(bb_pool);
      child->up = node;
      child->solved = 0;
      child->objval = child->infsum = 0.0;
      child->j = j;
      child->type = 'U';
      child->bound = floor(get_value(m+j));
      child->left = last_node;
      child->right = NULL;
      child->temp = NULL;
      /* add the second subproblem to the active list */
      if (first_node == NULL)
         first_node = child;
      else
         last_node->right = child;
      last_node = child;
      active++;
      return;
}

/*----------------------------------------------------------------------
-- delete_node - delete the problem from branch-and-bound tree.
--
-- This routine deletes the specified problem from the branch-and-bound
-- tree. If after that the parent problem has no child subproblems, the
-- routine also deletes it recursively from the tree. */

static void delete_node(struct BBNODE *node)
{     struct BBNODE *parent;
      insist(node != NULL);
loop: /* determine the parent */
      parent = node->up;
      /* detach the current node from its parent */
      if (parent != NULL)
      {  if (parent->left == node)
            parent->left = NULL;
         else if (parent->right == node)
            parent->right = NULL;
         else
            insist(parent != parent);
      }
      /* delete the current node */
      free_atom(bb_pool, node);
      /* if the parent exists and now has no childs, it also should be
         deleted */
      if (parent != NULL)
      {  if (parent->left == NULL && parent->right == NULL)
         {  node = parent;
            goto loop;
         }
      }
      return;
}

/*----------------------------------------------------------------------
-- choose_node - choose active problem using the best projection.
--
-- This routine chooses an active problem which should be solved next
-- (this is needed after the branching process has been terminated).
--
-- If there is no integer feasible solution, the routine chooses an
-- active problem which has smallest sum of integer infeasibilities.
-- Otherwise, if the current best solution has been found, the routine
-- chooses an active problem using the best projection heuristuc. */

static struct BBNODE *choose_node(void)
{     struct BBNODE *this = NULL, *node;
      if (!found)
      {  /* no integer feasible solution has been found yet */
         double minsum = DBL_MAX;
         for (node = first_node; node != NULL; node = node->right)
         {  if (minsum > node->infsum)
               this = node, minsum = node->infsum;
         }
      }
      else
      {  /* there is the current best solution is already known */
         double best = DBL_MAX;
         for (node = first_node; node != NULL; node = node->right)
         {  double dz, val;
            /* dz estimates degradation of the objective function per
               unit of the sum of integer infeasibilities */
            dz = (best_objval - lp_objval) / root_node->infsum;
            /* val estimates optimal value of the objective function if
               the sum of integer infeasibilities would be zero */
            val = node->objval + dz * node->infsum;
            /* choose the problem which have the best estimated optimal
               value of the objective function */
            if (best > val) this = node, best = val;
         }
      }
      return this;
}

/*----------------------------------------------------------------------
-- terminate - terminate branch-and-bound environment.
--
-- This routine free all memory allocated to data structures which are
-- used by the branch-and-bound solver. */

static void terminate(void)
{     ufree(rsm->type);
      ufree(rsm->lb);
      ufree(rsm->ub);
      delete_mat(rsm->A);
      ufree(rsm->posx);
      ufree(rsm->indb);
      ufree(rsm->indn);
      ufree(rsm->tagn);
      if (rsm->efi != NULL) delete_efi(rsm->efi);
      if (rsm->rfi != NULL) delete_rfi(rsm->rfi);
      ufree(rsm);
      ufree(bbar);
      ufree(lp_indb);
      ufree(lp_indn);
      ufree(lp_tagn);
      if (lp_efi != NULL) delete_efi(lp_efi);
      if (lp_rfi != NULL) delete_rfi(lp_rfi);
      delete_pool(bb_pool);
      return;
}

/*----------------------------------------------------------------------
-- mip1_driver - driver routine.
--
-- This routine is a driver for the branch-and-bound procedure which is
-- intended for solving mixed integer linear programming problems. */

int mip1_driver(LP *_lp, LPSOL *_sol, struct mip1_cp *_cp)
{     struct BBNODE *node;
      int child = 0, i, j, ret, tag;
      double *c, *dvec = NULL;
      /* initialize branch-and-bound environment */
      initialize(_lp, _sol, _cp);
      /* build the expanded vector of coefficients of the objective
         function c = (0, c'), where 0 is zero subvector for auxiliary
         variables, c' is the original vector of coefficients for
         structural variables (in order to simplify program logic the
         case of maximization is reduced to the case of minimization) */
      c = ucalloc(1+m+n, sizeof(double));
      for (i = 1; i <= m; i++) c[i] = 0.0;
      switch (lp->dir)
      {  case '-':
            for (j = 1; j <= n; j++) c[m+j] = + lp->c[j];
            break;
         case '+':
            for (j = 1; j <= n; j++) c[m+j] = - lp->c[j];
            break;
         default:
            insist(lp->dir != lp->dir);
      }
      /* if the dual steepest edge technique is required, create the
         vector delta */
      if (cp->steep) dvec = ucalloc(1+m, sizeof(double));
      /* create the root problem and add it to the active list */
      root_node = node = get_atom(bb_pool);
      node->up = NULL;
      node->solved = 0;
      node->objval = node->infsum = 0.0;
      node->j = 0;
      node->type = '?';
      node->bound = 0.0;
      node->left = node->right = node->temp = NULL;
      first_node = last_node = node;
      active++;
loop: /* main loop starts here */
      /* node points to an active subproblem which should be solved */
      insist(node != NULL && !node->solved);
      /* construct initial basis for the subproblem (any dual feasible
         basis may be used) */
      if (child)
      {  /* the flag child means that the current active subproblem is
            a child of the last solved subproblem whose (optimal) basis
            was stored in RSM common block; in this case the basis may
            be used as initial one for solving the current subproblem;
            therefore it is enough only to tight (lower or upper) bound
            of a branching variable for the current subproblem */
         new_bound(node->j, node->type, node->bound);
      }
      else
      {  /* otherwise, if the flag child is not set, the optimal basis
            of the root problem is used as initial one for solving the
            current active subproblem */
         /* restore optimal basis of the root problem */
         reset_rsm();
         /* assign new bounds to branching variables to construct the
            current subproblem */
         set_bounds(node);
         /* reset the vector delta (if the dual steepest edge technique
            is used) */
         if (cp->steep) for (i = 1; i <= m; i++) dvec[i] = 1.0;
      }
      /* igitur, the initial basis in RSM block is optimal for some
         parent problem, therefore it is dual feasible for the current
         subproblem; use the dual simplex method in order to find an
         optimal basis solution of the current subproblem */
      ret = rsm_dual(rsm, monit, c, cp->tol_bnd, cp->tol_dj,
         cp->tol_piv, dvec, cp->relax);
      /* remove the current subproblem from the active list, because it
         has been solved */
      remove_node(node);
      /* analyze return code */
      if (ret == 4 && monit_flag == 0) ret = 1;
      switch (ret)
      {  case 0:
            /* optimal solution found */
            break;
         case 1:
            /* problem has no feasible solution */
            goto cut;
         case 2:
            /* the current basis solution became dual infeasible due to
               numerical instability */
            display();
            error("Numerical stability lost");
            error("Sorry, basis recovery procedure not implemented");
            ret = 2;
            goto done;
         case 3:
            /* the basis matrix became singular or ill-conditioned due
               to unsuccessful choice of the pivot element on the last
               dual simplex iteration */
            display();
            error("Numerical problems with basis matrix");
            error("Sorry, basis recovery procedure not implemented");
            ret = 2;
            goto done;
         case 4:
            /* iteration limit exceeded; search terminated */
            display();
            print("ITERATION LIMIT EXCEEDED; SEARCH TERMINATED");
            ret = 1;
            goto done;
         default:
            insist(ret != ret);
      }
      /* the dual simplex method has found an optimal solution of the
         current subproblem */
      /* compute values of all basic variables */
      eval_bbar(rsm, bbar);
      /* round off values of those (basic) integer structural variables
         which are close to the corresponding integer values or to their
         (integer) bounds; this operation is extremely important for
         correct branching */
      round_sol();
      /* compute value of the objective function */
      node->objval = eval_objfun();
      /* compute sum of integer infeasibilities */
      node->infsum = eval_infsum();
      /* store the solution of the root problem */
      if (node->up == NULL)
      {  int status = 'O';
         insist(root_node == node);
         for (j = 1; j <= n; j++)
         {  if (!is_int(j))
            {  status = 'I';
               break;
            }
         }
         store_sol(status, node->objval);
      }
      /* if only the initial solution is required, print an appropriate
         message and return */
      if (node->up == NULL && cp->what == 0)
      {  switch (sol->status)
         {  case 'O':
               found = 1;
               best_objval = node->objval;
               display();
               print("Initial solution is INTEGER OPTIMAL");
               break;
            case 'I':
               print("Initial solution is INTEGER INFEASIBLE");
               break;
            default:
               insist(sol->status != sol->status);
         }
         ret = 0;
         goto done;
      }
      /* if value of the objective function for the current subproblem
         is not better than for the best solution found before, further
         branching has no sense */
      if (found
         && check_rr(node->objval, best_objval, cp->tol_obj) >= -1)
         goto cut;
      /* choose a branching variable xS[j] among structural integer
         variables which are integer infeasible */
      switch (cp->branch)
      {  case 0:
            /* branch on the first integer variable */
            for (j = 1; j <= n; j++) if (!is_int(j)) break;
            if (j > n) j = 0;
            tag = 'U';
            break;
         case 1:
            /* branch on the last integer variable */
            for (j = n; j >= 1; j--) if (!is_int(j)) break;
            tag = 'U';
            break;
         case 2:
            /* branch using heuristic by Driebeek and Tomlin */
            j = choose_branch(&tag);
            break;
         default:
            insist(cp->branch != cp->branch);
      }
      /* if the choice is impossible, the current basis solution being
         integer feasible is a new best solution */
      if (j == 0)
      {  /* new best integer feasible basis solution has been found */
         found = 1;
         best_objval = node->objval;
         display();
         /* store this new solution */
         store_sol('F', node->objval);
         /* if only some integer feasible solution is required, print
            an appropriate message and return */
         if (cp->what == 1)
         {  print("INTEGER FEASIBLE SOLUTION FOUND");
            ret = 0;
            goto done;
         }
         /* branching is finished */
         goto cut;
      }
      /* structural integer variable xS[j] has been chosen for further
         branching */
      insist(1 <= j && j <= n);
      /* splite the current problem on two child subproblems and add
         them to the active list (the parent subproblem is kept in the
         tree) */
      split_node(node, j);
      /* choose an active problem (that is one of child subproblems of
         the current problem) which should be considered next */
      switch (tag)
      {  case 'L':
            /* that one where xS[j] >= new lower bound */
            node = node->left;
            insist(node->type == 'L');
            break;
         case 'U':
            /* that one where xS[j] <= new upper bound */
            node = node->right;
            insist(node->type == 'U');
            break;
         default: insist(tag != tag);
      }
      /* note that the new current problem is a child of the old current
         problem */
      child = 1;
      /* continue the search */
      goto loop;
cut:  /* further branching for the current problem has no sense because
         its solution is not better than the current best solution or it
         has no primal feasible solution at all; this allows to cut off
         a whole subtree whose root is the current problem */
      /* delete the current problem (and parent problems which will have
         no child subproblems) from the branch-and-bound tree */
      delete_node(node);
      /* in the active list there may be other problems whose solutions
         even they would be integer feasible can't be better than the
         current best solution (this is determined using analysis of the
         corresponding parent problems, because the active problems have
         not been solved yet); such active problems also can be removed
         from the active list and deleted from the tree */
      if (found)
      {  /* look up the active list */
         node = first_node;
         while (node != NULL)
         {  struct BBNODE *next;
            /* save pointer to the next active node (although deleting
               active node may involve deleting other nodes, the pointer
               to the next active node remains correct, because other
               nodes are parents and don't belong to the active list) */
            next = node->right;
            /* if solution of the parent problem is not better than the
               current best solution, its child problem also can't have
               better solution and can be deleted */
            if (node->up != NULL && check_rr(node->up->objval,
               best_objval, cp->tol_obj) >= -1)
            {  /* remove the problem from the active list */
               remove_node(node);
               /* and delete it from the branch-and-bound tree */
               delete_node(node);
            }
            /* consider the next active problem */
            node = next;
         }
      }
      /* since branching has been terminated, it is necessary to choose
         some problem from the active list (so called "backtrack") */
      switch (cp->btrack)
      {  case 0:
            /* choose a node using FIFO strategy */
            node = first_node;
            break;
         case 1:
            /* choose a node using LIFO strategy */
            node = last_node;
            break;
         case 2:
            /* choose a node using the best projection heuristic */
            node = choose_node();
            break;
         default:
            insist(cp->btrack != cp->btrack);
      }
      /* if the choice is impossible, the active list is empty; thus,
         the current best solution is integer optimal solution of the
         MIP problem */
      if (node == NULL)
      {  /* active list is empty */
         display();
         if (!found)
         {  print("PROBLEM HAS NO INTEGER FEASIBLE SOLUTION");
            sol->status = 'N';
         }
         else
         {  print("INTEGER OPTIMAL SOLUTION FOUND");
            sol->status = 'O';
         }
         ret = 0;
         goto done;
      }
      /* branching was terminated, therefore the new current problem is
         NOT a child of old current problem */
      child = 0;
      /* continue the search */
      goto loop;
done: /* free working arrays */
      ufree(c);
      if (cp->steep) ufree(dvec);
      /* terminate branch-and-bound environment */
      terminate();
      /* return to the calling program */
      return ret;
}

/* eof */
