/* glpmat/load_mat.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <string.h>
#include "glpmat.h"

/*----------------------------------------------------------------------
-- load_mat - read sparse matrix from text file using plain format.
--
-- *Synopsis*
--
-- #include "glpmat.h"
-- MAT *load_mat(char *fname);
--
-- *Description*
--
-- The load_mat routine reads a sparse matrix from the text file, whose
-- name is the character string fname, using plain format (this format
-- is described in comments to the routine save_mat; note that the order
-- of matrix elements may be arbitrary).
--
-- *Returns*
--
-- If the operation was successful, the load_mat routine returns a
-- pointer to the loaded matrix. Otherwise the routine returns NULL. */

MAT *load_mat(char *fname)
{     DATA *ds;
      MAT *A = NULL;
      int m, n, nz, i, j, k;
      double aij;
      char str[255+1];
      print("load_mat: reading sparse matrix from `%s'...", fname);
      ds = open_data(fname);
      if (ds == NULL) goto fail;
      if (scan_data(ds, 's', str, 0)) goto fail;
      if (strcmp(str, "MAT"))
      {  error("%s:%d: marker `MAT' not found", ds->fn, ds->cn);
         goto fail;
      }
      if (scan_data(ds, 'i', &m, 'i', &n, 'i', &nz, 0)) goto fail;
      if (m <= 0)
      {  error("%s:%d: invalid number of rows", ds->fn, ds->cn);
         goto fail;
      }
      if (n <= 0)
      {  error("%s:%d: invalid number of columns", ds->fn, ds->cn);
         goto fail;
      }
      if (nz < 0)
      {  error("%s:%d: invalid number of elements", ds->fn, ds->cn);
         goto fail;
      }
      A = create_mat(m, n);
      for (k = 1; k <= nz; k++)
      {  if (scan_data(ds, 'i', &i, 'i', &j, 'e', &aij, 0)) goto fail;
         if (!(1 <= i && i <= m))
         {  error("%s:%d: row index out of range", ds->fn, ds->cn);
            goto fail;
         }
         if (!(1 <= j && j <= n))
         {  error("%s:%d: column index out of range", ds->fn, ds->cn);
            goto fail;
         }
         new_elem(A, i, j, aij);
      }
      if (scan_data(ds, 's', str, 0)) goto fail;
      if (strcmp(str, "EOD"))
      {  error("%s:%d: marker `EOD' not found", ds->fn, ds->cn);
         goto fail;
      }
      close_data(ds);
      return A;
fail: if (ds != NULL) close_data(ds);
      if (A != NULL) delete_mat(A);
      return NULL;
}

/* eof */
