/* glpavl/insert_by_key.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glpavl.h"
#include "glpset.h"

/*----------------------------------------------------------------------
-- insert_by_key - insert new node with given key to AVL-tree.
--
-- *Synopsis*
--
-- #include "glpavl.h:
-- AVLNODE *insert_by_key(AVLTREE *tree, void *key);
--
-- *Description*
--
-- The insert_by_key routine creates a new node with given key and
-- inserts this node to the AVL-tree. (To perform this operation the
-- keys comparing routine should be defined.)
--
-- The insertion is performed in any case whether there are nodes with
-- the same key in the tree or not. To check whether the node with the
-- same key already exists the find_by_key routine should be used.
--
-- It's assumed that data representing a key of the inserted node and
-- which key points to has yet been allocated. This allocation should be
-- kept at least while the node is in the tree or while the tree exists.
--
-- After return the fileds type and link of the inserted node will
-- contain binary zeros. These fileds are available through the pointer
-- returned by the insert_by_key routine and may be used for arbitrary
-- purposes.
--
-- *Returns*
--
-- The insert_by_key routine returns a pointer to the inserted node. */

AVLNODE *insert_by_key(AVLTREE *tree, void *key)
{     AVLNODE *p, *q, *r;
      short int flag;
      if (tree->comp == NULL)
         fault("insert_by_key: keys comparing routine not defined");
      /* search for an appropriate place to insert the new node */
      p = NULL; q = tree->root;
      while (q != NULL)
      {  p = q;
         if (tree->comp(key, p->key) <= 0)
         {  flag = 0;
            q = p->left;
            p->rank++;
         }
         else
         {  flag = 1;
            q = p->right;
         }
      }
      /* create and insert the new node */
      r = get_atom(tree->pool);
      r->key = key; r->type = 0; r->link = NULL;
      r->rank = 1; r->up = p;
      r->flag = (short int)(p == NULL ? 0 : flag);
      r->bal = 0; r->left = NULL; r->right = NULL;
      tree->size++;
      if (p == NULL)
         tree->root = r;
      else
         if (flag == 0) p->left = r; else p->right = r;
      /* go up to the root and correct all nodes affected by the
         insertion */
      while (p != NULL)
      {  if (flag == 0)
         {  /* the height of the left subtree of [p] is increased */
            if (p->bal > 0)
            {  p->bal = 0;
               break;
            }
            if (p->bal < 0)
            {  rotate_avl(tree, p);
               break;
            }
            p->bal = -1; flag = p->flag; p = p->up;
         }
         else
         {  /* the height of the right subtree of [p] is increased */
            if (p->bal < 0)
            {  p->bal = 0;
               break;
            }
            if (p->bal > 0)
            {  rotate_avl(tree, p);
               break;
            }
            p->bal = +1; flag = p->flag; p = p->up;
         }
      }
      /* if the root reached, the height of entire tree is increased */
      if (p == NULL) tree->height++;
      return r;
}

/* eof */
