/* glprsm.h */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#ifndef _GLPRSM_H
#define _GLPRSM_H

#include <time.h>
#include "glpefi.h"
#include "glpmat.h"
#include "glprfi.h"

#define rsm_btran             glp_btran
#define rsm_change_basis      glp_change_basis
#define rsm_check_basis       glp_check_basis
#define rsm_check_bbar        glp_check_bbar
#define rsm_check_cbar        glp_check_cbar
#define rsm_check_dev         glp_check_dev
#define rsm_check_dzeta       glp_check_dzeta
#define rsm_check_gamma       glp_check_gamma
#define rsm_defect            glp_defect
#define rsm_dual_col          glp_dual_col
#define rsm_dual_opt          glp_dual_opt
#define rsm_dual_row          glp_dual_row
#define rsm_exact_dzeta       glp_exact_dzeta
#define rsm_exact_gamma       glp_exact_gamma
#define rsm_find_feas         glp_find_feas
#define rsm_ftran             glp_ftran
#define rsm_get_xn            glp_get_xn
#define rsm_harris_col        glp_harris_col
#define rsm_harris_row        glp_harris_row
#define rsm_infsum            glp_infsum
#define rsm_init_dzeta        glp_init_dzeta
#define rsm_init_gamma        glp_init_gamma
#define rsm_invert            glp_invert
#define rsm_objval            glp_objval
#define rsm_pivot_col         glp_pivot_col
#define rsm_pivot_row         glp_pivot_row
#define rsm_primal_opt        glp_primal_opt
#define rsm_scaling           glp_scaling
#define rsm_set_ap            glp_set_ap
#define rsm_set_aq            glp_set_aq
#define rsm_set_bbar          glp_set_bbar
#define rsm_set_bp            glp_set_bp
#define rsm_set_cbar          glp_set_cbar
#define rsm_srt_pi            glp_set_pi
#define rsm_update_dzeta      glp_update_dzeta
#define rsm_update_gamma      glp_update_gamma

typedef struct RSM RSM;

struct RSM
{     /* revised simplex method common block */
      /* --- problem data segment --- */
      int m;
      /* number of rows (auxiliary variables) */
      int n;
      /* number of columns (structural variables) */
      int *type; /* int type[1+m+n]; */
      /* type[0] is not used; type[k] defines the type of variable x[k]
         (1 <= k <= m+n) as follows: */
#define LP_FR     0  /* free variable:    -inf <  x[k] <  +inf */
#define LP_LO     1  /* lower bound:      l[k] <= x[k] <  +inf */
#define LP_UP     2  /* upper bound:      -inf <  x[k] <= u[k] */
#define LP_DB     3  /* double bound:     l[k] <= x[k] <= u[k] */
#define LP_FX     4  /* fixed variable:   l[k]  = x[k]  = u[k] */
      double *lb; /* double lb[1+m+n]; */
      /* lb[0] is not used; lb[k] is the lower bound of variable x[k]
         (1 <= k <= m+n); if x[k] has no lower bound, lb[k] is zero */
      double *ub; /* double ub[1+m+n]; */
      /* ub[0] is not used; ub[k] is the upper bound of variable x[k]
         (1 <= k <= m+n); if x[k] has no upper bound, ub[k] is zero;
         if x[k] is fixed variable, lb[k] is equal to ub[k] */
      MAT *A; /* MAT A[1:m,1:m+n]; */
      /* expanded matrix of constraint coefficients A~ = (I | -A),
         where I is the unity matrix of order m, A is the source matrix
         of constraint coefficients from struct LP; using the matrix A~
         the system of equality constraints may be written as A~*x = 0,
         where x = (xR, xS) is the united vector of all variables, xR is
         the subvector of auxiliary variables x[1], ..., x[m], xS is the
         subvector of structural variables x[m+1], ..., x[m+n] */
      int dir;
      /* optimization direction flag: */
#define LP_MIN 0  /* objective function should be MINimized */
#define LP_MAX 1  /* objective function should be MAXimized */
      int obj;
      /* row (auxiliary variable) number that defines the objective
         function (1 <= obj <= m); obj may be equal to zero to indicate
         that the objective function is identically equal to zero */
      /* --- quadratic programming segment --- */
      int qp_flag;
      /* this flag is set in case of solving quadratic programming
         problem */
      int mm;
      /* numbers of rows of original QP problem */
      int nn;
      /* numbers of columns of original QP problem */
      /* should note that m = n = mm + nn */
      /* --- problem scaling segment --- */
      double *R; /* double R[1+m]; */
      /* row scaling diagonal matrix; R[0] is not used; R[1], ..., R[m]
         are diagonal elements of the matrix R */
      double *S; /* double S[1+n]; */
      /* column scaling diagonal matrix; S[0] is not used; S[1], ...,
         S[n] are diagonal elements of the matrix S */
      /* the matrix of constraint coefficients for the scaled problem
         has the form R*A*S, where A is the original (unscaled) matrix;
         if the scaling is not used, R and S are unity matrices */
      /* --- basis solution segment --- */
      int *posx; /* int posx[1+m+n]; */
      /* posx[0] is not used; posx[k] is the position of the variable
         x[k] (1 <= k <= m+n) in the vector xB of basis variables or in
         the vector xN of non-basis variables:
         if posx[k] = +i then x[k] = xB[i] (1 <= i <= m)
         if posx[k] = -j then x[k] = xN[j] (1 <= j <= n) */
      int *indb; /* int indb[1+m]; */
      /* indb[0] is not used; indb[i] = k if xB[i] = x[k] */
      int *indn; /* int indn[1+n]; */
      /* indn[0] is not used; indn[j] = k if xN[j] = x[k] */
      int *tagn; /* int tagn[1+n]; */
      /* tagn[0] is not used; tagn[j] is a tag used for the non-basis
         variable xN[j] (1 <= j <= n): */
#define LP_NL     1  /* non-basis variable on lower bound */
#define LP_NU     2  /* non basis variable on upper bound */
#define LP_NF     3  /* non-basis free variable */
#define LP_NS     4  /* non-basis fixed variable */
      /* --- simplex method segment (primal & dual) --- */
      EFI *efi;
      /* if EFI (elimination form of the inverse) is used to represent
         the current basis matrix B, efi points to EFI; otherwise the
         efi field is equal to NULL */
      RFI *rfi;
      /* if RFI (Reid's form of the inverse) is used to represent the
         current basis matrix B, rfi points to RFI; otherwise the rfi
         field is equal to NULL */
      double *bbar; /* double bbar[1+m]; */
      /* bbar[0] is not used; bbar[i] is the current value of the basis
         variable xB[i] (1 <= i <= m) */
      double *cost; /* double cost[1+m+n]; */
      /* cost[0] is not used; cost[k] is a coefficient of the auxiliary
         objective function at the variable x[k] (1 <= k <= m+n); this
         objective function is to be always minimized */
      double *pi; /* double pi[1+m]; */
      /* pi[0] is not used; pi[i] is a simplex multiplier corresponding
         to the i-th equality constraint (1 <= i <= m) */
      double *cbar; /* double cbar[1+n]; */
      /* cbar[0] is not used; cbar[j] is the reduced cost (marginal) of
         the non-basis variable xN[j] (1 <= j <= n); they correspond to
         the auxiliary objective function */
      double *dzeta; /* double dzeta[1+m]; */
      /* dzeta[0] is not used; dzeta[i] is the weight coefficient for
         the i-th row (1 <= i <= m) of the current simplex table; these
         weights are used for the dual steepest edge pricing (in order
         to use the standard "textbook" pricing these weights should be
         set to 1) */
      double *gamma; /* double gamma[1+n]; */
      /* gamma[0] is not used; gamma[j] is the weight coefficient for
         the j-th column (1 <= j <= n) of the current simplex table;
         these weights are used for the primal steepest edge pricing
         (in order to use the standard "textbook" pricing these weights
         should be set to 1) */
      int p;
      /* number of the basis variable xB[p] which has been chosen to
         leave the basis (if p = 0 then choice is impossible; if p < 0
         then the chosen non-basis variable xN[q] should be moved from
         the current bound to the opposite one) */
      int tagp;
      /* it is a tag which should be assigned to the basis variable
         xB[p] when this variable will have left the basis */
      double *bp; /* double bp[1+m]; */
      /* p-th row of inv(B); bp[0] is not used; elements are placed in
         locations bp[1], ..., bp[m] */
      double *ap; /* double ap[1+n]; */
      /* ap[0] is not used; ap[j] is the coefficient at the non-basis
         variable xN[j] in p-th row of the current simplex table that
         corresponds to the basis variable xB[p] (1 <= p <= m), i.e.
         xB[p] = ap[1]*xN[1] + ... + ap[n]*xN[n] */
      int q;
      /* number of the non-basis variable xN[q] which has been chosen
         to enter the basis (if q = 0 then choice is impossible) */
      double *aq; /* double aq[1+m]; */
      /* aq[0] is not used; aq[i] is the coefficient at the non-basis
         variable xN[q] in i-th row of the current simplex table that
         corresponds to the basis variable xB[i] (1 <= i <= m), i.e.
         xB[i] = ... + aq[i]*xN[q] + ... */
      /* --- control parameters segment --- */
      int prim_steep;
      /* entering variable option (primal simplex):
         0 - use "textbook" technique
         1 - use steepest edge technique (this means updating gamma on
             each primal simplex iteration) */
      int prim_relax;
      /* leaving variable option (primal simplex):
         0 - use "textbook" ratio test
         1 - use a technique proposed by P.Harris */
      int dual_steep;
      /* leaving variable option (dual simplex):
         0 - use "textbook" technique
         1 - use steepest edge technique (this means updating dzeta on
             each dual simplex iteration) */
      int dual_relax;
      /* entering variable option (dual simplex):
         0 - use "textbook" ratio test
         1 - use a technique proposed by P.Harris */
      double tol_bnd;
      /* tolerance used for checking primal feasibility */
      double tol_dj;
      /* tolerance used for checking dual feasibility */
      double tol_piv;
      /* tolerance used for pivoting */
      int check_basis;
      /* if this flag is set, the routines check basis segment data
         structures for correctness on each simplex iteration (only for
         debugging) */
      int check_dzeta;
      /* if this flag is set, the routines check the accuracy of the
         updated weights dzeta comparing them with exact values on each
         simplex iteration (only for debugging) */
      int check_gamma;
      /* if this flag is set, the routines check the accuracy of the
         updated weights gamma comparing them with exact values on each
         simplex iteration (only for debugging) */
      int fin_out;
      /* if this flag is set, the searching routines display final
         solution at the end of the stage */
      /* --- statistics segment --- */
      int iter;
      /* iteration count */
      clock_t t_last;
      /* most recent time at which visual information was displayed */
};

extern double *rsm_btran(RSM *rsm, double z[]);
/* perform backward transformation (BTRAN) */

extern int rsm_change_basis(RSM *rsm);
/* change basis */

extern void rsm_check_basis(RSM *rsm);
/* check basis segment for correctness */

extern int rsm_check_bbar(RSM *rsm, double tol);
/* check current solution for numerical stability (primal) */

extern int rsm_check_cbar(RSM *rsm, double tol);
/* check current solution for numerical stability (dual) */

extern int rsm_check_dev(double x, double x0, double tol);
/* check deviation from given value */

extern double rsm_check_dzeta(RSM *rsm);
/* check accuracy of updated weights (dual) */

extern double rsm_check_gamma(RSM *rsm);
/* check accuracy of updated weights (primal) */

extern int rsm_defect(RSM *rsm);
/* determine defect of current basis solution */

extern double rsm_dual_col(RSM *rsm, double tol);
/* choose non-basis variable (dual simplex) */

extern int rsm_dual_opt(RSM *rsm);
/* find optimal solution by dual simplex method */

extern int rsm_dual_row(RSM *rsm, double tol);
/* choose basis variable (dual simplex) */

extern double rsm_exact_dzeta(RSM *rsm, int i);
/* compute exact weight (dual steepest edge) */

extern double rsm_exact_gamma(RSM *rsm, int j);
/* compute exact weight (steepest edge technique) */

extern int rsm_find_feas(RSM *rsm);
/* find primal feasible solution */

extern double *rsm_ftran(RSM *rsm, double z[], int save);
/* perform forward transformation (FTRAN) */

extern double rsm_get_xn(RSM *rsm, int j);
/* get current value of non-basis variable */

extern double rsm_harris_col(RSM *rsm, double tol, double tol1);
/* choose non-basis variable (dual, Harris technique) */

extern double rsm_harris_row(RSM *rsm, double tol, double tol1);
/* choose basis variable (primal, Harris technique) */

extern double rsm_infsum(RSM *rsm);
/* compute sum of infeasibilities */

extern void rsm_init_dzeta(RSM *rsm);
/* initialize weights (dual steepest edge) */

extern void rsm_init_gamma(RSM *rsm);
/* initialize weights (steepest edge technique) */

extern int rsm_invert(RSM *rsm);
/* reinvert basis matrix */

extern double rsm_objval(RSM *rsm);
/* obtain current value of objective function */

extern void rsm_pivot_col(RSM *rsm, double tol);
/* choose non-basis variable (primal simplex) */

extern double rsm_pivot_row(RSM *rsm, double tol);
/* choose basis variable (primal simplex) */

extern int rsm_primal_opt(RSM *rsm);
/* find optimal solution by primal simplex method */

extern void rsm_scaling(RSM *rsm);
/* scale linear programming problem data */

extern void rsm_set_ap(RSM *rsm);
/* compute pivot row of simplex table */

extern void rsm_set_aq(RSM *rsm);
/* compute pivot column of simplex table */

extern void rsm_set_bbar(RSM *rsm);
/* compute current values of basis variables */

extern void rsm_set_bp(RSM *rsm);
/* compute p-th row of the inverse inv(B) */

extern void rsm_set_cbar(RSM *rsm);
/* compute reduced costs of non-basis variables */

extern void rsm_set_pi(RSM *rsm);
/* compute simplex multipliers */

extern void rsm_update_dzeta(RSM *rsm, double w[]);
/* update weights (dual steepest edge) */

extern void rsm_update_gamma(RSM *rsm, double y[], double w[]);
/* update weights (steepest edge technique) */

#endif

/* eof */
