/* glpapi.h */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#ifndef _GLPAPI_H
#define _GLPAPI_H

#include "glpk.h"
#include "glpavl.h"
#include "glpstr.h"

#define glp glp_workspace

typedef struct GLPWKSP GLPWKSP;
typedef struct GLPITEM GLPITEM;
typedef struct GLPCOEF GLPCOEF;

struct GLPWKSP
{     /* GLPK API workspace */
      /* --- memory management segment --- */
      POOL *str_pool;
      /* memory pool for segmented character strings */
      POOL *item_pool;
      /* memory pool for allocating GLPITEM objects */
      POOL *coef_pool;
      /* memory pool for allocating GLPCOEF objects */
      /* --- problem data segment --- */
      AVLTREE *row_tab;
      /* row names look-up table */
      GLPITEM *first_row;
      /* pointer to the first row */
      GLPITEM *last_row;
      /* pointer to the last row */
      GLPITEM *this_row;
      /* pointer to the current row */
      AVLTREE *col_tab;
      /* column names look-up table */
      GLPITEM *first_col;
      /* pointer to the first column */
      GLPITEM *last_col;
      /* pointer to the last column */
      GLPITEM *this_col;
      /* pointer to the current column */
      GLPCOEF *this_coef;
      /* pointer to the current constraint coefficient */
      /* --- control parameters segment --- */
      char mps_bnd_name[8+1]; /* used by the glp_read_mps routine */
      /* this character string specifies the name of bound vector that
         should be used in the problem; if this parameter is an empty
         string, the first bound vector is used */
      char mps_obj_name[8+1]; /* used by the glp_read_mps routine */
      /* this character string specifies the name of row that should be
         used as the objective function; if this parameter is an empty
         string, the first row of N type (i.e. free row) is used as the
         objective function */
      int mps_one_entry; /* used by the glp_write_mps routine */
      /* if this flag is set, the routine keeps fields 5 and 6 empty;
         otherwise the routine uses all fields */
      int mps_pedantic; /* used by the glp_write_mps routine */
      /* if this flag is set, the routine never omits column and vector
         names; otherwise the routine omits those names every time when
         it's possible */
      char mps_rhs_name[8+1]; /* used by the glp_read_mps routine */
      /* this character string specifies the name of right-hand side
         (RHS) vector that should be used in the problem; if this
         parameter is an empty string, the first RHS vector is used */
      char mps_rng_name[8+1]; /* used by the glp_read_mps routine */
      /* this character string specifies the name of range vector that
         should be used in the problem; if this parameter is an empty
         string, the first range vector is used */
      int mps_skip_empty; /* used by the glp_write_mps routine */
      /* if this flag is set, the routine writes all columns, ibcluding
         empty ones (i.e. those which have no constraint coefficients);
         otherwise the routine write only non-empty columns */
      int mps_use_names; /* used by the glp_write_mps routine */
      /* if this flag is set, the routine tries to generate standard
         MPS names using orifinal row and column names as templates;
         otherwise the routine generates plain names using sequential
         numbers of rows and columns */
      int obj_dir; /* used by the solver */
      /* optimization flag:
         GLP_INI - only initial solution should be found
         GLP_ANY - only feasible solution should be found
         GLP_MIN - objective function should be minimized
         GLP_MAX - objective function should be maximized */
      GLPITEM *obj_row; /* used by the solver */
      /* pointer to the objective function row; NULL indicates that the
         objective function is identically equal to zero */
      STR *problem; /* informative */
      /* problem name (1 to 255 chars) */
      int round; /* used by the solver */
      /* if this flag is set, the solver replaces values of those basis
         variables which are near to zero, by exact zeros; otherwise the
         solver remains values of all basis variables "as is" */
      int scale; /* used by the solver */
      /* if this flag is set, the solver performs automatic scaling the
         problem before solving; otherwise the scaling is not used */
      int spx_form; /* used by the solver */
      /* this parameter specifies what form of the basis matrix should
         be used by the solver:
         GLP_EFI    - EFI
         GLP_RFI_BG - RFI + Bartels & Golub updating technique
         GLP_RFI_FT - RFI + Forrest & Tomlin updating technique */
      int spx_relax; /* used by the solver */
      /* if this flag is set, the solver uses two-pass ratio test (for
         both primal and dual simplex) proposed by P.Harris; otherwise
         the solver uses the standard "textbook" ratio test */
      int spx_steep; /* used by the solver */
      /* if this flag is set, the solver uses the steepest edge pricing
         (for both primal and dual simplex) proposed by Goldfarb & Reid;
         otherwise the solver uses the standard "textbook" pricing */
      int spx_use_dual; /* used by the solver */
      /* if this flag is set, the solver uses the dual simplex method
         for searching for feasible solution; otherwise the solver uses
         the primal simplex method */
      int status; /* reported by the solver */
      /* solution status:
         GLP_UNDEF  - solution status is undefined
         GLP_OPT    - solution is optimal
         GLP_FEAS   - soultion is feasible
         GLP_INFEAS - solution is infeasible
         GLP_NOFEAS - the problem has no feasible solution
         GLP_UNBND  - the problem has unbounded solution */
      int sum_aij; /* used by the solver */
      /* if this flag is set, the solver replaces multiplets (i.e.
         elements of the constraint matrix that have identical row and
         column indices) by their sum before solving the problem;
         otherwise the solver considers multiplets as incorrect data */
      double tol_aij; /* used by the solver */
      /* this parameter specifies the absolute tolerance which is used
         by the solver to drop small elements of the original constraint
         matrix; if this tolerance is equal to zero, the solver uses the
         original matrix without any changes */
      double tol_bnd; /* used by the solver */
      /* this parameter specifies the relative tolerance which is used
         by the solver for checking primal feasibility of the current
         basis solution */
      double tol_dj; /* used by the solver */
      /* this parameter specifies the relative tolerance which is used
         by the solver for checking dual feasibility of the current
         basis solution */
      double tol_piv; /* used by the solver */
      /* this parameter specifies the relative tolerance which is used
         by the solver for pivoting */
};

struct GLPITEM
{     /* row (auxiliary variable) or column (structural variable) */
      STR *name;
      /* row/column name (1 to 255 chars) */
      int type;
      /* row/column type:
         GLP_FR - free variable:  -inf <  x[k] <  +inf
         GLP_LO - lower bound:    l[k] <= x[k] <  +inf
         GLP_UP - upper bound:    -inf <  x[k] <= u[k]
         GLP_DB - double bound:   l[k] <= x[k] <= u[k]
         GLP_FX - fixed variable: l[k]  = x[k]  = u[k] */
      double lb;
      /* lower bound; if row/column has no lower bound, lb is zero */
      double ub;
      /* upper bound; if row/column has no upper bound, ub is zero;
         in case of fixed variable (GLP_FX) lb is equal to ub */
      GLPCOEF *ptr;
      /* pointer to the linked list of constraint coefficients placed
         in this row/column (this linked list is unordered) */
      int seqn;
      /* this field is used by GLP routines for different purposes */
      int tagx;
      /* row/column status:
         GLP_BS - basis variable
         GLP_NL - non-basis variable on lower bound
         GLP_NU - non basis variable on upper bound
         GLP_NF - non-basis free variable
         GLP_NS - non-basis fixed variable */
      double valx;
      /* row/column primal activity */
      double dx;
      /* row/column dual activity */
      GLPITEM *next;
      /* pointer to the next row/column (created row/column is always
         added to the end of this linked list) */
};

struct GLPCOEF
{     /* constraint coefficient */
      GLPITEM *row;
      /* pointer to the row */
      GLPITEM *col;
      /* pointer to the column */
      double val;
      /* coefficient value */
      GLPCOEF *next1;
      /* pointer to the next coefficient in the same row */
      GLPCOEF *next2;
      /* pointer to the next coefficient in the same column */
};

extern GLPWKSP *glp;
/* pointer to the active workspace */

#endif

/* eof */
