/* glprsm/rsm_harris_row.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <float.h>
#include <math.h>
#include "glprsm.h"

/*----------------------------------------------------------------------
-- rsm_harris_row - choose basis variable (primal, Harris technique).
--
-- *Synopsis*
--
-- #include "glprsm.h"
-- double rsm_harris_row(RSM *rsm, double tol, double tol1);
--
-- *Description*
--
-- The rsm_harris_row performs the same operation as the rsm_pivot_row
-- routine, but using a technique proposed by P.Harris. This technique
-- is a two-pass technique. On the first pass a relaxed steplength tmax
-- is computed; it is maximal allowable absolute change of the chosen
-- non-basis variable xN[q] in valid direction, when each basis variable
-- xB[i] is yet inside its relaxed bounds:
--
--    lB[i] - gap <= xB[i] <= uB[i] + gap,
--
-- where gap = tol1*max(|lB[i]|,1) or gap = tol1*max(|uB[i]|,1) that
-- depends on what bound is relaxed. On the second pass for each basis
-- variable xB[i] a textbook steplength teta[i] is computed (in the
-- program code teta[i] is temp); it is maximal allowable absolute
-- change of the chosen non-basis variable xN[q] in valid direction,
-- when xB[i] is yet inside its original bounds (when all other basis
-- variables are ignored):
--
--    lB[i] <= xB[i] <= uB[i].
--
-- Following the Harris' technique the routine chooses that basis
-- variable xB[p], for which teta[p] <= tmax and which has the largest
-- absolute value of the influence coefficient aq[p].
--
-- Should note that if xB[p] is chosen using the Harris' technique,
-- in the adjacent basis some basis variables may slightly violate their
-- bounds being inside its relaxed, not original bounds.
--
-- The advantage of the Harris' techinque over standard technique is
-- that in case of degeneracy (i.e. when some basis variables are near
-- to their bounds) this technique allows to choose the basis variable
-- having larger (in absolute value) coefficient aq[p], that essentialy
-- improves numerical stability of all operations related to the basis
-- matrix.
--
-- For details see:
-- P.M.J.Harris. Pivot selection methods of the Devex lp code. Math.
-- Prog., 5, 1973, 1-28.
--
-- *Returns*
--
-- The rsm_harris_row routine returns absolute change of the non-basis
-- variable xN[q], which will happen when this variable will enter the
-- basis. */

double rsm_harris_row(RSM *rsm, double tol, double tol1)
{     int m = rsm->m, i, k, tag;
      double *lb = rsm->lb, *ub = rsm->ub, *bbar = rsm->bbar;
      double big, eps, temp, tmax, teta;
#     define gap(bnd) (tol1 * (fabs(bnd) > 1.0 ? fabs(bnd) : 1.0))
      /* compute the absolute tolerance eps using the given relative
         tolerance tol */
      big = 0.0;
      for (i = 1; i <= m; i++)
         if (big < fabs(rsm->aq[i])) big = fabs(rsm->aq[i]);
      eps = tol * big;
      /* turn to the case of increasing xN[q] in order to simplify
         program logic (remember that the objective function should be
         minimized) */
      if (rsm->cbar[rsm->q] > 0.0)
         for (i = 1; i <= m; i++) rsm->aq[i] = - rsm->aq[i];
      /* initial settings for the first pass */
      k = rsm->indn[rsm->q]; /* x[k] = xN[q] */
      if (rsm->type[k] == LP_DB)
         tmax = (ub[k] - lb[k]) + gap(ub[k] - lb[k]);
      else
         tmax = DBL_MAX;
      /* the first look through the list of basis variables */
      for (i = 1; i <= m; i++)
      {  /* if the coefficient aq[i] is too small, it is assumed that
            xB[i] doesn't depend on xN[q] */
         if (rsm->aq[i] == 0.0 || fabs(rsm->aq[i]) < eps) continue;
         /* analyze main cases */
         k = rsm->indb[i]; /* x[k] = xB[i] */
         if (rsm->type[k] == LP_FR)
         {  /* xB[i] is free variable */
            continue;
         }
         else if (rsm->type[k] == LP_LO)
         {  /* xB[i] has lower bound */
            if (rsm->aq[i] > 0.0) continue;
lo_1:       temp = ((lb[k] - gap(lb[k])) - bbar[i]) / rsm->aq[i];
         }
         else if (rsm->type[k] == LP_UP)
         {  /* xB[i] has upper bound */
            if (rsm->aq[i] < 0.0) continue;
up_1:       temp = ((ub[k] + gap(ub[k])) - bbar[i]) / rsm->aq[i];
         }
         else if (rsm->type[k] == LP_DB)
         {  /* xB[i] has both lower and upper bounds */
            if (rsm->aq[i] < 0.0) goto lo_1; else goto up_1;
         }
         else if (rsm->type[k] == LP_FX)
         {  /* xB[i] is fixed variable */
            if (rsm->aq[i] < 0.0) goto lo_1; else goto up_1;
         }
         else
            insist(rsm->type[k] != rsm->type[k]);
         /* if xB[i] slightly violates its (relaxed!) bound, temp is
            negative; in this case it is assumes thst xB[i] is exactly
            on its (relaxed!) bound, so temp is replaced by zero */
         if (temp < 0.0) temp = 0.0;
         /* compute maximal allowable change of xN[q] */
         if (tmax > temp) tmax = temp;
      }
      /* initial settings for the second pass */
      rsm->p = 0, rsm->tagp = -1, teta = DBL_MAX, big = 0.0;
      k = rsm->indn[rsm->q]; /* x[k] = xN[q] */
      if (rsm->type[k] == LP_DB)
      {  temp = ub[k] - lb[k];
         if (temp <= tmax)
            rsm->p = -1, rsm->tagp = -1, teta = temp, big = 1.0;
      }
      /* the second look through the list of the basis variable */
      for (i = 1; i <= m; i++)
      {  /* if the coefficient aq[i] is too small, it is assumed that
            xB[i] doesn't depend on xN[q] */
         if (rsm->aq[i] == 0.0 || fabs(rsm->aq[i]) < eps) continue;
         /* analyze main cases */
         k = rsm->indb[i]; /* x[k] = xB[i] */
         if (rsm->type[k] == LP_FR)
         {  /* xB[i] is free variable */
            continue;
         }
         else if (rsm->type[k] == LP_LO)
         {  /* xB[i] has lower bound */
            if (rsm->aq[i] > 0.0) continue;
lo_2:       temp = (lb[k] - bbar[i]) / rsm->aq[i];
            tag = LP_NL;
         }
         else if (rsm->type[k] == LP_UP)
         {  /* xB[i] has upper bound */
            if (rsm->aq[i] < 0.0) continue;
up_2:       temp = (ub[k] - bbar[i]) / rsm->aq[i];
            tag = LP_NU;
         }
         else if (rsm->type[k] == LP_DB)
         {  /* xB[i] has both lower and upper bounds */
            if (rsm->aq[i] < 0.0) goto lo_2; else goto up_2;
         }
         else if (rsm->type[k] == LP_FX)
         {  /* xB[i] is fixed variable */
            temp = 0.0;
            tag = LP_NS;
         }
         else
            insist(rsm->type[k] != rsm->type[k]);
         /* if xB[i] slightly violates its (original!) bound, temp is
            negative; in this case it is assumed that xB[i] is exactly
            on its (original!) bound, so temp is replaced by zero */
         if (temp < 0.0) temp = 0.0;
         /* apply the Harris' rule */
         if (temp <= tmax && big < fabs(rsm->aq[i]))
            rsm->p = i, rsm->tagp = tag, teta = temp,
            big = fabs(rsm->aq[i]);
      }
      /* restore original signs of the coefficient aq[i] */
      if (rsm->cbar[rsm->q] > 0.0)
         for (i = 1; i <= m; i++) rsm->aq[i] = - rsm->aq[i];
      return teta;
#     undef gap
}

/* eof */
