/* glprsm/rsm_change_basis.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glprsm.h"
#include "glpset.h"

/*----------------------------------------------------------------------
-- rsm_change_basis - change basis.
--
-- *Synopsis*
--
-- #include "glprsm.h"
-- int rsm_change_basis(RSM *rsm);
--
-- *Description*
--
-- The rsm_change_basis routine changes the current basis, making the
-- chosen basis variable xB[p] to be non-basis and the chosen non-basis
-- variable xN[q] to be basis. Geometrically this change corresponds to
-- the movement from the current vertex of the feasible polyhedra to
-- the adjacent vertex along the edge. Algebraically it means that p-th
-- column of the current basis matrix B is changed by q-th column of
-- the matrix N (the latter consists of those columns of the expanded
-- constraint matrix A~, which correspond to non-basis variables). In
-- the special case, when q < 0, the routine just transfers the choosen
-- non-basis variable xN[q] from the current bound to the opposite one,
-- and the basis remains unchanged.
--
-- Since the current basis matrix is changed, this routine also updates
-- some representation of this matrix. If the representation becomes too
-- long or inaccurate, the rsm_change_basis routine performs reinversion
-- of the basis matrix.
--
-- *Rerturns*
--
-- If no reinversion was performed, the routine returns zero. Otherwise
-- the rsm_change_basis routine returns an exit code which was reported
-- by the partucular routine that performed reinversion. In the case of
-- error the calling program should analyze representation of the basis
-- matrix in order to exclude dependent columns from the basis matrix
-- replacing them by columns corresponding to auxiliary variables (such
-- columns are unity vectors). */

int rsm_change_basis(RSM *rsm)
{     int m = rsm->m, n = rsm->n, p = rsm->p, q = rsm->q, ret = 0,
         k, kp, kq;
      if (rsm->check_basis) rsm_check_basis(rsm);
      if (p < 0)
      {  /* xN[q] goes from the current bound to the opposite one */
         k = rsm->indn[q]; /* x[k] = xN[q] */
         insist(rsm->type[k] == LP_DB);
         insist(rsm->tagn[q] == LP_NL || rsm->tagn[q] == LP_NU);
         rsm->tagn[q] = (rsm->tagn[q] == LP_NL) ? LP_NU : LP_NL;
      }
      else
      {  /* xB[p] leaves the basis, xN[q] enters the basis */
         insist(1 <= p && p <= m);
         insist(1 <= q && q <= n);
         kp = rsm->indb[p]; kq = rsm->indn[q];
         rsm->posx[kq] = +p; rsm->posx[kp] = -q;
         rsm->indb[p]  = kq; rsm->indn[q]  = kp;
         rsm->tagn[q] = rsm->tagp;
         /* update representation of the basis matrix */
         if (rsm->efi != NULL)
            ret = update_efi(rsm->efi, rsm->p);
         else if (rsm->rfi != NULL)
            ret = update_rfi(rsm->rfi, rsm->p);
         else
            insist(2 + 2 == 5);
         /* if representation became inaccurate or too long, reinvert
            the basis matrix */
         if (ret != 0) ret = rsm_invert(rsm);
      }
      if (rsm->check_basis) rsm_check_basis(rsm);
      rsm->iter++;
      return ret;
}

/* eof */
