/* glpsol.c */

/*----------------------------------------------------------------------
-- This file is a part of the GNU LPK package.
--
-- Copyright (C) 2000 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                    Department for Applied Informatics,
--                    Moscow Aviation Institute, Moscow, Russia.
--                    All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include "glpk.h"
#include "glpset.h"

/*----------------------------------------------------------------------
-- This program is a demo solver based on the revised simplex method.
-- It is intended for solving large scale linear programming problems.
----------------------------------------------------------------------*/

static char *version = "GLPSOL -- GLPK Demo Solver, Version 1.1";

static char *mps_fname = "";
/* the name of the input text file that contains LP problem data in MPS
   format */

static char *out_fname = "";
/* the name of the output text file, to which the final basis solution
   should be sent using printable format (optional) */

/*----------------------------------------------------------------------
-- display_help - display help.
--
-- This routine displays help information about the program as it is
-- required by the GNU Coding Standard. */

static void display_help(char *my_name)
{     printf("Usage: %s [options...] mps-file\n", my_name);
      printf("\n");
      printf("Options:\n");
      printf("   --ini             only initial basis solution should b"
         "e found\n");
      printf("   --any             only feasible basis solution should "
         "be found\n");
      printf("   --min             objective function should be minimiz"
         "ed (default)\n");
      printf("   --max             objective function should be maximiz"
         "ed\n");
      printf("   -o filename, --output filename\n");
      printf("                     send final solution to filename usin"
         "g printable\n");
      printf("                     format\n");
      printf("   --textbook        use options --noscale, --efi, --prim"
         "al, --nosteep,\n");
      printf("                     and --norelax by default\n");
      printf("   --scale           scale the problem (default)\n");
      printf("   --noscale         do not scale the problem\n");
      printf("   --efi             use EFI\n");
      printf("   --rfi-bg          use RFI + Bartels & Golub updating t"
         "echnique\n");
      printf("                     (default)\n");
      printf("   --rfi-ft          use RFI + Forrest & Tomlin updating "
         "technique\n");
      printf("   --primal          find feasible solution using primal "
         "simplex\n");
      printf("                     (default)\n");
      printf("   --dual            find feasible solution using dual si"
         "mplex\n");
      printf("   --steep           use steepest edge technique (default"
         ")\n");
      printf("   --nosteep         use standard \"textbook\" pricing\n")
         ;
      printf("   --relax           use Harris' two-pass ratio test (def"
         "ault)\n");
      printf("   --norelax         use standard \"textbook\" ratio test"
         "\n");
      printf("   --sum             enable multiplets in constraint matr"
         "ix (will be\n");
      printf("                     replaced by their sum)\n");
      printf("   --nosum           disable multiplets in constraint mat"
         "rix (default)\n");
      printf("   -h, --help        display this help information and ex"
         "it\n");
      printf("   -v, --version     display program version and exit\n");
      printf("\n");
      printf("Please, report bugs to <bug-glpk@gnu.org>\n");
      exit(EXIT_SUCCESS);
      /* no return */
}

/*----------------------------------------------------------------------
-- display_version - display version.
--
-- This routine displays version information for the program as it is
-- required by the GNU Coding Standard. */

static void display_version(void)
{     printf("%s\n", version);
      printf("Copyright (C) 2000 Andrew Makhorin <mao@mai2.rcnet.ru>, M"
         "osc.Av.Inst.\n");
      printf("This program is free software; you may redistribute it un"
         "der the terms of\n");
      printf("the GNU General Public License. This program has absolute"
         "ly no warranty.\n");
      exit(EXIT_SUCCESS);
      /* no return */
}

/*----------------------------------------------------------------------
-- process_cmdline - process command line parameters.
--
-- This routine processes parameters specified in command line. */

static void process_cmdline(int argc, char *argv[])
{     int k;
      for (k = 1; k < argc; k++)
      {  if (strcmp(argv[k], "--ini") == 0)
            assert(glp_set_ipar("obj_dir", GLP_INI) == 0);
         else if (strcmp(argv[k], "--any") == 0)
            assert(glp_set_ipar("obj_dir", GLP_ANY) == 0);
         else if (strcmp(argv[k], "--min") == 0)
            assert(glp_set_ipar("obj_dir", GLP_MIN) == 0);
         else if (strcmp(argv[k], "--max") == 0)
            assert(glp_set_ipar("obj_dir", GLP_MAX) == 0);
         else if (strcmp(argv[k], "-o") == 0 ||
                  strcmp(argv[k], "--output") == 0)
         {  k++;
            if (k == argc)
            {  fprintf(stderr, "No output file name specified\n");
               exit(EXIT_FAILURE);
            }
            if (out_fname[0] != '\0')
            {  fprintf(stderr, "Only one output file allowed\n");
               exit(EXIT_FAILURE);
            }
            out_fname = argv[k];
         }
         else if (strcmp(argv[k], "--textbook") == 0)
         {  assert(glp_set_ipar("scale", GLP_NO) == 0);
            assert(glp_set_ipar("spx_form", GLP_EFI) == 0);
            assert(glp_set_ipar("spx_use_dual", GLP_NO) == 0);
            assert(glp_set_ipar("spx_steep", GLP_NO) == 0);
            assert(glp_set_ipar("spx_relax", GLP_NO) == 0);
         }
         else if (strcmp(argv[k], "--scale") == 0)
            assert(glp_set_ipar("scale", GLP_YES) == 0);
         else if (strcmp(argv[k], "--noscale") == 0)
            assert(glp_set_ipar("scale", GLP_NO) == 0);
         else if (strcmp(argv[k], "--efi") == 0)
            assert(glp_set_ipar("spx_form", GLP_EFI) == 0);
         else if (strcmp(argv[k], "--rfi-bg") == 0)
            assert(glp_set_ipar("spx_form", GLP_RFI_BG) == 0);
         else if (strcmp(argv[k], "--rfi-ft") == 0)
            assert(glp_set_ipar("spx_form", GLP_RFI_FT) == 0);
         else if (strcmp(argv[k], "--primal") == 0)
            assert(glp_set_ipar("spx_use_dual", GLP_NO) == 0);
         else if (strcmp(argv[k], "--dual") == 0)
            assert(glp_set_ipar("spx_use_dual", GLP_YES) == 0);
         else if (strcmp(argv[k], "--steep") == 0)
            assert(glp_set_ipar("spx_steep", GLP_YES) == 0);
         else if (strcmp(argv[k], "--nosteep") == 0)
            assert(glp_set_ipar("spx_steep", GLP_NO) == 0);
         else if (strcmp(argv[k], "--relax") == 0)
            assert(glp_set_ipar("spx_relax", GLP_YES) == 0);
         else if (strcmp(argv[k], "--norelax") == 0)
            assert(glp_set_ipar("spx_relax", GLP_NO) == 0);
         else if (strcmp(argv[k], "--sum") == 0)
            assert(glp_set_ipar("sum_aij", GLP_YES) == 0);
         else if (strcmp(argv[k], "--nosum") == 0)
            assert(glp_set_ipar("sum_aij", GLP_NO) == 0);
         else if (strcmp(argv[k], "-h") == 0 ||
                  strcmp(argv[k], "--help") == 0)
            display_help(argv[0]);
         else if (strcmp(argv[k], "-v") == 0 ||
                  strcmp(argv[k], "--version") == 0)
            display_version();
         else if (argv[k][0] == '-' ||
                 (argv[k][0] == '-' && argv[k][1] == '-'))
         {  fprintf(stderr, "Invalid option `%s'; try %s --help\n",
               argv[k], argv[0]);
            exit(EXIT_FAILURE);
         }
         else
         {  if (mps_fname[0] != '\0')
            {  fprintf(stderr, "Only one input file allowed\n");
               exit(EXIT_FAILURE);
            }
            mps_fname = argv[k];
         }
      }
      return;
}

/*----------------------------------------------------------------------
-- main - main routine.
--
-- This main routine is called by the control program and manages the
-- solving process. */

int main(int argc, char *argv[])
{     int obj_dir;
      clock_t t_start;
      /* initialize GLPK API */
      glp_initialize();
      /* process command line parameters */
      process_cmdline(argc, argv);
      /* read LP problem data from MPS file */
      if (mps_fname[0] == '\0')
      {  fprintf(stderr, "No input MPS file specified\n");
         exit(EXIT_FAILURE);
      }
      glp_get_ipar("obj_dir", &obj_dir);
      if (glp_read_mps(mps_fname) != 0)
      {  fprintf(stderr, "Can't read MPS file\n");
         exit(EXIT_FAILURE);
      }
      glp_set_ipar("obj_dir", obj_dir);
      /* solve LP problem */
      t_start = clock();
      if (glp_simplex() != 0)
      {  fprintf(stderr, "Can't solve LP problem\n");
         exit(EXIT_FAILURE);
      }
      /* display statistics */
      printf("Time used: %.1f secs\n",
         (double)(clock() - t_start) / (double)CLOCKS_PER_SEC);
      printf("Memory used: %.1fM (%d bytes)\n",
         (double)mem_tpeak / (1024.0 * 1024.0), mem_tpeak);
      /* write final solution found by the solver */
      if (out_fname[0] != '\0' && glp_print_sol(out_fname) != 0)
      {  fprintf(stderr, "Can't write solution\n");
         exit(EXIT_FAILURE);
      }
      /* terminate GLPK API */
      glp_terminate();
      /* exit to the control program */
      return 0;
}

/* eof */
