/* glprsm/rsm_scaling.c */

/*----------------------------------------------------------------------
-- This file is a part of the GNU LPK package.
--
-- Copyright (C) 2000 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                    Department for Applied Informatics,
--                    Moscow Aviation Institute, Moscow, Russia.
--                    All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glprsm.h"

/*----------------------------------------------------------------------
-- rsm_scaling - scale linear programming problem data.
--
-- *Synopsis*
--
-- #include "glprsm.h"
-- void rsm_scaling(RSM *rsm);
--
-- *Description*
--
-- The rsm_scaling routine performs scaling of the linear programming
-- problem data. At first the routine computes diagonal scaling matrices
-- R and S. After that the routine uses the matrices R and S to scale
-- lower and upper bounds of variables and constraint matrix. New scaled
-- data are stored in the structure RSM which rsm points to instead old
-- unscaled data.
--
-- In order to scale constraint matrix the rsm_scaling routine uses
-- geometric mean scaling that follows equilibration scaling.
--
-- Let the formulation of the original (unscaled) problem has the form:
--
--    xR = A * xS       (equality constraints)
--    lR <= xR <= uR    (bounds of auxiliary variables)
--    lS <= xS <= uS    (bounds of structural variables)
--
-- As a result of scaling the original constraint matrix A is replaced
-- by the scaled matrix A' = R*A*S, where R and S are diagonal scaling
-- matrices. Thus, the formulation of the scaled problem will have the
-- form:
--
--    xR' = A' * xS'
--    lR' <= xR' <= uR'
--    lS' <= xS' <= uS'
--
-- where:
--
--    A'  = R * A * S
--    xR' = R * xR
--    lR' = R * lR
--    uR' = R * uR
--    xS' = inv(S) * xS
--    lS' = inv(S) * lS
--    uS' = inv(S) * uS
--
-- The scaling matrices R and S are saved in the common block and used
-- later in order to descale the computed solution. */

void rsm_scaling(RSM *rsm)
{     int m = rsm->m, n = rsm->n, i, j, k;
      /* compute scaling matrices */
      {  MAT *A;
         A = create_mat(m, n);
         submatrix(A, rsm->A, 1, m, m+1, m+n);
         gm_scaling(A, rsm->R, rsm->S, 0, 0.01, 20);
         eq_scaling(A, rsm->R, rsm->S, 0);
         delete_mat(A);
      }
      /* scale bounds of auxiliary variables */
      for (i = 1; i <= m; i++)
      {  k = i;
         rsm->lb[k] *= rsm->R[i];
         rsm->ub[k] *= rsm->R[i];
      }
      /* scale bound of structural variables */
      for (j = 1; j <= n; j++)
      {  k = m+j;
         rsm->lb[k] /= rsm->S[j];
         rsm->ub[k] /= rsm->S[j];
      }
      /* scale matrix of constraint coefficients */
      for (j = 1; j <= n; j++)
      {  ELEM *e;
         k = m+j;
         for (e = rsm->A->col[k]; e != NULL; e = e->col)
            e->val *= (rsm->R[e->i] * rsm->S[j]);
      }
      return;
}

/* eof */
