/* glprsm/rsm_pivot_row.c */

/*----------------------------------------------------------------------
-- This file is a part of the GNU LPK package.
--
-- Copyright (C) 2000 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                    Department for Applied Informatics,
--                    Moscow Aviation Institute, Moscow, Russia.
--                    All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <assert.h>
#include <float.h>
#include <math.h>
#include "glprsm.h"

/*----------------------------------------------------------------------
-- rsm_pivot_row - choose basis variable (primal simplex).
--
-- *Synopsis*
--
-- #include "glprsm.h"
-- double rsm_pivot_row(RSM *rsm, double tol);
--
-- *Description*
--
-- The rsm_pivot_routine performs the ratio test in order to choose the
-- basis variable xB[p] (i.e. pivot row of the current simplex table),
-- which will leave the basis. It is assumed that q is the number of
-- the non-basis variable xN[q] that was chosen to enter the basis, and
-- the array aq contains q-th column of the current simplex table that
-- shows an influence of xN[q] to all basis variables xB. If xN[q]
-- reaches its opposite bound before any basis variable, the routine
-- sets p to -1. If the current solution is unbounded and hence it is
-- impossible to choose xB[p], the routine sets p to zero. The routine
-- also sets tagp, which shows to what bound the chosen basis variable
-- should be set when it will leave the basis.
--
-- The parameter tol is a relative tolerance. If the coefficient aq[i]
-- is too small, i.e. if |aq[i]| < eps, where eps = tol*max|aq[*]|, then
-- basis variable xB[i] is not considered by the routine.
--
-- The rsm_pivot_routine tries to choose such basis variable xB[i],
-- which reaches its bound before any other basis variables, when the
-- non-basis variable xN[q] is changing in valid direction (increasing,
-- if cbar[q] < 0, or decreasing, if cbar[q] > 0). Linear dependence
-- between xB[i] and xN[q] is the following:
--
--    xB[i] = ... + aq[i] * xN[q] + ...
--
-- *Returns*
--
-- The rsm_pivot_row routine returns absolute change of the non-basis
-- variable xN[q], which will happen when this variable will enter the
-- basis. */

double rsm_pivot_row(RSM *rsm, double tol)
{     int m = rsm->m, i, k, tag;
      double big, eps, temp, teta;
      /* compute the absolute tolerance eps using the given relative
         tolerance tol */
      big = 0.0;
      for (i = 1; i <= m; i++)
         if (big < fabs(rsm->aq[i])) big = fabs(rsm->aq[i]);
      eps = tol * big;
      /* turn to the case of increasing xN[q] in order to simplify
         program logic (remember that the objective function should be
         minimized) */
      if (rsm->cbar[rsm->q] > 0.0)
         for (i = 1; i <= m; i++) rsm->aq[i] = - rsm->aq[i];
      /* initial settings */
      k = rsm->indn[rsm->q]; /* x[k] = xN[q] */
      if (rsm->type[k] == LP_DB)
      {  rsm->p = -1;
         rsm->tagp = -1;
         teta = rsm->ub[k] - rsm->lb[k];
         big = 1.0;
      }
      else
      {  rsm->p = 0;
         rsm->tagp = -1;
         teta = DBL_MAX;
         big = 0.0;
      }
      /* look through the list of basis variables */
      for (i = 1; i <= m; i++)
      {  /* if the coefficient aq[i] is too small, it is assumed that
            xB[i] doesn't depend on xN[q] */
         if (rsm->aq[i] == 0.0 || fabs(rsm->aq[i]) < eps) continue;
         /* analyze main cases */
         k = rsm->indb[i]; /* x[k] = xB[i] */
         if (rsm->type[k] == LP_FR)
         {  /* xB[i] is free variable */
            continue;
         }
         else if (rsm->type[k] == LP_LO)
         {  /* xB[i] has lower bound */
            if (rsm->aq[i] > 0.0) continue;
lo:         temp = (rsm->lb[k] - rsm->bbar[i]) / rsm->aq[i];
            tag = LP_NL;
         }
         else if (rsm->type[k] == LP_UP)
         {  /* xB[i] has upper bound */
            if (rsm->aq[i] < 0.0) continue;
up:         temp = (rsm->ub[k] - rsm->bbar[i]) / rsm->aq[i];
            tag = LP_NU;
         }
         else if (rsm->type[k] == LP_DB)
         {  /* xB[i] has both lower and upper bounds */
            if (rsm->aq[i] < 0.0) goto lo; else goto up;
         }
         else if (rsm->type[k] == LP_FX)
         {  /* xB[i] is fixed variable */
            temp = 0.0;
            tag = LP_NS;
         }
         else
            assert(rsm->type[k] != rsm->type[k]);
         /* if xB[i] slightly violates its bound, temp is negative;
            in this case it is assumes that xB[i] is exactly on its
            bound, therefore temp is replaced by zero */
         if (temp < 0.0) temp = 0.0;
         /* apply minimal ratio test */
         if (teta > temp || teta == temp && big < fabs(rsm->aq[i]))
         {  rsm->p = i;
            rsm->tagp = tag;
            teta = temp;
            big = fabs(rsm->aq[i]);
         }
      }
      /* restore original signs of the coefficients aq[i] */
      if (rsm->cbar[rsm->q] > 0.0)
         for (i = 1; i <= m; i++) rsm->aq[i] = - rsm->aq[i];
      return teta;
}

/* eof */
