/* glprsm/rsm_init_gamma.c */

/*----------------------------------------------------------------------
-- This file is a part of the GNU LPK package.
--
-- Copyright (C) 2000 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                    Department for Applied Informatics,
--                    Moscow Aviation Institute, Moscow, Russia.
--                    All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glprsm.h"

/*----------------------------------------------------------------------
-- rsm_init_gamma - initialize weights (steepest edge technique).
--
-- *Synopsis*
--
-- #include "glprsm.h"
-- void rsm_init_gamma(RSM *rsm);
--
-- *Description*
--
-- The routine rsm_init_gamma initializes weights, which are used in
-- steepest edge technique, and stores them to the array gamma. It is
-- assumed that B = I and N = -A, i.e. the simplex table is the same
-- as the constraint matrix A.
--
-- The routine uses the following formula:
--
--    gamma[j] = 1 + SUM alfa[i,j]^2, j = 1, ..., n,
--                    i
--
-- where alfa[i,j] = a[i,j] is an element of the matrix A.
--
-- For details see:
-- D.Goldfarb, J.K.Reid. A practicable steepest-edge algorithm. Math.
-- Prog., 12, 1977, 361-71. */

void rsm_init_gamma(RSM *rsm)
{     ELEM *e;
      int n = rsm->n, j, k;
      double t;
      for (j = 1; j <= n; j++)
      {  k = rsm->indn[j];
         t = 1.0;
         for (e = rsm->A->col[k]; e != NULL; e = e->col)
            t += e->val * e->val;
         rsm->gamma[j] = t;
      }
      return;
}

/* eof */
