/* glprsm/rsm_init_dzeta.c */

/*----------------------------------------------------------------------
-- This file is a part of the GNU LPK package.
--
-- Copyright (C) 2000 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                    Department for Applied Informatics,
--                    Moscow Aviation Institute, Moscow, Russia.
--                    All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glprsm.h"

/*----------------------------------------------------------------------
-- rsm_init_dzeta - initialize weights (dual steepest edge).
--
-- *Synopsis*
--
-- #include "glprsm.h"
-- void rsm_init_dzeta(RSM *rsm);
--
-- *Description*
--
-- The routine rsm_init_dzeta initializes weights, which are used in
-- dual steepest edge technique, and stores them to the array dzeta.
-- It is assumed that B = I and N = -A, i.e. the initial simplex table
-- is the same as the constraint matrix A.
--
-- The routine uses the following formula:
--
--    dzeta[i] = 1 + SUM alfa[i,j]^2, i = 1, ..., m,
--                    j
--
-- where alfa[i,j] = a[i,j] is an element of the matrix A. */

void rsm_init_dzeta(RSM *rsm)
{     ELEM *e;
      int m = rsm->m, n = rsm->n, i, j, k;
      for (i = 1; i <= m; i++) rsm->dzeta[i] = 1.0;
      for (j = 1; j <= n; j++)
      {  k = rsm->indn[j];
         for (e = rsm->A->col[k]; e != NULL; e = e->col)
            rsm->dzeta[e->i] += e->val * e->val;
      }
      return;
}

/* eof */
