/* glprsm/rsm_check_bbar.c */

/*----------------------------------------------------------------------
-- This file is a part of the GNU LPK package.
--
-- Copyright (C) 2000 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                    Department for Applied Informatics,
--                    Moscow Aviation Institute, Moscow, Russia.
--                    All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include "glprsm.h"

/*----------------------------------------------------------------------
-- rsm_check_bbar - check current solution for numerical stability.
--
-- *Synopsis*
--
-- #include "glprsm.h"
-- int rsm_check_bbar(RSM *rsm, double tol);
--
-- *Description*
--
-- The rsm_check_bbar routine checks numerical stability of the current
-- basis solution. The solution is considered as stable if:
--
--    lB[i] - eps <= bbar[i] <= uB[i] + eps for all i = 1, 2, ..., m,
--
-- where bbar[i] is the current value of the basis variable xB[i],
-- lB[i] and uB[i] are lower and upper bounds of the variable xB[i],
-- eps = tol*max(|lB[i]|,1) or eps = tol*max(|uB[i]|,1) (depending on
-- what bound is checked), tol > 0 is the given tolerance.
--
-- Should note that for checking this routine uses bounds of variables
-- which may differ from the original bounds.
--
-- *Returns*
--
-- If the current basis solution is stable, the routine returns zero.
-- Otherwise the routine returns non-zero. */

int rsm_check_bbar(RSM *rsm, double tol)
{     int m = rsm->m, i, k;
      for (i = 1; i <= m; i++)
      {  k = rsm->indb[i]; /* x[k] = xB[i] */
         if (rsm->type[k] == LP_LO || rsm->type[k] == LP_DB ||
             rsm->type[k] == LP_FX)
         {  /* xB[i] has lower bound */
            if (rsm_check_dev(rsm->bbar[i], rsm->lb[k], tol) == -2)
               return 1;
         }
         if (rsm->type[k] == LP_UP || rsm->type[k] == LP_DB ||
             rsm->type[k] == LP_FX)
         {  /* xB[i] has upper bound */
            if (rsm_check_dev(rsm->bbar[i], rsm->ub[k], tol) == +2)
               return 1;
         }
      }
      return 0;
}

/* eof */
