/* glprfi/update_rfi.c */

/*----------------------------------------------------------------------
-- This file is a part of the GNU LPK package.
--
-- Copyright (C) 2000 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                    Department for Applied Informatics,
--                    Moscow Aviation Institute, Moscow, Russia.
--                    All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <assert.h>
#include <math.h>
#include <stddef.h>
#include "glpgel.h"
#include "glprfi.h"
#include "glpset.h"

/*----------------------------------------------------------------------
-- update_rfi - update RFI for adjacent basis matrix.
--
-- *Synopsis*
--
-- #include "glprfi.h"
-- int update_rfi(RFI *rfi, int p);
--
-- *Description*
--
-- The update_rfi routine recomputes RFI corresponding to the current
-- basis matrix B, so that the updated RFI will correspond to the new
-- (adjacent) basis matrix Bnew, where Bnew is a result of change p-th
-- column of B by other column.
--
-- Note that new p-th column of the basis matrix is passed implicitly to
-- this routine: the update_rfi routine assumes that the transformed new
-- p-th column was saved before by the rfi_ftran routine.
--
-- *Returns*
--
-- The update_rfi routine returns one of the following codes:
--
-- 0 - RFI has been successfully updated;
-- 1 - RFI became inaccurate;
-- 2 - RFI became too long.
--
-- If the returned code is non-zero, RFI should be rebuilt by the means
-- of the build_rfi routine. */

static RFI *_rfi = NULL;

static void func(int i, int p, double f)
{     /* add the next term to the eta-file H */
      RFI *rfi = _rfi;
      i = rfi->P->row[i], p = rfi->P->row[p];
      app_term(rfi->H, i, p, f);
      return;
}

#define iU(i) (rfi->P->col[i])
/* converts row number of V to row number of U */

#define jU(j) (rfi->Q->row[j])
/* converts column number of V to column number of U */

int update_rfi(RFI *rfi, int p)
{     ELEM *e;
      int m = rfi->m, i, k1, k2;
      double big, *v, drop = 1e-15;
      _rfi = rfi;
      if (!(1 <= p && p <= m))
         fault("update_rfi: invalid column number");
      if (!rfi->flag)
         fault("update_rfi: transformed column not ready");
      /* the current basis matrix is B = H*V; the new basis matrix is
         Bnew = H*Vnew, where Vnew differs from V only by p-th column,
         which is inv(H)*(new p-th column of B) (i.e. this is partially
         transformed column saved by the rfi_ftran routine) */
      v = rfi->col; /* new p-th column of V */
      /* compute maximal absolute value of elements of column v */
      big = 0.0;
      for (i = 1; i <= m; i++)
      {  double t = fabs(v[i]);
         if (big < t) big = t;
      }
      /* change p-th column of V by v = inv(H)*(new p-th column of B)
         ignoring relatively small elements to improve sparsity of V */
      clear_line(rfi->V, -p);
      for (i = 1; i <= m; i++)
      {  if (v[i] == 0.0 || fabs(v[i]) < drop * big) continue;
         new_elem(rfi->V, i, p, v[i]);
      }
      /* now Bnew = H*Vnew, however the matrix Unew = P*Vnew*Q is not
         upper triangular and has the following form:

              1     k1     k2   m
         1    x x x * x x x x x x
              . x x * x x x x x x
              . . x * x x x x x x
         k1   . . . * x x x x x x
              . . . * x x x x x x
              . . . * . x x x x x
              . . . * . . x x x x
         k2   . . . * . . . x x x
              . . . . . . . . x x
              . . . . . . . . . x

         elements of changed column of Unew that corresponds to the
         column v are marked by '*'; other non-zeros of Unew are marked
         by 'x'; k1 is number of changed column of Unew which can be
         determined by p and row permutation matrix P; k2 is maximal
         row number which contain non-zero element of changed column */
      k1 = jU(p), k2 = 0;
      for (e = rfi->V->col[p]; e != NULL; e = e->col)
         if (k2 < iU(e->i)) k2 = iU(e->i);
      /* if k1 > k2, the matrix Unew is singular since u[k1,k1] = 0 */
      if (k1 > k2) return 1;
      /* try to minimize the size of "bump" (i.e. the size of submatrix
         of U formed by rows and columns k1, k1+1, ..., k2) using Reid's
         technique; this affects only matrices P and Q, therefore the
         main equality Bnew = H*Vnew remains actual */
      if (rfi->reid && k1 < k2)
      {  min_bump(rfi->P, rfi->V, rfi->Q, &k1, &k2, rfi->rs, rfi->cs,
            (int *)rfi->col, (int *)rfi->work);
         assert(k1 <= k2);
      }
      /* now k1 <= k2; if k1 < k2, the matrix Unew should be transformed
         to the upper diagonal form by means of special version of
         gaussian elimination (Bartels & Golub or Forrest & Tomlin) that
         takes into account special structure of the matrix Unew; all
         elementary gaussian transformation are accumulated in the eta
         file H to provide the main equality Bnew = H*Vnew; in the case
         of k1 = k2 the matrix Unew is upper diagonal yet and therefore
         no transformations are needed; however since u[k1,k1] may be
         zero (as a result of Reid's structural transformation), the
         routine performing gaussian elimination is used not to perform
         it but only to check that u[k1,k1] is non-zero */
      {  int dum = 0, ret;
         switch (rfi->tech)
         {  case RFI_BG:
               /* Bartels & Golub technique */
               ret = gel_bg(rfi->P, rfi->V, rfi->Q, k1, k2, func, 0.35,
                  1e-5, &dum, rfi->work);
               break;
            case RFI_FT:
               /* Forrest & Tomlin technique */
               ret = gel_ft(rfi->P, rfi->V, rfi->Q, k1, k2, func, 1e-5,
                  &dum, rfi->work);
               break;
            default:
               assert(rfi->tech != rfi->tech);
         }
         /* if gaussian elimination routine returned detected that the
            matrix U is near to singular or inaccurate, RFI should be
            rebuilt */
         if (ret != 0) return 1;
      }
      /* if the total size of matrices H and V are too big, RFI should
         be rebuilt */
      {  int nzH = rfi->H->pool->count;
         int nzV = rfi->V->pool->count;
         if ((nzH + nzV) > 2 * (rfi->nzH0 + rfi->nzV0)) return 2;
      }
      /* the transformed column is no longer valid */
      rfi->flag = 0;
      /* RFI has been successfully updated */
      return 0;
}

/* eof */
