/* glpeta/ht_solve.c */

/*----------------------------------------------------------------------
-- This file is a part of the GNU LPK package.
--
-- Copyright (C) 2000 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                    Department for Applied Informatics,
--                    Moscow Aviation Institute, Moscow, Russia.
--                    All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glpeta.h"
#include "glpset.h"

/*----------------------------------------------------------------------
-- ht_solve - solve transposed linear system H'*x = b using eta-file.
--
-- *Synopsis*
--
-- #include "glpeta.h"
-- double *ht_solve(HFILE *eta, double x[]);
--
-- *Description*
--
-- The ht_solve routine solves the system H'*x = b, where H' is a matrix
-- transposed to the matrix H which is represented by the eta-file which
-- eta points to, x is dense vector of unknowns, b is dense vector of
-- right-hand sides. On entry the array x should contain elements of the
-- vector b in locations x[1], x[2], ..., x[n], where n is the order of
-- the system. On exit this array will contain the vector x in the same
-- locations.
--
-- *Returns*
--
-- The ht_solve routine returns a pointer to the array x. */

double *ht_solve(HFILE *eta, double x[])
{     HTERM *e;
      for (e = eta->tail; e != NULL; e = e->prev)
      {  if (e->j == e->i)
         {  if (e->val == 0.0)
               fault("ht_solve: singular matrix");
            x[e->j] /= e->val;
         }
         else
            x[e->j] -= e->val * x[e->i];
      }
      return x;
}

/* eof */
