/* glpeta/h_solve.c */

/*----------------------------------------------------------------------
-- This file is a part of the GNU LPK package.
--
-- Copyright (C) 2000 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                    Department for Applied Informatics,
--                    Moscow Aviation Institute, Moscow, Russia.
--                    All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glpeta.h"
#include "glpset.h"

/*----------------------------------------------------------------------
-- h_solve - solve linear system H*x = b using eta-file.
--
-- *Synopsis*
--
-- #include "glpeta.h"
-- double *h_solve(HFILE *eta, double x[]);
--
-- *Description*
--
-- The h_solve routine solves the system H*x = b, where H is the matrix
-- represented by the eta-file which eta points to, x is dense vector of
-- unknowns, b is dense vector of right-hand sides. On entry the array x
-- should contain elements of the vector b in locations x[1], x[2], ...,
-- x[n], where n is the order of the system. On exit this array will
-- contain the vector x in the same locations.
--
-- *Returns*
--
-- The h_solve routine returns a pointer to the array x. */

double *h_solve(HFILE *eta, double x[])
{     HTERM *e;
      for (e = eta->head; e != NULL; e = e->next)
      {  if (e->i == e->j)
         {  if (e->val == 0.0)
               fault("h_solve: singular matrix");
            x[e->i] /= e->val;
         }
         else
            x[e->i] -= e->val * x[e->j];
      }
      return x;
}

/* eof */
