/* glpeta/app_term.c */

/*----------------------------------------------------------------------
-- This file is a part of the GNU LPK package.
--
-- Copyright (C) 2000 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                    Department for Applied Informatics,
--                    Moscow Aviation Institute, Moscow, Russia.
--                    All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glpeta.h"
#include "glpset.h"

/*----------------------------------------------------------------------
-- app_term - append new eta-term to eta-file.
--
-- *Synopsis*
--
-- #include "glpeta.h"
-- HTERM *app_term(HFILE *eta, int i, int j, double val);
--
-- *Description*
--
-- The app_term appends new eta-term to the eta-file which eta points
-- to. The new eta-term corresponds to the square matrix H(t+1) (where
-- t is the total numbers of terms before appending) that differs from
-- the unity matrix only in one element h[i,j] = val.
--
-- Should note that if i = j and val = 1, or if i != j and val = 0, the
-- eta-term H(t+1) formally is unity matrix. Therefore in this two cases
-- the routine actually doesn't append such eta-term.
--
-- *Returns*
--
-- The app_term routine returns a pointer to the appended eta-term, or
-- NULL, if the eta-term was not appended because of two cases mentioned
-- above. */

HTERM *app_term(HFILE *eta, int i, int j, double val)
{     HTERM *e;
      if (!(1 <= i && i <= eta->n && 1 <= j && j <= eta->n))
         fault("app_term: row or column number out of range");
      if ((i == j && val == 1.0) || (i != j && val == 0.0))
         e = NULL;
      else
      {  e = get_atom(eta->pool);
         e->i = i; e->j = j; e->val = val;
         e->prev = eta->tail; e->next = NULL;
         if (eta->head == NULL)
            eta->head = e;
         else
            eta->tail->next = e;
         eta->tail = e;
      }
      return e;
}

/* eof */
