/* glpavl/insert_by_pos.c */

/*----------------------------------------------------------------------
-- This file is a part of the GNU LPK package.
--
-- Copyright (C) 2000 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                    Department for Applied Informatics,
--                    Moscow Aviation Institute, Moscow, Russia.
--                    All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glpavl.h"
#include "glpset.h"

/*----------------------------------------------------------------------
-- insert_by_pos - insert new node to given position of AVL-tree.
--
-- *Synopsis*
--
-- #include "glpavl.h"
-- AVLNODE *insert_by_pos(AVLTREE *tree, int pos);
--
-- *Description*
--
-- The insert_by_pos routine creates a new node and inserts this node
-- to the given position pos of the AVL-tree.
--
-- It is necessary that 1 <= pos <= N+1, where N = tree.size is the
-- total number of nodes in the AVL-tree before insertion.
--
-- If we renumber all nodes of the AVL-tree from 1 to N in that order
-- in which nodes are visited by the next_node routine, then a new node
-- will be placed between nodes that have numbers pos-1 and pos (if pos
-- is 1, a new node is placed before the first node, and if pos is N+1,
-- a new node is placed after the last node). So, after insertion a new
-- node will be placed in the given position pos.
--
-- After return the fileds type and link of the inserted node will
-- contain binary zeros. These fileds are available through the pointer
-- returned by the insert_by_key routine and may be used for arbitrary
-- purposes.
--
-- It is not recommended to use the insert_by_pos routine together with
-- other routines that suppose keyed nodes (insert_by_key, find_by_key,
-- next_by_key).
--
-- *Returns*
--
-- The insert_by_pos routine returns a pointer to the inserted node. */

AVLNODE *insert_by_pos(AVLTREE *tree, int pos)
{     AVLNODE *p, *q, *r;
      short int flag;
      if (!(1 <= pos && pos <= tree->size+1))
         fault ("insert_by_pos: invalid position");
      /* search for an appropriate place to insert the new node */
      p = NULL; q = tree->root;
      while (q != NULL)
      {  p = q;
         if (pos <= p->rank)
         {  flag = 0;
            q = p->left;
            p->rank++;
         }
         else
         {  flag = 1;
            q = p->right;
            pos -= p->rank;
         }
      }
      /* create and insert the new node */
      r = get_atom(tree->pool);
      r->key = NULL; r->type = 0; r->link = NULL;
      r->rank = 1; r->up = p;
      r->flag = (short int)(p == NULL ? 0 : flag);
      r->bal = 0; r->left = NULL; r->right = NULL;
      tree->size++;
      if (p == NULL)
         tree->root = r;
      else
         if (flag == 0) p->left = r; else p->right = r;
      /* go up to the root and correct all nodes affected by the
         insertion */
      while (p != NULL)
      {  if (flag == 0)
         {  /* the height of the left subtree of [p] is increased */
            if (p->bal > 0)
            {  p->bal = 0;
               break;
            }
            if (p->bal < 0)
            {  rotate_avl(tree, p);
               break;
            }
            p->bal = -1; flag = p->flag; p = p->up;
         }
         else
         {  /* the height of the right subtree of [p] is increased */
            if (p->bal < 0)
            {  p->bal = 0;
               break;
            }
            if (p->bal > 0)
            {  rotate_avl(tree, p);
               break;
            }
            p->bal = +1; flag = p->flag; p = p->up;
         }
      }
      /* if the root reached, the height of entire tree is increased */
      if (p == NULL) tree->height++;
      return r;
}

/* eof */
