/* glpapi/glp_set_bounds.c */

/*----------------------------------------------------------------------
-- This file is a part of the GNU LPK package.
--
-- Copyright (C) 2000 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                    Department for Applied Informatics,
--                    Moscow Aviation Institute, Moscow, Russia.
--                    All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glpapi.h"

/*----------------------------------------------------------------------
-- glp_set_bounds - set bounds of the current row or column.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- int glp_set_bounds(int what, int type, double lb, double ub);
--
-- *Description*
--
-- If what = GLP_ROW, the glp_set_bounds routine sets (changes) the
-- type and bounds of the current row.
--
-- If what = GLP_COL, the glp_set_bounds routine sets (changes) the
-- type and bounds of the current column.
--
-- Parameters type, lb, and ub specify respectively the type, the lower
-- bound, and the upper bound, which should be set for the current row
-- or column:
--
--     Type          Bounds              Note
--    ------    -----------------    -------------
--    GLP_FR    -inf <  x <  +inf    free variable
--    GLP_LO      lb <= x <  +inf    lower bound
--    GLP_UP    -inf <  x <=  ub     upper bound
--    GLP_DB      lb <= x <=  ub     double bound
--    GLP_FX            x  =  lb     fixed variable
--
-- where x is the auxiliary (in case of row) or structural (in case of
-- column) variable.
--
-- If the current row/column has no lower or/and upper bound, the
-- corresponding values of lb or/and ub are ignored. If the current
-- row/column is of GLP_FX type, the fixed value should be specified by
-- the parameter lb, and the parameter ub is ignored.
--
-- *Returns*
--
-- 0 - no errors;
-- 1 - the current row/column is undefined;
-- 2 - the parameter what is invalid, or the parameter type is invalid,
--     or lb >= ub (in case of type = GLP_DB only). */

int glp_set_bounds(int what, int type, double lb, double ub)
{     GLPITEM *item;
      switch (what)
      {  case GLP_ROW:
            item = glp->this_row;
            break;
         case GLP_COL:
            item = glp->this_col;
            break;
         default:
            return 2;
      }
      if (item == NULL) return 1;
      switch (type)
      {  case GLP_FR:
            item->type = GLP_FR;
            item->lb = item->ub = 0.0;
            break;
         case GLP_LO:
            item->type = GLP_LO;
            item->lb = lb, item->ub = 0.0;
            break;
         case GLP_UP:
            item->type = GLP_UP;
            item->lb = 0.0, item->ub = ub;
            break;
         case GLP_DB:
            item->type = GLP_DB;
            item->lb = lb, item->ub = ub;
            break;
         case GLP_FX:
            item->type = GLP_FX;
            item->lb = item->ub = lb;
            break;
         default:
            return 2;
      }
      return 0;
}

/* eof */
