/* glpmat/v_solve.c */

/*----------------------------------------------------------------------
-- This file is a part of the GNU LPK package.
--
-- Copyright (C) 2000 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                    Department for Applied Informatics,
--                    Moscow Aviation Institute, Moscow, Russia.
--                    All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glpmat.h"
#include "glpset.h"

/*----------------------------------------------------------------------
-- v_solve - solve implicit upper triangular system V*x = b.
--
-- *Synopsis*
--
-- #include "glpmat.h"
-- double *v_solve(PER *P, MAT *V, PER *Q, double x[], double work[]);
--
-- *Description*
--
-- The v_solve routine solves the system V*x = b, where U = P*V*Q is
-- upper triangular matrix, V is the given sparse matrix, P and Q are
-- permutation matrices, x is dense vector of unknowns, b is dense
-- vector of right-hand sides. The matrix U should be non-singular, i.e.
-- all its diagonal elements should be non-zeros. On entry the array x
-- should contain elements of the vector b in locations x[1], x[2], ...,
-- x[n], where n is the order of the system. On exit this array will
-- contain the vector x in the same locations. The auxiliary array work
-- should have at least 1+n elements, where n is order of the system
-- (if this parameter is NULL, the routine automatically allocates and
-- frees the working array).
--
-- *Returns*
--
-- The v_solve routine returns a pointer to the array x. */

double *v_solve(PER *P, MAT *V, PER *Q, double x[], double _work[])
{     ELEM *e;
      int i, j, flag = 1;
      double uii, *work = _work;
      if (V->m != V->n)
         fault("v_solve: matrix is not quadratic");
      if (!(P->n == V->m && Q->n == V->m))
         fault("v_solve: permutation matrices have invalid order");
      if (_work == NULL) work = ucalloc(1+V->m, sizeof(double));
      /* compute the vector b~ = P*b */
      for (i = 1; i <= V->m; i++) work[i] = x[i];
      per_vec(x, P, work);
      /* solve the system U*x~ = b~, where U = P*V*Q */
      for (i = V->m; i >= 1; i--)
      {  /* flag = 1 means that x~[i+1] = ... = x~[n] = 0; therefore if
            b~[i] = 0 then x~[i] = 0 */
         if (flag && x[i] == 0.0) continue;
         uii = 0.0;
         for (e = V->row[P->row[i]]; e != NULL; e = e->row)
         {  j = Q->row[e->j];
            if (j < i)
               fault("v_solve: matrix P*V*Q is not upper triangular");
            if (j == i)
               uii = e->val;
            else
               x[i] -= e->val * x[j];
         }
         if (uii == 0.0)
            fault("v_solve: matrix is singular");
         x[i] /= uii;
         if (x[i] != 0.0) flag = 0.0;
      }
      /* compute the vector x = Q*x~ */
      for (i = 1; i <= V->m; i++) work[i] = x[i];
      per_vec(x, Q, work);
      if (_work == NULL) ufree(work);
      return x;
}

/* eof */
