/* glpmat/submatrix.c */

/*----------------------------------------------------------------------
-- This file is a part of the GNU LPK package.
--
-- Copyright (C) 2000 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                    Department for Applied Informatics,
--                    Moscow Aviation Institute, Moscow, Russia.
--                    All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glpmat.h"
#include "glpset.h"

/*----------------------------------------------------------------------
-- submatrix - copy submatrix.
--
-- *Synopsis*
--
-- #include "glpmat.h"
-- MAT *submatrix(MAT *B, MAT *A, int i1, int i2, int j1, int j2);
--
-- *Description*
--
-- The submatrix routine copies a submatrix of the sparse matrix A to
-- the sparse matrix B. The copied submatrix consists of rows i1 to i2
-- and column j1 to j2 of the matrix A. The matrix B should have m rows
-- and n columns, where m = i2 - i1 + 1 and n = j2 - j1 + 1. The matrix
-- A remains unchanged.
--
-- *Returns*
--
-- The submatrix routine returns a pointer to the matrix B. */

MAT *submatrix(MAT *B, MAT *A, int i1, int i2, int j1, int j2)
{     ELEM *e;
      int i, j;
      if (!(1 <= i1 && i1 <= i2 && i2 <= A->m))
         fault("submatrix: invalid row numbers");
      if (!(1 <= j1 && j1 <= j2 && j2 <= A->n))
         fault("submatrix: invalid column numbers");
      if (!(B->m == i2 - i1 + 1 && B->n == j2 - j1 + 1))
         fault("submatrix: invalid dimension of target matrix");
      clear_mat(B);
      for (i = i1; i <= i2; i++)
      {  for (e = A->row[i]; e != NULL; e = e->row)
         {  j = e->j;
            if (j1 <= j && j <= j2)
               new_elem(B, i - i1 + 1, j - j1 + 1, e->val);
         }
      }
      return B;
}

/* eof */
