/* glpmat/show_mat.c */

/*----------------------------------------------------------------------
-- This file is a part of the GNU LPK package.
--
-- Copyright (C) 2000 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                    Department for Applied Informatics,
--                    Moscow Aviation Institute, Moscow, Russia.
--                    All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glpimg.h"
#include "glpmat.h"
#include "glpset.h"

/*----------------------------------------------------------------------
-- show_mat - output matrix pattern as raster image.
--
-- *Synopsis*
--
-- #include "glpmat.h"
-- int show_mat(MAT *A, int type, char *fname);
--
-- *Description*
--
-- The show_mat routine creates pattern of the matrix A in the form of
-- raster image and writes this image using default format to the file,
-- whose name is the character string fname.
--
-- The parameter type defines the type of the image.
--
-- If type is 0, the background is white, matrix elements are black.
--
-- If type is 1, the following colors are used:
--
-- gray:    there is no element in the position a[i,j] (background)
-- white:   a[i,j] > 0
-- cyan:    a[i,j] < 0
-- green:   a[i,j] = 0
-- red:     there are several elements in the position a[i,j].
--
-- In case of monochrome pattern the amount of memory is (m*n)/8 bytes,
-- and in case of color pattern the amount of memory is (m*n)/2 bytes,
-- where m and n are respectively number of rows and number of columns
-- of the matrix A.
--
-- *Returns*
--
-- If the operation completed successfully, the routine returns zero.
-- Otherwise the routine returns non-zero. */

int show_mat(MAT *A, int type, char *fname)
{     IMG *bmp;
      ELEM *e;
      int m = A->m, n = A->n, i, j, ret;
      print("show_mat: writing matrix pattern to `%s'...", fname);
      if (type == 0)
      {  /* monochrome pattern */
         bmp = create_img(IMG_2, n, m);
         for (i = 1; i <= m; i++)
         {  for (j = 1; j <= n; j++)
               set_imgval(bmp, j-1, i-1, IMG_WHITE);
            for (e = A->row[i]; e != NULL; e = e->row)
            {  j = e->j;
               set_imgval(bmp, j-1, i-1, IMG_BLACK);
            }
         }
         ret = save_img(bmp, fname);
         delete_img(bmp);
      }
      else if (type == 1)
      {  /* color pattern */
         bmp = create_img(IMG_16, n, m);
         for (i = 1; i <= m; i++)
         {  for (j = 1; j <= n; j++)
               set_imgval(bmp, j-1, i-1, IMG_DARKGRAY);
            for (e = A->row[i]; e != NULL; e = e->row)
            {  j = e->j;
               if (get_imgval(bmp, j-1, i-1) != IMG_DARKGRAY)
                  set_imgval(bmp, j-1, i-1, IMG_LIGHTRED);
               else if (e->val > 0.0)
                  set_imgval(bmp, j-1, i-1, IMG_WHITE);
               else if (e->val < 0.0)
                  set_imgval(bmp, j-1, i-1, IMG_LIGHTCYAN);
               else /* e->val == 0.0 */
                  set_imgval(bmp, j-1, i-1, IMG_LIGHTGREEN);
            }
         }
         ret = save_img(bmp, fname);
         delete_img(bmp);
      }
      else
         fault("show_mat: invalid image type");
      return ret;
}

/* eof */
