/* glpimg/save_as_tiff.c */

/*----------------------------------------------------------------------
-- This file is a part of the GNU LPK package.
--
-- Copyright (C) 2000 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                    Department for Applied Informatics,
--                    Moscow Aviation Institute, Moscow, Russia.
--                    All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <assert.h>
#include <errno.h>
#include <stdio.h>
#include <string.h>
#include "glpimg.h"
#include "glpset.h"

/*----------------------------------------------------------------------
-- save_as_tiff - write raster image using TIFF format.
--
-- *Synopsis*
--
-- #include "glpimg.h"
-- int save_as_tiff(IMG *img, char *fname);
--
-- *Description*
--
-- The save_as_tiff routine writes the raster image using uncompressed
-- TIFF format to the binary file, whose name is the character string
-- fname.
--
-- *Returns*
--
-- If the operation completed successfully, the routine returns zero.
-- Otherwise the routine returns non-zero. */

static void put_byte(FILE *fp, int val)
{     unsigned char b = (unsigned char)val;
      fwrite(&b, sizeof(char), 1, fp);
      return;
}

static void put_word(FILE *fp, int val) /* big endian */
{     put_byte(fp, val);
      put_byte(fp, val >> 8);
      return;
}

static void put_dword(FILE *fp, int val) /* big endian */
{     put_word(fp, val);
      put_word(fp, val >> 16);
      return;
}

int save_as_tiff(IMG *img, char *fname)
{     FILE *fp;
      int loc_0111, loc_0140, loc_011A, loc_011B, k, x, y, temp;
      fp = fopen(fname, "wb");
      if (fp == NULL)
      {  error("save_as_tiff: can't create `%s' - %s", fname,
            strerror(errno));
         goto fail;
      }
      /* 8-byte image file header */
      /* byte ordering used within the file = 0x4949 (big endian) */
      put_byte(fp, 0x49);
      put_byte(fp, 0x49);
      /* TIFF version number = 42 = 0x2A */
      put_word(fp, 0x2A);
      /* offset (in bytes) of the first (and the only) Image File
         Directory (IFD); it follows the image file header */
      put_dword(fp, 8);
      /*** IFD starts here ***/
      /* number of 12-byte IFD entries */
      put_word(fp, 17);
      /* subfile type */
      put_word(fp, 0x00FE);
      put_word(fp, 4 /* long */);
      put_dword(fp, 1);
      put_dword(fp, 0);
      /* image width (in pixels) */
      put_word(fp, 0x0100);
      put_word(fp, 3 /* short */);
      put_dword(fp, 1);
      put_dword(fp, img->xsize);
      /* image heigth (in pixels) */
      put_word(fp, 0x0101);
      put_word(fp, 3 /* short */);
      put_dword(fp, 1);
      put_dword(fp, img->ysize);
      /* bits per sample */
      put_word(fp, 0x0102);
      put_word(fp, 3 /* short */);
      put_dword(fp, 1);
      switch (img->type)
      {  case IMG_2:    put_dword(fp, 1); break;
         case IMG_4:    put_dword(fp, 4); break;
         case IMG_16:   put_dword(fp, 4); break;
         case IMG_256:  put_dword(fp, 8); break;
         default: assert(img->type != img->type);
      }
      /* compression */
      put_word(fp, 0x0103);
      put_word(fp, 3 /* short */);
      put_dword(fp, 1);
      put_dword(fp, 1 /* no compression */);
      /* photometric interpretation */
      put_word(fp, 0x0106);
      put_word(fp, 3 /* short */);
      put_dword(fp, 1);
      put_dword(fp, 3 /* color palette used */);
      /* fill order */
      put_word(fp, 0x010A);
      put_word(fp, 3 /* short */);
      put_dword(fp, 1);
      put_dword(fp, 1 /* most significant fill first */);
      /* strip offset */
      put_word(fp, 0x0111);
      put_word(fp, 4 /* long */);
      put_dword(fp, 1);
      loc_0111 = ftell(fp);
      put_dword(fp, 0 /* will be stored later */);
      /* orientation */
      put_word(fp, 0x0112);
      put_word(fp, 3 /* short */);
      put_dword(fp, 1);
      put_dword(fp, 1 /* left upper corner */);
      /* sample per pixel */
      put_word(fp, 0x0115);
      put_word(fp, 3 /* short */);
      put_dword(fp, 1);
      put_dword(fp, 1 /* if color palette used */);
      /* rows per strip */
      put_word(fp, 0x0116);
      put_word(fp, 3 /* short */);
      put_dword(fp, 1);
      put_dword(fp, img->ysize);
      /* strip byte counts */
      put_word(fp, 0x0117);
      put_word(fp, 4 /* long */);
      put_dword(fp, 1);
      switch (img->type)
      {  case IMG_2:
            put_dword(fp, ((img->xsize + 7) / 8) * img->ysize);
            break;
         case IMG_4:
            put_dword(fp, ((img->xsize + 1) / 2) * img->ysize);
            break;
         case IMG_16:
            put_dword(fp, ((img->xsize + 1) / 2) * img->ysize);
            break;
         case IMG_256:
            put_dword(fp, img->xsize * img->ysize);
            break;
         default:
            assert(img->type != img->type);
      }
      /* horizontal resolution */
      put_word(fp, 0x011A);
      put_word(fp, 5 /* ratio */);
      put_dword(fp, 1);
      loc_011A = ftell(fp);
      put_dword(fp, 0 /* will be stored later */);
      /* vertical resolution */
      put_word(fp, 0x011B);
      put_word(fp, 5 /* ratio */);
      put_dword(fp, 1);
      loc_011B = ftell(fp);
      put_dword(fp, 0 /* will be stored later */);
      /* planar configuration */
      put_word(fp, 0x011C);
      put_word(fp, 3 /* short */);
      put_dword(fp, 1);
      put_dword(fp, 1 /* contiguous */);
      /* resolution unit */
      put_word(fp, 0x0128);
      put_word(fp, 3 /* short */);
      put_dword(fp, 1);
      put_dword(fp, 3 /* centimeter */);
      /* color map */
      put_word(fp, 0x0140);
      put_word(fp, 3 /* short */);
      switch (img->type)
      {  case IMG_2:    put_dword(fp, 3 * 2);   break;
         case IMG_4:    put_dword(fp, 3 * 16);  break;
         case IMG_16:   put_dword(fp, 3 * 16);  break;
         case IMG_256:  put_dword(fp, 3 * 256); break;
         default: assert(img->type != img->type);
      }
      loc_0140 = ftell(fp);
      put_dword(fp, 0 /* will be stored later */);
      /*** end of IFD ***/
      /* align on dword boundary */
      while (ftell(fp) % 4) put_byte(fp, '?');
      /* x resolution */
      temp = ftell(fp);
      fseek(fp, loc_011A, SEEK_SET);
      put_dword(fp, temp);
      fseek(fp, 0, SEEK_END);
      put_dword(fp, img->xres);
      put_dword(fp, 100);
      /* y resolution */
      temp = ftell(fp);
      fseek(fp, loc_011B, SEEK_SET);
      put_dword(fp, temp);
      fseek(fp, 0, SEEK_END);
      put_dword(fp, img->yres);
      put_dword(fp, 100);
      /* color map */
      temp = ftell(fp);
      fseek(fp, loc_0140, SEEK_SET);
      put_dword(fp, temp);
      fseek(fp, 0, SEEK_END);
      switch (img->type)
      {  case IMG_2:
            for (k = 0; k < 2; k++)
            {  put_byte(fp, img->rgb[k].red);
               put_byte(fp, img->rgb[k].red);
            }
            for (k = 0; k < 2; k++)
            {  put_byte(fp, img->rgb[k].green);
               put_byte(fp, img->rgb[k].green);
            }
            for (k = 0; k < 2; k++)
            {  put_byte(fp, img->rgb[k].blue);
               put_byte(fp, img->rgb[k].blue);
            }
            break;
         case IMG_4:
            for (k = 0; k < 16; k++)
            {  put_byte(fp, img->rgb[k & 3].red);
               put_byte(fp, img->rgb[k & 3].red);
            }
            for (k = 0; k < 16; k++)
            {  put_byte(fp, img->rgb[k & 3].green);
               put_byte(fp, img->rgb[k & 3].green);
            }
            for (k = 0; k < 16; k++)
            {  put_byte(fp, img->rgb[k & 3].blue);
               put_byte(fp, img->rgb[k & 3].blue);
            }
            break;
         case IMG_16:
            for (k = 0; k < 16; k++)
            {  put_byte(fp, img->rgb[k].red);
               put_byte(fp, img->rgb[k].red);
            }
            for (k = 0; k < 16; k++)
            {  put_byte(fp, img->rgb[k].green);
               put_byte(fp, img->rgb[k].green);
            }
            for (k = 0; k < 16; k++)
            {  put_byte(fp, img->rgb[k].blue);
               put_byte(fp, img->rgb[k].blue);
            }
            break;
         case IMG_256:
            for (k = 0; k < 256; k++)
            {  put_byte(fp, img->rgb[k].red);
               put_byte(fp, img->rgb[k].red);
            }
            for (k = 0; k < 256; k++)
            {  put_byte(fp, img->rgb[k].green);
               put_byte(fp, img->rgb[k].green);
            }
            for (k = 0; k < 256; k++)
            {  put_byte(fp, img->rgb[k].blue);
               put_byte(fp, img->rgb[k].blue);
            }
            break;
         default:
            assert(img->type != img->type);
      }
      /* strip data (pixels) */
      temp = ftell(fp);
      fseek(fp, loc_0111, SEEK_SET);
      put_dword(fp, temp);
      fseek(fp, 0, SEEK_END);
      switch (img->type)
      {  case IMG_2:
            for (y = 0; y < img->ysize; y++)
            {  temp = 0, k = 0;
               for (x = 0; x < img->xsize; x++)
               {  temp = (temp << 1) | get_imgval(img, x, y), k++;
                  if (k == 8) put_byte(fp, temp), temp = 0, k = 0;
               }
               if (k > 0)
               {  while (k < 8) temp <<= 1, k++;
                  put_byte(fp, temp);
               }
            }
            break;
         case IMG_4:
         case IMG_16:
            for (y = 0; y < img->ysize; y++)
            {  temp = 0, k = 0;
               for (x = 0; x < img->xsize; x++)
               {  temp = (temp << 4) | get_imgval(img, x, y), k += 4;
                  if (k == 8) put_byte(fp, temp), temp = 0, k = 0;
               }
               if (k > 0)
               {  while (k < 8) temp <<= 4, k += 4;
                  put_byte(fp, temp);
               }
            }
            break;
         case IMG_256:
            for (y = 0; y < img->ysize; y++)
            {  for (x = 0; x < img->xsize; x++)
                  put_byte(fp, get_imgval(img, x, y));
            }
            break;
         default:
            assert(img->type != img->type);
      }
      fflush(fp);
      if (ferror(fp))
      {  error("save_as_tiff: can't write to `%s' - %s", fname,
            strerror(errno));
         goto fail;
      }
      fclose(fp);
      return 0;
fail: if (fp != NULL) fclose(fp);
      return 1;
}

/* eof */
