/* glpimg/reset_rgb.c */

/*----------------------------------------------------------------------
-- This file is a part of the GNU LPK package.
--
-- Copyright (C) 2000 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                    Department for Applied Informatics,
--                    Moscow Aviation Institute, Moscow, Russia.
--                    All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <assert.h>
#include "glpimg.h"

/*----------------------------------------------------------------------
-- reset_rgb - initialize color map table by default spectrum.
--
-- *Synopsis*
--
-- #include "glpimg.h"
-- void reset_rgb(IMG *img);
--
-- *Description*
--
-- The reset_rgb routine sets all entries of the color map table for
-- the raster image to default standard values.
--
-- Normally, the video BIOS programs the video DAC registers with a
-- default spectrum of color values. Registers 00H through 0FH contain
-- the default gamut of CGA-compatible colors. Registers 10H through
-- 1FH contain a gray scale of gradually increasing intensity. The next
-- 216 registers (20H through F7H) contain three groups of 72 colors,
-- with the first group (registers 20H through 67H) at high intensity,
-- the second (registers 68H through AFH) at an intermediate intensity,
-- and the third (registers B0H through F7H) at low intensity. Each
-- 72-color group is made up of three ranges of colors of decreasing
-- saturation (increasing whiteness); each range varies smoothly in hue
-- from blue to red to green. */

static RGB dac[256] =
{     /* 00-0F: CGA-compatible default colors */
      { 0x00, 0x00, 0x00 }, { 0x00, 0x00, 0xA8 },
      { 0x00, 0xA8, 0x00 }, { 0x00, 0xA8, 0xA8 },
      { 0xA8, 0x00, 0x00 }, { 0xA8, 0x00, 0xA8 },
      { 0xA8, 0x54, 0x00 }, { 0xA8, 0xA8, 0xA8 },
      { 0x54, 0x54, 0x54 }, { 0x54, 0x54, 0xFF },
      { 0x54, 0xFF, 0x54 }, { 0x54, 0xFF, 0xFF },
      { 0xFF, 0x54, 0x54 }, { 0xFF, 0x54, 0xFF },
      { 0xFF, 0xFF, 0x54 }, { 0xFF, 0xFF, 0xFF },
      /* 10-1F: gray scale */
      { 0x00, 0x00, 0x00 }, { 0x14, 0x14, 0x14 },
      { 0x20, 0x20, 0x20 }, { 0x2C, 0x2C, 0x2C },
      { 0x38, 0x38, 0x38 }, { 0x44, 0x44, 0x44 },
      { 0x50, 0x50, 0x50 }, { 0x60, 0x60, 0x60 },
      { 0x70, 0x70, 0x70 }, { 0x80, 0x80, 0x80 },
      { 0x90, 0x90, 0x90 }, { 0xA0, 0xA0, 0xA0 },
      { 0xB4, 0xB4, 0xB4 }, { 0xC8, 0xC8, 0xC8 },
      { 0xE0, 0xE0, 0xE0 }, { 0xFF, 0xFF, 0xFF },
      /* 20-37: b-r-g-b (high intensity, high saturation) */
      { 0x00, 0x00, 0xFF }, { 0x40, 0x00, 0xFF },
      { 0x7C, 0x00, 0xFF }, { 0xBC, 0x00, 0xFF },
      { 0xFF, 0x00, 0xFF }, { 0xFF, 0x00, 0xBC },
      { 0xFF, 0x00, 0x7C }, { 0xFF, 0x00, 0x40 },
      { 0xFF, 0x00, 0x00 }, { 0xFF, 0x40, 0x00 },
      { 0xFF, 0x7C, 0x00 }, { 0xFF, 0xBC, 0x00 },
      { 0xFF, 0xFF, 0x00 }, { 0xBC, 0xFF, 0x00 },
      { 0x7C, 0xFF, 0x00 }, { 0x40, 0xFF, 0x00 },
      { 0x00, 0xFF, 0x00 }, { 0x00, 0xFF, 0x40 },
      { 0x00, 0xFF, 0x7C }, { 0x00, 0xFF, 0xBC },
      { 0x00, 0xFF, 0xFF }, { 0x00, 0xBC, 0xFF },
      { 0x00, 0x7C, 0xFF }, { 0x00, 0x40, 0xFF },
      /* 38-4F: b-r-g-b (high intensity, moderate saturation) */
      { 0x7C, 0x7C, 0xFF }, { 0x9C, 0x7C, 0xFF },
      { 0xBC, 0x7C, 0xFF }, { 0xDC, 0x7C, 0xFF },
      { 0xFF, 0x7C, 0xFF }, { 0xFF, 0x7C, 0xDC },
      { 0xFF, 0x7C, 0xBC }, { 0xFF, 0x7C, 0x9C },
      { 0xFF, 0x7C, 0x7C }, { 0xFF, 0x9C, 0x7C },
      { 0xFF, 0xBC, 0x7C }, { 0xFF, 0xDC, 0x7C },
      { 0xFF, 0xFF, 0x7C }, { 0xDC, 0xFF, 0x7C },
      { 0xBC, 0xFF, 0x7C }, { 0x9C, 0xFF, 0x7C },
      { 0x7C, 0xFF, 0x7C }, { 0x7C, 0xFF, 0x9C },
      { 0x7C, 0xFF, 0xBC }, { 0x7C, 0xFF, 0xDC },
      { 0x7C, 0xFF, 0xFF }, { 0x7C, 0xDC, 0xFF },
      { 0x7C, 0xBC, 0xFF }, { 0x7C, 0x9C, 0xFF },
      /* 50-67: b-r-g-b (high intensity, low saturation) */
      { 0xB4, 0xB4, 0xFF }, { 0xC4, 0xB4, 0xFF },
      { 0xD8, 0xB4, 0xFF }, { 0xE8, 0xB4, 0xFF },
      { 0xFF, 0xB4, 0xFF }, { 0xFF, 0xB4, 0xE8 },
      { 0xFF, 0xB4, 0xD8 }, { 0xFF, 0xB4, 0xC4 },
      { 0xFF, 0xB4, 0xB4 }, { 0xFF, 0xC4, 0xB4 },
      { 0xFF, 0xD8, 0xB4 }, { 0xFF, 0xE8, 0xB4 },
      { 0xFF, 0xFF, 0xB4 }, { 0xE8, 0xFF, 0xB4 },
      { 0xD8, 0xFF, 0xB4 }, { 0xC4, 0xFF, 0xB4 },
      { 0xB4, 0xFF, 0xB4 }, { 0xB4, 0xFF, 0xC4 },
      { 0xB4, 0xFF, 0xD8 }, { 0xB4, 0xFF, 0xE8 },
      { 0xB4, 0xFF, 0xFF }, { 0xB4, 0xE8, 0xFF },
      { 0xB4, 0xD8, 0xFF }, { 0xB4, 0xC4, 0xFF },
      /* 68-7F: b-r-g-b (moderate intensity, high saturation) */
      { 0x00, 0x00, 0x70 }, { 0x1C, 0x00, 0x70 },
      { 0x38, 0x00, 0x70 }, { 0x54, 0x00, 0x70 },
      { 0x70, 0x00, 0x70 }, { 0x70, 0x00, 0x54 },
      { 0x70, 0x00, 0x38 }, { 0x70, 0x00, 0x1C },
      { 0x70, 0x00, 0x00 }, { 0x70, 0x1C, 0x00 },
      { 0x70, 0x38, 0x00 }, { 0x70, 0x54, 0x00 },
      { 0x70, 0x70, 0x00 }, { 0x54, 0x70, 0x00 },
      { 0x38, 0x70, 0x00 }, { 0x1C, 0x70, 0x00 },
      { 0x00, 0x70, 0x00 }, { 0x00, 0x70, 0x1C },
      { 0x00, 0x70, 0x38 }, { 0x00, 0x70, 0x54 },
      { 0x00, 0x70, 0x70 }, { 0x00, 0x54, 0x70 },
      { 0x00, 0x38, 0x70 }, { 0x00, 0x1C, 0x70 },
      /* 80-97: b-r-g-b (moderate intensity, moderate saturation) */
      { 0x38, 0x38, 0x70 }, { 0x44, 0x38, 0x70 },
      { 0x54, 0x38, 0x70 }, { 0x60, 0x38, 0x70 },
      { 0x70, 0x38, 0x70 }, { 0x70, 0x38, 0x60 },
      { 0x70, 0x38, 0x54 }, { 0x70, 0x38, 0x44 },
      { 0x70, 0x38, 0x38 }, { 0x70, 0x44, 0x38 },
      { 0x70, 0x54, 0x38 }, { 0x70, 0x60, 0x38 },
      { 0x70, 0x70, 0x38 }, { 0x60, 0x70, 0x38 },
      { 0x54, 0x70, 0x38 }, { 0x44, 0x70, 0x38 },
      { 0x38, 0x70, 0x38 }, { 0x38, 0x70, 0x44 },
      { 0x38, 0x70, 0x54 }, { 0x38, 0x70, 0x60 },
      { 0x38, 0x70, 0x70 }, { 0x38, 0x60, 0x70 },
      { 0x38, 0x54, 0x70 }, { 0x38, 0x44, 0x70 },
      /* 98-AF: b-r-g-b (moderate intensity, low saturation) */
      { 0x50, 0x50, 0x70 }, { 0x58, 0x50, 0x70 },
      { 0x60, 0x50, 0x70 }, { 0x68, 0x50, 0x70 },
      { 0x70, 0x50, 0x70 }, { 0x70, 0x50, 0x68 },
      { 0x70, 0x50, 0x60 }, { 0x70, 0x50, 0x58 },
      { 0x70, 0x50, 0x50 }, { 0x70, 0x58, 0x50 },
      { 0x70, 0x60, 0x50 }, { 0x70, 0x68, 0x50 },
      { 0x70, 0x70, 0x50 }, { 0x68, 0x70, 0x50 },
      { 0x60, 0x70, 0x50 }, { 0x58, 0x70, 0x50 },
      { 0x50, 0x70, 0x50 }, { 0x50, 0x70, 0x58 },
      { 0x50, 0x70, 0x60 }, { 0x50, 0x70, 0x68 },
      { 0x50, 0x70, 0x70 }, { 0x50, 0x68, 0x70 },
      { 0x50, 0x60, 0x70 }, { 0x50, 0x58, 0x70 },
      /* B0-C7: b-r-g-b (low intensity, high saturation) */
      { 0x00, 0x00, 0x40 }, { 0x10, 0x00, 0x40 },
      { 0x20, 0x00, 0x40 }, { 0x30, 0x00, 0x40 },
      { 0x40, 0x00, 0x40 }, { 0x40, 0x00, 0x30 },
      { 0x40, 0x00, 0x20 }, { 0x40, 0x00, 0x10 },
      { 0x40, 0x00, 0x00 }, { 0x40, 0x10, 0x00 },
      { 0x40, 0x20, 0x00 }, { 0x40, 0x30, 0x00 },
      { 0x40, 0x40, 0x00 }, { 0x30, 0x40, 0x00 },
      { 0x20, 0x40, 0x00 }, { 0x10, 0x40, 0x00 },
      { 0x00, 0x40, 0x00 }, { 0x00, 0x40, 0x10 },
      { 0x00, 0x40, 0x20 }, { 0x00, 0x40, 0x30 },
      { 0x00, 0x40, 0x40 }, { 0x00, 0x30, 0x40 },
      { 0x00, 0x20, 0x40 }, { 0x00, 0x10, 0x40 },
      /* C8-DF: b-r-g-b (low intensity, moderate saturation) */
      { 0x20, 0x20, 0x40 }, { 0x28, 0x20, 0x40 },
      { 0x30, 0x20, 0x40 }, { 0x38, 0x20, 0x40 },
      { 0x40, 0x20, 0x40 }, { 0x40, 0x20, 0x38 },
      { 0x40, 0x20, 0x30 }, { 0x40, 0x20, 0x28 },
      { 0x40, 0x20, 0x20 }, { 0x40, 0x28, 0x20 },
      { 0x40, 0x30, 0x20 }, { 0x40, 0x38, 0x20 },
      { 0x40, 0x40, 0x20 }, { 0x38, 0x40, 0x20 },
      { 0x30, 0x40, 0x20 }, { 0x28, 0x40, 0x20 },
      { 0x20, 0x40, 0x20 }, { 0x20, 0x40, 0x28 },
      { 0x20, 0x40, 0x30 }, { 0x20, 0x40, 0x38 },
      { 0x20, 0x40, 0x40 }, { 0x20, 0x38, 0x40 },
      { 0x20, 0x30, 0x40 }, { 0x20, 0x28, 0x40 },
      /* E0-F7: b-r-g-b (low intensity, low saturation) */
      { 0x2C, 0x2C, 0x40 }, { 0x30, 0x2C, 0x40 },
      { 0x34, 0x2C, 0x40 }, { 0x3C, 0x2C, 0x40 },
      { 0x40, 0x2C, 0x40 }, { 0x40, 0x2C, 0x3C },
      { 0x40, 0x2C, 0x34 }, { 0x40, 0x2C, 0x30 },
      { 0x40, 0x2C, 0x2C }, { 0x40, 0x30, 0x2C },
      { 0x40, 0x34, 0x2C }, { 0x40, 0x3C, 0x2C },
      { 0x40, 0x40, 0x2C }, { 0x3C, 0x40, 0x2C },
      { 0x34, 0x40, 0x2C }, { 0x30, 0x40, 0x2C },
      { 0x2C, 0x40, 0x2C }, { 0x2C, 0x40, 0x30 },
      { 0x2C, 0x40, 0x34 }, { 0x2C, 0x40, 0x3C },
      { 0x2C, 0x40, 0x40 }, { 0x2C, 0x3C, 0x40 },
      { 0x2C, 0x34, 0x40 }, { 0x2C, 0x30, 0x40 },
      /* F8-FF: black */
      { 0x00, 0x00, 0x00 }, { 0x00, 0x00, 0x00 },
      { 0x00, 0x00, 0x00 }, { 0x00, 0x00, 0x00 },
      { 0x00, 0x00, 0x00 }, { 0x00, 0x00, 0x00 },
      { 0x00, 0x00, 0x00 }, { 0x00, 0x00, 0x00 },
};

void reset_rgb(IMG *img)
{     int c;
      switch (img->type)
      {  case IMG_2: /* 2 colors (1 bit per pixel) */
            img->rgb[0] = dac[0x00]; /* black */
            img->rgb[1] = dac[0x0F]; /* white */
            break;
         case IMG_4: /* 4 colors (2 bits per pixel) */
            img->rgb[0] = dac[0x00]; /* black */
            img->rgb[1] = dac[0x0B]; /* cyan */
            img->rgb[2] = dac[0x0D]; /* magenta */
            img->rgb[3] = dac[0x0F]; /* white */
            break;
         case IMG_16: /* 16 colors (4 bits per pixel) */
            for (c = 0x00; c <= 0x0F; c++) img->rgb[c] = dac[c];
            break;
         case IMG_256: /* 256 colors (8 bits per pixel) */
            for (c = 0x00; c <= 0xFF; c++) img->rgb[c] = dac[c];
            break;
         default:
            assert(img->type != img->type);
      }
      return;
}

/* eof */