package html;

import java.util.Hashtable;
import java.net.URL;
import java.io.IOException;
import java.awt.Image;
import java.awt.Color;

public class HtmlDocument
{
  private IntVector codes = new IntVector();
  private StringVector strings = new StringVector();
  private ImageVector images = new ImageVector();
  private String title = null;
  private String location = null;
  private String start = null;
  private Color bgColor = null;
  private Color textColor = null;
  private Color linkColor = null;
  private URL base = null;

  private String endOfText = null;

  public HtmlDocument(URL url) throws IOException
  {
    base = url;
    fillLocationAndStart(url);
    parse(new HtmlTokenizer(url.openStream()));
    codes.trimToSize();
    strings.trimToSize();
    images.trimToSize();
  }

  public String getTitle()
  {
    return title;
  }

  public String getLocation()
  {
    return location;
  }

  public String getStart()
  {
    return start;
  }

  public String getURLString()
  {
    if (start == null)
      return location;
    else
      return location + "#" + start;
  }

  synchronized protected void draw(HtmlPager p)
  {
    p.setBase(base);
    p.setColors(bgColor, textColor, linkColor);
    codes.reset();
    strings.reset();
    images.reset();
    while (codes.hasMoreElements())
    {
      int code = codes.nextElement();
      if (code >= 0)
        drawOpen(code, p);
      else
        drawClose(-code, p);
    }
    p.finish();
  }

  private void fillLocationAndStart(URL url)
  {
    String protocol = url.getProtocol();
    String host = url.getHost();
    int port = url.getPort();
    String file = url.getFile();

    while (file.startsWith("/"))
      file = file.substring(1);

    if (port < 0)
      location = protocol + "://" + host + "/" + file;
    else
      location = protocol + "://" + host + ":" + port + "/" + file;

    start = url.getRef();
  }

  private void parse(HtmlTokenizer tokenizer)
  {
    while (!tokenizer.eof())
    {
      if (!parseTextItem(tokenizer) &&
	  !parseOpenItem(tokenizer) &&
	  !parseCloseItem(tokenizer))
	tokenizer.getTagOrText();
    }
  }

  private boolean parseTextItem(HtmlTokenizer tokenizer)
  {
    String text;

    if (endOfText == null)
      text = tokenizer.getText();
    else
      text = tokenizer.getPreformattedText(endOfText);

    if (text != null)
    {
      codes.addElement(TEXT);
      strings.addElement(text);
      return true;
    }

    return false;
  }

  private boolean parseOpenItem(HtmlTokenizer tokenizer)
  {
    for (int code = 1; code < MAXCODES; code++)
    {
      Hashtable attr = tokenizer.getOpenTag(names[code]);
      if (attr != null)
      {
	codes.addElement(code);
	switch (code)
	{
	case A:
	  strings.addElement(get(attr, "HREF"));
	  strings.addElement(get(attr, "NAME"));
	  break;

	case BASE:
	  String href = get(attr, "HREF");
	  strings.addElement(href);
	  try { base = new URL(href); } catch (Exception e) {}
	  break;

	case BODY:
	  bgColor = getColor(get(attr, "BGCOLOR"), bgColor);
	  textColor = getColor(get(attr, "TEXT"), textColor);
	  linkColor = getColor(get(attr, "LINK"), linkColor);
	  break;

	case FONT:
	  strings.addElement(get(attr, "SIZE"));
	  strings.addElement(get(attr, "COLOR"));
	  break;

	case IMG:
	  int w = getInt(get(attr, "WIDTH"));
	  int h = getInt(get(attr, "HEIGHT"));
	  images.addElement(base, get(attr, "SRC"), w, h);
	  strings.addElement(get(attr, "ALIGN"));
	  break;

	case PRE:
	  endOfText = "";
	  break;

	case TITLE:
	  title = tokenizer.getText();
	  while (!tokenizer.getCloseTag(names[code]))
	    tokenizer.getTagOrText();
	  break;

	case LISTING:
	case XMP:
	  endOfText = "/" + names[code];
	  break;

	default:
	  break;
	}
        return true;
      }
    }

    return false;
  }

  private boolean parseCloseItem(HtmlTokenizer tokenizer)
  {
    for (int code = 1; code < MAXCODES; code++)
    {
      if (tokenizer.getCloseTag(names[code]))
      {
	codes.addElement(-code);
	if (code == PRE || code == LISTING || code == XMP)
	  endOfText = null;
	return true;
      }
    }

    return false;
  }

  private static final int TEXT 	= 0;
  private static final int A		= 1+TEXT;
  private static final int ADDRESS	= 1+A;
  private static final int B		= 1+ADDRESS;
  private static final int BASE		= 1+B;
  private static final int BIG		= 1+BASE;
  private static final int BLOCKQUOTE	= 1+BIG;
  private static final int BODY		= 1+BLOCKQUOTE;
  private static final int BR		= 1+BODY;
  private static final int CENTER	= 1+BR;
  private static final int CITE		= 1+CENTER;
  private static final int CODE		= 1+CITE;
  private static final int DD		= 1+CODE;
  private static final int DIR		= 1+DD;
  private static final int DL		= 1+DIR;
  private static final int DT		= 1+DL;
  private static final int EM		= 1+DT;
  private static final int FONT		= 1+EM;
  private static final int H1		= 1+FONT;
  private static final int H2		= 1+H1;
  private static final int H3		= 1+H2;
  private static final int H4		= 1+H3;
  private static final int H5		= 1+H4;
  private static final int H6		= 1+H5;
  private static final int HR		= 1+H6;
  private static final int I		= 1+HR;
  private static final int IMG		= 1+I;
  private static final int KBD		= 1+IMG;
  private static final int LI		= 1+KBD;
  private static final int LISTING	= 1+LI;
  private static final int MENU		= 1+LISTING;
  private static final int OL		= 1+MENU;
  private static final int P		= 1+OL;
  private static final int PRE		= 1+P;
  private static final int SAMP		= 1+PRE;
  private static final int SMALL	= 1+SAMP;
  private static final int STRONG	= 1+SMALL;
  private static final int TITLE	= 1+STRONG;
  private static final int TT		= 1+TITLE;
  private static final int UL		= 1+TT;
  private static final int VAR		= 1+UL;
  private static final int XMP		= 1+VAR;
  private static final int MAXCODES	= 1+XMP;

  protected static final String names[] =
  {
    null,
    "A",
    "ADDRESS",
    "B",
    "BASE",
    "BIG",
    "BLOCKQUOTE",
    "BODY",
    "BR",
    "CENTER",
    "CITE",
    "CODE",
    "DD",
    "DIR",
    "DL",
    "DT",
    "EM",
    "FONT",
    "H1",
    "H2",
    "H3",
    "H4",
    "H5",
    "H6",
    "HR",
    "I",
    "IMG",
    "KBD",
    "LI",
    "LISTING",
    "MENU",
    "OL",
    "P",
    "PRE",
    "SAMP",
    "SMALL",
    "STRONG",
    "TITLE",
    "TT",
    "UL",
    "VAR",
    "XMP"
  };

  private void drawOpen(int code, HtmlPager p)
  {
    int size;
    String href;
    String name;
    String align;
    String text;
    HtmlImage image;
    Color color;

    switch (code)
    {
    case TEXT:		text = strings.nextElement();
			p.drawText(text); return;
    case A:		href = strings.nextElement();
			name = strings.nextElement();
			p.pushAnchor(href, name); return;
    case ADDRESS:	p.pushItalic(); return;
    case B:		p.pushBold(); return;
    case BASE:		return;
    case BIG:		p.pushFontSize(p.getFontSize() - 1); return;
    case BLOCKQUOTE:	p.pushRightMargin(); p.pushLeftMargin(true); return;
    case BODY:		return;
    case BR:		p.drawNewLine(false); return;
    case CENTER:	p.pushCenter(); return;
    case CITE:		return;
    case CODE:		p.pushFixedFont(); return;
    case DD:		p.drawNewLine(false); return;
    case DIR:		p.pushLeftMargin(true); p.pushListButton(); return;
    case DL:		p.pushLeftMargin(true); return;
    case DT:		p.popLeftMargin(true); p.pushLeftMargin(false); return;
    case EM:		p.pushItalic(); return;
    case FONT:		size = getFontSize(strings.nextElement(), p);
			color = getColor(strings.nextElement(), p.getFontColor());
			p.pushFontColor(color);
			p.pushFontSize(size); return;
    case H1:		p.drawNewLine(true); pushHeader(1, p); return;
    case H2:		p.drawNewLine(true); pushHeader(2, p); return;
    case H3:		p.drawNewLine(true); pushHeader(3, p); return;
    case H4:		p.drawNewLine(true); pushHeader(4, p); return;
    case H5:		p.drawNewLine(true); pushHeader(5, p); return;
    case H6:		p.drawNewLine(true); pushHeader(6, p); return;
    case HR:		p.drawRule(); return;
    case I:		p.pushItalic(); return;
    case IMG:		align = strings.nextElement();
			image = images.nextElement();
			p.drawImage(image, align); return;
    case KBD:		p.pushFixedFont(); return;
    case LI:		p.drawNewLine(false); p.drawListItem(); return;
    case LISTING:	p.pushPreformatted(); return;
    case MENU:		p.pushLeftMargin(true); p.pushListButton(); return;
    case OL:		p.pushLeftMargin(true); p.pushListNumber(); return;
    case P:		p.drawNewLine(true); return;
    case PRE:		p.pushPreformatted(); return;
    case SAMP:		p.pushFixedFont(); return;
    case SMALL:		p.pushFontSize(p.getFontSize() + 1); return;
    case STRONG:	p.pushBold(); return;
    case TITLE:		return;
    case TT:		p.pushFixedFont(); return;
    case UL:		p.pushLeftMargin(true); p.pushListButton(); return;
    case VAR:		p.pushFixedFont(); p.pushBold(); return;
    case XMP:		p.pushPreformatted(); return;
    }
  }

  private void drawClose(int code, HtmlPager p)
  {
    switch (code)
    {
    case TEXT:		return;
    case A:		p.popAnchor(); return;
    case ADDRESS:	p.popFont(); return;
    case B:		p.popFont(); return;
    case BASE:		return;
    case BIG:		p.popFont(); return;
    case BLOCKQUOTE:	p.popRightMargin(); p.popLeftMargin(true); return;
    case BODY:		return;
    case BR:		return;
    case CENTER:	p.popCenter(); return;
    case CITE:		return;
    case CODE:		p.popFont(); return;
    case DD:		return;
    case DIR:		p.popList(); p.popLeftMargin(true); return;
    case DL:		p.popLeftMargin(true); return;
    case DT:		return;
    case EM:		p.popFont(); return;
    case FONT:		p.popFontColor(); p.popFont(); return;
    case H1:		popHeader(p); p.drawNewLine(true); return;
    case H2:		popHeader(p); p.drawNewLine(true); return;
    case H3:		popHeader(p); p.drawNewLine(true); return;
    case H4:		popHeader(p); p.drawNewLine(true); return;
    case H5:		popHeader(p); p.drawNewLine(true); return;
    case H6:		popHeader(p); p.drawNewLine(true); return;
    case HR:		return;
    case I:		p.popFont(); return;
    case IMG:		return;
    case KBD:		p.popFont(); return;
    case LI:		return;
    case LISTING:	p.popPreformatted(); return;
    case MENU:		p.popList(); p.popLeftMargin(true); return;
    case OL:		p.popList(); p.popLeftMargin(true); return;
    case P:		return;
    case PRE:		p.popPreformatted(); return;
    case SAMP:		p.popFont(); return;
    case SMALL:		p.popFont(); return;
    case STRONG:	p.popFont(); return;
    case TITLE:		return;
    case TT:		p.popFont(); return;
    case UL:		p.popList(); p.popLeftMargin(true); return;
    case VAR:		p.popFont(); p.popFont(); return;
    case XMP:		p.popPreformatted(); return;
    }
  }

  private void pushHeader(int size, HtmlPager p)
  {
    p.pushStandardFont();
    p.pushBold();
    p.pushFontSize(size);
  }
  private void popHeader(HtmlPager p)
  {
    p.popFont();
    p.popFont();
    p.popFont();
  }

  private static int getFontSize(String str, HtmlPager p)
  {
    if (str == null)
      return p.getFontSize();

    try
    {
      if (str.charAt(0) == '+')
	return p.getFontSize() - Integer.parseInt(str.substring(1), 10);
      else if (str.charAt(0) == '-')
	return p.getFontSize() + Integer.parseInt(str.substring(1), 10);
      else
	return 7 - Integer.parseInt(str, 10);
    }
    catch (Exception e)
    {
      return p.getFontSize();
    }
  }

  private static Color getColor(String str, Color defaultColor)
  {
    if (str == null)
      return defaultColor;

    if (str.charAt(0)=='#')
      str = str.substring(1);

    try
    {
      int r = Integer.parseInt(str.substring(0,2), 16);
      int g = Integer.parseInt(str.substring(2,4), 16);
      int b = Integer.parseInt(str.substring(4,6), 16);
      if (r < 0 || g < 0 || b < 0)
	return defaultColor;

      return new Color(r, g, b);
    }
    catch (Exception e)
    {
      return defaultColor;
    }
  }

  private static int getInt(String str)
  {
    try
    {
      return Integer.parseInt(str, 10);
    }
    catch (Exception e)
    {
      return -1;
    }
  }

  private static String get(Hashtable attr, String key)
  {
    return (String)attr.get(key);
  }
}
