/*
 * input.c - Helper routines for reading input from curses.
 *
 * Copyright (C) 2001  Southern Storm Software, Pty Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "defs.h"

#ifdef	__cplusplus
extern	"C" {
#endif

/*
 * Handle a window resize request from the terminal.
 */
static void HandleResize(void)
{
	struct winsize size;

	/* Get the new window size from the terminal and adjust accordingly */
	if(ioctl(0, TIOCGWINSZ, &size) >= 0)
	{
		resizeterm(size.ws_row, size.ws_col);
	}

	/* Update the entire screen */
	touchwin(curscr);
	wrefresh(curscr);
}

/*
 * Magic number that converts a time which is relative to
 * Jan 1, 1970 into a value which is relative to Jan 1, 0001.
 */
#if defined(WIN32) && !defined(__CYGWIN__)
#define	EPOCH_ADJUST	((CursesHelpInt64)62135596800LL)
#else
#define	EPOCH_ADJUST	((CursesHelpInt64)62135596800L)
#endif

int CursesHelpGetNextChar(CursesHelpInt64 timeToFire)
{
	struct timeval currentTime;
	struct timeval endTime;
	fd_set readSet;
	int result;
	int ch;

	/* Make sure that all curses drawing operations have been flushed */
	refresh();

	/* Determine when the timeout should end */
	if(timeToFire >= 0)
	{
		endTime.tv_sec = (long)((timeToFire / 10000000) - EPOCH_ADJUST);
		endTime.tv_usec = ((long)((timeToFire % 10000000) / 10));
	}
	else
	{
		endTime.tv_sec = 0;
		endTime.tv_usec = 0;
	}

	/* Continue selecting until timeout, input available, or a signal */
	for(;;)
	{
		FD_ZERO(&readSet);
		FD_SET(0, &readSet);
		if(timeToFire >= 0)
		{
			/* We need to select using a timeout */
			gettimeofday(&currentTime, (struct timezone *)0);
			if(currentTime.tv_sec > endTime.tv_sec ||
			   (currentTime.tv_sec == endTime.tv_sec &&
			    currentTime.tv_usec >= endTime.tv_usec))
			{
				/* A timeout has occurred */
				break;
			}
			if(currentTime.tv_usec <= endTime.tv_usec)
			{
				currentTime.tv_usec = endTime.tv_usec - currentTime.tv_usec;
				currentTime.tv_sec = endTime.tv_sec - currentTime.tv_sec;
			}
			else
			{
				currentTime.tv_usec = endTime.tv_usec + 1000000 -
									  currentTime.tv_usec;
				currentTime.tv_sec = endTime.tv_sec - currentTime.tv_sec - 1;
			}
			result = select(1, &readSet, (fd_set *)0,
						    (fd_set *)0, &currentTime);
		}
		else
		{
			/* There is no timeout, so select forever */
			result = select(1, &readSet, (fd_set *)0,
							(fd_set *)0, (struct timeval *)0);
		}
		if(result > 0)
		{
			/* There is input available */
			ch = getch();
			if(ch == ('L' & 0x1F) || ch == ('R' & 0x1F) || ch == KEY_REFRESH)
			{
				/* Refresh the screen */
				touchwin(curscr);
				wrefresh(curscr);
				ch = KEY_REFRESH;
			}
			if(ch >= 0x100)
			{
				/* Adjust the keycode to the C# version of the value */
				return (ch << 16);
			}
			else
			{
				return ch;
			}
		}
		else if(result == 0)
		{
			/* A timeout has occurred */
			break;
		}
		else
		{
			/* The select was interrupted, so process the pending signals */
			if(errno == EINTR)
			{
				if(CursesHelpSawSizeChange)
				{
					CursesHelpSawSizeChange = 0;
					if(CursesHelpSawInterrupt)
					{
						CursesHelpSawInterrupt = 0;
						return (KEY_MAX << 16);
					}
					else
					{
						HandleResize();
						return (KEY_RESIZE << 16);
					}
				}
				if(CursesHelpSawInterrupt)
				{
					CursesHelpSawInterrupt = 0;
					return (KEY_MAX << 16);
				}
			}
		}
	}

	/* We will get here on timeout */
	return -1;
}

#ifdef	__cplusplus
};
#endif
