// Copyright (C) 2000 Open Source Telecom Corporation.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// As a special exception to the GNU General Public License, permission is
// granted for additional uses of the text contained in its release
// of Bayonne as noted here.
//
// This exception is that permission is hereby granted to link Bayonne 
// with the Dialogic runtime libraries to produce a executable image
// without requiring Dialogic's sources to be supplied so long as each
// each source file so linked contains this exclusion.
//
// This exception does not however invalidate any other reasons why
// the resulting executable file might be covered by the GNU General
// public license or invalidate the licensing requirements of any
// other component or library.
//
// This exception applies only to the code released by OST under the
// name Bayonne.  If you copy code from other releases into a copy of
// Bayonne, as the General Public License permits, the exception does not
// apply to the code that you add in this way.  To avoid misleading
// anyone as to the status of such modified files, you must delete
// this exception notice from them.
//
// If you write modifications of your own to Bayonne, it is your choice
// whether to permit this exception to apply to your modifications.
// If you do not wish that, delete this exception notice, at which
// point the terms of your modification would be covered under the GPL
// as explicitly stated in "COPYING".

#include "driver.h"

#define	SIGEVTCHK	0x1010
#define SIGBITCHK	0x1111

DialogicConfig::DialogicConfig() :
Keydata("/bayonne/dialogic")
{
	static KEYDEF defkeys[] = {
	{"timeslots", "32"},
	{"voice", "1"},
	{"digital", "0"},
	{"routing", "digital"},
	{NULL, NULL}};

	Load(defkeys);
}

bool DialogicConfig::getRouting(void)
{
	const char *rtflag = getLast("routing");

	if(!stricmp(rtflag, "digital"))
		return false;

	return true;
}

DialogicDriver::DialogicDriver() :
Driver(), Thread(NULL, keythreads.priService())
{
	DialogicTrunk *trk;
	char d4xname[32];
	char dtiname[32];
	int dticards, voxcards;
	int dticard = 1, voxcard = 1, digital = 0;
	int card = 0;
	int voxcount, vox, dti, dticount;
	int id = 0;
	int ts, tslots, chdev, tsdev;
	int maxslots = getTimeslots();
	CT_DEVINFO ctinfo;

	status = DialogicTrunk::status;
	memset(status, ' ', sizeof(DialogicTrunk::status));

	running = false;

	voxcards = getVoice();
	dticards = getDigital();

	if(!voxcards)
	{
		slog(SLOG_ERROR) << "Dialogic driver missing cards; not initialized" << endl;
		maps = NULL;
		ports = NULL;
		groups = NULL;
		port_count = 0;
		port_used = 0;
		return;
	}

	if(dticards * maxslots > voxcards * 4)
		port_count = dticards * maxslots;
	else
		port_count = voxcards * 8;
	
	ports = new DialogicTrunk *[port_count];
	groups = new TrunkGroup *[port_count];
	maps = new DialogicTrunk *[port_count * 4 + 16];
	memset(ports, 0, sizeof(DialogicTrunk *) * port_count);
	memset(groups, 0, sizeof(TrunkGroup *) * port_count);
	memset(maps, 0, sizeof(DialogicTrunk *) * port_count * 4 + 16);

	while(voxcard <= voxcards)
	{
		sprintf(d4xname, "dxxxB%d", voxcard);
		chdev = dx_open(d4xname, 0);
		if(chdev < 0)
		{
			++voxcard;
			continue;
		}
	
		voxcount = ATDV_SUBDEVS(chdev);
		dx_close(chdev);
		
		for(vox = 1; vox <= voxcount; ++vox) 
		{
			sprintf(d4xname, "dxxxB%dC%d", voxcard, vox);
			chdev = dx_open(d4xname, 0);
			tsdev = 0;
			if(!dti && vox == 1)
			{
				dx_getctinfo(chdev, &ctinfo);
				switch(ctinfo.ct_nettype)
				{
				case CT_NTT1:
					dti = 1;
					dticount = 24;
					break;
				case CT_NTE1:
					dti = 1;
					dticount = 30;
					break;
				}
			}
			if(!dti)
			{
//				ports[id] = new DialogicTrunk(id, chdev, -1);
				maps[chdev] = ports[id];
				++id;
				break;
			}
			while(dti == 1 && dticard <= dticards)
			{
				sprintf(dtiname, "dtiB%d", dticard);
				tsdev = dt_open(dtiname, 0);
				if(tsdev < 0)
				{
					++dticard;
					continue;
				}
				dticount = ATDV_SUBDEVS(tsdev);
				dt_close(tsdev);
				break;
			}
			if(dti)
			{
				if(dticard > dticards)
					break;

				if(dti > dticount)
				{
					dti = 0;
					break;
				}
				sprintf(dtiname, "dtiB%dT%d", dticard, dti);
				tsdev = dt_open(dtiname, 0);
//				ports[id] = new DialogicTrunk(id, chdev, tsdev);
				maps[chdev] = ports[id];
				maps[tsdev] = ports[id];
				++id;
				++dti;
			}
		}
		++voxcard;
	}
				
	slog(SLOG_INFO) << "Dialogic driver loaded; capacity=" << id << endl;
	port_used = id;
}

DialogicDriver::~DialogicDriver()
{
	if(running)
		Terminate();

	Stop();
	if(ports)
		delete ports;

	if(maps)
		delete maps;

	if(groups)
		delete groups;
}

int DialogicDriver::Start(void)
{
	int count = 0;
	int port;

	if(active)
	{
		slog(SLOG_ERROR) << "driver already started" << endl;
		return 0;
	}

	for(port = 0; port < port_used; ++port)
		ports[port]->Start();

	slog(SLOG_INFO) << "driver starting..." << endl;

	if(!running)
		Thread::Start();

	active = true;
	return count;
}

void DialogicDriver::Stop(void)
{
	int port;
	if(!active)
		return;

	for(port = 0; port < port_used; ++port)
		ports[port]->Terminate();		

	if(ports)
		memset(ports, 0, sizeof(DialogicTrunk *) * port_count);

	active = false;
	slog(SLOG_INFO) << "driver stopping..." << endl;
}

void DialogicDriver::Run(void)
{
	int evdev;
	DialogicTrunk *trunk;
	TrunkEvent event;
	DX_CST *cst;
	DV_DIGIT *digbuf;
	unsigned short sig;
	short ind;
	int evtype;
	int digcnt, diglen;
	unsigned long tmask;
	char *d;

	running = true;
	setCancel(THREAD_CANCEL_IMMEDIATE);
	for(;;)
	{
		sr_waitevt(-1);
		evdev = sr_getevtdev();
		if(evdev < 0)
			continue;

		trunk = maps[evdev];
		if(!trunk)
			continue;

		memset(&event, 0, sizeof(event));
		evtype = sr_getevttype();

retry:
		switch(evtype)
		{
		case TDX_CALLP:
			switch(ATDX_CPTERM(trunk->chdev))
			{
			case CR_BUSY:
				event.id = TRUNK_CPA_BUSYTONE;
				break;
			case CR_CEPT:
				event.id = TRUNK_CPA_INTERCEPT;
				break;
			case CR_CNCT:
				event.id = TRUNK_CPA_CONNECT;
				break;
			case CR_FAXTONE:
				event.id = TRUNK_FAX_DETECT;
				break;
			case CR_NOANS:
				event.id = TRUNK_CPA_NOANSWER;
				break;
			case CR_NODIALTONE:
				event.id = TRUNK_CPA_NODIALTONE;
				break;
			case CR_NORB:
				event.id = TRUNK_CPA_NORINGBACK;
				break;
			}
			if(event.id)
			{
				trunk->putEvent(&event);
				break;
			}
			evtype = TDX_DIAL;
			goto retry;
		case TDX_GETDIG:
			digbuf = &trunk->digbuf;
			diglen = strlen(digbuf->dg_value);
			digcnt = 0;
			while(digcnt < diglen)
			{
				event.id = TRUNK_DTMF_KEYUP;
				event.parm.dtmf.duration = 40;
				event.parm.dtmf.digit = trunk->getDigit(digbuf->dg_value[digcnt++]);
				trunk->putEvent(&event);
			}
		case TDX_PLAY:
		case TDX_RECORD:
		case TDX_DIAL:			
		case TDX_PLAYTONE:
		case TDX_ERROR:
			tmask = ATDX_TERMMSK(trunk->chdev);
			switch(trunk->interface)
			{
			case NT_ANALOG:
				if(tmask & TM_LCOFF)
				{
					event.id = TRUNK_LINE_WINK; 
					trunk->putEvent(&event);
					break;
				}
			case NT_T1:
				if((ATDT_TSSGBIT(trunk->tsdev) & DTSG_RCVA) == 0)
				{
					if(trunk->flags.offhook)
						event.id = TRUNK_CALL_DISCONNECT;
					else
						event.id = TRUNK_CALL_RINGING;
					trunk->putEvent(&event);
					break;
				}
			case NT_E1:
				if((ATDT_TSSGBIT(trunk->tsdev) & DTSG_RCVA) != 0)
				{
					if(trunk->flags.offhook)
						event.id = TRUNK_CALL_DISCONNECT;
					else
						event.id = TRUNK_CALL_RINGING;
					trunk->putEvent(&event);
					break;
				}
			}
			if(event.id)
				break;

			if((tmask & TM_IDDTIME) || (tmask & TM_MAXTIME))
			{
				event.id = TRUNK_TIMER_EXPIRED;
				trunk->putEvent(&event);
				break;
			}

			if(tmask & TM_TONE)
			{
				event.id = TRUNK_CPA_DIALTONE;
				trunk->putEvent(&event);
				break;
			}

			if(tmask & TM_DIGIT)
			{
			}
	
			switch(evtype)
			{
			case TDX_PLAYTONE:
			case TDX_DIAL:
			case TDX_CALLP:
				event.id = TRUNK_TONE_IDLE;
				break;
			case TDX_PLAY:
			case TDX_RECORD:
				event.id = TRUNK_AUDIO_IDLE;
			}
			if(event.id)
				trunk->putEvent(&event);
				
			break;
		case TDX_SETHOOK:
			if(trunk->flags.offhook)
				event.id = TRUNK_ON_HOOK;
			else
				event.id = TRUNK_OFF_HOOK;
			trunk->putEvent(&event);
			break;
		case TDX_CST:
			cst = (DX_CST *) sr_getevtdatap();
			switch(cst->cst_event)
			{
			case DE_RINGS:
				event.id = TRUNK_RINGING_ON;
				trunk->putEvent(&event);
				break;
			}
			break;
		case DTEV_SIG:
			sig = (unsigned short)(*((unsigned short *)sr_getevtdatap()));
			for(ind = 0; ind < 4; ++ind)
			{
				if(!(sig & (SIGEVTCHK << ind)))
					continue;
				switch(sig & (SIGBITCHK << ind))
				{
				case DTMM_AON:
					if(trunk->interface == NT_T1)
						event.id = TRUNK_CALL_RINGING;
					else
						event.id = TRUNK_CALL_DISCONNECT;
					trunk->putEvent(&event);
					break;
				case DTMM_AOFF:
					if(trunk->interface == NT_E1)
						event.id = TRUNK_CALL_RINGING;
					else
						event.id = TRUNK_CALL_DISCONNECT;
					trunk->putEvent(&event);
					break;
				}
			}
			break;
		}
	} 
}

Trunk *DialogicDriver::getTrunkPort(int id)
{
	if(id < 0 || id >= port_count)
		return NULL;

	if(!ports)
		return NULL;

	return (Trunk *)ports[id];
}

void DialogicDriver::Hold(void)
{
	EnterMutex();
	if(!level)
		sr_hold();
	++level;
	LeaveMutex();
}

void DialogicDriver::Release(void)
{
	EnterMutex();
	--level;
	if(!level)
		sr_release();
	LeaveMutex();
}

DialogicDriver dialogicivr;
