// Copyright (C) 2000 Open Source Telecom Corporation.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include <ctype.h>
#include "server.h"

char Trunk::digit[16] = {
	'0', '1', '2', '3',
	'4', '5', '6', '7',
	'8', '9', '*', '#',
	'a', 'b', 'c', 'd'};

Trunk::Trunk(int port) :
ScriptInterp((aaScript *)driver, keymemory.getSymbolSize(), keymemory.getPageSize())
{
	int i;
	static char *names[] = {"date", "time", "duration", "count", "rings"};

	group = driver->getTrunkGroup(port);
	group->incCapacity();

	flags.dtmf = flags.offhook = flags.reset = false;
	flags.trunk = TRUNK_MODE_INACTIVE;
	flags.dsp = DSP_MODE_INACTIVE;
	flags.script = false;
	id = port;
	rings = 0;
	start = idle = 0;
	thread = NULL;
	script = NULL;
	session = NULL;
	manager = NULL;

	dtmf.bin.id = "digits";
	dtmf.bin.flags.readonly = false;
	dtmf.bin.flags.system = true;
	dtmf.bin.flags.initial = false;
	dtmf.bin.flags.commit = true;
	dtmf.bin.flags.size = 32;
	dtmf.bin.next = NULL;

	for(i = 0; i < 5; ++i)
	{
		numbers[i].sym.id = names[i];
		numbers[i].sym.flags.readonly = true;
		numbers[i].sym.flags.system = true;
		numbers[i].sym.flags.initial = false;
		numbers[i].sym.flags.commit = false;
		numbers[i].sym.flags.size = 10;
		numbers[i].sym.next = NULL;
	}
}

int Trunk::getDigit(char dig)
{
	int i;

	dig = tolower(dig);
	for(i = 0; i < 16; ++i)
	{
		if(digit[i] == dig)
			return i;
	}
	return -1;
}

unsigned long Trunk::getMask(void)
{
	if(manager)
		return manager->getMask();

	return ScriptInterp::getMask();
}

void Trunk::setDTMFDetect(void)
{
	if(isActive())
		setDTMFDetect(getMask() & 0x00000008);
	else
		setDTMFDetect(false);
}

bool Trunk::ScriptStep(void)
{
	if(manager)
		return manager->Step();
	return Step();
}

void Trunk::Commit(scriptsymbol_t *sym)
{
	if(sym == &dtmf.bin)
		digits = strlen(dtmf.bin.data);
}

void Trunk::Logout(void)
{
	if(!session)
		return;

	if(database)
		database->Logout(this);

	session = NULL;
}

void Trunk::setConstant(const char *id, const char *data)
{
	scriptsymbol_t *sym = ScriptSymbol::getEntry(id, strlen(data));
	if(!sym)
		return;

	if(!sym->flags.initial)
		return;
	
	strcpy(sym->data, data);
	sym->flags.initial = false;
	sym->flags.readonly = true;
}

scriptsymbol_t *Trunk::getEntry(const char *name, int size)
{
	int i;
	time_t now;
	struct tm *dt;
	struct tm tbuf;

	if(*name == '%')	// bug in interp
		++name;

	if(!stricmp(name, "digits"))
	{
		dtmf.bin.data[digits] = 0;
		return &dtmf.bin;
	}

	for(i = 0; i < 5; ++i)
	{
		if(!stricmp(name, numbers[i].sym.id))
			break;
	}

	if(i >= 5)
		return ScriptSymbol::getEntry(name, size);

	if(i < 3)
		time(&now);
	
	switch(i)
	{
	case 4:
		sprintf(numbers[4].sym.data, "%d", rings);
		break;
	case 3:
		sprintf(numbers[3].sym.data, "%d", digits);
		break;
	case 2:
		if(start)
			sprintf(numbers[2].sym.data, "%ld", now - start);
		else
			numbers[2].sym.data[0] = 0;
		break;
	case 0:
		dt = localtime_r(&now, &tbuf);
		sprintf(numbers[0].sym.data, "%04d%02d%02d",
			dt->tm_year + 1900, dt->tm_mon + 1, dt->tm_mday);
		break;
	case 1:
		dt = localtime_r(&now, &tbuf);
		sprintf(numbers[1].sym.data, "%02d%02d%02d",
			dt->tm_hour, dt->tm_min, dt->tm_sec);
	}

	return &numbers[i].sym;
}

bool Trunk::Attach(const char *name)
{
	ScriptCommand *cmd = getCommand();
	char buf[65];
	scriptsymbol_t *sym;
	time_t now;
	struct tm *dt;
	struct tm tbuf;

	// This concerns me.  Why are we returning false?
	// What, in fact, is this case?

	// Since we just set TRUNK_MODE_OUTGOING, in our parent
	// stack frame, I'll kill the flags.script case if we
	// are outbound scheduled.

	//Matt B

	if(flags.trunk != TRUNK_MODE_OUTGOING) 
	{
	    if(flags.script)
	        return false;
	}

	flags.reset = flags.dtmf = flags.offhook = flags.once = false;
	tgi.pid = 0;

	if(manager)
	{
		if(!manager->postAccept(name))
			return false;
	}
	else if(!ScriptInterp::Attach(name))
		return false;
	
	flags.script = true;
	setSymbol("annotation", 160);
	setSymbol("played", 12);
	setSymbol("recorded", 12);
	setSymbol("created", 20);

	setSymbol("extension", 8);
	setSymbol("extension", cmd->getLast("extension"));
	
	setSymbol("trim", 8);
	setSymbol("trim", "600");

	setSymbol("language", 16);
	setSymbol("language", cmd->getLast("language"));

	setSymbol("voice", 16);
	setSymbol("voice", cmd->getLast("voice"));

	setSymbol("volume", 3);
	setSymbol("volume", group->getLast("volume"));

	sprintf(buf, "%03d", id);
	setConstant("id", buf);

	sprintf(buf, "%d", driver->getTrunkCount());
	setConstant("ports", buf);

	group->getSchedule(buf);
	setConstant("schedule", buf);

	time(&now);
	dt = localtime_r(&now, &tbuf);
	sprintf(buf, "%04d%02d%02d",
		dt->tm_year + 1900, dt->tm_mon + 1, dt->tm_mday);
	setConstant("startdate", buf);
	sprintf(buf, "%02d%02d%02d",
		dt->tm_hour, dt->tm_min, dt->tm_sec);
	setConstant("starttime", buf);

	setConstant("version", cmd->getLast("version"));
	setConstant("server", cmd->getLast("server"));
	setConstant("driver", plugins.getDriverName());
	setConstant("node", cmd->getLast("node"));
	setConstant("group", group->getName());
	setConstant("start", name);

	cdrc = 0;
	if(!start)
		time(&start);

	if(database)
		database->Initialize(this);

	debug->DebugState(this, "attach script");
	return true;
}

void Trunk::Detach(void)
{
	char buffer[256];
	int i = 0;
	strstream str(buffer, 256);
	char *val;

	if(!flags.script)
		return;

	Logout();
	buffer[0] = 0;
	while(i < cdrc)
	{
		val = getContent(cdrv[i++]);
		if(!val)
			continue;
			
		str << val;
	}

	if(cdrc)
	{
		str << endl;
		str << ends;
		audit(buffer);
	}

	digits = 0;
	cdrc = 0;
	if(manager)
		manager->postDetach();
	else
		ScriptInterp::Detach();
	ScriptSymbol::Purge();
	start = 0;
	flags.script = false;
	if(tgi.pid)
		kill(tgi.pid, SIGHUP);
	tgi.pid = 0;
	if(database)
		database->Finalize(this);
	debug->DebugState(this, "detach script");
}

void Trunk::stopServices(void)
{
	if(thread)
	{
		if(thread->isExiting())
			delete thread;
	}
	thread = NULL;
}

int Trunk::getTimeout(void)
{
	ScriptCommand *cmd = getCommand();
	char *opt = getValue(cmd->getLast("timeout"));
	return atoi(opt);
}

int Trunk::getInterdigit(void)
{
	ScriptCommand *cmd = getCommand();
	char *opt = getValue(cmd->getLast("interdigit"));
	return atoi(opt);
}

unsigned short Trunk::getDigitMask(void)
{
	static char *digits = "0123456789*#abcd";
	unsigned short mask = 0;
	char *cp = getValue(NULL);
	char *dp;

	if(!cp)
		return 0;
	
	while(*cp)
	{
		dp = strchr(digits, tolower(*cp));
		++cp;
		if(dp)
			mask |= (1 << (int)(dp - digits));
	}
	return mask;
}

bool Trunk::TrunkSignal(trunksignal_t signal)
{
	if(manager)
		return manager->postSignal(signal);

	if(!signal)
	{
		Advance();
		return true;
	}
	else
		return Signal((unsigned long)(signal) - 1);
}

