// Copyright (C) 2000 Open Source Telecom Corporation.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include "server.h"
#include <pwd.h>
#include <math.h>

KeyLocal::KeyLocal() :
Keydata("/bayonne/localize")
{
	static KEYDEF keydefs[] = {
	{"primarycurrency", "dollar"},
	{"primarychange", "cent"},
	{"convertcurrency", "1.00"},
	{NULL, NULL}};

	Load(keydefs);
}

KeyPaths::KeyPaths() :
Keydata("/bayonne/paths")
{
	static KEYDEF defpaths[] = {
	{"libpath", "/usr/lib/bayonne"},
	{"libexec", "/usr/lib"},
	{"tgipath", "/usr/lib/tgi:/bin:/usr/bin"},
	{"datafiles", "/var/bayonne"},
	{"scripts", "/usr/share/aascripts"},
	{"prompts", "/usr/share/aaprompts"},
	{"spool", "/var/spool/bayonne"},
	{"runfiles", "/var/run"},
	{"precache", "/var/bayonne/cache"},
	{"config", "bayonne runtime configuration"},
	{NULL, NULL}};

	Load("~bayonne/paths");
	Load(defpaths);
}

KeyServer::KeyServer() :
Keydata("/bayonne/server")
{
	static KEYDEF defkeys[] = {
	{"user", "mail"},
	{"default", "default"},
	{"nodes", "12"},
	{"token", "&"},
	{NULL, NULL}};

	struct passwd *pwd;
	char namebuf[128];
	char *cp;

	Load("~bayonne/server");
	if(!getLast("node"))
	{
		gethostname(namebuf, sizeof(namebuf) - 1);
		cp = strchr(namebuf, '.');
		if(cp)
			*cp = 0;
		setValue("node", namebuf);
	}
	Load(defkeys);

	uid = getuid();
	gid = getgid();
	if(uid)
		return;

	pwd = getpwnam(getLast("user"));
	if(pwd)
	{
		uid = pwd->pw_uid;
		gid = pwd->pw_gid;
	}
	else
	{
		uid = 0xffff;
		gid = 0xffff;
	}
	endpwent();
}

void KeyServer::loadGroups(void)
{
	char *group;

	new TrunkGroup();
	slog(SLOG_DEBUG) << "loading default trunk group" << endl;
	
	group = (char *)TrunkGroup::first->getLast("groups");
	if(!group)
		return;

	group = strtok(group, " \t\n");
	while(group)
	{
		slog(SLOG_DEBUG) << "loading " << group << " trunk group" << endl;
		new TrunkGroup(group);
		group = strtok(NULL, " \t\n");
	}
}

TrunkGroup *KeyServer::getGroup(char *name)
{
	TrunkGroup *grp = TrunkGroup::first;

	while(grp)
	{
		if(!stricmp(name, grp->getName()))
			break;
		grp = grp->next;
	}
	
	return grp;
}

KeyThreads::KeyThreads() :
Keydata("/bayonne/threads")
{
	static KEYDEF defpaths[] = {
	{"audio", "0"},
	{"priority", "0"},
	{"gateways", "0,1"},
	{"services", "1,1"},
	{"database", "0"},
	{"managers", "0"},
	{"scheduler", "0"},
	{"resetdelay", "18"},
	{"stepdelay", "36"},
	{"stepinterval", "18"},
	{"interval", "15"},
	{"refresh", "5"},
	{"policy", "other"},
	{"pages", "0"},
	{NULL, NULL}};

	const char *cp = getLast("pri");
	
	if(cp)
		setValue("priority", cp);

	Load(defpaths);
}

int KeyThreads::getPolicy()
{
	const char *cp = getLast("policy");
#ifdef	SCHED_RR
	if(!stricmp(cp, "rr"))
		return SCHED_RR;
#endif
#ifdef	SCHED_FIFO
	if(!stricmp(cp, "fifo"))
		return SCHED_FIFO;
#endif
	return 0;
}

KeyMailbox::KeyMailbox() :
Keydata("/bayonne/mailbox")
{
	static KEYDEF defkeys[] = {
	{"count", "99"},
	{"limit", "30"},
	{"quota", "30"},
	{"minimum", "150"},
	{"maximum", "300"},
	{"password", "999"},
	{NULL, NULL}};

	Load("~bayonne/mailbox");
	Load(defkeys);
}

KeyMemory::KeyMemory() :
Keydata("/bayonne/memory")
{
	static KEYDEF defpaths[] = {
	{"symbols", "64"},
	{"page", "1024"},
	{NULL, NULL}};

	Load(defpaths);
}

int KeyThreads::priResolver(void)
{
	const char *cp = getLast("resolver");
	if(cp)
		return atoi(cp);

	return 0;
}

int KeyThreads::getResolver(void)
{
	char buf[32];
	int cnt;
	const char *cp = getLast("resolver");
	
	if(!cp)
		return 0;

	strcpy(buf, cp);
	cp = strchr(buf, ',');
	if(cp)
	{
		++cp;
		while(*cp == ' ' || *cp == '\t')
			++cp;
		return atoi(cp);
	}
	return 15;
}

int KeyThreads::getServices(void)
{
	char buf[32];
	int cnt;

	strcpy(buf, getLast("services"));
	char *cp = strchr(buf, ',');
	if(cp)
	{
		++cp;
		while(*cp == ' ' || *cp == '\t')
			++cp;
		cnt = atoi(cp);
		if(cnt > 0)
			return cnt;
	}
	return 1;
}

int KeyThreads::getGateways(void)
{
	char buf[32];

	strcpy(buf, getLast("gateways"));
	char *cp = strchr(buf, ',');
	if(cp)
	{
		++cp;
		if(*cp == ' ' || *cp == '\t')
			++cp;
		return atoi(cp);
	}
	else
		return 1;
}

Plugins::Plugins() :
Keydata("/bayonne/plugins")
{
	static KEYDEF defplugs[] = {
		{"database", "default"},
		{NULL, NULL}};

	Load("~bayonne/plugins");
	Load(defplugs);

	if(getLast("driver"))
		return;

#if defined(HAVE_LINUX_TELEPHONY_H)
	setValue("driver", "phonedev");
#elif defined(have_montecarlo_h) || defined(HAVE_MONTECARLO_H)
	setValue("driver", "pika");
#else
	setValue("driver", "dummy");
#endif
	
}

Plugins::~Plugins()
{
	dynunload();
}

char *Plugins::getDriverName(void)
{
	static char name[33];

	const char *drv = getLast("driver");
	char *cp = strrchr(drv, '/');
	if(cp)
		++cp;
	else
		cp = (char *)drv;
	strncpy(name, cp, 32);
	name[33] = 0;
	cp = strrchr(name, '.');
	if(!cp)
		return name;

	if(!strcmp(cp, ".ivr"))
		*cp = 0;
	return name;
}

void Plugins::loadManagers(void)
{
	char path[256];
	char list[412];
	char *cp;

	cp = (char *)getLast("managers");
	if(!cp)
		return;

	strcpy(list, cp);
	cp = strtok(list, " \t\n;,");
	while(cp)
	{
		if(!*cp)
			break;

		if(*cp == '/')
			path[0] = 0;
		else
		{
			strcpy(path, keypaths.getLibpath());
			strcat(path, LIB_VERSION);
		}
		strcat(path, cp);
		if(*cp != '/')
			strcat(path, ".nmi");

		new DSO(path);
		cp = strtok(NULL, " \t\n;,");
	}
}

void Plugins::loadAuditing(void)
{
	char path[256];
	char list[412];
	char *cp;

	cp = (char *)getLast("auditing");
	if(!cp)
		return;

	strcpy(list, cp);
	cp = strtok(list, " \t\n;,");
	while(cp)
	{
		if(!*cp)
			break;

		if(*cp == '/')
			path[0] = 0;
		else
		{
			strcpy(path, keypaths.getLibpath());
			strcat(path, LIB_VERSION);
		}
		strcat(path, cp);
		if(*cp != '/')
			strcat(path, ".aud");

		new DSO(path);
		cp = strtok(NULL, " \t\n;,");
	}
}

void Plugins::loadFunctions(void)
{
	char path[256];
	char list[512];
	char *cp;

	cp = (char *)getLast("functions");
	if(!cp)
		return;

	strcpy(list, cp);
	cp = strtok(list, " \t\n;,");
	while(cp)
	{
		if(!*cp)
			break;

		if(*cp == '/')
			path[0] = 0;
		else
		{
			strcpy(path, keypaths.getLibpath());
			strcat(path, LIB_VERSION);
		}
		strcat(path, cp);
		if(*cp != '/')
			strcat(path, ".fun");
		new DSO(path);
		cp = strtok(NULL, " \t\n;,");
	}
}

void Plugins::loadModules(void)
{
	char path[256];
	char list[512];
	char *cp;

	cp = (char *)getLast("modules");
	if(!cp)
		return;

	strcpy(list, cp);
	cp = strtok(list, " \t\n;,");
	while(cp)
	{
		if(!*cp)
			break;

		if(*cp == '/')
			path[0] = 0;
		else
		{
			strcpy(path, keypaths.getLibpath());
			strcat(path, LIB_VERSION);
		}
		strcat(path, cp);
		if(*cp != '/')
			strcat(path, ".mod");
		new DSO(path);
		cp = strtok(NULL, " \t\n;,");
	}
}

void Plugins::loadTGI(void)
{
	char path[256];
	char list[512];
	char *cp;

	cp = (char *)getLast("tgi");
	if(!cp)
		return;

	strcpy(list, cp);
	cp = strtok(list, " \t\n;,");
	while(cp)
	{
		if(!*cp)
			break;

		if(*cp == '/')
			path[0] = 0;
		else
		{
			strcpy(path, keypaths.getLibpath());
			strcat(path, LIB_VERSION);
		}
		strcat(path, cp);
		if(*cp != '/')
			strcat(path, ".tgi");
		new DSO(path);
		cp = strtok(NULL, " \t\n;,");
	}
}


void Plugins::loadTranslators(void)
{
	char path[256];
	char list[512];
	char *cp;

	cp = (char *)getLast("languages");
	if(!cp)
		return;

	strcpy(list, cp);
	cp = strtok(list, " \t\n;,");
	while(cp)
	{
		if(!*cp)
			break;

		if(*cp == '/')
			path[0] = 0;
		else
		{
			strcpy(path, keypaths.getLibpath());
			strcat(path, LIB_VERSION);
		}

		strcat(path, cp);
		if(*cp != '/')
			strcat(path, ".tts");
		new DSO(path);
		cp = strtok(NULL, " \t\n;,");
	}
}

void Plugins::loadExtensions(void)
{
	char path[256];
	strcpy(path, keypaths.getLibpath());
	strcat(path, LIB_VERSION);
	strcat(path, "bayonne.ext");
	if(canAccess(path))
		new DSO(path);
}

void Plugins::loadDatabase(void)
{
	char path[256];
	char *d = (char *)getLast("database");

	if(*d != '/')
	{
		strcpy(path, keypaths.getLibpath());
		strcat(path, LIB_VERSION);
		strcat(path, d);
		strcat(path, ".dba");			
		d = path;
	}
	if(canAccess(d))
		new DSO(d);
}


void Plugins::loadDebug(void)
{
	char path[256];
	char *d = (char *)getLast("debug");

	if(!d)
	{
		new Debug();
		return;
	}

	if(*d != '/')
	{
		strcpy(path, keypaths.getLibpath());
		strcat(path, LIB_VERSION);
		strcat(path, d);
		strcat(path, ".dbg");			
		d = path;
	}
	new DSO(d);
}
	
DSO *Plugins::loadStatMon(void)
{
	char path[256];
	char *s = (char *)getLast("statmon");

	if(!s)
		return NULL;

	if(*s != '/')
	{
		strcpy(path, keypaths.getLibpath());
		strcat(path, LIB_VERSION);
		strcat(path, s);
		strcat(path, ".mon");
		s = path;
	}
	return new DSO(s);
}

DSO *Plugins::loadDriver(void)
{
	char path[256];
	char *drv = (char *)getLast("driver");
	
	if(*drv != '/')
	{
		strcpy(path, keypaths.getLibpath());
		strcat(path, LIB_VERSION);
		strcat(path, drv);
		strcat(path, ".ivr");
		drv = path;
	}
	
	return new DSO(drv);
}

KeyTones::KeyTones() :
Keydata("/bayonne/tones")
{
	char *tones = (char *)getLast("tones");
	int v1, v2, v3;
	char *tone;

	tones = strtok(tones, " \t\n");
	while(tones)
	{
		v1 = v2 = v3 = 0;
		tone = (char *)getLast(tones);
		if(!tone)
		{
			tones = strtok(NULL, " \t\n");
			continue;
		}
		sscanf(tone, "%d %d %d", &v1, &v2, &v3);
		if(v3)
			new phTone(tones, v3, v1, v2);
		else
			new phTone(tones, v2, v1);
		tones = strtok(NULL, " \t\n");
	}
}

TrunkGroup *TrunkGroup::first = NULL;

TrunkGroup::TrunkGroup(char *name) :
Keydata("/bayonne/trunks"), CallStat()
{
	KEYDEF keys[] = {
		{"answer", "1"},
		{"hangup", "100"},
		{"ringtime", "6"},
		{"flash", "200"},
		{"dialtone", "800"},
		{"dialspeed", "160"},
		{"volume", "80"},
		{NULL, NULL}};

	int count = driver->getTrunkCount();
	int i;
	char *cp;
	char namebuf[65];
	
	schedule[0] = 0;
	Load("~bayonne/trunks");

	if(name)
	{
		strcpy(namebuf, "/bayonne/");
		strcat(namebuf, name);
		strcat(namebuf, "-trunks");
		Load(namebuf);
		*namebuf = '~';
		Load(namebuf);
		cp = (char *)getLast("trunks");
		if(cp)
			cp = strtok(cp, " ,;\t\n");
		while(cp)
		{
			i = atoi(cp);
			if(i >= 0 && i < count)
				driver->groups[i] = this;
			cp = strtok(NULL, " ,;\t\n");
		}
	}
	else
	{
		for(i = 0; i < count; ++i) {
			driver->groups[i] = this;
		}
		name = "*";
	}

	setValue("name", name);

	Load(keys);
	next = NULL;

	if(first)
		first->next = this;
	else
		first = this;
}	

const char *TrunkGroup::getSchedule(char *buf)
{
	const char *scr = getLast("script");	// check for .conf override
	if(scr)
		return scr;

	scheduler.ReadLock();
	if(schedule[0])
		strcpy(buf, schedule);
	else if(this == first)
		strcpy(buf, keyserver.getDefault());
	else
		strcpy(buf, getName());
	scheduler.Unlock();
	return buf;
}

void TrunkGroup::setSchedule(const char *str)
{
	if(!str)
		str = "";

	scheduler.WriteLock();
	strncpy(schedule, str, sizeof(schedule) - 1);
	schedule[sizeof(schedule) - 1] = 0;
	scheduler.Unlock();
}

phTone *phTone::first = NULL;

int phTone::ulaw[256] = { 
	0,0,1,1,2,2,2,2,3,3,3,3,3,3,3,3,
        4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,4,
        5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
        5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
        6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,
        6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,
        6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,
        6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,
        7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
        7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
        7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
        7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
        7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
        7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
        7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
        7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7};

phTone::phTone(const char *n, timeout_t dur, unsigned f) 
{
	int i, sign, exponent, mantissa;
	int sample;

	next = first;
	first = this;

        duration = dur;
        samples = new unsigned char[dur * 8];
        freq1 = f;
        freq2 = 0;
        strncpy(name, n, 32);
        name[32] = 0; 

	double freq = (f * M_PI * 2) / 8000.;
	double pos = 0;

	for(i = 0; i < dur * 8; ++i)
	{
		sample = (int)(sin(pos) * 30000);
		pos += freq;
		sign = (sample >> 8) & 0x80;
		if(sign != 0) sample = -sample;
		sample += 0x84;
		exponent = ulaw[(sample >> 7) & 0xff];
		mantissa = (sample >> (exponent + 3)) & 0x0f;
		samples[i] = -(sign | (exponent << 4) | mantissa);
		if(!samples[i])
			samples[i] = 0x02;
	}
}

phTone::phTone(const char *n, timeout_t dur, unsigned f1, unsigned f2) 
{
	int i, sign, exponent, mantissa;
	int sample;

	next = first;
	first = this;

        duration = dur;
        samples = new unsigned char[dur * 8];
        freq1 = f1;
        freq2 = f2;
        strncpy(name, n, 32);
        name[32] = 0; 

	double fa1 = (f1 * M_PI * 2) / 8000.;
	double fa2 = (f2 * M_PI * 2) / 8000.;
	double pos1 = 0, pos2 = 0;

	for(i = 0; i < dur * 8; ++i)
	{
		sample = (int)((sin(pos1) + sin(pos2)) * 15000);
		pos1 += fa1;
		pos2 += fa2;
		sign = (sample >> 8) & 0x80;
		if(sign != 0) sample = -sample;
		sample += 0x84;
		exponent = ulaw[(sample >> 7) & 0xff];
		mantissa = (sample >> (exponent + 3)) & 0x0f;
		samples[i] = -(sign | (exponent << 4) | mantissa);
		if(!samples[i])
			samples[i] = 0x02;
	}
}

 
phTone::~phTone()
{
	if(samples)
		delete samples;
}


void phTone::Clear(void)
{
	if(samples)
		delete samples;

	samples = NULL;
}

phTone *getphTone(const char *name) 
{
	phTone *tone = phTone::first;
	while(tone)
	{
		if(!stricmp(tone->name, name))
			break;
		tone = tone->next;
	}
	return tone;
}		

KeyServer keyserver;
KeyPaths keypaths;
KeyLocal keylocal;
KeyMailbox keymailbox;
KeyTones keytones;
KeyThreads keythreads;
KeyMemory keymemory;
Plugins plugins;
Scheduler scheduler;

ScriptSymbol Trunk::globals(keymemory.getSymbolSize(), keymemory.getPageSize());

