// Copyright (C) 2000 Open Source Telecom Corporation.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include "driver.h"
#include <sys/ioctl.h>
#include <sys/stat.h>

PhonedevPlay::PhonedevPlay(PhonedevTrunk *trunk) :
Service((Trunk *)trunk, keythreads.priAudio()), AudioFile()
{
	reset = false;
	driver = trunk->getDriver();
	dev = trunk->getDevice();
}

PhonedevPlay::~PhonedevPlay()
{
	char buffer[12];
	Terminate();
	sprintf(buffer, "%d", getPosition());
	trunk->setSymbol("played", buffer);
	Close();
	if(reset)
	{
		ioctl(dev, PHONE_PLAY_STOP); 
		dspReset();
	}
}

char *PhonedevPlay::getContinuation(void)
{
	return getPlayfile();
}

void PhonedevPlay::Initial(void)
{
	int codec, framing, len, count = 0;
	char buffer[32];
	char *fn = getPlayfile();
	struct stat ino;
	struct tm *dt;
	struct tm tbuf;
	char *ann;
	
	trunk->getName(buffer);
	ioctl(dev, PHONE_PLAY_VOLUME, data->play.volume * 10);
	stat(fn, &ino);
	Open(fn);	
	if(!isOpen())
	{
		slog(SLOG_ERROR) << fn << ": cannot open" << endl;
		Failure();
	}

	dt = localtime_r(&ino.st_ctime, &tbuf);
	sprintf(buffer, "%04d%02d%02d,%02d%02d%02d",
		dt->tm_year + 1900, dt->tm_mon + 1, dt->tm_mday,
		dt->tm_hour, dt->tm_min, dt->tm_sec);
	trunk->setSymbol("created", buffer);
	ann = getAnnotation();
	if(ann)
		trunk->setSymbol("annotation", ann);
	
	switch(getEncoding())
	{
	case MULAW_AUDIO_ENCODING:
		codec = ULAW;
		framing = 20;
		samples = 24000;
		break;
	case ALAW_AUDIO_ENCODING:
		codec = ALAW;
		samples = 24000;
		len = 80;
	default:
		slog(SLOG_ERROR) << buffer << ": unsupported codec required" << endl;
		Failure();
	}

	reset = true;
	ioctl(dev, PHONE_PLAY_DEPTH, phivr.getAudioBuffers());

	switch(driver)
	{
#ifdef	IXJ_PHONEDEV
	case ixj_driver:
		if(framing == 20)
			framing = 30;
		break;
#endif
	}

	while(framing != ioctl(dev, PHONE_FRAME, framing) && count++ < 5)
		Sleep(20);

	if(count > 4)
	{
		slog(SLOG_ERROR) << buffer << ": cannot set framing" << endl;
 		Failure();
	}

	if(ioctl(dev, PHONE_PLAY_CODEC, codec))
	{
		slog(SLOG_ERROR) << buffer << ": cannot set codec" << endl;
		Failure();
	}
	samples = (samples * framing) / 3000;
	bufsize = tobytes(getEncoding(), samples);
}

void PhonedevPlay::Run(void)
{
	char name[33];
	char buffer[bufsize];
	audioerror_t status = AUDIO_SUCCESS;
	int len = 0;

	trunk->getName(name);
	if(ioctl(dev, PHONE_PLAY_START))
	{
		slog(SLOG_ERROR) << name << ": failed channel playback" << endl;
		Failure();
	}

	while(status == AUDIO_SUCCESS && !stopped)
	{	
		status = getSamples(buffer, samples);
		if(status == AUDIO_READ_INCOMPLETE || status == AUDIO_SUCCESS)
			len = ::write(dev, buffer, bufsize);	
	}
	if(status == AUDIO_READ_FAILURE)
		slog(SLOG_ERROR) << name << ": failed playback" << endl;
	ioctl(dev, PHONE_PLAY_STOP);
	reset = false;
	Success();
}

bool PhonedevTrunk::playHandler(TrunkEvent *event)
{
	switch(event->id)
	{
	case TRUNK_SERVICE_SUCCESS:
		TrunkSignal(TRUNK_SIGNAL_STEP);
		handler = &PhonedevTrunk::stepHandler;
		return true;
	case TRUNK_SERVICE_FAILURE:
		setSymbol("error", "play-failed");
		TrunkSignal(TRUNK_SIGNAL_ERROR);
		handler = &PhonedevTrunk::stepHandler;
		return true;
	case TRUNK_ENTER_STATE:
		debug->DebugState(this, "play");
		flags.dsp = DSP_MODE_VOICE;
		status[id] = 'p';
		if(!flags.offhook)
		{
			setHookState(true);
			setTimer(250);
			return true;
		}
	case TRUNK_TIMER_EXPIRED:
		setDTMFDetect();
		thread = new PhonedevPlay(this);
		thread->Start();
		return true;
	}
	return false;
}

