# GNU Solfege - eartraining for GNOME
# Copyright (C) 2000, 2001  Tom Cato Amundsen
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import gtk, gnome.uiconsts
import abstract, const, gu, mpd, soundcard
from i18n import _
import random, os
import string
import cfg

RHYTHMS = ("c4", "c8 c8", "c16 c16 c16 c16", "c8 c16 c16",
           "c16 c16 c8", "c16 c8 c16", "c8. c16", "c16 c8.",
           "r4", "r8c8", "r8 c16 c16", "r16 c16 c8", "r16c8c16",
           "r16 c16 c16 c16", "r8 r16 c16", "r16 c8.",
           "c12 c12 c12", "r12 c12 c12",
           "c12 r12 c12", "c12 c12 r12", "r12 r12 c12", "r12 c12 r12",
           )


class Teacher(abstract.Teacher):
    OK = 0
    ERR_PICKY = 1
    ERR_NO_ELEMS = 2
    def __init__(self, exname, app):
        abstract.Teacher.__init__(self, exname, app)
    def new_question(self):
        """returns:
               Teacher.ERR_PICKY : if the question is not yet solved and the
                                   teacher is picky (== you have to solve the
                                   question before a new is asked).
               Teacher.OK : if a new question was created.
        """
        if self.m_timeout_handle:
            gtk.timeout_remove(self.m_timeout_handle)
            self.m_timeout_handle = None

        if self.get_bool('config/picky_on_new_question') \
                 and self.q_status in [const.QSTATUS_NEW, const.QSTATUS_WRONG]:
             return Teacher.ERR_PICKY

        self.q_status = const.QSTATUS_NO

        norest_v = []
        v = []
        for x in range(len(RHYTHMS)):
            if self.get_bool("rhythm_element_%i" % x):
                if not (RHYTHMS[x][0] == "r"
                        and self.get_bool("not_start_with_rest")):
                    norest_v.append(x)
                v.append(x)
        if not v:
            return Teacher.ERR_NO_ELEMS
        self.m_question = [random.choice(norest_v)]
        for x in range(1, self.get_int("num_beats")):
            self.m_question.append(random.choice(v))
        self.q_status = const.QSTATUS_NEW
        return Teacher.OK
    def get_music_string(self):
        s = ""
        for x in range(self.get_int("count_in")):
            s = s + "d4 "
        for k in self.m_question:
            s = s + RHYTHMS[k] + " "
        return r"\staff{%s}" % s
    def play_question(self):
        if self.q_status == const.QSTATUS_NO:
            return
        score = mpd.parser.parse_to_score_object(self.get_music_string())
        track = score.get_midi_events_with_channel(9, cfg.get_int('config/preferred_instrument_velocity'))[0]
        track.prepend_bpm(self.get_int("bpm"))
        track.replace_note(mpd.notename_to_int("c"),
                           self.get_int("rhythm_perc"), 9)
        track.replace_note(mpd.notename_to_int("d"),
                           self.get_int("countin_perc"), 9)
        soundcard.synth.play_track(track)
        score.help__del__()
    def guess_answer(self, a):
        assert self.q_status in [const.QSTATUS_NEW, const.QSTATUS_WRONG]
        v = []
        for idx in range(len(self.m_question)):
           v.append(self.m_question[idx] == a[idx])
        if filter(lambda a: a == 0, v) == []:
            self.q_status = const.QSTATUS_SOLVED
            self.maybe_auto_new_question()
            self.m_app.play_happy_sound()
            return 1
        else:
            self.q_status = const.QSTATUS_WRONG
            self.m_app.play_sad_sound()

def load_xpm(parent, filename):
    pix, mask = gtk.create_pixmap_from_xpm(parent, None,
                                           os.path.join("xpm", filename))
    return gtk.GtkPixmap(pix, mask)

def load_rhythm_xpm(parent, idx):
    return load_xpm(parent, "rhythm-" + reduce(lambda x, y:\
              x+y, string.split(RHYTHMS[idx])) + ".xpm")

class RhythmViewer(gtk.GtkFrame):
    def __init__(self, parent):
        gtk.GtkFrame.__init__(self)
        self.set_shadow_type(gtk.SHADOW_IN)
        self.g_parent = parent
        self.g_box = gtk.GtkHBox()
        self.g_box.show()
        self.g_box.set_spacing(gnome.uiconsts.PAD_SMALL)
        self.g_box.set_border_width(gnome.uiconsts.PAD)
        self.add(self.g_box)
        self.m_data = []
        # the number of rhythm elements the viewer is supposed to show
        self.m_num_beats = 0
        self.g_face = None
        self.__timeout = None
    def set_num_beats(self, i):
        self.m_num_beats = i
    def clear(self):
        self.g_box.foreach(lambda o: o.destroy())
        self.m_data = []
    def create_holders(self):
        """
        create those |__| that represents one beat
        """
        if self.__timeout:
            gtk.timeout_remove(self.__timeout)
        self.clear()
        for x in range(self.m_num_beats):
            self.add_elem("holder.xpm")
        self.m_data = []
    def clear_wrong_part(self):
        """When the user have answered the question, this method is used
        to clear all but the first correct elements."""
        # this assert is always true because if there is no rhythm element,
        # then there is a rhythm holder ( |__| )
        assert self.m_num_beats == len(self.g_parent.m_t.m_question)
        self.g_face.destroy()
        self.g_face = None
        for n in range(self.m_num_beats):
            if self.m_data[n] != self.g_parent.m_t.m_question[n]:
                break
        for x in range(n, len(self.g_box.children())):
            self.g_box.children()[n].destroy()
        self.m_data = self.m_data[:n]
        for x in range(n, self.m_num_beats):
            self.add_elem("holder.xpm")
    def add_elem(self, fn):
        pix = load_xpm(self, fn)
        pix.show()
        self.g_box.pack_start(pix, gtk.FALSE)
    def add_rhythm_element(self, i):
        assert len(self.m_data) <= self.m_num_beats
        if len(self.g_box.children()) >= self.m_num_beats:
            self.g_box.children()[self.m_num_beats-1].destroy()
        pix = load_rhythm_xpm(self, i)
        pix.show()
        self.g_box.pack_start(pix, gtk.FALSE)
        self.g_box.reorder_child(pix, len(self.m_data))
        self.m_data.append(i)
    def backspace(self):
        if len(self.m_data) > 0:
            if self.g_face:
                self.g_face.destroy()
                self.g_face = None
            self.g_box.children()[len(self.m_data)-1].destroy()
            pix = load_xpm(self, "holder.xpm")
            pix.show()
            self.g_box.pack_start(pix, gtk.FALSE)
            del self.m_data[-1]
    def mark_wrong(self, idx):
        box = gtk.GtkVBox()
        r = self.g_box.children()[idx]
        self.g_box.children()[idx].reparent(box)
        pix = load_xpm(self, "rhythm-wrong.xpm")
        box.pack_start(pix)
        self.g_box.pack_start(box, gtk.FALSE)
        self.g_box.reorder_child(box, idx)
        box.show_all()
    def len(self):
        "return the number of rhythm elements currently viewed"
        return len(self.m_data)
    def sad_face(self):
        pix = load_xpm(self, "sadface-32x32.xpm")
        pix.show()
        self.g_face = ev = gtk.GtkEventBox()
        ev.connect('button_press_event', self.on_sadface_event)
        ev.show()
        ev.add(pix)
        self.g_box.pack_start(ev, gtk.FALSE)
    def happy_face(self):
        pix = load_xpm(self, "happyface-32x32.xpm")
        pix.show()
        self.g_face = ev = gtk.GtkEventBox()
        ev.connect('button_press_event', self.on_happyface_event)
        ev.show()
        ev.add(pix)
        self.g_box.pack_start(ev, gtk.FALSE)
    def on_happyface_event(self, obj, event):
        if event.type == gtk.GDK.BUTTON_PRESS and event.button == 1:
            self.g_parent.new_question()
    def on_sadface_event(self, obj, event):
        if event.type == gtk.GDK.BUTTON_PRESS and event.button == 1:
            self.clear_wrong_part()
    def flash(self, s):
        self.clear()
        l = gtk.GtkLabel(s)
        l.set_name("Feedback")
        l.set_alignment(0.0, 0.5)
        l.show()
        self.g_box.pack_start(l, gtk.TRUE, gtk.TRUE)
        self.__timeout = gtk.timeout_add(const.NORMAL_WAIT, self.unflash)
    def unflash(self, *v):
        self.__timeout = None
        self.clear()


class Gui(abstract.Gui):
    def __init__(self, teacher, window):
        abstract.Gui.__init__(self, teacher, window)
        self.m_key_bindings = {'new_ak': self.new_question,
                               'repeat_ak': self.m_t.play_question,
                               'give_up_ak': self.show_answer,
                               'backspace_ak': self.on_backspace}
        hbox = gu.bHBox(self.practise_box, gtk.FALSE)
        for i in range(0, 8):
            hbox.pack_start(self.xpmbutton(i), gtk.FALSE)
        hbox = gu.bHBox(self.practise_box, gtk.FALSE)
        for i in range(8, 16):
            hbox.pack_start(self.xpmbutton(i), gtk.FALSE)
        hbox = gu.bHBox(self.practise_box, gtk.FALSE)
        for i in range(16, 22):
            hbox.pack_start(self.xpmbutton(i), gtk.FALSE)
        #-------
        self.practise_box.pack_start(gtk.GtkHBox(), gtk.FALSE, 
                                     padding=gnome.uiconsts.PAD_SMALL)
        self.g_rhythm_viewer = RhythmViewer(self)
        self.g_rhythm_viewer.set_usize(-1, 52)
        self.g_rhythm_viewer.create_holders()
        self.practise_box.pack_start(self.g_rhythm_viewer, gtk.FALSE)

        # action area
        self.g_new = gu.bButton(self.action_area, _("New"), self.new_question)
        self.g_repeat = gu.bButton(self.action_area, _("Repeat"),
                          lambda btn, self=self: self.m_t.play_question())
        self.g_repeat.set_sensitive(gtk.FALSE)
        self.g_show = gu.bButton(self.action_area, _("Give up"), self.show_answer)
        self.g_show.set_sensitive(gtk.FALSE)

        self.g_backspace = gu.bButton(self.action_area, _("Backspace"),
                     self.on_backspace)
        self.g_backspace.set_sensitive(gtk.FALSE)
        self.practise_box.show_all()
        ##############
        # config_box #
        ##############
        frame = gtk.GtkFrame(_("Rhythms to use in question"))
        self.config_box.pack_start(frame, gtk.FALSE)
        vbox = gtk.GtkVBox()
        vbox.set_border_width(gnome.uiconsts.PAD_SMALL)
        frame.add(vbox)
        hbox = gu.bHBox(vbox, gtk.FALSE)
        for i in range(0, 8):
            hbox.pack_start(self.xpmcheckbutton(i), gtk.FALSE)
        hbox = gu.bHBox(vbox, gtk.FALSE)
        for i in range(8, 16):
            hbox.pack_start(self.xpmcheckbutton(i), gtk.FALSE)
        hbox = gu.bHBox(vbox, gtk.FALSE)
        for i in range(16, 22):
            hbox.pack_start(self.xpmcheckbutton(i), gtk.FALSE)
        #--------    
        self.config_box.pack_start(gtk.GtkHBox(), gtk.FALSE,
                                   padding=gnome.uiconsts.PAD_SMALL)

        table = gtk.GtkTable(2, 4, gtk.FALSE)
        self.config_box.pack_start(table, gtk.FALSE)
        ###
        label = gtk.GtkLabel(_("Number of beats in question:"))
        label.set_alignment(1.0, 0.5)
        table.attach(label, 0, 1, 0, 1, xpadding=gnome.uiconsts.PAD_SMALL,
                      xoptions=gtk.FILL)
        table.attach(gu.nSpinButton(self.m_exname, "num_beats",
                     gtk.GtkAdjustment(4, 1, 100, 1, 10)),
                     1, 2, 0, 1, xoptions=gtk.FILL)
        self.g_rhythm_perc_combo = combo = gu.PercussionNameCombo(self.m_exname, "rhythm_perc", "Side Stick")
        table.attach(combo, 2, 3, 0, 1, xoptions=gtk.FILL)
        label = gtk.GtkLabel(_("Count in before question:"))
        label.set_alignment(1.0, 0.5)
        table.attach(label, 0, 1, 1, 2, xpadding=gnome.uiconsts.PAD_SMALL,
                     xoptions=gtk.FILL)
        table.attach(gu.nSpinButton(self.m_exname, "count_in",
                     gtk.GtkAdjustment(2, 0, 10, 1, 10)),
                     1, 2, 1, 2, xoptions=gtk.FILL)
        combo = gu.PercussionNameCombo(self.m_exname, "countin_perc",
                                       "Claves")
        table.attach(combo, 2, 3, 1, 2, xoptions=gtk.FILL)
        #-----
        self.config_box.pack_start(gtk.GtkHBox(), gtk.FALSE,
                                   padding=gnome.uiconsts.PAD_SMALL)
        #------
        hbox = gu.bHBox(self.config_box, gtk.FALSE)
        hbox.set_spacing(gnome.uiconsts.PAD_SMALL)
        hbox.pack_start(gu.nCheckButton(self.m_exname,
                 "not_start_with_rest",
                 _("Don't start the question with a rest")), gtk.FALSE)
        sep = gtk.GtkVSeparator()
        hbox.pack_start(sep, gtk.FALSE)
        hbox.pack_start(gtk.GtkLabel(_("Beats per minute:")), gtk.FALSE)
        spin = gu.nSpinButton(self.m_exname, 'bpm',
                 gtk.GtkAdjustment(60, 20, 240, 1, 10))
        hbox.pack_start(spin, gtk.FALSE)
        self._add_auto_new_question_gui(self.config_box)
        self.config_box.show_all()
    def xpmbutton(self, i):
        "used by the constructor"
        pix = load_rhythm_xpm(self, i)
        pix.show()
        btn = gtk.GtkButton()
        btn.add(pix)
        btn.show()
        btn.connect('clicked', self.guess_element, i)
        return btn
    def xpmcheckbutton(self, i):
        pix = load_rhythm_xpm(self, i)
        pix.show()
        btn = gu.nCheckButton(self.m_exname, "rhythm_element_%i" % i, default_value=1)
        btn.add(pix)
        btn.show()
        return btn
    def on_backspace(self, widget=None):
        if self.m_t.q_status == const.QSTATUS_SOLVED:
           return
        self.g_rhythm_viewer.backspace()
        if not self.g_rhythm_viewer.m_data:
            self.g_backspace.set_sensitive(gtk.FALSE)
    def guess_element(self, sender, i):
        if self.m_t.q_status == const.QSTATUS_NO:
            self.g_rhythm_viewer.flash(_("Click 'New' to begin."))
            return
        if self.m_t.q_status == const.QSTATUS_SOLVED:
            return
        if self.g_rhythm_viewer.len() == len(self.m_t.m_question):
            self.g_rhythm_viewer.clear_wrong_part()
        self.g_rhythm_viewer.add_rhythm_element(i)
        if self.g_rhythm_viewer.len() == len(self.m_t.m_question):
            if self.m_t.guess_answer(self.g_rhythm_viewer.m_data):
                self.g_rhythm_viewer.happy_face()
                self.g_new.set_sensitive(gtk.TRUE)
                self.g_backspace.set_sensitive(gtk.FALSE)
                self.g_show.set_sensitive(gtk.FALSE)
            else:
                v = []
                for idx in range(len(self.m_t.m_question)):
                    v.append(self.m_t.m_question[idx] == self.g_rhythm_viewer.m_data[idx])
                for x in range(len(v)):
                    if not v[x]:
                        self.g_rhythm_viewer.mark_wrong(x)
                self.g_rhythm_viewer.sad_face()
        else:
            self.g_backspace.set_sensitive(gtk.TRUE)
    def new_question(self, widget=None):
        g = self.m_t.new_question()
        if g == Teacher.OK:
            self.g_rhythm_viewer.set_num_beats(self.get_int('num_beats'))
            self.g_rhythm_viewer.create_holders()
            self.g_show.set_sensitive(gtk.TRUE)
            self.g_repeat.set_sensitive(gtk.TRUE)
            self.g_new.set_sensitive(
               not self.get_bool('config/picky_on_new_question'))
            self.m_t.play_question()
        elif g == Teacher.ERR_PICKY:
            self.g_rhythm_viewer.flash(_("You have to solve this question first."))
        elif g == Teacher.ERR_NO_ELEMS:
            self.g_rhythm_viewer.flash(_("You have to configure this exercise properly"))
        else:
            raise "NEVER REACH THIS"
    def on_start_practise(self):
        self.g_rhythm_viewer.flash(_("Click 'New' to begin."))
    def on_end_practise(self):
        self.g_new.set_sensitive(gtk.TRUE)
        self.g_repeat.set_sensitive(gtk.FALSE)
        self.g_show.set_sensitive(gtk.FALSE)
        self.g_rhythm_viewer.create_holders()
        self.m_t.end_practise()
    def show_answer(self, widget=None):
        if self.m_t.q_status == const.QSTATUS_NO:
            return
        self.g_rhythm_viewer.clear()
        for i in self.m_t.m_question:
            self.g_rhythm_viewer.add_rhythm_element(i)
        self.m_t.q_status = const.QSTATUS_SOLVED
        self.g_new.set_sensitive(gtk.TRUE)
        self.g_show.set_sensitive(gtk.FALSE)
        self.g_backspace.set_sensitive(gtk.FALSE)

