/* sha256.c
 *
 * The sha256 hash function.
 *
 * See http://csrc.nist.gov/publications/fips/fips180-2/fips180-2.pdf
 */

/* nettle, low-level cryptographics library
 *
 * Copyright (C) 2001 Niels Mller
 *  
 * The nettle library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 *
 * The nettle library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
 * License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with the nettle library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 */

/* Modelled after the sha1.c code by Peter Gutmann. */

#if HAVE_CONFIG_H
# include "config.h"
#endif

#include <assert.h>
#include <stdlib.h>
#include <string.h>

#include "sha.h"

#include "macros.h"
#include "nettle-write.h"

/* Generated by the shadata program. */
static const uint32_t
K[64] =
{
  0x428a2f98UL, 0x71374491UL, 0xb5c0fbcfUL, 0xe9b5dba5UL, 
  0x3956c25bUL, 0x59f111f1UL, 0x923f82a4UL, 0xab1c5ed5UL, 
  0xd807aa98UL, 0x12835b01UL, 0x243185beUL, 0x550c7dc3UL, 
  0x72be5d74UL, 0x80deb1feUL, 0x9bdc06a7UL, 0xc19bf174UL, 
  0xe49b69c1UL, 0xefbe4786UL, 0xfc19dc6UL, 0x240ca1ccUL, 
  0x2de92c6fUL, 0x4a7484aaUL, 0x5cb0a9dcUL, 0x76f988daUL, 
  0x983e5152UL, 0xa831c66dUL, 0xb00327c8UL, 0xbf597fc7UL, 
  0xc6e00bf3UL, 0xd5a79147UL, 0x6ca6351UL, 0x14292967UL, 
  0x27b70a85UL, 0x2e1b2138UL, 0x4d2c6dfcUL, 0x53380d13UL, 
  0x650a7354UL, 0x766a0abbUL, 0x81c2c92eUL, 0x92722c85UL, 
  0xa2bfe8a1UL, 0xa81a664bUL, 0xc24b8b70UL, 0xc76c51a3UL, 
  0xd192e819UL, 0xd6990624UL, 0xf40e3585UL, 0x106aa070UL, 
  0x19a4c116UL, 0x1e376c08UL, 0x2748774cUL, 0x34b0bcb5UL, 
  0x391c0cb3UL, 0x4ed8aa4aUL, 0x5b9cca4fUL, 0x682e6ff3UL, 
  0x748f82eeUL, 0x78a5636fUL, 0x84c87814UL, 0x8cc70208UL, 
  0x90befffaUL, 0xa4506cebUL, 0xbef9a3f7UL, 0xc67178f2UL, 
};

/* Initialize the SHA values */

void
sha256_init(struct sha256_ctx *ctx)
{
  /* Initial values, also generated by the shadata program. */
  static const uint32_t H0[_SHA256_DIGEST_LENGTH] =
  {
    0x6a09e667UL, 0xbb67ae85UL, 0x3c6ef372UL, 0xa54ff53aUL, 
    0x510e527fUL, 0x9b05688cUL, 0x1f83d9abUL, 0x5be0cd19UL, 
  };

  memcpy(ctx->state, H0, sizeof(H0));

  /* Initialize bit count */
  ctx->count_low = ctx->count_high = 0;
  
  /* Initialize buffer */
  ctx->index = 0;
}

#define SHA256_INCR(ctx) ((ctx)->count_high += !++(ctx)->count_low)

void
sha256_update(struct sha256_ctx *ctx,
	      unsigned length, const uint8_t *buffer)
{
  if (ctx->index)
    { /* Try to fill partial block */
      unsigned left = SHA256_DATA_SIZE - ctx->index;
      if (length < left)
	{
	  memcpy(ctx->block + ctx->index, buffer, length);
	  ctx->index += length;
	  return; /* Finished */
	}
      else
	{
	  memcpy(ctx->block + ctx->index, buffer, left);

	  _nettle_sha256_compress(ctx->state, ctx->block, K);
	  SHA256_INCR(ctx);
	  
	  buffer += left;
	  length -= left;
	}
    }
  while (length >= SHA256_DATA_SIZE)
    {
      _nettle_sha256_compress(ctx->state, buffer, K);
      SHA256_INCR(ctx);

      buffer += SHA256_DATA_SIZE;
      length -= SHA256_DATA_SIZE;
    }
  /* Buffer leftovers */
  /* NOTE: The corresponding sha1 code checks for the special case length == 0.
   * That seems supoptimal, as I suspect it increases the number of branches. */
  
  memcpy(ctx->block, buffer, length);
  ctx->index = length;
}

/* Final wrapup - pad to SHA1_DATA_SIZE-byte boundary with the bit pattern
   1 0* (64-bit count of bits processed, MSB-first) */

static void
sha256_final(struct sha256_ctx *ctx)
{
  uint32_t bitcount_high;
  uint32_t bitcount_low;
  int i;

  i = ctx->index;
  
  /* Set the first char of padding to 0x80.  This is safe since there is
     always at least one byte free */

  assert(i < SHA256_DATA_SIZE);
  ctx->block[i++] = 0x80;

  if (i > (SHA1_DATA_SIZE - 8))
    { /* No room for length in this block. Process it and
       * pad with another one */
      memset(ctx->block + i, 0, SHA256_DATA_SIZE - i);
      _nettle_sha256_compress(ctx->state, ctx->block, K);

      i = 0;
    }

  if (i < (SHA256_DATA_SIZE - 8))
    memset(ctx->block + i, 0, (SHA256_DATA_SIZE - 8) - i);

  /* There are 512 = 2^9 bits in one block */
  bitcount_high = (ctx->count_high << 9) | (ctx->count_low >> 23);
  bitcount_low = (ctx->count_low << 9) | (ctx->index << 3);

  /* This is slightly inefficient, as the numbers are converted to
     big-endian format, and will be converted back by the compression
     function. It's probably not worth the effort to fix this. */
  WRITE_UINT32(ctx->block + (SHA256_DATA_SIZE - 8), bitcount_high);
  WRITE_UINT32(ctx->block + (SHA256_DATA_SIZE - 4), bitcount_low);

  _nettle_sha256_compress(ctx->state, ctx->block, K);
}

void
sha256_digest(struct sha256_ctx *ctx,
	      unsigned length,
	      uint8_t *digest)
{
  assert(length <= SHA256_DIGEST_SIZE);

  sha256_final(ctx);
  _nettle_write_be32(length, digest, ctx->state);
  sha256_init(ctx);
}

/* sha224 variant. FIXME: Move to seperate file? */

void
sha224_init(struct sha256_ctx *ctx)
{
  /* Initial values. I's unclear how they are chosen. */
  static const uint32_t H0[_SHA256_DIGEST_LENGTH] =
  {
    0xc1059ed8, 0x367cd507, 0x3070dd17, 0xf70e5939,
    0xffc00b31, 0x68581511, 0x64f98fa7, 0xbefa4fa4,
  };

  memcpy(ctx->state, H0, sizeof(H0));

  /* Initialize bit count */
  ctx->count_low = ctx->count_high = 0;
  
  /* Initialize buffer */
  ctx->index = 0;
}

void
sha224_digest(struct sha256_ctx *ctx,
	      unsigned length,
	      uint8_t *digest)
{
  assert(length <= SHA224_DIGEST_SIZE);

  sha256_final(ctx);
  _nettle_write_be32(length, digest, ctx->state);
  sha224_init(ctx);
}
