/*-------------------------------------------------------------------------*/
/* GNU Prolog                                                              */
/*                                                                         */
/* Part  : Prolog Compiler                                                 */
/* File  : top_comp.c                                                      */
/* Descr.: compiler main (shell) program                                   */
/* Author: Daniel Diaz                                                     */
/*                                                                         */
/* Copyright (C) 1999 Daniel Diaz                                          */
/*                                                                         */
/* GNU Prolog is free software; you can redistribute it and/or modify it   */
/* under the terms of the GNU General Public License as published by the   */
/* Free Software Foundation; either version 2, or any later version.       */
/*                                                                         */
/* GNU Prolog is distributed in the hope that it will be useful, but       */
/* WITHOUT ANY WARRANTY; without even the implied warranty of              */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU        */
/* General Public License for more details.                                */
/*                                                                         */
/* You should have received a copy of the GNU General Public License along */
/* with this program; if not, write to the Free Software Foundation, Inc.  */
/* 59 Temple Place - Suite 330, Boston, MA 02111, USA.                     */
/*-------------------------------------------------------------------------*/
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <sys/stat.h>
#include <dirent.h>
#include <unistd.h>
#include <errno.h>
#include <ctype.h>
#include <sys/param.h>
#include <sys/types.h>

#include <sys/types.h>
#include <sys/wait.h>

#include "decode_hexa.c"
#include "copying.c"
#include "../EnginePl/config.h"

#ifdef M_ix86_cygwin
#include <process.h>
#endif




/*---------------------------------*/
/* Constants                       */
/*---------------------------------*/

#define MAX_SUB_DIRS               1024

#define MAX_FILES                  1024

#define CMD_LINE_MAX_OPT           4096
#define CMD_LINE_LENGTH            (MAXPATHLEN+CMD_LINE_MAX_OPT+1)

#define TEMP_FILE_PREFIX           GPLC

#define OBJ_FILE_OBJ_BEGIN         "obj_begin"
#define OBJ_FILE_OBJ_END           "obj_end"
#define OBJ_FILE_ALL_PL_BIPS       "all_pl_bips"
#define OBJ_FILE_ALL_FD_BIPS       "all_fd_bips"
#define OBJ_FILE_TOP_LEVEL         "top_level"
#define OBJ_FILE_DEBUGGER          "debugger"

#define EXE_FILE_PL2WAM            "pl2wam"
#define EXE_FILE_WAM2MA            "wam2ma"
#define EXE_FILE_MA2ASM            "ma2asm"
#define EXE_FILE_ASM               "as"
#define EXE_FILE_FD2C              "fd2c"
#define EXE_FILE_CC                CC
#define EXE_FILE_LINK              CC
#define EXE_FILE_STRIP             "strip"




#define FILE_PL                    0
#define FILE_WAM                   1
#define FILE_MA                    2
#define FILE_ASM                   3
#define FILE_OBJ                   4

#define FILE_FD                    5
#define FILE_C                     6
#define FILE_LINK                  7




#define PL_SUFFIX                  ".pl"
#define PL_SUFFIX_ALTERNATE        ".pro"
#define WAM_SUFFIX                 ".wam"
#define WBC_SUFFIX                 ".wbc"
#define MA_SUFFIX                  ".ma"
#define ASM_SUFFIX                 ".s"
#define FD_SUFFIX                  ".fd"
#define C_SUFFIX                   ".c"
#define C_SUFFIX_ALTERNATE         "|.C|.CC|.cc|.cxx|.c++|.cpp|"
#define OBJ_SUFFIX                 ".o"

#define CC_COMPILE_ONLY_OPT        "-c "
#define CC_INCLUDE_OPT             "-I"
#define CC_OUTPUT_OPT              "-o "
#define LINK_OUTPUT_OPT            "-o "




/*---------------------------------*/
/* Type Definitions                */
/*---------------------------------*/

typedef struct
    {
     char *name;
     char *suffix;
     int   type;
     char *work_name1;
     char *work_name2;
    }FileInf;


typedef struct
    {
     char *exe_name;
     char  opt[CMD_LINE_MAX_OPT];
     char *out_opt;
    }CmdInf;




/*---------------------------------*/
/* Global Variables                */
/*---------------------------------*/

char    *start_path;

int      devel_mode=0;
char    *devel_dir[MAX_SUB_DIRS];

FileInf  file[MAX_FILES];
int      nb_file=0;

int      stop_after=FILE_LINK;
int      verbose=0;
char    *file_name_out=NULL;

int      def_local_size =-1;
int      def_global_size=-1;
int      def_trail_size =-1;
int      def_cstr_size  =-1;
int      fixed_sizes=0;
int      needs_stack_file=0;

int      bc_mode     =0;
int      no_top_level=0;
int      min_pl_bips =0;
int      min_fd_bips =0;
int      no_debugger =0;
int      no_fd_lib   =0;
int      strip       =0;

int      no_decode_hex=0;

char     warn[1024]="";

char    *temp_dir=NULL;
int      no_del_temp_files=0;

CmdInf   cmd_pl2wam={ EXE_FILE_PL2WAM," ","-o " };
CmdInf   cmd_wam2ma={ EXE_FILE_WAM2MA," ","-o " };
CmdInf   cmd_ma2asm={ EXE_FILE_MA2ASM," ","-o " };
CmdInf   cmd_asm   ={ EXE_FILE_ASM,   " ","-o " };
CmdInf   cmd_fd2c  ={ EXE_FILE_FD2C,  " ","-o " };
CmdInf   cmd_cc    ={ EXE_FILE_CC,    " ",CC_OUTPUT_OPT };
CmdInf   cmd_link  ={ EXE_FILE_LINK,  " ",LINK_OUTPUT_OPT };

char    *cc_fd2c_flags=CFLAGS " ";




char    *suffixes[]=
           { PL_SUFFIX, WAM_SUFFIX, MA_SUFFIX, ASM_SUFFIX, OBJ_SUFFIX,
             FD_SUFFIX, C_SUFFIX, NULL };




/*---------------------------------*/
/* Function Prototypes             */
/*---------------------------------*/

char     *Search_Path           (char *file);

void      Init_Develop_Dir      (void);

void      Determine_Pathnames   (void);
void      Compile_Files         (void);
void      New_Work_File         (FileInf *f,int stage,int stop_after);
void      Free_Work_File2       (FileInf *f);
void      Compile_Cmd           (CmdInf *c,FileInf *f);
void      Link_Cmd              (void);
void      Exec_One_Cmd          (char *str,int no_decode_hex);
void      Delete_Temp_File      (char *name);

void      Find_File             (char *file,char *suff,char *file_path);

void      Fatal_Error           (char *format,...);
void      Parse_Arguments       (int argc,char *argv[]);
void      Display_Help          (void);


#define Record_Link_Warn_Option(i) sprintf(warn+strlen(warn),"%s ",argv[i])


#define Before_Cmd(cmd)                                                     \
     if (verbose)                                                           \
         fprintf(stderr,"%s\n",cmd);

#define After_Cmd(error)                                                    \
     if (error)                                                             \
         Fatal_Error("compilation failed");




char *last_opt;

#define Check_Arg(i,str)           (last_opt=str,strncmp(argv[i],str,strlen(argv[i]))==0)
#define Add_Last_Option(opt)       sprintf(opt+strlen(opt),"%s ",last_opt)
#define Add_Option(i,opt)          sprintf(opt+strlen(opt),"%s ",argv[i])




/*-------------------------------------------------------------------------*/
/* MAIN                                                                    */
/*                                                                         */
/*-------------------------------------------------------------------------*/
int main(int argc,char *argv[])

{
 static
 char  resolved[MAXPATHLEN];
 static
 char  buff[MAXPATHLEN];
 char *p;

 if ((start_path=getenv(ENV_VARIABLE))!=NULL)
     goto path_found;

 strcpy(buff,argv[0]);
 if (strcmp(buff+strlen(buff)-strlen(EXE_SUFFIX),EXE_SUFFIX)!=0)
     strcat(buff,EXE_SUFFIX);

 if (access(buff,X_OK)==0)
     start_path=buff;
  else
     if ((start_path=Search_Path(buff))==NULL)
         goto path_not_found;

#if defined(__unix__) || defined(__CYGWIN__)
 if (realpath(start_path,resolved+1)==NULL)           /* +1 for eventual @ */
     goto path_not_found;
#else
 strcpy(resolved+1,start_path);                       /* +1 for eventual @ */
#endif

 sprintf(buff,"/bin/%s",GPLC);
 if ((p=strstr(resolved+1,buff))!=NULL)
    {
     *p='\0';
     start_path=resolved+1;
     goto path_found;
    }         

 sprintf(buff,"/TopComp/%s",GPLC);
 if ((p=strstr(resolved+1,buff))!=NULL)
    {
     *p='\0';
     start_path=resolved;
     *start_path='@';                       /* to enforce development mode */
     goto path_found;
    }

path_not_found:
 Fatal_Error("cannot find the path for %s, set the environment variable %s",
             PROLOG_NAME,ENV_VARIABLE);

path_found:
 if (*start_path=='@')                                 /* development mode */
    {
     start_path++;
     devel_mode=1;
    }

 strcat(cmd_cc.opt,CFLAGS_MACHINE " " CFLAGS_REGS CC_COMPILE_ONLY_OPT);

 if (devel_mode)
     Init_Develop_Dir();
  else
     sprintf(cmd_cc.opt+strlen(cmd_cc.opt),"%s%s/include ",
             CC_INCLUDE_OPT,start_path);

 strcat(cmd_link.opt,LDFLAGS " ");

 Parse_Arguments(argc,argv);
 if (verbose)
     fprintf(stderr,"Path used: %s %s\n",start_path,
     (devel_mode) ? "(development mode)" : "");

 Compile_Files();

 return 0;
}




/*-------------------------------------------------------------------------*/
/* SEARCH_PATH                                                             */
/*                                                                         */
/*-------------------------------------------------------------------------*/
char *Search_Path(char *file)

{
#if defined(__unix__) || defined(__CYGWIN__)

 char *path=getenv("PATH");
 char *p;
 int   l;
 static
 char  buff[MAXPATHLEN];

 if (path==NULL)
     return NULL;

 p=path;
 for(;;)
    {
     if ((p=strchr(path,':'))!=NULL)
        {
         l=p-path;
         strncpy(buff,path,l);
        }
      else
        {
         strcpy(buff,path);
         l=strlen(buff);
        }

     buff[l++]='/';

     strcpy(buff+l,file);

     if (access(buff,X_OK)==0)
         return buff;

     if (p==NULL)
         break;

     path=p+1;
    }

 return NULL;
#else
 return search_path(file);
#endif
}




/*-------------------------------------------------------------------------*/
/* INIT_DEVELOP_DIR                                                        */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void Init_Develop_Dir(void)

{
 DIR            *dir;
 struct dirent  *cur_entry;
 char          **p;
 struct stat     info;
 static
 char            buff[MAXPATHLEN];


 dir=opendir(start_path);
 if (dir==NULL)
     Fatal_Error("Cannot access to %s",start_path);

 p=devel_dir;
 while((cur_entry=readdir(dir))!=NULL)
    {
     sprintf(buff,"%s/%s",start_path,cur_entry->d_name);
     if (*cur_entry->d_name!='.' && 
         stat(buff,&info)==0 && S_ISDIR(info.st_mode))
        {
         *p++=strdup(cur_entry->d_name);
         sprintf(cmd_cc.opt+strlen(cmd_cc.opt),"%s%s ",
                 CC_INCLUDE_OPT,buff);
        }
    }
 
 *p=NULL;

 closedir(dir);
}




/*-------------------------------------------------------------------------*/
/* COMPILE_FILES                                                           */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void Compile_Files(void)

{
 FileInf *f;
 int      stage;
 int      stage_end;
 int      l;
 FILE    *fd;


 if (stop_after<FILE_LINK)
    {
     if (*warn)
         fprintf(stderr,"link not done - ignored option(s): %s\n",warn);

     stage_end=stop_after;
     needs_stack_file=0;

     if (bc_mode)
        {
         suffixes[FILE_WAM]=WBC_SUFFIX;
         strcat(cmd_pl2wam.opt,"--wam-for-byte-code ");
        }
    }
  else
     stage_end=FILE_ASM;

 if (needs_stack_file)
    {
     f=file+nb_file;

     f->work_name2=NULL;
     New_Work_File(f,FILE_WAM,10000);              /* to create work_name2 */
     f->name=f->work_name2;
     f->suffix=f->name+strlen(f->name)-strlen(suffixes[FILE_MA]);
     f->type=FILE_MA;
     f->work_name1=f->name,
     f->work_name2=NULL;

     if (verbose)
         fprintf(stderr,"creating stack size file: %s\n",f->name);

     if ((fd=fopen(f->name,"wt"))==NULL)
         Fatal_Error("cannot open stack size file (%s)",f->name);

     if (def_local_size>=0)
         fprintf(fd,"long global def_local_size = %d\n",def_local_size);
     if (def_global_size>=0)
         fprintf(fd,"long global def_global_size = %d\n",def_global_size);
     if (def_trail_size>=0)
         fprintf(fd,"long global def_trail_size = %d\n",def_trail_size);
     if (def_cstr_size>=0)
         fprintf(fd,"long global def_cstr_size = %d\n",def_cstr_size);
     if (fixed_sizes)
         fprintf(fd,"long global fixed_sizes = 1\n");

     fclose(fd);
    }

 if (verbose)
     fprintf(stderr,"\n*** Compiling\n");


 for(f=file;f->name;f++)
    {
     if (verbose && 
         (f->type==FILE_FD || f->type==FILE_C || f->type<=stage_end))
         fprintf(stderr,"\n--- file: %s\n",f->name);

     if (f->type==FILE_FD && stop_after>=FILE_ASM)
        {
         stage=FILE_FD;                /* to generate the correct C suffix */
         New_Work_File(f,stage,(stop_after==FILE_FD) ? stop_after : 10000);
         Compile_Cmd(&cmd_fd2c,f);
         if (stop_after!=FILE_FD)
            {
             stage=FILE_ASM;         /* to generate the correct obj suffix */
             New_Work_File(f,stage,stop_after);
	     l=strlen(cmd_cc.opt);                   /* add fd2c C options */
	     strcpy(cmd_cc.opt+l,cc_fd2c_flags);
             Compile_Cmd(&cmd_cc,f);
	     cmd_cc.opt[l]='\0';                            /* remove them */
	    }
         goto free_work_file;
        }

     if (f->type==FILE_C && stop_after>=FILE_ASM && stop_after!=FILE_FD)
        {
         stage=FILE_ASM;         /* to generate the correct obj suffix */
         New_Work_File(f,stage,stop_after);
         Compile_Cmd(&cmd_cc,f);
         goto free_work_file;
        }

     if (f->type==FILE_FD || f->type==FILE_C || 
         stop_after==FILE_FD || f->type>stop_after)
        {
         fprintf(stderr,"unused input file: %s\n",f->name);
         continue;
        }

     for(stage=f->type;stage<=stage_end;stage++)
        {
         New_Work_File(f,stage,stop_after);
         switch(stage)
            {
             case FILE_PL:
                 Compile_Cmd(&cmd_pl2wam,f);
                 break;

             case FILE_WAM:
                 Compile_Cmd(&cmd_wam2ma,f);
                 break;

             case FILE_MA:
                 Compile_Cmd(&cmd_ma2asm,f);
                 if (needs_stack_file && f==file+nb_file && 
                     !no_del_temp_files)
                    {
                     if (verbose)
                         fprintf(stderr,"deleting stack size file\n");
                     Delete_Temp_File(f->name);
                    }

                 break;

             case FILE_ASM:
                 Compile_Cmd(&cmd_asm,f);
                 break;
            }
        }

free_work_file:
     Free_Work_File2(f);             /* to suppress last useless temp file */
    }

 if (stop_after<FILE_LINK)
     return;

 if (verbose)
     fprintf(stderr,"\n*** Linking\n\n");

 Link_Cmd();

                                         /* removing temp files after link */
 for(f=file;f->name;f++)
     if (f->work_name1!=f->name)
         Delete_Temp_File(f->work_name1);
}




/*-------------------------------------------------------------------------*/
/* NEW_WORK_FILE                                                           */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void New_Work_File(FileInf *f,int stage,int stop_after)

{
 static
 char  buff[MAXPATHLEN];
 char *p;

 if (stage<stop_after)                               /* intermediate stage */
    {
     p=tempnam(temp_dir,TEMP_FILE_PREFIX);
     sprintf(buff,"%s%s",p,suffixes[stage+1]);
     free(p);
    }
  else                                                      /* final stage */
     if (file_name_out)                       /* specified output filename */
         strcpy(buff,file_name_out);
      else
        {
         strcpy(buff,f->name);
         strcpy(buff+(f->suffix-f->name),suffixes[stage+1]);
 	}

 Free_Work_File2(f);
 f->work_name2=strdup(buff);
}




/*-------------------------------------------------------------------------*/
/* FREE_WORK_FILE2                                                         */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void Free_Work_File2(FileInf *f)

{
 if (f->work_name2!=NULL)
    {
     if (f->work_name1!=f->name)
         Delete_Temp_File(f->work_name1);

     f->work_name1=f->work_name2;
    }
}




/*-------------------------------------------------------------------------*/
/* COMPILE_CMD                                                             */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void Compile_Cmd(CmdInf *c,FileInf *f)

{
 static
 char buff[CMD_LINE_LENGTH];

 sprintf(buff,"%s%s%s%s %s",c->exe_name,c->opt,c->out_opt,
         f->work_name2,f->work_name1);

 Exec_One_Cmd(buff,1);
}




/*-------------------------------------------------------------------------*/
/* LINK_CMD                                                                */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void Link_Cmd(void)

{
 static
 char     file_out[MAXPATHLEN];
 static
 char     buff[CMD_LINE_LENGTH];
 FileInf *f;


 if (no_fd_lib)
     min_fd_bips=1;

 if (file_name_out==NULL)
    {
     f=file;
     strcpy(file_out,f->name);
     file_out[f->suffix-f->name]='\0';
     file_name_out=file_out;
    }

 sprintf(buff,"%s%s%s%s ",cmd_link.exe_name,cmd_link.opt,cmd_link.out_opt,
         file_name_out);

 Find_File(OBJ_FILE_OBJ_BEGIN,OBJ_SUFFIX,buff+strlen(buff));
 strcat(buff," ");

 for(f=file;f->name;f++)
     sprintf(buff+strlen(buff),"%s ",f->work_name1);

 if (!min_pl_bips)
    {
     Find_File(OBJ_FILE_ALL_PL_BIPS,OBJ_SUFFIX,buff+strlen(buff));
     strcat(buff," ");
    }

#ifndef NO_USE_FD_SOLVER
 if (!min_fd_bips)
    {
     Find_File(OBJ_FILE_ALL_FD_BIPS,OBJ_SUFFIX,buff+strlen(buff));
     strcat(buff," ");
    }
#endif

 if (!no_top_level)
    {
     Find_File(OBJ_FILE_TOP_LEVEL,OBJ_SUFFIX,buff+strlen(buff));
     strcat(buff," ");
    }

 if (!no_debugger)
    {
     Find_File(OBJ_FILE_DEBUGGER,OBJ_SUFFIX,buff+strlen(buff));
     strcat(buff," ");
    }

#ifndef NO_USE_FD_SOLVER
 if (!no_fd_lib)
    {
     Find_File(LIB_BIPS_FD,"",buff+strlen(buff));
     strcat(buff," ");

     Find_File(LIB_ENGINE_FD,"",buff+strlen(buff));
     strcat(buff," ");
    }
#endif

 Find_File(LIB_BIPS_PL,"",buff+strlen(buff));
 strcat(buff," ");

 Find_File(OBJ_FILE_OBJ_END,OBJ_SUFFIX,buff+strlen(buff));
 strcat(buff," ");

 Find_File(LIB_ENGINE_PL,"",buff+strlen(buff));
 strcat(buff," ");

#ifndef NO_USE_LINEDIT
 Find_File(LIB_LINEDIT,"",buff+strlen(buff));
 strcat(buff," ");
#endif

 strcat(buff,LDLIBS " ");

 Exec_One_Cmd(buff,no_decode_hex);

 if (strip && *EXE_FILE_STRIP!='\0')
    {
     sprintf(buff,"%s %s%s",EXE_FILE_STRIP,file_name_out,EXE_SUFFIX);
     Exec_One_Cmd(buff,1);
    }
}




/*-------------------------------------------------------------------------*/
/* EXEC_ONE_CMD                                                            */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void Exec_One_Cmd(char *cmd,int no_decode_hex)

{
#if defined(__unix__) || defined(__CYGWIN__)
 char *arg[CMD_LINE_MAX_OPT];
 char *p;
 int   i=0;
 int   error;
 int   pipe_out[2];
 FILE *fpipe;
 static
 char  buff[CMD_LINE_LENGTH];

 Before_Cmd(cmd)
 p=cmd;
 for(;;)
    {
     while(*p==' ')
         p++;

     if (*p=='\0')
         break;

     arg[i++]=p;

     while(*p!=' ' && *p!='\0')
         p++;

     if (*p=='\0')
         break;

     *p++='\0';
    }

 arg[i]=NULL;


 if (no_decode_hex==0)
    {
     if (pipe(pipe_out)<0)
         Fatal_Error("error trying to execute %s",cmd);
    }
#ifdef M_ix86_cygwin
  else
    {
     error=spawnvp(_P_WAIT,arg[0],(const char * const *) arg);
     if (error== -1)
        {
         fprintf(stderr,"cannot execute ");
         perror(arg[0]);
        }
     goto finish;
    }
#endif

 fflush(stdout);
 fflush(stderr);
 error=vfork();
 if (error== -1)
     Fatal_Error("cannot create a new process for %s",arg[0]);

 if (error==0)                                            /* child process */
    {
     if (no_decode_hex==0)
        {
         close(pipe_out[0]);
         if (pipe_out[1]!=1)
            {
             dup2(pipe_out[1],1);
             close(pipe_out[1]);
            }
         dup2(1,2);
        }
     execvp(arg[0],arg);                          /* only returns on error */
     fprintf(stderr,"cannot execute ");
     perror(arg[0]);
     exit(1);
    }

 if (no_decode_hex==0)
    {
     close(pipe_out[1]);
     fpipe=fdopen(pipe_out[0],"rt");

     if (fpipe==NULL)
         Fatal_Error("error trying to execute %s",cmd);

     for(;;)
        {
         fgets(buff,sizeof(buff)-1,fpipe);
         if (feof(fpipe))
             break;

         fputs(Decode_Hexa(buff,"predicate(%s)",1,1),stderr);
        }

     if (fclose(fpipe))
         Fatal_Error("error after executing %s",cmd);
    }
 wait(&error);
#ifdef M_ix86_cygwin
finish:
#endif
 After_Cmd(error)

#else

 int error;

 Before_Cmd(cmd)
 error=system(cmd);
 error >>= 8;
 if (error== -1 || error==127)
     Fatal_Error("error trying to execute %s",cmd);

 After_Cmd(error)
#endif
}




/*-------------------------------------------------------------------------*/
/* DELETE_TEMP_FILE                                                        */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void Delete_Temp_File(char *name)

{
 if (no_del_temp_files)
     return;

#if 1
 if (verbose)
     fprintf(stderr,"delete %s\n",name);
#endif

 unlink(name);
}




/*-------------------------------------------------------------------------*/
/* FIND_FILE                                                               */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void Find_File(char *file,char *suff,char *file_path)

{
 char   name[MAXPATHLEN];
 char **p;

 sprintf(name,"%s%s",file,suff);
 if (!devel_mode)
    {
     sprintf(file_path,"%s/lib/%s",start_path,name);
     if (access(file_path,F_OK)==0)
         return;
    }
  else
     for(p=devel_dir;*p;p++)
        {
         sprintf(file_path,"%s/%s/%s",start_path,*p,name);
         if (access(file_path,F_OK)==0)
             return;
        }

 Fatal_Error("cannot locate file %s",name);
}




/*-------------------------------------------------------------------------*/
/* FATAL_ERROR                                                             */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void Fatal_Error(char *format,...)

{
 FileInf *f;
 va_list  arg_ptr;

 va_start(arg_ptr,format);
 vfprintf(stderr,format,arg_ptr);
 va_end(arg_ptr);

 fprintf(stderr,"\n");

 if (no_del_temp_files)
     exit(1);

 if (verbose)
     fprintf(stderr,"deleting temporary files before exit\n");

 for(f=file;f->name;f++)
    {
     if (f->work_name1 && f->work_name1!=f->name && 
         (file_name_out==NULL || strcmp(f->work_name1,file_name_out)!=0))
         Delete_Temp_File(f->work_name1);

     if (f->work_name2 && f->work_name2!=f->name && 
         (file_name_out==NULL || strcmp(f->work_name2,file_name_out)!=0))
         Delete_Temp_File(f->work_name2);
    }

 exit(1);
}




/*-------------------------------------------------------------------------*/
/* PARSE_ARGUMENTS                                                         */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void Parse_Arguments(int argc,char *argv[])

{
 int       i,file_name_out_i;
 char    **p,*q;
 FileInf  *f=file;


 for(i=1;i<argc;i++)
    {
     if (*argv[i]=='-' && argv[i][1]!='\0')
        {
         if (Check_Arg(i,"-o") || Check_Arg(i,"--output"))
            {
             file_name_out_i=i;
             if (++i>=argc)
                 Fatal_Error("FILE missing after %s option",last_opt);

             file_name_out=argv[i];
             continue;
            }

         if (Check_Arg(i,"-W") || Check_Arg(i,"--wam-for-native"))
            {
             stop_after=FILE_PL;
             bc_mode=0;
             continue;
            }

         if (Check_Arg(i,"-w") || Check_Arg(i,"--wam-for-byte-code"))
            {
             stop_after=FILE_PL;
             bc_mode=1;
             continue;
            }

         if (Check_Arg(i,"-M") || Check_Arg(i,"--mini-assembly"))
            {
             stop_after=FILE_WAM;
             bc_mode=0;
             continue;
            }

         if (Check_Arg(i,"-S") || Check_Arg(i,"--assembly"))
            {
             stop_after=FILE_MA;
             bc_mode=0;
             continue;
            }

         if (Check_Arg(i,"-c") || Check_Arg(i,"--object"))
            {
             stop_after=FILE_ASM;
             bc_mode=0;
             continue;
            }

         if (Check_Arg(i,"-F") || Check_Arg(i,"--fd-to-c"))
            {
             stop_after=FILE_FD;
             bc_mode=0;
             continue;
            }

         if (Check_Arg(i,"--comment"))
            {
             Add_Last_Option(cmd_wam2ma.opt);
             Add_Last_Option(cmd_ma2asm.opt);
             continue;
            }

         if (Check_Arg(i,"--temp-dir"))
            {
             if (++i>=argc)
                 Fatal_Error("PATH missing after %s option",last_opt);

             temp_dir=argv[i];
             continue;
            }

         if (Check_Arg(i,"--no-del-temp-files"))
            {
             no_del_temp_files=1;
             continue;
            }

         if (Check_Arg(i,"--no-decode-hexa"))
            {
             no_decode_hex=1;
             continue;
            }

         if (Check_Arg(i,"--version") || Check_Arg(i,"-v") || 
             Check_Arg(i,"--verbose"))
            {
             Display_Copying("Prolog compiler");
             if (Check_Arg(i,"--version"))
                 exit(0);

             verbose=1;
             continue;
	    }

         if (Check_Arg(i,"-h") || Check_Arg(i,"--help"))
            {
             Display_Help();
             exit(0);
	    }

         if (Check_Arg(i,"--pl-state"))
            {
             if (++i>=argc)
                 Fatal_Error("FILE missing after %s option",last_opt);

             if (access(argv[i],R_OK)!=0)
                {
                 perror(argv[i]);
                 exit(1);
                }

             Add_Last_Option(cmd_pl2wam.opt);
             last_opt=argv[i];
             Add_Last_Option(cmd_pl2wam.opt);
             continue;
            }

         if (Check_Arg(i,"--no-inline")           || 
             Check_Arg(i,"--no-reorder")          ||
             Check_Arg(i,"--no-reg-opt")          ||
             Check_Arg(i,"--min-reg-opt")         ||
             Check_Arg(i,"--no-opt-last-subterm") ||
             Check_Arg(i,"--fast-math")           ||
             Check_Arg(i,"--keep-void-inst")      ||
             Check_Arg(i,"--no-susp-warn")        ||
             Check_Arg(i,"--no-singl-warn")       ||
             Check_Arg(i,"--no-redef-error")      ||
             Check_Arg(i,"--no-call-c")           ||
             Check_Arg(i,"--compile-msg")         ||
             Check_Arg(i,"--statistics"))
            {
             Add_Last_Option(cmd_pl2wam.opt);
             continue;
            }

         if (Check_Arg(i,"--c-compiler"))
            {
             if (++i>=argc)
                 Fatal_Error("FILE missing after %s option",last_opt);

             cmd_cc.exe_name=argv[i];
             continue;
            }

         if (Check_Arg(i,"-C"))
            {
             if (++i>=argc)
                 Fatal_Error("OPTION missing after %s option",last_opt);

             Add_Option(i,cmd_cc.opt);
             cc_fd2c_flags="";       /* if C options specified do not take */
             continue;               /* into account default fd2c C options*/
            }

         if (Check_Arg(i,"-A"))
            {
             if (++i>=argc)
                 Fatal_Error("OPTION missing after %s option",last_opt);

             Add_Option(i,cmd_asm.opt);
             continue;
            }

         if (Check_Arg(i,"--local-size"))
            {
             Record_Link_Warn_Option(i);
             if (++i>=argc)
                 Fatal_Error("SIZE missing after %s option",last_opt);
             def_local_size=strtol(argv[i],&q,10);
             if (*q || def_local_size<0)
                 Fatal_Error("invalid stack size (%s)",argv[i]);
             Record_Link_Warn_Option(i);
             needs_stack_file=1;
             continue;
            }

         if (Check_Arg(i,"--global-size"))
            {
             Record_Link_Warn_Option(i);
             if (++i>=argc)
                 Fatal_Error("SIZE missing after %s option",last_opt);
             def_global_size=strtol(argv[i],&q,10);
             if (*q || def_global_size<0)
                 Fatal_Error("invalid stack size (%s)",argv[i]);
             Record_Link_Warn_Option(i);
             needs_stack_file=1;
             continue;
            }

         if (Check_Arg(i,"--trail-size"))
            {
             Record_Link_Warn_Option(i);
             if (++i>=argc)
                 Fatal_Error("SIZE missing after %s option",last_opt);
             def_trail_size=strtol(argv[i],&q,10);
             if (*q || def_trail_size<0)
                 Fatal_Error("invalid stack size (%s)",argv[i]);
             Record_Link_Warn_Option(i);
             needs_stack_file=1;
             continue;
            }

         if (Check_Arg(i,"--cstr-size"))
            {
             Record_Link_Warn_Option(i);
             if (++i>=argc)
                 Fatal_Error("SIZE missing after %s option",last_opt);
             def_cstr_size=strtol(argv[i],&q,10);
             if (*q || def_cstr_size<0)
                 Fatal_Error("invalid stack size (%s)",argv[i]);
             Record_Link_Warn_Option(i);
             needs_stack_file=1;
             continue;
            }

         if (Check_Arg(i,"--fixed-sizes"))
            {
             Record_Link_Warn_Option(i);
             fixed_sizes=1;
             needs_stack_file=1;
             continue;
            }

         if (Check_Arg(i,"--no-top-level"))
            {
             Record_Link_Warn_Option(i);
             no_top_level=1;
             no_debugger=1;
             continue;
            }

         if (Check_Arg(i,"--no-debugger"))
            {
             Record_Link_Warn_Option(i);
             no_debugger=1;
             continue;
            }

         if (Check_Arg(i,"--min-pl-bips"))
            {
             Record_Link_Warn_Option(i);
             min_pl_bips=1;
             continue;
            }

         if (Check_Arg(i,"--min-fd-bips"))
            {
             Record_Link_Warn_Option(i);
             min_fd_bips=1;
             continue;
            }

         if (Check_Arg(i,"--min-bips") || Check_Arg(i,"--min-size"))
            {
             Record_Link_Warn_Option(i);
             no_top_level=no_debugger=min_pl_bips=min_fd_bips=1;
             if (Check_Arg(i,"--min-size"))
                 strip=1;
             continue;
            }

         if (Check_Arg(i,"--no-fd-lib"))
            {
             Record_Link_Warn_Option(i);
             no_fd_lib=1;
             continue;
            }

         if (Check_Arg(i,"-s") || Check_Arg(i,"--strip"))
            {
             Record_Link_Warn_Option(i);
             strip=1;
             continue;
            }

         if (Check_Arg(i,"-L"))
            {
             Record_Link_Warn_Option(i);
             if (++i>=argc)
                 Fatal_Error("OPTION missing after %s option",last_opt);

             Add_Option(i,cmd_link.opt);
             Record_Link_Warn_Option(i);
             continue;
            }

         Fatal_Error("unknown option %s - try %s --help",argv[i],GPLC);
        }

     if (nb_file==MAX_FILES-1)           /* reserve 1 for stack sizes file */
         Fatal_Error("too many files (max=%d)",MAX_FILES);

     nb_file++;

     f->name=argv[i];

     if ((f->suffix=strrchr(argv[i],'.'))==NULL)
          f->suffix=argv[i]+strlen(argv[i]);

     if (strcmp(PL_SUFFIX_ALTERNATE,f->suffix)==0)
         f->type=FILE_PL;
      else
         if ((q=strstr(C_SUFFIX_ALTERNATE,f->suffix)) &&
             q[-1]=='|' && q[strlen(f->suffix)]=='|')
             f->type=FILE_C;
          else
            {
             f->type=FILE_LINK;
             for(p=suffixes;*p;p++)
                 if (strcmp(*p,f->suffix)==0)
                    {
                     f->type=p-suffixes;
                     break;
                    }
        }

     f->work_name1=f->name;
     f->work_name2=NULL;

     if (f->type!=FILE_LINK && access(f->name,R_OK)!=0)
        {
         perror(f->name);
         exit(1);
	}

     f++;
    }


 if (f==file)
    {
     if (verbose)
         exit(0);               /* -verbose with no files same as -version */
      else
         Fatal_Error("no input file specified");
    }

 f->name=NULL;

 if (nb_file>1 && stop_after<FILE_LINK && file_name_out)
    {
     Record_Link_Warn_Option(file_name_out_i);
     Record_Link_Warn_Option(file_name_out_i+1);
     file_name_out=NULL;
    }
}




/*-------------------------------------------------------------------------*/
/* DISPLAY_HELP                                                            */
/*                                                                         */
/*-------------------------------------------------------------------------*/
void Display_Help(void)

#define L(msg)  fprintf(stderr,"%s\n",msg);

{
 fprintf(stderr,"Usage: %s [OPTION]... FILE...\n",GPLC);
 L(" ")
 L("General options:")
 L("  -o FILE, --output FILE      set output file name")
 L("  -W, --wam-for-native        stop after producing WAM file(s)")
 L("  -w, --wam-for-byte-code     stop after producing WAM for byte-code file(s) (force --no-call-c)")
 L("  -M, --mini-assembly         stop after producing mini-assembly file(s)")
 L("  -S, --assembly              stop after producing assembly file(s)")
 L("  -F, --fd-to-c               stop after producing C file(s) from FD file(s)")
 L("  -c, --object                stop after producing object file(s)")
 L("  --temp-dir PATH             use PATH as directory for temporary files")
 L("  --no-del-temp               do not delete temporary files")
 L("  --no-decode-hexa            do not decode hexadecimal predicate names")
 L("  -v, --verbose               print executed commands")
 L("  -h, --help                  print this help and exit")
 L("  --version                   print version number and exit")
 L(" ")
 L("Prolog to WAM compiler options:")
 L("  --pl-state FILE             read FILE to set the initial Prolog state")
 L("  --no-inline                 do not inline predicates")
 L("  --no-reorder                do not reorder predicate arguments")
 L("  --no-reg-opt                do not optimize registers")
 L("  --min-reg-opt               minimally optimize registers")
 L("  --no-opt-last-subterm       do not optimize last subterm compilation")
 L("  --fast-math                 fast mathematical mode (assume integer arithmetic)")
 L("  --keep-void-inst            keep void instructions in the output file")
 L("  --no-susp-warn              do not show warnings for suspicious predicates")
 L("  --no-singl-warn             do not show warnings for named singleton variables")
 L("  --no-redef-error            do not show errors for built-in redefinitions")
 L("  --no-call-c                 do not allow the use of fd_tell, '$call_c',...")
 L("  --compile-msg               print a compile message")
 L("  --statistics                print statistics information")
 L(" ")
 L("WAM to mini-assembly translator options:")
 L("  --comment                   include comments in the output file")
 L(" ")
 L("Mini-assembly to assembly translator options:")
 L("  --comment                   include comments in the output file")
 L(" ")
 L("C Compiler options:")
 L("  --c-compiler FILE           use FILE as C compiler")
 L("  -C OPTION                   pass OPTION to the C compiler")
 L(" ")
 L("Assembler options:")
 L("  -A OPTION                   pass OPTION to the assembler")
 L(" ")
 L("Linker options:")
 L("  --local-size N              set default local  stack size to N Kb")
 L("  --global-size N             set default global stack size to N Kb")
 L("  --trail-size N              set default trail  stack size to N Kb")
 L("  --cstr-size N               set default cstr   stack size to N Kb")
 L("  --fixed-sizes               do not consult environment variables at run-time")
 L("  --no-top-level              do not link the top-level (force --no-debugger)")
 L("  --no-debugger               do not link the Prolog/WAM debugger")
 L("  --min-pl-bips               link only used Prolog built-in predicates")
 L("  --min-fd-bips               link only used FD solver built-in predicates")
 L("  --min-bips                  same as: --no-top-level --min-pl-bips --min-fd-bips") 
 L("  --min-size                  same as: --min-bips --strip") 
 L("  --no-fd-lib                 do not look for the FD library (maintenance only)")
 L("  -s, --strip                 strip the executable")
 L("  -L OPTION                   pass OPTION to the linker")
 L("")
 L("Report bugs to bug-prolog@gnu.org.")
}

#undef L
