
#define DEFINE_GLOBALS 1

#include <config.h>
#include "sharedanced.h"
#ifdef HAVE_SETLOCALE
# include <locale.h>
#endif
#include "log.h"
#include "daemonize.h"
#include "expire.h"
#include "process.h"
#include "sharedanced_p.h"

#ifdef WITH_DMALLOC
# include <dmalloc.h>
#endif

static void die_mem(void)
{
    logfile(LOG_ERR, _("Out of memory"));
    
    exit(EXIT_FAILURE);
}

static void sigchld(int fd, short event, void *ev_)
{
    struct event *ev = ev_;

    (void) fd;
    (void) event;
    signal_del(ev);
    _exit(EXIT_SUCCESS);
}

static void usage(void)
{
    puts("\n" PACKAGE_STRING " - " __DATE__ "\n");
    fputs(_(
            "--ip=xxx          (-i) listen to this address (default=any)\n"
            "--port=xxx        (-p) listen to that port (default=1042)\n"
            "--backlog=xxx     (-b) change the backlog\n"
            "--directory=xxx   (-d) directory where the data should be stored\n"
            "--readsize=xxx    (-r) read buffer size\n"
            "--maxreadsize=xxx (-R) max read size\n"            
            "--expiration=xxx  (-e) expiration of data in seconds\n"
            "--timeout=xxx     (-t) disconnect after xxx seconds without data\n"
            "--debug           (-D) verbose debug mode\n"
            "--daemonize       (-B) operate in background\n"
            "\n"
            "Report bugs and suggestions to "
            ), stdout);
    puts(PACKAGE_BUGREPORT ".\n");

    exit(EXIT_SUCCESS);
}

int main(int argc, char *argv[])
{
    struct addrinfo hints, *res;
    struct event ev;
    struct event sigchld_ev;
    int on;    
    int listen_fd;
    int fodder;
    int option_index;

#ifdef HAVE_SETLOCALE
    setlocale(LC_ALL, "");
#endif
    bindtextdomain(PACKAGE, LOCALEDIR);
    textdomain(PACKAGE);
    
    if (argc <= 1) {
        usage();
    }

#ifndef SAVE_DESCRIPTORS
    openlog("sharedanced", LOG_PID, syslog_facility);
#endif    

    timeout.tv_sec = (time_t) DEFAULT_TIMEOUT;
    timeout.tv_usec = 0U;
    
    while ((fodder = getopt_long(argc, argv, GETOPT_OPTIONS, long_options,
                                 &option_index)) != -1) {
        switch (fodder) {
        case 'h': {
            usage();
        }
        case 'B': {
            daemonize = 1;
            break;
        }
        case 'D': {
            debug = 1;
            break;
        }
        case 'i': {
            if ((listen_address_s = strdup(optarg)) == NULL) {
                die_mem();
            }
            break;
        }
        case 'p': {
            if ((port_s = strdup(optarg)) == NULL) {
                die_mem();
            }
            break;
        }
        case 'b': {
            if ((backlog = atoi(optarg)) < 1) {
                logfile(LOG_ERR, _("Invalid backlog"));
                return 1;
            }
            break;
        }
        case 'r': {
            if ((read_chunk_size = (size_t) strtoul(optarg, NULL, 10))
                < (size_t) 1U) {
                logfile(LOG_ERR, _("Invalid read chunk size"));
                return 1;
            }
            break;
        }
        case 'R': {
            if ((max_read_size = (size_t) strtoul(optarg, NULL, 10))
                < (size_t) 10U) {
                logfile(LOG_ERR, _("Invalid max read size"));
                return 1;
            }
            break;
        }
        case 'e': {
            if ((expiration = (time_t) strtoul(optarg, NULL, 10))
                < (time_t) 1) {
                logfile(LOG_ERR, _("Invalid expiration time"));
                return 1;
            }
            break;
        }
        case 't': {
            if ((timeout.tv_sec = (time_t) strtoul(optarg, NULL, 10))
                < (time_t) 1) {
                logfile(LOG_ERR, _("Invalid timeout"));
                return 1;
            }
            break;
        }
        case 'd': {
            if ((storage_dir = strdup(optarg)) == NULL) {
                die_mem();
            }
            strlen_storage_dir = strlen(storage_dir);
            break;
        }
        default: {
            usage();
        }
        }
    }    
    memset(&hints, 0, sizeof hints);
    hints.ai_flags = AI_PASSIVE;
    hints.ai_family = AF_UNSPEC;
    hints.ai_socktype = SOCK_STREAM;
    hints.ai_addr = NULL;
    if ((on = getaddrinfo(listen_address_s, port_s, &hints, &res)) != 0 ||
        (res->ai_family != AF_INET && res->ai_family != AF_INET6)) {
        logfile(LOG_ERR, _("Unable to create the listening socket: [%s]"),
                gai_strerror(on));
        return 1;
    }
    on = 1;
    if ((listen_fd = socket(res->ai_family, SOCK_STREAM, IPPROTO_TCP)) == -1 ||
        setsockopt(listen_fd, SOL_SOCKET, SO_REUSEADDR,
                   (char *) &on, sizeof on) != 0 ||
        bind(listen_fd, (struct sockaddr *) res->ai_addr,
             (socklen_t) res->ai_addrlen) != 0 ||
        listen(listen_fd, backlog) != 0) {
        logfile(LOG_ERR, _("Unable to create the socket: [%s]"),
                strerror(errno));
        freeaddrinfo(res);
        return 1;
    }
    freeaddrinfo(res);

    dodaemonize();
    
    if ((expire_pid = fork()) == (pid_t) -1) {
        logfile(LOG_ERR, _("Unable to fork: [%s]"), strerror(errno));
        (void) close(listen_fd);        
        return 1;
    }
    if (expire_pid == (pid_t) 0) {
        (void) close(listen_fd);
        expire();
        _exit(0);
    }        
    event_init();
    signal_set(&sigchld_ev, SIGCHLD, sigchld, &ev);
    signal_add(&sigchld_ev, NULL);
    event_set(&ev, listen_fd, EV_READ | EV_PERSIST, new_client, &ev);
    event_add(&ev, NULL);
    event_dispatch();
    (void) close(listen_fd);
    (void) kill(expire_pid, SIGTERM);

#ifndef SAVE_DESCRIPTORS
    closelog();
#endif        
    
    return 0;
}
