/* NcDBMutil.c
 *
 * Copyright (c) 1999-2005 Mike Gleason, NCEMRSoft.
 * All rights reserved.
 */

#ifdef HAVE_CONFIG_H
#	include "config.h"
#elif defined(WIN32)
#	include "config.h.win32"
#endif

#ifdef HAVE_UNISTD_H
#	include <unistd.h>
#endif

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include <fcntl.h>
#include <ctype.h>
#include <errno.h>
#include <signal.h>

#include <ncdbm.h>

#if !defined(HAVE_SLEEP) && !defined(sleep)
#	define sleep(a)
#endif

NcDBMFile gDB;
NcDBMFilePtr gDBptr = NULL;
const char *gDBFileName = NULL;
int gVerbose = 0;
int gWarnings = 1;
int gDelim = ',';
int gEmptyDataOK = 0;
FILE *gStdErr;
int gReOpenInBatchMode = 0;
int gHSize = NCDBM_DEFAULT_HSIZE;

#ifdef WIN32
#	include "getopt.h"
#	define optind gOptInd
#	define optarg gOptArg
#	define getopt Getopt
#else
#	ifdef NEED_GETOPT_H
#		include <getopt.h>
#	endif
#	ifdef NEED_GETOPT_EXTERN_DECLS
		/* extern int optind; don't need this yet */
		extern char *optarg;
#	endif
#endif


static int
OpenOrCreateDBForWriting(void)
{
	int e;

	e = NcDBM_Open(&gDB, gDBFileName, NCDBM_O_WRONLY|NCDBM_O_CREAT, 00666, gHSize);
	if (e != NCDBM_NO_ERR) {
		fprintf(gStdErr, "Could not open %s for writing: %s.\n", gDBFileName, NcDBM_Strerror(e));
		return (-1);
	}
	gDBptr = &gDB;
	return (0);
}	/* OpenOrCreateDBForWriting */




static int
OpenDBForReadingAndWriting(void)
{
	int e;

	e = NcDBM_Open(&gDB, gDBFileName, NCDBM_O_RDWR|NCDBM_O_CREAT, 00666, gHSize);
	if (e != NCDBM_NO_ERR) {
		fprintf(gStdErr, "Could not open %s for read/write: %s.\n", gDBFileName, NcDBM_Strerror(e));
		perror("open");
		return (-1);
	}
	gDBptr = &gDB;
	return (0);
}	/* OpenDBForReadingAndWriting */




static int
OpenDBForReading(void)
{
	int e;

	e = NcDBM_Open(&gDB, gDBFileName, NCDBM_O_RDONLY, 0, gHSize);
	if (e != NCDBM_NO_ERR) {
		fprintf(gStdErr, "Could not open %s for reading: %s.\n", gDBFileName, NcDBM_Strerror(e));
		return (-1);
	}
	gDBptr = &gDB;
	return (0);
}	/* OpenDBForReading */




static void
DBClose(void)
{
	NcDBMFilePtr dbp;

	if (gDBptr != NULL) {
		dbp = gDBptr;
		gDBptr = NULL;
		NcDBM_Close(dbp);
	}
}	/* DBClose */




static void
ExitStuff(void)
{
	if (gDBptr != NULL) {
		DBClose();
		fprintf(gStdErr, "Database closed.\n");
	}
}	/* ExitStuff */




static void
Bye(int signum)
{
	static int once = 0;

	if (++once == 1) {
		fprintf(gStdErr, "\nCaught signal %d, exiting.\n", signum);
		DBClose();
		exit(3);
	}
}	/* Bye */




static int
Add1(char *line, NcDBMInsertMode insertmode)
{
	NcDBMDatum key, d;
	int rc;
	char *datastart;
	char keybuf[256], z[4];

	key.dptr = strncpy(keybuf, line, sizeof(keybuf));
	keybuf[sizeof(keybuf) - 1] = '\0';
	datastart = strchr(keybuf, gDelim);
	if (datastart == NULL) {
		z[0] = '\0';
		d.dptr = z;
		d.dsize = 1;
		key.dsize = (int) strlen(key.dptr) + 1;
		if (key.dsize > ((int) sizeof(keybuf) - 3)) {
			fprintf(gStdErr, "Key field is too large in \"%s\".\n", line);
			return (-1);
		}
		if (gEmptyDataOK == 0) {
			fprintf(gStdErr, "Key field not delimited by \'%c\' in \"%s\".\n", gDelim, line);
			return (-1);
		}
	} else {
		*datastart++ = '\0';
		key.dsize = (int) (datastart - keybuf);
		d.dptr = (char *) line + key.dsize;
		d.dsize = (int) strlen(d.dptr) + 1;
	}

	rc = NcDBM_Insert(gDBptr, key, d, insertmode);

	if (rc < 0) {
		fprintf(gStdErr, "Could not insert record \"%s\" : %s.\n", line, NcDBM_Strerror(rc));
		return (-1);
	} else if (rc > 0) {
		fprintf(gStdErr, "Could not insert record \"%s\" : key field already exists.\n", line);
		return (-1);
	}
	return (0);
}	/* Add1 */




static int
Add(char *line, NcDBMInsertMode insertmode)
{
	if (OpenOrCreateDBForWriting() < 0)
		return (1);
	return (Add1(line, insertmode));
}	/* Add */




static int
Import(const char *ifile, NcDBMInsertMode insertmode)
{
	FILE *fp;
	char buf[256];
	char *cp;
	int errs = 0;

	fp = fopen(ifile, "r");
	if (fp == NULL) {
		perror(ifile);
		return (-1);
	}

	if (OpenOrCreateDBForWriting() < 0) {
		fclose(fp);
		return (-1);
	}
	(void) NcDBM_ExclusiveLock(gDBptr, F_WRLCK);	/* lock for entire import. */

	buf[sizeof(buf) - 1] = '\0';
	while (fgets(buf, sizeof(buf) - 1, fp) != NULL) {
		cp = buf + strlen(buf) - 1;
		if (*cp == '\n')
			*cp = '\0';
		if (Add1(buf, insertmode) < 0)
			errs++;
	}

	(void) NcDBM_ExclusiveLock(gDBptr, F_UNLCK);	/* unlock. */
	fclose(fp);
	return (errs);
}	/* Import */




static int
Del(char *line)
{
	NcDBMDatum key;
	char *datastart;
	char keybuf[256];
	int rc;

	key.dptr = strncpy(keybuf, line, sizeof(keybuf));
	keybuf[sizeof(keybuf) - 1] = '\0';
	datastart = strchr(keybuf, gDelim);
	if (datastart == NULL) {
		key.dsize = (int) strlen(key.dptr) + 1;
		if (key.dsize > ((int) sizeof(keybuf) - 3)) {
			fprintf(gStdErr, "Key field is too large in \"%s\".\n", line);
			return (-1);
		}
	} else {
		*datastart++ = '\0';
		key.dsize = (int) (datastart - keybuf);
	}

	if ((rc = NcDBM_Delete(gDBptr, key)) != 0) {
		fprintf(gStdErr, "Could not remove record for \"%s\" : %s.\n", keybuf, NcDBM_Strerror(rc));
		return (-1);
	}
	return (0);
}	/* Del */




static int
Query(char *line)
{
	NcDBMDatum d, key;
	char *datastart;
	char keybuf[256];
	int rc;

	key.dptr = strncpy(keybuf, line, sizeof(keybuf));
	keybuf[sizeof(keybuf) - 1] = '\0';
	datastart = strchr(keybuf, gDelim);
	if (datastart == NULL) {
		key.dsize = (int) strlen(key.dptr) + 1;
		if (key.dsize > ((int) sizeof(keybuf) - 3)) {
			fprintf(gStdErr, "Key field is too large in \"%s\".\n", line);
			return (-1);
		}
	} else {
		*datastart++ = '\0';
		key.dsize = (int) (datastart - keybuf);
	}

	rc = NcDBM_Query(gDBptr, key, &d);
	if (rc != NCDBM_NO_ERR) {
		fprintf(gStdErr, "Could not query record for \"%s\" : %s.\n", keybuf, NcDBM_Strerror(rc));
		return (-1);
	}

	printf("%s%c%s\n", keybuf, gDelim, d.dptr);
	return (0);
}	/* Query */




static int
Batch(const char *bfile)
{
	FILE *fp;
	char buf[400];
	char *cp;
	int errs = 0;
	int op;

	fp = fopen(bfile, "r");
	if (fp == NULL) {
		perror(bfile);
		return (-1);
	}

	if (OpenDBForReadingAndWriting() < 0) {
		fclose(fp);
		return (-1);
	}

	gStdErr = stdout;
	memset(buf, 0, sizeof(buf));
	while (fgets(buf, sizeof(buf) - 1, fp) != NULL) {
		cp = buf + strlen(buf) - 1;
		if (*cp == '\n')
			*cp = '\0';

		cp = buf;
		op = *cp++;
		if ((op == '\0') || (*cp == '\0'))
			continue;
		if (isspace(*cp))
			cp++;
		if (isupper(op))
			op = tolower(op);
		if (gVerbose != 0)
			printf("> %c %s\n", op, cp);
		switch (op) {
		case 'a':
		case 'i':
			/* insert */
			if (gReOpenInBatchMode != 0) {
				/* Useful only for performance testing, where each
				 * transaction requires a full open/transaction/close sequence.
				 */
				DBClose();
				if (OpenDBForReadingAndWriting() < 0) {
					fclose(fp);
					return (-1);
				}
			}
			if (Add1(cp, NCDBM_INSERT) < 0)
				errs++;
			break;
		case 'u':
			/* update */
			if (gReOpenInBatchMode != 0) {
				/* Useful only for performance testing, where each
				 * transaction requires a full open/transaction/close sequence.
				 */
				DBClose();
				if (OpenDBForReadingAndWriting() < 0) {
					fclose(fp);
					return (-1);
				}
			}
			if (Add1(cp, NCDBM_REPLACE) < 0)
				errs++;
			break;
		case 'q':
			/* query */
			if (gReOpenInBatchMode != 0) {
				/* Useful only for performance testing, where each
				 * transaction requires a full open/transaction/close sequence.
				 */
				DBClose();
				if (OpenDBForReading() < 0) {
					fclose(fp);
					return (-1);
				}
			}
			if (Query(cp) < 0)
				errs++;
			break;
		case 'd':
			/* delete */
			if (gReOpenInBatchMode != 0) {
				/* Useful only for performance testing, where each
				 * transaction requires a full open/transaction/close sequence.
				 */
				DBClose();
				if (OpenDBForReadingAndWriting() < 0) {
					fclose(fp);
					return (-1);
				}
			}
			if (Del(cp) < 0)
				errs++;
			break;
		}
	}
	fclose(fp);
	return (errs);
}	/* Batch */




static int
Export(void)
{
	NcDBMDatum key;
	NcDBMDatum d;
	NcDBMIterState itstate;
	int e;

	if (OpenDBForReading() < 0) {
		return (-1);
	}

	(void) NcDBM_ExclusiveLock(gDBptr, F_RDLCK);	/* lock for entire export. */
	e = NcDBM_FirstKey(gDBptr, &key, &d, &itstate);
	if (e == NCDBM_ERR_NO_MORE_RECORDS) {
		fprintf(gStdErr, "(database is empty)\n");
		(void) NcDBM_ExclusiveLock(gDBptr, F_UNLCK);	/* unlock. */
		return (-1);
	} else if (e != NCDBM_NO_ERR) {
		fprintf(gStdErr, "Export failed : %s.\n", NcDBM_Strerror(e));
		(void) NcDBM_ExclusiveLock(gDBptr, F_UNLCK);	/* unlock. */
		return (-1);
	}

	/* Print first record. */
	fprintf(stdout, "%s%c%s\n", key.dptr, gDelim, d.dptr);
	free(d.dptr);
	free(key.dptr);

	for (;;) {
		e = NcDBM_NextKey(gDBptr, &key, &d, &itstate);
		if (e == NCDBM_ERR_NO_MORE_RECORDS)
			break;
		if (e != NCDBM_NO_ERR) {
			fprintf(gStdErr, "Export failed : %s.\n", NcDBM_Strerror(e));
			(void) NcDBM_ExclusiveLock(gDBptr, F_UNLCK);	/* unlock. */
			return (-1);
		}

		/* Print each record. */
		fprintf(stdout, "%s%c%s\n", key.dptr, gDelim, d.dptr);
		free(d.dptr);
		free(key.dptr);
	}

	(void) NcDBM_ExclusiveLock(gDBptr, F_UNLCK);	/* unlock. */
	return (0);
}	/* Export */




static void
Usage(void)
{
	FILE *fp;

#ifdef HAVE_POPEN
	fp = popen("more", "w");
	if (fp == NULL)
		fp = stdout;
#else
	fp = stdout;
#endif

	fprintf(fp, "\nUsages:\n");
	fprintf(fp, "------\n");
	fprintf(fp, "\nAdd a record:\n");
	fprintf(fp, "  NcDBMutil -f <database.db> -F <delim> -a \"keyfield<delim>data...\"\n");
	fprintf(fp, "\nUpdate an existing record:\n");
	fprintf(fp, "  NcDBMutil -f <database.db> -F <delim> -u \"keyfield<delim>data...\"\n");
	fprintf(fp, "\nDelete an existing record:\n");
	fprintf(fp, "  NcDBMutil -f <database.db> -d <keyfield>\n");
	fprintf(fp, "\nQuery an existing record:\n");
	fprintf(fp, "  NcDBMutil -f <database.db> -q <keyfield>\n");
	fprintf(fp, "\nExport a database into a textual format:\n");
	fprintf(fp, "  NcDBMutil -f <database.db> -F <delim> -x\n");
	fprintf(fp, "\nImport a pre-formatted text file into a database:\n");
	fprintf(fp, "  NcDBMutil -f <database.db> -F <delim> -i <filename>\n");
	fprintf(fp, "\nImport, replacing records as needed:\n");
	fprintf(fp, "  NcDBMutil -f <database.db> -F <delim> -r <filename>\n");
	fprintf(fp, "\n");

#ifdef HAVE_POPEN
	if (fp != stdout)
		(void) pclose(fp);
#endif
	exit(5);
}	/* Usage */



static void
CheckPwFileName(void)
{
	if (gDBFileName == NULL) {
		fprintf(gStdErr, "No database file specified.\n");
		sleep(2);
		Usage();
	}
}	/* CheckPwFileName */




int
main(int argc, char **argv)
{
	int c, es = 0;
	char *addline = NULL, *delline = NULL, *queryline = NULL;
	const char *batchfile = NULL;
	const char *importfile = NULL;
	int export = 0;
	NcDBMInsertMode insertmode = NCDBM_INSERT;

	gStdErr = stderr;

	/* Use default behavior. */
#ifdef SIGINT
	signal(SIGINT, SIG_DFL);
#endif
#ifdef SIGALRM
	signal(SIGALRM, SIG_DFL);
#endif
#ifdef SIGQUIT
	signal(SIGQUIT, SIG_DFL);
#endif
#ifdef SIGTSTP
	signal(SIGTSTP, SIG_IGN);
#endif
#ifdef SIGHUP
	signal(SIGHUP, SIG_DFL);
#endif

#ifdef HAVE_ATEXIT
	atexit(ExitStuff);
#endif

	while ((c = getopt(argc, argv, "a:b:d:ef:F:q:i:r:u:vwxC:B:J")) > 0) switch (c) {
		case 'a':
			addline = optarg;
			break;
		case 'b':
			batchfile = optarg;
			break;
		case 'd':
			delline = optarg;
			break;
		case 'e':
		case 'x':
			export = 1;
			break;
		case 'F':
			gDelim = optarg[0];
			break;
		case 'f':
			gDBFileName = optarg;
			break;
		case 'i':
			importfile = optarg;
			break;
		case 'r':
			importfile = optarg;
			insertmode = NCDBM_REPLACE;
			break;
		case 'u':
			addline = optarg;
			insertmode = NCDBM_REPLACE;
			break;
		case 'v':
			gVerbose++;
			break;
		case 'w':
			gWarnings = !gWarnings;
			break;
		case 'q':
			queryline = optarg;
			break;
		case 'C':
			break;
		case 'B':
			break;
		case 'J':
			gReOpenInBatchMode = 1;
			break;
		default:
			Usage();
	}

#ifdef SIGHUP
	(void) signal(SIGHUP, Bye);
#endif
#ifdef SIGINT
	(void) signal(SIGINT, Bye);
#endif
#ifdef SIGTERM
	(void) signal(SIGTERM, Bye);
#endif

	if (addline != NULL) {
		CheckPwFileName();
		es = Add(addline, insertmode);
	} else if (queryline != NULL) {
		CheckPwFileName();

		if (OpenDBForReading() < 0) {
			es = 1;
		} else {
			es = Query(queryline);
		}
	} else if (delline != NULL) {
		CheckPwFileName();

		if (OpenDBForReadingAndWriting() < 0) {
			es = 1;
		} else {
			es = Del(delline);
		}
	} else if (batchfile != NULL) {
		CheckPwFileName();
		es = Batch(batchfile);
	} else if (export != 0) {
		CheckPwFileName();
		es = Export();
	} else if (importfile != NULL) {
		CheckPwFileName();
		es = Import(importfile, insertmode);
	} else {
		fprintf(gStdErr, "No operation specified.\n");
		sleep(2);
		Usage();
	}

	DBClose();
	exit((es == 0) ? 0 : 1);
	/*NOTREACHED*/
	return 0;
	/*NOTREACHED*/
}	/* main */
