/* firstnxt.c */

#ifdef HAVE_CONFIG_H
#	include "config.h"
#elif defined(WIN32)
#	include "config.h.win32"
#	include <windows.h>
#	include <io.h>
#endif

#ifdef HAVE_UNISTD_H
#	include <unistd.h>
#	include <sys/types.h>
#endif

#ifdef HAVE_FCNTL_H
#	include <fcntl.h>
#endif

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stddef.h>
#include <ctype.h>
#include <errno.h>

#include "internal.h"
#include "ncdbm.h"

/* Note: No locking is done here; use NcDBM_ExclusiveLock() before
 * you call this function.
 */

int
NcDBM_NextKey(
	NcDBMFile *const db,
	NcDBMDatum *const nxtkey,
	NcDBMDatum *const content,
	NcDBMIterState *itp)
{
	FILE *fp;
	NcDBM_hash32_t prime;
	off_t o;
	NcDBMRecordHeader rh;
	char rkey[NCDBM_MAX_KEY_SIZE + 1];
	int rksize, dsize;
	size_t ntoread;
	char *cp;

	if ((db == NULL) || (strncmp(db->magic, NCDBM_MAGIC_STR, sizeof(db->magic) - 1) != 0)) {
		/* corrupt or closed. */
		return (NCDBM_ERR_MEM_HOSED);
	}

	if (itp == NULL) {
		return (NCDBM_ERR_BAD_PARAMETER);
	}

	fp = db->fp;

	/* If we were at the end of the chain, find a new chain
	 * to traverse.
	 */
	if (itp->ho == 0) {
		if (itp->n > 0)
			itp->h++;
		o = db->hOff + (off_t) (itp->h * sizeof(NcDBM_off_t));
		if (fseek(fp, (off_t) o, SEEK_SET) != 0) {
			return (NCDBM_ERR_SEEK);
		}

		for (prime = db->prime; itp->h < prime; itp->h++, o += (off_t) sizeof(NcDBM_off_t)) {
			if (fread(&itp->ho, sizeof(itp->ho), 1, fp) != (size_t) 1) {
				return (NCDBM_ERR_WRITE);
			}
			if (itp->ho != 0) {
				/* Found the next non-empty chain. */
				break;
			}
		}

		if (itp->h == prime) {
			return (NCDBM_ERR_NO_MORE_RECORDS);
		}
	}

	itp->n++;

	/* Now read the next record header from the current chain. */
	if (fseek(fp, (off_t) itp->ho, SEEK_SET) != 0) {
		return (NCDBM_ERR_SEEK);
	}
	
	if (fread(&rh, sizeof(NcDBMRecordHeader), (size_t) 1, fp) != (size_t) 1) {
		return (NCDBM_ERR_READ);
	}

	/* We read the header, now update the state variable so that
	 * we will fetch the next record on the next invocation.
	 */
	itp->ho = (NcDBM_hash32_t) rh.nextOffset;
	
	rksize = (int) rh.keySize;
	memcpy(rkey, rh.keyPrefix, sizeof(rh.keyPrefix));

	if (rksize > NCDBM_KEY_PREFIX_SIZE) {
		cp = rkey + NCDBM_KEY_PREFIX_SIZE;
		ntoread = (size_t) rksize - NCDBM_KEY_PREFIX_SIZE;
		if (fread(cp, ntoread, (size_t) 1, fp) != (size_t) 1) {
			return (NCDBM_ERR_READ);
		}
	}

	if (nxtkey != NULL) {
		/* Copy the key field, if requested. */
		nxtkey->dptr = malloc((size_t) (rksize + 1));
		if (nxtkey->dptr == NULL)
			return (NCDBM_ERR_MALLOC);

		memcpy(nxtkey->dptr, rkey, (size_t) rksize);
		nxtkey->dptr[rksize] = '\0';
	}

	/* Now copy the record contents, if requested. */
	if (content != NULL) {
		dsize = (int) rh.recSize;
		content->dptr = malloc((size_t) (dsize + 1));
		if (content->dptr == NULL)
			return (NCDBM_ERR_MALLOC);
	
		if (fread(content->dptr, (size_t) dsize, (size_t) 1, fp) != (size_t) 1) {
			return (NCDBM_ERR_READ);
		}
		content->dptr[dsize] = '\0';
		content->dsize = dsize;
	}

	return (NCDBM_NO_ERR);
}	/* NcDBM_NextKey */




/* Note: No locking is done here; use NcDBM_ExclusiveLock() before
 * you call this function.
 */

int
NcDBM_FirstKey(
	NcDBMFile *const db, 
	NcDBMDatum *const firstkey, 
	NcDBMDatum *const content,
	NcDBMIterState *const itp)
{
	if ((db == NULL) || (strncmp(db->magic, NCDBM_MAGIC_STR, sizeof(db->magic) - 1) != 0)) {
		/* corrupt or closed. */
		return (NCDBM_ERR_MEM_HOSED);
	}

	if (itp == NULL) {
		return (NCDBM_ERR_BAD_PARAMETER);
	}

	memset(itp, 0, sizeof(NcDBMIterState));

	return (NcDBM_NextKey(db, firstkey, content, itp));
}	/* NcDBM_FirstKey */
