/* delete.c */

#ifdef HAVE_CONFIG_H
#	include "config.h"
#elif defined(WIN32)
#	include "config.h.win32"
#	include <windows.h>
#	include <io.h>
#endif

#ifdef HAVE_UNISTD_H
#	include <unistd.h>
#	include <sys/types.h>
#endif

#ifdef HAVE_FCNTL_H
#	include <fcntl.h>
#endif

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stddef.h>
#include <ctype.h>
#include <errno.h>

#include "internal.h"
#include "ncdbm.h"

int
NcDBM_Delete0(
	NcDBMFile *const db,
	NcDBM_hash32_t h,
	NcDBMRecordHeader *rh,
	off_t prevho,
	NcDBMRecordHeader *prevrh)
{
	off_t ho;
	NcDBM_off_t newho0;
	FILE *fp;

	/* We will update the previous record in the chain's header so
	 * that it points to the next record listed in the record that
	 * we want to delete.  This will effectively re-link the chain,
	 * and then orphan the record we want to delete.  If there is
	 * no previous record, then the first-level hash needs to be
	 * updated instead.
	 */
	fp = db->fp;

	if (prevho == 0) {
		/* Update first-level hash entry. */

		/* Find the first-level hash entry for this hash value. */
		ho = db->hOff + (off_t) (h * sizeof(NcDBM_off_t));

		/* This effectively locks the entire chain for ourselves. */
		if (NcDBM_LockInt(db, F_WRLCK, ho) < 0)
			return (NCDBM_ERR_LOCK);

		if (fseek(fp, (off_t) ho, SEEK_SET) != 0) {
			(void) NcDBM_LockInt(db, F_UNLCK, ho);
			return (NCDBM_ERR_SEEK);
		}

		/* Write a new element into the first-level hash.
		 * We will then have a chain of exactly one record,
		 * which we just wrote.
		 */
		newho0 = rh->nextOffset;
		if (fwrite(&newho0, (size_t) sizeof(newho0), (size_t) 1, fp) != (size_t) 1) {
			(void) NcDBM_LockInt(db, F_UNLCK, ho);
			return (NCDBM_ERR_WRITE);
		}

		if (fflush(fp) != 0) {
			(void) NcDBM_LockInt(db, F_UNLCK, ho);
			return (NCDBM_ERR_WRITE);
		}

		(void) NcDBM_LockInt(db, F_UNLCK, ho);
	} else {
		prevrh->nextOffset = rh->nextOffset;

		/* This effectively locks the entire chain for ourselves. */
		if (NcDBM_LockInt(db, F_WRLCK, prevho) < 0)
			return (NCDBM_ERR_LOCK);

		/* Go back to where the last record in the chain is (again). */
		if (fseek(fp, (off_t) prevho, SEEK_SET) != 0) {
			(void) NcDBM_LockInt(db, F_UNLCK, prevho);
			return (NCDBM_ERR_SEEK);
		}

		/* Write out the updated header. */
		if (fwrite(prevrh, (size_t) sizeof(NcDBMRecordHeader), (size_t) 1, fp) != (size_t) 1) {
			(void) NcDBM_LockInt(db, F_UNLCK, prevho);
			return (NCDBM_ERR_WRITE);
		}

		if (fflush(fp) != 0) {
			(void) NcDBM_LockInt(db, F_UNLCK, prevho);
			return (NCDBM_ERR_WRITE);
		}

		(void) NcDBM_LockInt(db, F_UNLCK, prevho);
	}

	/* Note that we do not actually remove the data record itself.
	 * It has simply been unlinked from the chain.
	 */
	return (NCDBM_NO_ERR);
}	/* NcDBM_Delete0 */



int
NcDBM_Delete(NcDBMFile *const db, const NcDBMDatum key)
{
	NcDBM_hash32_t h;
	NcDBMRecordHeader rh, prevrh;
	off_t o, prevho;
	int e;

	if ((db == NULL) || (strncmp(db->magic, NCDBM_MAGIC_STR, sizeof(db->magic) - 1) != 0)) {
		/* corrupt or closed. */
		return (NCDBM_ERR_MEM_HOSED);
	}

	if ((key.dptr == NULL) || (key.dsize < 1)) {
		return (NCDBM_ERR_BAD_PARAMETER);
	}

	if (key.dsize > NCDBM_MAX_KEY_SIZE) {
		return (NCDBM_ERR_KEY_TOO_LARGE);
	}

	/* First, compute the key's hash value. */
	h = db->func((unsigned char *) key.dptr, key.dsize);
	h = h % db->prime;

	e = NcDBM_Query0(db, key, h, NULL, &rh, &o, &prevho, &prevrh);

	if (e == NCDBM_NO_ERR) {
		/* Found the record, now unlink it. */
		e = NcDBM_Delete0(db, h, &rh, prevho, &prevrh);
	}

	return (e);
}	/* NcDBM_Delete */
