/* core.c */

#ifdef HAVE_CONFIG_H
#	include "config.h"
#elif defined(WIN32)
#	include "config.h.win32"
#	include <windows.h>
#	include <io.h>
#	include <share.h>
#endif

#ifdef HAVE_UNISTD_H
#	include <unistd.h>
#	include <sys/types.h>
#endif

#ifdef HAVE_FCNTL_H
#	include <fcntl.h>
#endif

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stddef.h>
#include <ctype.h>
#include <errno.h>

#ifdef NCDBM_DEBUG
#	include <time.h>
#	include <stdarg.h>
#endif

#include "internal.h"
#include "ncdbm.h"

static char gNcDBMVersion[] = "@(#) NcDBM 1.1.5 ** Copyright 2000-2005 Mike Gleason. All rights reserved.";
int gNcDBM_Unused = 0;

#ifdef NCDBM_DEBUG
int gNcDBM_DebugLevel = 9;
FILE *gNcDBM_DebugFp = NULL;
#endif

#ifdef NCDBM_DEBUG
/*VARARGS*/
void
NcDBM_Debug(const int level, const char *const fmt, ...)
{
	va_list ap;
	char dstr[40];
	char buf[512];
	time_t now;
	char spaces[64];

	if (gNcDBM_DebugLevel >= level) {
		va_start(ap, fmt);
#ifdef HAVE_VSNPRINTF
		(void) vsnprintf(buf, sizeof(buf) - 1, fmt, ap);
		buf[sizeof(buf) - 1] = '\0';
#else
		(void) vsprintf(buf, fmt, ap);
#endif
		va_end(ap);

		time(&now);
		dstr[0] = '\0';
		(void) strftime(dstr, sizeof(dstr), "%Y-%m-%d %H:%M:%S", localtime(&now));

		if (gNcDBM_DebugFp == NULL)
			gNcDBM_DebugFp = stderr;

		memset(spaces, ' ', sizeof(spaces));
		if (level >= 9) {
			spaces[(9 - 1) * 2] = '\0';
		} else {
			spaces[(level - 1) * 2] = '\0';
		}
		if (gNcDBM_DebugFp != NULL) {
			(void) fprintf(gNcDBM_DebugFp, "%s <%u> %s%s\n",
				dstr,
				(unsigned int) getpid(),
				spaces,
				buf
			);
			(void) fflush(gNcDBM_DebugFp);
		}
	}
}	/* NcDBM_Debug */
#endif

NcDBM_hash32_t
NcDBM_off_t_to_NcDBM_hash32_t(const off_t a)
{
	/* The database file format was designed to use 31-bit offsets,
	 * so even though we're really truncating the hi 32 bits of
	 * a 64-bit off_t (if off_t is 64 bits), it doesn't matter
	 * because that's what we want to do.
	 */
	return ((NcDBM_hash32_t) (a & 0x7FFFFFFF));
}	/* NcDBM_off_t_to_NcDBM_hash32_t */




NcDBM_hash32_t
NcDBM_DefaultHashFunction(const unsigned char *ucp, int dsize)
{
	unsigned int accum = 0;
	int c;

	while (--dsize >= 0) {
		c = *ucp++;
		accum <<= 1;	/* TODO */
		accum += c;
	}

	return (accum);
}	/* NcDBM_DefaultHashFunction */



void
NcDBM_SetHashFunction(NcDBMFile *const db, NcDBMHashFunction func)
{
	if ((db == NULL) || (strncmp(db->magic, NCDBM_MAGIC_STR, sizeof(db->magic) - 1) != 0)) {
		return;
	}

	db->func = func;
}	/* NcDBM_SetHashFunction */




int
NcDBM_LockFile(NcDBMFile *const db, int lock)
{
#ifdef HAVE_FCNTL
	struct flock l;

	if (db->useLocks != 0) {
		l.l_type = lock;
		l.l_start = (off_t) 0;
		l.l_whence = SEEK_SET;
		l.l_len = (off_t) 0;	/* everything == 0 */

		if (fcntl(fileno(db->fp), F_SETLKW, &l) < 0) {
			NCDBM_TRACE((1, "Failed to LockFile(%d) (fd=%d): %s", lock, fileno(db->fp), strerror(errno)));
			return (NCDBM_ERR_LOCK);
		}
		NCDBM_TRACE((1, "LockFile(%d)", lock));
	}
	return (NCDBM_NO_ERR);
#elif defined(WIN32)
	OVERLAPPED ol;
	HANDLE hFile;
	int fd;

	if (db->useLocks != 0) {
		/* The way we get hFile only works for MS Visual C. */
		fd = _fileno(db->fp);
		if (fd < 3)
			return (NCDBM_ERR_LOCK);

		hFile = (HANDLE) _get_osfhandle(fd);
		if ((hFile == NULL) || (hFile == INVALID_HANDLE_VALUE))
			return (NCDBM_ERR_LOCK);

		if (db->isWinNT) {
			ol.hEvent = 0;
			ol.Internal = 0;
			ol.InternalHigh = 0;
			ol.Offset = 0;
			ol.OffsetHigh = 0;

			if (lock == F_UNLCK) {
				if (! UnlockFileEx(
					hFile,
					(DWORD) 0,	// reserved
					(DWORD) 0xFFFFFFFF,	// low dword
					(DWORD) 0x7FFFFFFF, // hi dword
					&ol
				)) {
					return (NCDBM_ERR_LOCK);
				}
			} else if (lock == F_RDLCK) {
				if (! LockFileEx(
					hFile,
					(DWORD) 0,	// Don't exclusive lock for reads...
					(DWORD) 0,	// reserved
					(DWORD) 0xFFFFFFFF,	// low dword
					(DWORD) 0x7FFFFFFF, // hi dword
					&ol
				)) {
					return (NCDBM_ERR_LOCK);
				}
			} else /* F_WRLCK */ {
				if (! LockFileEx(
					hFile,
					(DWORD) LOCKFILE_EXCLUSIVE_LOCK,
					(DWORD) 0,	// reserved
					(DWORD) 0xFFFFFFFF,	// low dword
					(DWORD) 0x7FFFFFFF, // hi dword
					&ol
				)) {
					return (NCDBM_ERR_LOCK);
				}
			}
		} else {
			if (lock == F_UNLCK) {
				if (! UnlockFile(
					hFile,
					(DWORD) 0,			// off low dword
					(DWORD) 0,			// off hi dword
					(DWORD) 0xFFFFFFFF,	// n low dword
					(DWORD) 0x7FFFFFFF // n hi dword
				)) {
					return (NCDBM_ERR_LOCK);
				}
			} else {
				if (! LockFile(
					hFile,
					(DWORD) 0,			// off low dword
					(DWORD) 0,			// off hi dword
					(DWORD) 0xFFFFFFFF,	// n low dword
					(DWORD) 0x7FFFFFFF // n hi dword
				)) {
					return (NCDBM_ERR_LOCK);
				}
			}
		}
	}
	return (NCDBM_NO_ERR);
#else
	return (NCDBM_NO_ERR);
#endif
}	/* NcDBM_LockFile */



int
NcDBM_LockInt(NcDBMFile *const db, int lock, off_t o)
{
#ifdef HAVE_FCNTL
	struct flock l;

	if (db->useLocks != 0) {
		l.l_type = lock;
		l.l_start = o;
		l.l_whence = SEEK_SET;
		l.l_len = (off_t) sizeof(NcDBM_off_t);	/* lock 4 bytes */

		if (fcntl(fileno(db->fp), F_SETLKW, &l) < 0) {
			NCDBM_TRACE((1, "Failed to LockInt(%d,%ld) (fd=%d): %s", lock, (long) o, fileno(db->fp), strerror(errno)));
			return (NCDBM_ERR_LOCK);
		}
		NCDBM_TRACE((1, "LockInt(%d,%ld)", lock, (long) o));
	}
	return (NCDBM_NO_ERR);
#elif defined(WIN32)
	OVERLAPPED ol;
	HANDLE hFile;
	int fd;
	LARGE_INTEGER o_li;

	if (db->useLocks != 0) {
		/* The way we get hFile only works for MS Visual C. */
		fd = _fileno(db->fp);
		if (fd < 3)
			return (NCDBM_ERR_LOCK);

		hFile = (HANDLE) _get_osfhandle(fd);
		if ((hFile == NULL) || (hFile == INVALID_HANDLE_VALUE))
			return (NCDBM_ERR_LOCK);

		o_li.HighPart = 0;
		o_li.LowPart = 0;
		o_li.QuadPart = (LONGLONG) o;

		if (db->isWinNT) {
			ol.hEvent = 0;
			ol.Internal = 0;
			ol.InternalHigh = 0;
			ol.Offset = o_li.LowPart;
			ol.OffsetHigh = o_li.HighPart;

			if (lock == F_UNLCK) {
				if (! UnlockFileEx(
					hFile,
					(DWORD) 0,	// reserved
					(DWORD) sizeof(NcDBM_off_t),	// low dword
					(DWORD) 0, // hi dword
					&ol
				)) {
					return (NCDBM_ERR_LOCK);
				}
			} else if (lock == F_RDLCK) {
				if (! LockFileEx(
					hFile,
					(DWORD) 0,	// Don't exclusive lock for reads...
					(DWORD) 0,	// reserved
					(DWORD) sizeof(NcDBM_off_t),	// low dword
					(DWORD) 0, // hi dword
					&ol
				)) {
					return (NCDBM_ERR_LOCK);
				}
			} else /* F_WRLCK */ {
				if (! LockFileEx(
					hFile,
					(DWORD) LOCKFILE_EXCLUSIVE_LOCK,
					(DWORD) 0,	// reserved
					(DWORD) sizeof(NcDBM_off_t),	// low dword
					(DWORD) 0, // hi dword
					&ol
				)) {
					return (NCDBM_ERR_LOCK);
				}
			}
		} else {
			// Win9x doesn't implement LockFileEx()
			// (and thus doesn't support read locks)
			//
			if (lock == F_UNLCK) {
				if (! UnlockFile(
					hFile,
					(DWORD) o_li.LowPart,			// off low dword
					(DWORD) o_li.HighPart,			// off hi dword
					(DWORD) sizeof(NcDBM_off_t),	// n low dword
					(DWORD) 0						// n hi dword
				)) {
					return (NCDBM_ERR_LOCK);
				}
			} else {
				if (! LockFile(
					hFile,
					(DWORD) o_li.LowPart,			// off low dword
					(DWORD) o_li.HighPart,			// off hi dword
					(DWORD) sizeof(NcDBM_off_t),	// n low dword
					(DWORD) 0						// n hi dword
				)) {
					return (NCDBM_ERR_LOCK);
				}
			}
		}
	}
	return (NCDBM_NO_ERR);
#else
	return (NCDBM_NO_ERR);
#endif
}	/* NcDBM_LockInt */




int
NcDBM_EmptyDatabase(NcDBMFile *const db, int needseek)
{
	NcDBMFileHeader fileHeader;
	FILE *fp;
	NcDBM_hash32_t i, prime;
	NcDBM_off_t o;
	int e;

	if ((e = NcDBM_LockFile(db, F_WRLCK)) < 0)
		return (e);

	fp = db->fp;
	if ((needseek != 0) && (fseek(fp, 0L, SEEK_SET) != 0)) {
		(void) NcDBM_LockFile(db, F_UNLCK);
		return (NCDBM_ERR_SEEK);
	}

	memset(&fileHeader, 0, sizeof(NcDBMFileHeader));
	prime = fileHeader.prime = db->prime;
	strncpy(fileHeader.magic, NCDBM_MAGIC_STR, sizeof(fileHeader.magic) - 1);

	if (fwrite(&fileHeader, sizeof(fileHeader), 1, fp) != (size_t) 1) {
		(void) NcDBM_LockFile(db, F_UNLCK);
		return (NCDBM_ERR_WRITE);
	}

	/* Write out the complete hash table, including
	 * a trailer element (-1 == 0xFFFFFFFF).
	 *
	 * Since this is a new database, each hash element
	 * will be 0 (does not point to a record).
	 */
	o = 0;
	for (i=0; i<prime; i++) {
		if (fwrite(&o, sizeof(o), 1, fp) != (size_t) 1) {
			(void) NcDBM_LockFile(db, F_UNLCK);
			return (NCDBM_ERR_WRITE);
		}
	}

	/* Note that this trailer element we write as the last
	 * element in the hash array also serves a special
	 * purpose.  When we're doing an insert, we need to lock
	 * out other applications adding records, so we denote
	 * an insert lock by locking this position in the file.
	 * We don't want to lock the first byte in the file, or
	 * some such, since we don't want to hold up other apps
	 * from reading the file header.
	 */
	o = (off_t) -1;
	if (fwrite(&o, sizeof(o), 1, fp) != (size_t) 1) {
		(void) NcDBM_LockFile(db, F_UNLCK);
		return (NCDBM_ERR_WRITE);
	}

	db->dirty = 1;
	fflush(fp);
	(void) NcDBM_LockFile(db, F_UNLCK);
	return (NCDBM_NO_ERR);
}	/* NcDBM_EmptyDatabase */




int
NcDBM_VerifyDatabaseHeader(NcDBMFile *const db, int needseek)
{
	NcDBMFileHeader fh;
	NcDBM_uint32_t i;

	/* Note: We don't lock the file here for reading.
	 * Technically we should, but for speed we skip this
	 * step since we could in theory be called every
	 * time they want to lookup a record.  All this
	 * does is verify that the header appear sane,
	 * and in the rare case that some other process
	 * is updating the header, we'll either get a
	 * short read and err out (okay) or the header
	 * will appear valid enough (okay).
	 */
	 
#ifdef unix
	if ((needseek != 0) && (lseek(fileno(db->fp), 0, SEEK_SET) != (off_t) 0)) {
		return (NCDBM_ERR_NOT_NCDBM);
	}

	if (read(fileno(db->fp), &fh, sizeof(fh)) != (size_t) sizeof(fh)) {
		return (NCDBM_ERR_NOT_NCDBM);
	}
#elif defined(windows)
	if ((needseek != 0) && (_lseek(fileno(db->fp), 0, SEEK_SET) != (off_t) 0)) {
		return (NCDBM_ERR_NOT_NCDBM);
	}

	if (_read(_fileno(db->fp), &fh, sizeof(fh)) != (size_t) sizeof(fh)) {
		return (NCDBM_ERR_NOT_NCDBM);
	}
#else
	if ((needseek != 0) && (fseek(db->fp, 0L, SEEK_SET) != 0)) {
		return (NCDBM_ERR_NOT_NCDBM);
	}

	if (fread(&fh, sizeof(fh), (size_t) 1, db->fp) != (size_t) 1) {
		return (NCDBM_ERR_NOT_NCDBM);
	}
#endif

	if (strncmp(fh.magic, NCDBM_MAGIC_STR, sizeof(fh.magic) - 1) != 0) {
		/* header does not look like one of ours. */
		memcpy(&i, fh.magic, sizeof(i));
		if ((i == BDB_MAGIC_INT1) || (i == BDB_MAGIC_INT2)) {
			return (NCDBM_ERR_FILE_IS_BDB);
		} else if ((i == BDB_MAGIC_INT3) || (i == BDB_MAGIC_INT4)) {
			return (NCDBM_ERR_FILE_IS_BDB);
		} else if ((i == GDBM_MAGIC_INT1) || (i == GDBM_MAGIC_INT2)) {
			return (NCDBM_ERR_FILE_IS_GDBM);
		} else if (strncmp(fh.magic, GDBM_MAGIC_STR, strlen(GDBM_MAGIC_STR)) == 0) {
			return (NCDBM_ERR_FILE_IS_GDBM);
		}
		return (NCDBM_ERR_NOT_NCDBM);
	}

	db->prime = fh.prime;
	return (NCDBM_NO_ERR);
}	/* NcDBM_VerifyDatabaseHeader */




int
NcDBM_Open(NcDBMFile *const db, const char *const dbPathName, const int how, const int mode, const int hsize)
{
	const char *mStr;
	FILE *fp;
	int existed, e;
	int needseek;

#ifdef __SUNPRO_C
#if ((SIZEOF_PTR == 8) && (defined(HAVE_LONG_LONG)))
	unsigned long long aln = ((unsigned long long) db) % sizeof(NcDBMFile *);
#else
	unsigned long aln = ((unsigned long) db) % sizeof(NcDBMFile *);
#endif
	if (aln != 0) {
		/* Damn Sun compiler (__SUNPRO_C) did not align the structure correctly. */
		return (NCDBM_ERR_SUN_C_BUG);
	}
#endif

	gNcDBM_Unused = mode || gNcDBMVersion[0];	/* Shut up the compiler ... param is reserved for future use*/
	memset(db, 0, sizeof(NcDBMFile));

	if (how & NCDBM_O_RDWR) {
		if (how & NCDBM_O_TRUNC) {
			/* Not a true truncation!
			 * But we have to do this
			 * because fopen() syntax
			 * varies too much.
			 */
			(void) remove(dbPathName);
		}
		mStr = "r+" NCDBM_BINARY;
#if defined(windows)
		existed = 0;
		fp = _fsopen(dbPathName, "rb", _SH_DENYNO);
		if (fp != NULL) {
			(void) fclose(fp);
			existed = 1;
		}	
#else
		existed = 0;
		fp = fopen(dbPathName, "r");
		if (fp != NULL) {
			(void) fclose(fp);
			existed = 1;
		}	
#endif
		if ((!existed) && ((how & NCDBM_O_CREAT) != 0)) {
			/* Just create an empty file. */
#ifdef windows
			fp = _fsopen(dbPathName, "w", _SH_DENYNO);
#else
			fp = fopen(dbPathName, "w");
#endif
			if (fp == NULL)
				return (NCDBM_ERR_OPEN);
			(void) fclose(fp);
		}
	} else if (how & NCDBM_O_WRONLY) {
		/* We will still need to
		 * open the file in read/write.
		 */
		if (how & NCDBM_O_TRUNC) {
			(void) remove(dbPathName);
		}
		mStr = "r+" NCDBM_BINARY;
#if defined(windows)
		existed = 0;
		fp = _fsopen(dbPathName, "rb", _SH_DENYNO);
		if (fp != NULL) {
			(void) fclose(fp);
			existed = 1;
		}	
#else
		existed = 0;
		fp = fopen(dbPathName, "r");
		if (fp != NULL) {
			(void) fclose(fp);
			existed = 1;
		}	
#endif
		if ((!existed) && ((how & NCDBM_O_CREAT) != 0)) {
			/* Just create an empty file. */
#ifdef windows
			fp = _fsopen(dbPathName, "w", _SH_DENYNO);
#else
			fp = fopen(dbPathName, "w");
#endif
			if (fp == NULL)
				return (NCDBM_ERR_OPEN);
			(void) fclose(fp);
		}
	} else {
		/* read only */
		if (how & NCDBM_O_TRUNC) {
			return (-1);
		}
		mStr = "r";
		existed = 1;
	}

#ifdef windows
	fp = _fsopen(dbPathName, mStr, _SH_DENYNO);
#else
	fp = fopen(dbPathName, mStr);
#endif

	if (fp == NULL) {
		return (NCDBM_ERR_OPEN);
	}

	setvbuf(fp, db->fbuf, _IOFBF, sizeof(db->fbuf));
	needseek = 0;
	db->fp = fp;
	if (hsize < 2)
		db->prime = NCDBM_DEFAULT_HSIZE;
	else
		db->prime = hsize;

#ifdef windows
	{
		OSVERSIONINFO osv;

		db->isWinNT = 0;
		ZeroMemory(&osv, sizeof(osv));
		osv.dwOSVersionInfoSize = sizeof(osv);
		if (GetVersionEx(&osv)) {
			if (osv.dwPlatformId == VER_PLATFORM_WIN32_NT)
				db->isWinNT = 1;
		}
	}
#endif

	if (!existed) {
		/* Need to construct file header */
		if ((e = NcDBM_EmptyDatabase(db, needseek)) < 0) {
			fclose(fp);
			db->fp = NULL;
			return (e);
		}
		needseek = 1;
	}

	if ((e = NcDBM_VerifyDatabaseHeader(db, needseek)) < 0) {
		fclose(fp);
		db->fp = NULL;
		return (e);
	}

	db->func = NcDBM_DefaultHashFunction;
	strncpy(db->magic, NCDBM_MAGIC_STR, sizeof(db->magic) - 1);

	/* Pre-compute where the first level hash sits. */
	db->hOff = (off_t) sizeof(NcDBMFileHeader);

	/* And also the start of the data records. */
	db->dOff = db->hOff + ((db->prime + 1) * (off_t) sizeof(NcDBM_off_t));

	db->useLocks = 1;
	return (NCDBM_NO_ERR);
}	/* NcDBM_Open */




int
NcDBM_Close(NcDBMFile *const db)
{
	int e;

	if ((db == NULL) || (strncmp(db->magic, NCDBM_MAGIC_STR, sizeof(db->magic) - 1) != 0)) {
		/* corrupt, or already closed. */
		return (NCDBM_ERR_MEM_HOSED);
	}

	e = NCDBM_NO_ERR;
	if (db->dirty != 0) {
		if (fflush(db->fp) != 0)
			e = NCDBM_ERR_WRITE;

#ifdef HAVE_FSYNC
		if (fsync(fileno(db->fp)) < 0)
			e = NCDBM_ERR_WRITE;
#endif
	}
	(void) fclose(db->fp);
	memset(db, 0, sizeof(NcDBMFile));
	return (e);
}	/* NcDBM_Close */




int
NcDBM_Query0(
	NcDBMFile *const db,
	const NcDBMDatum key,
	NcDBM_hash32_t h,
	NcDBMDatum *const content,
	NcDBMRecordHeader *const rhp,
	off_t *hop,
	off_t *phop,
	NcDBMRecordHeader *prevrhp)
{
	NcDBM_off_t ho0;
	off_t ho, prevho;
	off_t o;
	FILE *fp;
	NcDBMRecordHeader rh;
	int rksize;
	int dsize, cdsize;
	int e = NCDBM_NO_ERR;
#ifdef unix
	int nused, nleft;
	NcDBMRecordHeader *trhp;
	char *cp;
#elif defined(windows)
	int nused, nleft;
	NcDBMRecordHeader *trhp;
	char *cp;
#else
	char rkey[NCDBM_MAX_KEY_SIZE + 1];
#endif

	/* Find the first-level hash entry for this hash value. */
	o = db->hOff + (off_t) (h * sizeof(NcDBM_off_t));

	/* This effectively locks the entire chain for ourselves. */
	if (NcDBM_LockInt(db, F_RDLCK, o) < 0)
		return (NCDBM_ERR_LOCK);

	fp = db->fp;
#ifdef unix
	(void) fflush(fp);
	if (lseek(fileno(fp), o, SEEK_SET) != o) {
		(void) NcDBM_LockInt(db, F_UNLCK, o);
		return (NCDBM_ERR_SEEK);
	}

	ho = 0;
	ho0 = 0;
	if (read(fileno(fp), &ho0, sizeof(ho0)) != sizeof(ho0)) {
		(void) NcDBM_LockInt(db, F_UNLCK, o);
		return (NCDBM_ERR_READ);
	}
	ho = (off_t) ho0;
#elif defined(windows)
	(void) fflush(fp);
	if (_lseek(_fileno(fp), o, SEEK_SET) != o) {
		(void) NcDBM_LockInt(db, F_UNLCK, o);
		return (NCDBM_ERR_SEEK);
	}

	ho = 0;
	ho0 = 0;
	if (_read(_fileno(fp), &ho0, sizeof(ho0)) != sizeof(ho0)) {
		(void) NcDBM_LockInt(db, F_UNLCK, o);
		return (NCDBM_ERR_READ);
	}
	ho = (off_t) ho0;
#else
	if (fseek(fp, (off_t) o, SEEK_SET) != 0) {
		(void) NcDBM_LockInt(db, F_UNLCK, o);
		return (NCDBM_ERR_SEEK);
	}

	ho = 0;
	ho0 = 0;
	if (fread(&ho0, sizeof(ho0), (size_t) 1, fp) != (size_t) 1) {
		(void) NcDBM_LockInt(db, F_UNLCK, o);
		return (NCDBM_ERR_READ);
	}
	ho = (off_t) ho0;
#endif

	if (prevrhp != NULL) {
		memset(prevrhp, 0, sizeof(NcDBMRecordHeader));
	}

	for (prevho = 0;;) {
		if (ho == 0) {
			/* We've exhaused the chain for this hash value,
			 * and no record matched the key.
			 */
			e = NCDBM_NOT_FOUND;

			/* Copy the file position where the record started, if requested. */
			if (hop != NULL) {
				*hop = ho;
			}

			/* Copy the file position where the previous record in the chain
			 * started, if requested.
			 */
			if (phop != NULL) {
				*phop = prevho;
			}

			(void) NcDBM_LockInt(db, F_UNLCK, o);
			return (e);
		}

#ifdef unix
		if (lseek(fileno(fp), (off_t) ho, SEEK_SET) == (off_t) -1) {
			(void) NcDBM_LockInt(db, F_UNLCK, o);
			return (NCDBM_ERR_SEEK);
		}

		trhp = (NcDBMRecordHeader *) db->fbuf;

		(void) memset(trhp, 0, sizeof(db->fbuf));
		if (read(fileno(fp), trhp, (int) sizeof(db->fbuf)) < (int) sizeof(NcDBMRecordHeader)) {
			(void) NcDBM_LockInt(db, F_UNLCK, o);
			return (NCDBM_ERR_READ);
		}

		/* Note: we may read past the keyPrefix, but since we were guaranteed to read
		 * at least 512 bytes we have enough room to fit the entire record header
		 * and maximum key size.
		 */
		rksize = (int) trhp->keySize;
		if (memcmp(trhp->keyPrefix, key.dptr, (size_t) rksize) == 0) {
			/* Found the record. */
			memcpy(&rh, trhp, sizeof(rh));
			break;
		}

		/* Read the next record that matched this hash value (if any). */
		prevho = ho;
		ho = (off_t) trhp->nextOffset;

		/* Update the saved previous record header, if asked. */
		if (prevrhp != NULL) {
			memcpy(prevrhp, trhp, sizeof(NcDBMRecordHeader));
		}
#elif defined(windows)
		if (_lseek(_fileno(fp), (off_t) ho, SEEK_SET) == (off_t) -1) {
			(void) NcDBM_LockInt(db, F_UNLCK, o);
			return (NCDBM_ERR_SEEK);
		}

		trhp = (NcDBMRecordHeader *) db->fbuf;

		(void) memset(trhp, 0, sizeof(db->fbuf));
		if (_read(_fileno(fp), trhp, (int) sizeof(db->fbuf)) < (int) sizeof(NcDBMRecordHeader)) {
			(void) NcDBM_LockInt(db, F_UNLCK, o);
			return (NCDBM_ERR_READ);
		}

		/* Note: we may read past the keyPrefix, but since we were guaranteed to read
		 * at least 512 bytes we have enough room to fit the entire record header
		 * and maximum key size.
		 */
		rksize = (int) trhp->keySize;
		if (memcmp(trhp->keyPrefix, key.dptr, (size_t) rksize) == 0) {
			/* Found the record. */
			memcpy(&rh, trhp, sizeof(rh));
			break;
		}

		/* Read the next record that matched this hash value (if any). */
		prevho = ho;
		ho = (off_t) trhp->nextOffset;

		/* Update the saved previous record header, if asked. */
		if (prevrhp != NULL) {
			memcpy(prevrhp, trhp, sizeof(NcDBMRecordHeader));
		}
#else
		if (fseek(fp, (off_t) ho, SEEK_SET) != 0) {
			(void) NcDBM_LockInt(db, F_UNLCK, o);
			return (NCDBM_ERR_SEEK);
		}

		if (fread(&rh, sizeof(NcDBMRecordHeader), (size_t) 1, fp) != (size_t) 1) {
			(void) NcDBM_LockInt(db, F_UNLCK, o);
			return (NCDBM_ERR_READ);
		}

		rksize = (int) rh.keySize;
		if (key.dsize == rksize) {
			if (rksize <= NCDBM_KEY_PREFIX_SIZE) {
				if (memcmp(rh.keyPrefix, key.dptr, (size_t) rksize) == 0) {
					/* Found the record. */
					break;
				} else {
					/* This wasn't the record; continue on to the next in the chain. */
				}
			} else {
				if (memcmp(rh.keyPrefix, key.dptr, (size_t) NCDBM_KEY_PREFIX_SIZE) == 0) {
					/* The prefix matched.  But now we have to load
					 * the remaining portion of the key to finish
					 * the compare.
					 */
	
					if (fread(rkey, (size_t) (rksize - NCDBM_KEY_PREFIX_SIZE), (size_t) 1, fp) != (size_t) 1) {
						(void) NcDBM_LockInt(db, F_UNLCK, o);
						return (NCDBM_ERR_READ);
					}

					if (memcmp(rkey, key.dptr + NCDBM_KEY_PREFIX_SIZE, (size_t) (rksize - NCDBM_KEY_PREFIX_SIZE)) == 0) {
						/* Found the record. */
						break;
					} else {
						/* This wasn't the record; continue on to the next in the chain. */
					}
				} else {
					/* This wasn't the record; continue on to the next in the chain. */
				}
			}
		}
		/* Read the next record that matched this hash value (if any). */
		prevho = ho;
		ho = (off_t) rh.nextOffset;

		/* Update the saved previous record header, if asked. */
		if (prevrhp != NULL) {
			memcpy(prevrhp, &rh, sizeof(rh));
		}
#endif	/* unix */
	}

	/* Now copy the record contents, if requested. */
	if (content != NULL) {
		if (content->dptr != NULL) {
			cdsize = content->dsize - 1;
			if ((int) rh.recSize < cdsize)
				dsize = (int) rh.recSize;
			else
				dsize = cdsize;
		} else {
			dsize = (int) rh.recSize;
			content->dptr = malloc((size_t) (dsize + 1));
			content->dsize = dsize;
			if (content->dptr == NULL) {
				(void) NcDBM_LockInt(db, F_UNLCK, o);
				return (NCDBM_ERR_MALLOC);
			}
		}

#ifdef unix
		/* OK, we've already read in the record header and key.
		 * Hopefully the record contents was small enough to
		 * also fit in the buffer we read;  if not, we'll read
		 * the rest of it.
		 */
		nused = (int) sizeof(NcDBMRecordHeader);
		if (rksize > NCDBM_KEY_PREFIX_SIZE) {
			nused += rksize - NCDBM_KEY_PREFIX_SIZE;
		}
		cp = db->fbuf + nused;
		nleft = (int) sizeof(db->fbuf) - nused;
		if (dsize < nleft) {
			memcpy(content->dptr, cp, (size_t) dsize);
		} else {
			memcpy(content->dptr, cp, (size_t) nleft);
			if (read(fileno(fp), content->dptr + nleft, dsize - nleft) != (dsize - nleft)) {
				(void) NcDBM_LockInt(db, F_UNLCK, o);
				return (NCDBM_ERR_READ);
			}
		}
#elif defined(windows)
		/* OK, we've already read in the record header and key.
		 * Hopefully the record contents was small enough to
		 * also fit in the buffer we read;  if not, we'll read
		 * the rest of it.
		 */
		nused = (int) sizeof(NcDBMRecordHeader);
		if (rksize > NCDBM_KEY_PREFIX_SIZE) {
			nused += rksize - NCDBM_KEY_PREFIX_SIZE;
		}
		cp = db->fbuf + nused;
		nleft = sizeof(db->fbuf) - nused;
		if (dsize < nleft) {
			memcpy(content->dptr, cp, (size_t) dsize);
		} else {
			memcpy(content->dptr, cp, (size_t) nleft);
			if (_read(_fileno(fp), content->dptr + nleft, dsize - nleft) != (dsize - nleft)) {
				(void) NcDBM_LockInt(db, F_UNLCK, o);
				return (NCDBM_ERR_READ);
			}
		}
#else
		if (fread(content->dptr, (size_t) dsize, (size_t) 1, fp) != (size_t) 1) {
			(void) NcDBM_LockInt(db, F_UNLCK, o);
			return (NCDBM_ERR_READ);
		}
#endif
		content->dptr[dsize] = '\0';
	}

	/* Copy the record header information, if requested. */
	if (rhp != NULL) {
		*rhp = rh;
	}

	/* Copy the file position where the record started, if requested. */
	if (hop != NULL) {
		*hop = ho;
	}

	/* Copy the file position where the previous record in the chain
	 * started, if requested.
	 */
	if (phop != NULL) {
		*phop = prevho;
	}

	(void) NcDBM_LockInt(db, F_UNLCK, o);
	return (e);
}	/* NcDBM_Query0 */



int
NcDBM_Query(NcDBMFile *const db, const NcDBMDatum key, NcDBMDatum *const content)
{
	NcDBM_hash32_t h;
	int e;

	if ((db == NULL) || (strncmp(db->magic, NCDBM_MAGIC_STR, sizeof(db->magic) - 1) != 0)) {
		/* corrupt or closed. */
		return (NCDBM_ERR_MEM_HOSED);
	}

	if ((key.dptr == NULL) || (key.dsize < 1) || (content == NULL)) {
		return (NCDBM_ERR_BAD_PARAMETER);
	}

	if (key.dsize > NCDBM_MAX_KEY_SIZE) {
		return (NCDBM_ERR_KEY_TOO_LARGE);
	}

	/* First, compute the key's hash value. */
	h = db->func((unsigned char *) key.dptr, key.dsize);
	h = h % db->prime;
	content->dptr = NULL;

	e = NcDBM_Query0(
		db,
		key,
		h,
		content,
		NULL,
		NULL,
		NULL,
		NULL
	);

	return (e);
}	/* NcDBM_Query */




int
NcDBM_Exists(NcDBMFile *const db, const NcDBMDatum key)
{
	NcDBM_hash32_t h;
	int e;

	if ((db == NULL) || (strncmp(db->magic, NCDBM_MAGIC_STR, sizeof(db->magic) - 1) != 0)) {
		/* corrupt or closed. */
		return (NCDBM_ERR_MEM_HOSED);
	}

	if ((key.dptr == NULL) || (key.dsize < 1)) {
		return (NCDBM_ERR_BAD_PARAMETER);
	}

	if (key.dsize > NCDBM_MAX_KEY_SIZE) {
		return (NCDBM_ERR_KEY_TOO_LARGE);
	}

	/* First, compute the key's hash value. */
	h = db->func((unsigned char *) key.dptr, key.dsize);
	h = h % db->prime;

	e = NcDBM_Query0(
		db,
		key,
		h,
		NULL,
		NULL,
		NULL,
		NULL,
		NULL
	);

	return (e);
}	/* NcDBM_Exists */

