/* insert.c */

#ifdef HAVE_CONFIG_H
#	include "config.h"
#elif defined(WIN32)
#	include "config.h.win32"
#	include <windows.h>
#	include <io.h>
#endif

#ifdef HAVE_UNISTD_H
#	include <unistd.h>
#	include <sys/types.h>
#endif

#ifdef HAVE_FCNTL_H
#	include <fcntl.h>
#endif

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stddef.h>
#include <ctype.h>
#include <errno.h>

#include "internal.h"
#include "ncdbm.h"

int
NcDBM_Insert(NcDBMFile *const db, NcDBMDatum key, NcDBMDatum content, NcDBMInsertMode flags)
{
	NcDBM_hash32_t h;
	NcDBMRecordHeader rh, prevrh;
	off_t o, trailero, prevho, newho;
	NcDBM_off_t newho0;
	char *cp;
	FILE *fp;
	int e;
	int ksize;
	size_t ntowrite;

	if ((db == NULL) || (strncmp(db->magic, NCDBM_MAGIC_STR, sizeof(db->magic) - 1) != 0)) {
		/* corrupt or closed. */
		return (NCDBM_ERR_MEM_HOSED);
	}

	if ((key.dptr == NULL) || (key.dsize < 1) || (content.dptr == NULL) || (content.dsize < 1)) {
		return (NCDBM_ERR_BAD_PARAMETER);
	}

	if (key.dsize > NCDBM_MAX_KEY_SIZE) {
		return (NCDBM_ERR_KEY_TOO_LARGE);
	}

	/* First, compute the key's hash value. */
	h = db->func((unsigned char *) key.dptr, key.dsize);
	h = h % db->prime;

	e = NcDBM_Query0(
		db,
		key,
		h,
		NULL,
		&rh,
		&o,
		&prevho,
		(flags == NCDBM_REPLACE) ? &prevrh : NULL
	);

	if (e == NCDBM_NOT_FOUND) {
		/* We can insert it into the database. */
		memset(&rh, 0, sizeof(rh));
		ksize = key.dsize;
		rh.keySize = (unsigned short) ksize;
		if (ksize <= NCDBM_KEY_PREFIX_SIZE) {
			/* We only need to use the standard prefix. */
			strncpy(rh.keyPrefix, key.dptr, (size_t) ksize);
		} else {
			/* The key is longer than our prefix, so we will
			 * have to write additional key bytes after the
			 * record header.
			 */
			strncpy(rh.keyPrefix, key.dptr, (size_t) NCDBM_KEY_PREFIX_SIZE);
		}
		if (content.dsize > 0x7FFF) {
			return (NCDBM_ERR_RECORD_TOO_LARGE);
		}
		rh.recSize = (unsigned short) (content.dsize);
		/* memset did: rh.nextOffset = 0; */

		/* Race condition?
		 *
		 * If two processes both "newho = ftell(fp)" at the same time,
		 * they'll both write their record at the same location.
		 *
		 * To prevent this, we have a special "insert lock" which any
		 * process appending data to the end of the data file must use.
		 * This insert lock is noted by locking the trailer element
		 * in the toplevel hash.  Remember, the toplevel hash on disk
		 * is an array of db->prime elements + 1, and that "+1" element
		 * contains the trailer node which delineates the end of the
		 * hash and the start of the record data.
		 *
		 * That trailer node also serves us here as the insert lock,
		 * so we lock that position in the datafile when we do an insert.
		 * The reason we don't use some other convenient file position,
		 * such as the first byte in the file, is that we don't want to
		 * protect the datafile header, since a process doing a query
		 * could open it for reading and try to verify the header.  On
		 * a system with mandatory locking (such as Win32), locking
		 * byte 0 (or some other header byte) would cause other processes
		 * to be unable to read the header.
		 *
		 * Since we don't re-link the hashchain until after the record
		 * contents have been appended to the file, this insert lock
		 * can be used as the only lock for the record data as well.
		 * We don't need to lock the area of the file that the new
		 * record will occupy since the chain hasn't been changed yet
		 * with a record pointer to that data.
		 */

		fp = db->fp;
	
		/* Find the first-level hash entry for this hash value. */
		trailero = (off_t) (db->hOff + (db->prime * sizeof(NcDBM_off_t)));

		if (NcDBM_LockInt(db, F_WRLCK, trailero) < 0)
			return (NCDBM_ERR_LOCK);
		
		if (fseek(fp, 0, SEEK_END) != 0) {
			(void) NcDBM_LockInt(db, F_UNLCK, trailero);
			return (NCDBM_ERR_SEEK);
		}
		
		newho = (off_t) ftell(fp);
		if (newho == -1) {
			(void) NcDBM_LockInt(db, F_UNLCK, trailero);
			return (NCDBM_ERR_SEEK);
		}

		/* Write the record header, including key prefix. */
		if (fwrite(&rh, (size_t) sizeof(rh), (size_t) 1, fp) != (size_t) 1) {
			(void) NcDBM_LockInt(db, F_UNLCK, trailero);
			return (NCDBM_ERR_WRITE);
		}
		
		if (ksize > NCDBM_KEY_PREFIX_SIZE) {
			/* We need to append the remaining portion of the key next. */
			cp = key.dptr + NCDBM_KEY_PREFIX_SIZE;
			ntowrite = (size_t) ksize - NCDBM_KEY_PREFIX_SIZE;
			if (fwrite(cp, ntowrite, (size_t) 1, fp) != (size_t) 1) {
				/* We now have a garbage record header,
				 * but nothing points to it (not so bad).
				 */
				(void) NcDBM_LockInt(db, F_UNLCK, trailero);
				return (NCDBM_ERR_WRITE);
			}
		}
		
		/* Now write the record contents. */
		if (fwrite(content.dptr, (size_t) content.dsize, (size_t) 1, fp) != (size_t) 1) {
			/* We now have a garbage record header,
			 * but nothing points to it (not so bad).
			 */
			(void) NcDBM_LockInt(db, F_UNLCK, trailero);
			return (NCDBM_ERR_WRITE);
		}

		if (fflush(fp) != 0) {
			(void) NcDBM_LockInt(db, F_UNLCK, trailero);
			return (NCDBM_ERR_WRITE);
		}

		/* Done appending to database file. */
		(void) NcDBM_LockInt(db, F_UNLCK, trailero);

		/* We now have a record appended to the database,
		 * so now we need to change the table so that it
		 * is officially registered in the chain.
		 */
		if (prevho == 0) {
			/* This key is a new entry into the hash. */

			/* Find the first-level hash entry for this hash value. */
			o = db->hOff + (off_t) (h * sizeof(NcDBM_off_t));

			if (NcDBM_LockInt(db, F_WRLCK, o) < 0)
				return (NCDBM_ERR_LOCK);

			if (fseek(fp, (off_t) o, SEEK_SET) != 0) {
				(void) NcDBM_LockInt(db, F_UNLCK, o);
				return (NCDBM_ERR_SEEK);
			}

			/* Write a new element into the first-level hash.
			 * We will then have a chain of exactly one record,
			 * which we just wrote.
			 */
			newho0 = NcDBM_off_t_to_NcDBM_hash32_t(newho);
			if (fwrite(&newho0, (size_t) sizeof(newho0), (size_t) 1, fp) != (size_t) 1) {
				(void) NcDBM_LockInt(db, F_UNLCK, o);
				return (NCDBM_ERR_WRITE);
			}		

			if (fflush(fp) != 0) {
				(void) NcDBM_LockInt(db, F_UNLCK, o);
				return (NCDBM_ERR_WRITE);
			}
			
			/* Done updating chain. */
			(void) NcDBM_LockInt(db, F_UNLCK, o);
		} else {
			if (NcDBM_LockInt(db, F_WRLCK, prevho) < 0)
				return (NCDBM_ERR_LOCK);
			
			/* Go back to where the last record in the chain is. */
			if (fseek(fp, (off_t) prevho, SEEK_SET) != 0) {
				(void) NcDBM_LockInt(db, F_UNLCK, prevho);
				return (NCDBM_ERR_SEEK);
			}

			/* Read the last item in the chain's record header. */
			if (fread(&rh, (size_t) sizeof(rh), (size_t) 1, fp) != (size_t) 1) {
				(void) NcDBM_LockInt(db, F_UNLCK, prevho);
				return (NCDBM_ERR_READ);
			}

			/* Change the chain so that this record now points
			 * to the new record we just wrote.
			 */
			rh.nextOffset = NcDBM_off_t_to_NcDBM_hash32_t(newho);

			/* Go back to where the last record in the chain is (again). */
			if (fseek(fp, (off_t) prevho, SEEK_SET) != 0) {
				(void) NcDBM_LockInt(db, F_UNLCK, prevho);
				return (NCDBM_ERR_SEEK);
			}

			/* Write out the updated header. */
			if (fwrite(&rh, (size_t) sizeof(rh), (size_t) 1, fp) != (size_t) 1) {
				(void) NcDBM_LockInt(db, F_UNLCK, prevho);
				return (NCDBM_ERR_WRITE);
			}		

			if (fflush(fp) != 0) {
				(void) NcDBM_LockInt(db, F_UNLCK, prevho);
				return (NCDBM_ERR_WRITE);
			}
				
			/* Done updating chain. */
			(void) NcDBM_LockInt(db, F_UNLCK, prevho);
		}

		/* Done inserting. */
		db->dirty = 1;
	} else if (e == NCDBM_NO_ERR) {
		/* There is already a record.  We can replace it
		 * if the caller said to, otherwise we return
		 * an error saying the record exists.
		 */
		if (flags != NCDBM_REPLACE)
			return (NCDBM_ERR_ALREADY_EXISTS);

		/* If the record length is greater than or equal to the
		 * size of the new contents, we can recycle this 
		 * area in the file.  Otherwise, we have to mark this
		 * area as deleted and append a new record to the end
		 * of the file.
		 */
		fp = db->fp;
		if ((int) rh.recSize >= content.dsize) {
			/* write new record contents on top of previous. */

			/* This effectively locks the entire chain for ourselves.
			 * We also may write to any record's data in the chain,
			 * without having to lock the area occupied by the data.
			 */
			if (NcDBM_LockInt(db, F_WRLCK, o) < 0)
				return (NCDBM_ERR_LOCK);

			if (fseek(fp, (off_t) o, SEEK_SET) < 0) {
				(void) NcDBM_LockInt(db, F_UNLCK, o);
				return (NCDBM_ERR_SEEK);
			}
	
			/* update the record header */
			rh.recSize = (unsigned short) content.dsize;
			if (fwrite(&rh, (size_t) sizeof(rh), (size_t) 1, fp) != (size_t) 1) {
				(void) NcDBM_LockInt(db, F_UNLCK, o);
				return (NCDBM_ERR_WRITE);
			}

			if (fflush(fp) != 0) {
				(void) NcDBM_LockInt(db, F_UNLCK, o);
				return (NCDBM_ERR_WRITE);
			}

			/* Find the start of the data to overwrite
			 * Basically we skip the rest of the key to get to it.
			 */
			ksize = key.dsize;
			o += sizeof(rh);
			if (ksize > NCDBM_KEY_PREFIX_SIZE)
				o += ksize - NCDBM_KEY_PREFIX_SIZE;

			if (fseek(fp, (off_t) o, SEEK_SET) != 0) {
				/* record is now corrupt! */
				(void) NcDBM_LockInt(db, F_UNLCK, o);
				return (NCDBM_ERR_SEEK);
			}

			if (fwrite(content.dptr, (size_t) content.dsize, (size_t) 1, fp) != (size_t) 1) {
				/* record is now corrupt! */
				(void) NcDBM_LockInt(db, F_UNLCK, o);
				return (NCDBM_ERR_WRITE);
			}

			if (fflush(fp) != 0) {
				(void) NcDBM_LockInt(db, F_UNLCK, o);
				return (NCDBM_ERR_WRITE);
			}

			(void) NcDBM_LockInt(db, F_UNLCK, o);
		} else {
			/* Delete this record and add a new one. */
			e = NcDBM_Delete0(db, h, &rh, prevho, &prevrh);

			if (e == NCDBM_NO_ERR) {
				/* This could be better optimized. */
				e = NcDBM_Insert(db, key, content, flags);
			}
		}

		/* Done replacing. */
		db->dirty = 1;
		if (fflush(fp) != 0) {
			return (NCDBM_ERR_WRITE);
		}
	} else {
		return (e);
	}

	return (NCDBM_NO_ERR);
}	/* NcDBM_Insert */




int NcDBM_ExclusiveLock(NcDBMFile *const db, int lockmode)
{
	/* The purpose of this function is so that you can
	 * avoid repeated locking/unlocking for each record
	 * of a data import.  We can speed up the import
	 * significantly by doing a single lock at the
	 * start of the import, doing the entire import,
	 * and then unlocking the file.  The problem is
	 * that this will lock out other processes for
	 * what could be a very long time, which is why
	 * you should use discretion when using this.
	 *
	 * The way we do this is to lock the file using
	 * our standard function, and then temporarily
	 * turn locking off.  Then the functions that
	 * would normally lock during individual records
	 * will not do this.
	 */
	if (lockmode != F_UNLCK) {
		/* lock */
		db->ouseLocks = db->useLocks;
		if (NcDBM_LockFile(db, lockmode) < 0) {
			db->useLocks = db->ouseLocks;
			NCDBM_TRACE((1, "Failed to ExclusiveLock(%d).", lockmode));
			return (NCDBM_ERR_LOCK);
		}
		NCDBM_TRACE((1, "ExclusiveLock(%d).", lockmode));
		db->useLocks = 0;
		NCDBM_TRACE((2, "db->useLocks = %d, db->ouseLocks = %d.", db->useLocks, db->ouseLocks));
	} else {
		/* unlock (F_UNLCK) */
		db->useLocks = db->ouseLocks;
		if (NcDBM_LockFile(db, lockmode) < 0) {
			NCDBM_TRACE((1, "Failed to ExclusiveUnlock(%d).", lockmode));
		} else {
			NCDBM_TRACE((1, "ExclusiveUnlock(%d).", lockmode));
		}
		NCDBM_TRACE((2, "db->useLocks = %d, db->ouseLocks = %d.", db->useLocks, db->ouseLocks));
	}
	return (NCDBM_NO_ERR);
}	/* NcDBM_ExclusiveLock */
