#!/usr/bin/perl -w

#############################################################################
#
# Copyright (C) 2004-2006, NcFTP Software.
# All Rights Reserved.
# Version: 2.8.4
#
#############################################################################
#
# How to use this CGI script:
#
# This CGI script can be used to allow system administrators to provide
# web-based reports based off of the data from the NcFTPd Server log files.
#
#   The full documentation is at: http://www.ncftp.com/ncftpd/doc/reports/
#
#
# Install this script in an appropriate directory so that your web server
# can access this file and run it as the web server user.  Typically, this
# means you put the script in a "cgi-bin" directory.  You may also need
# to configure the web server so that files with the .cgi extension are
# executable (for example, on Apache, you would configure httpd.conf so a
# "AddHandler cgi-script .cgi" line is present).  You can also rename this
# script to reports.pl, if you have Perl scripts set up already.
#
# Since you most likely do not want world access to your reporting data,
# make sure the script is in a password protected directory (i.e. an
# .htaccess file is present and configured properly).  For highest security,
# have the script only run from an https:// URL.
#
# This script needs to be able to read the NcFTPd log files, so make sure
# the web server user ("httpd", "web", "www", or whatever your user your
# web service runs as) has read access to the files _and_ read+execute
# permission on all directories leading up to the log files.  For example,
# if your log files are in /var/log/ncftpd, check the permissions on /var,
# /var/log, and /var/log/ncftpd to be sure the user can access each
# directory.  (To have NcFTPd generate the log files automatically with
# the correct ownerships, use the general.cf "log-owner" option;
# http://www.ncftp.com/ncftpd/doc/config/g/log-owner.html)
#
# Similarly, the script needs to be able to read the NcFTPd configuration
# files (general.cf, domain.cf), which means those files need to be
# readable by the web server user.
#
# If you want the reports to be able to output graphs, then the setup is
# more complicated.  Set the $plotting_graphs variable below to "no"
# if you don't want the graphing capability, otherwise do the following:
#
#   (1) Create a directory for this script to write the report data files,
#       which is writable by the web server. For example, if your web
#       server software serves the /home/httpd/htdocs directory,
#       create your directory somewhere within that directory tree.
#       This directory should only be used by this script; we suggest
#       a directory like /home/httpd/htdocs/tmp/ncftpd.
#   (2) Skip to the next section below and set the $DocumentRoot and
#       $DocumentRoot_subdir_for_reports variables.  With the example
#       directory /home/httpd/htdocs/tmp/ncftpd, your $DocumentRoot would
#       be "/home/httpd/htdocs" and the subdirectory to write to would
#       be "tmp/ncftpd".
#   (3) Don't forget to make the directory writable by the web user,
#       e.g., "chmod 770 /home/httpd/htdocs/tmp/ncftpd" and
#       "chown webuser:webgroup /home/httpd/htdocs/tmp/ncftpd".
#   (4) Make sure you have gnuplot installed and have set the $gnuplot
#       variable below to the pathname.
#   (5) Set the $plotting_graphs variable below to "yes".
#   (6) Create a cron task to clean the directory.  This script has some
#       functionality to do that built-in.  Here's an example crontab
#       entry which purges reports older than 15 minutes:
#          0 * * * * /home/httpd/cgi-bin/reports.cgi -k 15 >/dev/null 2>&1
#
# Don't forget to make the script file readable and executable by the
# web server user (i.e. "chmod 755 reports.cgi").
#
# Run your web browser and enter in the URL to the script and you're
# done! If there are problems, they will be reported on the web page;
# if there are problems running the report script, check your web server
# log files (i.e. "error_log", etc.).
#
# (Skip to the next section)
#############################################################################

require 5.004;
use strict;
use CGI qw(:all escapeHTML);
use CGI::Carp qw(fatalsToBrowser);
use File::Basename;
use IPC::Open2;
use Time::Local;
use POSIX;
use Getopt::Std;

#############################################################################
# Change the following values as needed.
#############################################################################

my ($general_cf)			= "/usr/local/etc/ncftpd/general.cf";
my ($domain_cf)				= "/usr/local/etc/ncftpd/domain.cf";
my ($plotting_graphs)			= "yes";
my ($gnuplot)				= "/usr/bin/gnuplot";
my ($DocumentRoot)			= "/home/httpd/htdocs";
my ($DocumentRoot_subdir_for_reports)	= "tmp/ncftpd";

#############################################################################
# No more user-configurable options below this point.
#############################################################################

$CGI::POST_MAX=1024 * 4;		# max 4K posts
$CGI::DISABLE_UPLOADS = 1;		# no uploads
my ($png_graph_output)			= "yes";
my ($gnuplot_lines_style)		= "lines";
my ($gnuplot_boxes_style)		= "boxes";	# Could try "boxes fs solid"
##my ($gnuplot_png_term) 		= "png small color";	# Older version of gnuplot needs this
my ($gnuplot_png_term) 			= "png small";
my ($gnuplot_gif_term) 			= "gif small";
my ($th_bgcolor)			= "#A2D0FF";
my ($tc_bgcolor)			= "#9bbad6";
my ($td_bgcolor)			= "#e4ecf6";
my ($t_cellspacing)			= "0";
my ($t_cellpadding)			= "5";
my ($t_border)				= "1";
my ($t_bgcolor)				= "black";
my ($last_field)			= 60;
my (%cookies) 				= ();
my ($started_html) 			= 0;
my ($finished_html) 			= 0;
my ($debug) 				= 0;
my ($op)				= "";
my ($script)				= script_name();
my ($odir)				= "";
my ($relodir)				= "";
my (@domains)				= ();
my ($domain_filter)			= "ALL";
my ($t0)				= -1;
my ($t1)				= -1;
my ($d0)				= "";
my ($d1)				= "";
my ($stn0)				= "";
my ($stn1)				= "";
my ($df0)				= "";
my ($df1)				= "";
my (%domain_log_xfer)			= ();
my (%domain_log_sess)			= ();
my ($log_stat)				= "";
my (@logfiles)				= ();
my ($logsused)				= 0;
my ($logidx)				= 0;
my ($logtype)				= "log-xfer";
my ($log_is_open)			= 0;
my ($logpath)				= 0;
my (@logfields)				= ();
my (@stats)				= ();
my ($logline)				= "";
my ($nloglines)				= 0;
my ($nlogbytes)				= 0;
my (@warnings)				= ();
my ($report_title)			= "";
my ($table_started)			= 0;
my ($nrows)				= 0;
my ($topN)				= 0;
my ($topN_warning)			= 0;
my ($nrows_warning)			= 1;
my ($Xtype) 				= "Download";
my ($xtype) 				= "R";
my (%event_types)			= ();
my (%event_mask)			= ();
my ($sort_by)				= "";
my ($otdir)				= "";
my ($relotdir)				= "";
my ($xlabel) 				= "";
my ($xtics)				= "";
my ($gp_term)				= "";
my ($gp_fmt)				= "";
my (@window_stats)			= ();
my ($window_lines_used) 		= 0;
my ($window_size) 			= 0;
my ($curwin) 				= 0;
my ($win) 				= 0;
my ($wt0) 				= 0;
my ($wt1) 				= 0;
my (@stat_windows)			= ();
my ($num_stat_windows)			= 0;
my (@window_stat_field_type)		= ();
my ($start_dir)				= "";
my ($purge_minimum_seconds)		= 900;
my ($start_time)			= time();
my ($using_html)			= 0;
my ($title) 				= "";
my ($browser_supports_multipart) 	= 0;
my ($browser) 				= "Unknown";
my ($boundary)				= "";
my ($use_multipart_when_possible)	= "yes";
my ($using_multipart) 			= 0;
my ($num_multi_parts)			= 0;
my ($ticker)				= 0;
my ($ticker_interval)			= 3;

sub CookieHeader
{
	if (scalar(values %cookies) > 0) {
		return header(-cookie=>[values(%cookies)]);
	} else {
		return header();
	}
}	# CookieHeader




sub MultipartStatus
{
	my ($statusPage) = $_[0] || "";
	my ($sectionTitle) = $_[1] || "NcFTPd Reports: Processing...";
	
	return if (($using_multipart == 0) || ($started_html > 0));
	if ($num_multi_parts++ == 0) {
		#
		# Setup page for multi-part.  This means we have to
		# be sure and finish the multi-part page later.
		#
		# Use line-buffered output so web browser sees our
		# output immediately.
		#
		$| = 1;
		print
			"Content-Type: multipart/x-mixed-replace;boundary=\"$boundary\"\r\n\r\n",
			"WARNING: Your browser does not accept multi-part pages.\r\n\r\n";
		alarm($ticker_interval);
	}

	print
		"--${boundary}\r\nContent-Type: text/html\r\n\r\n",
		start_html($sectionTitle), "\n\n",
		"<p>",
		$statusPage,
		"\n",
		end_html(),
		"\n";
	
	$ticker = 0;
}	# MultipartStatus



sub StartHTML
{
	$title = $_[0] || "";

	if ($finished_html) {
		die("Already wrote a complete HTML page to remote browser.");
	} elsif ($started_html++ == 0) {
		if ($num_multi_parts > 0) {
			print "--${boundary}\r\n";
		}
		$| = 0;
		print CookieHeader(), start_html($title), "\n\n";
		# DebugParams();
	} else {
		# Start another (non-multipart) section in the same page!
		print "\n\n\n<hr>\n<h3>$title</h3>\n\n";
		return (0);
	}
	return (1);
}	# StartHTML




sub CheckForMultipartCapability
{
	my ($uagent) = user_agent() || "";
	
	if ($uagent eq "") {
		$browser = "Shell Prompt";
		$browser_supports_multipart = 1;	# For debugging
	} elsif ($uagent =~ /(Opera)/) {
		$browser = $1;
		$browser_supports_multipart = 1;
	} elsif ($uagent =~ /(MSIE|Safari|KHTML)/) {
		$browser = $1;
		$browser_supports_multipart = 0;
	} elsif ($uagent =~ /Mozilla\/.*Gecko/) {
		$browser = "Mozilla";
		$browser_supports_multipart = 1;
	} else {
		$browser = "Unknown";
		$browser_supports_multipart = 0;
	}
	$boundary = '================section-boundary================';
	if (($browser_supports_multipart) && ($use_multipart_when_possible eq "yes")) {
		$using_multipart = 1;
	}
}	# CheckForMultipartCapability




sub EndHTML
{
	if (($started_html > 0) && ($finished_html++ == 0)) {
		print end_html(), "\n";
		print "\r\n--${boundary}--\r\n" if ($num_multi_parts > 0);
	}
}	# EndHTML




sub ErrorPage
{
	StartHTML("NcFTPd Reports: Error");
	print "<p><b>Error</b>: ";
	print @_;
	print "</p>";
}	# ErrorPage




sub LsLd
{
	my ($output, @lsitems) = @_;
	local (*LSR, *LSW);
	my ($pid);
	my ($rc) = 0;
	my ($line);
	my ($old_sigpipe);

	$old_sigpipe = $SIG{PIPE};
	$SIG{PIPE} = 'IGNORE';

	$$output = "";
	$pid = open2(\*LSR, \*LSW, "/bin/ls", "-ld", @lsitems);
	if ($pid > 0) {
		close(LSW);
		while (defined($line = <LSR>)) {
			$$output .= $line;
		}
		close(LSR);
		waitpid($pid, 0);
		$rc = 1;
	}

	$SIG{PIPE} = $old_sigpipe;
	return ($rc);
}	# LsLd




sub NotAccessibleThenPrintError
{
	my ($pathname, $ftype) = @_;
	my ($need_r) = 0;
	my ($need_w) = 0;
	my ($need_x) = 0;
	my ($fail) = 0;

	$ftype = "-r--" if (! defined($ftype));
	return (-2) if (($ftype !~ /^[d\-]/) || (! defined($pathname)) || ($pathname eq ""));

	# Coalesce multiple slashes
	$pathname =~ s-/{2,}-/-g;

	# Remove trailing slashes
	$pathname =~ s-/+$--;

	$need_r = 1 if (index($ftype, "r") >= 0);
	$need_w = 1 if (index($ftype, "w") >= 0);
	$need_x = 1 if (index($ftype, "x") >= 0);
	$ftype = substr($ftype, 0, 1);

	$fail++ if (($ftype eq "-") && (! -f $pathname));
	$fail++ if (($ftype eq "d") && (! -d $pathname));
	if (! $fail) {
		#
		# It exists at least.
		#
		$fail++ if (($need_r) && (! -r _));
		$fail++ if (($need_w) && (! -w _));
		$fail++ if (($need_x) && (! -x _));
	}

	return (0) if ($fail == 0);

	StartHTML("Access error");

	if ($ftype eq "-") {
		if ($need_x) {
			print p(b("Error: "), "Cannot run <tt>$pathname</tt>."), "\n";
		} elsif (($need_r) && ($need_w)) {
			print p(b("Error: "), "Cannot access the file <tt>$pathname</tt> for reading and writing."), "\n";
		} elsif ($need_r) {
			print p(b("Error: "), "Cannot access the file <tt>$pathname</tt> for reading."), "\n";
		} elsif ($need_w) {
			print p(b("Error: "), "Cannot access the file <tt>$pathname</tt> for writing."), "\n";
		} else {
			print p(b("Error: "), "Cannot access the file <tt>$pathname</tt>."), "\n";
		}
	} else {
		if (($need_r) && ($need_w) && ($need_x)) {
			print p(b("Error: "), "Cannot access the directory <tt>$pathname</tt> for traversal (execution), listing (reading), creating and removing (writing)"), "\n";
		} elsif (($need_r) && ($need_x)) {
			print p(b("Error: "), "Cannot access the directory <tt>$pathname</tt> for traversal (execution) and listing (reading)"), "\n";
		} elsif (($need_w) && ($need_x)) {
			print p(b("Error: "), "Cannot access the directory <tt>$pathname</tt> for traversal (execution), and creating/removing (writing)"), "\n";
		} else {
			if ($need_x) {
				print p(b("Error: "), "Cannot access the directory <tt>$pathname</tt> for traversal (execution)"), "\n";
			}
			if ($need_r) {
				print p(b("Error: "), "Cannot access the directory <tt>$pathname</tt> for listing (reading)"), "\n";
			}
			if($need_w) {
				print p(b("Error: "), "Cannot access the directory <tt>$pathname</tt> for creating and removing (writing)"), "\n";
			}
		}
	}

	my ($usr) = getpwuid($>) || "UID $>";
	my (@gids) = split(/\s/, $));
	my ($grps) = "";
	my ($gid, $gid1, $grp, $skipgrp, $ngrp);
	$gid1 = "none";

	for $gid (@gids) {
		$skipgrp = 0;
		if ($gid1 eq "none") {
			$gid1 = $gid;
		} else {
			$skipgrp = 1 if ($gid == $gid1);
		}
		if (! $skipgrp) {
			$grp = getgrgid($gid);
			# $grp .= " ($gid)";
			if ($grps eq "") {
				$grps = tt($grp);
			} else {
				$grps .= ", " . tt($grp);
			}
		}
	}

	print p(
		tt(script_name()),
		"is running as user ", tt($usr), " and ",
		scalar(@gids) > 1 ? "groups" : "group",
		"$grps."
	), "\n";

	my ($lsitem) = $pathname;
	my (@lsitems) = ();
	while ($lsitem ne "") {
		push(@lsitems, $lsitem);
		$lsitem =~ s-/[^/]+$--;
	}

	my ($stuff);
	if (LsLd(\$stuff, @lsitems)) {;
		print p(ul(pre(escapeHTML($stuff)))), "\n";
	}

	$fail = 0;
	for $lsitem (reverse(@lsitems)) {
		if (! -e $lsitem) {
			my ($parent_dir) = $lsitem;
			if ($lsitem =~ /^(.*)\//) {
				$parent_dir = $1;
			} else {
				# Don't give us relative paths
				$parent_dir = "..";
			}
			if (($need_w) && (! -w $parent_dir)) {
				print p(b("Reason:"), "parent directory", tt($parent_dir), "is not writable."), "\n";
			} elsif ((($need_r) || ($need_x)) && (! -x $parent_dir)) {
				print p(b("Reason:"), "parent directory", tt($parent_dir), "is not traversable (executable)."), "\n";
			} else {
				print p(b("Reason: "), tt($lsitem), "does not exist."), "\n";
			}
			$fail++;
			last;
		} elsif ((-d $lsitem) && (! -x $lsitem)) {
			print p(b("Reason: "), "directory", tt($lsitem), "is not traversable (executable)."), "\n";
			$fail++;
			last;
		}
	}

	if ($fail == 0) {
		#
		# Then the item exists -- but it earlier failed one of our
		# other tests, so print what that was.
		#
		if ($ftype eq "d") {
			$ftype = "directory";
		} else {
			$ftype = "file";
		}
		stat($pathname);

		
		if (($need_r) && (! -r _)) {
			print p(b("Reason: "), "the $ftype", tt($pathname), "is not readable."), "\n";
		}
		if (($need_w) && (! -w _)) {
			print p(b("Reason: "), "the $ftype", tt($pathname), "is not writable."), "\n";
		}
		if (($need_x) && (! -x _)) {
			print p(b("Reason: "), "the $ftype", tt($pathname), "is not executable."), "\n";
		}
	}

	return (-1);
}	# NotAccessibleThenPrintError




sub DebugParams
{
	my (@params) = param();
	my ($parm);

	print "\n\n", comment("Cookies and CGI parameters are listed here."), "\n";
	print "<TABLE border>\n";
	print "\t", Tr(th({-bgcolor=>"black"}, "<font color=white>Parameters</font>")), "\n";
	if (scalar(@params) == 0) {
		print "\t", Tr(td({-align=>"center"}, "(none)")), "\n";
	} else {
		print "\t<TR><TD>\n";
		print "\t\t<TABLE cellspacing=\"5\">\n";
		for $parm (@params) {
			$parm = escapeHTML($parm);
			print "\t\t\t", Tr(th({-align=>"right"}, $parm), td(escapeHTML(param($parm)))), "\n";
		}
		print "\t\t</TABLE>\n";
		print "\t</TR></TD>\n";
	}

	print "\t", Tr(th({-bgcolor=>"black"}, "<font color=white>Cookies</font>")), "\n";
	@params = cookie();
	if (scalar(@params) == 0) {
		print "\t", Tr(td({-align=>"center"}, "(none)")), "\n";
	} else {
		print "\t<TR><TD>\n";
		print "\t\t<TABLE cellspacing=\"5\">\n";
		for $parm (@params) {
			print "\t\t\t", Tr(th({-align=>"right"}, $parm), td(escapeHTML(cookie($parm)))), "\n";
		}
		print "\t\t</TABLE>\n";
		print "\t</TR></TD>\n";
	}

	print "</TABLE>\n\n";
}	# DebugParams




sub DebugDateRange
{
	my ($d0a, $d1a);

	$d0a = POSIX::strftime("%Y-%m-%d %H:%M:%S", localtime($t0));
	$d1a = POSIX::strftime("%Y-%m-%d %H:%M:%S", localtime($t1));

	print "<!--\n";
	print "start: t0 = $t0   in: $d0   out: $d0a\n";
	print "end:   t1 = $t1   in: $d1   out: $d1a\n";
	print "-->\n";
}	# DebugDateRange




sub Mkdirs
{
	my ($path, $mode) = @_;
	my ($subpath);
	my ($i, $n, $e);
	my (@nodes);
	my($absolute) = 0;

	if (!defined($mode)) { $mode = 00777; }		# Umask will affect.
	@nodes = split(/\/+/, $path);

	$n = scalar(@nodes);
	return 1 if ($n <= 0);		# It was just the root directory

	if ($nodes[0] eq "") {
		#
		# Absolute paths will leave us with an
		# empty first node.
		#
		$absolute = 1;
		shift(@nodes);
		$n--;
	}

	#
	# Find the greatest part of the path that
	# already exists.  We will then create the
	# remaining nodes.
	#
	while (1) {
		$subpath = "";
		if ($absolute) { $subpath = "/"; }
		for ($i = 0; $i < $n; $i++) {
			$subpath .= $nodes[$i] . "/";
		}
		chop($subpath);
		last if (-d "$subpath");
		if (--$n <= 0) {
			$subpath = "";
			$n = 0;
			last;
		}
	}

	#
	# We now have in $subpath the portion of $path
	# that already exists.  We now just need to
	# create the remaining node(s).  We have in $n
	# the number of successive nodes which exist.
	#
	$e = $n;
	$n = scalar(@nodes);
	return 1 if ($e == $n);			# Entire path already exists

	for ($i = $e ; $i < $n; $i++) {
		if (($subpath eq "") && (! $absolute)) {
			$subpath = $nodes[$i];
		} else {
			$subpath .= "/" . $nodes[$i];
		}
		return 0 if (! mkdir($subpath,$mode));
	}

	return 1;				# Finished
}	# Mkdirs




sub LoadGeneralCf
{
	my ($line);
	my ($var, $val);
	my ($set_name) = "ALL";

	if (! open(CF, "< $general_cf")) {
		if (NotAccessibleThenPrintError($general_cf, "-r--") >= 0) {

			ErrorPage("Could not load <tt>general.cf</tt> file <tt>$general_cf</tt>: $?.\n");
		}
		return (0);
	}
	while (defined($line = <CF>)) {
		next unless ($line =~ /^\s*([0-9A-Za-z_\-]+)\s*\=\s*(\S+)/);
		$var = $1;
		$val = $2;
		$log_stat = $val if ($var eq "log-stat");
	}
	close(CF);
	return (1);
}	# LoadGeneralCf




sub LoadDomainCf
{
	my ($line);
	my ($var, $val);
	my ($set_name);

	if (! open(CF, "< $domain_cf")) {
		if (NotAccessibleThenPrintError($domain_cf, "-r--") >= 0) {

			ErrorPage("Could not load <tt>domain.cf</tt> file <tt>$domain_cf</tt>: $?.\n");
		}
		return (0);
	}
	while (defined($line = <CF>)) {
		next unless ($line =~ /^\s*([0-9A-Za-z_\-]+)\s*\=\s*(\S+)/);
		$var = $1;
		$val = $2;
		$val =~ s/\@SETNAME\@/$set_name/g;
		if ($var eq "set-name") {
			$set_name = $val;
			push(@domains, $set_name);
		}
		$domain_log_xfer{$set_name} = $val if ($var eq "log-xfer");
		$domain_log_sess{$set_name} = $val if ($var eq "log-session");
	}
	close(CF);
	return (1);
}	# LoadDomainCf




sub GetLogList
{
	my (%logtrack) = ();
	my ($t);
	my ($domain);
	my ($prevlogpath) = "";
	my ($logpath);
	my ($prevlogdir) = "";
	my ($logdir) = "";
	my (@domains_to_check) = ();
	
	die("Assertion failed") if (($t0 == 0) || ($t0 == -1) || ($t1 == 0) || ($t1 == -1) || (! defined($t0)) || (! defined($t1)));
	
	if ($domain_filter eq "ALL") {
		@domains_to_check = @domains;
	} else {
		push(@domains_to_check, $domain_filter);
	}

	for ($t = $t0 ; $t <= $t1; $t += 3600) {
		for $domain (@domains_to_check) {
			if ($logtype eq "log-xfer") {
				$logpath = POSIX::strftime($domain_log_xfer{$domain}, localtime($t));
			} elsif ($logtype eq "log-stat") {
				$logpath = POSIX::strftime($log_stat, localtime($t));
			} else {
				$logpath = POSIX::strftime($domain_log_sess{$domain}, localtime($t));
			}
			next if ($logpath eq $prevlogpath);
			$prevlogpath = $logpath;
			$logdir = dirname($logpath);
			if ($logdir ne $prevlogdir) {
				return (0) if (NotAccessibleThenPrintError($logdir, "dr-x") < 0);
				$logdir = $prevlogdir;
			}
			if (! -e $logpath) {
				if ($logtype ne "log-stat") {
					push(@warnings, "Missing domain <tt>$domain</tt> log file: <tt>$logpath</tt>.");
				}
			} elsif (! exists($logtrack{$logpath})) {
				$logtrack{$logpath} = $t;
				push(@logfiles, $logpath);
				return (0) if (NotAccessibleThenPrintError($logpath, "-r--") < 0);
			}
		}
	}
	return (1);
}	# GetLogList




sub PrepareToIterateLogs
{
	die("Assertion failed") if ((! $logtype) || (! $domain_filter));
	return (0) unless LoadDomainCf();
	return (0) unless LoadGeneralCf();
	return (0) unless GetLogList();
	if (scalar(@logfiles) == 0) {
		ErrorPage("No <tt>$logtype</tt> logs found between $d0 and $d1.\n");
		return (0);
	}
	return (1);
}	# PrepareToIterateLogs



sub BuildPlotdataFile
{
	my (@statlogfields) = ();
	my ($interval, $fld, $i);
	my ($plotdata, $start_ts, $flds)	= @_;
	my (@fldnums) = split(/,/, $flds);
	my ($nf) = scalar(@fldnums);
	
	if ($nf < 2) {
		if (! open(PDATA1, "> $plotdata")) {
			ErrorPage("Could not create $plotdata: $?.\n");
			close(LOG);
			return (-1);
		}
	} else {
		if (($nf >= 1) && (! open(PDATA1, "> $plotdata.1"))) {
			ErrorPage("Could not create $plotdata.1: $?.\n");
			close(LOG);
			return (-1);
		}
		if (($nf >= 2) && (! open(PDATA2, "> $plotdata.2"))) {
			ErrorPage("Could not create $plotdata.2: $?.\n");
			close(LOG);
			close(PDATA1);
			return (-1);
		}
		if (($nf >= 3) && (! open(PDATA3, "> $plotdata.3"))) {
			ErrorPage("Could not create $plotdata.3: $?.\n");
			close(LOG);
			close(PDATA1);
			close(PDATA2);
			return (-1);
		}
		if (($nf >= 4) && (! open(PDATA4, "> $plotdata.4"))) {
			ErrorPage("Could not create $plotdata.4: $?.\n");
			close(LOG);
			close(PDATA1);
			close(PDATA2);
			close(PDATA3);
			return (-1);
		}
		if (($nf >= 5) && (! open(PDATA5, "> $plotdata.5"))) {
			ErrorPage("Could not create $plotdata.5: $?.\n");
			close(LOG);
			close(PDATA1);
			close(PDATA2);
			close(PDATA3);
			close(PDATA4);
			return (-1);
		}
	}

	for ($i=0; $i<$num_stat_windows; $i++) {
		@statlogfields = split(/,/, $stat_windows[$i]);
		$interval = int(($statlogfields[2 - 1] - $start_ts) / $window_size);
		
		printf PDATA1 ("%d %s\n", $interval, $statlogfields[$fldnums[1 - 1] - 1]) if ($nf >= 1);
		printf PDATA2 ("%d %s\n", $interval, $statlogfields[$fldnums[2 - 1] - 1]) if ($nf >= 2);
		printf PDATA3 ("%d %s\n", $interval, $statlogfields[$fldnums[3 - 1] - 1]) if ($nf >= 3);
		printf PDATA4 ("%d %s\n", $interval, $statlogfields[$fldnums[4 - 1] - 1]) if ($nf >= 4);
		printf PDATA5 ("%d %s\n", $interval, $statlogfields[$fldnums[5 - 1] - 1]) if ($nf >= 5);
	}

	close(PDATA1) if ($nf >= 1);
	close(PDATA2) if ($nf >= 2);
	close(PDATA3) if ($nf >= 3);
	close(PDATA4) if ($nf >= 4);
	close(PDATA5) if ($nf >= 5);
	
	return (0);
}	# BuildPlotdataFile



sub BuildGnuplotScriptFile
{
	my ($gpscript, $gp_term, $title, $xlabel, $xtics, $img, $gpcmd) = @_;
	
	if (open(GPSCRIPT, "> $gpscript")) {
print GPSCRIPT <<EOF;
			set term $gp_term
			set title '$title'
			set xlabel '$xlabel'
			set xtics $xtics
			set size 0.9,0.5
			set output '$img'

EOF
		print GPSCRIPT $gpcmd, "\n";
		close(GPSCRIPT);
	}
}	# BuildGnuplotScriptFile



sub RunGnuplot
{
	my ($gpscript, $gpresult) = @_;
	
	# This is safe because all of the variables below are
	# internal and cannot be modified by the web client.
	#
	system("'$gnuplot' '$gpscript' > '$gpresult' 2>&1");
}	# RunGnuplot



sub GetNextLogLine
{
	my ($ts, $flds);
	
	if ($nloglines == 0) {
		MultipartStatus(sprintf("Processing %d log file%s...", scalar(@logfiles), (scalar(@logfiles) == 1) ? "" : "s"));
	}
	
LOGLOOP: while (1) {
		if ($log_is_open) {
			$logline = <LOG>;	
			if (! defined($logline)) {
				$log_is_open = 0;
				close(LOG);
				next LOGLOOP;
			}
			chomp($logline);

			next LOGLOOP unless ($logline =~ /^(\d{4}\-\d\d\-\d\d\s\d\d:\d\d:\d\d)\s\#\S+\s+\|\s*(.+)/);
			$ts = $1;
			$flds = $2;
			next LOGLOOP unless (defined($ts) && defined($flds));
			next LOGLOOP unless (($stn0 le $ts) && ($ts le $stn1));

			$nloglines++;
			$nlogbytes += length($logline) + 1;
			@logfields = split(/,/, $flds);
			unshift(@logfields, $ts);
			
			if ($ticker) {
				if ($nlogbytes > 1048576) {
					MultipartStatus(sprintf("Processing logs ($nloglines lines, %.1f MB)...", $nlogbytes / 1048576));
				} else {
					MultipartStatus(sprintf("Processing logs ($nloglines lines, %d kB)...", int(($nlogbytes + 512) / 1024)));
				}
			}
			return (1);
		} else {
			$logpath = $logfiles[$logidx++];
			if (! defined($logpath)) {
				# No more log data
				return (0);
			}
			if (! open(LOG, "< $logpath")) {
				ErrorPage("Could not open $logpath: $?.\n");
			} else {
				# print "<p>Opened <tt>$logpath</tt>.</p>\n";
				$logsused++;
				$log_is_open = 1;
			}
		}
	}
}	# GetNextLogLine




sub ValidateDateRange
{
	my ($start_B, $start_m, $start_d, $start_Y, $start_H, $start_M);
	my ($end_B, $end_m, $end_d, $end_Y, $end_H, $end_M);
	my ($nErrs) = 0;
	my (%months) = ();
	my ($m, $i);

	for ($i = 1; $i <= 12; $i++) {
		$m = POSIX::strftime("%B", 0, 0, 12, 15, $i - 1, 99);
		$months{$m} = $i;
	}

	$start_B = param("start_B");
	$start_d = param("start_d");
	$start_Y = param("start_Y");
	$start_H = param("start_H");
	$start_M = param("start_M");
	$end_B = param("end_B");
	$end_d = param("end_d");
	$end_Y = param("end_Y");
	$end_H = param("end_H");
	$end_M = param("end_M");

	if (
		defined($start_B) &&
		defined($start_d) &&
		defined($start_Y) &&
		defined($start_H) &&
		defined($start_M) &&
		defined($end_B) &&
		defined($end_d) &&
		defined($end_Y) &&
		defined($end_H) &&
		defined($end_M)
	) {
		$nErrs++ if ($start_Y < 1997);
		$start_m = $months{$start_B};
		$nErrs++ if (! defined($start_m));
		$nErrs++ if (($start_d < 1) || ($start_d > 31));
		$nErrs++ if (($start_H < 0) || ($start_H > 23));
		$nErrs++ if (($start_M < 0) || ($start_M > 59));

		$nErrs++ if ($end_Y < 1997);
		$end_m = $months{$end_B};
		$nErrs++ if (! defined($end_m));
		$nErrs++ if (($end_d < 1) || ($end_d > 31));
		$nErrs++ if (($end_H < 0) || ($end_H > 23));
		$nErrs++ if (($end_M < 0) || ($end_M > 59));

		$t0 = timelocal(0, $start_M, $start_H, $start_d, $start_m - 1, $start_Y - 1900);
		$nErrs++ if ((! defined($t0)) || ($t0 <= 0));

		$t1 = timelocal(0, $end_M, $end_H, $end_d, $end_m - 1, $end_Y - 1900);
		$nErrs++ if ((! defined($t1)) || ($t1 <= 0));
		$nErrs++ if ($t0 >= $t1);
		$d0 = sprintf("%04d-%02d-%02d %02d:%02d:%02d", $start_Y, $start_m, $start_d, $start_H, $start_M, 0);
		$d1 = sprintf("%04d-%02d-%02d %02d:%02d:%02d", $end_Y, $end_m, $end_d, $end_H, $end_M, 0);
		$stn0 = POSIX::strftime("%Y-%m-%d %H:%M:%S", localtime($t0));
		$stn1 = POSIX::strftime("%Y-%m-%d %H:%M:%S", localtime($t1));
		if (($start_H == 0) && ($start_M == 0)) {
			$df0 = POSIX::strftime("%A, %B %d, %Y", localtime($t0));
		} else {
			$df0 = POSIX::strftime("%A, %B %d, %Y, %I:%M %p %Z", localtime($t0));
		}
		if (($end_H == 0) && ($end_M == 0)) {
			$df1 = POSIX::strftime("%A, %B %d, %Y", localtime($t1));
		} else {
			$df1 = POSIX::strftime("%A, %B %d, %Y, %I:%M %p %Z", localtime($t1));
		}
		return (1) if ($nErrs == 0);
	}
	ErrorPage("Invalid date range.");
	return (0);
}	# ValidateDateRange




sub EndStatWindow
{
	my ($swt0, $swt1);
	my ($dsec, $usec, $dt, $ut, $dkb, $ukb, $ndl, $nul);
	my ($m33, $m56, $m256, $e100M, $e1G);
	
	return unless ($window_lines_used > 0);
	$swt0 = POSIX::strftime("%Y-%m-%d %H:%M:%S", localtime($wt0));
	$swt1 = POSIX::strftime("%Y-%m-%d %H:%M:%S", localtime($wt1));
	$window_stats[1 - 1] = $swt0;
	$window_stats[2 - 1] = $wt0;
	$window_stats[3 - 1] = $swt1;
	$window_stats[4 - 1] = $wt1;
	
	$m33 = $window_stats[23 - 1];
	$m56 = $window_stats[24 - 1];
	$m256 = $window_stats[25 - 1];
	$e100M = $window_stats[27 - 1];
	$e1G = $window_stats[28 - 1];
	
	$dkb = $window_stats[41 - 1];
	$dsec = $window_stats[45 - 1];
	$ndl = $window_stats[32 - 1];
	$ukb = $window_stats[55 - 1];
	$usec = $window_stats[59 - 1];
	$nul = $window_stats[46 - 1];
	
	$window_stats[$last_field + 1 - 1] = $dkb / 1024.0;	# downloads_MB
	
	if ($dsec < 1) { $dt = 0; } else { $dt = $dkb / $dsec; }
	$window_stats[$last_field + 2 - 1] = $dt;		# downloads_avg_thruput

	if ($ndl == 0) {
		$window_stats[$last_field + 3 - 1] = 0;
		$window_stats[$last_field + 4 - 1] = 0;
		$window_stats[$last_field + 5 - 1] = 0;
		$window_stats[$last_field + 6 - 1] = 0;
		$window_stats[$last_field + 7 - 1] = 0;
	} else {
		$window_stats[$last_field + 3 - 1] = 100.0 * $window_stats[34 - 1] / $ndl;
		$window_stats[$last_field + 4 - 1] = 100.0 * $window_stats[35 - 1] / $ndl;
		$window_stats[$last_field + 5 - 1] = 100.0 * $window_stats[36 - 1] / $ndl;
		$window_stats[$last_field + 6 - 1] = 100.0 * $window_stats[37 - 1] / $ndl;
		$window_stats[$last_field + 7 - 1] = 100.0 * $window_stats[38 - 1] / $ndl;
	}
	
	$window_stats[$last_field + 8 - 1] = $ukb / 1024.0;	# downloads_MB
	
	if ($usec < 1) { $ut = 0; } else { $ut = $ukb / $usec; }
	$window_stats[$last_field + 9 - 1] = $ut;		# downloads_avg_thruput

	if ($nul == 0) {
		$window_stats[$last_field + 10 - 1] = 0;
		$window_stats[$last_field + 11 - 1] = 0;
		$window_stats[$last_field + 12 - 1] = 0;
		$window_stats[$last_field + 13 - 1] = 0;
		$window_stats[$last_field + 14 - 1] = 0;
	} else {
		$window_stats[$last_field + 10 - 1] = 100.0 * $window_stats[48 - 1] / $nul;
		$window_stats[$last_field + 11 - 1] = 100.0 * $window_stats[49 - 1] / $nul;
		$window_stats[$last_field + 12 - 1] = 100.0 * $window_stats[50 - 1] / $nul;
		$window_stats[$last_field + 13 - 1] = 100.0 * $window_stats[51 - 1] / $nul;
		$window_stats[$last_field + 14 - 1] = 100.0 * $window_stats[52 - 1] / $nul;
	}
	
	$window_stats[$last_field + 15 - 1] = $m33 + $m56 + $m256;
	$window_stats[$last_field + 16 - 1] = $e100M + $e1G;
	$stat_windows[$num_stat_windows] = join(',', @window_stats);
	$num_stat_windows++;
}	# EndStatWindow




sub NewStatWindow
{
	my ($i, $g);
	my ($gapstart, $gapend) = @_;
	
	if ($window_size > 0) {
		@window_stats = ();
		for ($i = 0; $i < 60; $i++) {
			$window_stats[$i] = 0;
		}
		for ($g = $gapstart; $g <= $gapend; $g++) {
			$wt0 = $g * $window_size;
			$wt1 = $wt0 + $window_size - 1;
			EndStatWindow();
		}
	}
	@window_stats = ();
	$window_lines_used = 0;
	@window_stat_field_type = ();
}	# NewStatWindow




sub SumStatLogs
{
	my ($statlogsused)			= 0;
	my ($statlogidx)			= 0;
	my ($statlog_is_open)			= 0;
	my ($statlogpath)			= 0;
	my (@statlogfields)			= ();
	my ($statlogline)			= "";
	my ($ct0, $ct1);
	my ($i);
	my ($total_stat_lines)			= 0;
	my ($nstatlogbytes)			= 0;
	
	MultipartStatus("Processing log files...");
	NewStatWindow(1, 0);
STATLOGLOOP: while (1) {
		if ($statlog_is_open) {
			$statlogline = <LOG>;	
			if (! defined($statlogline)) {
				$statlog_is_open = 0;
				close(LOG);
				next STATLOGLOOP;
			}
			chomp($statlogline);
			next STATLOGLOOP unless ($statlogline =~ /^\d{4}\-\d\d\-\d\d\s\d\d:\d\d:\d\d,(\d{9,10}),\d{4}\-\d\d\-\d\d\s\d\d:\d\d:\d\d,(\d{9,10}),/);
			$ct0 = $1;
			$ct1 = $2;
			next STATLOGLOOP unless (($ct0 >= $t0) && ($ct0 <= $t1) && ($ct1 >= $t0) && ($ct1 <= $t1) && ($ct0 > 880000000) && ($ct1 > 880000000));
			
			$total_stat_lines++;
			$nstatlogbytes += length($statlogline) + 1;
			
			if ($ticker) {
				if ($nstatlogbytes > 1048576) {
					MultipartStatus(sprintf("Processing logs ($total_stat_lines lines, %.1f MB)...", $nstatlogbytes / 1048576));
				} else {
					MultipartStatus(sprintf("Processing logs ($total_stat_lines lines, %d kB)...", int(($nstatlogbytes + 512) / 1024)));
				}
			}
			
			if ($window_size > 0) {
				$win = int($ct0 / $window_size);
				if ($win != $curwin) {
					if ($win < $curwin) {
						ErrorPage("Stat logs must be sorted in chronological order.");
						return (-1);
					}
					EndStatWindow();
					NewStatWindow($curwin + 1, $win - 1);
					$curwin = $win;
					$wt0 = $win * $window_size;
					$wt1 = $wt0 + $window_size - 1;
				}
			}
			
			# Add to the total counts.
			#
			@statlogfields = split(/,/, $statlogline);
			
			if ($total_stat_lines <= 1) {
				@stats = @statlogfields;
				for ($i = 0; $i < 60; $i++) {
					next if defined($statlogfields[$i]);
					$stats[$i] = 0;
				}
			} else {
				for ($i = 5; $i <= 59; $i++) {
					next unless defined($statlogfields[$i - 1]);
					$stats[$i - 1] += $statlogfields[$i - 1];
				}
				
				# Special-case for Field 60
				# which is a maximum, not a count
				#
				my ($simul) = 0;
				if (defined($statlogfields[60 - 1])) {
					$simul = $statlogfields[60 - 1];
				}
				$stats[60 - 1] = $simul if ($simul > $stats[60 - 1]);
			}
			
			# Add to the counts for this window.
			#
			if ($window_lines_used == 0) {
				@window_stats = @statlogfields;
				for ($i = 0; $i < 60; $i++) {
					if (defined($statlogfields[$i])) {
						if (! defined($window_stat_field_type[$i])) {
							$window_stat_field_type[$i] = "%s";
							$window_stat_field_type[$i] = "%d" if ($statlogfields[$i] =~ /^\d+$/);
							$window_stat_field_type[$i] = "%.2f" if ($statlogfields[$i] =~ /^\d+\.\d+$/);
						}
						next;
					}
					$window_stats[$i] = 0;
				}
			} else {
				for ($i = 5; $i <= 59; $i++) {
					if (defined($statlogfields[$i])) {
						if (! defined($window_stat_field_type[$i])) {
							$window_stat_field_type[$i] = "%s";
							$window_stat_field_type[$i] = "%d" if ($statlogfields[$i] =~ /^\d+$/);
							$window_stat_field_type[$i] = "%.2f" if ($statlogfields[$i] =~ /^\d+\.\d+$/);
						}
						$window_stats[$i - 1] += $statlogfields[$i - 1];
					}
				}
				
				# Special-case for Field 60
				# which is a maximum, not a count
				#
				my ($simul) = 0;
				$simul = $statlogfields[60 - 1] if (defined($statlogfields[60 - 1]));
				if ($simul > $window_stats[60 - 1]) {
					$window_stat_field_type[$i] = "%d";
					$window_stats[60 - 1] = $simul;
				}
			}
			
			if ($window_size == 0) {
				# Track range of lines used.
				if ($window_lines_used == 0) {
					$wt0 = $ct0;
					$wt1 = $ct1;
				} else {
					if ($ct0 < $wt0) {
						$wt0 = $ct0;
					}
					if ($ct1 > $wt0) {
						$wt1 = $ct1;
					}
				}
			}
			
			$window_lines_used++;
		} else {
			$statlogpath = $logfiles[$statlogidx++];
			if (! defined($statlogpath)) {
				# No more log data
				EndStatWindow();
				return (0);
			}
			if (! open(LOG, "< $statlogpath")) {
				ErrorPage("Could not open $statlogpath: $?.\n");
			} else {
				$statlogsused++;
				$statlog_is_open = 1;
			}
		}
	}
}	# SumStatLogs




sub GeneralReportGraph
{
	my ($plotdata, $gpscript, $gpresult, $img, $relimg);
	my ($name, $title, $flds, $gp_cmd) = @_;
	
	$gpscript = "$otdir/$name.script";
	$plotdata = "$otdir/$name.data";
	$gpresult = "$otdir/$name.result";
	$img = "$otdir/$name.$gp_fmt";
	$relimg = "$relotdir/$name.$gp_fmt";
	$gp_cmd =~ s/plotdata/$plotdata/g;
	
	BuildPlotdataFile($plotdata, $t0, $flds);
	BuildGnuplotScriptFile($gpscript, $gp_term, $title, $xlabel, $xtics, $img, $gp_cmd);
	RunGnuplot($gpscript, $gpresult);
	print "<p><hr><p>\n";
	print "<p><center><img src=\"$relimg\" alt=\"$title\">";
}	# GeneralReportGraph




sub CheckGraphingSetup
{
	my ($errs) = 0;
	return (1) if ($plotting_graphs eq "no");
	
	if (NotAccessibleThenPrintError($gnuplot, "---x") < 0) { $errs++; }
	
	if ($DocumentRoot !~ /^\//) {
		ErrorPage("The <tt>\$DocumentRoot</tt> variable needs to be set to an absolute path.");
		$errs++;
	}
	$DocumentRoot_subdir_for_reports =~ s/^$DocumentRoot\/*//;
	if ($DocumentRoot_subdir_for_reports !~ /^\//) {
		$DocumentRoot_subdir_for_reports = "/$DocumentRoot_subdir_for_reports";
	}
	$odir = "${DocumentRoot}${DocumentRoot_subdir_for_reports}";
	$relodir = $DocumentRoot_subdir_for_reports;

	if (NotAccessibleThenPrintError($odir, "d--x") < 0) {
		$errs++;
	}
	return (0) if ($errs > 0);
	return (1);
}	# CheckGraphingSetup




sub GeneralReport
{
	my ($dname);
	
	# Setup the temporary directory
	umask(022);
	if ($plotting_graphs ne "no") {
		$dname = POSIX::strftime("rpt.%Y%m%d.%H%M.$$", localtime(time()));
		$otdir = "$odir/$dname";
		$relotdir = "$relodir/$dname";
		if (! Mkdirs($otdir, 00755)) {
			if (NotAccessibleThenPrintError($odir, "drwx") >= 0) {
				ErrorPage("Could not create temporary output directory <tt>$otdir</tt>.");
			}
			return;
		}
	}
	
	return unless ValidateDateRange();
	
	my ($difftimedays) = int(($t1 - $t0) / 86400);
	my ($difftimesecs) = ($t1 - $t0);
	my ($start_YYYY_mm_dd) = POSIX::strftime("%Y-%m-%d", localtime($t0));
	my ($end_YYYY_mm_dd) = POSIX::strftime("%Y-%m-%d", localtime($t1));
	$xlabel = "Hours between $start_YYYY_mm_dd and $end_YYYY_mm_dd";
	my ($name);
	
	if ($difftimedays <= 3) {
		$window_size=3600;
		if ($start_YYYY_mm_dd eq $end_YYYY_mm_dd) {
			$xlabel="Hours of $start_YYYY_mm_dd";
			$xtics=1;
		} else {
			$xtics=4;
		}
	} elsif ($difftimedays <= 7) {
		$window_size=3600;
		$xtics=12;
	} elsif ($difftimedays <= 14) {
		$window_size=3600;
		$xtics=24;
	} elsif ($difftimedays <= 125) {
		$window_size=86400;
		$xlabel = "Days between $start_YYYY_mm_dd and $end_YYYY_mm_dd";
		$xtics=7;
	} elsif ($difftimedays <= 800) {
		$window_size=604800;
		$xlabel = "Weeks between $start_YYYY_mm_dd and $end_YYYY_mm_dd";
		$xtics=4;
	} else {
		$window_size=2592000;
		$xlabel="30-day periods between $start_YYYY_mm_dd and $end_YYYY_mm_dd";
		$xtics=3;
	}
	
	my ($intervalmin) = int($window_size / 60);
	my ($intervalright) = int($difftimesecs / $window_size);
	
	$logtype = "log-stat";
	$domain_filter = "ALL";
	return unless PrepareToIterateLogs();
	SumStatLogs();

	my ($session_n) 				= $stats[5 - 1];
	my ($session_refused) 				= $stats[6 - 1];
	my ($session_denied) 				= $stats[7 - 1];
	my ($session_login) 				= $stats[8 - 1];
	my ($session_failedLogin) 			= $stats[9 - 1];
	my ($session_secs) 				= $stats[10 - 1];
	my ($session_dirListings) 			= $stats[11 - 1];
	my ($session_withDirListing) 			= $stats[12 - 1];
	my ($session_withDownload) 			= $stats[13 - 1];
	my ($session_withUpload) 			= $stats[14 - 1];
	my ($session_client_unknown) 			= $stats[15 - 1];
	my ($session_client_NcFTP) 			= $stats[16 - 1];
	my ($session_client_Netscape) 			= $stats[17 - 1];
	my ($session_client_InternetExplorer) 		= $stats[18 - 1];
	my ($session_client_MacOSX) 			= $stats[19 - 1];
	my ($session_client_reserved1) 			= $stats[20 - 1];
	my ($session_client_reserved2) 			= $stats[21 - 1];
	my ($session_client_reserved3) 			= $stats[22 - 1];
	my ($session_modem_33kbit) 			= $stats[23 - 1];
	my ($session_modem_56kbit) 			= $stats[24 - 1];
	my ($session_modem_256kbit) 			= $stats[25 - 1];
	my ($session_modem_10Mbit) 			= $stats[26 - 1];
	my ($session_modem_100Mbit) 			= $stats[27 - 1];
	my ($session_modem_1Gbit) 			= $stats[28 - 1];
	my ($session_reserved1) 			= $stats[29 - 1];
	my ($session_reserved2) 			= $stats[30 - 1];
	my ($session_reserved3) 			= $stats[31 - 1];
	
	my ($download_n) 				= $stats[32 - 1];
	my ($download_result_ok) 			= $stats[33 - 1];
	my ($download_result_abor) 			= $stats[34 - 1];
	my ($download_result_error) 			= $stats[35 - 1];
	my ($download_result_incomplete) 		= $stats[36 - 1];
	my ($download_result_noent) 			= $stats[37 - 1];
	my ($download_result_perm) 			= $stats[38 - 1];
	my ($download_result_reserved1) 		= $stats[39 - 1];
	my ($download_result_reserved2) 		= $stats[40 - 1];
	my ($download_kb) 				= $stats[41 - 1];
	my ($download_reserved1) 			= $stats[42 - 1];
	my ($download_reserved2) 			= $stats[43 - 1];
	my ($download_reserved3) 			= $stats[44 - 1];
	my ($download_secs) 				= $stats[45 - 1];
	
	my ($upload_n) 					= $stats[46 - 1];
	my ($upload_result_ok) 				= $stats[47 - 1];
	my ($upload_result_abor) 			= $stats[48 - 1];
	my ($upload_result_error) 			= $stats[49 - 1];
	my ($upload_result_incomplete) 			= $stats[50 - 1];
	my ($upload_result_noent) 			= $stats[51 - 1];
	my ($upload_result_perm) 			= $stats[52 - 1];
	my ($upload_result_reserved1) 			= $stats[53 - 1];
	my ($upload_result_reserved2) 			= $stats[54 - 1];
	my ($upload_kb) 				= $stats[55 - 1];
	my ($upload_reserved1) 				= $stats[56 - 1];
	my ($upload_reserved2) 				= $stats[57 - 1];
	my ($upload_reserved3) 				= $stats[58 - 1];
	my ($upload_secs) 				= $stats[59 - 1];
	
	my ($period_max_simul_users) 			= $stats[60 - 1];
	### end parsing parameters ###
	
	my ($download_mb, $download_gb, $download_tb);
	my ($upload_mb, $upload_gb, $upload_tb);
	my ($session_modem_detected);
	my ($avg_session_min);
	my ($avg_session_sec);
	my ($download_kbsec);
	my ($linesUsed);
	my ($perc_netscape);
	my ($perc_ie);
	my ($perc_ncftp);
	my ($perc_macosx);
	my ($perc_reserved1);
	my ($perc_reserved2);
	my ($perc_reserved3);
	my ($perc_unknown);
	my ($upload_kbsec);
	
	StartHTML("NcFTPd Reports: General Report");
	# DebugParams();
	
	if ($session_n != 0) {
		$download_mb = $download_kb / (1024.0);
		$download_gb = $download_kb / (1024.0 * 1024.0);
		$download_tb = $download_kb / (1024.0 * 1024.0 * 1024.0);
		
		$upload_mb = $upload_kb / (1024.0);
		$upload_gb = $upload_kb / (1024.0 * 1024.0);
		$upload_tb = $upload_kb / (1024.0 * 1024.0 * 1024.0);
		
		$session_modem_detected =
			$session_modem_33kbit +
			$session_modem_56kbit +
			$session_modem_256kbit +
			$session_modem_10Mbit +
			$session_modem_100Mbit +
			$session_modem_1Gbit;
		
		if ($download_secs < 1) {
			$download_kbsec = 0;
		} else {
			$download_kbsec = $download_kb / $download_secs;
		}
		if ($upload_secs < 1) {
			$upload_kbsec = 0;
		} else {
			$upload_kbsec = $upload_kb / $upload_secs;
		}
		if ($session_login < 1) {
			$avg_session_min = 0;
			$avg_session_sec = 0;
			$perc_netscape = 0;
			$perc_ie = 0;
			$perc_ncftp = 0;
			$perc_macosx = 0;
			$perc_reserved1 = 0;
			$perc_reserved2 = 0;
			$perc_reserved3 = 0;
			$perc_unknown = 100.0;
		} else {
			$avg_session_sec = $session_secs / $session_login;
			$avg_session_min = int($avg_session_sec / 60);
			$avg_session_sec = int($avg_session_sec) % 60;
			$perc_netscape = 100.0 * $session_client_Netscape / $session_login;
			$perc_ie = 100.0 * $session_client_InternetExplorer / $session_login;
			$perc_ncftp = 100.0 * $session_client_NcFTP / $session_login;
			$perc_macosx = 100.0 * $session_client_MacOSX / $session_login;
			$perc_reserved1 = 100.0 * $session_client_reserved1 / $session_login;
			$perc_reserved2 = 100.0 * $session_client_reserved2 / $session_login;
			$perc_reserved3 = 100.0 * $session_client_reserved3 / $session_login;
			$perc_unknown = 100.0 * $session_client_unknown / $session_login;
		}
		
		print "<table>\n";
		printf("\t<tr><th align=left width=500>Total number of user sessions</th><td width=150>%d</td></tr>\n", $session_n);
		
		printf("\t<tr><th align=left>&nbsp;&nbsp;&nbsp;&nbsp;Connections refused (server full)</th>");
		printf("<td>%d</td><td><small>(%.2f%%)</small></td></tr>\n", $session_refused, $session_refused * 100.0 / $session_n);
		
		printf("\t<tr><th align=left>&nbsp;&nbsp;&nbsp;&nbsp;Connections denied (IP blocked)</th>");
		printf("<td>%d</td><td><small>(%.2f%%)</small></td></tr>\n", $session_denied, $session_denied * 100.0 / $session_n);
		
		printf("\t<tr><th align=left>&nbsp;&nbsp;&nbsp;&nbsp;Failed logins</th>");
		printf("<td>%d</td><td><small>(%.2f%%)</small></td></tr>\n", $session_failedLogin, $session_failedLogin * 100.0 / $session_n);
		
		printf("\t<tr><th align=left>&nbsp;&nbsp;&nbsp;&nbsp;Successful logins</th>");
		printf("<td>%d</td><td><small>(%.2f%%)</small></td></tr>\n", $session_login, $session_login * 100.0 / $session_n);
		
		if ($session_login > 0) {
			printf("\t<tr><th align=left>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Average login duration</th>");
			printf("<td>%d:%02d</td><td><small><i>minutes:seconds</i></small></tr>\n", $avg_session_min, $avg_session_sec);
			
			printf("\t<tr><th align=left>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Logins with at least one complete download</th>");
			printf("<td>%d</td><td><small>(%.2f%%)</small></tr>\n", $session_withDownload, 100.0 * $session_withDownload / $session_login);
			
			printf("\t<tr><th align=left>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Logins with at least one complete upload</th>");
			printf("<td>%d</td><td><small>(%.2f%%)</small></tr>\n", $session_withUpload, 100.0 * $session_withUpload / $session_login);
			
			printf("\t<tr><th align=left>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Logins with at least one successful directory listing</th>");
			printf("<td>%d</td><td><small>(%.2f%%)</small></tr>\n", $session_withDirListing, 100.0 * $session_withDirListing / $session_login);
			
			printf("\t<tr><th align=left>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Average number of directory listings per login</th>");
			printf("<td>%.1f</td><td></tr>\n", $session_dirListings / $session_login);
		}
		
		printf("\t<tr><th align=left>&nbsp;&nbsp;&nbsp;&nbsp;Maximum simultaneous users</th>");
		printf("<td>%d</td><td></td></tr>\n", $period_max_simul_users);
		
		print "</table>\n\n\n<table>\n";
		printf("\t<tr><th align=left width=500>Number of downloads</th><td width=150>%d</td></tr>\n", $download_n);
		
		if ($download_n > 0) {
			printf("\t<tr><th align=left>&nbsp;&nbsp;&nbsp;&nbsp;Explicitly aborted by client</th>");
			printf("<td>%d</td><td><small>(%.2f%%)</small></td></tr>\n", $download_result_abor, $download_result_abor * 100.0 / $download_n);
			
			printf("\t<tr><th align=left>&nbsp;&nbsp;&nbsp;&nbsp;Connection to client lost</th>");
			printf("<td>%d</td><td><small>(%.2f%%)</small></td></tr>\n", $download_result_incomplete, $download_result_incomplete * 100.0 / $download_n);
			
			printf("\t<tr><th align=left>&nbsp;&nbsp;&nbsp;&nbsp;Non-existent file requested by client</th>");
			printf("<td>%d</td><td><small>(%.2f%%)</small></td></tr>\n", $download_result_noent, $download_result_noent * 100.0 / $download_n);
			
			printf("\t<tr><th align=left>&nbsp;&nbsp;&nbsp;&nbsp;Access denied to file requested by client</th>");
			printf("<td>%d</td><td><small>(%.2f%%)</small></td></tr>\n", $download_result_perm, $download_result_perm * 100.0 / $download_n);
			
			printf("\t<tr><th align=left>&nbsp;&nbsp;&nbsp;&nbsp;Miscellaneous I/O error</th>");
			printf("<td>%d</td><td><small>(%.2f%%)</small></td></tr>\n", $download_result_error, $download_result_error * 100.0 / $download_n);
			
			printf("\t<tr><th align=left>&nbsp;&nbsp;&nbsp;&nbsp;Complete</th>");
			printf("<td>%d</td><td><small>(%.2f%%)</small></td></tr>\n", $download_result_ok, $download_result_ok * 100.0 / $download_n);
			
			printf("\t<tr><th align=left>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Kilobytes transferred</th>");
			printf("<td>%d</td><td><small>kB</small></td></tr>\n", $download_kb);
			
			if ($download_tb > 1.0) {
				printf("\t<tr><th align=left>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Terabytes transferred</th>");
				printf("<td>%.2f</td><td><small>TB</small></td></tr>\n", $download_tb);
			} elsif ($download_gb > 1.0) {
				printf("\t<tr><th align=left>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Gigabytes transferred</th>");
				printf("<td>%.2f</td><td><small>GB</small></td></tr>\n", $download_gb);
			} elsif ($download_mb > 1.0) {
				printf("\t<tr><th align=left>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Megabytes transferred</th>");
				printf("<td>%.2f</td><td><small>MB</small></td></tr>\n", $download_mb);
			}
			
			printf("\t<tr><th align=left>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Average throughput</th>");
			printf("<td>%.2f</td><td><small>kB/sec</small></td></tr>\n", $download_kbsec);
		}
		print "</table>\n";
		
		if ($session_modem_detected > 0) {
			print "<table>\n";
			
			printf("\t<tr><th align=left width=500>Percentage of users transferring at 33.6k modem speed</th>");
			printf("<td width=150>%.2f%%</td></tr>\n", 100 * $session_modem_33kbit / $session_modem_detected);
			
			printf("\t<tr><th align=left width=500>&nbsp;&nbsp;&nbsp;&nbsp;at 56 kbit/sec</th>");
			printf("<td width=150>%.2f%%</td></tr>\n", 100 * $session_modem_56kbit / $session_modem_detected);
			
			printf("\t<tr><th align=left width=500>&nbsp;&nbsp;&nbsp;&nbsp;at 256 kbit/sec</th>");
			printf("<td width=150>%.2f%%</td></tr>\n", 100 * $session_modem_256kbit / $session_modem_detected);
			
			printf("\t<tr><th align=left width=500>&nbsp;&nbsp;&nbsp;&nbsp;at 10 Mbit/sec</th>");
			printf("<td width=150>%.2f%%</td></tr>\n", 100 * $session_modem_10Mbit / $session_modem_detected);
			
			printf("\t<tr><th align=left width=500>&nbsp;&nbsp;&nbsp;&nbsp;at 100 Mbit/sec</th>");
			printf("<td width=150>%.2f%%</td></tr>\n", 100 * $session_modem_100Mbit / $session_modem_detected);
			
			printf("\t<tr><th align=left width=500>&nbsp;&nbsp;&nbsp;&nbsp;at more than 100 Mbit/sec</th>");
			printf("<td width=150>%.2f%%</td></tr>\n", 100 * $session_modem_1Gbit / $session_modem_detected);
			
			print "</table>\n";
		}
		
		if ($perc_unknown < 99.9) {
			print "<table>\n";
			if ($perc_netscape > 0.0) {
				printf("\t<tr><th align=left width=500>Users using Netscape browsers (<a href=\"http://www.mozilla.org/products/mozilla1.x/\">Mozilla</a>, <a href=\"http://www.mozilla.org/products/firefox/\">Firefox</a>, <a href=\"http://channels.netscape.com/ns/browsers/default.jsp\">Navigator</a>)</th><td width=150>%.2f%%</td></tr>\n", $perc_netscape);
			}
			if ($perc_ie > 0.0) {
				printf("\t<tr><th align=left width=500>Users using <a href=\"http://www.microsoft.com/windows/ie/default.asp\">Internet Explorer</a></th><td width=150>%.2f%%</td></tr>\n", $perc_ie);
			}
			if ($perc_ncftp > 0.0) {
				printf("\t<tr><th align=left width=500>Users using <a href=\"http://www.NcFTP.com/ncftp/\">NcFTP Client</a></th><td width=150>%.2f%%</td></tr>\n", $perc_ncftp);
			}
			if ($perc_macosx > 0.0) {
				printf("\t<tr><th align=left width=500>Users using Mac OS X (<a href=\"http://www.apple.com/safari/\">Safari</a>)</th><td width=150>%.2f%%</td></tr>\n", $perc_macosx);
			}
			if ($perc_reserved1 > 0.0) {
				printf("\t<tr><th align=left width=500>Users using Other Type 1</th><td width=150>%.2f%%</td></tr>\n", $perc_reserved1);
			}
			if ($perc_reserved2 > 0.0) {
				printf("\t<tr><th align=left width=500>Users using Other Type 2</th><td width=150>%.2f%%</td></tr>\n", $perc_reserved2);
			}
			if ($perc_reserved3 > 0.0) {
				printf("\t<tr><th align=left width=500>Users using Other Type 3</th><td width=150>%.2f%%</td></tr>\n", $perc_reserved3);
			}
			printf("\t<tr><th align=left width=500>Users using other FTP clients</th><td width=150>%.2f%%</td></tr>\n", $perc_unknown);
			print "</table>\n";
		}
		
		$linesUsed++;
	} else {
		printf("<P>Sorry, there was no activity in the logs for the selected time period.\n");
	}

	if ($plotting_graphs ne "no") {
		if ($png_graph_output ne "no") {
			$gp_term = $gnuplot_png_term;
			$gp_fmt = "png";
		} else {
			$gp_term = $gnuplot_gif_term;
			$gp_fmt = "gif";
		}

		GeneralReportGraph("logins", "Number of logins", "8", "plot [0:$intervalright] [0: ] 'plotdata' title '' with $gnuplot_boxes_style");
		GeneralReportGraph("failedlogins", "Failed logins", "6,7,9", "plot [0:$intervalright] [0: ] 'plotdata.1' title 'refused' with $gnuplot_lines_style, 'plotdata.2' title 'denied' with $gnuplot_lines_style, 'plotdata.3' title 'invalid name/password' with $gnuplot_lines_style");
		GeneralReportGraph("useful_sessions", "Useful sessions", "12,13,14", "plot [0:$intervalright] [0: ] 'plotdata.1' title 'with a download' with $gnuplot_lines_style, 'plotdata.2' title 'with an upload' with $gnuplot_lines_style, 'plotdata.3' title 'with a directory listing' with $gnuplot_lines_style");
		GeneralReportGraph("simul", "Simultaneous users", "60", "plot [0:$intervalright] [0: ] 'plotdata' title '' with $gnuplot_boxes_style");
		GeneralReportGraph("modems", "Types of user connections", sprintf("%d,%d,%d", $last_field + 15, 26, $last_field + 16), "plot [0:$intervalright] [0: ] 'plotdata.1' title '33.6/56k/256k modem' with $gnuplot_lines_style, 'plotdata.2' title '10 Mbit' with $gnuplot_lines_style, 'plotdata.3' title '100 Mbit or better' with $gnuplot_lines_style");
		GeneralReportGraph("downloads", "Number of downloads", "32", "plot [0:$intervalright] [0: ] 'plotdata' title '' with $gnuplot_boxes_style");
		GeneralReportGraph("downloads_MB", "Megabytes downloaded", sprintf("%d", $last_field + 1), "plot [0:$intervalright] [0: ] 'plotdata' title '' with $gnuplot_boxes_style");
		GeneralReportGraph("downloads_avg_thruput", "kB/sec download throughput", sprintf("%d", $last_field + 2), "plot [0:$intervalright] [0: ] 'plotdata' title '' with $gnuplot_boxes_style");
		GeneralReportGraph("dlcompletion", "Download error percentage", sprintf("%d,%d,%d,%d,%d", $last_field + 3, $last_field + 4, $last_field + 5, $last_field + 6, $last_field + 7), "plot [0:$intervalright] [0:100] 'plotdata.1' title 'user abort' with $gnuplot_lines_style, 'plotdata.2' title 'OS error' with $gnuplot_lines_style, 'plotdata.3' title 'incomplete' with $gnuplot_lines_style, 'plotdata.4' title 'no such file' with $gnuplot_lines_style, 'plotdata.5' title 'permission denied' with $gnuplot_lines_style");
		GeneralReportGraph("uploads", "Number of uploads", "46", "plot [0:$intervalright] [0: ] 'plotdata' title '' with $gnuplot_boxes_style");
		GeneralReportGraph("uploads_MB", "Megabytes uploaded", sprintf("%d", $last_field + 8), "plot [0:$intervalright] [0: ] 'plotdata' title '' with $gnuplot_boxes_style");
		GeneralReportGraph("uploads_avg_thruput", "kB/sec upload throughput", sprintf("%d", $last_field + 9), "plot [0:$intervalright] [0: ] 'plotdata' title '' with $gnuplot_boxes_style");
		GeneralReportGraph("ulcompletion", "Upload error percentage", sprintf("%d,%d,%d,%d,%d", $last_field + 10, $last_field + 11, $last_field + 12, $last_field + 13, $last_field + 14), "plot [0:$intervalright] [0:100] 'plotdata.1' title 'user abort' with $gnuplot_lines_style, 'plotdata.2' title 'OS error' with $gnuplot_lines_style, 'plotdata.3' title 'incomplete' with $gnuplot_lines_style, 'plotdata.4' title 'no such file' with $gnuplot_lines_style, 'plotdata.5' title 'permission denied' with $gnuplot_lines_style");
	}
}	# GeneralReport




sub ReportStart
{
	$report_title = $_[0];
	StartHTML("NcFTPd Reports: $report_title");
	# DebugParams();
	DebugDateRange();
}	# ReportHeader




sub ReportHeader
{
	my ($li);

	print "\n<h3>$report_title</h3>\n\n",
		"<p><ul>\n",
		"<li><b>From:</b> $df0</li>\n",
		"<li><b>To:</b> $df1</li>\n";
	print "<li><b>Domain:</b> $domain_filter</li>\n" if ($domain_filter ne "ALL");
	for $li (@_) {
		print "<li>$li</li>\n";
	}
	print "</ul>\n\n";
}	# ReportHeader




sub ReportTrailer
{
	TableEnd();
	if (($topN_warning) && ($nrows > $topN)) {
		print "<p><B>Note:</B> Limit of $topN rows has been reached.</p>\n";
	} elsif (($nrows_warning) && ($nrows == 0)) {
		print "<p><B>Note:</B> No activity in the logs for this time period.</p>\n";
	}
}	# ReportTrailer




sub TableStart
{
	print "\n<p>\n<table",
		defined($t_cellspacing) ? " cellspacing=$t_cellspacing" : "",
		defined($t_cellpadding) ? " cellpadding=$t_cellpadding" : "",
		defined($t_border) ? " border=$t_border" : "",
		defined($t_bgcolor) ? " bgcolor=\"$t_bgcolor\"" : "",
		">\n";
	++$table_started;
}	# TableStart




sub TableEnd
{
	if (($nrows > 0) && ($table_started--)) {
		print "</table>\n\n";
	}
}	# TableEnd




sub TableHeaderRow
{
	my ($f, $f0, $c);
	my ($width) = "";

	print "\t<tr bgcolor=\"$th_bgcolor\">\n";
	for $f0 (@_) {
		$width = "";
		$f = $f0;
		$c = substr($f, 0, 1);
		if ($c eq "_") {
			$c = substr($f, 1, 1);
			$f = substr($f, 2);	# Unescaped mode
		} else {
			$f = substr($f, 1);
			if ($f =~ /^:(\d+):(.*)/) {
				$width = " width=" . $1;
				$f = $2;
			}
			$f = escapeHTML($f);
		}
		if ($c eq "-") {
			print "\t\t<th${width} align=right>$f</th>\n";
		} elsif ($c eq "+") {
			print "\t\t<th${width} align=left>$f</th>\n";
		} elsif ($c eq "=") {
			print "\t\t<th${width} align=center>$f</th>\n";
		} else {
			print "\t\t<th>*** BUG ***</th>\n";
		}
	}
	print "\t\t</tr>\n",
		"\t<!------------------------------------------------------------------------>\n";
}	# TableHeaderRow




sub TableRow
{
	my ($f, $f0, $c);

	print "\t<tr bgcolor=\"$td_bgcolor\" valign=baseline>\n";
	for $f0 (@_) {
		$f = $f0;
		$c = substr($f, 0, 1);
		if ($c eq "_") {
			$c = substr($f, 1, 1);
			$f = substr($f, 2);	# Unescaped mode
		} else {
			$f = escapeHTML(substr($f, 1));
		}
		$f = "&nbsp;" if ($f eq "");
		if ($c eq "-") {
			print "\t\t<td align=right>$f</td>\n";
		} elsif ($c eq "+") {
			print "\t\t<td align=left>$f</td>\n";
		} elsif ($c eq "=") {
			print "\t\t<td align=center>$f</td>\n";
		} elsif ($c eq "/") {
			print "\t\t<td align=left><tt>$f</tt></td>\n";
		} elsif ($c eq "#") {
			print "\t\t<td align=right bgcolor=\"$tc_bgcolor\">$f</td>\n";
		} elsif ($c eq ".") {
			print "\t\t<td>$f</td>\n";
		} else {
			print "\t\t<td>*** BUG ***</td>\n";
		}
	}
	print "\t\t</tr>\n";
}	# TableHeaderRow




sub SetTopN
{
	$nrows = 0;
	$topN = param("topN") || $_[0];
	$topN = $_[0] if ($topN < 0);
	$topN = $_[1] if (($topN > $_[1]) || ($topN < 0));
	$topN_warning = $_[2];
}	# SetTopN




sub SetXtype
{
	$logtype = "log-xfer";
	$Xtype = $_[0];
	if ($Xtype eq "Download") {
		$xtype = "R";	# RETR (Retrieve)
	} else {
		$xtype = "S";	# STOR (Store)
	}
}	# SetXtype




sub SetSortBy
{
	$sort_by = "logins";
	if (defined(param("sort_by"))) {
		$sort_by = lc(param("sort_by"));
	}
}	# SetSortBy




sub SetEventMask
{
	$event_types{"R"} = "Download";
	$event_types{"S"} = "Upload";
	$event_types{"T"} = "Listing";
	$event_types{"D"} = "Delete";
	$event_types{"M"} = "Mkdir";
	$event_types{"C"} = "Chmod";
	$event_types{"L"} = "Symlink";
	$event_types{"N"} = "Rename";

	$event_mask{"R"} = 0;
	$event_mask{"S"} = 0;
	$event_mask{"T"} = 0;
	$event_mask{"D"} = 0;
	$event_mask{"M"} = 0;
	$event_mask{"C"} = 0;
	$event_mask{"L"} = 0;
	$event_mask{"N"} = 0;

	$event_mask{"R"} = 1 if ((defined(param("x_R"))) && (lc(param("x_R")) eq "on"));
	$event_mask{"S"} = 1 if ((defined(param("x_S"))) && (lc(param("x_S")) eq "on"));
	$event_mask{"T"} = 1 if ((defined(param("x_T"))) && (lc(param("x_T")) eq "on"));
	$event_mask{"D"} = 1 if ((defined(param("x_D"))) && (lc(param("x_D")) eq "on"));
	$event_mask{"M"} = 1 if ((defined(param("x_M"))) && (lc(param("x_M")) eq "on"));
	$event_mask{"C"} = 1 if ((defined(param("x_C"))) && (lc(param("x_C")) eq "on"));
	$event_mask{"L"} = 1 if ((defined(param("x_L"))) && (lc(param("x_L")) eq "on"));
	$event_mask{"N"} = 1 if ((defined(param("x_N"))) && (lc(param("x_N")) eq "on"));
}	# SetEventMask




sub TopDownloadsReport
{
	my (%tfiles) = ();
	my (@tfiles_sorted) = ();
	my ($prevcount) = -1;
	my ($prevrank) = -1;
	my ($rank) = 0;
	my ($count) = 0;
	my ($tfile);

	return unless ValidateDateRange();
	$domain_filter = param("domain") || "ALL";
	SetTopN(20, 1000, 0);
	SetXtype($_[0]);
	return unless PrepareToIterateLogs();

	while (GetNextLogLine()) {
		next unless ($logfields[1] eq $xtype);
		next unless ($logfields[10] eq "OK");
		if (! exists($tfiles{$logfields[2]})) {
			$tfiles{$logfields[2]} = 1;
		} else {
			++$tfiles{$logfields[2]};
		}
	}

	MultipartStatus(sprintf("Sorting %d items...", scalar(%tfiles)));
	@tfiles_sorted = sort { $tfiles{$b} <=> $tfiles{$a} } keys(%tfiles);

	ReportStart("Top $topN ${Xtype}s");
	ReportHeader();
	TableStart();
	TableHeaderRow("-Rank", "+Pathname", "-${Xtype}s");

	foreach $tfile (@tfiles_sorted) {
		$count = $tfiles{$tfiles_sorted[$nrows]};
		last if (++$nrows > $topN);
		$rank = ($prevcount == $count) ? $prevrank : $nrows;
		TableRow(sprintf("##%d.", $rank), "/${tfile}", "-${count}");
		$prevrank = $rank;
		$prevcount = $count;
	}
	TableEnd();
}	# TopDownloadsReport




sub TransferredFiles
{
	my ($R_count) = 0;
	my (%R_status) = ();
	my ($R_MB) = 0.0;
	my ($perc);

	return unless ValidateDateRange();
	$domain_filter = param("domain") || "ALL";
	SetTopN(20, 1000, 1);
	SetXtype($_[0]);
	return unless PrepareToIterateLogs();

	$R_status{"OK"} = 0;
	$R_status{"ABOR"} = 0;
	$R_status{"INCOMPLETE"} = 0;
	$R_status{"PERM"} = 0;
	$R_status{"NOENT"} = 0;
	$R_status{"ERROR"} = 0;

	ReportStart("${Xtype}ed Files");
	ReportHeader();
	while (GetNextLogLine()) {
		next unless ($logfields[1] eq $xtype);
		last if (++$nrows > $topN);
		if ($nrows == 1) {
			TableStart();
			TableHeaderRow("+When", "+Pathname", "-Size", "+By", "+Completion");
		}
		TableRow("+$logfields[0]", "/$logfields[2]", "-$logfields[3]", "+$logfields[6] \@ $logfields[8]", "+$logfields[10]");
		$R_count++;
		$R_status{$logfields[10]}++;
		$R_MB += $logfields[3] / (1024.0 * 1024.0);
	}

	if ($nrows >= 1) {
		TableEnd();
		if ($R_MB < 1.0) {
			$b = "KBytes";
			$R_MB *= 1024.0;
		} elsif ($R_MB >= 1000.0) {
			$b = "GBytes";
			$R_MB /= 1024.0;
		} else {
			$b = "MBytes";
		}
		$perc = sprintf("%.1f%%", 100.0 * $R_status{"OK"} / $R_count);
		$R_MB = sprintf("%.2f", $R_MB);
		TableStart();
		TableHeaderRow("+:100:${Xtype}s", "+$b", "+OK", "+OK %", "+INCOMPLETE", "+ABOR", "+PERM", "+NOENT", "+ERROR");
		TableRow("-$R_count", "-$R_MB", "-$R_status{'OK'}", "-$perc", "-$R_status{'INCOMPLETE'}", "-$R_status{'ABOR'}", "-$R_status{'PERM'}", "-$R_status{'NOENT'}", "-$R_status{'ERROR'}");
		TableEnd();
	}

	ReportTrailer();
}	# TransferredFiles




sub TransferSummary
{
	my ($printEach) = ((defined(param("printEach"))) && (lc(param("printEach")) eq "on")) ? 1 : 0;
	my ($pathname) = param("pathname");
	my ($matchExact) = 0;
#	my ($matchRegex) = 0;	# Unimplemented until we can sanitize user REs
	my (%matched_pathnames) = ();
	my (%matched_pathnames_OK) = ();
	my (%matched_pathnames_ABOR) = ();
	my (%matched_pathnames_INCOMPLETE) = ();
	my (%matched_pathnames_PERM) = ();
	my (%matched_pathnames_NOENT) = ();
	my (%matched_pathnames_ERROR) = ();

	if ((! defined ($pathname)) || ($pathname eq "")) {
		ErrorPage("Invalid pathname.");
		return;
	}
	$matchExact = 1 if (substr($pathname, 0, 1) eq "/");

	return unless ValidateDateRange();
	$domain_filter = param("domain") || "ALL";
	SetXtype($_[0]);
	return unless PrepareToIterateLogs();

	ReportStart("${Xtype} Summary");
	ReportHeader("<b>Query:</b> <tt>" . escapeHTML($pathname) . "</tt>");
	$nrows = 0;
	while (GetNextLogLine()) {
		next unless ($logfields[1] eq $xtype);
		if ($matchExact) {
			next if ($logfields[2] ne $pathname);
		} else {
			next if (index($logfields[2], $pathname) == (-1));
		}

		if (! exists($matched_pathnames{$logfields[2]})) {
			$matched_pathnames{$logfields[2]} = 1;
			$matched_pathnames_OK{$logfields[2]} = 0;
			$matched_pathnames_ABOR{$logfields[2]} = 0;
			$matched_pathnames_INCOMPLETE{$logfields[2]} = 0;
			$matched_pathnames_PERM{$logfields[2]} = 0;
			$matched_pathnames_NOENT{$logfields[2]} = 0;
			$matched_pathnames_ERROR{$logfields[2]} = 0;
		} else {
			$matched_pathnames{$logfields[2]}++;
		}

		if ($logfields[10] eq "OK") {
			$matched_pathnames_OK{$logfields[2]}++;
		} elsif ($logfields[10] eq "ABOR") {
			$matched_pathnames_ABOR{$logfields[2]}++;
		} elsif ($logfields[10] eq "INCOMPLETE") {
			$matched_pathnames_INCOMPLETE{$logfields[2]}++;
		} elsif ($logfields[10] eq "PERM") {
			$matched_pathnames_PERM{$logfields[2]}++;
		} elsif ($logfields[10] eq "NOENT") {
			$matched_pathnames_NOENT{$logfields[2]}++;
		} elsif ($logfields[10] eq "ERROR") {
			$matched_pathnames_ERROR{$logfields[2]}++;
		}

		++$nrows;
		next unless ($printEach);
		if ($nrows == 1) {
			TableStart();
			TableHeaderRow("+When", "+Pathname", "-Size", "+By", "+Completion");
		}
		TableRow("+$logfields[0]", "/$logfields[2]", "-$logfields[3]", "+$logfields[6] \@ $logfields[8]", "+$logfields[10]");
	}

	if ($nrows > 0) {
		TableEnd();

		if ($printEach) {
			# Explain this second table
			print "<BR><P><h3>Itemized breakdown of matched files</h3>\n";
		}
		TableStart();
		TableHeaderRow("+Pathname", "+OK", "+INCOMPLETE", "+ABOR", "+PERM", "+NOENT", "+ERROR");
		my (@matched_pathnames_sorted) = sort { $matched_pathnames{$b} <=> $matched_pathnames{$a} } keys(%matched_pathnames);
		for $pathname (@matched_pathnames_sorted) {
			TableRow("/$pathname",
				exists($matched_pathnames_OK{$pathname}) ? "-$matched_pathnames_OK{$pathname}" : "_.&nbsp;",
				exists($matched_pathnames_INCOMPLETE{$pathname}) ? "-$matched_pathnames_INCOMPLETE{$pathname}" : "_.&nbsp;",
				exists($matched_pathnames_ABOR{$pathname}) ? "-$matched_pathnames_ABOR{$pathname}" : "_.&nbsp;",
				exists($matched_pathnames_PERM{$pathname}) ? "-$matched_pathnames_PERM{$pathname}" : "_.&nbsp;",
				exists($matched_pathnames_NOENT{$pathname}) ? "-$matched_pathnames_NOENT{$pathname}" : "_.&nbsp;",
				exists($matched_pathnames_ERROR{$pathname}) ? "-$matched_pathnames_ERROR{$pathname}" : "_.&nbsp;"
			);
		}
		TableEnd();
	}

	$nrows_warning = 0;
	ReportTrailer();
	if ($nrows == 0) {
		print "<p>No matches found for <tt>", escapeHTML($pathname), "</tt>.\n";
	}
}	# TransferSummary



sub TrafficSummaryReport
{
	my ($R_count) = 0;
	my (%R_status) = ();
	my ($R_MB) = 0.0;
	my ($S_count) = 0;
	my (%S_status) = ();
	my ($S_MB) = 0.0;
	my ($T_count) = 0;
	my (%T_status) = ();
	my ($D_count) = 0;
	my ($M_count) = 0;
	my ($C_count) = 0;
	my ($L_count) = 0;
	my ($N_count) = 0;
	my ($perc, $b);

	return unless ValidateDateRange();
	$domain_filter = param("domain") || "ALL";
	return unless PrepareToIterateLogs();

	$R_status{"OK"} = 0;
	$R_status{"ABOR"} = 0;
	$R_status{"INCOMPLETE"} = 0;
	$R_status{"PERM"} = 0;
	$R_status{"NOENT"} = 0;
	$R_status{"ERROR"} = 0;

	$S_status{"OK"} = 0;
	$S_status{"ABOR"} = 0;
	$S_status{"INCOMPLETE"} = 0;
	$S_status{"PERM"} = 0;
	$S_status{"NOENT"} = 0;
	$S_status{"ERROR"} = 0;

	$T_status{"OK"} = 0;
	$T_status{"ABOR"} = 0;
	$T_status{"INCOMPLETE"} = 0;
	$T_status{"PERM"} = 0;
	$T_status{"NOENT"} = 0;
	$T_status{"ERROR"} = 0;

	ReportStart("Traffic Summary");
	ReportHeader();
	while (GetNextLogLine()) {
		if ($logfields[1] eq "R") {
			$R_count++;
			$R_status{$logfields[10]}++;
			$R_MB += $logfields[3] / (1024.0 * 1024.0);
		} elsif ($logfields[1] eq "S") {
			$S_count++;
			$S_status{$logfields[10]}++;
			$S_MB += $logfields[3] / (1024.0 * 1024.0);
		} elsif ($logfields[1] eq "T") {
			$T_count++;
			$T_status{$logfields[3]}++;
		} elsif ($logfields[1] eq "D") {
			$D_count++;
		} elsif ($logfields[1] eq "M") {
			$M_count++;
		} elsif ($logfields[1] eq "C") {
			$C_count++;
		} elsif ($logfields[1] eq "L") {
			$L_count++;
		} elsif ($logfields[1] eq "N") {
			$N_count++;
		}
		++$nrows;
	}

	if ($nrows >= 1) {
		# [R]
		if ($R_MB < 1.0) {
			$b = "KBytes";
			$R_MB *= 1024.0;
		} elsif ($R_MB >= 1000.0) {
			$b = "GBytes";
			$R_MB /= 1024.0;
		} else {
			$b = "MBytes";
		}
		if ($R_count > 0) {
			$perc = sprintf("%.1f%%", 100.0 * $R_status{"OK"} / $R_count);
		} else {
			$perc = "";
		}
		$R_MB = sprintf("%.2f", $R_MB);
		TableStart();
		TableHeaderRow("+:100:Downloads", "+$b", "+OK", "+OK %", "+INCOMPLETE", "+ABOR", "+PERM", "+NOENT", "+ERROR");
		TableRow("-$R_count", "-$R_MB", "-$R_status{'OK'}", "-$perc", "-$R_status{'INCOMPLETE'}", "-$R_status{'ABOR'}", "-$R_status{'PERM'}", "-$R_status{'NOENT'}", "-$R_status{'ERROR'}");
		TableEnd();

		# [S]
		print "<BR>";
		if ($S_MB < 1.0) {
			$b = "KBytes";
			$S_MB *= 1024.0;
		} elsif ($S_MB >= 1000.0) {
			$b = "GBytes";
			$S_MB /= 1024.0;
		} else {
			$b = "MBytes";
		}
		if ($S_count > 0) {
			$perc = sprintf("%.1f%%", 100.0 * $S_status{"OK"} / $S_count);
		} else {
			$perc = "";
		}
		$S_MB = sprintf("%.2f", $S_MB);
		TableStart();
		TableHeaderRow("+:100:Uploads", "+$b", "+OK", "+OK %", "+INCOMPLETE", "+ABOR", "+PERM", "+NOENT", "+ERROR");
		TableRow("-$S_count", "-$S_MB", "-$S_status{'OK'}", "-$perc", "-$S_status{'INCOMPLETE'}", "-$S_status{'ABOR'}", "-$S_status{'PERM'}", "-$S_status{'NOENT'}", "-$S_status{'ERROR'}");
		TableEnd();
		
		if ($T_count > 0) {
			# [T]
			print "<BR>";
			$perc = sprintf("%.1f%%", 100.0 * $T_status{"OK"} / $T_count);
			TableStart();
			TableHeaderRow("+:100:Listings", "+OK", "+OK %", "+INCOMPLETE", "+ABOR", "+PERM", "+NOENT", "+ERROR");
			TableRow("-$T_count", "-$T_status{'OK'}", "-$perc", "-$T_status{'INCOMPLETE'}", "-$T_status{'ABOR'}", "-$T_status{'PERM'}", "-$T_status{'NOENT'}", "-$T_status{'ERROR'}");
			TableEnd();
		}

		if ($D_count > 0) {
			# [D]
			print "<BR>";
			TableStart();
			TableHeaderRow("+:100:Deletes");
			TableRow("-$D_count");
			TableEnd();
		}

		if ($M_count > 0) {
			# [M]
			print "<BR>";
			TableStart();
			TableHeaderRow("+:100:Mkdirs");
			TableRow("-$M_count");
			TableEnd();
		}

		if ($C_count > 0) {
			# [C]
			print "<BR>";
			TableStart();
			TableHeaderRow("+:100:Chmods");
			TableRow("-$C_count");
			TableEnd();
		}

		if ($L_count > 0) {
			# [L]
			print "<BR>";
			TableStart();
			TableHeaderRow("+:100:Symlinks");
			TableRow("-$L_count");
			TableEnd();
		}

		if ($N_count > 0) {
			# [N]
			print "<BR>";
			TableStart();
			TableHeaderRow("+:100:Renames");
			TableRow("-$N_count");
			TableEnd();
		}

	}
	ReportTrailer();
}	# TrafficSummaryReport




sub RealUsersReport
{
	my ($user);
	my (%u_logins) = ();
	my (@u_logins_sorted) = ();
	my (%u_ktot) = ();
	my (%u_ndl) = ();
	my (%u_kdl) = ();
	my (%u_bdl) = ();
	my (%u_nul) = ();
	my (%u_kul) = ();
	my (%u_bul) = ();
	my (%u_nls) = ();
	my (%u_last) = ();
	my ($count_hash);
	my ($k);

	return unless ValidateDateRange();
	$domain_filter = param("domain") || "ALL";
	$logtype = "log-session";
	SetSortBy();
	return unless PrepareToIterateLogs();

	while (GetNextLogLine()) {
		next if ($logfields[1] eq "");
		next if ($logfields[1] eq "anonymous");
		next if ($logfields[20] <= 0);
		++$nrows;
		$user = $logfields[1];
		if (! exists($u_logins{$user})) {
			$u_logins{$user} = 0;
			$u_ktot{$user} = 0;
			$u_ndl{$user} = 0;
			$u_kdl{$user} = 0;
			$u_bdl{$user} = 0;
			$u_nul{$user} = 0;
			$u_kul{$user} = 0;
			$u_bul{$user} = 0;
			$u_nls{$user} = 0;
			$u_last{$user} = "";
		}
		$u_logins{$user} += $logfields[20];
		$u_last{$user} = "$logfields[0]|$logfields[3]";
		if ($logfields[6] > 0) {
			$k = (($logfields[6] + 512.0) / 1024.0);
			$u_kdl{$user} += $k;
			$u_ktot{$user} += $k;
			$u_bdl{$user} += ($logfields[6]);
		}
		$u_ndl{$user} += $logfields[9];
		if ($logfields[7] > 0) {
			$k = (($logfields[7] + 512.0) / 1024.0);
			$u_kul{$user} += $k;
			$u_ktot{$user} += $k;
			$u_bul{$user} += ($logfields[7]);
		}
		$u_nul{$user} += $logfields[10];
		$u_nls{$user} += $logfields[27];
	}
	
	ReportStart("Authenticated Users Login Summary");
	ReportHeader();
	if ($nrows > 0) {
		my ($last_time, $last_host, $mdl, $mul);
		if ($sort_by eq "downloads") {
			MultipartStatus(sprintf("Sorting %d items...", scalar(%u_ndl)));
			@u_logins_sorted = sort { $u_ndl{$b} <=> $u_ndl{$a} } keys(%u_ndl);
			$count_hash = \%u_ndl;
		} elsif ($sort_by eq "mbytes downloaded") {
			MultipartStatus(sprintf("Sorting %d items...", scalar(%u_kdl)));
			@u_logins_sorted = sort { $u_kdl{$b} <=> $u_kdl{$a} } keys(%u_kdl);
			$count_hash = \%u_kdl;
		} elsif ($sort_by eq "uploads") {
			MultipartStatus(sprintf("Sorting %d items...", scalar(%u_nul)));
			@u_logins_sorted = sort { $u_nul{$b} <=> $u_nul{$a} } keys(%u_nul);
			$count_hash = \%u_nul;
		} elsif ($sort_by eq "mbytes uploaded") {
			MultipartStatus(sprintf("Sorting %d items...", scalar(%u_kul)));
			@u_logins_sorted = sort { $u_kul{$b} <=> $u_kul{$a} } keys(%u_kul);
			$count_hash = \%u_kul;
		} elsif ($sort_by eq "mbytes total") {
			MultipartStatus(sprintf("Sorting %d items...", scalar(%u_ktot)));
			@u_logins_sorted = sort { $u_ktot{$b} <=> $u_ktot{$a} } keys(%u_ktot);
			$count_hash = \%u_ktot;
		} elsif ($sort_by eq "listings") {
			MultipartStatus(sprintf("Sorting %d items...", scalar(%u_nls)));
			@u_logins_sorted = sort { $u_nls{$b} <=> $u_nls{$a} } keys(%u_nls);
			$count_hash = \%u_nls;
		} else {
			MultipartStatus(sprintf("Sorting %d items...", scalar(%u_logins)));
			@u_logins_sorted = sort { $u_logins{$b} <=> $u_logins{$a} } keys(%u_logins);
			$count_hash = \%u_logins;
		}
		$nrows = scalar(@u_logins_sorted);
		print "<P>Total number of unique authenticated users: <b>", $nrows, "</b>.\n"; 
		$nrows = 0;
		TableStart();
		TableHeaderRow("+User", "-Logins", "+Last", "+From", "-Downloads", "-MBytes", "-Uploads", "-MBytes", "-Listings");
		for $user (@u_logins_sorted) {
			++$nrows;
			($last_time, $last_host) = split(/\|/, $u_last{$user});
			if ($u_kdl{$user} < 2097152) {
				$mdl = sprintf("%.2f", ($u_bdl{$user}) / 1048576.0);
			} else {
				$mdl = sprintf("%.2f", ($u_kdl{$user} > 0.0) ? (($u_kdl{$user}) / 1024.0) : 0.0);
			}
			if ($u_kul{$user} < 2097152) {
				$mul = sprintf("%.2f", ($u_bul{$user}) / 1048576.0);
			} else {
				$mul = sprintf("%.2f", ($u_kul{$user} > 0.0) ? (($u_kul{$user}) / 1024.0) : 0.0);
			}
			# $mdl .= " ($u_bdl{$user})";
			# $mul .= " ($u_bul{$user})";
			TableRow("+$user", "-$u_logins{$user}", "+$last_time", "+$last_host", "-$u_ndl{$user}", "-$mdl", "-$u_nul{$user}", "-$mul", "-$u_nls{$user}");
		}
	}
	ReportTrailer();
}	# RealUsersReport




sub RealUserQuery
{
	my ($user) = param("user");
	my ($completion);
	my ($event);
	my ($size);
	my ($pathname);

	if ((! defined ($user)) || ($user eq "")) {
		ErrorPage("Invalid username to query.");
		return;
	}
	return unless ValidateDateRange();
	$domain_filter = param("domain") || "ALL";
	SetTopN(-1, 10000, 1);
	SetEventMask();
	return unless PrepareToIterateLogs();

	ReportStart("Activity Summary for User $user");
	ReportHeader();
	while (GetNextLogLine()) {
		$event = $logfields[1];
		next unless ((defined($event_mask{$event})) && ($event_mask{$event} == 1));
		next unless ($logfields[6] eq $user);
		last if (++$nrows > $topN);
		if ($nrows == 1) {
			TableStart();
			TableHeaderRow("+Date", "+Event", "+Item", "-Size", "=Completion", "+From");
		}
		$completion = $size = "";
		$pathname = $logfields[2];
		if (($event eq "R") || ($event eq "S")) {
			$size = $logfields[3];
			$completion = $logfields[10];
		} elsif ($event eq "T") {
			$completion = $logfields[3];
		} elsif ($event eq "C") {
			$pathname = "($logfields[3]) $logfields[2]";
		} elsif (($event eq "L") || ($event eq "N")) {
			$pathname .= " -> $logfields[4]";
		}
		TableRow("+$logfields[0]", "+$event_types{$event}", "/$pathname", "-$size", "=$completion", "+$logfields[8]");
	}
	ReportTrailer();
}	# RealUserQuery




sub TopEmailAddressesReport
{
	my ($user);
	my (%u_logins) = ();
	my (@u_logins_sorted) = ();
	my (%u_ktot) = ();
	my (%u_ndl) = ();
	my (%u_kdl) = ();
	my (%u_bdl) = ();
	my (%u_nul) = ();
	my (%u_kul) = ();
	my (%u_bul) = ();
	my (%u_nls) = ();
	my (%u_last) = ();
	my ($prevcount) = -1;
	my ($prevrank) = -1;
	my ($rank) = 0;
	my ($count) = 0;
	my ($count_hash);
	my ($k);

	return unless ValidateDateRange();
	$domain_filter = param("domain") || "ALL";
	$logtype = "log-session";
	SetTopN(20, 1000, 0);
	SetSortBy();
	return unless PrepareToIterateLogs();

	while (GetNextLogLine()) {
		next if ($logfields[1] ne "anonymous");
		next if ($logfields[20] <= 0);
		++$nrows;
		$user = $logfields[2];
		$user =~ s/\@/\ \@\ /g;
		if (! exists($u_logins{$user})) {
			$u_logins{$user} = 0;
			$u_ktot{$user} = 0;
			$u_ndl{$user} = 0;
			$u_kdl{$user} = 0;
			$u_bdl{$user} = 0;
			$u_nul{$user} = 0;
			$u_kul{$user} = 0;
			$u_bul{$user} = 0;
			$u_nls{$user} = 0;
			$u_last{$user} = "";
		}
		$u_logins{$user} += $logfields[20];
		$u_last{$user} = "$logfields[0]|$logfields[3]";
		if ($logfields[6] > 0) {
			$k = (($logfields[6] + 512.0) / 1024.0);
			$u_kdl{$user} += $k;
			$u_ktot{$user} += $k;
			$u_bdl{$user} += ($logfields[6]);
		}
		$u_ndl{$user} += $logfields[9];
		if ($logfields[7] > 0) {
			$k = (($logfields[7] + 512.0) / 1024.0);
			$u_kul{$user} += $k;
			$u_ktot{$user} += $k;
			$u_bul{$user} += ($logfields[7]);
		}
		$u_nul{$user} += $logfields[10];
		$u_nls{$user} += $logfields[27];
	}

	ReportStart("Anonymous Users Login Summary (by e-mail address)");
	ReportHeader("<b>Limit:</b> to top <b>$topN</b> users");
	if ($nrows > 0) {
		my ($last_time, $last_host, $mdl, $mul);
		if ($sort_by eq "downloads") {
			MultipartStatus(sprintf("Sorting %d items...", scalar(%u_ndl)));
			@u_logins_sorted = sort { $u_ndl{$b} <=> $u_ndl{$a} } keys(%u_ndl);
			$count_hash = \%u_ndl;
		} elsif ($sort_by eq "mbytes downloaded") {
			MultipartStatus(sprintf("Sorting %d items...", scalar(%u_kdl)));
			@u_logins_sorted = sort { $u_kdl{$b} <=> $u_kdl{$a} } keys(%u_kdl);
			$count_hash = \%u_kdl;
		} elsif ($sort_by eq "uploads") {
			MultipartStatus(sprintf("Sorting %d items...", scalar(%u_nul)));
			@u_logins_sorted = sort { $u_nul{$b} <=> $u_nul{$a} } keys(%u_nul);
			$count_hash = \%u_nul;
		} elsif ($sort_by eq "mbytes uploaded") {
			MultipartStatus(sprintf("Sorting %d items...", scalar(%u_kul)));
			@u_logins_sorted = sort { $u_kul{$b} <=> $u_kul{$a} } keys(%u_kul);
			$count_hash = \%u_kul;
		} elsif ($sort_by eq "mbytes total") {
			MultipartStatus(sprintf("Sorting %d items...", scalar(%u_ktot)));
			@u_logins_sorted = sort { $u_ktot{$b} <=> $u_ktot{$a} } keys(%u_ktot);
			$count_hash = \%u_ktot;
		} elsif ($sort_by eq "listings") {
			MultipartStatus(sprintf("Sorting %d items...", scalar(%u_nls)));
			@u_logins_sorted = sort { $u_nls{$b} <=> $u_nls{$a} } keys(%u_nls);
			$count_hash = \%u_nls;
		} else {
			MultipartStatus(sprintf("Sorting %d items...", scalar(%u_logins)));
			@u_logins_sorted = sort { $u_logins{$b} <=> $u_logins{$a} } keys(%u_logins);
			$count_hash = \%u_logins;
		}
		$nrows = scalar(@u_logins_sorted);
		print "<P>Total number of unique e-mail addresses: <b>", $nrows, "</b>.\n"; 
		$nrows = 0;
		TableStart();
		TableHeaderRow("-Rank", "+User", "-Logins", "+Last", "+From", "-Downloads", "-MBytes", "-Uploads", "-MBytes", "-Listings");
		for $user (@u_logins_sorted) {
			last if (++$nrows > $topN);
			$count = $$count_hash{$user};
			$rank = ($prevcount == $count) ? $prevrank : $nrows;
			($last_time, $last_host) = split(/\|/, $u_last{$user});
			if ($u_kdl{$user} < 2097152) {
				$mdl = sprintf("%.2f", ($u_bdl{$user}) / 1048576.0);
			} else {
				$mdl = sprintf("%.2f", ($u_kdl{$user} > 0.0) ? (($u_kdl{$user}) / 1024.0) : 0.0);
			}
			if ($u_kul{$user} < 2097152) {
				$mul = sprintf("%.2f", ($u_bul{$user}) / 1048576.0);
			} else {
				$mul = sprintf("%.2f", ($u_kul{$user} > 0.0) ? (($u_kul{$user}) / 1024.0) : 0.0);
			}
			# $mdl .= " ($u_bdl{$user})";
			# $mul .= " ($u_bul{$user})";
			TableRow(sprintf("##%d.", $rank), "+$user", "-$u_logins{$user}", "+$last_time", "+$last_host", "-$u_ndl{$user}", "-$mdl", "-$u_nul{$user}", "-$mul", "-$u_nls{$user}");
			$prevrank = $rank;
			$prevcount = $count;
		}
	}
	ReportTrailer();
}	# TopEmailAddressesReport




sub TopClientIPAddressesReport
{
	my ($user);
	my (%u_logins) = ();
	my (@u_logins_sorted) = ();
	my (%u_ktot) = ();
	my (%u_ndl) = ();
	my (%u_kdl) = ();
	my (%u_bdl) = ();
	my (%u_nul) = ();
	my (%u_kul) = ();
	my (%u_bul) = ();
	my (%u_nls) = ();
	my (%u_last) = ();
	my ($prevcount) = -1;
	my ($prevrank) = -1;
	my ($rank) = 0;
	my ($count) = 0;
	my ($count_hash);
	my ($k);
	my ($incl_anon) = ((defined(param("incl_anon"))) && (lc(param("incl_anon")) eq "on")) ? 1 : 0;
	my ($incl_real) = ((defined(param("incl_real"))) && (lc(param("incl_real")) eq "on")) ? 1 : 0;

	return unless ValidateDateRange();
	$domain_filter = param("domain") || "ALL";
	$logtype = "log-session";
	SetTopN(20, 1000, 0);
	SetSortBy();
	return unless PrepareToIterateLogs();

	while (GetNextLogLine()) {
		next if ($logfields[1] eq "");
		if ((! $incl_anon) && (! $incl_real)) {
			$user = $logfields[3];
		} else {
			next if ((! $incl_anon) && ($logfields[1] eq "anonymous"));
			next if ((! $incl_real) && ($logfields[1] ne "anonymous"));
			$user = "$logfields[1] \@ $logfields[3]";
		}
		next if ($logfields[20] <= 0);
		++$nrows;
		if (! exists($u_logins{$user})) {
			$u_logins{$user} = 0;
			$u_ktot{$user} = 0;
			$u_ndl{$user} = 0;
			$u_kdl{$user} = 0;
			$u_bdl{$user} = 0;
			$u_nul{$user} = 0;
			$u_kul{$user} = 0;
			$u_bul{$user} = 0;
			$u_nls{$user} = 0;
			$u_last{$user} = "";
		}
		$u_logins{$user} += $logfields[20];
		$u_last{$user} = "$logfields[0]|$logfields[3]";
		if ($logfields[6] > 0) {
			$k = (($logfields[6] + 512.0) / 1024.0);
			$u_kdl{$user} += $k;
			$u_ktot{$user} += $k;
			$u_bdl{$user} += ($logfields[6]);
		}
		$u_ndl{$user} += $logfields[9];
		if ($logfields[7] > 0) {
			$k = (($logfields[7] + 512.0) / 1024.0);
			$u_kul{$user} += $k;
			$u_ktot{$user} += $k;
			$u_bul{$user} += ($logfields[7]);
		}
		$u_nul{$user} += $logfields[10];
		$u_nls{$user} += $logfields[27];
	}
	
	ReportStart("User Login Summary By IP Address");
	ReportHeader("<b>Limit:</b> to top <b>$topN</b> users");
	if ($nrows > 0) {
		my ($last_time, $last_host, $mdl, $mul);
		if ($sort_by eq "downloads") {
			MultipartStatus(sprintf("Sorting %d items...", scalar(%u_ndl)));
			@u_logins_sorted = sort { $u_ndl{$b} <=> $u_ndl{$a} } keys(%u_ndl);
			$count_hash = \%u_ndl;
		} elsif ($sort_by eq "mbytes downloaded") {
			MultipartStatus(sprintf("Sorting %d items...", scalar(%u_kdl)));
			@u_logins_sorted = sort { $u_kdl{$b} <=> $u_kdl{$a} } keys(%u_kdl);
			$count_hash = \%u_kdl;
		} elsif ($sort_by eq "uploads") {
			MultipartStatus(sprintf("Sorting %d items...", scalar(%u_nul)));
			@u_logins_sorted = sort { $u_nul{$b} <=> $u_nul{$a} } keys(%u_nul);
			$count_hash = \%u_nul;
		} elsif ($sort_by eq "mbytes uploaded") {
			MultipartStatus(sprintf("Sorting %d items...", scalar(%u_kul)));
			@u_logins_sorted = sort { $u_kul{$b} <=> $u_kul{$a} } keys(%u_kul);
			$count_hash = \%u_kul;
		} elsif ($sort_by eq "mbytes total") {
			MultipartStatus(sprintf("Sorting %d items...", scalar(%u_ktot)));
			@u_logins_sorted = sort { $u_ktot{$b} <=> $u_ktot{$a} } keys(%u_ktot);
			$count_hash = \%u_ktot;
		} elsif ($sort_by eq "listings") {
			MultipartStatus(sprintf("Sorting %d items...", scalar(%u_nls)));
			@u_logins_sorted = sort { $u_nls{$b} <=> $u_nls{$a} } keys(%u_nls);
			$count_hash = \%u_nls;
		} else {
			MultipartStatus(sprintf("Sorting %d items...", scalar(%u_logins)));
			@u_logins_sorted = sort { $u_logins{$b} <=> $u_logins{$a} } keys(%u_logins);
			$count_hash = \%u_logins;
		}
		$nrows = scalar(@u_logins_sorted);
		print "<P>Total number of unique users: <b>", $nrows, "</b>.\n"; 
		$nrows = 0;
		TableStart();
		TableHeaderRow("-Rank", "+User", "-Logins", "+Last", "-Downloads", "-MBytes", "-Uploads", "-MBytes", "-Listings");
		for $user (@u_logins_sorted) {
			last if (++$nrows > $topN);
			$count = $$count_hash{$user};
			$rank = ($prevcount == $count) ? $prevrank : $nrows;
			($last_time, $last_host) = split(/\|/, $u_last{$user});
			if ($u_kdl{$user} < 2097152) {
				$mdl = sprintf("%.2f", ($u_bdl{$user}) / 1048576.0);
			} else {
				$mdl = sprintf("%.2f", ($u_kdl{$user} > 0.0) ? (($u_kdl{$user}) / 1024.0) : 0.0);
			}
			if ($u_kul{$user} < 2097152) {
				$mul = sprintf("%.2f", ($u_bul{$user}) / 1048576.0);
			} else {
				$mul = sprintf("%.2f", ($u_kul{$user} > 0.0) ? (($u_kul{$user}) / 1024.0) : 0.0);
			}
			# $mdl .= " ($u_bdl{$user})";
			# $mul .= " ($u_bul{$user})";
			TableRow(sprintf("##%d.", $rank), "+$user", "-$u_logins{$user}", "+$last_time", "-$u_ndl{$user}", "-$mdl", "-$u_nul{$user}", "-$mul", "-$u_nls{$user}");
			$prevrank = $rank;
			$prevcount = $count;
		}
	}
	ReportTrailer();
}	# TopClientIPAddressesReport




sub ProcessFile
{
	my ($pathname, $ftype, $size, $mtime) = @_;
	
	return unless ($pathname =~ /\/(rpt\.)?\d{8}\.\d{4}\.\d{4}/);
	return unless ($mtime < ($start_time - $purge_minimum_seconds));
	
	if ($ftype eq "d") {
		# printf("  %s %s\n", $ftype, $pathname);
		if (! rmdir($pathname)) {
			warn "Could not remove directory \"$pathname\": $!\n";
		}
	} else {
		# printf("  %s %s\n", $ftype, $pathname);
		if (! unlink($pathname)) {
			warn "Could not remove directory \"$pathname\": $!\n";
		}
	}
}	# ProcessFile




sub ProcessDirectory
{
	my ($directory) = $_[0];
	my ($filename, $pathname, $subdir);
	my ($dev,$ino,$mode,$nlink,$uid,$gid,$rdev,$size,$atime,$mtime,$ctime,$blksize,$blocks);
	my ($d_dev,$d_ino,$d_mode,$d_nlink,$d_uid,$d_gid,$d_rdev,$d_size,$d_atime,$d_mtime,$d_ctime,$d_blksize,$d_blocks);
	my ($sfx);
	my ($tstr);
	my ($nitems);
	my (@subdirs) = ();

	($d_dev,$d_ino,$d_mode,$d_nlink,$d_uid,$d_gid,$d_rdev,$d_size,$d_atime,$d_mtime,$d_ctime,$d_blksize,$d_blocks) = stat($directory);
	if (! opendir(DIR, $directory)) {
		warn "Could not open directory $directory: $!\n";
		return;
	}

	$directory = "" if ($directory eq "/");
	$nitems = 0;
	while (defined($filename = readdir(DIR))) {
		next if ($filename eq ".");
		next if ($filename eq "..");
		$nitems++;
		$pathname = $directory . "/" . $filename;
		($dev,$ino,$mode,$nlink,$uid,$gid,$rdev,$size,$atime,$mtime,$ctime,$blksize,$blocks) = lstat($pathname);
		if (! defined($dev)) {
			warn "$pathname: $!\n";
			next;
		}

		if (-l _) {
			ProcessFile($pathname, "l", $size, $mtime);
		} elsif (-f _) {
			ProcessFile($pathname, "-", $size, $mtime);
		} elsif (-d _) {
			push(@subdirs, $filename);
		}
	}
	closedir(DIR);

	#
	# Now print the entry for the directory itself.
	#
	if (defined($d_mtime)) {
		ProcessFile($directory, "d", $nitems, $d_mtime);
	}

	#
	# Now do the rest of the subdirectories.
	# We have waited until we closed the parent directory
	# to conserve file descriptors.
	#
	if (scalar(@subdirs) > 0) {
		for $subdir (@subdirs) {
			$subdir = "$directory/$subdir";
			ProcessDirectory($subdir);
		}
	}
}	# ProcessDirectory




sub PurgeOldReports
{
	my ($directory);
	
	if ($DocumentRoot !~ /^\//) {
		ErrorPage("The <tt>\$DocumentRoot</tt> variable needs to be set to an absolute path.");
		return;
	}
	$DocumentRoot_subdir_for_reports =~ s/^$DocumentRoot\/*//;
	if ($DocumentRoot_subdir_for_reports !~ /^\//) {
		$DocumentRoot_subdir_for_reports = "/$DocumentRoot_subdir_for_reports";
	}

	$directory = "${DocumentRoot}${DocumentRoot_subdir_for_reports}";
	$start_dir = "${directory}/";
	ProcessDirectory($directory);
}	# PurgeOldReports




sub MainMenu
{
	my ($t0, $t1);
	my ($i, $j, $k);
	my ($start_B, $start_d, $start_Y, $start_H, $start_M);
	my ($start_B_menu, $start_d_menu, $start_Y_menu, $start_H_menu, $start_M_menu);
	my ($end_B, $end_d, $end_Y, $end_H, $end_M);
	my ($end_B_menu, $end_d_menu, $end_Y_menu, $end_H_menu, $end_M_menu);
	my (@months) = ();
	my ($m);
	my ($samplefile) = "/home/ftp/pub/filename";
	my ($sampleuser) = "root";
	my ($date_menu) = "";
	my ($domain_menu) = "";
	my ($domain);
	my ($transaction_type_menu) = "";
	my ($sort_by_menu) = "";

	for ($i = 1; $i <= 12; $i++) {
		push(@months, POSIX::strftime("%B", 0, 0, 12, 15, $i - 1, 99));
	}

	$t1 = time();
	$t0 = $t1 - (14 * 86400);

	($end_B, $end_d, $end_Y, $end_H, $end_M) = split(/\|/, POSIX::strftime("%B|%d|%Y|%H|%M", localtime($t1)));
	($start_B, $start_d, $start_Y, $start_H, $start_M) = split(/\|/, POSIX::strftime("%B|%d|%Y|%H|%M", localtime($t0)));

	#
	# Prepare the start time menus
	#
	$start_B_menu = "<select name=\"start_B\">\n";
	for ($i = 1; $i <= 12; $i++) {
		$m = $months[int($i) - 1];
		if ($start_B eq $m) {
			$start_B_menu .= "\t<option selected>$m</option>\n";
		} else {
			$start_B_menu .= "\t<option>$m</option>\n";
		}
	}
	$start_B_menu .= "</select>\n";

	$start_d_menu = "<select name=\"start_d\">\n";
	for ($i = 1; $i <= 31; $i++) {
		if ($start_d == $i) {
			$start_d_menu .= "\t<option selected>$i</option>\n";
		} else {
			$start_d_menu .= "\t<option>$i</option>\n";
		}
	}
	$start_d_menu .= "</select>\n";

	$start_Y_menu = "<select name=\"start_Y\">\n";
	for ($i = 1997; $i <= int($end_Y); $i++) {
		if (int($start_Y) == $i) {
			$start_Y_menu .= "\t<option selected>$i</option>\n";
		} else {
			$start_Y_menu .= "\t<option>$i</option>\n";
		}
	}
	$start_Y_menu .= "</select>\n";

	$start_H_menu = "<select name=\"start_H\">\n";
	for ($i = 0; $i <= 23; $i++) {
		if ($i == 0) {
			$j = sprintf("%02d", $i);
			$start_H_menu .= "\t<option selected>$j</option>\n";
		} else {
			$j = sprintf("%02d", $i);
			$start_H_menu .= "\t<option>$j</option>\n";
		}
	}
	$start_H_menu .= "</select>\n";

	$start_M_menu = "<select name=\"start_M\">\n";
	for ($i = 0; $i < 60; $i += 15) {
		if ($i == 0) {
			$j = sprintf("%02d", $i);
			$start_M_menu .= "\t<option selected>$j</option>\n";
		} else {
			$j = sprintf("%02d", $i);
			$start_M_menu .= "\t<option>$j</option>\n";
		}
	}
	$start_M_menu .= "</select>\n";

	#
	# Prepare the end time menus
	#
	$end_B_menu = "<select name=\"end_B\">\n";
	for ($i = 1; $i <= 12; $i++) {
		$m = $months[int($i) - 1];
		if ($end_B eq $m) {
			$end_B_menu .= "\t<option selected>$m</option>\n";
		} else {
			$end_B_menu .= "\t<option>$m</option>\n";
		}
	}
	$end_B_menu .= "</select>\n";

	$end_d_menu = "<select name=\"end_d\">\n";
	for ($i = 1; $i <= 31; $i++) {
		if ($end_d == $i) {
			$end_d_menu .= "\t<option selected>$i</option>\n";
		} else {
			$end_d_menu .= "\t<option>$i</option>\n";
		}
	}
	$end_d_menu .= "</select>\n";

	$end_Y_menu = "<select name=\"end_Y\">\n";
	for ($i = 1997; $i <= int($end_Y); $i++) {
		if (int($end_Y) == $i) {
			$end_Y_menu .= "\t<option selected>$i</option>\n";
		} else {
			$end_Y_menu .= "\t<option>$i</option>\n";
		}
	}
	$end_Y_menu .= "</select>\n";

	$end_H_menu = "<select name=\"end_H\">\n";
	for ($i = 0; $i <= 23; $i++) {
		if (int($end_H) == $i) {
			$j = sprintf("%02d", $i);
			$end_H_menu .= "\t<option selected>$j</option>\n";
		} else {
			$j = sprintf("%02d", $i);
			$end_H_menu .= "\t<option>$j</option>\n";
		}
	}
	$end_H_menu .= "</select>\n";

	$end_M_menu = "<select name=\"end_M\">\n";
	$k = int((int($end_M) / 15)) * 15;
	for ($i = 0; $i < 60; $i += 15) {
		if ($i == $k) {
			$j = sprintf("%02d", $i);
			$end_M_menu .= "\t<option selected>$j</option>\n";
		} else {
			$j = sprintf("%02d", $i);
			$end_M_menu .= "\t<option>$j</option>\n";
		}
	}
	$end_M_menu .= "</select>\n";

	$date_menu = "
<ul><table>
<tr>
<td>Starting Date:</td>
<td>
$start_B_menu
$start_d_menu
$start_Y_menu
</td>
<td>Time:</td>
<td>
$start_H_menu
$start_M_menu
</td></tr>
	
<tr>
<td>Ending Date:</td>
<td>
$end_B_menu
$end_d_menu
$end_Y_menu
</td>
<td>Time:</td>
<td>
$end_H_menu
$end_M_menu
</td></tr>
</table></ul>";

	$transaction_type_menu = "
<P>Transaction types to include:
<ul>
<table>
<tr>
<td><input type=checkbox name=\"x_R\" checked /> Download</td>
<td><input type=checkbox name=\"x_S\" checked /> Upload</td>
<td><input type=checkbox name=\"x_T\" checked /> Directory Listing</td>
</tr>
<tr>
<td><input type=checkbox name=\"x_D\" checked /> Delete</td>
<td><input type=checkbox name=\"x_M\" checked /> Mkdir</td>
<td><input type=checkbox name=\"x_C\" checked /> Chmod</td>
</tr>
<tr>
<td><input type=checkbox name=\"x_N\" checked /> Rename</td>
<td><input type=checkbox name=\"x_L\" checked /> Symlink</td>
</tr>
</table>
</ul>";

	$sort_by_menu = "
Sort by:
<select name=\"sort_by\">
<option selected>Logins</option>
<option>Downloads</option>
<option>MBytes Downloaded</option>
<option>Uploads</option>
<option>MBytes Uploaded</option>
<option>MBytes Total</option>
<option>Listings</option>
</select>";

	LoadDomainCf();
	$domain_menu = "<select name=\"domain\">\n";
	$domain_menu .= "\t<option selected>ALL</option>\n";
	for $domain (@domains) {
		$domain_menu .= "\t<option>$domain</option>\n";
	}
	$domain_menu .= "</select>\n";

	$using_multipart = 0;
	StartHTML("NcFTPd Reports");
#	DebugParams();

	#####################################################################
	# Main Menu
	#####################################################################

	print <<EOF;
<h3><i>NcFTPd</i> Reports Selector</h3>
<p>Select a report to run from the menu below, or simply scroll down
this page.
<p>
<ul>
	<li><a href="#general">General Report</a></li>
	<li><a href="#topdl">Top Downloads</a></li>
	<li><a href="#dlallsum">Downloaded Files</a></li>
	<li><a href="#dl1sum">Downloaded File Summary</a></li>
	<li><a href="#ulallsum">Uploaded Files</a></li>
	<li><a href="#ul1sum">Uploaded File Summary</a></li>
	<li><a href="#traffic">Traffic Summary</a></li>
	<li><a href="#realusers">Authenticated Users Login Summary</a></li>
	<li><a href="#realuserquery">Authenticated User Activity Summary</a></li>
	<li><a href="#ipsum">User Login Summary By IP Address</a></li>
	<li><a href="#emsum">Anonymous User Login Summary By E-Mail Address</a></li>
</ul>
<p>
Some reports can be configured to only use data from a specific
virtual domain, however, this requires that the domain have its
own log files (i.e. you used different pathnames for the
<TT>log-session</TT> option).
<p><hr><p>
EOF
	#####################################################################
	# General report form
	#####################################################################

	print <<EOF;
<a name=\"general\"></a>
<P>The <b>General Report</b> provides a wide variety
of statistics and information, based upon the
near real-time <TT>stat</TT> logs that <I>NcFTPd</I>
generates.
<p>Report complexity: <b>low</b>
<ul>
<small>
Relatively quick to generate since it summarizes data that
has been pre-computed.  Drawing the graphs may take a few
moments, however.
</small>
</ul>
<p>To run this report, select the timespan that you
want the report to cover, and then click the <b>submit</b>
button.
<form action=\"${script}?general\" method=\"post\">

$date_menu

<p>
<input type="hidden" name="general" value="" />
<input value="Submit" type="submit" />
</form>
<p><hr><p>
EOF

	#####################################################################
	# Top Download report form
	#####################################################################

	print <<EOF;
<a name=\"topdl\"></a>
<P>The <b>Top Downloads</b> report shows the most popular
files downloaded within a period of time.
<p>Report complexity: <b>high</b>
<ul>
<small>
<P>This report can be resource intensive because the entire
span of log data must be sorted first, and then examined.
Therefore, the complexity increases with larger timespans.
</small>
</ul>
<p>To run this report, select the timespan that you
want the report to cover, and then click the <b>submit</b>
button.
<form action=\"${script}?topdl\" method=\"post\">

$date_menu

<p>Show the top
<input type=text name=\"topN\" value=\"20\" size=4>
downloads for virtual domain: 

$domain_menu

<p>
<input type="hidden" name="topdl" value="" />
<input value="Submit" type="submit" />
</form>
<p><hr><p>
EOF

	#####################################################################
	# Downloaded Files report form
	#####################################################################

	print <<EOF;
<a name=\"dlallsum\"></a>
<P>The <b>Downloaded Files</b> report shows all
files downloaded within a period of time.
<p>Report complexity: <b>medium</b>
<ul>
<small>
<P>This report can easily produce an overwhelming amount
of output, since it essentially re-formats the log data.
It requires a sequential scan of the <tt>xfer</tt> logs.
</small>
</ul>
<p>To run this report, select the timespan that you
want the report to cover, and then click the <b>submit</b>
button.
<form action=\"${script}?dlsum\" method=\"post\">

$date_menu

<p>Show a maximum of
<input type=text name=\"topN\" value=\"1000\" size=5>
downloads for virtual domain: 

$domain_menu

<p>
<input type="hidden" name="dlsum" value="" />
<input value="Submit" type="submit" />
</form>
<p><hr><p>
EOF

	#####################################################################
	# Downloaded File Summary form
	#####################################################################

	print <<EOF;
<a name=\"dl1sum\"></a>
<P>The <b>Downloaded File Summary</b> lets you
find out how many times a specific file was downloaded.
You specify the complete or partial pathname (but no symlinks!) and
whether you just want a count, or a list of all matches.
<p>Report complexity: <b>medium</b>
<ul>
<small>
<P>This report can produce a lot of
of output, depending on the number of hits.
It requires a sequential scan of the <tt>xfer</tt> logs.
</small>
</ul>
<p>To run this report, select the timespan that you
want the report to cover, and then click the <b>submit</b>
button.
<form action=\"${script}?dlquery\" method=\"post\">

$date_menu

<P>For virtual domain: 
$domain_menu

<P>Pathname of file to look for:
<input type=text name=\"pathname\" value=\"$samplefile\" size=60>

<P>
Print each transaction:
<input type=checkbox name=\"printEach\" checked />

<p>
<input type="hidden" name="dlquery" value="" />
<input value="Submit" type="submit" />
</form>
<p><hr><p>
EOF

	#####################################################################
	# Uploaded Files report form
	#####################################################################

	print <<EOF;
<a name=\"ulallsum\"></a>
<P>The <b>Uploaded Files</b> report shows all
files uploaded within a period of time.
<p>Report complexity: <b>medium</b>
<ul>
<small>
<P>This report can easily produce an overwhelming amount
of output, since it essentially re-formats the log data.
It requires a sequential scan of the <tt>xfer</tt> logs.
</small>
</ul>
<p>To run this report, select the timespan that you
want the report to cover, and then click the <b>submit</b>
button.
<form action=\"${script}?ulsum\" method=\"post\">

$date_menu

<p>Show a maximum of
<input type=text name=\"topN\" value=\"1000\" size=5>
uploads for virtual domain: 

$domain_menu

<p>
<input type="hidden" name="ulsum" value="" />
<input value="Submit" type="submit" />
</form>
<p><hr><p>
EOF


	#####################################################################
	# Uploaded File Summary form
	#####################################################################

	print <<EOF;
<a name=\"ul1sum\"></a>
<P>The <b>Uploaded File Summary</b> lets you
find out how many times a specific file was uploaded.
You specify the complete or partial pathname (but no symlinks!) and
whether you just want a count, or a list of all matches.
<p>Report complexity: <b>medium</b>
<ul>
<small>
<P>This report can produce a lot of
of output, depending on the number of hits.
It requires a sequential scan of the <tt>xfer</tt> logs.
</small>
</ul>
<p>To run this report, select the timespan that you
want the report to cover, and then click the <b>submit</b>
button.
<form action=\"${script}?ulquery\" method=\"post\">

$date_menu

<P>For virtual domain: 
$domain_menu

<P>Pathname of file to look for:
<input type=text name=\"pathname\" value=\"$samplefile\" size=60>
<P>
Print each transaction:
<input type=checkbox name=\"printEach\" checked />

<p>
<input type="hidden" name="ulquery" value="" />
<input value="Submit" type="submit" />
</form>
<p><hr><p>
EOF


	#####################################################################
	# Traffic summary report form
	#####################################################################

	print <<EOF;
<a name=\"traffic\"></a>
<P>The <b>Traffic Summary</b> shows the number
of transfers and total bytes transferred.  This can
tell you how much data is going through the server.
<p>Report complexity: <b>medium</b>
<ul>
<small>
This requires a sequential scan of the <tt>xfer</tt> logs.
</small>
</ul>
<p>To run this report, select the timespan that you
want the report to cover, and then click the <b>submit</b>
button.
<form action=\"${script}?traffic\" method=\"post\">

$date_menu

<P>For virtual domain: 
$domain_menu

<p>
<input type="hidden" name="traffic" value="" />
<input value="Submit" type="submit" />
</form>
<p><hr><p>
EOF


	#####################################################################
	# Authenticated Users Login Summary report form
	#####################################################################

	print <<EOF;
<a name=\"realusers\"></a>
<P>The <b>Authenticated Users Login Summary</b> shows a synopsis of
activity by users which have logged in with usernames and passwords
(i.e. users who did not login anonymously).
<p>Report complexity: <b>medium</b>
<ul>
<small>
<P>It requires a sequential scan of the <tt>session</tt> logs.
</small>
</ul>
<p>To run this report, select the timespan that you
want the report to cover, and then click the <b>submit</b>
button.
<form action=\"${script}?realusers\" method=\"post\">

$date_menu

<P>For virtual domain: 
$domain_menu

<p>
$sort_by_menu

<p>
<input type="hidden" name="realusers" value="" />
<input value="Submit" type="submit" />
</form>
<p><hr><p>
EOF


	#####################################################################
	# Authenticated User Activity Summary form
	#####################################################################

	print <<EOF;
<a name=\"realuserquery\"></a>
<P>The <b>Authenticated User Activity Summary</b> itemizes transactions
performed by a particular user.
<p>Report complexity: <b>medium</b>
<ul>
<small>
<P>It requires a sequential scan of the <tt>xfer</tt> logs.
</small>
</ul>
<p>To run this report, select the timespan that you
want the report to cover, and then click the <b>submit</b>
button.
<form action=\"${script}?realuserquery\" method=\"post\">

$date_menu

<P>For virtual domain: 
$domain_menu

<P>User login to look for:
<input type=text name=\"user\" value=\"$sampleuser\" size=8>

$transaction_type_menu
<p>
<input type="hidden" name="realuserquery" value="" />
<input value="Submit" type="submit" />
</form>
<p><hr><p>
EOF


	#####################################################################
	# IP summary report form
	#####################################################################

	print <<EOF;
<a name=\"ipsum\"></a>
<P>The <b>User Login Summary By IP Address</b> report tells you the number
of unique users that used your server, and a list of the most frequent.
Hostnames will be used in place of IP addresses if the log files already have
resolved the IP address.
<p>Report complexity: <b>high</b>
<ul>
<small>
<P>This report requires a sequential scan of the
<tt>session</tt> logs, and needs to be able to sort
a potentially large dataset.
</small>
</ul>
<p>To run this report, select the timespan that you
want the report to cover, and then click the <b>submit</b>
button.
<form action=\"${script}?ipsum\" method=\"post\">

$date_menu

<p>Show the top
<input type=text name=\"topN\" value=\"20\" size=4>
client IP addresses using virtual domain: 

$domain_menu

<p>
<input type=checkbox name=\"incl_anon\" checked /> Include anonymous users
<input type=checkbox name=\"incl_real\" checked /> Include non-anonymous users

<p>
$sort_by_menu

<p>
<input type="hidden" name="ipsum" value="" />
<input value="Submit" type="submit" />
</form>
<p><hr><p>
EOF


	#####################################################################
	# Email summary report form
	#####################################################################

	print <<EOF;
<a name=\"emsum\"></a>
<P>The <b>Anonymous User Login Summary By E-Mail Address</b> tells you the
number of unique anonymous users (differentiated by the e-mail address that is
collected as the anonymous user's password) that used your server, and a list of
the most frequent.
This report is no longer very useful, as most FTP client programs use a
default anonymous password.  Prior to 1996, it was considered common courtesy
to use your e-mail address as the anonymous user's password.
<p>Report complexity: <b>high</b>
<ul>
<small>
<P>This report requires a sequential scan of the
<tt>session</tt> logs, and needs to be able to sort
a potentially large dataset.
</small>
</ul>
<p>To run this report, select the timespan that you
want the report to cover, and then click the <b>submit</b>
button.
<form action=\"${script}?emsum\" method=\"post\">

$date_menu

<p>Show the top
<input type=text name=\"topN\" value=\"20\" size=4>
anonymous users using virtual domain: 

$domain_menu

<p>
$sort_by_menu

<p>
<input type="hidden" name="emsum" value="" />
<input value="Submit" type="submit" />
</form>
<p><hr><p>
EOF

	#####################################################################
	# End of report selection
	#####################################################################
}	# MainMenu




sub Main
{
	my (%opts);
	
	umask(077);
	if (defined(request_method()) && (request_method() ne "")) {
		$using_html = 1;
		CheckForMultipartCapability();
	}
	
	getopts("k:", \%opts);
	if (exists($opts{"k"})) {
		if ($using_html) {
			ErrorPage("You are not allowed to do that from a web browser.");
			EndHTML();
			return;
		}
		$purge_minimum_seconds = $opts{"k"} * 60;
		if ($purge_minimum_seconds < 60) { $purge_minimum_seconds = 60; }
		PurgeOldReports();
		exit(0);
	}
	
	if (defined(param("general"))) {
		if (CheckGraphingSetup()) {
			GeneralReport();
		}
	} elsif (defined(param("topdl"))) {
		TopDownloadsReport("Download");
	} elsif (defined(param("dlsum"))) {
		TransferredFiles("Download");
	} elsif (defined(param("dlquery"))) {
		TransferSummary("Download");
	} elsif (defined(param("ulsum"))) {
		TransferredFiles("Upload");
	} elsif (defined(param("ulquery"))) {
		TransferSummary("Upload");
	} elsif (defined(param("traffic"))) {
		TrafficSummaryReport();
	} elsif (defined(param("realusers"))) {
		RealUsersReport();
	} elsif (defined(param("realuserquery"))) {
		RealUserQuery();
	} elsif (defined(param("emsum"))) {
		TopEmailAddressesReport();
	} elsif (defined(param("ipsum"))) {
		TopClientIPAddressesReport();
	} else {
		# Unknown op.
		if (CheckGraphingSetup()) {
			MainMenu();
		}
	}
	EndHTML();	# Finish our page, if we started writing one
}	# Main

$SIG{ALRM} = sub { $ticker++; alarm($ticker_interval); };
$SIG{CHLD} = 'IGNORE';
Main();
