#!/usr/bin/perl -w
#
# Copyright (C) 2003-2004, NcFTP Software.
# All Rights Reserved.
#

#############################################################################
#
# How to use this CGI script:
#
# This CGI script can be used to allow system administrators to manipulate
# NcFTPd password databases using a web-based interface instead of running
# "ncftpd_passwd" directly.
#
# For highest security, do not install this script at all and continue to
# manually run "ncftpd_passwd".
#
# (1) Create the file /etc/ftp.passwd.conf.  "chown root" and "chmod 600"
#     the file.  Then edit the file and add a line that contains the complete
#     pathname to the password database, followed by a space, followed by
#     the username the web server runs the CGI as, followed by a comma,
#     followed by the tokens "allow-setuid-root,allow-mkdir".
#     A sample line could look like this assuming the web server user is "web":
#        /usr/local/etc/ncftpd/passwd.db web,allow-setuid-root,allow-mkdir
#
# (2) Create a setuid-root COPY of ncftpd_passwd which is only accessible
#     by your web server user. Example:
#        cp /usr/local/sbin/ncftpd_passwd /usr/local/sbin/web_ncftpd_passwd
#        chown root /usr/local/sbin/web_ncftpd_passwd
#        chgrp web /usr/local/sbin/web_ncftpd_passwd
#        chmod 4710 /usr/local/sbin/web_ncftpd_passwd
#     Then scroll down below and set the variable $ncftpd_passwd to the
#     pathname of the copy of ncftpd_passwd that you just made.
#
# (3) This script needs a private directory in order to write session
#     token files (one for each logged in admin user).  This can be any
#     directory you want, but it must be accessible only by the web server
#     user.  Example, using /var/spool/ncftpd/useradmin:
#        mkdir /var/spool/ncftpd
#        chmod 755 /var/spool/ncftpd
#        mkdir /var/spool/ncftpd/useradmin
#        chown web /var/spool/ncftpd/useradmin
#        chgrp web /var/spool/ncftpd/useradmin
#        chmod 770 /var/spool/ncftpd/useradmin
#     Then scroll down below and set the variable $statedir to the
#     pathname of the directory that you just made.
#
# (4) This script will prompt the user for a password.  Scroll down below
#     and set the $admin_password variable to the password you want to use.
#     Do NOT set this to your machine's root password!!!  This is a password
#     only to be used by this script, so don't use the same password of any
#     other important password.  The purpose of the password is to allow your
#     sysadmins in, and keep out end-users and hackers.
#
# (5) Scroll below and set the $log_file_name variable to the pathname to
#     log messages to.  This is optional, so if you don't want a log, set
#     to empty string ("").  If you use a log, make sure the web user
#     can access the directory containing the log file.  For example, if
#     you have the log file reside in /var/log/ncftpd, you may need to:
#       chmod 775 /var/log/ncftpd
#       chgrp web /var/log/ncftpd
#     so that the web user can create the log file there.
#
# (6) Scroll below and set the $database_file_name variable to the pathname to
#     the NcFTPd password database you wish to edit via web browser.
#
# (7) Move this script to a cgi-bin directory, preferably one only
#     accessible by an https:// URL, since the form data transmits the
#     passwords in cleartext otherwise.  Make sure the script is
#     executable (chmod 755 useradmin.pl) and is accessible by the
#     web user.
#
# (8) Run your web browser and enter in the URL to the script and you're
#     done!
#############################################################################



#############################################################################
# (Skip to the next section)
#############################################################################

use strict;
use CGI qw(:all escapeHTML);
use CGI::Carp qw(fatalsToBrowser);
use FileHandle;
use IPC::Open2;
use IPC::Open3;
use POSIX qw(strftime);

#############################################################################
# Change these!
#############################################################################

my ($ncftpd_passwd)		= "/usr/local/sbin/web_ncftpd_passwd";
my ($statedir)			= "/var/spool/ncftpd/useradmin";
my ($admin_password)		= "test123";
my ($log_file_name)		= "/var/log/ncftpd/useradmin.log";
my ($database_file_name)	= "/usr/local/etc/ncftpd/passwd.db";
my ($debug) 			= 0;

#############################################################################
# Optional: if want fields pre-populated, set the $default_* fields to
# something other than undef.  If you want them pre-populated and not
# changeable, set the $required_* fields instead.
#############################################################################

my ($default_password)		= undef;
my ($default_uid)		= undef;
my ($default_gids)		= undef;
my ($default_full_name)		= undef;
my ($default_home_dir)		= undef;
my ($default_shell)		= undef;
my ($default_quota_kbytes)	= undef;
my ($default_quota_nfiles)	= undef;
my ($default_bandwidth_dnload)	= undef;
my ($default_bandwidth_upload)	= undef;
my ($default_umask)		= undef;
my ($default_restricted)	= 1;
my ($default_user_permissions)	= undef;

my ($required_password)		= undef;
my ($required_uid)		= undef;
my ($required_gids)		= undef;
my ($required_full_name)	= undef;
my ($required_home_dir)		= undef;
my ($required_shell)		= "";		# Shell isn't used anyway
my ($required_quota_kbytes)	= undef;
my ($required_quota_nfiles)	= undef;
my ($required_bandwidth_dnload)	= undef;
my ($required_bandwidth_upload)	= undef;
my ($required_umask)		= undef;
my ($required_restricted)	= undef;
my ($required_user_permissions)	= undef;

my ($want_create_home_dir_checkbox)	= 1;
my ($create_home_dir_choice)		= 1;	# Checkbox default is on
my ($create_home_dir_mode)		= "00700";	# Mode bits, octal str
my ($want_umask_field)			= 1;
my ($want_user_permissions_field)	= 1;
my ($want_restriction_field)		= 0;	# Do not use unless u-always-restrict-virtual-users=no
my ($default_record_sortby)		= "username";	# username, uid, gid, or dir

#############################################################################
# No more user-configurable options below this point.
#############################################################################



# Consider a session expired after 15 minutes.
my ($stateexpiration)			= 15;

$CGI::POST_MAX=1024 * 4;		# max 4K posts
$CGI::DISABLE_UPLOADS = 1;		# no uploads
my (%cookies) 				= ();
my ($minimum_ncftpd_version)		= "2.8.0";
my ($started_html) 			= 0;
my ($finished_html) 			= 0;
my ($starttime)				= time();
my ($statefile)				= "";
my ($login_expired)			= 0;
my ($op)				= "";
my ($script)				= script_name();
my ($print_cmd_as_comment)		= 0;
my (@ncftpd_passwd_args)		= ();
my ($ncftpd_passwd_cmd) 		= "";
my ($ncftpd_passwd_stdout)		= "";
my ($ncftpd_passwd_stderr)		= "";
my ($ncftpd_passwd_status)		= -1;
my ($log_is_open)			= 0;




sub CookieHeader
{
	if (scalar(values %cookies) > 0) {
		return header(-cookie=>[values(%cookies)]);
	} else {
		return header();
	}
}	# CookieHeader




sub StartHTML
{
	my ($title) = $_[0] || "";

	if ($finished_html) {
		die("Already wrote a complete HTML page to remote browser.");
	} elsif ($started_html++ == 0) {
		print CookieHeader(), start_html($title), "\n\n";
		DebugParams() if ($debug);
	} else {
		# Start another section!
		print "\n\n\n<hr>\n<h3>$title</h3>\n\n";
	}
}	# StartHTML




sub EndHTML
{
	if (($started_html > 0) && ($finished_html++ == 0)) {
		print end_html(), "\n";
	}
}	# EndHTML




sub DebugParams
{
	my (@params) = param();
	my ($parm);

	print "\n\n", comment("Cookies and CGI parameters are listed here."), "\n";
	print "<TABLE border>\n";
	print "\t", Tr(th({-bgcolor=>"black"}, "<font color=white>Parameters</font>")), "\n";
	if (scalar(@params) == 0) {
		print "\t", Tr(td({-align=>"center"}, "(none)")), "\n";
	} else {
		print "\t<TR><TD>\n";
		print "\t\t<TABLE cellspacing=\"5\">\n";
		for $parm (@params) {
			$parm = escapeHTML($parm);
			print "\t\t\t", Tr(th({-align=>"right"}, $parm), td(escapeHTML(param($parm)))), "\n";
		}
		print "\t\t</TABLE>\n";
		print "\t</TR></TD>\n";
	}

	print "\t", Tr(th({-bgcolor=>"black"}, "<font color=white>Cookies</font>")), "\n";
	@params = cookie();
	if (scalar(@params) == 0) {
		print "\t", Tr(td({-align=>"center"}, "(none)")), "\n";
	} else {
		print "\t<TR><TD>\n";
		print "\t\t<TABLE cellspacing=\"5\">\n";
		for $parm (@params) {
			print "\t\t\t", Tr(th({-align=>"right"}, $parm), td(escapeHTML(cookie($parm)))), "\n";
		}
		print "\t\t</TABLE>\n";
		print "\t</TR></TD>\n";
	}

	print "</TABLE>\n\n";
}	# DebugParams




sub LsLd
{
	my ($output, @lsitems) = @_;
	local (*LSR, *LSW);
	my ($pid);
	my ($rc) = 0;
	my ($line);
	my ($old_sigpipe);

	$old_sigpipe = $SIG{PIPE};
	$SIG{PIPE} = 'IGNORE';

	$$output = "";
	$pid = open2(\*LSR, \*LSW, "/bin/ls", "-ld", @lsitems);
	if ($pid > 0) {
		close(LSW);
		while (defined($line = <LSR>)) {
			$$output .= $line;
		}
		close(LSR);
		waitpid($pid, 0);
		$rc = 1;
	}

	$SIG{PIPE} = $old_sigpipe;
	return ($rc);
}	# LsLd




sub NotAccessibleThenPrintError
{
	my ($pathname, $ftype) = @_;
	my ($need_r) = 0;
	my ($need_w) = 0;
	my ($need_x) = 0;
	my ($fail) = 0;

	$ftype = "-r--" if (! defined($ftype));
	return (-2) if (($ftype !~ /^[d\-]/) || (! defined($pathname)) || ($pathname eq ""));

	# Coalesce multiple slashes
	$pathname =~ s-/{2,}-/-g;

	# Remove trailing slashes
	$pathname =~ s-/+$--;

	$need_r = 1 if (index($ftype, "r") >= 0);
	$need_w = 1 if (index($ftype, "w") >= 0);
	$need_x = 1 if (index($ftype, "x") >= 0);
	$ftype = substr($ftype, 0, 1);

	$fail++ if (($ftype eq "-") && (! -f $pathname));
	$fail++ if (($ftype eq "d") && (! -d $pathname));
	if (! $fail) {
		#
		# It exists at least.
		#
		$fail++ if (($need_r) && (! -r _));
		$fail++ if (($need_w) && (! -w _));
		$fail++ if (($need_x) && (! -x _));
	}

	return (0) if ($fail == 0);

	StartHTML("Access error");

	if ($ftype eq "-") {
		if ($need_x) {
			print p(b("Error: "), "Cannot run <tt>$pathname</tt>."), "\n";
		} elsif (($need_r) && ($need_w)) {
			print p(b("Error: "), "Cannot access the file <tt>$pathname</tt> for reading and writing."), "\n";
		} elsif ($need_r) {
			print p(b("Error: "), "Cannot access the file <tt>$pathname</tt> for reading."), "\n";
		} elsif ($need_w) {
			print p(b("Error: "), "Cannot access the file <tt>$pathname</tt> for writing."), "\n";
		} else {
			print p(b("Error: "), "Cannot access the file <tt>$pathname</tt>."), "\n";
		}
	} else {
		if (($need_r) && ($need_w) && ($need_x)) {
			print p(b("Error: "), "Cannot access the directory <tt>$pathname</tt> for traversal (execution), listing (reading), creating and removing (writing)"), "\n";
		} elsif (($need_r) && ($need_x)) {
			print p(b("Error: "), "Cannot access the directory <tt>$pathname</tt> for traversal (execution) and listing (reading)"), "\n";
		} elsif (($need_w) && ($need_x)) {
			print p(b("Error: "), "Cannot access the directory <tt>$pathname</tt> for traversal (execution), and creating/removing (writing)"), "\n";
		} else {
			if ($need_x) {
				print p(b("Error: "), "Cannot access the directory <tt>$pathname</tt> for traversal (execution)"), "\n";
			}
			if ($need_r) {
				print p(b("Error: "), "Cannot access the directory <tt>$pathname</tt> for listing (reading)"), "\n";
			}
			if($need_w) {
				print p(b("Error: "), "Cannot access the directory <tt>$pathname</tt> for creating and removing (writing)"), "\n";
			}
		}
	}

	my ($usr) = getpwuid($>) || "UID $>";
	my (@gids) = split(/\s/, $));
	my ($grps) = "";
	my ($gid, $gid1, $grp, $skipgrp, $ngrp);
	$gid1 = "none";

	for $gid (@gids) {
		$skipgrp = 0;
		if ($gid1 eq "none") {
			$gid1 = $gid;
		} else {
			$skipgrp = 1 if ($gid == $gid1);
		}
		if (! $skipgrp) {
			$grp = getgrgid($gid);
			# $grp .= " ($gid)";
			if ($grps eq "") {
				$grps = tt($grp);
			} else {
				$grps .= ", " . tt($grp);
			}
		}
	}

	print p(
		tt(script_name()),
		"is running as user ", tt($usr), " and ",
		scalar(@gids) > 1 ? "groups" : "group",
		"$grps."
	), "\n";

	my ($lsitem) = $pathname;
	my (@lsitems) = ();
	while ($lsitem ne "") {
		push(@lsitems, $lsitem);
		$lsitem =~ s-/[^/]+$--;
	}

	my ($stuff);
	if (LsLd(\$stuff, @lsitems)) {;
		print p(ul(pre(escapeHTML($stuff)))), "\n";
	}

	$fail = 0;
	for $lsitem (reverse(@lsitems)) {
		if (! -e $lsitem) {
			my ($parent_dir) = $lsitem;
			if ($lsitem =~ /^(.*)\//) {
				$parent_dir = $1;
			} else {
				# Don't give us relative paths
				$parent_dir = "..";
			}
			if (($need_w) && (! -w $parent_dir)) {
				print p(b("Reason:"), "parent directory", tt($parent_dir), "is not writable."), "\n";
			} elsif ((($need_r) || ($need_x)) && (! -x $parent_dir)) {
				print p(b("Reason:"), "parent directory", tt($parent_dir), "is not traversable (executable)."), "\n";
			} else {
				print p(b("Reason: "), tt($lsitem), "does not exist."), "\n";
			}
			$fail++;
			last;
		} elsif ((-d $lsitem) && (! -x $lsitem)) {
			print p(b("Reason: "), "directory", tt($lsitem), "is not traversable (executable)."), "\n";
			$fail++;
			last;
		}
	}

	if ($fail == 0) {
		#
		# Then the item exists -- but it earlier failed one of our
		# other tests, so print what that was.
		#
		if ($ftype eq "d") {
			$ftype = "directory";
		} else {
			$ftype = "file";
		}
		stat($pathname);

		
		if (($need_r) && (! -r _)) {
			print p(b("Reason: "), "the $ftype", tt($pathname), "is not readable."), "\n";
		}
		if (($need_w) && (! -w _)) {
			print p(b("Reason: "), "the $ftype", tt($pathname), "is not writable."), "\n";
		}
		if (($need_x) && (! -x _)) {
			print p(b("Reason: "), "the $ftype", tt($pathname), "is not executable."), "\n";
		}
	}

	return (-1);
}	# NotAccessibleThenPrintError




sub Log
{
	return unless ((defined($log_file_name)) && ($log_file_name ne ""));

	if (! $log_is_open) {
		if (! open(LOG, ">> $log_file_name")) {
			StartHTML("Configuration Error");
			print p("Could not open the log file designated by \$log_file_name (<tt>$log_file_name</tt>) for writing.\nEdit the script and change the value of \$log_file_name to \"\" if you do not want a log, or fix the following problem:"), "\n";

			NotAccessibleThenPrintError($log_file_name, "--w-");
			EndHTML();
			exit(1);
		}
		$log_is_open++;
	}

	my ($pfx) = strftime("%Y-%m-%d %H:%M:%S | ", localtime(time()));
	print LOG $pfx;
	print LOG @_ if (scalar(@_) > 0);
	print LOG "\n";
}	# Log




sub CheckForSavedState
{
	my ($mtime);

	$statefile = "";
	if (cookie("state")) {
		$statefile = cookie("state");
		unless ($statefile =~ /^\d{15}$/) {
			# illegal state -- faked data?
			#
			if ($statefile ne "logged off") {
				warn "illegal session cookie: [$statefile]\n";
			}
			$statefile = "";
		}
	}

	if ($statefile) {
		$statefile = sprintf("%s/state_%s", $statedir, $statefile);
		$mtime = (stat($statefile))[9];
		if ((! defined($mtime)) || ($mtime < (time() - ($stateexpiration * 60)))) {
			# State file was too old.
			# Make them re-login.
			#
			#warn "expired session cookie: [$statefile]\n";
			$login_expired = 1;
			unlink($statefile);
			$statefile = "";
		} else {
			return (1);
		}
	}

	# Assign a new state.
	#
	$statefile = sprintf("%s/state_%05u%05u%05u", $statedir, (int(rand(100000))), $starttime % 100000, (int(rand(100000))));
	return (0);
}	# CheckForSavedState




sub TouchSessionCookie
{
	my($statekey) = $statefile;
	$statekey =~ s/^.*state_//;
	#
	# Note: we have the cookie expire in 30 days, but that is because
	# we maintain the session expiration on the server.  We want this
	# to be long so we can display a "login expired" message.
	#
	$cookies{"state"} = cookie(-PATH=>script_name(), -NAME=>"state", -VALUE=>$statekey, -EXPIRES=>sprintf("+%ud", 30));
}	# TouchSessionCookie




sub SaveState
{
#TODO: USE FILELOCKING#
	my($msgid);

	return (0) if ($statefile eq "");

	unless (open(STATEFILE, ">$statefile")) {
		StartHTML("Configuration Error");
		print p("Could not create the session token file in the designated \$statedir directory (<tt>$statedir</tt>).\nEdit the script and change the value of \$statedir or fix the following problem:"), "\n";

		NotAccessibleThenPrintError($statefile, "--w-");
		EndHTML();
		exit(1);
	}

	printf STATEFILE ("LOGIN from %s\n", remote_host());
	close(STATEFILE);
}	# SaveState




sub Login
{
	my ($login_incorrect) = 0;

	if (CheckForSavedState()) {
		#
		# Already logged in and still active
		#
		return (1);
	}

	if ((defined(param("op"))) && (lc(param("op")) eq "quit")) {
		Logoff();
		return (0);
	}

	if (defined(param("admin_pass"))) {
		#
		# Process supplied credentials.
		#
		if ($admin_password eq "test123") {
			StartHTML("Configuration Error");
			print p("Edit this script and change the <tt>\$admin_password</tt> -- this script does not allow you to use factory default password.");
			EndHTML();
			exit(1);
		}

		my ($ver) = "0";
		my ($youhave) = "";
		my ($minver);
		if ($minimum_ncftpd_version =~ /(\d+)\.(\d+)\.(\d+)/) {
			$minver = sprintf("%d%02d%02d",  $1, $2, $3);
		}
		if (RunNcftpd_passwd("-b") >= 0) {
			if ($ncftpd_passwd_stdout =~ /NcFTPd (\d+)\.(\d+)\.(\d+)/) {
				my ($ver2) = $1 . "." . $2 . "." . $3;
				$ver = sprintf("%d%02d%02d",  $1, $2, $3);
				$youhave = "  Your version of <tt>ncftpd_passwd</tt> is $ver2.";
				Log("VERSION of ncftpd_passwd is $ver2; minimum needed is $minimum_ncftpd_version");
			}
		}
		if ($ver < $minver) {
			StartHTML("Configuration Error");
			print p("This script requires the <tt>ncftpd_passwd</tt> utility program from NcFTPd $minimum_ncftpd_version or newer.$youhave  Perhaps you forgot to update your <tt>web_ncftpd_passwd</tt> when you upgraded.  View the instructions in the script file for details.");
			EndHTML();
			exit(1);
		}
		
		if (param("admin_pass") eq $admin_password) {
			SaveState();
			Log("LOGIN from ", remote_host());
			return (1);
		}
		Log("LOGIN FAILED: bad password from ", remote_host());
		++$login_incorrect;
		sleep(3);
	}

	#
	# Ask for credentials.
	#
	StartHTML("NcFTPd: Virtual User Administration: Login");
	print h3("Login"), "\n";

	print p("This administration program needs to be able to use cookies.  If your browser does not allow cookies, you will not be able to login.  Your login session will expire after ", $stateexpiration, " minutes of inactivity."), "\n";
	if ($login_incorrect) {
		print p("<font color=red>Login incorrect.</font>  Please try again."), "\n";
	} elsif ($login_expired) {
		Log("EXPIRED");
		print p("<font color=red>Your login session has expired.</font>  Please login again to continue."), "\n";
	}

	my ($formurl) = self_url();
	print start_form(-action=>$formurl, -method=>"POST"), "\n";
	my ($op) = param("op");
	print hidden(-name=>"op", -value=>$op), "\n" if (defined($op));
	print table(
		Tr(th({-align=>"right"}, "Admin Password:"), td(password_field(-name=>"admin_pass", -size=>16))), "\n",
		Tr(th(), td(submit(-name=>"Login", -value=>"Login"))), "\n"
	);
	print end_form(), "\n";
	return (0);
}	# Login




sub Logoff
{
	my($statekey) = $statefile;
	$statekey =~ s/^.*state_//;
	$cookies{"state"} = cookie(-PATH=>script_name(), -NAME=>"state", -VALUE=>$statekey, -EXPIRES=>"now");
	unlink($statefile);
	$statefile = "";

	Log("LOGOFF from ", remote_host());
	StartHTML("NcFTPd: Virtual User Administration: Logoff");
	print h3("Logoff"), "\n";

	print p("You are now logged off."), "\n";
	print p("You may", (a({href=>"$script?op=menu"}, "Login")), "again now."), "\n";
}	# Logoff




sub RunNcftpd_passwd
{
	local (*PIPE_stdin, *PIPE_stdout, *PIPE_stderr);
	my ($done_stdout);
	my ($done_stderr);
	my ($pid, $arg);
	my ($old_sigpipe);
	my ($c_arg) = 0;
	
	#
	# Create displayable version of argument list
	#
	@ncftpd_passwd_args = @_;
	$ncftpd_passwd_cmd = "$ncftpd_passwd";
	foreach $arg (@ncftpd_passwd_args) {
		$c_arg++ if ($arg eq "-c");
		if (($arg =~ /([^:]+:)([^:]+)(:.*)/) && ($c_arg != 0)) {
			$ncftpd_passwd_cmd .= " '" . $1 . "********" . $3 . "'";
		} elsif ($arg =~ /[\:\s\|\;\$\&\*\?\(\()\~\`\\]/) {
			$ncftpd_passwd_cmd .= " '$arg'";
		} elsif ($arg eq "") {
			$ncftpd_passwd_cmd .= " ''";
		} else {
			$ncftpd_passwd_cmd .= " $arg";
		}
	}
	Log("Attempting to run:  $ncftpd_passwd_cmd");

	if ($debug) {
		StartHTML("Debug");
		print h3("Command"), "\n";
		print p(ul(pre(escapeHTML($ncftpd_passwd_cmd)))), "\n";
	}

	return (-1) if (NotAccessibleThenPrintError($ncftpd_passwd, "---x") < 0);

	$ncftpd_passwd_status = -1;
	$ncftpd_passwd_stdout = "";
	$ncftpd_passwd_stderr = "";

	$old_sigpipe = $SIG{PIPE};
	$SIG{PIPE} = 'IGNORE';

	$pid = open3(\*PIPE_stdin, \*PIPE_stdout, \*PIPE_stderr,
		$ncftpd_passwd, @ncftpd_passwd_args);

	if ($pid < 2) {
		StartHTML("Error");
		print p(b("Error: "), "Could not run ncftpd_passwd."), "\n";
		print p(b("Reason: "), "open3 failed: $!"), "\n";
		print p("The command attempted was:", "\n");
		print ul(pre(escapeHTML($ncftpd_passwd_cmd))), "\n";
		if (defined($old_sigpipe)) { $SIG{PIPE} = $old_sigpipe; }
		return (-2);
	}

	close(PIPE_stdin);

	$done_stdout = $done_stderr = 0;
	while ((! $done_stdout) || (! $done_stderr)) {
		my ($rin, $rout);
		my ($nread, $data);

		$rin = '';
		if ($done_stdout == 0) {
			vec($rin, fileno(PIPE_stdout), 1) = 1;
		}
		if ($done_stderr == 0) {
			vec($rin, fileno(PIPE_stderr), 1) = 1;
		}

		# Wait forever until data input or EOF is ready
		$rout = $rin;
		$nread = select($rout, undef, undef, undef);
		if (($nread <= 0) || (! defined($rout))) {
			close(PIPE_stdout) unless ($done_stdout);
			close(PIPE_stderr) unless ($done_stderr);
			if (defined($old_sigpipe)) { $SIG{PIPE} = $old_sigpipe; }
			StartHTML("Error");
			print p(b("Error: "), "select() failed."), "\n";
			return (-3);
		}

		if ((! $done_stdout) && (vec($rout, fileno(PIPE_stdout), 1) == 1)) {
			$nread = sysread(PIPE_stdout, $data, 256);
			if ($nread <= 0) {
				close(PIPE_stdout);
				$done_stdout = 1;
			} else {
				$ncftpd_passwd_stdout .= $data;
			}
		}
		
		if ((! $done_stderr) && (vec($rout, fileno(PIPE_stderr), 1) == 1)) {
			$nread = sysread(PIPE_stderr, $data, 256);
			if ($nread <= 0) {
				close(PIPE_stderr);
				$done_stderr = 1;
			} else {
				$ncftpd_passwd_stderr .= $data;
			}
		}
	}

	$ncftpd_passwd_status = ($? >> 8) if (waitpid($pid, 0) > 0);
	if (defined($old_sigpipe)) { $SIG{PIPE} = $old_sigpipe; }
	Log("ncftpd_passwd exit status = $ncftpd_passwd_status");
	Log("ncftpd_passwd STDERR = {\n$ncftpd_passwd_stderr\n}\n") if ($ncftpd_passwd_stderr ne "");

	return (1);
}	# RunNcftpd_passwd




sub PrintErrorFooter
{
	if ($ncftpd_passwd_stderr ne "") {
		print "\n\n\n";
		print "<hr>\n" if ($ncftpd_passwd_stdout ne "");
		print "<h3>Error Messages</h3>\n";
		print p(ul(pre(escapeHTML($ncftpd_passwd_stderr)))), "\n";
		print "\n\n\n<hr><h3>Command Attempted</h3>\n";
		print p(ul(pre(escapeHTML($ncftpd_passwd_cmd)), p("Exit status code: ", b($ncftpd_passwd_status)))), "\n";
	} elsif ($print_cmd_as_comment) {
		print "\n\n\n<!-- command attempted was:\n$ncftpd_passwd_cmd\nexit status = $ncftpd_passwd_status\n-->\n\n\n";
	}
}	# PrintErrorFooter




sub PasswdDBExport
{
	my ($attachment_name) = $database_file_name;

	$attachment_name =~ s/^.*\///;
	$attachment_name =~ s/\.db/.txt/i;

	Log("EXPORT $database_file_name");
	if ((RunNcftpd_passwd("-f", $database_file_name, "-x") >= 0) && ($ncftpd_passwd_stdout ne "")) {
		print header(-cookie=>[values(%cookies)], -attachment=>$attachment_name, -type=>'text/plain');
		print $ncftpd_passwd_stdout;
	} else {
		StartHTML("Error");
		PrintErrorFooter();
		print p({-align=>"right"}, i(a({href=>"$script?op=menu"}, "Return to Main Menu"))), "\n";
	}
}	# PasswdDBExport




sub PasswdDBUIDSummary
{
	my ($rec);
	my (@fields);
	my ($nrecords) = 0;
	my ($uid, $start_uid, $end_uid);
	my (@uids) = ();
	my ($smallest_uid) = (-666);
	my ($largest_uid) = (-666);

	Log("UID_REPORT $database_file_name");
	if ((RunNcftpd_passwd("-f", $database_file_name, "-x") >= 0) && ($ncftpd_passwd_stdout ne "")) {
		StartHTML("Report");
		while ($ncftpd_passwd_stdout =~ /^(.*)$/igm) {
			$rec = $1;
			$nrecords++;

			# Records are in /etc/passwd format.
			@fields = split(/:/, $rec);
			$uid = $fields[2];
			if ($nrecords == 1) {
				$smallest_uid = $largest_uid = $uid;
			} elsif ($uid > $largest_uid) {
				$largest_uid = $uid;
			} elsif ($uid < $smallest_uid) {
				$smallest_uid = $uid;
			}
			push(@uids, $uid);
		}

		
		print "<table>\n";
		print "<tr><th align=left>Number of records in database:</th><td align=right>$nrecords</td></tr>\n";
		print "<tr><th align=left>Lowest UID in use:</th><td align=right>$smallest_uid</td></tr>\n";
		print "<tr><th align=left>Highest UID in use:</th><td align=right>$largest_uid</td></tr>\n";
		print "<tr><td>&nbsp;</td></tr>\n";
		print "<tr><th align=left>All UIDs in use:</th></tr>\n";
		print "</table>\n";

		print "<p><ul>\n";
		$start_uid = -666;
		$end_uid = -666;
		foreach $uid (sort { $a <=> $b } @uids) {
			if ($start_uid == -666) {
				$start_uid = $uid;
				$end_uid = $uid;
				next;
			}
			if ($uid == ($end_uid + 1)) {
				$end_uid = $uid;
			} elsif ($uid != $end_uid) {
				if ($start_uid == $end_uid) {
					print "$start_uid<BR>\n";
				} else {
					print "$start_uid .. $end_uid<BR>\n";
				}
				$start_uid = $uid;
				$end_uid = $uid;
			}
		}
		if ($start_uid == $end_uid) {
			print "$start_uid<BR>\n";
		} else {
			print "$start_uid .. $end_uid<BR>\n";
		}
		print "</ul></p>\n";
		PrintErrorFooter();
		print p({-align=>"right"}, i(a({href=>"$script?op=menu"}, "Return to Main Menu"))), "\n";
	} else {
		StartHTML("Error");
		PrintErrorFooter();
		print p({-align=>"right"}, i(a({href=>"$script?op=menu"}, "Return to Main Menu"))), "\n";
	}
}	# PasswdDBUIDSummary



sub PasswdRecordListHeader
{
	my ($newurl) = self_url();
	my ($sortby) = lc($default_record_sortby);
	my ($sortby3);
	my ($h_User) = "User";
	my ($h_UID) = "UID";
	my ($h_GIDs) = "GIDs";
	my ($h_Directory) = "Directory";
	my ($h_Full_Name) = "Full Name";

	if (defined(param("sortby"))) {
		$sortby = lc(param("sortby"));
		CGI::delete('sortby');
	}
	$sortby3 = substr($sortby, 0, 3);

	if ($sortby3 eq "uid") { $h_UID = ">" . $h_UID . "<"; }
	if ($sortby3 eq "gid") { $h_GIDs = ">" . $h_GIDs . "<"; }
	if ($sortby3 eq "dir") { $h_Directory = ">" . $h_Directory . "<"; }
	if ($sortby3 eq "ful") { $h_Full_Name = ">" . $h_Full_Name . "<"; }
	if ($sortby3 eq "use") { $h_User = ">" . $h_User . "<"; }

	$newurl = self_url();

	print "<table cellspacing=5>\n\n";
	print "<tr valign=bottom>\n";
	print "\t<th></th>\n";
	print "\t<th align=left><a href=\"${newurl}&sortby=username\">$h_User</a></th>\n";
	print "\t<th align=center><a href=\"${newurl}&sortby=uid\">$h_UID</a></th>\n";
	print "\t<th align=left><a href=\"${newurl}&sortby=gid\">$h_GIDs</a></th>\n";
	print "\t<th align=left><a href=\"${newurl}&sortby=fullname\">$h_Full_Name</a></th>\n";
	print "\t<th align=left><a href=\"${newurl}&sortby=dir\">$h_Directory</a></th>\n";
	print "\t<th align=center>Quota<br>kB</th>\n";
	print "\t<th align=center>Quota<br>files</th>\n";
	print "\t<th align=center>Bandwidth<br>Download</th>\n";
	print "\t<th align=center>Bandwidth<br>Upload</th>\n";
	print "\t<th align=center>Umask</th>\n" if ($want_umask_field);
	print "\t<th align=center>Restricted</th>\n" if ($want_restriction_field);
	print "\t<th align=center>User Permissions</th>\n" if ($want_user_permissions_field);
	print "</tr>\n\n";
}	# PasswdRecordListHeader



sub FormatKB
{
	my ($fld) = $_[0];

	return "" unless ($fld ne "");
	if ($fld >= 1024 * 1024) {
		$fld = sprintf("%.2f&nbsp;GB", $fld / (1024.0 * 1024.0));
		$fld =~ s/\.00//;
	} elsif ($fld >= 1024) {
		$fld = sprintf("%.2f&nbsp;MB", $fld / 1024.0);
		$fld =~ s/\.00//;
	} else {
		$fld = "$fld&nbsp;kB";
	}
	$fld .= $_[1] if (defined($_[1]));
	return ($fld);
}	# FormatKB




sub PasswdRecordListItem
{
	my ($orecord) = $_[0];
	my (@fields) = split(/:/, $orecord);
	my ($oop) = param("op");
	my ($op);
	my ($newurl);
	my ($umsk) = "";
	my ($perm) = "";
	my ($restr) = "";
	my ($fullname);
	my ($gids);
	
	param("op", $op = ($_[1] || "edit"));
	param("orecord", $orecord);
	$newurl = self_url();

	printf("<tr align=left valign=baseline>\n");
	printf("\t<td><small>[<a href=\"%s\">$op</a>]</small></td>\n", $newurl);

	# Use whitespace rather than zero fields (mean the same thing)
	# so those that do have quotas stand out when printed.
	#
	$fields[7] = "" if ((! defined($fields[7])) || ($fields[7] == 0));
	$fields[8] = "" if ((! defined($fields[8])) || ($fields[8] == 0));
	$fields[9] = "" if ((! defined($fields[9])) || ($fields[9] == 0));
	$fields[10] = "" if ((! defined($fields[10])) || ($fields[10] == 0));
	$fields[11] = "" if (! defined($fields[11]));
	$fields[12] = "" if ((! defined($fields[12])) || ($fields[12] == 0));
	$fields[13] = "" if ((! defined($fields[13])) || ($fields[13] == 0));

	$fields[7] = FormatKB($fields[7]);
	$fields[8] = FormatKB($fields[8]);
	$fields[12] = FormatKB($fields[12], "/sec");
	$fields[13] = FormatKB($fields[13], "/sec");

	if ($fields[11] =~ /umask=(\d+)/) { $umsk = $1; }
	if ($fields[11] =~ /perm[^=]*=([^,\n\t]*)/) { $perm = $1; }
	if ($fields[11] =~ /restr[^=]*=(\d+)/) { $restr = $1; }
	if ($fields[11] =~ /restr[^=]*=[yt]/i) { $restr = 1; }
	if ($fields[11] =~ /restr[^=]*=[nf]/i) { $restr = 0; }
	$restr = "yes" if (($restr ne "") && ($restr != 0));
	$restr = "no" if (($restr ne "") && ($restr ne "yes") && ($restr == 0));
	$perm = escapeHTML($perm);
	$perm =~ s/\s/\&nbsp;/g;
	
	$fullname = escapeHTML($fields[4]);
	$fullname =~ s/\s/\&nbsp;/g;
	$gids = escapeHTML($fields[3]);
	if ($gids =~ /^([^,]+),(.*)/) {
		$gids = $1 . "<small>," . $2 . "</small>";
	}
	
	printf("\t<td><tt>%s</tt></td>\n", escapeHTML($fields[0]));
	printf("\t<td align=right>%s</td>\n", escapeHTML($fields[2]));
	printf("\t<td align=left>%s</td>\n", $gids);
	printf("\t<td align=left>%s</td>\n", $fullname);
	printf("\t<td><tt>%s</tt></td>\n", escapeHTML($fields[5]));
	printf("\t<td align=right>%s</td>\n", ($fields[7]));
	printf("\t<td align=right>%s</td>\n", escapeHTML($fields[9]));
	printf("\t<td align=right>%s</td>\n", ($fields[12]));
	printf("\t<td align=right>%s</td>\n", ($fields[13]));
	printf("\t<td align=left>%s</td>\n", escapeHTML($umsk)) if ($want_umask_field);
	printf("\t<td align=center>%s</td>\n", escapeHTML($restr)) if ($want_restriction_field);
	printf("\t<td align=left><nobr><tt>%s</tt></nobr></td>\n", $perm) if ($want_user_permissions_field);
	printf("</tr>\n");

	param("op", $oop) if (defined($oop));
	param("orecord", "");
}	# PasswdRecordListItem




sub PasswdRecordListTrailer
{
	print "</table>\n\n\n";
	# print "<hr>\n";
	# print p(pre(escapeHTML($ncftpd_passwd_stdout))), "\n";
}	# PasswdRecordListTrailer




sub PasswdDBList
{
	my ($title) = "List $database_file_name";
	my ($sortby) = lc($default_record_sortby);
	my ($sortby3);

	StartHTML("NcFTPd: Virtual User Administration: $title");

	if (defined(param("sortby"))) {
		$sortby = lc(param("sortby"));
	}

	Log("LIST $database_file_name");
	return unless (RunNcftpd_passwd("-f", $database_file_name, "-x") >= 0);
	if ($ncftpd_passwd_stdout ne "") {
		my ($record);
		my (@records) = ();
		$sortby3 = substr($sortby, 0, 3);
		if ($sortby3 eq "uid") {
			@records = sort {
				my ($anum, $bnum);
				$anum = (split(/:/, $a))[2] || 0;
				$bnum = (split(/:/, $b))[2] || 0;
				$anum <=> $bnum
			} (split(/\n/, $ncftpd_passwd_stdout));
		} elsif ($sortby3 eq "gid") {
			@records = sort {
				my ($anum, $bnum);
				$anum = (split(/:/, $a))[3] || "";
				$bnum = (split(/:/, $b))[3] || "";
				$anum = (split(/,/, $anum))[0] || 0;
				$bnum = (split(/,/, $bnum))[0] || 0;
				$anum <=> $bnum
			} (split(/\n/, $ncftpd_passwd_stdout));
		} elsif ($sortby3 eq "ful") {
			@records = sort {
				my ($aname, $bname);
				$aname = (split(/:/, $a))[4] || "";
				$bname = (split(/:/, $b))[4] || "";
				$aname cmp $bname
			} (split(/\n/, $ncftpd_passwd_stdout));
		} elsif ($sortby3 eq "dir") {
			@records = sort {
				my ($aname, $bname);
				$aname = (split(/:/, $a))[5] || "";
				$bname = (split(/:/, $b))[5] || "";
				$aname cmp $bname
			} (split(/\n/, $ncftpd_passwd_stdout));
		} else {	# username
			@records = sort {
				my ($aname, $bname);
				$aname = (split(/:/, $a))[0] || "";
				$bname = (split(/:/, $b))[0] || "";
				$aname cmp $bname
			} (split(/\n/, $ncftpd_passwd_stdout));
		}

		PasswdRecordListHeader();
		foreach $record (@records) {
			PasswdRecordListItem($record, "edit");
		}
		PasswdRecordListTrailer();
	} else {
	}
	PrintErrorFooter();
	print p({-align=>"right"}, i(a({href=>"$script?op=menu"}, "Return to Main Menu"))), "\n";
}	# PasswdDBList




sub PasswdDBQuery
{
	my ($title) = "Query by name";
	my ($query_op, $query_op_name) = @_;

	StartHTML("NcFTPd: Virtual User Administration: $title");

	if (! defined(param("name"))) {
		#
		# Ask them which user to query.
		#
		my ($formurl) = self_url();
		print start_form(-action=>$formurl, -method=>"POST"), "\n";
		my ($op) = param("op");
		param("query_op", $query_op);
		print hidden(-name=>"query_op", -value=>$query_op), "\n" if (defined($query_op));
		print hidden(-name=>"op", -value=>$op), "\n" if (defined($op));
		print table(
			Tr(th({-align=>"right"}, "Username:"), td(textfield(-name=>"name", -size=>16))), "\n",
			Tr(th(), td(submit(-name=>$query_op_name))), "\n"
		);
		print end_form(), "\n";
	} else {
		#
		# We have the username, now look it up and display the results.
		#
		$query_op = param("query_op") || "edit";
		if ($query_op ne "query") {
			Log("QUERY (in order to $query_op) ", param("name"));
		} else {
			Log("QUERY ", param("name"));
		}
		return unless (RunNcftpd_passwd("-f", $database_file_name, "-q", param("name")) >= 0);
		if ($ncftpd_passwd_status == 0) {
			$query_op = "edit" if ($query_op) eq "query";
			if ($query_op eq "delete") {
				print h3("Are you sure you want to <font color=red>delete</font> this user?"), "\n";
			}

			my ($record);
			my (@records) = sort {
				my ($aname, $bname);
				$aname = (split(/:/, $a))[0] || "";
				$bname = (split(/:/, $b))[0] || "";
				$aname cmp $bname
			} (split(/\n/, $ncftpd_passwd_stdout));

			PasswdRecordListHeader();
			foreach $record (@records) {
				PasswdRecordListItem($record, $query_op);
			}
			PasswdRecordListTrailer();
		}
		PrintErrorFooter();
	}
	print p({-align=>"right"}, i(a({href=>"$script?op=menu"}, "Return to Main Menu"))), "\n";
}	# PasswdDBQuery




sub PasswdDBDelete
{
	my ($title);
	my ($orecord);
	my ($user_to_delete);

	return unless (defined($orecord = param("orecord")));

	$user_to_delete = (split(/:/, $orecord))[0];
	$title = "Delete user $user_to_delete";

	StartHTML("NcFTPd: Virtual User Administration: $title");

	Log("DELETE: [$orecord]");

	#print "\n\n\n<!--\n";
	#print "DELETE = [", (escapeHTML($orecord)), "]\n";
	#print "-->\n\n\n";

	return unless (RunNcftpd_passwd("-f", $database_file_name, "-d", $user_to_delete) >= 0);
	print h3("Command Output"), "\n";
	print p(ul(pre(escapeHTML($ncftpd_passwd_stdout)))), "\n";
	PrintErrorFooter();
	print p({-align=>"right"}, i(a({href=>"$script?op=menu"}, "Return to Main Menu"))), "\n";
}	# PasswdDBDelete




sub PasswdDBEdit
{
	my ($title);
	my ($edit_op_name);
	my ($edit_op);
	my ($orecord) = "";
	my ($pass_confirm) = "";
	my (@fields) = ();
	my (@boxes) = ();
	my ($umsk) = "";
	my ($perm) = "";
	my ($restr) = "";

	$restr = $default_restricted if ($want_restriction_field);
	
	if ((defined(param("orecord"))) && (defined(param("edit_op")))) {
		#
		# Submitting the completed add/change form
		#
		my (@form_errors) = ();
		my ($fld, $fldname, $record, $msg);
		my ($enc_it) = 0;	# Encrypt a new password?
		my ($mkd_it) = 0;	# Mkdir $HOME?

		$record = "";
		$orecord = param("orecord");
		$edit_op = param("edit_op");
		if ($edit_op eq "add") {
			$title = "Add new user";
			$edit_op_name = "Add User";
			$fields[0] = param("f_username") if defined(param("f_username"));
		} else {
			@fields = split(/:/, $orecord);	
			$pass_confirm = $fields[1] if (defined($fields[1]));
			$title = "Edit user " . $fields[0];
			$edit_op_name = "Modify User";
		}

		$enc_it = 1 if ((defined(param("f_dbuser_password"))) && (param("f_dbuser_password") ne ""));
		$fields[1] = param("f_dbuser_password") if ((defined(param("f_dbuser_password"))) && (param("f_dbuser_password") ne ""));
		$enc_it = 0 if ((defined(param("f_encpass"))) && (param("f_encpass") ne ""));
		$fields[1] = param("f_encpass") if ((defined(param("f_encpass"))) && (param("f_encpass") ne ""));
		$pass_confirm = param("f_dbuser_password_confirm") if ((defined(param("f_dbuser_password_confirm"))) && (param("f_dbuser_password_confirm") ne ""));
		$fields[2] = param("f_uid") if ((defined(param("f_uid"))) && (param("f_uid") ne ""));
		$fields[3] = param("f_gids") if ((defined(param("f_gids"))) && (param("f_gids") ne ""));
		$fields[4] = param("f_gecos") if ((defined(param("f_gecos"))) && (param("f_gecos") ne ""));
		$fields[5] = param("f_homedir") if ((defined(param("f_homedir"))) && (param("f_homedir") ne ""));
		$mkd_it = 1 if ((defined(param("f_mkdir"))) && (lc(param("f_mkdir")) eq "on"));
		$fields[6] = param("f_shell") if ((defined(param("f_shell"))) && (param("f_shell") ne ""));
		$fields[7] = param("f_qbh") if ((defined(param("f_qbh"))) && (param("f_qbh") ne ""));
		$fields[8] = param("f_qbs") if ((defined(param("f_qbs"))) && (param("f_qbs") ne ""));
		$fields[9] = param("f_qfh") if ((defined(param("f_qfh"))) && (param("f_qfh") ne ""));
		$fields[10] = param("f_qfs") if ((defined(param("f_qfs"))) && (param("f_qfs") ne ""));
		$fields[11] = param("f_qflags") if ((defined(param("f_qflags"))) && (param("f_qflags") ne ""));
		$fields[12] = param("f_bdn") if ((defined(param("f_bdn"))) && (param("f_bdn") ne ""));
		$fields[13] = param("f_bup") if ((defined(param("f_bup"))) && (param("f_bup") ne ""));
		$umsk = param("f_umask") if ((defined(param("f_umask"))) && (param("f_umask") ne ""));
		$perm = param("f_perm") if ((defined(param("f_perm"))) && (param("f_perm") ne ""));
		$restr = 1 if ((defined(param("f_restr"))) && (lc(param("f_restr")) eq "on"));
		$restr = 0 if ((defined(param("f_restr"))) && (lc(param("f_restr")) ne "on"));
		$restr = 0 if ((defined(param("f_restr"))) && (lc(param("f_restr")) eq ""));
		$restr = 0 if ((! defined(param("f_restr"))) && ($want_restriction_field));

		$fld = $fields[0];
		$fldname = "username";
		if (! defined($fld)) {
			push(@form_errors, "Missing $fldname field.");
		} elsif (($fld eq "") || ($fld =~ /:/))  {
			push(@form_errors, "Invalid $fldname field (\"$fld\").");
		}

		$fields[1] = $required_password if (defined($required_password));
		$fld = $fields[1];
		$fldname = "password";
		if (! defined($fld)) {
			push(@form_errors, "Missing $fldname field.");
		} elsif (($fld eq "") || ($fld =~ /:/))  {
			push(@form_errors, "Invalid $fldname field (\"$fld\").");
		}
		if ($fields[1] ne $pass_confirm) {
			push(@form_errors, "The two passwords typed do not match.  Type the same password carefully in both the \"Password\" and \"Password (confirm)\" fields.");
		}

		$fields[2] = $required_uid if (defined($required_uid));
		$fld = $fields[2];
		$fldname = "UID";
		if (! defined($fld)) {
			push(@form_errors, "Missing $fldname field.");
		} elsif ($fld !~ /^\d+$/) {
			push(@form_errors, "Invalid $fldname field (\"$fld\" is not a whole number).");
		}

		$fields[3] = $required_gids if (defined($required_gids));
		$fld = $fields[3];
		$fldname = "GID";
		if (! defined($fld)) {
			push(@form_errors, "Missing $fldname field.");
		} elsif ($fld =~ /[^\d,\s]/) {
			push(@form_errors, "Invalid $fldname field (\"$fld\" is not a comma-delimited list of whole numbers such as \"555\" or \"555,777,778\").");
		} else {
			$fields[3] =~ s/\s//g;
			$fields[3] =~ s/,,+/,/g;
			$fields[3] =~ s/^,//g;
			$fields[3] =~ s/^$//g;
		}

		$fields[4] = $required_full_name if (defined($required_full_name));
		$fld = $fields[4];
		$fldname = "Full Name (also known as GECOS)";
		if (! defined($fld)) {
			$fields[4] = "";	# Allowed to be empty
		} elsif ($fld =~ /:/)  {
			push(@form_errors, "Invalid $fldname field (\"$fld\").");
		}

		$fields[5] = $required_home_dir if (defined($required_home_dir));
		$fld = $fields[5];
		$fldname = "Home Directory";
		if (! defined($fld)) {
			push(@form_errors, "Missing $fldname field.");
		} elsif (($fld !~ /^\//) || ($fld =~ /:/))  {
			push(@form_errors, "Invalid $fldname field (\"$fld\" is not an absolute UNIX directory pathname).");
		}

		$fields[6] = $required_shell if (defined($required_shell));
		$fld = $fields[6];
		$fldname = "User Shell";
		if ((! defined($fld)) || ($fld eq "")) {
			$fields[6] = "/bin/false";
		} elsif (($fld !~ /^\//) || ($fld =~ /:/))  {
			push(@form_errors, "Invalid $fldname field (\"$fld\" is not an absolute UNIX pathname).");
		}

		$fields[7] = $required_quota_kbytes if (defined($required_quota_kbytes));
		$fld = $fields[7];
		$fldname = "Hard Quota for Bytes Used";
		if ((! defined($fld)) || ($fld eq "")) {
			$msg = "no-op";		# Allowed to be empty
		} elsif ($fld !~ /^(\d+)([kmMgG])?$/) {
			push(@form_errors, "Invalid $fldname field (\"$fld\" is not a whole number).");
		} else {
			$fields[7] = $1;
			$fields[7] *= 1024 if (uc($2) eq "M");
			$fields[7] *= 1024 * 1024 if (uc($2) eq "G");
		}

		$fld = $fields[8];
		$fldname = "Soft Quota for Bytes Used";
		if ((! defined($fld)) || ($fld eq "")) {
			$msg = "no-op";		# Allowed to be empty
		} elsif ($fld !~ /^(\d+)([kmMgG])?$/) {
			push(@form_errors, "Invalid $fldname field (\"$fld\" is not a whole number).");
		} else {
			$fields[8] = $1;
			$fields[8] *= 1024 if (uc($2) eq "M");
			$fields[8] *= 1024 * 1024 if (uc($2) eq "G");
		}

		$fields[9] = $required_quota_nfiles if (defined($required_quota_nfiles));
		$fld = $fields[9];
		$fldname = "Hard Quota for Number of Files";
		if ((! defined($fld)) || ($fld eq "")) {
			$msg = "no-op";		# Allowed to be empty
		} elsif ($fld !~ /^\d+$/) {
			push(@form_errors, "Invalid $fldname field (\"$fld\" is not a whole number).");
		}

		$fld = $fields[10];
		$fldname = "Soft Quota for Number of Files";
		if ((! defined($fld)) || ($fld eq "")) {
			$msg = "no-op";		# Allowed to be empty
		} elsif ($fld !~ /^\d+$/) {
			push(@form_errors, "Invalid $fldname field (\"$fld\" is not a whole number).");
		}

		$fields[12] = $required_bandwidth_dnload if (defined($required_bandwidth_dnload));
		$fld = $fields[12];
		$fldname = "Bandwidth Download Quota in kBytes/sec";
		if ((! defined($fld)) || ($fld eq "")) {
			$msg = "no-op";		# Allowed to be empty
		} elsif ($fld !~ /^(\d+)([kmMgG])?$/) {
			push(@form_errors, "Invalid $fldname field (\"$fld\" is not a whole number).");
		} else {
			$fields[12] = $1;
			$fields[12] *= 1024 if (uc($2) eq "M");
			$fields[12] *= 1024 * 1024 if (uc($2) eq "G");
		}

		$fields[13] = $required_bandwidth_upload if (defined($required_bandwidth_upload));
		$fld = $fields[13];
		$fldname = "Bandwidth Upload Quota in kBytes/sec";
		if ((! defined($fld)) || ($fld eq "")) {
			$msg = "no-op";		# Allowed to be empty
		} elsif ($fld !~ /^(\d+)([kmMgG])?$/) {
			push(@form_errors, "Invalid $fldname field (\"$fld\" is not a whole number).");
		} else {
			$fields[13] = $1;
			$fields[13] *= 1024 if (uc($2) eq "M");
			$fields[13] *= 1024 * 1024 if (uc($2) eq "G");
		}

		# The 12th field (Field 11) is now a special use field for several variables.
		#
		$umsk = $required_umask if (defined($required_umask));
		$restr = $required_restricted if (defined($required_restricted));
		$perm = $required_user_permissions if (defined($required_user_permissions));
		$fld = "";
		if ($umsk ne "") {
			if ($umsk !~ /^[0-7]{1,3}$/) {
				push(@form_errors, "Invalid umask (\"$umsk\").");
			} else {
				$umsk = sprintf("%03d", $umsk);
				$fld .= ",umask=$umsk";
			}
		}
		if (($want_restriction_field) && (defined($restr)) && ($restr ne "")) {
			if ($restr == 0) {
				$fld .= ",restricted=no";
			} else {
				$fld .= ",restricted=yes";
			}
		}
		if ($perm ne "") {
			if ($perm !~ /^[\ \-\*\+0acCdDlLMnoOrRtuwW]*$/) {
				push(@form_errors, "Invalid user permission string (\"$perm\").");
			} else {
				$fld .= ",perm=$perm";
			}
		}
		$fld =~ s/[:\t\r\n]//g;
		$fld =~ s/^,//;
		$fields[11] = $fld;

		for (my $ix = 0; $ix <= 13; $ix++) {
			if (! defined($fields[$ix])) { $fields[$ix] = ""; }
		}
		$record = join(':', @fields);
		StartHTML("NcFTPd: Virtual User Administration: $title");

		if ($edit_op eq "add") {
			Log("ADD: [$record]");
			#print "\n\n\n<!--\n";
			#print "NEW = [", (escapeHTML($record)), "]\n";
			#print "-->\n\n\n";
		} else {
			Log("CHANGE FROM: [$orecord]");
			Log("CHANGE TO:   [$record]");
			#print "\n\n\n<!--\n";
			#print "OLD = [", (escapeHTML($orecord)), "]\n";
			#print "NEW = [", (escapeHTML($record)), "]\n";
			#print "-->\n\n\n";
		}

		if (scalar(@form_errors) > 0) {
			print "<h3>Error</h3>\n";
			for $msg (@form_errors) {
				print p(escapeHTML($msg)), "\n";
			}
		} else {
			my ($aflag) = ($edit_op eq "add") ? "-a" : "-u";
			my (@ncftpd_passwd_args) = (
				"-f",
				$database_file_name
			);
			push(@ncftpd_passwd_args, "-c") if ($enc_it); 
			push(@ncftpd_passwd_args, $aflag);
			push(@ncftpd_passwd_args, $record);

			if ($mkd_it) {
				push(@ncftpd_passwd_args, "-M");
				push(@ncftpd_passwd_args, $create_home_dir_mode);
			}
			
			if ($fields[5] eq "/ftp/Data/FTP") {
				#
				# Custom hack for TheFund
				#
				if (! $mkd_it) {
					push(@ncftpd_passwd_args, "-M");
					push(@ncftpd_passwd_args, $create_home_dir_mode);
				}
				push(@ncftpd_passwd_args, "-J");
				push(@ncftpd_passwd_args, "/ftp/Data/FTP/Users/" . $fields[0]);
			}

			return unless (RunNcftpd_passwd(@ncftpd_passwd_args) >= 0);
			print h3("Command Output"), "\n";
			print p(ul(pre(escapeHTML($ncftpd_passwd_stdout)))), "\n";
			PrintErrorFooter();
		}
	} else {
		#
		# Need to present a form for the admin to add/change.
		#
		if (defined(param("orecord"))) {
			$orecord = param("orecord");
			@fields = split(/:/, $orecord);	
			$title = "Edit user " . $fields[0];
			$edit_op_name = "Modify User";
			StartHTML("NcFTPd: Virtual User Administration: $title");
		} else {
			$orecord = "";
			$title = "Add new user";
			$edit_op_name = "Add User";
			StartHTML("NcFTPd: Virtual User Administration: $title");
		}

		#
		# Present user modification form
		#
		my ($formurl) = self_url();
		print start_form(-action=>$formurl, -method=>"POST"), "\n";
		my ($op) = param("op");
		print hidden(-name=>"op", -value=>$op), "\n" if (defined($op));
		print hidden(-name=>"orecord", -value=>$orecord);

		if ($edit_op_name eq "Add User") {
			$fields[1] = $default_password if (defined($default_password));
			$fields[14] = $default_password if (defined($default_password));
			$fields[2] = $default_uid if (defined($default_uid));
			$fields[3] = $default_gids if (defined($default_gids));
			$fields[4] = $default_full_name if (defined($default_full_name));
			$fields[5] = $default_home_dir if (defined($default_home_dir));
			$fields[6] = $default_shell if (defined($default_shell));
			$fields[7] = $default_quota_kbytes if (defined($default_quota_kbytes));
			$fields[8] = "";
			$fields[9] = $default_quota_nfiles if (defined($default_quota_nfiles));
			$fields[10] = "";
			$fields[11] = "";
			$fields[12] = $default_bandwidth_dnload if (defined($default_bandwidth_dnload));
			$fields[13] = $default_bandwidth_upload if (defined($default_bandwidth_upload));
		}

		if ($edit_op_name eq "Modify User") {
			#
			# By default, leave blank to denote that the existing
			# password should be kept.
			#
			$fields[1] = $fields[14] = undef;
		}

		if (defined($fields[1])) {
			$boxes[1] = password_field(-name=>"f_dbuser_password", -size=>16, -value=>$fields[1]);
		} else {
			$boxes[1] = password_field(-name=>"f_dbuser_password", -size=>16);
		}
		if (defined($fields[14])) {
			$boxes[14] = password_field(-name=>"f_dbuser_password_confirm", -size=>16, -value=>$fields[14]);
		} else {
			$boxes[14] = password_field(-name=>"f_dbuser_password_confirm", -size=>16);
		}

		if ($fields[11] =~ /umask=(\d+)/) { $umsk = $1; }
		if ($fields[11] =~ /perm[^=]*=([^,\n\t]*)/) { $perm = $1; }
		if ($fields[11] =~ /restr[^=]*=(\d+)/) { $restr = $1; }
		if ($fields[11] =~ /restr[^=]*=[yt]/i) { $restr = 1; }
		if ($fields[11] =~ /restr[^=]*=[nf]/i) { $restr = 0; }
		
		$boxes[2] = textfield(-name=>"f_uid", -size=>6, -value=>$fields[2]);
		$boxes[3] = textfield(-name=>"f_gids", -size=>16, -value=>$fields[3]);
		$boxes[4] = textfield(-name=>"f_gecos", -size=>30, -value=>$fields[4]);
		$boxes[5] = textfield(-name=>"f_homedir", -size=>50, -value=>$fields[5]);
		$boxes[6] = textfield(-name=>"f_shell", -size=>30, -value=>$fields[6]);
		$boxes[7] = textfield(-name=>"f_qbh", -size=>8, -value=>$fields[7]);
		$boxes[8] = "";
		$boxes[9] = textfield(-name=>"f_qfh", -size=>8, -value=>$fields[9]);
		$boxes[10] = "";
		$boxes[11] = "";
		$boxes[12] = textfield(-name=>"f_bdn", -size=>8, -value=>$fields[12]);
		$boxes[13] = textfield(-name=>"f_bup", -size=>8, -value=>$fields[13]);
		$boxes[15] = checkbox("f_mkdir", $create_home_dir_choice, "on", "Create?");
		$boxes[15] = "" unless ($want_create_home_dir_checkbox);
		$boxes[16] = textfield(-name=>"f_umask", -size=>3, -value=>$umsk);
		$restr = 1 if (! defined($restr));
		$boxes[17] = checkbox("f_restr", $restr, "on", "");
		$boxes[18] = textfield(-name=>"f_perm", -size=>30, -value=>$perm);
				
		my ($user_table_row);
		if ($edit_op_name eq "Modify User") {
			print hidden(-name=>"f_username", -value=>$fields[0]);
			print hidden(-name=>"edit_op", -value=>"update");
			$user_table_row = Tr(th({-align=>"right"}, "Username:"), td($fields[0]), td({-width=>"30%"}, ""));
		} else {
			print hidden(-name=>"edit_op", -value=>"add");
			$user_table_row = Tr(th({-align=>"right"}, "Username:"), td(textfield(-name=>"f_username", -size=>16, -value=>$fields[0])), td({-width=>"30%"}, ""));

			$boxes[1] = escapeHTML($required_password) if (defined($required_password));
			$boxes[14] = escapeHTML($required_password) if (defined($required_password));
			$boxes[2] = escapeHTML($required_uid) if (defined($required_uid));
			$boxes[3] = escapeHTML($required_gids) if (defined($required_gids));
			$boxes[4] = escapeHTML($required_full_name) if (defined($required_full_name));

			if (defined($required_home_dir)) {
				$boxes[5] = "<tt>" . escapeHTML($required_home_dir) . "</tt>";
				$boxes[15] = "" if (-d $required_home_dir);
			}
			if (defined($required_shell)) {
				$boxes[6] = "<tt>" . escapeHTML($required_shell) . "</tt>";
			}
			$boxes[7] = escapeHTML($required_quota_kbytes) if (defined($required_quota_kbytes));
			$boxes[9] = escapeHTML($required_quota_nfiles) if (defined($required_quota_nfiles));
			$boxes[12] = escapeHTML($required_bandwidth_dnload) if (defined($required_bandwidth_dnload));
			$boxes[13] = escapeHTML($required_bandwidth_upload) if (defined($required_bandwidth_upload));
			$boxes[16] = escapeHTML($required_umask) if (defined($required_umask));
			$boxes[17] = escapeHTML($required_restricted) if (defined($required_restricted));
			if (defined($required_user_permissions)) {
				$boxes[18] = "<tt>" . escapeHTML($required_user_permissions) . "</tt>";
			}
			$boxes[16] = textfield(-name=>"f_umask", -size=>3, -value=>$umsk);
			$boxes[17] = checkbox("f_restr", $restr, "on", "");
			$boxes[18] = textfield(-name=>"f_perm", -size=>30, -value=>$perm);
		}
		print "\n\n<table width=\"100%\">\n";

		print $user_table_row, "\n";

		print Tr(th({-align=>"right"}, "Password:"), td($boxes[1]), td(i("Leave blank to keep existing password."))), "\n";

		print Tr(th({-align=>"right"}, "Password&nbsp;<small>(confirm)</small>:"), td($boxes[14]), td(i(""))), "\n";

		print Tr(th({-align=>"right"}, "UID:"), td($boxes[2])), "\n";

		print Tr(th({-align=>"right"}, "GIDs:"), td($boxes[3]), td(i("For membership in more than one group, use a comma-delimited list."))), "\n";

		print Tr(th({-align=>"right"}, "Full&nbsp;Name:"), td($boxes[4]), td(i("This field is optional."))), "\n";

		print Tr(th({-align=>"right"}, "Home&nbsp;Directory:"), td($boxes[5], "&nbsp;&nbsp;", $boxes[15], "&nbsp;&nbsp;")), "\n";

		if (defined($required_shell) && ($required_shell eq "") && (($edit_op_name eq "Add User") || (($edit_op_name eq "Modify User") && (($fields[6] eq "/bin/false") || ($fields[6] eq ""))))) {
			print "<!-- shell field omitted -->\n";
		} else {
			print Tr(th({-align=>"right"}, "Shell:"), td($boxes[6]), td(i("Will not be run, so it can be left empty."))), "\n";
		}
		
		print Tr(th({-align=>"right"}, "Disk&nbsp;Quota:"), td(
			$boxes[7],
			" kBytes" .
			"&nbsp;&nbsp;&nbsp;" .
			$boxes[9],
			" files "
			), td(i("Set to zero or leave empty for no limit."))), "\n";
		print Tr(th({-align=>"right"}, "Bandwidth&nbsp;Limit:"), td(
			$boxes[12],
			" download kBytes/second" .
			"&nbsp;&nbsp;&nbsp;" .
			$boxes[13],
			" upload kBytes/second" .
			"&nbsp;&nbsp;&nbsp;"
			), td(i("Set to zero or leave empty for no limit."))), "\n";

		print Tr(th({-align=>"right"}, "Umask:"), td($boxes[16]), td(i("Leave empty unless you want to override the setting of <nobr><tt><a href=\"http://www.ncftp.com/ncftpd/doc/config/d/u-umask.html\">u-umask</a></tt></nobr> for this specific user."))), "\n" if ($want_umask_field);
		print Tr(th({-align=>"right"}, "Restricted:"), td($boxes[17]), td(i("Restricted to home directory?"))), "\n" if ($want_restriction_field);
		print Tr(th({-align=>"right"}, "User&nbsp;Permissions:"), td($boxes[18]), td(i("Leave empty unless you want to override the setting of <nobr><tt><a href=\"http://www.ncftp.com/ncftpd/doc/config/g/u-restrict-mode.html\">u-restrict-mode</a></tt></nobr> for this specific user."))), "\n" if ($want_user_permissions_field);

		print Tr(th(), td(submit(-name=>$edit_op_name))), "\n";

		print "</table>\n\n\n";
		print end_form(), "\n";
	}

	print p({-align=>"right"}, i(a({href=>"$script?op=menu"}, "Return to Main Menu"))), "\n";
}	# PasswdDBEdit




sub MainMenu
{
	Log("MENU");
	StartHTML("NcFTPd: Virtual User Administration: Main Menu");

	print "<p>Select the function you wish to perform on the <i>NcFTPd</i> password database <tt>", $database_file_name, "</tt>:\n";
	print ul(
		li(a({href=>"$script?op=list"},"List"), " all users"),
		li(a({href=>"$script?op=query"},"Query"), " a specific user by name"),
		li(a({href=>"$script?op=edit"},"Edit"), " an existing user"),
		li(a({href=>"$script?op=add"},"Add"), " a new user"),
		li(a({href=>"$script?op=delete"},"Delete"), " an existing user"),
		li(a({href=>"$script?op=export"},"Export"), " database into text format"),
		li(a({href=>"$script?op=uidreport"},"Report"), " on User ID (UID) usage"),
		li(a({href=>"$script?op=quit"},"Logoff")),
	);

}	# MainMenu




sub Main
{
	umask(077);
	if (Login()) {
		TouchSessionCookie();

		if (! defined(param("op"))) {
			MainMenu();
		} else {
			$op = lc(param("op"));
			if ($op eq "menu") {
				MainMenu();
			} elsif ($op eq "list") {
				PasswdDBList();
			} elsif ($op eq "query") {
				PasswdDBQuery("query", "Query");
			} elsif ($op eq "edit") {
				if (! param("orecord")) {
					PasswdDBQuery("edit", "Edit");
				} else {
					PasswdDBEdit();
				}
			} elsif ($op eq "add") {
				PasswdDBEdit();
			} elsif ($op eq "delete") {
				if (! param("orecord")) {
					PasswdDBQuery("delete", "Delete");
				} else {
					PasswdDBDelete();
				}
			} elsif ($op eq "export") {
				PasswdDBExport();
			} elsif ($op eq "uidreport") {
				PasswdDBUIDSummary();
			} elsif ($op eq "quit") {
				Logoff();
			} else {
				# Unknown op.
				MainMenu();
			}
		}

		SaveState();	# Update time of last activity
	}
	EndHTML();	# Finish our page, if we started writing one
}	# Main

Main();
