#!/usr/bin/perl -w
#
# This script is Copyright (C) 2002-2006 by Mike Gleason, NcFTP Software.
# All Rights Reserved.
#
# This script tries to use the "lsof" utility if the package included it.
# Thanks to Vic Abell <abe@purdue.edu> of the Purdue University
# Computing Center for providing "lsof".
#
use strict;
use Socket;
use File::Copy;
use POSIX qw(strftime);

# Porting checklist:
#   df behavior
#   du -s -k
#   ps behavior
#   echo -n / echo \c
#   rc scripts
#   lsof
#   /var/log preference
#   adduser program
#   uninstall_ncftpd.pl
#

my ($verbose) 				= 1;
my ($force_install)			= 0;
my ($prefix) 				= "";
my ($eprefix) 				= "";
my ($bindir) 				= "";
my ($sbindir) 				= "";
my ($datadir)				= "";
my ($sysconfdir)			= "";
my ($runtimedatadir)			= "/var/run";
my ($logdir)				= "/var/log";
my ($libdir)				= "";
my ($includedir)			= "";
my ($mandir)				= "";
my ($ftpuser_preferred)			= "ftp";
my ($ftpgroup_preferred)		= "ftp";
my ($ftpuser)				= $ftpuser_preferred;
my ($ftpgroup)				= $ftpgroup_preferred;
my ($ftpuid)				= -1;
my ($ftpgid)				= -1;
my ($webuser)				= "";
my ($webgroup)				= "";
my ($webuid)				= -1;
my ($webgid)				= -1;
my ($host)				= "";
my ($OS)				= "";
my ($SYS)				= "";
my ($linux_libc)			= "";
my ($ftphome)				= "";
my ($ncftpd_running_at_time_of_install)	= 0;
my ($ftp_service_running_at_time_of_install) = "";
my ($lsof)				= "";
my ($perl)				= "";
my ($ftpport)				= 21;
my ($ftpservicename)			= "ftp";
my ($install_version)			= "";
my ($existing_version)			= "";
my ($existing_ncftpd)			= "";
my ($existing_general_cf)		= "";
my ($existing_domain_cf)		= "";
my ($new_general_cf)			= "";
my ($new_domain_cf)			= "";
my ($new_ncftpd)			= "";
my ($confdir)				= "";
my ($rptbindir)				= "";
my ($install_src_root)			= "";
my ($extra_paths)			= "";
my ($create_new_configs)		= 0;
my ($inittab_preferred)			= 0;
my ($run_ncftpd_verbose) 		= 0;
my ($startup_error_log)			= "";
my ($rc_startup_script)			= "";
my ($inittab_startup_script)		= "";
my ($linux_distro)			= "";
my ($linux_distro_class)		= "";
my ($linux_distro_version)		= "";
my ($linux_distro_version_num)		= 0;
my ($linux_distro_release_file)		= "";
my ($install_log)			= "";
my ($tee_output_to_log)			= 1;	# Change to 0 to if this script hangs.
my ($log_time_spec) 			= "%H:%M:%S";
my ($pager) 				= "";
my ($pager_pid)				= 0;
my (@uninstall_items)			= ();
my ($uninstaller)			= "";
my ($binary_compat_err)			= "";
my ($remove_ftp_from_ftpusers)		= 1;
my ($use_upstart)			= -1;	# -1 means we will try to determine this automatically.
my ($initctl)				= "";
my ($upstart_jobs_dir) 		= "/etc/init";
my (@upstart_competing_services)	= ();
my ($use_systemd)			= -1;	# -1 means we will try to determine this automatically.
my ($systemctl)				= "";
my ($systemd_services_dir) 		= "/lib/systemd/system";
my (@systemd_competing_services)	= ();
my ($use_launchd)			= -1;	# -1 means we will try to determine this automatically.
my ($launchctl)				= "";
my ($launchd_jobs_dir) 			= "/System/Library/LaunchDaemons";
my (@launchd_competing_services)	= ( "com.apple.ftpd:ftp.plist:ftpd" );
my ($original_command_line)		= "";



sub Usage
{
	print <<EOF;
Usage: $0 [options]
Options: [defaults in brackets after descriptions]
Invocation:
  --help                  Print this message
  --quiet, --silent       Do not print `checking...' messages
  --debug                 Print debugging messages
Directory and file names:
  --prefix=PREFIX         Install architecture-independent files in PREFIX
                          [/usr/local]
  --exec-prefix=EPREFIX   Install architecture-dependent files in EPREFIX
                          [same as prefix]
  --bindir=DIR            User executables in DIR [EPREFIX/bin]
  --sbindir=DIR           System admin executables in DIR [EPREFIX/sbin]
  --sysconfdir=DIR        Read-only single-machine data in DIR [PREFIX/etc]
  --runtimedatadir=DIR    Dynamic runtime state information DIR [$runtimedatadir]
  --logdir=DIR            Create log files in DIR [$logdir]
  --lsof=FILE             Use FILE as path to list-open-files program
Features and packages:
  --port=NUMBER           Install on port NUMBER [21]
  --create-new-configs    Do not try to reuse existing configs, if present
  --inittab               Run NcFTPd from /etc/inittab rather than an RC script
  --enable-verbose-mode   Enable verbose logging mode by default
  --leave-ftp-in-ftpusers Do not try to remove "ftp" from /etc/ftpusers
EOF
	Exit(2);
}	# Usage



sub StripExtraSlashes
{
	my ($dir) = @_;
	$dir =~ s/\/{2,}/\//g;
	return ($dir) if ($dir eq "/");
	$dir =~ s/\/+$//g;
	return ($dir);
}	# StripExtraSlashes	




sub Getopts
{
	my($i, $arg, $val);
	my (@ARGVcopy) = @ARGV;

	$original_command_line = $0;
	for $arg (@ARGVcopy) {
		if ($arg =~ /\s/) { $arg = "'" . $arg . "'"; }
		$original_command_line .= " " . $arg;
	}

	for ($i = 0; $i < scalar(@ARGV); $i++) {
		$arg = $ARGV[$i];
		if ($arg eq "--help") {
			Usage();
		} elsif ($arg eq "--force") {
			$force_install = 1;
		} elsif ($arg eq "--create-new-configs") {
			$create_new_configs = 1;
		} elsif ($arg eq "--leave-ftp-in-ftpusers") {
			$remove_ftp_from_ftpusers = 0;
		} elsif ($arg eq "--inittab") {
			if (! -f "/etc/inittab") {
				Exit(1, "This system does not use /etc/inittab.\n");
			}
			$inittab_preferred = 1;
		} elsif ($arg eq "--silent") {
			$verbose = 0;
		} elsif ($arg eq "--quiet") {
			$verbose = 0;
		} elsif ($arg eq "--debug") {
			$verbose = 2;
		} elsif ($arg =~ /^-{1,2}(v+)(=(.*))?$/) {
			if (defined($3)) {
				$run_ncftpd_verbose = int($3);
			} else {
				$run_ncftpd_verbose = length($1);
			}
		} elsif ($arg =~ /^(--enable-verbose-mode|--with-verbose-mode)(=(.*))?$/) {
			$run_ncftpd_verbose = 1;
			if (defined($3)) {
				$run_ncftpd_verbose = int($3);
			}
		} elsif ($arg =~ /^--debug=(.*)/) {
			$val = int($1);
			Exit(1, "$val is not a valid debugging level.\n")
				if (($val < 0) || ($val > 10));
			$verbose = $val;
		} elsif ($arg =~ /^--port=(.*)/) {
			$val = int($1);
			Exit(1, "$val is not a valid port number.\n")
				if (($val < 1) || ($val >= 65535));
			$ftpport = $val;
		} elsif ($arg =~ /^--lsof=(.*)/) {
			$val = StripExtraSlashes($1);
			Exit(1, "$val is not an absolute pathname.\n")
				if ($val !~ /^\//);
			$lsof = $val;
		} elsif ($arg =~ /^--prefix=(.*)/) {
			$val = StripExtraSlashes($1);
			Exit(1, "$val is not an absolute pathname.\n")
				if ($val !~ /^\//);
			$prefix = $val;
		} elsif ($arg =~ /^--exec-prefix=(.*)/) {
			$val = StripExtraSlashes($1);
			Exit(1, "$val is not an absolute pathname.\n")
				if ($val !~ /^\//);
			$eprefix = $val;
		} elsif ($arg =~ /^--bindir=(.*)/) {
			$val = StripExtraSlashes($1);
			Exit(1, "$val is not an absolute pathname.\n")
				if ($val !~ /^\//);
			$bindir = $val;
		} elsif ($arg =~ /^--sbindir=(.*)/) {
			$val = StripExtraSlashes($1);
			Exit(1, "$val is not an absolute pathname.\n")
				if ($val !~ /^\//);
			$sbindir = $val;
		} elsif ($arg =~ /^--datadir=(.*)/) {
			$val = StripExtraSlashes($1);
			Exit(1, "$val is not an absolute pathname.\n")
				if ($val !~ /^\//);
			$datadir = $val;
		} elsif ($arg =~ /^--sysconfdir=(.*)/) {
			$val = StripExtraSlashes($1);
			Exit(1, "$val is not an absolute pathname.\n")
				if ($val !~ /^\//);
			$sysconfdir = $val;
		} elsif ($arg =~ /^--runtimedatadir=(.*)/) {
			$val = StripExtraSlashes($1);
			Exit(1, "$val is not an absolute pathname.\n")
				if ($val !~ /^\//);
			$runtimedatadir = $val;
		} elsif ($arg =~ /^--logdir=(.*)/) {
			$val = StripExtraSlashes($1);
			Exit(1, "$val is not an absolute pathname.\n")
				if ($val !~ /^\//);
			$logdir = $val;
		} elsif ($arg =~ /^--libdir=(.*)/) {
			$val = StripExtraSlashes($1);
			Exit(1, "$val is not an absolute pathname.\n")
				if ($val !~ /^\//);
			$libdir = $val;
		} elsif ($arg =~ /^--includedir=(.*)/) {
			$val = StripExtraSlashes($1);
			Exit(1, "$val is not an absolute pathname.\n")
				if ($val !~ /^\//);
			$includedir = $val;
		} elsif ($arg =~ /^--mandir=(.*)/) {
			$val = StripExtraSlashes($1);
			Exit(1, "$val is not an absolute pathname.\n")
				if ($val !~ /^\//);
			$mandir = $val;
		} else {
			printf ("Argument not recognized: \"%s\".\nTry \"%s --help\" command line options.\n", $arg, $0);
			Exit(3);
		}
	}

	$prefix = "/usr/local" if (! $prefix);
	$eprefix = $prefix if (! $eprefix);
	$bindir = $eprefix . "/bin" if (! $bindir);
	$sbindir = $eprefix . "/sbin" if (! $sbindir);
	$datadir = $prefix . "/share" if (! $datadir);
	$sysconfdir = $prefix . "/etc" if (! $sysconfdir);
	$runtimedatadir = "/var" if (! $runtimedatadir);
	$libdir = $eprefix . "/lib" if (! $libdir);
	$includedir = $prefix . "/include" if (! $includedir);
	$mandir = $prefix . "/man" if (! $mandir);

	$confdir = "$sysconfdir/ncftpd";
	$rptbindir = "$sysconfdir/ncftpd/rptbin";
	$new_general_cf = "$confdir/general.cf";
	$new_domain_cf = "$confdir/domain.cf";
	$new_ncftpd = "$sbindir/ncftpd";

	Log(2, "Original command line: $original_command_line\n");
	if ($verbose > 1) {
		Log(1, "prefix=$prefix\n");
		Log(1, "eprefix=$eprefix\n");
		Log(1, "bindir=$bindir\n");
		Log(1, "sbindir=$sbindir\n");
		Log(1, "datadir=$datadir\n");
		Log(1, "sysconfdir=$sysconfdir\n");
		Log(1, "runtimedatadir=$runtimedatadir\n");
		Log(1, "logdir=$logdir\n");
	#	Log(1, "libdir=$libdir\n");
	#	Log(1, "includedir=$includedir\n");
	#	Log(1, "mandir=$mandir\n");
		Log(1, "port=$ftpport\n");
		Log(1, "run_ncftpd_verbose=$run_ncftpd_verbose\n");
		Log(1, "\n");
	}
}	# Getopts




sub Mkdirs
{
	my ($path, $mode, $uid, $gid) = @_;
	my ($subpath);
	my ($i, $n, $e);
	my (@nodes);
	my($absolute) = 0;

	if (!defined($mode)) { $mode = 00755; }		# Umask will NOT affect.
	@nodes = split(/\/+/, $path);

	$n = scalar(@nodes);
	return 1 if ($n <= 0);		# It was just the root directory

	if ($nodes[0] eq "") {
		#
		# Absolute paths will leave us with an
		# empty first node.
		#
		$absolute = 1;
		shift(@nodes);
		$n--;
	}

	#
	# Find the greatest part of the path that
	# already exists.  We will then create the
	# remaining nodes.
	#
	while (1) {
		$subpath = "";
		if ($absolute) { $subpath = "/"; }
		for ($i = 0; $i < $n; $i++) {
			$subpath .= $nodes[$i] . "/";
		}
		chop($subpath);
		last if (-d "$subpath");
		if (--$n <= 0) {
			$subpath = "";
			$n = 0;
			last;
		}
	}

	#
	# We now have in $subpath the portion of $path
	# that already exists.  We now just need to
	# create the remaining node(s).  We have in $n
	# the number of successive nodes which exist.
	#
	$e = $n;
	$n = scalar(@nodes);
	return 1 if ($e == $n);			# Entire path already exists

	for ($i = $e ; $i < $n; $i++) {
		if (($subpath eq "") && (! $absolute)) {
			$subpath = $nodes[$i];
		} else {
			$subpath .= "/" . $nodes[$i];
		}
		return 0 if (! mkdir($subpath,$mode));
		chown($uid, $gid, $subpath) if (($> == 0) && (defined($uid)) && (defined($gid)));
		chmod($mode, $subpath);
		UninstallLog("Rmdir(\"$subpath\");\n");
	}

	return 1;				# Finished
}	# Mkdirs




sub RunProgram
{
	my ($exit_if_not_present) = shift(@_);
	my (@args) = @_;
	my ($t0) = time();
	my $rc = system(@args);
	my ($elapsed) = time() - $t0;
	my ($arg) = "";
	my (@ARGVcopy) = @args;
	my ($command_line) = "";
	my ($elapstr) = "";

	if ($elapsed > 1) {
		$elapstr = sprintf(" (finished in %d seconds)", $elapsed);
	}

	for $arg (@ARGVcopy) {
		# Use this for Q&D logging purpose only; it doesn't handle quotes properly.
		if ($arg =~ /\'/) { $arg = "\"" . $arg . "\""; }
		elsif ($arg =~ /\"/) { $arg = "'" . $arg . "'"; }
		elsif ($arg =~ /\s/) { $arg = "'" . $arg . "'"; }
		$command_line .= " " if ($command_line ne "");
		$command_line .= $arg;
	}

	if ($rc == 0) {
		Log(1, "RunProgram ($command_line) OK$elapstr.\n");
		return (1);	# OK
	} elsif ($? == -1) {
		Log(0,  "RunProgram ($command_line) failed: $!$elapstr.\n");
		Exit(1) if ($exit_if_not_present);
	} elsif ($? & 127) {
		Log(0, sprintf("RunProgram ($command_line) died with signal %d%s$elapstr.\n", ($? & 127), (($? & 128) ? " (core dumped)" : "")));
		Exit(1);
	} else {
		Log(1, sprintf("RunProgram ($command_line) failed (es=%d)$elapstr.\n", ($? >> 8)));
	}
	return (0);		# Failed
}	# RunProgram




sub RunProgramIfPresent
{
	RunProgram(0, @_);
}	# RunProgramIfPresent




sub RunProgramOrExit
{
	RunProgram(1, @_);
}	# RunProgramOrExit




sub DiskFreeBytes
{
	my ($path) = @_;
	my ($dfresult) = "";	
	my (@dflines);
	my (@column_headers);
	my (@column_values);
	my ($blocksize) = 0;

	$path = "\'$path\'";
	if ($^O eq "sco3.2v5.0") {
		$dfresult = `df -P -k $path 2>/dev/null`;
	} elsif ($^O eq "sunos") {
		$dfresult = `/usr/bin/df $path 2>/dev/null`;
	} else {
		$dfresult = `df -k $path 2>/dev/null`;
	}

	#
	# This is an ugly hack, but it's pick your poison:
	# Parse "df" output which varies per platform,
	# or try to use even more variants of statfs
	# and friends from within Perl.
	#
	# The "df" option also implies that the user must
	# be running in English for this to work, but this
	# is still the most feasible option.
	#
	@dflines = split(/\r?\n/, $dfresult);
	return (-1) if (scalar(@dflines) < 2);

# aix> df -k /usr/local/bin
# Filesystem    1024-blocks      Free %Used    Iused %Iused Mounted on
# /dev/hd2           749568     20136   98%    28087    15% /usr

# solaris> df -k /usr/local/bin
# Filesystem            kbytes    used   avail capacity  Mounted on
# /dev/dsk/c0t0d0s3    3527406 1175661 2316471    34%    /opt

# sunos> /usr/bin/df /usr/local
# Filesystem            kbytes    used   avail capacity  Mounted on
# /dev/sd0h             665662  380527  218569    64%    /home

# unixware7> df -k /usr/local/bin
# filesystem          kbytes   used     avail    capacity  mounted on
# /dev/root           2369587  729785   1639802  31%       /

# openunix8> df -k /usr/local/bin
# filesystem          kbytes   used     avail    capacity  mounted on
# /dev/root           8209215  1601741  6607474  20%       /

# linux> df -k /usr/local/bin
# Filesystem           1k-blocks      Used Available Use% Mounted on
# /dev/sdb5             31245640  15312240  14346180  52% /usr

# linux (Fedora 4 with logical volume groups) > df -k /usr/local/bin
# Filesystem           1K-blocks      Used Available Use% Mounted on
# /dev/mapper/VolGroup00-LogVol00
#                        3301112   2709544    421176  87% /

# openbsd> df -k /usr/local/bin
# Filesystem  1K-blocks     Used    Avail Capacity  Mounted on
# /dev/sd0a     1770994   233886  1448559    14%    /

# bsd/os> df -k /usr/local/bin
# Filesystem  1024-blocks     Used    Avail Capacity  Mounted on
# /dev/sd0h       3998700   921434  2877331    24%    /usr

# irix> df -k /usr/local/bin
# Filesystem             Type  kbytes     use     avail  %use Mounted on
# /dev/root               xfs  1962860  1568024   394836  80  /

# tru64unix> df -k /usr/local/bin
# Filesystem     1024-blocks        Used   Available Capacity  Mounted on
# usr_domain#usr     2150400      782697     1307000    38%    /usr
#

# scosv> df -P -k /usr/local/bin
# Filesystem         1024-blocks     Used Available Capacity Mounted on
# /dev/root              4893578   909216   3984362      19% /usr/local

	@column_headers = split(' ', $dflines[0]);
	@column_values = split(' ', $dflines[1]);

	# Hack for Linux with logical volumes
	if ((scalar(@column_values) == 1) && ($dflines[2] =~ /^\s/)) {
		# Filesystem           1K-blocks      Used Available Use% M...
		# /dev/mapper/VolGroup00-LogVol00
		#                        3301112   2709544    421176  87% /
		@column_values = split(' ', $dflines[2]);
		unshift(@column_values, "/dev/null");
	}

	if ((scalar(@column_headers) >= 5) && (scalar(@column_values) >= 5)) {
		if ($column_headers[1] eq "1k-blocks") {
			$blocksize = 1024;
		} elsif ($column_headers[1] eq "1024-blocks") {
			$blocksize = 1024;
		} elsif ($column_headers[1] eq "1K-blocks") {
			$blocksize = 1024;
		} elsif (($column_headers[1] eq "kbytes") && ($column_headers[2] eq "used")) {
			$blocksize = 1024;
		} elsif (($column_headers[2] eq "kbytes") && ($column_headers[3] eq "use")) {
			$blocksize = 1024;
		}

		if ($blocksize == 1024) {
			if ($column_headers[2] eq "Free") {
				return ($blocksize * $column_values[2]);
			} elsif ($column_headers[3] eq "avail") {
				return ($blocksize * $column_values[3]);
			} elsif ($column_headers[3] eq "Avail") {
				return ($blocksize * $column_values[3]);
			} elsif ($column_headers[3] eq "Available") {
				return ($blocksize * $column_values[3]);
			} elsif ($column_headers[4] eq "avail") {
				return ($blocksize * $column_values[4]);
			}
		}
	}

# hp-ux> > df -k /usr/local/bin
# /software/common       (/dev/vg00/lvol14      ) :  2235975 total allocated Kb
#                                                     408093 free allocated Kb
#                                                    1827882 used allocated Kb
#                                                         81 % allocation used
	if ($dfresult =~ /(\d+).free.allocated.Kb/s) {
		return (int($1) * 1024);
	}

	return (-1);
}	# DiskFreeBytes




sub PsGrep
{
	my ($pattern, $match_mode, $output, $pscmd) = @_;
	my (@column_headers);
	my (@column_values);
	my ($column_number);
	my ($line, $col);
	my ($pid_column) = 0;
	my ($command_pos) = 0;
	my ($command) = 0;
	my ($two_column_format) = 0;
	my (@pids) = ();

	return () unless (defined($pattern));
	$match_mode = "program" unless (defined($match_mode));
	$output = "pids" unless (defined($output));

	if ((! defined($pscmd)) || (! $pscmd)) {
		if ($^O =~ /^(solaris|aix|dec_osf|sco3.2v5.0|svr4)$/) {
			$pscmd = "/bin/ps -e -o pid,args";
		} elsif ($^O =~ /^(linux)$/) {
			$pscmd = "/bin/ps -w -e -o pid,args";
			if (open(PSCOMMAND, "/bin/ps --version |")) {
				$line = <PSCOMMAND>;
				close(PSCOMMAND);
				$pscmd = "/bin/ps auxw" if ($line =~ /version\s1/);
			}
		} elsif ($^O =~ /^(hpux)$/) {
			$pscmd = "/bin/ps -ef";
		} elsif ($^O =~ /^(unixware|openunix)$/) {
			$pscmd = "/usr/bin/ps -ef";
		} elsif ($^O =~ /^(freebsd|netbsd|openbsd|bsdos)$/) {
			$pscmd = "/bin/ps -axw -o pid,command";
		} elsif ($^O =~ /^(darwin|rhapsody)$/) {
			$pscmd = "/bin/ps -auxw";
		} elsif ($^O eq "irix") {
			$pscmd = "/sbin/ps -e -o pid,args";
		} elsif ($^O eq "sunos") {
			$pscmd = "/bin/ps -axw";
		} else {
			$pscmd = "ps -ef";
		}
	}
	$two_column_format = 1 if ($pscmd =~ /pid,/);

	if (open(PSCOMMAND, "$pscmd 2>/dev/null |")) {
		unless (defined($line = <PSCOMMAND>)) {
			close(PSCOMMAND);
			return ();
		}
		if ($two_column_format) {
			$pid_column = 0;
			if ($match_mode eq "program") {
				$pattern = '^' . $pattern . '$'; 
			}
			while (defined($line = <PSCOMMAND>)) {
				chomp($line);
				@column_values = split(' ', $line, 2);
				$command = $column_values[1];
				$command =~ s/://;
				next unless defined($command);

				if ($match_mode ne "full") {
					my (@a) = split(' ', $command);
					$command = $a[0];
					next unless defined($command);
				}
				if ($match_mode eq "program") {
					my (@a) = split(/\//, $command);
					$command = $a[-1];
					next unless defined($command);
					if ($command =~ /^[\(\[](.*)[\)\]]$/) {
						$command = $1;
						next unless defined($command);
					}
				}
				if ($command =~ /${pattern}/) {
					if ($output eq "long") {
						push(@pids, $line);
					} else {
						push(@pids, $column_values[$pid_column]);
					}
				}
			}
		} else {
			$command_pos = index($line, "COMMAND");
			$command_pos = index($line, "CMD") if ($command_pos <= 0);
			unless ($command_pos > 0) {
				close(PSCOMMAND);
				return ();
			}
			@column_headers = split(' ', $line);
			unless (scalar(@column_headers) > 0) {
				close(PSCOMMAND);
				return ();
			}
			$column_number = 0;
			for $col (@column_headers) {
				$pid_column = $column_number if ($col eq "PID");
				$column_number++;
			}
			unless ($pid_column >= 0) {
				close(PSCOMMAND);
				return ();
			}
			if ($match_mode eq "program") {
				$pattern = '^' . $pattern . '$'; 
			}
			while (defined($line = <PSCOMMAND>)) {
				chomp($line);
				@column_values = split(' ', $line);
				$command = substr($line, $command_pos);
				$command =~ s/://;
				if ($match_mode ne "full") {
					my (@a) = split(' ', $command);
					$command = $a[0];
				}
				if ($match_mode eq "program") {
					my (@a) = split(/\//, $command);
					$command = $a[-1];
					if ($command =~ /^[\(\[](.*)[\)\]]$/) {
						$command = $1;
					}
				}
				if ($command =~ /${pattern}/) {
					if ($output eq "long") {
						push(@pids, $line);
					} else {
						push(@pids, $column_values[$pid_column]);
					}
				}
			}
		}
		close(PSCOMMAND);
	}

	return (@pids);
}	# PsGrep




sub SearchPath 
{
	my ($prog, $pathstotryfirst, $pathstoignore) = @_;
	return ("") if ((! defined($prog)) || (! $prog));

	my (@ignore_paths) = ();
	my ($PATH) = $ENV{"PATH"};
	$PATH = "/bin:/usr/bin" if ((! defined($PATH)) || (! $PATH));
	if ((defined($pathstotryfirst)) && ($pathstotryfirst ne "")) {
		$PATH = "$pathstotryfirst:$PATH";
	}
	if ((defined($pathstoignore)) && ($pathstoignore ne "")) {
		@ignore_paths = split(/:/, $pathstoignore);
	}

	my (@path) = split(/:/, $PATH);
	my ($dir, $pathprog, $idir, $ignore);
	for $dir (@path) {
		$ignore = 0;
		for $idir (@ignore_paths) {
			if ($idir eq $dir) {
				$ignore = 1;
			}
		}
		next if ($ignore);
		$pathprog = "$dir/$prog";
		return ($pathprog) if (-x $pathprog);
	}
	return ("");
}	# SearchPath




sub UninstallLog
{
	my (@more_uninst_items) = @_;
	push(@uninstall_items, @more_uninst_items);
}	# UninstallLog




sub Log
{
	my ($v, @args) = @_;
	my ($arg);

	return if (! defined($v));
	my ($tstr) = "";
	$tstr = POSIX::strftime("$log_time_spec ", localtime(time())) if ($log_time_spec ne "");
	$tstr = "" if ($args[0] =~ /^\s/);

	if (scalar(@args)) {
		for $arg (@args) {
			if ($install_log ne "") {
				if ($verbose > $v) {
					print $arg;
				} else {
					print LOG $tstr, $arg;
				}
			} else {
				print $arg if ($verbose > $v);
			}
		}
	}
}	# Log




sub CreateCustomUninstaller
{
	my ($line);
	my (@uninstaller) = ();
	my ($uninstall_item);
	my (@reversed_uninstall_items);
	my ($ltime) = strftime("%Y-%m-%d %H:%M:%S %z", localtime(time()));
	my ($skip) = 0;

	if (($uninstaller ne "") && (open(UNINST, "<$uninstaller"))) {
		@uninstaller = <UNINST>;
		close(UNINST);

		foreach $line (@uninstaller) {
			if ($line eq "#---prologue---#\n") {
				$line .= "#---begin prologue set by $0 on $ltime---#\n";
				$line .= "\t\$OS = \"$OS\";\n";
				$line .= "\t\$SYS = \"$SYS\";\n";
				$line .= "\t\$linux_distro = \"$linux_distro\";\n";
				$line .= "\t\$ftpport = $ftpport;\n";
				$line .= "#---end prologue set by $0 on $ltime---#\n";
			} elsif ($line eq "#---host-specific---#\n") {
				$line .= "#---begin host-specifics set by $0 on $ltime---#\n";
				$line .= "\tIsThereAnFTPServerServing();\n";
				@reversed_uninstall_items = reverse(@uninstall_items);
				for $uninstall_item (@reversed_uninstall_items) {
					chomp($uninstall_item);
					$line .= "\t$uninstall_item\n";
				}
				$line .= "\tIsThereAnFTPServerServing();\n";
				$line .= "\trename(\"$uninstaller\", \"$uninstaller.ran.once.already.pl\");\n";
				$line .= "\tExit(0);\n";
				$line .= "#---end host-specifics set by $0 on $ltime---#\n";
			} elsif ($line =~ /^#---begin/) {
				#
				# Omit existing #---begin / #---end blocks if they exist.
				#
				$line = "";
				$skip = 1;
			} elsif ($line =~ /^#---end/) {
				$line = "";
				$skip = 0;
			} elsif ($skip == 1) {
				$line = "";
			}
		}

		if (open(UNINST, ">$uninstaller")) {
			print UNINST @uninstaller;
			close(UNINST);
			chmod(0700, $uninstaller);
		}
	}
}	# CreateCustomUninstaller




sub Exit
{
	my ($es, @reason) = @_;
	if (scalar(@reason)) {
		Log(0, @reason);
	}
	if ($install_log ne "") {
		if ($confdir ne "") {
			print "\nInstallation log saved as $confdir/install_ncftpd.log.\n";
			UninstallLog("Remove(\"$confdir/install_ncftpd.log\");\n");
			unlink("$confdir/install_ncftpd.log");
			if (move($install_log, "$confdir/install_ncftpd.log")) {
				$install_log = "$confdir/install_ncftpd.log";
			} else {
				print "\nOops, the installation log is at $install_log.\n";
			}
		} else {
			print "\nInstallation log saved as $install_log.\n";
			UninstallLog("Remove(\"$install_log\");\n");
		}
		$install_log = "";
		CreateCustomUninstaller();
		my ($ltime) = strftime("%Y-%m-%d %H:%M:%S %z", localtime(time()));

		# The following line won't be in the log file if the move()
		# above was to a different filesystem. But that's OK.
		#
		if ($tee_output_to_log != 0) {
			print "+++ Log closed at $ltime +++\n";
			alarm(3);
			close(LOG);
			close(STDERR);
			close(STDOUT);
			close(TEE);
		} else {
			print LOG "--- Log closed at $ltime ---\n";
			close(LOG);
		}
	}
	exit($es);
}	# Exit




sub OpenLog
{
	$install_log = "/etc/install_ncftpd.log";

	$| = 1;
	unlink($install_log);
	if (open(LOG, ">>$install_log")) {
		my ($ofh) = select(LOG);
		$| = 1;
		select($ofh);

		my ($ltime) = strftime("%Y-%m-%d %H:%M:%S %z", localtime(time()));

		if ($tee_output_to_log) {
			if (! open(TEE, "| tee -a $install_log")) {
				print STDERR "Warning: could not open a tee. Your log file will be missing any data printed by external programs, but these data will be shown on your screen.\n";
				$tee_output_to_log = 0;
			} else {
				$ofh = select(TEE);
				$| = 1;
				select($ofh);
				if (! open(STDOUT, ">&TEE")) {
					close(TEE);
					print STDERR "Warning: could not redirect STDOUT to TEE. Your log file will be missing any data printed by external programs, but these data will be shown on your screen.\n";
					$tee_output_to_log = 0;
				} elsif (! open(STDERR, ">&TEE")) {
					# Hopefully this never happens because now STDOUT is closed.
					die;
					close(TEE);
					print STDOUT "Warning: could not redirect STDERR to TEE. Your log file will be missing any data printed by external programs, but these data will be shown on your screen.\n";
					$tee_output_to_log = 0;
				} else {
					# TEE and log opened OK
					$| = 1;
					print LOG "+++ Log created at $ltime +++\n";	# Use +++ for TEE
				}
			}
		} else {
			print LOG "--- Log created at $ltime ---\n";
		}
	} else {
		$install_log = "";
	}
}	# OpenLog




sub OpenPager
{
	if (-t STDOUT) {
		if ((defined($ENV{"PAGER"})) && (-x $ENV{"PAGER"})) {
			$pager = $ENV{"PAGER"};
		} elsif (-x "/usr/bin/less") {
			$pager = "/usr/bin/less";
		} elsif (-x "/usr/bin/more") {
			$pager = "/usr/bin/more";
		} elsif (-x "/bin/more") {
			$pager = "/bin/more";
		} elsif (-x "/usr/bin/pg") {
			$pager = "/usr/bin/pg";
		}
		if ($pager ne "") {
			$pager_pid = open(STDOUT, "| $pager");
			if ($pager_pid > 0) {
				$SIG{PIPE} = 'IGNORE';
				$| = 1;
			}
		}
	} else {
		$| = 1;
	}
}	# OpenPager




sub SetOSVar
{
	my ($os)				= "";
	my ($os_v)				= "";
	my ($os_r)				= "";
	my ($archp)				= "";
	my (@os_r)				= (0,0,0);
	my (@os_v)				= (0,0,0);
	my ($arch)				= "";
	my ($line);

	$host = lc( `uname -n 2>/dev/null` );
	chomp($host);
	Log(2, "host = $host\n");

	$os = lc( `uname -s 2>/dev/null` );
	chomp($os);
	Log(2, "os = $os\n");

	$os_v = lc( `uname -v 2>/dev/null` );
	chomp($os_v);
	if ($os_v =~ /(\d+(\.\d+)*)/) {
		# Get just the version digit string
		$os_v = $1;
		@os_v = split(/\./, $os_v);
	}
	Log(2, "os_v = $os_v\n");

	$os_r = lc( `uname -r 2>/dev/null` );
	chomp($os_r);
	if ($os_r =~ /(\d+(\.\d+)*)/) {
		# Get just the version digit string
		$os_r = $1;
		@os_r = split(/\./, $os_r);
	}
	Log(2, "os_r = $os_r\n");

	$arch = lc( `uname -m 2>/dev/null` );
	chomp($arch);
	Log(2, "arch = $arch\n");

	$archp = lc( `uname -p 2>/dev/null` );
	chomp($archp);
	Log(2, "archp = $archp\n");

	if ($os eq "osf1") {
		if (($os_r[0] == 3) || ($os_r[0] == 4)) {
			$OS = "digitalunix${os_r}-$arch";
			$SYS = "digitalunix";
		} else {
			$OS = "tru64unix${os_r}-$arch";
			$SYS = "tru64unix";
		}
	} elsif ($os eq "aix") {
		$OS = "aix${os_v}.${os_r}";
		$SYS = "aix";
	} elsif ($os eq "irix") {
		$OS = "irix${os_r}";
		$SYS = "irix";
	} elsif ($os eq "irix64") {
		$OS = "irix64_${os_r}";
		$SYS = "irix64";
	} elsif ($os eq "hp-ux") {
		$OS = "hpux${os_r}";
		$SYS = "hpux";
	} elsif ($os eq "freebsd") {
		$OS = "freebsd${os_r}-$arch";
		$SYS = "freebsd";
	} elsif ($os eq "netbsd") {
		$OS = "netbsd${os_r}-$arch";
		$SYS = "netbsd";
	} elsif ($os eq "openbsd") {
		$OS = "openbsd${os_r}-$arch";
		$SYS = "openbsd";
	} elsif ($os =~ "^sco") {
		$OS = "scosv";
		$SYS = "sco";
	} elsif ($os =~ "^dynix") {
		$OS = "dynixptx${os_v}";
		$SYS = "dynixptx";
	} elsif ($os eq "linux") {
		my ($libc) = "";

		$arch = "x86" if ($arch =~ /86$/);
	
		my ($tmpfile, $rnd);
		$rnd = int(rand(10000));
		$tmpfile = $0;
		$tmpfile =~ s/^.*\///;
		$tmpfile = "$tmpfile.$$.$rnd";
		if ((-d "/root") && ($> == 0)) {
			$tmpfile = "/root/$tmpfile";
		} else {
			# FIXME: /tmp race
			#
			$tmpfile = "/tmp/$tmpfile";
		}
		Log(2, "$tmpfile\n");
		unlink("$tmpfile", "$tmpfile.c", "$tmpfile.o");
		Exit(1, "Could not open $tmpfile.c: $!\n") unless open(TMPCFILE, "> $tmpfile.c");
		print TMPCFILE <<EOF;
#include <stdio.h>
#include <stdlib.h>
#ifdef HAVE_GNU_LIBC_VERSION_H
#	include <gnu/libc-version.h>
#else
extern const char *gnu_get_libc_version(void);
extern const char *gnu_get_libc_release(void);
#endif

int main(void)
{
	const char *ver = gnu_get_libc_version();
	const char *rel = gnu_get_libc_release();

	fprintf(stdout, "glibc%s\\n", ver);
	exit(0);
}
EOF
		Log(2, "created $tmpfile.c\n");
		close(TMPCFILE);

		my ($CC);
		if (-f "/usr/bin/gcc") {
			$CC = "/usr/bin/gcc";
		} elsif (-f "/usr/bin/cc") {
			$CC = "/usr/bin/cc";
		} else {
			$CC = "cc";
		}

		if (-f "/usr/include/gnu/libc-version.h") {
			system("$CC", "-DHAVE_GNU_LIBC_VERSION_H", "$tmpfile.c", "-o", "$tmpfile") if ($CC ne "cc");
		} else {
			system("$CC", "$tmpfile.c", "-o", "$tmpfile") if ($CC ne "cc");
		}
		if (-f "$tmpfile") {
			Log(2, "compiled $tmpfile.c to $tmpfile\n");
			$libc=`$tmpfile`;
			chomp($libc);
			Log(2, "cc_libc = $libc\n");
		}
		unlink("$tmpfile", "$tmpfile.c", "$tmpfile.o");

		$OS = "linux-$arch";
		$SYS = "linux";

		my ($ldd_libc) = `/usr/bin/ldd --version 2>/dev/null`;
		# ldd (GNU libc) 2.12
		if ($ldd_libc =~ /(GNU libc|GLIBC)\D+(\d+\.\d+(\.\d+)?)/i) {
			$ldd_libc = "glibc" . $2;
			Log(2, "ldd_libc = $ldd_libc\n");
		} else {
			$ldd_libc = "";
		}

		# ldd /bin/echo
		#       linux-vdso.so.1 =>  (0x00007fff172f2000)
		#       libc.so.6 => /lib/x86_64-linux-gnu/libc.so.6 (0x00007ff53a7eb000)
		#       /lib64/ld-linux-x86-64.so.2 (0x00007ff53abbe000)

		my ($libc_path) = "";
		my ($libc_dir) = "";
		if (open(LDD, "ldd /bin/echo |")) {
			while (defined(my $line = <LDD>)) {
				if ($line =~ /^\s*libc.so[^\/]+(\/\S+)/) {
					$libc_path = $1;
				}
			}
			close(LDD);
		}
		if ($libc_path =~ /(.+)\//) {
			$libc_dir = $1;
		}

		my ($f_libc) = "";
		my ($nf_libc) = 0;
		my @lib_libc = ();
		push(@lib_libc, <$libc_dir/libc-*.*.*.so>) if ($libc_dir ne "");
		push(@lib_libc, <$libc_dir/libc-*.*.so>) if ($libc_dir ne "");
		push(@lib_libc, </lib64/libc-*.*.*.so>);
		push(@lib_libc, </lib64/libc-*.*.so>);
		push(@lib_libc, </lib/libc-*.*.so>);
		push(@lib_libc, </lib/libc-*.*.*.so>);
		my ($highest_libc_ver) = 0;
		my ($tmp_libc_ver) = 0;
		my ($libc_detection_mode) = "specific file query";

		for my $li (@lib_libc) {
			next unless (-f $li);
			if ($li =~ /libc-((\d+)\.(\d+)(\.(\d+)?))/) {
				Log(2, "f_libc found glibc" . $1 . "\n");
				$nf_libc++;
				$tmp_libc_ver = defined($5) ? $5 : 0; 
				$tmp_libc_ver += $2 * 10000 + $3 * 100;
				if ($highest_libc_ver < $tmp_libc_ver) {
					$highest_libc_ver = $tmp_libc_ver;
					$f_libc = "glibc" . $1;
				}       
			}       
		}       
		Log(2, "f_libc best was $f_libc\n") if (($f_libc ne "") && ($nf_libc >= 2));

		if ($libc =~ /^glibc/) {
			$libc_detection_mode = "CC";
			if ($libc =~ /^(glibc\D*d+\.\d+)/) {
				$libc = $1;
			}
		} elsif ($ldd_libc =~ /glibc\D*(\d+.\d+)/) {
			$libc_detection_mode = "ldd";
			$libc = "glibc" . $1;
		} elsif ($f_libc =~ /libc\D*(\d+.\d+)/) {
			$libc_detection_mode = "inspection of /lib and /lib64";
			$libc = "glibc" . $1;
		} elsif (-f "/lib/libc.so.6.1") {
			$libc = "glibc2.0";
		} elsif (-f "/lib/libc.so.5") {
			$libc = "libc5";
		} else {
			$libc_detection_mode = "guess of last resort";
			$libc = "glibc2.7";
		}
		$linux_libc = $libc;
		Log(1, "Found linux_libc = \"$linux_libc\" detected via $libc_detection_mode\n");

		$linux_distro = "unknown";
		if (-f "/etc/SuSE-release") {
			$linux_distro = "SuSE";
			$linux_distro_release_file = "/etc/SuSE-release";
		} elsif (-f "/etc/turbolinux-release") {
			$linux_distro = "TurboLinux";
			$linux_distro_release_file = "/etc/turbolinux-release";
		} elsif (-f "/etc/slackware-version") {
			$linux_distro = "Slackware";
			$linux_distro_release_file = "/etc/slackware-version";
		} elsif (-f "/etc/slackware-release") {
			$linux_distro = "Slackware";
			$linux_distro_release_file = "/etc/slackware-release";
		} elsif (-f "/etc/mandriva-release") {
			$linux_distro = "Mandriva";
			$linux_distro_release_file = "/etc/mandriva-release";
		} elsif (-f "/etc/mandrake-release") {
			$linux_distro = "Mandrake";
			$linux_distro_release_file = "/etc/mandrake-release";
		} elsif (-f "/etc/cobalt-release") {
			$linux_distro = "Cobalt";
			$linux_distro_release_file = "/etc/cobalt-release";
		} elsif (-f "/etc/debian_version") {
			# Includes Ubuntu, Mint
			$linux_distro_class = "Debian";
			$linux_distro = "Debian";	# May be overridden down below by /etc/lsb-release
			$linux_distro_release_file = "/etc/debian_version";
		} elsif (-f "/etc/conectiva-release") {
			$linux_distro = "Conectiva";
			$linux_distro_release_file = "/etc/conectiva-release";
		} elsif (open(RH, "</etc/redhat-release")) {
			$line = <RH>;
			close(RH);
			$linux_distro_class = "Red Hat";
			$linux_distro = "Red Hat variant";
			if ($line =~ /^Red\ Hat/) {
				$linux_distro = "Red Hat";
				if ($line =~ /^Red Hat\D+([0-9.]+)/i) {
					$linux_distro_version = $1;
				}
				$linux_distro_release_file = "/etc/redhat-release";
			} elsif ($line =~ /^Fedora/) {
				$linux_distro = "Fedora Core";
				if ($line =~ /^Fedora release\D+([0-9.]+)/i) {
					$linux_distro_version = $1;
				}
				$linux_distro_release_file = "/etc/redhat-release";
			} elsif ($line =~ /^CentOS/) {
				if ($line =~ /^CentOS release\D+([0-9.]+)/i) {
					$linux_distro_version = $1;
				}
				$linux_distro = "CentOS";
				$linux_distro_release_file = "/etc/redhat-release";
			} elsif ($line =~ /^LinuxPPC/) {
				$linux_distro = "LinuxPPC";
			}
		} elsif (open(ETCISSUE, "</etc/issue")) {
			while (defined($line = <ETCISSUE>)) {
				if ($line =~ /(Caldera|Cobalt|Debian|LinuxPPC|OpenSuSE|SuSE|TurboLinux|Slackware|Mandrake|Mandriva|Conectiva|Red\ Hat|Fedora(\ Core)?|Ubuntu|Mint)/) {
					$linux_distro = $1;
					last;
				} elsif ($line =~ /S\.u\.S\.E/i) {
					$linux_distro = "SuSE";
				}
			}
			close(ETCISSUE);
		}
		if (($linux_distro eq "unknown") || ($linux_distro eq "Debian")) {
			if (open(LSB, "</etc/lsb-release")) {
				my ($maybe_linux_distro_version) = "";
				my ($maybe_linux_distro) = "";
				while (defined($line = <LSB>)) {
					chomp($line);
					if ($line =~ /^\s*DISTRIB_ID\s*=\s*(\S.*)/) {
						$maybe_linux_distro = $1;
					} elsif ($line =~ /^\s*DISTRIB_RELEASE\s*=\s*(\S.*)/) {
						$maybe_linux_distro_version = $1;
					}
				}
				close(LSB);
				if ($maybe_linux_distro ne "") {
					$linux_distro = $maybe_linux_distro;
					if ($maybe_linux_distro_version ne "") {
						$linux_distro_version = $maybe_linux_distro_version;
					}
				}
			}
		}
		if ($linux_distro eq "unknown") {
			my (@unknown_releases) = glob("/etc/*-release");
			if (scalar(@unknown_releases) > 0) {
				$linux_distro_release_file = $unknown_releases[0];
				$linux_distro = $unknown_releases[0];
				$linux_distro =~ s/\-release$//;
				$linux_distro =~ s-^.*/--;
				Log(1, "Detected unknown linux_distro=$linux_distro.\n");
			}
		}
		if ($linux_distro_release_file ne "") {
			if (open(RH, "< $linux_distro_release_file")) {
				while (defined($line = <RH>)) {
					if ($line =~ /^\s*VERSION\s*=\s*(\S*)/i) {
						$linux_distro_version = $1;
					} elsif ($line =~ /(release|version)\s*(\S*)/i) {
						$linux_distro_version = $2;
					}
				}
				close(RH);
				if ($linux_distro_version =~ /(\d+(\.\d+)?)/) {
					$linux_distro_version_num = $1;
				}
			}
		}
		if ($linux_distro_version ne "") {
			Log(1, "Linux Distribution: $linux_distro (Version $linux_distro_version)\n");
		} else {
			Log(1, "Linux Distribution: $linux_distro\n");
		}
		Log(1, "Linux LIBC Version: $linux_libc\n");
	} elsif ($os eq "bsd/os") {
		$OS = "bsdos${os_r}";
		$SYS = "bsdos";
	} elsif ($os eq "ultrix") {
		$OS = "ultrix-${arch}";
		$SYS = "ultrix";
	} elsif ($os eq "openunix") {
		$OS = "openunix${os_v}";
		$SYS = "openunix";
	} elsif ($os eq "unixware") {
		$OS = "unixware${os_v}";
		$SYS = "unixware";
	} elsif (($os eq "darwin") || ($os eq "rhapsody") || ($os =~ /^macos/))  {
		$SYS = "macosx";
		$OS = "macosx";
		if (open(PLIST, "< /System/Library/CoreServices/SystemVersion.plist")) {
			my ($doc) = "";
			my ($line) = "";
			while (defined($line = <PLIST>)) { $doc .= $line; }
			close(PLIST);
			$doc =~ s/\s+//gs;
			if ($doc =~ /<key>ProductVersion<\/key><string>([^<]+)<\/string>/i) {
				$os_r = $1;
				$OS = "macosx" . $os_r;
			}
		}
	} elsif ($os eq "sunos") {
		$arch = "sparc" if (! $arch);
		$archp = "$arch" if (! $archp);
		if (($os_r[0] == 5) && ($os_r[1] >= 7)) {
			$os_r =~ s/^\d+\.//;
			$OS = "solaris${os_r}-$archp";
			$SYS = "solaris";
		} elsif (($os_r[0] == 5) && ($os_r[1] < 7)) {
			$os_r =~ s/^5/2/;
			$OS = "solaris${os_r}-$archp";
			$SYS = "solaris";
		} elsif ($os_r[0] == 4) {
			$OS = "sunos${os_r}-$archp";
			$SYS = "sunos";
		} else {
			$OS = "solaris${os_r}-$archp";
			$SYS = "solaris";
		}
	} else {
		$OS = "$os";
		$SYS = "$os";
	}

	Log(1, "OS = $OS ; SYS = $SYS\n");
}	# SetOSVar




sub SetOSDefaults
{
	# Set log dir to OS defaults, unless we already have a /var/log.
	#
	if ((! -d $logdir) || (-l $logdir)) {
		if ($SYS eq "aix") {
			$logdir = "/var/adm";
		} elsif ($SYS =~ /^irix/) {
			$logdir = "/var/adm";
		} elsif ($SYS eq "hpux") {
			$logdir = "/var/spool/log";
		} elsif ($SYS eq "sco") {
			$logdir = "/var/adm/log";
		} elsif ($SYS eq "unixware") {
			$logdir = "/var/adm/log";
		} elsif ($SYS eq "openunix") {
			$logdir = "/var/adm/log";
		}
	}
	$startup_error_log = "$logdir/ncftpd/startup_errors";
}	# SetOSDefaults




sub VerifyOSPrerequisites
{
	# We could remove this check, as it is now well past the EOL of Mac OS X 10.2.
	if (($OS =~ /^macosx10\.[012](\D|$)/) && (! -x "/usr/bin/ftp")) {
		Exit(1, "NcFTPd for Mac OS X requires the BSD Subsystem.  This is located on the Mac OS X installation disc.\n");
	}
}	# VerifyOSPrerequisites




sub SeeIfNcFTPdIsBinaryCompatible
{
	my ($f) = @_;
	my ($osdir, $ver, $n);

	$osdir = $f;
	$osdir =~ s/\/[^\/]+$//;
	$n = "$osdir/ncftpd";
	Log(1, "Try to run \"$n -b\".\n");
	$ver = "";
	$ver = `$n -b 2>&1` || "";
	chomp($ver);
	# i.e: "NcFTPd 2.7.1/849"
	if ($ver =~ /^NcFTPd\s([1-9]\.\d+\.\d+)\/\d+/) {
		Log(2, "  result = [$ver]\n");
		$ver = $1;
		$install_version = $ver;
		$install_src_root = $osdir;
		$binary_compat_err = "";
		return (1);
	} else {
		Log(1, "  result = [$ver]\n");
		$binary_compat_err = $ver;
		if ($SYS eq "openbsd") {
			unlink("$n.core");
		}
	}
	return (0);
}	# SeeIfNcFTPdIsBinaryCompatible




sub VerifyLocation
{
	my ($wd) = $0;
	my (@os_ncftpd);
	my ($f);

	$wd =~ s/\/*install_ncftpd.pl$//;	
	$wd = "." if (! $wd);
	if ($wd ne ".") {
		chdir($wd) or Exit(1, "Could not change directory to \"$wd\": $!\n");
	}
	$wd = `pwd 2>/dev/null`;
	chomp($wd);
	$wd = "." if (! $wd);
	Exit(1, "Could not find install_ncftpd.pl in directory \"$wd\"!\n") if (! -f "install_ncftpd.pl");
	$install_src_root = $wd;

	$install_version = "";
	@os_ncftpd = glob("$wd/*/ncftpd");
	if (scalar(@os_ncftpd) > 0) {
		foreach $f (@os_ncftpd) {
			$f =~ s-/ncftpd$--;
			$f =~ s-^.*/--;
		}
		my (@skipped_os_ncftpd) = ();

		sub version_int {
			my ($sver) = $_[0] || "0.0";
			if ($sver =~ /(\d+)\D+(\d+)\D+(\d+)/) {
				return (10000 * int($1) + 100 * int($2) + int($3));
			} elsif ($sver =~ /(\d+)\D+(\d+)/) {
				return (10000 * int($1) + 100 * int($2));
			} elsif ($sver =~ /(\d+)/) {
				return (10000 * int($1));
			}
			return 0;
		}

		if ($SYS eq "linux") {
			my (@os_ncftpd2) = ();
			foreach $f (@os_ncftpd) {
				my ($aver, $sver);
				$sver = $linux_libc;
				$sver =~ s/libc5/0/;
				$sver = version_int($sver);

				$aver = $f;
				$aver =~ s/libc5/0/;
				$aver = version_int($aver);

				if ($aver <= $sver) {
					push(@os_ncftpd2, $f);
				} else {
					push(@skipped_os_ncftpd, $f);
				}
			}
			@os_ncftpd = @os_ncftpd2;
			@os_ncftpd = sort {
				if ($a eq $linux_libc) {
					return (-1);
				} elsif ($b eq $linux_libc) {
					return (1);
				} else {
					my ($aver, $bver);

					$aver = $a;
					$aver =~ s/libc5/0/;
					$aver = version_int($aver);

					$bver = $b;
					$bver =~ s/libc5/0/;
					$bver = version_int($bver);

					$bver <=> $aver;
				}
			} @os_ncftpd;
		} else {
			@os_ncftpd = sort {
				if ($a eq $OS) {
					return (-1);
				} elsif ($b eq $OS) {
					return (1);
				} else {
					my ($aver, $bver);

					$aver = version_int($a);
					$bver = version_int($b);

					$bver <=> $aver;
				}
			} @os_ncftpd;
		}
		if (scalar(@skipped_os_ncftpd) > 0) {
			Log(2, "List of platform subdirectories that will not be used as they appear to be aimed at newer systems than this one:\n");
			for $f (@skipped_os_ncftpd) {
				Log(2, "  " . $f . "\n");
			}
		}
		Log(2, "List of platform subdirectories found that will be checked:\n");
		for $f (@os_ncftpd) {
			Log(2, "  " . $f . "\n");
		}
		for $f (@os_ncftpd) {
			last if (SeeIfNcFTPdIsBinaryCompatible("$wd/$f/ncftpd"));
		}
		if ($install_version eq "") {
			Log(0, "\nERROR: The package you downloaded will not run on this platform.\n");
			Log(0, "       Check http://www.ncftp.com/download/ for a package for $OS.\n");
			Exit(1);
		}
	} elsif (! -f "$wd/ncftpd") {
		Log(0, "\nERROR: There is no \"ncftpd\" program in the same directory as this script,\n$wd/install_ncftpd.pl.\n");
		Exit(1);
	} else {
		if (! SeeIfNcFTPdIsBinaryCompatible("$wd/ncftpd")) {
			if ($SYS eq "openbsd") {
				# Look for a message similar to this:
				# /tmp/ncftpd-2.8.2/ncftpd: can't load library 'libc.so.30.1'
				my ($libc_so_link);
				my (@libc_sos, $libc_so);
				if (($binary_compat_err =~ /can.?t\ load\ library.*(libc\.so\.[0-9\.]+)/) || ($binary_compat_err =~ /(libc\.so\.[^:]+): No\ such\ file\ or\ directory/)) {
					# Try to work around this by symlinking
					# the version of libc.so we are using
					# to the latest version on this host.
					#
					$libc_so_link = "/usr/lib/" . $1;
					@libc_sos = glob("/usr/lib/libc.so.*");
					$libc_so = $libc_sos[0];
					if ((Symlink($libc_so, $libc_so_link, "do not overwrite")) && (! SeeIfNcFTPdIsBinaryCompatible("$wd/ncftpd"))) {
						# Remove the link we made first,
						# since it didn't work.
						#
						unlink($libc_so_link);
						Log(0, "\nERROR: The package you downloaded will not run on this platform.\n");
						Log(0, "       Check http://www.ncftp.com/download/ for a package for $OS.\n");
						Exit(1);
					}
				} else {
					Log(0, "\nERROR: The package you downloaded will not run on this platform.\n");
					Log(0, "       Check http://www.ncftp.com/download/ for a package for $OS.\n");
					Exit(1);
				}
			} elsif ($SYS eq "macosx") {
				my $archp = lc( `uname -p 2>/dev/null` );
				chomp($archp);
				Log(1, "The binaries do not run on this arch (\"$archp\").\n");
				if ($archp eq "powerpc") {
					Log(0, "\nERROR: This package is only for Mac OS X systems with Intel CPUs.\n");
					Log(0, "       Check http://www.ncftp.com/download/ for the PowerPC version.\n");
					Exit(1);
				} else {
					Log(0, "\nERROR: This package is only for Mac OS X systems with PowerPC CPUs.\n");
					Log(0, "       Check http://www.ncftp.com/download/ for the Intel version.\n");
					Exit(1);
				}
			} else {
				Log(0, "\nERROR: The package you downloaded will not run on this platform.\n");
				Log(0, "       Check http://www.ncftp.com/download/ for a package for $OS.\n");
				Exit(1);
			}
		}
	}

	Log(1, "Install source root is $install_src_root.\n");
	if ($install_src_root eq $wd) {
		return "$install_src_root/extra:$install_src_root";
	} else {
		return "$install_src_root/extra:$install_src_root:$wd";
	}
}	# VerifyLocation




sub VerifyPackageContents
{
	my ($f, $path);
	my (@required_files) = (
		"LICENSE",
		"conf/general.cf-dist",
		"conf/domain.cf-dist",
		"ncftpd",
		"ncftpd_edquota",
		"ncftpd_passwd",
		"ncftpd_repquota",
		"ncftpd_spy",
		"extra/ncftpd.init",
		"extra/ncftpd.sh",
	);
	my (@required_dirs) = (
		"extra",
	);
	for $f (@required_files) {
		$path = "$install_src_root/$f";
		if (! -f $path) {
			Log(0, "Missing required file for installation: $path.\n") if ($force_install);
			Exit(1, "Missing required file for installation: $path.\n") if (! $force_install);
		}
	}
	for $f (@required_dirs) {
		$path = "$install_src_root/$f";
		if (! -d $path) {
			Log(0, "Missing required directory for installation: $path.\n") if ($force_install);
			Exit(1, "Missing required directory for installation: $path.\n") if (! $force_install);
		}
	}
}	# VerifyPackageContents




sub VerifyFreeSpace
{
	my ($bytesfree, $kbytesfree, $duresult, $kbytesneeded);

	$bytesfree = DiskFreeBytes($eprefix);
	if ($bytesfree == (-1)) {
		Log(0, "Could not determine free space on $prefix.\n");
		return;
	}
	$kbytesfree = $bytesfree / 1024;
	Log(1, sprintf("$eprefix kbytes free: %.2f\n", $kbytesfree));

	if ($^O eq "sunos") {
		$duresult = `cd $install_src_root && du -s $install_src_root 2>&1`;
	} else {
		$duresult = `cd $install_src_root && du -s -k $install_src_root 2>&1`;
	}
	chomp($duresult);
	if ($duresult !~ /^(\d+)/) {
		Log(0, "Could not determine disk usage for $install_src_root.\n");
		Log(0, "Result was [$duresult]\n");
		return;
	}
	$kbytesneeded = int($1) + 100;
	Log(1, sprintf("$install_src_root kbytes usage: %.2f\n", $kbytesneeded));
	if ($kbytesneeded > $kbytesfree) {
		Log(0, sprintf("ERROR: %u kB of free space is required on $prefix,\n", $kbytesneeded));
		Log(0, sprintf("       but only %.2f kB available.\n", $kbytesfree));
		Exit(1) if (! $force_install);
	}
}	# VerifyFreeSpace




sub SetPATHEnvironmentVar
{
	my ($path_prefix) = @_;
	my (@path_prefix_list) = ();
	my ($HOME) = $ENV{"HOME"};
	my ($newPATH) = "";
	my (@origPATHs) = split(/:/, $ENV{"PATH"});
	my (@paths) = ();
	my (@pre_paths);
	my ($dir);
	my (%found_paths);

	$HOME = "/" unless (defined($HOME));
	$ENV{"PATH"} = "/bin:/usr/bin" if (! defined($ENV{"PATH"}));
	@origPATHs = split(/:/, $ENV{"PATH"});
	$extra_paths = $path_prefix if (defined($path_prefix));
	@path_prefix_list = split(/:/, $path_prefix) if defined($path_prefix);
	if ($> != 0) {
		#
		# PATH for unprivileged user.
		#
		@paths = (
			@path_prefix_list,
			".",
			"$HOME/$OS/bin",
			"$HOME/bin/powerpc-apple-macos",
			"$HOME/bin",
			"$HOME/sh",
			"/opt/SUNWspro/bin",
			"/opt/ansic/bin",
			"/usr/vac/bin",
			"/usr/ccs/bin",
			"/udk/usr/ccs/bin",
			"/usr/local/bin",
			"/usr/local/sbin",
			"/bin",
			"/usr/bin",
			"/usr/contrib/bin",
			"/usr/X11R6/bin",
			"/usr/bin/X11R6",
			"/usr/X11/bin",
			"/usr/bin/X11",
			"/usr/sbin",
			"/sbin",
			"/etc",
			"/usr/ucb",
			"/usr/bsd",
			"/usr/freeware/bin",
			"/usr/openwin/bin",
			@origPATHs
		);
	} else {
		#
		# PATH for Superuser.
		#

		my (@pre_paths) = ();
		if ($HOME ne "/") {
			@pre_paths = (
				"$HOME/bin/powerpc-apple-macos",
				"$HOME/bin",
				"$HOME/sh",
				);
		}

		@paths = (
			@path_prefix_list,
			@pre_paths,
			"/bin",
			"/usr/bin",
			"/usr/sbin",
			"/sbin",
			"/usr/local/sbin",
			"/etc",
			"/usr/local/bin",
			"/usr/ucb",
			"/usr/bsd",
			"/opt/SUNWspro/bin",
			"/opt/ansic/bin",
			"/usr/vac/bin",
			"/usr/ccs/bin",
			"/udk/usr/ccs/bin",
			"/usr/contrib/bin",
			"/usr/X11R6/bin",
			"/usr/bin/X11R6",
			"/usr/X11/bin",
			"/usr/bin/X11",
			"/usr/freeware/bin",
			"/usr/openwin/bin",
			@origPATHs
		);
	}

	for $dir (@paths) {
		if (-d $dir) {
			if (! defined($found_paths{$dir})) {
				$found_paths{$dir} = 1;
				if ($newPATH eq "") {
					$newPATH = $dir;
				} else {
					$newPATH .= ":" . $dir;
				}
			}
		}
	}

	Log(1, "Setting PATH to \"$newPATH\".\n");
	$ENV{"PATH"} = $newPATH;
}	# SetPATHEnvironmentVar




sub LookupFTPServiceName
{
	return if ($ftpport == 21);
	return if ((defined($ftpservicename)) && ($ftpservicename ne "") && ($ftpservicename ne "ftp"));

	my ($ftpdataport) = $ftpport - 1;
	my ($name,$aliases,$port,$proto) = getservbyport($ftpport, "tcp");
	if (defined($name)) {
		$ftpservicename = $name;
	} else {
		($name,$aliases,$port,$proto) = getservbyname("ftp-$port", "tcp");
		if (defined($port) && defined($name) && ($port == $ftpport)) {
			$ftpservicename = $name;
		} elsif ((-f "/etc/services") && (open(ETCSERVICES, ">>/etc/services"))) {
			print ETCSERVICES "\n#\n# Added by install_ncftpd.pl\n#\n";
			print ETCSERVICES "ftp-$ftpport-data\t$ftpdataport/tcp\n";
			print ETCSERVICES "ftp-$ftpport\t$ftpport/tcp\t\t\t# Alternate FTP Service port\n";
			Log(0, "Added \"ftp-$ftpport\" to /etc/services.\n");
			close(ETCSERVICES);
			UninstallLog("RemoveFromEtcServices(\"ftp-$ftpport-data\", \"ftp-$ftpport\");\n");
		}
	}
}	# LookupFTPServiceName



sub StopAndUnloadLaunchdService
{
	my ($srv, $plist, $procname) = @_;
	my ($tprocs) = 0;
	if ($srv =~ /^([^:]+)\:([^:]+)\:([^:]+)$/) {
		$srv = $1;
		$plist = $2;
		$procname = $3;
	} else {
		$plist = "$srv.plist" if (! defined($plist));
		$procname = "unknown_process" if (! defined($procname));
	}
	$plist = "$launchd_jobs_dir/$plist" unless ($plist =~ /^\//);
	#
	# Unfortunately Launchd does not give us the
	# path to the plist for a given job. This could
	# cause problems if the job we want to disable
	# does not have its plist in /System/Library/LaunchDaemons.
	#
	if (IsServiceCurrentlyRunningFromLaunchd($srv)) {
		# Loaded and running.
		Log(1, "Stopping $srv in launchd.\n");
		if (RunProgramOrExit($launchctl, "stop", $srv)) {
			# In case launchctl returns immediately rather than waiting for the processes to die.
			#
			# We probably don't need to do this check, as it appears to wait,
			# which means we don't need to know the procname, which means we
			# don't really need that as a parameter.
			#
			for (my $i = 0; $i <= 5; $i ++) {
				Log(1, "  ...waiting $i for $procname to exit...\n") if ($i > 0);
				sleep($i);
				my (@liveprocs) = PsGrep($procname, "program", "pids");
				$tprocs = scalar(@liveprocs);
				last if ($tprocs == 0);
			}
			Log(0, "ERROR: $tprocs $procname FTP server process(es) appear to be still running, even after we tried stopping $srv in launchd.\n") if ($tprocs != 0);

			# Stopped.
			UninstallLog("system(\"$launchctl\", \"start\", \"$srv\");\n");
			Log(0, "Disabling (unloading) $srv in launchd.\n");
			if (RunProgramOrExit($launchctl, "unload", "-w", $plist)) {
				# Disabled.
				UninstallLog("system(\"$launchctl\", \"load\", \"-w\", \"$plist\");\n");
			} else {
				Log(0, "Could not unload $srv from launchd. This may cause problems the next time the system boots.\n");
			}
		} else {
			Log(0, "Could not stop $srv in launchd. This may cause problems the next time the system boots.\n");
		}
	} elsif (RunProgramOrExit($launchctl, "list", $srv)) {
		# Loaded, but not running.
		Log(0, "Disabling (unloading) $srv in launchd.\n");
		if (RunProgramOrExit($launchctl, "unload", "-w", $plist)) {
			UninstallLog("system(\"$launchctl\", \"load\", \"-w\", \"$plist\");\n");
		} else {
			Log(0, "Could not disable $srv in launchd. This may cause problems the next time the launch boots.\n");
		}
	}
}	# StopAndUnloadLaunchdService




sub CheckStartupSystem
{
	$systemctl = SearchPath("systemctl");
	if ($use_systemd < 0) {
		$use_systemd = ($systemctl ne "") ? 1 : 0;
		Log(1, "Will try to use systemd.\n") if ($use_systemd);
	}

	$initctl = SearchPath("initctl");
	if (($use_upstart < 0) && ($use_systemd <= 0)) {
		$use_upstart = ($initctl ne "") ? 1 : 0;
		Log(1, "Will try to use upstart.\n") if ($use_upstart);
	}

	$launchctl = SearchPath("launchctl");
	if ($use_launchd < 0) {
		$use_launchd = ($launchctl ne "") ? 1 : 0;
		Log(1, "Will try to use launchd.\n") if ($use_launchd);
	}

	if ($systemctl ne "") {
		# Get a list of all systemd services that could be enabled (and running).
		my (%available_systemd_services) = ();
		if (open(F, "systemctl list-unit-files --type=service |")) {
			while (defined(my $line = <F>)) {
				if ($line =~ /^\s*(\S+\.service)/) {
					my ($srv) = $1;
					$available_systemd_services{$srv} = 1;
				}
			}
			close(F);
		}

		# Get the list of other known services that would conflict with us.
		my (@srvs) = ();
		if (open(F, "$install_src_root/extra/ncftpd.service")) {
			while (defined(my $line = <F>)) {
				if ($line =~ /^Conflicts\s*=\s*(\S.*\S)*/i) {
					@srvs = split(/\s+/, $1);
				}
			}
			close(F);
		}

		# See if those services are known to this system.
		@systemd_competing_services = ();
		foreach my $srv (@srvs) {
			if (exists($available_systemd_services{$srv})) {
				push(@systemd_competing_services, $srv);
			}
		}

		# Block services that are present here and are known conflicts.
		foreach my $srv (@systemd_competing_services) {
			if (RunProgramOrExit($systemctl, "is-enabled", $srv)) {
				Log(0, "Disabling $srv in systemd.\n");
				if (RunProgramOrExit($systemctl, "--quiet", "disable", $srv)) {
					# Disabled, but may still be running ("is-active").
					UninstallLog("system(\"$systemctl\", \"enable\", \"$srv\");\n");
				} else {
					Log(0, "Could not disable $srv in systemd. This may cause problems the next time the system boots.\n");
				}
			}
		}
	}

	if ($initctl ne "") {
		# Get a list of all upstart services that could be enabled (and running).
		my (%available_upstart_services) = ();
		if (open(F, "initctl list |")) {
			while (defined(my $line = <F>)) {
				if ($line =~ /^(\S.*)\s.*\sprocess (\d+)/) {
					my ($srv) = $1;
					$available_upstart_services{$srv} = $2;	# running
				} elsif ($line =~ /^(\S.*)\s/) {
					my ($srv) = $1;
					$available_upstart_services{$srv} = 0;	# not running
				}
			}
			close(F);
		}

		# Get the list of other known services that would conflict with us.
		#
		# NOTE: we actually get this list from our systemd file.
		#
		my (@srvs) = ();
		if (open(F, "$install_src_root/extra/ncftpd.service")) {
			while (defined(my $line = <F>)) {
				if ($line =~ /^Conflicts\s*=\s*(\S.*\S)*/i) {
					my ($srv) = $1;
					$srv =~ s/\.service//ig;
					@srvs = split(/\s+/, $srv);
				}
			}
			close(F);
		}

		# See if those services are known to this system.
		@upstart_competing_services = ();
		foreach my $srv (@srvs) {
			if (exists($available_upstart_services{$srv})) {
				push(@upstart_competing_services, $srv);
			}
		}

		# Block services that are present here and are known conflicts.
		foreach my $srv (@upstart_competing_services) {
			next unless (exists($available_upstart_services{$srv}));
			if ($available_upstart_services{$srv} > 0) {
				# Service was running; stop it now.
				if (RunProgramOrExit($initctl, "stop", $srv)) {
					UninstallLog("system(\"$initctl\", \"start\", \"$srv\");\n");
				} else {
					Log(0, "ERROR: Could not stop $srv in Upstart (initctl). This will cause the installation to fail if its associated processes remain running.\n");
				}
			}
			UninstallLog("system(\"$initctl\", \"reload-configuration\");\n");
			if (SafeRename("$upstart_jobs_dir/$srv.conf", "$upstart_jobs_dir/$srv.conf.disabled") ne "") {
				Log(0, "Disabling $srv in upstart.\n");
				RunProgramOrExit($initctl, "reload-configuration");
			} else {
				Log(0, "Could not disable $srv in upstart. This may cause problems the next time the system boots.\n");
			}
		}
	}

	if ($launchctl ne "") {
		# Block services that are present here and are known conflicts.
		foreach my $srv (@launchd_competing_services) {
			StopAndUnloadLaunchdService($srv);
		}
	}
}	# CheckStartupSystem




sub IsServiceCurrentlyRunningFromLaunchd
{
	my ($srv) = $_[0];
	my ($pid) = 0;

	Log(1, "Checking for running job named \"$srv\" in Launchd.\n");
	if (($launchctl ne "") && (defined($srv)) && ($srv ne "") && (open(LAUNCHCTL, "$launchctl list $srv |"))) {
		while (defined(my $line = <LAUNCHCTL>)) {
			if ($line =~ /\DPID\D+(\d+)/) {
				$pid = $1;
				Log(1, "Job named \"$srv\" is currently running (PID $pid) from Launchd.\n");
			}
		}
		close(LAUNCHCTL);
	}
	Log(1, "Job named \"$srv\" is not currently running from Launchd.\n") if ($pid != 0);
	return ($pid);
}	# IsServiceCurrentlyRunningFromLaunchd




sub TestLsof
{
	Log(0, "Checking if \"$lsof\" works. This may require up to 30 seconds.\n");
	if (open(LSOFTEST, "$lsof $lsof 2>&1 |")) {
		my (@lsof_result) = <LSOFTEST>;
		close(LSOFTEST);
		my (@lsof_lines) = grep(/COMMAND/, @lsof_result);
		if (scalar(@lsof_lines) > 0) {
			Log(1, "Test run of $lsof succeeded.\n");
			return (1);
		} else {
			Log(1, "Test run of $lsof failed:\n");
			my ($i);
			for $i (@lsof_result) {
				chomp($i);
				Log(1, "  $i\n");
			}
		}
	}
	$lsof = "";
	return (0);
}	# TestLsof



sub FindLsof
{
	return (1) if (($lsof ne "") && (-x $lsof));
	$lsof = SearchPath("lsof", "/sbin:/usr/sbin:/usr/local/sbin:/usr/bin:/bin");
	return (1) if (TestLsof());
	$lsof = SearchPath("lsof", "", "/sbin:/usr/sbin:/usr/local/sbin:/usr/bin:/bin");
	return (1) if (TestLsof());
	return (0);
}	# FindLsof




sub FindPerl
{
	return (1) if (($perl ne "") && (-x $perl));
	$perl = SearchPath("perl");
	$perl = "/usr/bin/perl" if ($perl eq "");
	return (1);
}	# FindPerl




sub AbortIfSomethingWeDontKnowHowToDisableIsUsingPort21()
{
	my (@lsof_result);

	if (! FindLsof()) {
		Log(1, "Could not find lsof.\n");
	} elsif (open(LSOF, "$lsof -w -i tcp:$ftpport | sed -n '1p;/LISTEN/p;' 2>/dev/null |")) {
		@lsof_result = <LSOF>;
		close(LSOF);
		if ((defined($lsof_result[0])) && ($lsof_result[0] =~ /^COMMAND/)) {
			my ($lines_printed) = 0;
			for (my $line = 1; $line < scalar(@lsof_result); $line++) {
				my (@junk) = split(' ', $lsof_result[$line], 2);
				my ($program) = $junk[0];
				# Note that "ftpd" will match wu-ftpd, etc.
				if ($program !~ /(ncftpd|xinetd|inetd|ftpd)/i) {
					if (! $lines_printed) {
						Log(0, "\nERROR: You must first uninstall your current FTP server software.\n");
						Log(0, "The following processes are using the FTP service port ($ftpport):\n\n");
						Log(0, "  ", $lsof_result[0]);
					}
					Log(0, "  ", $lsof_result[$line]);
					$lines_printed++;
				} elsif ($program =~ /ncftpd/) {
					$ncftpd_running_at_time_of_install++;
					$ftp_service_running_at_time_of_install = "ncftpd";
				} elsif ($program =~ /xinetd/) {
					$ftp_service_running_at_time_of_install = "xinetd";
				} elsif ($program =~ /inetd/) {
					$ftp_service_running_at_time_of_install = "inetd";
				} else {
					$ftp_service_running_at_time_of_install = $program;
				}
			}
			if ($lines_printed) {
				Log(0, "\n");
				Log(0, "You can try killing these processes and restarting the installation,\n");
				Log(0, "but you must disable the other software's startup script. Otherwise,\n");
				Log(0, "the next time you reboot the machine NcFTPd will not be able to start.\n");
				Exit(1);
			}
		} elsif ((defined($lsof_result[0])) && ($lsof_result[0] !~ /^\s*$/)) {
			Log(0, "Unexpected result from $lsof:\n");
			for (my $line = 1; $line < scalar(@lsof_result); $line++) {
				Log(0, "  " . $lsof_result[$line]);
			}
		} else {
			Log(1, "No processes are currently handling FTP.\n");
		}
		if ($ncftpd_running_at_time_of_install > 0) {
			Log(0, "NcFTPd is currently handling FTP.\n");
		} elsif ($ftp_service_running_at_time_of_install eq "inetd") {
			Log(0, "Inetd is currently handling FTP.\n");
		} elsif ($ftp_service_running_at_time_of_install eq "xinetd") {
			Log(0, "Xinetd is currently handling FTP.\n");
		} elsif ($ftp_service_running_at_time_of_install ne "") {
			Log(0, "Something else ($ftp_service_running_at_time_of_install) is currently handling FTP.\n");
			Log(0, "\n");
			Log(0, "I will try killing these processes and continuing the installation,\n");
			Log(0, "but you may need to disable the other software's startup script. Otherwise,\n");
			Log(0, "the next time you reboot the machine NcFTPd will not be able to start.\n");
			Log(0, "\n");
			$ftp_service_running_at_time_of_install =~ s-^.*/--;
		}
	} else {
		Log(1, "Could not run lsof, so we won't be able to determine which process, if any, is running on port $ftpport.\n");
	}
}	# AbortIfSomethingWeDontKnowHowToDisableIsUsingPort21




sub CreateEtcShells
{
	return if (-f "/etc/shells");

	my (@shells_found) = ();
	my ($shell);
	my ($path);
	my ($aix_shells_found) = 0;

	if (open(AIXSHELLS, "/etc/security/login.cfg")) {
		#
		# AIX has information about its shells in this file.
		#
		# Example:
		#	shells = /bin/sh,/bin/bsh,/bin/csh,/bin/ksh,/bin/tsh,/usr/bin/sh,/usr/bin/bsh,/usr/bin/csh,/usr/bin/ksh,/usr/bin/tsh,/usr/sbin/sliplogin
		#
		my (@logincfgdata) = <AIXSHELLS>;
		close(AIXSHELLS);
		my (@shells_line) = grep(/^\s*shells\s*=/, @logincfgdata);
		if (scalar(@shells_line) > 0) {
			if ($shells_line[0] =~ /\=\s*(\/.*)/) {
				$shells_line[0] = $1;
				@shells_found = split(/,/, $shells_line[0]);
				$aix_shells_found = scalar(@shells_found);
			}
		}
	}

	my (@paths) = ("/sbin", "/bin", "/bin/posix", "/usr/bin", "/usr/local/bin");
	my (@shells) = ("sh", "ksh", "csh", "bash", "tcsh", "zsh", "pdksh", "rksh", "passwd");

	for $path (@paths) {
		for $shell (@shells) {
			if (-x "$path/$shell") {
				my ($a);
				my ($b) = 0;
				for $a (@shells_found) {
					if ($a eq "$path/$shell") {
						$b = 1;
						last;
					}
				}
				push(@shells_found, "$path/$shell") if ($b == 0);
			}
		}
	}

	return if (scalar(@shells_found) == 0);
	if (open(ETCSHELLS, ">/etc/shells")) {
		print ETCSHELLS "# NcFTPd will not allow users to connect who are not using one of\n# these shells.\n";

		if ($aix_shells_found) {
			print ETCSHELLS "#\n";
			print ETCSHELLS "# This file was generated from /etc/security/login.cfg.\n";
			print ETCSHELLS "# If you change this file, also change /etc/security/login.cfg.\n";
		}

		print ETCSHELLS "#\n";
		for $shell (@shells_found) {
			print ETCSHELLS $shell, "\n";
		}	
		close(ETCSHELLS);
		chmod(0644, "/etc/shells");
		Log(0, "Created /etc/shells.\n");
		UninstallLog("Remove(\"/etc/shells\");\n");
	}
}	# CreateEtcShells




sub CreateEtcFtpusers
{
	my ($user);
	my ($t0, $t1);
	my ($uid);
	my (@users) = (	"root", "kroot", "broot", "croot", "toor",
			"4Dgifts",
			"adabas",
			"adm",
			"Administrator",
			"administrator",
			"amanda",
			"anon",
			"at",
			"auditor",
			"auth",
			"bin",
			"bind",
			"cmwlogin",
			"cron",
			"cyrus",
			"daemon",
			"db2",
			"db2as",
			"db2fenc1",
			"db2inst1",
			"dbadmin",
			"dbmaker",
			"demo",
			"demos",
			"diag",
			"dpbox",
			"empress",
			"EZsetup",
			"fax",
			"fib",
			"firewall",
			"fixadm",
			"fixlohn",
			"flexlm",
			"fnet",
			"games",
			"gdm",
			"gnats",
			"guest",
			"halt",
			"hpdb",
			"imap",
			"informix",
			"ingres",
			"irc",
			"ixess",
			"kmem",
			"listen",
			"lnx",
			"lp",
			"lpd",
			"mail",
			"man",
			"mdom",
			"mosaic",
			"mysql",
			"named",
			"netscape",
			"news",
			"noaccess",
			"nobody",
			"nobody4",
			"nobodyV",
			"nonroot",
			"nonuser",
			"notes",
			"nouser",
			"nps",
			"ntp",
			"nuucp",
			"operator",
			"oracle",
			"OutOfBox",
			"pop",
			"postfix",
			"postgres",
			"quota",
			"radius",
			"reboot",
			"rfindd",
			"ris",
			"seti",
			"sgiweb",
			"shutdown",
			"skyrix",
			"spooladm",
			"sql",
			"squid",
			"sync",
			"sys",
			"sysadm",
			"tacacs",
			"tcb",
			"tty",
			"unknown",
			"uucp",
			"uucpa",
			"virtuoso",
			"vmware",
			"web",
			"wnn",
			"www",
			"wwwrun",
			"xfs",
			"xten",
			"yard",
			);
	my (@users_found) = ();
	my ($number_of_ftpusers_loaded) = 0;

	if (open(ETCFTPUSERS, "</etc/ftpusers")) {
		while (defined($user = <ETCFTPUSERS>)) {
			chomp($user);
			next unless ($user =~ /^[A-Za-z]/);
			push (@users_found, $user);
		}
		close(ETCFTPUSERS);
	}
	$number_of_ftpusers_loaded = scalar(@users_found);

	$t0 = time();
	Log(1,"Checking system user database for /etc/ftpusers suggestions.\n");
	for $user (@users) {
		my (@found) = grep(/$user/, @users_found);
		next if (scalar(@found) > 0);
		$uid = getpwnam($user);
		push (@users_found, $user) if (defined($uid));
		$t1 = time();
		if (($t1 - $t0) > 15) {
			Log(0, "Warning: It is taking too long to lookup usernames.  Perhaps you're using NIS or LDAP?\n");
			Log(0, "         The /etc/ftpusers checks will be skipped.\n");
			return;
		}
	}

	return if (scalar(@users_found) == 0);

	if ($number_of_ftpusers_loaded == 0) {
		if (open(ETCFTPUSERS, ">/etc/ftpusers")) {
			for $user (@users_found) {
				print ETCFTPUSERS $user, "\n";
			}	
			close(ETCFTPUSERS);
			chmod(0644, "/etc/ftpusers");
			Log(0, "Created /etc/ftpusers.\n");
			UninstallLog("Remove(\"/etc/ftpusers\");\n");
		}
	} elsif ($number_of_ftpusers_loaded < scalar(@users_found)) {
		# Don't munge the system's existing file,
		# but warn about users we think should be
		# in there.
		#
		Log(0, "\n");
		Log(0, "Warning: your /etc/ftpusers file may be missing entries.\n");
		Log(0, "The /etc/ftpusers file is a list of users who should NOT\n");
		Log(0, "be allowed to login by FTP.  The following users were found\n");
		Log(0, "on your system, and are NOT in /etc/ftpusers.  Please review\n");
		Log(0, "the list below, and if one or more of the users below should\n");
		Log(0, "not be logging in by FTP, add them to /etc/ftpusers using\n");
		Log(0, "a text editor (such as ", defined($ENV{EDITOR}) ? $ENV{EDITOR} : "vi", ").\n\n");
		my ($i);
		my ($j) = 0;
		my ($n) = scalar(@users_found);
		for ($i = $number_of_ftpusers_loaded; $i < $n; $i++, $j++) {
			$user = $users_found[$i];
			Log(0, "    ") if (($j % 4) == 0);
			Log(0, sprintf("%-12s", $user));
			Log(0, "\n") if ((($j % 4) == 3) || ($i == ($n - 1)));
		}
		Log(0, "\n");
	}
}	# CreateEtcFtpusers




sub AddFTPGroup
{
	my ($i, $grp);
	my (@altgroups) = ("www", "web", "daemon", "nobody");

	$ftpgid = getgrnam($ftpgroup);
	if ((defined($ftpgid)) && ($ftpgid > 0)) {
		Log(1, "Using group $ftpgroup (GID $ftpgid)\n");
		return;
	}

	$ftpgid = getgrnam("_ftp");
	if ((defined($ftpgid)) && ($ftpgid > 0)) {
		$ftpgroup = "_ftp";
		Log(1, "Using group $ftpgroup (GID $ftpgid)\n");
		return;
	}

	#
	# The "ftp" group does not exist.
	# Try to add one.
	#
	Log(1, "$ftpgroup group does not exist\n");

	#
	# Pick a random GID.
	#
	for ($ftpgid = 50; $ftpgid < 400; $ftpgid += 10) {
		last if (! defined(getgrgid($ftpgid)));
	}

	if ($SYS eq "macosx") {
		if (SearchPath("niutil") ne "") {
			RunProgramOrExit("niutil", "-create",     "/", "/groups/$ftpgroup");
			RunProgramOrExit("niutil", "-createprop", "/", "/groups/$ftpgroup", "passwd", "*");
			RunProgramOrExit("niutil", "-createprop", "/", "/groups/$ftpgroup", "gid", "$ftpgid");
		}
	} elsif ($SYS eq "bsdos") {
		RunProgramIfPresent("/usr/sbin/addgroup", "-g", $ftpgid, $ftpgroup);
	} elsif ($SYS eq "aix") {
		RunProgramIfPresent("/usr/bin/mkgroup", "id=$ftpgid", $ftpgroup);
	} elsif ($SYS eq "sco") {
		RunProgramIfPresent("/etc/groupadd", "-g", $ftpgid, $ftpgroup);
	} elsif (($SYS =~ /^(openbsd|netbsd|solaris|linux|tru64unix|digitalunix|unixware|openunix|hpux)$/) && (-x "/usr/sbin/groupadd")) {
		RunProgramIfPresent("/usr/sbin/groupadd", "-g", $ftpgid, $ftpgroup);
	}

	$ftpgid = getgrnam($ftpgroup);
	if ((defined($ftpgid)) && ($ftpgid > 0)) {
		Log(0, "Created group $ftpgroup (UID $ftpgid)\n");
		UninstallLog("RemoveGroup(\"$ftpgroup\");\n");
		return;
	}

	# We'd prefer to be able to add the group, but we
	# can live without it since we can borrow commonly
	# available groupnames.
	#
	for $grp (@altgroups) {
		$ftpgid = getgrnam($grp);
		if ((defined($ftpgid)) && ($ftpgid > 0)) {
			$ftpgroup = $grp;
			Log(1, "Using alternate choice for group = $ftpgroup (UID $ftpgid)\n");
			return;
		}
	}
}	# AddFTPGroup




sub AddFTPUser
{
	my ($i, $usr);
	my (@altusers) = ("_www", "_sandbox", "www", "web", "daemon", "nobody");
	my ($noshell) = "";	# Most systems just leave the shell field blank.
	my (@shells) = ();
	my ($modified_etc_shells) = 0;

	$ftpuid = getpwnam($ftpuser);
	if ((defined($ftpuid)) && ($ftpuid > 0)) {
		Log(1, "Using user $ftpuser (UID $ftpuid)\n");
		return;
	}

	$ftpuid = getpwnam("_ftp");
	if ((defined($ftpuid)) && ($ftpuid > 0)) {
		$ftpuser = "_ftp";
		Log(1, "Using user $ftpuser (UID $ftpuid)\n");
		return;
	}

	#
	# The "ftp" user does not exist.
	# Try to add one.
	#
	Log(1, "$ftpuser user does not exist\n");

	$noshell = "/sbin/nologin" if (-x "/sbin/nologin");
	$noshell = "/dev/null" if ($SYS eq "macosx");
	$noshell = "/dev/null" if ($SYS =~ /^irix/);
	$noshell = "/usr/sbin/nologin" if ($SYS eq "freebsd");
	$noshell = "/sbin/nologin" if ($OS =~ /^freebsd4/);
	$noshell = "/nonexistent" if ($OS =~ /^freebsd[23]/);
	$noshell = "/bin/false" if ($SYS eq "tru64unix");	# Or else userdel fails
	$noshell = "/bin/false" if ($SYS eq "solaris");		# Or else useradd fails
	$noshell = "/bin/false" if ($SYS eq "unixware");	# Or else useradd fails
	$noshell = "/bin/false" if ($SYS eq "openunix");	# Or else useradd fails
	$noshell = "/bin/false" if ($SYS eq "hpux");		# Or else useradd fails

	if (($SYS =~ /^(forfutureuse)$/) && (-f "/etc/shells")) {
		#
		# Some systems won't let you add a user unless
		# you specify a valid shell program.  We don't
		# want to add the "ftp" user with a real shell,
		# so we add /bin/false to /etc/shells which
		# is harmless if actually logged in with.
		#
		if (open(ETCSHELLS, "</etc/shells")) {
			@shells = <ETCSHELLS>;
			Log(1, "Loading /etc/shells.\n");
			close(ETCSHELLS);
			if (scalar(grep(/$noshell/, @shells)) <= 0) {
				#
				# Our no-shell wasn't already in there,
				# add it temporarily.
				#
				Log(1, "$noshell was not in /etc/shells.\n");
				if (open(ETCSHELLS, ">>/etc/shells")) {
					print ETCSHELLS $noshell, "\n";
					close(ETCSHELLS);
					$modified_etc_shells = 1;
					Log(1, "Added $noshell to /etc/shells.\n");
				}
			}
		}
	}

	#
	# Pick a random UID.
	#
	$ftpuid = 14;
	$ftpuid = 214 if ($SYS eq "sco");	# Otherwise SCO useradd fails
	for ($ftpuid = 14; $ftpuid < (65534 - 42); $ftpuid += 41) {
		last if (! defined(getpwuid($ftpuid)));
	}

	if ($SYS eq "macosx") {
		$ftphome = "/Library/FTPServer/Documents";
		if (SearchPath("niutil") ne "") {
			RunProgramOrExit("niutil", "-create",     "/", "/users/$ftpuser");
			RunProgramOrExit("niutil", "-createprop", "/", "/users/$ftpuser", "shell", "$noshell");
			RunProgramOrExit("niutil", "-createprop", "/", "/users/$ftpuser", "passwd", "*");
			RunProgramOrExit("niutil", "-createprop", "/", "/users/$ftpuser", "realname", "FTP Server");
			RunProgramOrExit("niutil", "-createprop", "/", "/users/$ftpuser", "uid", $ftpuid);
			RunProgramOrExit("niutil", "-createprop", "/", "/users/$ftpuser", "gid", $ftpgid);
			RunProgramOrExit("niutil", "-createprop", "/", "/users/$ftpuser", "home", $ftphome);
			# RunProgramOrExit("niutil", "-createprop", "/", "/users/$ftpuser", "_shadow_passwd", "");
			RunProgramOrExit("niutil", "-createprop", "/", "/users/$ftpuser", "expire", "0");
			RunProgramOrExit("niutil", "-createprop", "/", "/users/$ftpuser", "change", "0");
		}
	} elsif ($SYS =~ /^irix/) {
		# There doesn't seem to be a corresponding groupmgmt.
		RunProgramIfPresent("/usr/sbin/passmgmt", "-a", "-u", $ftpuid, "-g", $ftpgid, "-c", "FTP User", "-s", $noshell, $ftpuser);
	} elsif ($SYS eq "bsdos") {
		RunProgramIfPresent("/usr/sbin/adduser", "-u", $ftpuid, "-g", $ftpgid, "-G", "FTP User", "-s", $noshell, "-H", "/usr/home/$ftpuser", "-P", "*", $ftpuser);
	} elsif ($SYS eq "aix") {
		# User will get created with a regular shell, 
		# but they can't login in since we set su,rlogin,login=false.
		#
		RunProgramIfPresent("/usr/bin/mkuser", "daemon=false", "expires=0", "gecos=FTP User", "id=$ftpuid", "pgrp=$ftpgroup", "login=false", "rlogin=false", "su=false", "umask=022", $ftpuser);
	} elsif ($SYS eq "sco") {
		RunProgramIfPresent("/etc/useradd", "-u", $ftpuid, "-g", $ftpgid, "-c", "FTP User", "-s", $noshell, $ftpuser);
	} elsif (($SYS =~ /^(openbsd|netbsd|linux|tru64unix|digitalunix|solaris|unixware|openunix|hpux)$/) && (-x "/usr/sbin/useradd")) {
		RunProgramIfPresent("/usr/sbin/useradd", "-u", $ftpuid, "-g", $ftpgid, "-c", "FTP User", "-s", $noshell, $ftpuser);
	} elsif ($OS =~ /^freebsd[56789]/) {
		if (open(ADDUSER, "| /usr/sbin/adduser -f - -w no -D -u $ftpuid")) {
			print ADDUSER "${ftpuser}::::::FTP User::nologin:\n";
			close(ADDUSER);
		}
	}
	
	if ($modified_etc_shells) {
		if (open(ETCSHELLS, ">/etc/shells")) {
			print ETCSHELLS @shells;
			close(ETCSHELLS);
			$modified_etc_shells = 0;
			Log(1, "Restored /etc/shells.\n");
			chmod(0644, "/etc/shells");
		}
	}

	$ftpuid = getpwnam($ftpuser);
	if ((defined($ftpuid)) && ($ftpuid > 0)) {
		Log(0, "Created user $ftpuser (UID $ftpuid)\n");
		UninstallLog("RemoveUser(\"$ftpuser\");\n");
		return;
	}

	# We'd prefer to be able to add the user, but we
	# can live without it since we can borrow commonly
	# available usernames.
	#
	for $usr (@altusers) {
		$ftpuid = getpwnam($usr);
		if ((defined($ftpuid)) && ($ftpuid > 0)) {
			$ftpuser = $usr;
			Log(1, "Using alternate choice for user = $ftpuser (UID $ftpuid)\n");
			return;
		}
	}
}	# AddFTPUser




sub CreateFTPHome
{
	my ($name,$passwd,$uid,$gid,$quota,$comment,$gcos,$dir,$shell,$expire);

	($name,$passwd,$uid,$gid,$quota,$comment,$gcos,$dir,$shell,$expire) = 
		getpwnam($ftpuser);
	if (defined($dir)) {
		if (($ftpuser eq "www") && ($dir eq "/Library/WebServer")) {
			#
			# For Mac OS X, we'll try to use "www", and the
			# htdocs directory (/Library/WebServer/Documents)
			# for ftp-home.
			#
			$ftphome = "/Library/WebServer/Documents";
			Log(1, "Using $ftphome for ftp-home\n");
		} elsif ($ftpuser ne $ftpuser_preferred) {
			#
			# If we couldn't add "ftp", NcFTPd will use one of
			# the alternate users.  In this case, NcFTPd will
			# also not use that alternate user's home directory
			# as ftp-home.
			#
			if (-d "/export/home") {
				$ftphome = "/export/home/ftp";
			} elsif (-d "/home") {
				$ftphome = "/home/ftp";
			} elsif (-d "/usr/local") {
				$ftphome = "/usr/local/ftp";
			} else {
				$ftphome = "/usr/ftp";
			}
			Log(0, "Using $ftphome for ftp-home\n");
		} else {
			$ftphome = $dir;
			Log(1, "Using $ftphome for ftp-home\n");
		}

		#
		# Note that NcFTPd can run if ftp-home doesn't exist.
		#
		if (! -d $ftphome) {
			my ($did_mkd) = 0;
			if (Mkdirs($ftphome, 00755)) {
				$did_mkd = 1;
			} else {
				my ($alt_ftphome) = "/opt/ftp/anonymous_ftp_root";
				if ((-d "/opt") && (Mkdirs($alt_ftphome, 00755))) {
					Log(0, "Using $alt_ftphome for ftp-home since $ftphome could not be created.\n");
					$ftphome = $alt_ftphome;
					$did_mkd = 1;
				}
			}
			if ($did_mkd) {
				Log(0, "Created ftp-home directory, $ftphome\n");
				if (open(README, ">$ftphome/README")) {
					print README "This is the anonymous FTP area on $host.\n";
					print README "Files for public access to the world can be placed here.\n";
					close(README);
					chmod(0644, "$ftphome/README");
					UninstallLog("Remove(\"$ftphome/README\");\n");
				}
			} else {
				Log(0, "Warning: Could not create ftp-home directory, $ftphome\n");
			}
		}

		#
		# Make sure ~ftp is NOT owned by ftp!
		#
		chmod(0755, $ftphome);
		chown(0, 1, $ftphome);
	}
}	# CreateFTPHome




sub IsThereAnFTPServerServing
{
	my ($v) = @_;
	my ($hardway) = 0;
	my ($serveraddr);

	$v = 0 if (! defined($v));
	$serveraddr = inet_aton("localhost") || inet_aton("127.0.0.1");
	$hardway = 1 if (! defined($serveraddr));

	$hardway = 1 if (! socket(FTPSOCKET, PF_INET, SOCK_STREAM, getprotobyname('tcp')));

	if ($hardway) {
		#
		# This older way doesn't work as well.
		#
		Log(2, "Using alternate method to tell if FTP server is running.\n");
		my ($ftp) = SearchPath("ftp", "/usr/bin");
		my ($echo) = SearchPath("echo", "/bin:/usr/bin");
		if ($ftp eq "") {
			Log(0, "/usr/bin/ftp is not present.\n");
			Log(0, "Cannot tell if FTP server is running.\n");
			return (0);
		}
		if ($echo eq "") {
			Log(0, "/bin/echo is not present.\n");
			$echo = "echo";		# Try anyway
		}
		if (open(FTP, "$echo quit | $ftp -d -n 127.0.0.1 2>/dev/null |")) {
			my (@ftpresult) = <FTP>;
			close(FTP);
			# Connected to localhost.
			# 220 Hurricane NcFTPd Server (free personal license) ready.
			# ftp> quit
			# ---> QUIT
			# 221 Goodbye.
			my (@quitresult) = grep(@ftpresult, "QUIT");
			if (scalar(@quitresult) > 0) {
				Log($v, "An FTP server is running.\n");
				return (1);
			}
		}
		Log($v, "No FTP server is currently running.\n");
		return (0);
	}

	$serveraddr = sockaddr_in($ftpport, $serveraddr);
	if (connect(FTPSOCKET, $serveraddr)) {
		print FTPSOCKET "QUIT\r\n";
		close(FTPSOCKET);
		Log($v, "An FTP server is running.\n");
		return (1);
	}

	close(FTPSOCKET);
	Log($v, "No FTP server is running.\n");
	return (0);
}	# IsThereAnFTPServerServing




sub CommentOutFTPLineInInetdConf
{
	my ($inetd_conf) = "/etc/inetd.conf";
	$inetd_conf = "/etc/inet/inetd.conf" if ($SYS eq "solaris");

	my ($inetd_conf_new) = "$inetd_conf.new";
	my ($line);
	my ($disabled, $disableprefix);

	return unless (-f $inetd_conf);

	$disabled = 0;
	if (open(INETDCONF, "<$inetd_conf")) {
		if (! open(INETDCONFNEW, ">$inetd_conf_new")) {
			Log(0, "Could not create the file $inetd_conf_new: $!\n");
			Log(0, "The \"inetd\" process will not be able to be configured.\n");
			close(INETDCONF);
			return (0);
		}

		$disableprefix = "";
		while (defined($line = <INETDCONF>)) {
			if ($line =~ /^$ftpservicename\b/) {
				$disableprefix = "#DISABLED by install_ncftpd.pl# ";
				$disabled++;
			}
			print INETDCONFNEW $disableprefix, $line;

			# Also disable continuation lines.
			$disableprefix = "" unless ($line =~ /\\$/);
		}
		close(INETDCONF);
		close(INETDCONFNEW);
	}

	if ($disabled) {
		chmod(0644, $inetd_conf);
		unlink($inetd_conf);
		rename($inetd_conf_new, $inetd_conf) or Exit(1, "Could not rename $inetd_conf_new to $inetd_conf: $!\n");
		chmod(0644, $inetd_conf);
		Log(0, "Disabled the \"$ftpservicename\" service in the $inetd_conf file.\n");
		UninstallLog("UncommentOutFTPLineInInetdConf(\"$inetd_conf\");\n");
		return (1);
	} else {
		Log(0, "$inetd_conf does not appear to be handling the \"$ftpservicename\" service.\n");
		unlink($inetd_conf_new);
		return (0);
	}
}	# CommentOutFTPLineInInetdConf




sub CommentOutFTPLineInXinetdConf
{
	my ($inetd_conf) = "/etc/xinetd.conf";
	my ($inetd_conf_new) = "$inetd_conf.new";
	my ($line);
	my ($disabled, $disableprefix);

	return unless (-f $inetd_conf);

	$disabled = 0;
	if (open(INETDCONF, "<$inetd_conf")) {
		if (! open(INETDCONFNEW, ">$inetd_conf_new")) {
			Log(0, "Could not create the file $inetd_conf_new: $!\n");
			Log(0, "The \"xinetd\" process will not be able to be configured.\n");
			close(INETDCONF);
			return (0);
		}

		$disableprefix = "";
		while (defined($line = <INETDCONF>)) {
			if ($line =~ /^[^#]*service\s+$ftpservicename\b/) {
				$disableprefix = "#DISABLED by install_ncftpd.pl# ";
				$disabled++;
			}
			print INETDCONFNEW $disableprefix, $line;

			# Disable rest of block.
			$disableprefix = "" if ($line =~ /^[^#]*\}/);
		}
		close(INETDCONF);
		close(INETDCONFNEW);
	}

	if ($disabled) {
		chmod(0644, $inetd_conf);
		unlink($inetd_conf);
		rename($inetd_conf_new, $inetd_conf) or Exit(1, "Could not rename $inetd_conf_new to $inetd_conf: $!\n");
		chmod(0644, $inetd_conf);
		Log(0, "Disabled the \"$ftpservicename\" service in the $inetd_conf file.\n");
		UninstallLog("UncommentOutFTPLineInXinetdConf(\"$inetd_conf\");\n");
		return (1);
	} else {
		Log(0, "$inetd_conf does not appear to be handling the \"$ftpservicename\" service.\n");
		unlink($inetd_conf_new);
		return (0);
	}
}	# CommentOutFTPLineInXinetdConf




sub CommentOutFTPLineInXinetdD
{
	my ($inetd_d) = "/etc/xinetd.d";
	my ($inetd_conf);
	my ($line);
	my ($disabled, $disableprefix, $tdisabled);

	return unless (-d $inetd_d);

	my (@inetd_d_files) = glob("$inetd_d/*");

	$tdisabled = 0;
	for $inetd_conf (@inetd_d_files) {
		next if (-d $inetd_conf);
		if (open(INETDCONF, "<$inetd_conf")) {
			my ($inetd_conf_new) = "$inetd_conf.new";
			if (! open(INETDCONFNEW, ">$inetd_conf_new")) {
				Log(0, "Could not create the file $inetd_conf_new: $!\n");
				Log(0, "The \"xinetd\" process will not be able to be configured.\n");
				close(INETDCONF);
				return (0);
			}

			$disableprefix = "";
			$disabled = 0;
			while (defined($line = <INETDCONF>)) {
				if ($line =~ /^[^#]*service\s+$ftpservicename\b/) {
					$disableprefix = "#DISABLED by install_ncftpd.pl# ";
					$disabled++;
				}
				print INETDCONFNEW $disableprefix, $line;

				# Disable rest of block.
				$disableprefix = "" if ($line =~ /^[^#]*\}/);
			}
			close(INETDCONF);
			close(INETDCONFNEW);

			if ($disabled) {
				chmod(0644, $inetd_conf);
				unlink($inetd_conf);
				rename($inetd_conf_new, $inetd_conf) or Exit(1, "Could not rename $inetd_conf_new to $inetd_conf: $!\n");
				chmod(0644, $inetd_conf);
				Log(0, "Disabled the \"$ftpservicename\" service in the $inetd_conf file.\n");
				$tdisabled++;
				UninstallLog("UncommentOutFTPLineInXinetdD(\"$inetd_conf\");\n");
			} else {
				unlink($inetd_conf_new);
			}
		}
	}

	if ($tdisabled > 0) {
		return (1);
	} else {
		Log(0, "$inetd_d does not appear to be handling the \"$ftpservicename\" service.\n");
		return (0);
	}
}	# CommentOutFTPLineInXinetdD




sub TellInetdToReloadConfiguration
{
	my (@pids) = PsGrep("inetd", "program", "pids");
	my ($pid) = 0;
	my ($npids) = scalar(@pids);

	if ($npids == 0) {
		Log(1, "Warning: No Inetd process is running?\n");
		return;
	}

	if ($npids > 1) {
		Log(1, "Warning: Multiple Inetd processes detected?\n");
		for $pid (@pids) {
			Log(1, "  PID $pid\n");
		}
	}

	for $pid (@pids) {
		Log(0, "Sending SIGHUP to Inetd running on PID $pid.\n");
		kill HUP => $pid;
	}

	Log(1, "Giving Inetd a few seconds to reload.\n");
	sleep(2);
}	# TellInetdToReloadConfiguration




sub TellXinetdToReloadConfiguration
{
	my (@pids) = PsGrep("xinetd", "program", "pids");
	my ($pid) = 0;
	my ($npids) = scalar(@pids);

	if ($npids == 0) {
		Log(1, "Warning: No Xinetd process is running?\n");
		return;
	}

	if ($npids > 1) {
		Log(1, "Warning: Multiple Xinetd processes detected?\n");
		for $pid (@pids) {
			Log(1, "  PID $pid\n");
		}
	}

	for $pid (@pids) {
		Log(0, "Sending SIGUSR2 to Xinetd running on PID $pid.\n");
		kill USR2 => $pid;
	}

	Log(1, "Giving Xinetd a few seconds to reload.\n");
	sleep(2);
}	# TellXinetdToReloadConfiguration




sub WaitForFTPServerToStart
{
	my ($i);
	my ($n) = @_;
	my ($isRunning) = 0;
	$n = 4 if ((! defined($n)) || ($n < 1));

	for ($i = 1; $i <= $n; $i++) {
		Log(0, "Checking if the FTP service has started.\n");
		$isRunning = IsThereAnFTPServerServing(1);
		last if ($isRunning == 1);
		Log(2, sprintf("Sleeping %d seconds.\n", (3 * $i)));
		sleep(3 * $i) if ($i < $n);
	}
	return ($isRunning);
}	# WaitForFTPServerToStart




sub WaitForFTPServerToExit
{
	my ($i);
	my ($n) = @_;
	my ($isRunning) = 0;
	$n = 4 if ((! defined($n)) || ($n < 1));

	for ($i = 1; $i <= $n; $i++) {
		$isRunning = IsThereAnFTPServerServing(1);
		last if ($isRunning == 0);
		Log(0, "Checking if the FTP service is still active.\n");
		Log(2, sprintf("Sleeping %d seconds.\n", (3 * $i)));
		sleep(3 * $i) if ($i < $n);
	}
	return (! $isRunning);
}	# WaitForFTPServerToExit




sub DisableInetdViaSMF				# solaris 10
{
	my ($inetadm) = "/usr/sbin/inetadm";
	my ($svc) = "network/ftp:default";
	my ($status);

	$status = `$inetadm 2>/dev/null | fgrep \"$svc\"`;
	return (0) if ($status eq "");
	Log(0, "Using inetadm to disable existing FTP service.\n");
	return (0) unless ($status =~ /online/);

	$status = `$inetadm -d \"$svc\" 2>&1 "`;
	$status = `$inetadm 2>/dev/null | fgrep \"$svc\"`;
	return (0) if ($status eq "");
	return (0) unless ($status =~ /disabled\s+disabled/);
	Log(1, "OK: $inetadm -d $svc\n");
	UninstallLog("\`$inetadm -e \\\"$svc\\\"\`;\n");
	return (1);
}	# DisableInetdViaSMF




sub DisableInetdHandlingOfFTP
{
	if (CommentOutFTPLineInInetdConf()) {
		TellInetdToReloadConfiguration();	
		if (! WaitForFTPServerToExit()) {
			Log(0, "\n");
			Log(0, "ERROR: FTP was disabled in the Inetd configuration file, but the\n"); 
			Log(0, "       FTP service is still accepting connections.  You need to\n");
			Log(0, "       manually shutdown and disable your existing FTP service then\n");
			Log(0, "       restart the installation process.\n");
			Log(0, "\n");
		}
	}
	if ((CommentOutFTPLineInXinetdConf()) || (CommentOutFTPLineInXinetdD())) {
		TellXinetdToReloadConfiguration();	
		if (! WaitForFTPServerToExit()) {
			Log(0, "\n");
			Log(0, "ERROR: FTP was disabled in the Xinetd configuration file, but the\n"); 
			Log(0, "       FTP service is still accepting connections.  You need to\n");
			Log(0, "       manually shutdown and disable your existing FTP service then\n");
			Log(0, "       restart the installation process.\n");
			Log(0, "\n");
		}
	}
	if (DisableInetdViaSMF()) {
		if (! WaitForFTPServerToExit()) {
			Log(0, "\n");
			Log(0, "ERROR: FTP was disabled using SMF (/usr/sbin/inetadm), but the\n"); 
			Log(0, "       FTP service is still accepting connections.  You need to\n");
			Log(0, "       manually shutdown and disable your existing FTP service then\n");
			Log(0, "       restart the installation process.\n");
			Log(0, "\n");
		}
	}
}	# DisableInetdHandlingOfFTP




sub KillFTPd
{
	my ($victim) = @_;
	my (@procs) = PsGrep($victim, "program", "pids");
	my (%origprocs);
	my ($proc);
	my ($nproc) = 0;

	if ((! defined($victim)) || ($victim eq "")) { return (-1); }
	return (0) if (scalar(@procs) == 0);
	Log(1, sprintf("Sending TERM signal to %d $victim processes.\n", scalar(@procs)));

	for $proc (@procs) {
		Log(2, "  ") if (($nproc % 10) == 0);
		Log(2, sprintf("%-7d", $proc));
		Log(2, "\n") if (($nproc % 10) == 9);
		$origprocs{$proc} = 1;
		$nproc++;
	}
	Log(2, "\n") if (($nproc % 10) != 9);

	kill TERM => @procs;
	sleep(3);
	my ($i);
	for ($i = 0; $i < 3; $i ++) {
		sleep(2);
		@procs = PsGrep($victim, "program", "pids");
		return (0) if (scalar(@procs) == 0);	# All are dead, done.
	}

	my ($newinstanceprocs) = 0;
	my (@stragglers) = ();
	for $proc (@procs) {
		if (defined($origprocs{$proc})) {
			push(@stragglers, $proc);
		} else {
			$newinstanceprocs++;
		}
	}

	if ($newinstanceprocs > 0) {
		Log(1, "New processes were born after the old ones were killed!\n");
	}
	return ($newinstanceprocs) if (scalar(@stragglers) == 0);

	Log(1, sprintf("Sending KILL signal to %d $victim processes.\n", scalar(@stragglers)));
	$nproc = 0;
	for $proc (@stragglers) {
		Log(2, "  ") if (($nproc % 10) == 0);
		Log(2, sprintf("%-7d", $proc));
		Log(2, "\n") if (($nproc % 10) == 9);
		$nproc++;
	}
	Log(2, "\n") if (($nproc % 10) != 9);
	kill KILL => @stragglers;			# KiLL 'Em ALL
	return ($newinstanceprocs);
}	# KillFTPd




sub KillAnyRunningFTPd
{
	if ($systemctl ne "") {
		my (@srvs) = @systemd_competing_services;
		push(@srvs, "ncftpd.service");
		foreach my $srv (@srvs) {
			if (RunProgramOrExit($systemctl, "--quiet", "is-active", $srv)) {
				Log(0, "Stopping $srv via systemd.\n");
				if (RunProgramOrExit($systemctl, "stop", $srv)) {
					# Stopped.
					UninstallLog("system(\"$systemctl\", \"start\", \"$srv\");\n");
				} else {
					Log(0, "ERROR: Could not stop $srv in systemd. This will cause the installation to fail if its associated processes remain running.\n");
				}
			}
		}
	}

	if ($initctl ne "") {
		my (@srvs) = @upstart_competing_services;
		push(@srvs, "ncftpd");

		# Get a list of all upstart services that could be enabled (and running).
		my (%available_upstart_services) = ();
		if (open(F, "initctl list |")) {
			while (defined(my $line = <F>)) {
				if ($line =~ /^(\S.*)\s.*\sprocess (\d+)/) {
					my ($srv) = $1;
					$available_upstart_services{$srv} = $2;	# running
				} elsif ($line =~ /^(\S.*)\s/) {
					my ($srv) = $1;
					$available_upstart_services{$srv} = 0;	# not running
				}
			}
			close(F);
		}

		# Block services that are present here and are known conflicts.
		foreach my $srv (@srvs) {
			next unless (exists($available_upstart_services{$srv}));
			if ($available_upstart_services{$srv} > 0) {
				# Service was running; stop it now.
				if (RunProgramOrExit($initctl, "stop", $srv)) {
					UninstallLog("system(\"$initctl\", \"start\", \"$srv\");\n");
				} else {
					Log(0, "ERROR: Could not stop $srv in Upstart (initctl). This will cause the installation to fail if its associated processes remain running.\n");
				}
			}
			if (SafeRename("$upstart_jobs_dir/$srv.conf", "$upstart_jobs_dir/$srv.conf.disabled") ne "") {
				Log(0, "Disabling $srv in upstart.\n");
				RunProgramOrExit($initctl, "reload-configuration");
			} else {
				Log(0, "Could not disable $srv in upstart. This may cause problems the next time the system boots.\n");
			}
		}
	}

	if ($launchctl ne "") {
		my (@srvs) = @launchd_competing_services;
		push(@srvs, "com.ncftp.ncftpd:com.ncftp.ncftpd.plist:ncftpd");
		foreach my $srv (@srvs) {
			StopAndUnloadLaunchdService($srv);
		}
	}

	KillFTPd($ftp_service_running_at_time_of_install);
	# KillFTPd("ftpd|in.ftpd|wu\-?ftpd|proftpd");
	KillFTPd(".*[Ff][Tt][Pp][Dd].*");
}	# KillAnyRunningFTPd




sub NoteExistingNcFTPdConfigurationFromCommandLine
{
	my ($line) = @_;

	if ($line =~ /\s(\/\S*\/general.cf)\b/) {
		my ($cf) = $1;
		if ((-f $cf) && ($existing_general_cf eq "")) {
			$existing_general_cf = $cf;
			Log(1, "Found existing NcFTPd config file: $existing_general_cf.\n");
		}
	}
	if ($line =~ /\s(\/\S*\/domain.cf)\b/) {
		my ($cf) = $1;
		if ((-f $cf) && ($existing_domain_cf eq "")) {
			$existing_domain_cf = $cf;
			Log(1, "Found existing NcFTPd config file: $existing_domain_cf.\n");
		}
	}
	if ($line =~ /(\/\S*\/ncftpd)\s/) {
		my ($n) = $1;
		if ((! -d $n) && (-x _) && ($existing_ncftpd eq "")) {
			$existing_ncftpd = $n;
			Log(1, "Found existing NcFTPd at $existing_ncftpd.\n");
		}
	}
}	# NoteExistingNcFTPdConfigurationFromCommandLine




sub DisableExistingFTPServersRunningFromInittab
{
	if (open(ETCINITTAB, "</etc/inittab")) {
		my (@inittab_lines) = <ETCINITTAB>;
		close(ETCINITTAB);

		# nc:345:respawn:/usr/local/etc/ncftpd/ncftpd.sh
		my ($ncftpd_inittab_entries) = 0;
		my ($ftpd_inittab_entries) = 0;
		my ($ftpd) = "NcFTPd";

		my ($line);
		foreach $line (@inittab_lines) {
			next if ($line =~ /^\s*#/);	# skip comments
			next unless ($line =~ /^\w+:\w*:(\w+):.*/);
			next if ($1 eq "off");		# skip "off" services
			if ($line =~ /ncftpd/) {
				Log(1, "Found NcFTPd entry in the /etc/inittab.\n");
				$ncftpd_inittab_entries++;
				$line = "#DISABLED by install_ncftpd.pl# $line";
				if ($ncftpd_inittab_entries > 1) {
					Log(0, "Multiple existing NcFTPd entries found in /etc/inittab -- all will be disabled.\n");
					next;
				}
				NoteExistingNcFTPdConfigurationFromCommandLine($line);
				$ftpd = "NcFTPd";
			} elsif ($line =~ /ftpd/i) {
				Log(1, "Found other ftpd entry in the /etc/inittab.\n");
				$ftpd_inittab_entries++;
				$line = "#DISABLED by install_ncftpd.pl# $line";
				if ($ftpd_inittab_entries > 1) {
					Log(0, "Multiple existing ftpd entries found in /etc/inittab -- all will be disabled.\n");
					next;
				}
				$ftpd = "ftpd" unless ($ncftpd_inittab_entries > 0);
			}
		}
		if (($ncftpd_inittab_entries > 0) || ($ftpd_inittab_entries > 0)) {
			chmod(0644, "/etc/inittab");
			UninstallLog("UncommentOutFTPLinesInInittab();\n");
			if (open(ETCINITTAB, ">/etc/inittab")) {
				print ETCINITTAB @inittab_lines;
				close(ETCINITTAB);
				chmod(0644, "/etc/inittab");
				Log(0, "Disabled $ftpd in /etc/inittab.\n");

				my ($init) = SearchPath("init", "/sbin:/usr/sbin:/usr/bin:/bin");
				$init = "init" unless ($init);
				Log(1, "Running \"$init q\" to reload modified /etc/inittab.\n");
				RunProgramOrExit($init, "q");

				# At this point, "init" should send a TERM signal
				# to ncftpd.  Some buggy "init" implementations do not,
				# so wait a bit, then try to manually kill off the
				# processes.
				#
				Log(1, "Waiting for $init to ask $ftpd to shutdown.\n");
				sleep(5);
				if (! WaitForFTPServerToExit(1)) {
					Log(0, "Apparently \"$init\" did not shutdown $ftpd.  We'll try to now.\n");
					# KillFTPd("ncftpd|ftpd|in.ftpd|wu\-?ftpd|pro[\-]?ftpd");
					KillFTPd(".*[Ff][Tt][Pp][Dd].*");
				}

				if (! WaitForFTPServerToExit()) {
					Log(0, "\n");
					Log(0, "ERROR: $ftpd was disabled in the /etc/inittab, but it is still\n"); 
					Log(0, "       accepting connections.  You need to shutdown NcFTPd, then\n");
					Log(0, "       restart the installation process.\n");
					Log(0, "\n");
					Exit(1);
				}
				return (1);		# Done -- no more NcFTPd processes are running.
			} else {
				Log(0, "Could not rewrite the /etc/inittab file: $!\n");
			}
		} else {
			Log(1, "Did not find NcFTPd or ftpd entries in the /etc/inittab.\n");
		}
	}
	return (0);	# Nothing disabled and killed
}	# DisableExistingFTPServersRunningFromInittab




sub DisableExistingNcFTPd
{
	my ($oCOLUMNS);
	my (@procs);

	#
	# This disables NcFTPd when it is not running from the
	# /etc/inittab.
	#

	$oCOLUMNS = $ENV{"COLUMNS"};
	$ENV{"COLUMNS"} = "132" if ((!defined($oCOLUMNS)) || ($oCOLUMNS < 132));
	@procs = PsGrep("respawn", "program", "long");
	$ENV{"COLUMNS"} = $oCOLUMNS if (defined($oCOLUMNS));

	if (scalar(@procs) > 0) {
		Log(1, "The \"respawn\" program is currently running.\n");
		my ($proc);
		for $proc (@procs) {
			if ($proc =~ /ncftpd/) {
				Log(0, "\n");
				Log(0, "ERROR: You appear to be using the \"respawn\" utility to launch NcFTPd:\n");
				Log(0, "\n       ", $proc, "\n\n");
				Log(0, "       You need to both kill this process *AND* make sure you disable\n");
				Log(0, "       respawn from running when the system boots.  Once that is\n");
				Log(0, "       completed, restart the installation process.\n");
				Exit(1);
			}
		}
	}
	
	$oCOLUMNS = $ENV{"COLUMNS"};
	$ENV{"COLUMNS"} = "132" if ((!defined($oCOLUMNS)) || ($oCOLUMNS < 132));
	@procs = PsGrep("ncftpd", "program", "long");
	$ENV{"COLUMNS"} = $oCOLUMNS if (defined($oCOLUMNS));

	if (scalar(@procs) > 0) {
		Log(0, "NcFTPd is currently running.\n");
		my ($proc);
		for $proc (@procs) {
			NoteExistingNcFTPdConfigurationFromCommandLine($proc);
		}
		Log(0, "Terminating the running instance of NcFTPd.\n");
		if (($systemctl ne "") && (RunProgramOrExit("$systemctl", "--quiet", "is-active", "ncftpd.service"))) {
			if (! RunProgramOrExit("$systemctl", "stop", "ncftpd.service")) {
				Log(0, "\n");
				Log(0, "Warning: NcFTPd could not be stopped using $systemctl.\n");
				Log(0, "         You may need to manually disable it from within systemd.\n");
			}
		}
		if (($initctl ne "") && (RunProgramOrExit("$initctl", "--quiet", "status", "ncftpd"))) {
			if (! RunProgramOrExit("$initctl", "stop", "ncftpd")) {
				Log(0, "\n");
				Log(0, "Warning: NcFTPd could not be stopped using $initctl.\n");
				Log(0, "         You may need to manually disable it from within systemd.\n");
			}
		}
		if (IsServiceCurrentlyRunningFromLaunchd("com.ncftp.ncftpd")) {
			if (! RunProgramOrExit("$launchctl", "stop", "ncftpd.service")) {
				Log(0, "\n");
				Log(0, "Warning: NcFTPd could not be stopped using $launchctl.\n");
				Log(0, "         You may need to manually disable it from within launchd.\n");
			}
		}
		if (KillFTPd("ncftpd") > 0) {
			Log(0, "\n");
			Log(0, "ERROR: NcFTPd respawned when it was killed.\n");
			Log(0, "       You need manually shutdown NcFTPd so it no longer respawns,\n");
			Log(0, "       and then restart the installation process.\n");
			Log(0, "\n");
			Exit(1);
		}

		if (! WaitForFTPServerToExit()) {
			Log(0, "\n");
			Log(0, "ERROR: NcFTPd was terminated, but some other process is still\n");
			Log(0, "       accepting connections.  You need to terminate those processes\n");
			Log(0, "       and prevent them from restarting at boot time.  Once that is\n");
			Log(0, "       completed, restart the installation process.\n");
			Log(0, "\n");
			Exit(1);
		}
	} else {
		Log(0, "NcFTPd is not currently running.\n");
	}
}	# DisableExistingNcFTPd




sub CheckIfSomeOtherFTPServerIsStillRunning
{
	if (IsThereAnFTPServerServing(1)) {
		Log(0, "\n");
		Log(0, "ERROR: An unknown process is still accepting FTP connections.\n"); 
		Log(0, "       You need to terminate those processes *and* prevent them\n");
		Log(0, "       from restarting at boot time.  Once that is completed,\n");
		Log(0, "       restart the installation process.\n");
		Log(0, "\n");
		Exit(1);
	}
}	# CheckIfSomeOtherFTPServerIsStillRunning




sub DisableRcFile
{
	my ($rc_script) = $_[0];
	my ($rc_file) = $rc_script;
	$rc_file =~ s/^.*\///;

	return (0) unless (-f $rc_script);
	my ($disabled_rc_script_path);
	my ($i) = 0;
	for ($i = 1; $i < 1000; $i++) {
		$disabled_rc_script_path = $rc_script;
		$disabled_rc_script_path =~ s/\/[^\/]+$//;
		$disabled_rc_script_path .= "/#DISABLED#_$rc_file";
		$disabled_rc_script_path .= ".$i" if ($i > 1);
		last if (! -f $disabled_rc_script_path);
	}
	return (RenameAndUninstallLog($rc_script, $disabled_rc_script_path));
}	# DisableRcFile




sub DisableRcDir
{
	my ($rc_dir) = @_;

	my (@rc_scripts) = glob("$rc_dir/[SKIP]*");
	return if (scalar(@rc_scripts) < 1);

	my ($rc_script, $rc_file, $script_type);
	for $rc_script (@rc_scripts) {
		next if (-d $rc_script);
		$rc_file = $rc_script;
		$rc_file =~ s/^.*\///;
		if ($rc_file =~ /(ftp|ftpd)\b/i) {
			#
			# This appears to be an rc script that
			# starts up NcFTPd or some other ftpd.
			# Disable it by renaming it with a prefix
			# that "rc" won't look for.
			#
			$script_type = "?";
			$script_type = "startup" if (substr($rc_file, 0, 1) eq "S");
			$script_type = "shutdown" if (substr($rc_file, 0, 1) eq "K");
			Log(0, "Disabling system $script_type script $rc_script.\n");
			DisableRcFile($rc_script);
		}
	}
}	# DisableRcDir




sub DisableRcScripts
{
	my (@rc_dirs) = (
		"rc0.d",
		"rc1.d",
		"rc2.d",
		"rc3.d",
		"rc4.d",
		"rc5.d",
		"rcS.d",
		"rc6.d",
	);

	my (@rc_parent_dirs) = (
		"/etc",			# Solaris
		"/etc/rc.d",		# Red Hat
		"/etc/init.d",		# ??
		"/sbin",		# HP-UX
		"/sbin/rc.d",		# ??
		"/sbin/init.d",		# SuSE
	);

	my ($rc_parent_dir, $rc_dir, $rc_dirpath);
	for $rc_parent_dir (@rc_parent_dirs) {
		for $rc_dir (@rc_dirs) {
			$rc_dirpath = "$rc_parent_dir/$rc_dir";
			if (-d $rc_dirpath) {
				DisableRcDir($rc_dirpath);
			}
		}
	}

	DisableRcFile("/System/Library/StartupItems/NcFTPd/NcFTPd");
}	# DisableRcScripts




sub NoteExistingNcFTPdVersion
{
	my ($cf, $cfdir, $n);

	if ($existing_ncftpd eq "") {
		$cf = "";
		$cf = $existing_general_cf if ($existing_general_cf ne "");
		$cf = $existing_domain_cf if ($existing_domain_cf ne "");
		if ($cf ne "") {
			$cfdir = $cf;
			$cfdir =~ s/\/[^\/]+$//;
			$n = "$cfdir/ncftpd";
			if (-x $n) {
				$existing_ncftpd = $n;
			}
		}
		if ($existing_ncftpd eq "") {
			if (-x $new_ncftpd) {
				$existing_ncftpd = $new_ncftpd;
			} else {
				$n = SearchPath("ncftpd", "", ".::$extra_paths");
				$existing_ncftpd = $n if ($n ne "");
			}
		}
	}

	if (($existing_general_cf eq "") && (-f $new_general_cf)) {
		$existing_general_cf = $new_general_cf;
		Log(1, "Found existing NcFTPd config file: $existing_general_cf.\n");
	}

	if (($existing_domain_cf eq "") && (-f $new_domain_cf)) {
		$existing_domain_cf = $new_domain_cf;
		Log(1, "Found existing NcFTPd config file: $existing_domain_cf.\n");
	}

	if (($existing_general_cf ne "") && ($existing_domain_cf eq "")) {
		$cf = $existing_general_cf;
		$cfdir = $cf;
		$cfdir =~ s/\/[^\/]+$//;
		$cf = "$cfdir/domain.cf";
		if (-f $cf) {
			$existing_domain_cf = $cf;
			Log(1, "Found existing NcFTPd config file: $existing_domain_cf.\n");
		}
	} elsif (($existing_general_cf eq "") && ($existing_domain_cf ne "")) {
		$cf = $existing_domain_cf;
		$cfdir = $cf;
		$cfdir =~ s/\/[^\/]+$//;
		$cf = "$cfdir/general.cf";
		if (-f $cf) {
			$existing_general_cf = $cf;
			Log(1, "Found existing NcFTPd config file: $existing_general_cf.\n");
		}
	} elsif (($existing_general_cf eq "") && ($existing_domain_cf eq "") && ($existing_ncftpd ne "")) {
		$cf = $existing_ncftpd;
		$cfdir = $cf;
		$cfdir =~ s/\/[^\/]+$//;
		$cf = "$cfdir/general.cf";
		if (-f $cf) {
			$existing_general_cf = $cf;
			Log(1, "Found existing NcFTPd config file: $existing_general_cf.\n");
		}
		$cf = $existing_ncftpd;
		$cfdir = $cf;
		$cfdir =~ s/\/[^\/]+$//;
		$cf = "$cfdir/domain.cf";
		if (-f $cf) {
			$existing_domain_cf = $cf;
			Log(1, "Found existing NcFTPd config file: $existing_domain_cf.\n");
		}
	}

	if ($existing_ncftpd eq "") {
		$cf = "";
		$cf = $existing_general_cf if ($existing_general_cf ne "");
		$cf = $existing_domain_cf if ($existing_domain_cf ne "");
		if ($cf ne "") {
			$cfdir = $cf;
			$cfdir =~ s/\/[^\/]+$//;
			$n = "$cfdir/ncftpd";
			if (-x $n) {
				$existing_ncftpd = $n;
			}
		}
	}

	if ($existing_ncftpd ne "") {
		my ($ver) = "";
		my ($ver2) = `$existing_ncftpd -b 2>&1`;
		$ver = $ver2 if (defined($ver2));
		if ($ver =~ /([1234]\.\d+\.\d+)/) {
			$ver = $1;
			$existing_version = $ver;
		} else {
			$ver = `strings $existing_ncftpd 2>/dev/null | fgrep '@(#) NcFTPd '`;
			if ($ver =~ /([1234]\.\d+\.\d+)/) {
				$ver = $1;
				$existing_version = $ver;
			} else {
				$ver = "unknown";
			}
		}
		Log(0, "Found existing NcFTPd (version $ver) at $existing_ncftpd.\n");
	}
}	# NoteExistingNcFTPdVersion




sub LoadExistingConfigs
{
	my ($line, $var, $val);

	if ($existing_general_cf ne "") {
		if (open(CF, "<$existing_general_cf")) {
			Log(1, "Loading existing settings from $existing_general_cf.\n");
			while (defined($line = <CF>)) {
				chomp($line);
				next unless ($line =~ /^([A-Za-z0-9_\-]+)\s*\=\s*([^\n\r]*)/);
				$var = $1;
				$val = $2;
				Log(2, "  general.cf: [$var] = [$val]\n");
			}
			close(CF);
		}
	}

	if ($existing_domain_cf ne "") {
		if (open(CF, "<$existing_domain_cf")) {
			Log(1, "Loading existing settings from $existing_domain_cf.\n");
			while (defined($line = <CF>)) {
				chomp($line);
				next unless ($line =~ /^([A-Za-z0-9_\-]+)\s*\=\s*([^\n\r]*)/);
				$var = $1;
				$val = $2;
				Log(2, "  domain.cf: [$var] = [$val]\n");
			}
			close(CF);
		}
	}
}	# LoadExistingConfigs




sub RenameAndUninstallLog
{
	my ($from_path) = $_[0];
	my ($to_path) = $_[1];

	if (rename($from_path, $to_path)) {
		Log(1, "Renamed $from_path to $to_path.\n");
		UninstallLog("Rename(\"$to_path\", \"$from_path\");\n");
		return (1);
	} else {
		Log(0, "Could not rename $from_path to $to_path: $!\n");
	}
	return (0);
}	# RenameAndUninstallLog




sub SafeRename
{
	my ($rnfm, $rnto) = @_;
	return ("") if (! -f $rnfm);

	$rnto = "$rnfm.orig" if (! defined($rnto));
	my ($ornto) = $rnto;
	my ($i);
	for ($i = 2; $i < 1000; $i++) {
		last if (! -f $rnto);
		$rnto = $ornto . "-$i";
	}
	if (RenameAndUninstallLog($rnfm, $rnto)) {
		return ($rnto);
	} else {
		return ("");
	}
}	# SafeRename




sub SafeRenameWithNcVersion
{
	my ($rnfm, $rnto) = @_;
	$rnto = "$rnfm-$existing_version" if ((! defined($rnto)) && ($existing_version ne ""));
	return (SafeRename($rnfm, $rnto));
}	# SafeRenameWithNcVersion




sub LookupWebUser
{
	my (@webusers) = ("www", "httpd", "http", "apache", "web");
	my ($xwebuser, $xwebgroup);
	my ($name,$passwd,$uid,$gid,$quota,$comment,$gcos,$dir,$shell,$expire);

	Log(1, "Checking system user database for a web server username.\n");
	for $xwebuser (@webusers) {
		($name,$passwd,$uid,$gid,$quota,$comment,$gcos,$dir,$shell,$expire) = 
			getpwnam($xwebuser);
		if (defined($name) && defined($uid) && defined($gid)) {
			$webuser = $xwebuser;
			$webuid = $uid;
			$webgid = $gid;
			$webgroup = "$gid";
			$xwebgroup = getgrgid($webgid);
			$webgroup = $xwebgroup if (defined($xwebgroup));
			Log(1, "Found web server user \"$webuser\" (UID $webuid), group \"$webgroup\" (GID $webgid) in system user database.\n");
			last;
		}
	}
}	# LookupWebUser




sub MkInstallDirs
{
	Mkdirs($bindir, 00755);
	Mkdirs($sbindir, 00755);
	Mkdirs($sysconfdir, 00755);
	Mkdirs($runtimedatadir, 00755);
	Mkdirs($logdir, 00755);

	# We don't use these yet...
	# Mkdirs($datadir, 00755);
	# Mkdirs($libdir, 00755);
	# Mkdirs($includedir, 00755);
	# Mkdirs($mandir, 00755);

	if ((defined($webgid)) && ($webgid != (-1))) {
		Mkdirs($confdir, 00750, 0, $webgid);
		Mkdirs($rptbindir, 00755, 0, $webgid);
		Mkdirs("$logdir/ncftpd", 00750, $webuid, $webgid);
	} else {
		Mkdirs($confdir, 00755);
		Mkdirs($rptbindir, 00755, 0, 0);
		Mkdirs("$logdir/ncftpd", 00750);
	}
}	# MkInstallDirs



sub FixScript
{
	my ($path, $headerline) = @_;
	my ($line);

	my ($dev, $ino, $mode, $nlink, $uid, $gid, $rdev, $size, $atime, $mtime, $ctime, $blksize, $blocks) = 	
		stat($path);
	return 0 unless (defined($mtime));

	if (open(SRC, "<$path")) {
		$line = <SRC>;
		if (! open(DST, ">$path.new")) {
			close(SRC);
			return 0;
		} else {
			print DST $headerline, "\n";
			while (defined($line = <SRC>)) {
				print DST $line;
			}
			close(DST);
			close(SRC);
			return 0 unless rename("$path.new", $path);
			chmod($mode, $path);
			utime($atime, $mtime, $path);
			return 1;
		}
	}
	return 0;
}	# FixScript




sub FixReportScripts
{
	return (0) unless (-d "${install_src_root}/rptbin");

	my (@scripts) = glob("${install_src_root}/rptbin/*.cgi");
	my ($script);
	my ($sh) = SearchPath("ksh");

	$sh = SearchPath("bash") unless ($sh ne "");
	if ($sh ne "") {
		foreach $script (@scripts) {
			FixScript($script, "#!${sh}");
		}
	}

	return (1);
}	# FixReportScripts




sub RemoveFtpFromEtcFtpusers
{
	return (0) if ($remove_ftp_from_ftpusers == 0);
	return (0) if ($existing_ncftpd ne "");	# Assume they wanted it that way

	my ($need_to_modify) = 0;
	my ($file_contents) = "";
	my ($user) = "";
	my ($disableprefix) = "#DISABLED by install_ncftpd.pl# ";

	if (open(ETCFTPUSERS, "</etc/ftpusers")) {
		while (defined($user = <ETCFTPUSERS>)) {
			chomp($user);
			if (($user =~ /^\s*#/) || ($user =~ /^\s*$/)) {
				# Keep junk line as-is.
				$file_contents .= $user . "\n";
				next;
			}
			if ($user =~ /^\s*(\S+)\s*$/) {
				if ($user ne $1) {
					Log(0, "Found extraneous whitespace in /etc/ftpusers.  Need to remove it.\n");
					$user = $1;
					$need_to_modify++;
				}
			}
			if ($user =~ /^(ftp|anonymous|_ftp)\s*$/) {
				Log(0, "Found $user in /etc/ftpusers.  Need to remove it so anonymous FTP can work.\n");
				$need_to_modify++;
				$file_contents .= $disableprefix . $user . "\n";
			} else {
				$file_contents .= $user . "\n";
			}
		}
		close(ETCFTPUSERS);
	}

	if ($need_to_modify > 0) {
		if (open(ETCFTPUSERS, ">/etc/ftpusers")) {
			print ETCFTPUSERS $file_contents;
			close(ETCFTPUSERS);
			# Modifying an existing file, so not necessary...
			# chown(0, 0, "/etc/ftpusers");
			# chmod(0644, "/etc/ftpusers");
			UninstallLog("UncommentFTPUsers();\n");
			Log(1, "Modified /etc/ftpusers successfully.\n");
		}
	}
}	# RemoveFtpFromEtcFtpusers




sub AddConfigVariable
{
	my ($newvar) = $_[0];
	my ($newval) = $_[1];
	my ($fileloc) = $_[2];
	my ($dupe_vars) = $_[3];
	my ($cffiletype) = $_[4];
	my ($cffilepath) = "";

	if ((! defined($newvar)) || ($newvar eq "") || (! defined($newval)) || ($newval eq "")) {
		return (0);
	}

	if ((defined($fileloc)) && ($fileloc =~ /^b/i)) {
		$fileloc = "bottom";
	} else {
		$fileloc = "top";
	}

	if ((defined($dupe_vars)) && ($dupe_vars =~ /^k/i)) {
		$dupe_vars = "keep";
	} elsif ((defined($dupe_vars)) && ($dupe_vars =~ /^u/i)) {
		$dupe_vars = "update";
	} else {
		$dupe_vars = "delete";
	}

	if ((defined($cffiletype)) && ($cffiletype =~ /^d/i)) {
		$cffilepath = $new_domain_cf;
	} else {
		$cffilepath = $new_general_cf;
	}

	my (@cfdata);
	my ($line, $var, $val);

	Log(1, sprintf("Adding \"%s=%s\" to %s of %s.\n", $newvar, $newval, $fileloc, $cffilepath));

	if (! open(CF, "< $cffilepath")) {
		Log(1, "Could not open: $cffilepath: $!\n");
		return (0);
	}
	@cfdata = <CF>;
	close(CF);

	my (@cfdata_new) = ();

	my ($at_top) = 1;
	my ($updates) = 0;

	foreach $line (@cfdata) {
		if ($line =~ /^\s*#/) {
			# Comment line
		} elsif ($at_top) {
			$at_top = 0;
			if ($fileloc eq "top") {
				$updates++;
				push(@cfdata_new, sprintf("%s=%s\t# (added by install_ncftpd.pl)\n", $newvar, $newval));
			}
		}
		if ($line =~ /^((#\s*)?[A-Za-z0-9_\-]+)\s*\=\s*([^\n\r]*)/) {
			$var = $1;
			$val = $3;
			if ($var eq $newvar) {
				if ($dupe_vars eq "delete") {
					# Do not include duplicate copy of this variable.
				} elsif ($dupe_vars eq "keep") {
					push(@cfdata_new, $line);
				} elsif (($dupe_vars eq "update") && ($fileloc ne "top")) {
					push(@cfdata_new, sprintf("%s=%s\t# (updated by install_ncftpd.pl)\n", $newvar, $newval));
					$updates++;
				}
			} else {
				push(@cfdata_new, $line);
			}
		} else {
			# Not a var=val line.
			push(@cfdata_new, $line);
		}
	}
	if (($fileloc eq "bottom") && ($updates == 0)) {
		push(@cfdata_new, sprintf("%s=%s\t# (added by install_ncftpd.pl)\n", $newvar, $newval));
	}

	if (! open(CF, ">$cffilepath")) {
		Exit(1, "Could not create: $cffilepath: $!\n");
	}
	print CF @cfdata_new;
	close(CF);
	# UninstallLog... currently undoable.
	return (1);
}	# AddConfigVariable




sub InstallDefaultConfigs
{
	my (@cfdata);
	my ($line, $var, $val, $oval, $ovar);

	Log(0, "Installing new NcFTPd config files.\n");

	if (! open(CF, "<$install_src_root/conf/general.cf-dist")) {
		Exit(1, "Could not open: $install_src_root/conf/general.cf-dist: $!\n");
	}
	@cfdata = <CF>;
	close(CF);

	foreach $line (@cfdata) {
		if ($line =~ /^REMOVE-THIS-LINE/) {
			$line = "# (removed by install_ncftpd.pl)\n";
			next;
		}
		next unless ($line =~ /^((#\s*)?[A-Za-z0-9_\-]+)\s*\=\s*([^\n\r]*)/);
		$var = $1;
		$val = $3;
		$oval = $val;
		$ovar = $var;
		$val =~ s-/var/log-$logdir-g;
		$val =~ s-/var/run-$runtimedatadir-g;
		$val =~ s-/home/ftp-$ftphome-g;
		$val =~ s-/usr/local/etc-$sysconfdir-g;
		$val =~ s-/usr/local/bin-$bindir-g;
		$val =~ s-/usr/local/sbin-$sbindir-g;
		$val =~ s-/usr/local-$prefix-g;
		$val = $ftpport if ($var =~ /#?port/);
		if (($var =~ /#?log-owner/) && ($webuser eq "")) {
			$var = "#log-owner";
		}
		if (($var =~ /#?log-group/) && ($webgroup eq "")) {
			$var = "#log-group";
		}
		if (($var =~ /#?log-owner/) && ($webuser ne "")) {
			$var = "log-owner";
			$val = $webuser;
		}
		if (($var =~ /#?log-group/) && ($webgroup ne "")) {
			$var = "log-group";
			$val = $webgroup;
		}
		if (($var =~ /#?log-umask/) && ($webuser ne "")) {
			$var = "log-umask";
			$val = "027";
		}
		if (($oval ne $val) || ($ovar ne $var)) {
			$line = "$var=$val\n";
			Log(2, "  set general.cf: $var=$val\n");
		}
	}

	if (! open(CF, ">$new_general_cf")) {
		Exit(1, "Could not create: $new_general_cf: $!\n");
	}
	print CF @cfdata;
	close(CF);
	Log(0, "Created $new_general_cf.\n");
	UninstallLog("Remove(\"$new_general_cf\");\n");

	#
	# Domain.cf
	#

	my ($addr, $myhostname, $myipaddr);
	my ($additionalsets) = 0;
	$myipaddr = "127.0.0.1";
	$myhostname = "localhost";
	Log(0, "Looking up IP address and hostname for this machine.\n");
	Log(2, "Running \"$install_src_root/ncftpd -e\".\n");
	$myhostname = `$install_src_root/ncftpd -e 2>/dev/null`;
	chomp($myhostname);
	Log(2, "  result = [$myhostname]\n");
	$myhostname = "localhost" if ((! defined($myhostname)) || ($myhostname eq ""));
	$addr = gethostbyname($myhostname);
	$myipaddr = inet_ntoa($addr) if (defined($addr));
	Log(2, "  ipaddr = [$myipaddr]\n");

	if (! open(CF, "<$install_src_root/conf/domain.cf-dist")) {
		Exit(1, "Could not open: $install_src_root/conf/domain.cf-dist: $!\n");
	}
	@cfdata = <CF>;
	close(CF);

	foreach $line (@cfdata) {
		if ($line =~ /^REMOVE-THIS-LINE/) {
			$line = "# (removed by install_ncftpd.pl)\n";
			next;
		}
		$additionalsets++ if ($line =~ /^#\ end\ of\ first/);
		next unless ($line =~ /^((#\s*)?[A-Za-z0-9_\-]+)\s*\=\s*([^\n\r]*)/);
		$var = $1;
		$val = $3;
		$oval = $val;
		$ovar = $var;
		$val =~ s-/var/log-$logdir-g;
		$val =~ s-/var/run-$runtimedatadir-g;
		$val =~ s-/home/ftp-$ftphome-g;
		$val =~ s-/usr/local/etc-$sysconfdir-g;
		$val =~ s-/usr/local/bin-$bindir-g;
		$val =~ s-/usr/local/sbin-$sbindir-g;
		$val =~ s-/usr/local-$prefix-g;
		$val = $ftphome if ($var =~ /#?ftp-home/);
		if ($additionalsets == 0) {
			$val = $myhostname if ($var =~ /#?set-name/);
			$val = $myhostname if ($var =~ /#?server-name/);
			$val = $myipaddr if ($var =~ /#?server-address/);
		}
		if (($oval ne $val) || ($ovar ne $var)) {
			$line = "$var=$val\n";
			Log(2, "  set domain.cf: $var=$val\n");
		}
	}

	if (! open(CF, ">$new_domain_cf")) {
		Exit(1, "Could not create: $new_domain_cf: $!\n");
	}
	print CF @cfdata;
	close(CF);
	Log(0, "Created $new_domain_cf.\n");
	UninstallLog("Remove(\"$new_domain_cf\");\n");
}	# InstallDefaultConfigs




sub InstallFile
{
	my ($srcfile, $dstdir, $dstmode, $dstuid, $dstgid, $copy_text_with_token_replacement) = @_;
	my ($srcpath, $dstpath, $dstfile);

	$dstfile = $srcfile;
	$dstfile =~ s-^.*/--;
	$srcpath = "$install_src_root/$srcfile";
	$dstpath = "$dstdir/$dstfile";

	if ($dstdir =~ /^\>(.+)$/) {
		$dstpath = $1;
	}

	my ($dev, $ino, $mode, $nlink, $uid, $gid, $rdev, $size, $atime, $mtime, $ctime, $blksize, $blocks) = 	
		stat($srcpath);
	if (! defined($mode)) {
		Exit(1, "ERROR: Could not stat $srcpath: $!\n");
		return (0);
	}

	$dstmode = $mode unless defined($dstmode);
	$dstuid = $uid unless defined($dstuid);
	$dstgid = $gid unless defined($dstgid);

	SafeRenameWithNcVersion($dstpath);
	if (defined($copy_text_with_token_replacement) && ($copy_text_with_token_replacement)) {
		if (! open(SRCPATH, "< $srcpath")) {
			Exit(1, "ERROR: Could not open $srcpath (and then copy it to $dstpath): $!\n");
			return (0);
		}
		if (! open(DSTPATH, "> $dstpath")) {
			close(SRCPATH);
			Exit(1, "ERROR: Could not open $dstpath for writing (and then copy into it the contents of $srcpath): $!\n");
			return (0);
		}
		while (defined(my $line = <SRCPATH>)) {
			$line =~ s!%PREFIX%!$prefix!ig;
			$line =~ s!%EPREFIX%!$eprefix!ig;
			$line =~ s!%BINDIR%!$bindir!ig;
			$line =~ s!%SBINDIR%!$sbindir!ig;
			$line =~ s!%DATADIR%!$datadir!ig;
			$line =~ s!%SYSCONFDIR%!$sysconfdir!ig;
			$line =~ s!%RUNTIMEDATADIR%!$runtimedatadir!ig;
			$line =~ s!%LOGDIR%!$logdir!ig;
			$line =~ s!%LIBDIR%!$libdir!ig;
			$line =~ s!%INCLUDEDIR%!$includedir!ig;
			$line =~ s!%MANDIR%!$mandir!ig;
			$line =~ s!%CONFDIR%!$confdir!ig;
			$line =~ s!%RPTBINDIR%!$rptbindir!ig;
			$line =~ s!%INSTALL_SRC_ROOT%!$install_src_root!ig;
			$line =~ s!%OS%!$OS!ig;
			$line =~ s!%SYS%!$SYS!ig;
			$line =~ s!%PERL%!$perl!ig;
			$line =~ s!%LSOF%!$lsof!ig;
			$line =~ s!%SYSTEMD_SERVICES_DIR%!$systemd_services_dir!ig;
			$line =~ s!%LAUNCHD_JOBS_DIR%!$launchd_jobs_dir!ig;
			$line =~ s!%UPSTART_JOBS_DIR%!$upstart_jobs_dir!ig;
			print DSTPATH $line;
		}
		close(SRCPATH);
		close(DSTPATH);
	} elsif (! copy($srcpath, $dstpath)) {
		Exit(1, "ERROR: Could not copy $srcpath to $dstpath: $!\n");
		return (0);
	}
	chmod($dstmode, $dstpath);
	chown($dstuid, $dstgid, $dstpath);
	utime($atime, $mtime, $dstpath);

	UninstallLog("Remove(\"$dstpath\");\n");
	Log(1, "Installed $dstpath.\n");
	return (1);
}	# InstallFile




sub Symlink
{
	my ($srcpath, $dstpath, $force) = @_;

	if ((-l $dstpath) || (-e $dstpath)) {
		if ($force eq "overwrite") {
			if (! unlink($dstpath)) {
				Log(1, "Could not remove existing $dstpath: $!\n");
				# But keep going
			}
		} elsif ($force eq "rename") {
			return 0 if (SafeRenameWithNcVersion($dstpath) eq "");
		} else {
			Log(1, "Warning: could not link $srcpath -> $dstpath: $dstpath already exists\n");
			return 0;
		}
	}
	if (! symlink($srcpath, $dstpath)) {
		Log(0, "Warning: could not link $srcpath -> $dstpath: $!\n");
		return 0;
	} else {
		Log(1, "  Linked $srcpath -> $dstpath.\n");
		UninstallLog("Remove(\"$dstpath\");\n");
	}
	return 1;
}	# Symlink



sub InstallConfigs
{
	my ($oldg, $oldd);
	my ($install_default_configs) = 1;

	Log(0, "Installing configuration files.\n");
	$oldg = SafeRenameWithNcVersion($new_general_cf);
	$oldd = SafeRenameWithNcVersion($new_domain_cf);
	if (($oldg ne "") && ($oldd ne "") && (! $create_new_configs)) {
		if (! copy($oldg, $new_general_cf)) {
			Log(0, "ERROR: Could not copy $oldg to $new_general_cf: $!\n");
		} else {
			UninstallLog("Rename(\"$oldg\", \"$new_general_cf\");\n");
		}
		if ($webgid != (-1)) {
			chmod(0640, $new_general_cf);
			chown(0, $webgid, $new_general_cf);
		} else {
			chmod(0600, $new_general_cf);
			chown(0, 0, $new_general_cf);
		}

		if (! copy($oldd, $new_domain_cf)) {
			Log(0, "ERROR: Could not copy $oldd to $new_domain_cf: $!\n");
		} else {
			UninstallLog("Rename(\"$oldd\", \"$new_domain_cf\");\n");
		}
		if ($webgid != (-1)) {
			chmod(0640, $new_domain_cf);
			chown(0, $webgid, $new_domain_cf);
		} else {
			chmod(0600, $new_domain_cf);
			chown(0, 0, $new_domain_cf);
		}
		if ((-f $new_general_cf) && (-f $new_domain_cf)) {
			Log(1, "Using copies of existing NcFTPd config files.\n");
			$install_default_configs = 0;
		} else {
			Log(0, "Existing NcFTPd config files could not be copied.\n");
		}
	}

	InstallDefaultConfigs() if ($install_default_configs);

	if ($run_ncftpd_verbose > 0) {
		if (AddConfigVariable("verbose", $run_ncftpd_verbose, "top", "delete-duplicate-var", "general.cf")) {
			$run_ncftpd_verbose = 0;
		}
	}
}	# InstallConfigs




sub InstallPrograms
{
	my ($binuid, $bingid);
	my ($name,$passwd,$uid,$gid,$quota,$comment,$gcos,$dir,$shell,$expire);

	$binuid = 0;
	$bingid = 0;

	($name,$passwd,$uid,$gid,$quota,$comment,$gcos,$dir,$shell,$expire) = 
		getpwnam("bin");
	if (defined($uid)) {
		$binuid = $uid;
		$bingid = $gid;
	}

	Log(0, "Installing programs.\n");
	InstallFile("ncftpd", $sbindir, 0751, $binuid, $bingid);
	InstallFile("ncftpd_spy", $sbindir, 0700, $binuid, $bingid);
	InstallFile("ncftpd_passwd", $sbindir, 0751, $binuid, $bingid);
	InstallFile("ncftpd_edquota", $sbindir, 0751, $binuid, $bingid);
	InstallFile("ncftpd_repquota", $sbindir, 0751, $binuid, $bingid);
	InstallFile("extra/pgrep.pl", $bindir, 0755, $binuid, $bingid);
	Symlink("$bindir/pgrep.pl", "$bindir/pkill.pl", "overwrite");
	InstallFile("extra/uninstall_ncftpd.pl", $confdir, 0700, 0, $bingid);
	$uninstaller = "$confdir/uninstall_ncftpd.pl";
}	# InstallPrograms




sub InstallStartupScript
{
	my (@initscript);
	my (@inittabscript);
	my (@startupitemscript);
	my ($initscriptpath);
	my ($inittabscriptpath);
	my ($inittabscript_redirect);
	my ($runlevels);
	my ($rc_local);
	my ($rc_local_already);
	my ($line, $nlines);
	my ($rc_link, @rc_links);
	my ($var, $val, %vars, $newval);
	my ($insserv);
	my ($bash) = "";
	my (@scripts_to_fix) = ();

	Log(0, "Configuring your system so NcFTPd starts automatically at system boot.\n");

	# Be a little bit conservative and only look in /bin and /usr/bin,
	# which should be mounted at startup.
	#
	$bash = "/bin/bash" if (-x "/bin/bash");
	$bash = "/usr/bin/bash" if (-x "/usr/bin/bash");

	if ($use_systemd) {
		if (! open(SCRIPT, "<$install_src_root/extra/ncftpd.systemd.init")) {
			Exit(1, "Could not open $install_src_root/extra/ncftpd.systemd.init: $!\n");
		}
	} elsif ($use_upstart) {
		if (! open(SCRIPT, "<$install_src_root/extra/ncftpd.upstart.init")) {
			Exit(1, "Could not open $install_src_root/extra/ncftpd.upstart.init: $!\n");
		}
	} elsif ($use_launchd) {
		if (! open(SCRIPT, "<$install_src_root/extra/ncftpd.launchd.init")) {
			Exit(1, "Could not open $install_src_root/extra/ncftpd.launchd.init: $!\n");
		}
	} elsif (! open(SCRIPT, "<$install_src_root/extra/ncftpd.init")) {
		Exit(1, "Could not open $install_src_root/extra/ncftpd.init: $!\n");
	}
	@initscript = <SCRIPT>;
	close(SCRIPT);

	if (! open(SCRIPT, "<$install_src_root/extra/ncftpd.sh")) {
		Exit(1, "Could not open $install_src_root/extra/ncftpd.sh: $!\n");
	}
	@inittabscript = <SCRIPT>;
	close(SCRIPT);

	%vars = ();
	$vars{"NCFTPD_PREFIX"} = $prefix;
	$vars{"NCFTPD_GENERAL_CF"} = $new_general_cf;
	$vars{"NCFTPD_DOMAIN_CF"} = $new_domain_cf;
	$vars{"NCFTPD_STARTUP_ERROR_LOG"} = $startup_error_log;
	$vars{"NCFTPD_PROG_PATH"} = "$sbindir:$bindir:\$PATH";
	$vars{"NCFTPD_PROG"} = "$new_ncftpd";
	if ($run_ncftpd_verbose > 0) {
		$val = "-v";
		$val = "-vv" if ($run_ncftpd_verbose == 2);
		$val = "-vvv" if ($run_ncftpd_verbose == 3);
		$vars{"NCFTPD_VERBOSITY"} = "$val";
	}

	$nlines = 0;
	foreach $line (@initscript) {
		$nlines++;
		if ($use_systemd) {
			# No header needed
		} elsif ($use_upstart) {
			# No header needed
		} elsif ($use_launchd) {
			# No header needed
		} elsif (($nlines == 1) && ($linux_distro eq "SuSE") && ($linux_distro_version_num > 7)) {
			$line .= "#\n";
			$line .= "### BEGIN INIT INFO\n";
			$line .= "# Provides: ncftpd\n";
			$line .= "# Required-Start: \$network \$remote_fs \$syslog \$named\n";
			$line .= "# Required-Stop: \$network \$remote_fs \$syslog \$named\n";
			$line .= "# Default-Start: 3 5\n";
			$line .= "# Default-Stop: 0 1 2 6\n";
			$line .= "# Short-Description: NcFTPd FTP Server\n";
			$line .= "# Description: NcFTPd File Transfer Protocol (FTP) server daemon\n";
			$line .= "#\tSee the online documentation at http://www.ncftp.com/ncftpd/doc/\n";
			$line .= "### END INIT INFO\n";
			$line .= "#\n";
		} elsif (($nlines == 1) && (-f "/sbin/chkconfig")) {
			$line .= "#\n";
			$line .= "# chkconfig: 345 85 15\n";
			$line .= "# description: Starts and stops NcFTPd (FTP) service\n";
			$line .= "# processname: ncftpd\n";
			$line .= "# config: $new_general_cf\n";
			$line .= "# config: $new_domain_cf\n";
			$line .= "#\n";
		}
		for $var (keys %vars) {
			if ($line =~ /^\s*$var\=\"?([^\"]*)\"?/) {
				$val = $1;
				$newval = $vars{$var};
				if ($newval ne $val) {
					chomp($line);
					$line = "#Configured by install_ncftpd.pl# $line\n";
					$line .= "$var=\"$newval\"\n";
				}
			}
		}
		if ($line =~ /#SOURCE\-RC/) {
			if ($SYS eq "hpux") {
				$line = "PATH=/usr/sbin:/sbin:/bin:/usr/bin\nexport PATH\n";
			} elsif ($SYS eq "linux") {
				if (($linux_distro eq "SuSE") && ($linux_distro_version_num > 7)) {
					$line = ". /etc/rc.status\n\n";
					$line .= "# Reset status of this service\n";
					$line .= "rc_reset\n\n";
				} elsif ($linux_distro eq "SuSE") {
					$line = ". /etc/rc.config\n";
				} else {
					$line = "";
					if (-f "/etc/sysconfig/network") {
						if (-f "/etc/init.d/functions") {
							$line .= "# Source function library.\n";
							$line .= ". /etc/init.d/functions\n";
							$line .= "\n";
						} elsif (-f "/etc/rc.d/init.d/functions") {
							$line .= "# Source function library.\n";
							$line .= ". /etc/rc.d/init.d/functions\n";
							$line .= "\n";
						}
						if ((-f "/etc/sysconfig/network") && (! $use_systemd) && (! $use_upstart) && (! $use_launchd)) {
							$line .= "# Get config.\n";
							$line .= ". /etc/sysconfig/network\n";
							$line .= "\n";
							$line .= "# Check that networking is up.\n";
							$line .= "if [ \"X\${NETWORKING}\" = \"Xno\" ]\n";
							$line .= "then\n";
							$line .= "	exit 0\n";
							$line .= "fi\n";
							$line .= "\n";
						}
					}
				}
			} else {
				$line = "";
			}
		} elsif ($line =~ /^PKILL\=/) {
			if (-x "/usr/bin/pkill") {
				$line = "PKILL=\"/usr/bin/pkill\"\n";
			} else {
				$line = "PKILL=\"${bindir}/pkill.pl\"\n";
			}
		} elsif ($line =~ /^PGREP\=/) {
			if (-x "/usr/bin/pgrep") {
				$line = "PGREP=\"/usr/bin/pgrep\"\n";
			} else {
				$line = "PGREP=\"${bindir}/pgrep.pl\"\n";
			}
		} elsif ($line =~ /ECHO\-N/) {
			chomp($line);
			$line =~ s/\s*#\s*ECHO.*$//;
			if ($SYS =~ /(aix|hpux|solaris)/) {
				$line =~ s/echo\s\-n/echo/;
				$line .= "\'\\c\'";
			}
			$line .= "\n";
		} elsif ($line =~ /ECHO\-E/) {
			chomp($line);
			if (($linux_distro eq "SuSE") && ($linux_distro_version_num > 7)) {
				if ($line =~ /rc_done/) {
					$line =~ s/echo.*/rc_status -v/;
				} elsif ($line =~ /rc_failed/) {
					$line =~ s/echo.*/rc_failed \"\$es\"/;
				}
			}
			$line =~ s/\s*#\s*ECHO.*$//;
			unless ($SYS eq "linux") {
				$line =~ s/echo\s\-e/echo/;
			}
			$line .= "\n";
		} elsif ($line =~ /EXIT/) {
			if (($linux_distro eq "SuSE") && ($linux_distro_version_num > 7)) {
				$line =~ s/exit.*/rc_exit/;
			}
		} elsif ($line =~ /#\s*Successfully\s*backgrounded/) {
			if ($SYS eq "hpux") {
				$line .= "\t\t\tes=4	# HP-UX rc script exit status for backgrounding\n";
			}
		} elsif ($line =~ /^#EXTRA/) {
			$line = "";
			if ($SYS eq "hpux") {
				$line .= "\n";
				$line .= "\'start_msg\')\n";
				$line .= "	echo \'Start NcFTPd\'\n";
				$line .= "	es=0\n";
				$line .= "	;;\n";
				$line .= "\n";
				$line .= "\'stop_msg\')\n";
				$line .= "	echo \'Stopping NcFTPd\'\n";
				$line .= "	es=0\n";
				$line .= "	;;\n";
			}
		}
	}

	foreach $line (@inittabscript) {
		for $var (keys %vars) {
			if ($line =~ /^\s*$var\=\"?([^\"]*)\"?/) {
				$val = $1;
				$newval = $vars{$var};
				if ($newval ne $val) {
					chomp($line);
					$line = "#Configured by install_ncftpd.pl# $line\n";
					$line .= "$var=\"$newval\"\n";
				}
			}
		}
	}

	@rc_links = ();
	$initscriptpath = "";
	$inittabscriptpath = "";
	$rc_local = "";
	$rc_local_already = 0;
	$runlevels = "2345";
	$inittabscript_redirect = "";
	$insserv = "";

	if ((-f "/etc/insserv.conf") && (-d "/etc/init.d"))  {
		# insserv is used by SuSE 10
		if (-x "/sbin/insserv") {
			$insserv = "/sbin/insserv";
		} elsif (-x "/usr/sbin/insserv") {
			$insserv = "/usr/sbin/insserv";
		}
	}

	if ($insserv ne "") {
		@rc_links = ();
		# Note: We use "NcFTPd" rather than "ncftpd" for the
		# initscript name, so that we can do a "pkill -x ncftpd"
		# and not terminate the initscript.
		#
		$initscriptpath = "/etc/init.d/NcFTPd";
	} elsif ($SYS eq "linux") {
		if ($linux_distro eq "SuSE") {
			if ($linux_distro_version_num > 7) {
				# SuSE 8.x
				@rc_links = (
					"/etc/init.d/rc2.d/S21ncftpd",
					"/etc/init.d/rc2.d/K21ncftpd",
					"/etc/init.d/rc3.d/S21ncftpd",
					"/etc/init.d/rc3.d/K21ncftpd",
					"/etc/init.d/rc5.d/S21ncftpd",
					"/etc/init.d/rc5.d/K21ncftpd",
				);
				if (-d "/etc/init.d") {
					$initscriptpath = "/etc/init.d/NcFTPd";
				} else {
					$initscriptpath = "/etc/NcFTPd.init";
				}
			} elsif (-d "/etc/init.d") {
				if (-d "/etc/rc2.d") {
					# ??
					@rc_links = (
						"/etc/rc2.d/S21ncftpd",
						"/etc/rc2.d/K21ncftpd",
						"/etc/rc3.d/S21ncftpd",
						"/etc/rc3.d/K21ncftpd",
					);
				} else {
					# SuSE 7.x
					@rc_links = (
						"/etc/init.d/rc2.d/S21ncftpd",
						"/etc/init.d/rc2.d/K21ncftpd",
						"/etc/init.d/rc3.d/S21ncftpd",
						"/etc/init.d/rc3.d/K21ncftpd",
					);
				}
				$initscriptpath = "/etc/init.d/NcFTPd";
			} else {
				# SuSE 6.x
				@rc_links = (
					"/sbin/init.d/rc2.d/S21ncftpd",
					"/sbin/init.d/rc2.d/K21ncftpd",
					"/sbin/init.d/rc3.d/S21ncftpd",
					"/sbin/init.d/rc3.d/K21ncftpd",
				);
				$initscriptpath = "/sbin/init.d/NcFTPd";
			}
		} elsif ($linux_distro =~ /(Red\ Hat|CentOS|Mandrake|Mandriva|Conectiva|TurboLinux|Caldera|Cobalt|LinuxPPC|Fedora\ Core)/) {

			if (-d "/etc/rc2.d") {
				@rc_links = (
					"/etc/rc2.d/S85ncftpd",
					"/etc/rc3.d/S85ncftpd",
					"/etc/rc5.d/S85ncftpd",
				);
			} else {
				@rc_links = (
					"/etc/rc.d/rc2.d/S85ncftpd",
					"/etc/rc.d/rc3.d/S85ncftpd",
					"/etc/rc.d/rc5.d/S85ncftpd",
				);
			}
			if ((-d "/etc/init.d") && (! -d "/etc/rc.d/init.d")) {
				$initscriptpath = "/etc/init.d/NcFTPd";
			} elsif (-d "/etc/rc.d/init.d") {
				$initscriptpath = "/etc/rc.d/init.d/NcFTPd";
			} else {
				$initscriptpath = "/etc/NcFTPd.init";
			}
		} elsif ($linux_distro eq "Slackware") {
			if (-d "/etc/rc.d") {
				$initscriptpath = "/etc/rc.d/rc.NcFTPd";
			} else {
				$initscriptpath = "/etc/NcFTPd.init";
			}
			$rc_local = "/etc/rc.d/rc.local";
		} elsif ($linux_distro_class eq "Debian") {
			@rc_links = (
				"/etc/rc0.d/K21ncftpd",
				"/etc/rc1.d/K21ncftpd",
				"/etc/rc2.d/S21ncftpd",
				"/etc/rc3.d/S21ncftpd",
				"/etc/rc4.d/S21ncftpd",
				"/etc/rc5.d/S21ncftpd",
				"/etc/rc6.d/K21ncftpd",
				"/etc/rcS.d/K21ncftpd",
			);
			if (-d "/etc/init.d") {
				$initscriptpath = "/etc/init.d/rc.NcFTPd";
			} else {
				$initscriptpath = "/etc/NcFTPd.init";
			}
		} else {
			Log(0, "This script has not been customized for your Linux distribution.\n");
			if ((-f "/etc/inittab") && (! -d "/etc/init") && (! $use_systemd) && (! $use_upstart) && (! $use_launchd)) {
				$inittabscriptpath = "$confdir/ncftpd.inittab.sh";
				Log(0, "  NcFTPd will be installed into the /etc/inittab, rather than use an rc script,\n");
				Log(0, "  which should work OK.\n");
			} else {
				# Missing inittab, or running Upstart/Systemd which ignores inittab file.
				$initscriptpath = "$confdir/ncftpd.init";
				Log(0, "  You will need to manually install a startup script so NcFTPd will be\n");
				Log(0, "  automatically started at boot time.  A sample startup script will be\n");
				Log(0, "  installed as $initscriptpath.\n");
				Log(0, "  If your system uses rc.d directories, you can simply create symbolic links\n");
				Log(0, "  to $initscriptpath, otherwise you can\n");
				Log(0, "  check to see if there is an \"rc.local\" script you can use to run\n");
				Log(0, "  $initscriptpath.\n");
			}
		}
	} elsif ($SYS eq "freebsd") {
		if (-d "/usr/local/etc/rc.d") {
			$initscriptpath = "/usr/local/etc/rc.d/NcFTPd.sh";
		} else {
			$initscriptpath = "/etc/rc.NcFTPd";
			$rc_local = "/etc/rc.local";
		}
	} elsif ($SYS eq "macosx") {
		#
		# Special setup for Mac OS X.
		# OS X uses a non-traditional way to run startup scripts.
		#
		# We simply choose to extract a pre-built tar which creates
		# the startup scripts in /System/Library/StartupItems.
		#
		if ($use_launchd == 0) {
			if ($verbose > 1) {
				Log(1, "/usr/bin/tar -x -C / -vpf \"$install_src_root/extra/osx_supplement.tar\"\n");
				RunProgramIfPresent("/usr/bin/tar", "-x", "-C", "/", "-vpf", "$install_src_root/extra/osx_supplement.tar");
			} else {
				RunProgramIfPresent("/usr/bin/tar", "-x", "-C", "/", "-pf", "$install_src_root/extra/osx_supplement.tar");
			}

			if (! open(SCRIPT, "</System/Library/StartupItems/NcFTPd/NcFTPd")) {
				Exit(1, "Could not open /System/Library/StartupItems/NcFTPd/NcFTPd: $!\n");
			}
			@startupitemscript = <SCRIPT>;
			close(SCRIPT);
			UninstallLog("Remove(\"/System/Library/StartupItems/NcFTPd/NcFTPd\");\n");
			UninstallLog("Remove(\"/System/Library/StartupItems/NcFTPd/StartupParameters.plist\");\n");
			UninstallLog("Rmdir(\"/System/Library/StartupItems/NcFTPd\");\n");

			foreach $line (@startupitemscript) {
				for $var (keys %vars) {
					if ($line =~ /^\s*$var\=\"?([^\"]*)\"?/) {
						$val = $1;
						$newval = $vars{$var};
						if ($newval ne $val) {
							chomp($line);
							$line = "    $var=\"$newval\"\t# from install_ncftpd.pl\n";
						}
					}
				}
			}

			if (! open(SCRIPT, ">/System/Library/StartupItems/NcFTPd/NcFTPd")) {
				Exit(1, "Could not rewrite /System/Library/StartupItems/NcFTPd/NcFTPd: $!\n");
			}
			print SCRIPT @startupitemscript;
			close(SCRIPT);
		}

		#
		# But copy the standard init script to a suitable location,
		# so they can use it to restart NcFTPd.
		#
		$initscriptpath = "$sbindir/ncftpd.init";
	} elsif ($SYS eq "sco") {
		@rc_links = (
			"/etc/rc2.d/S92ncftpd",
		);
		$initscriptpath = "/etc/init.d/NcFTPd";
	} elsif (($SYS eq "unixware") || ($SYS eq "openunix")) {
		@rc_links = (
			"/etc/rc2.d/S75ncftpd",
		);
		$initscriptpath = "/etc/init.d/NcFTPd";
	} elsif ($SYS eq "aix") {
		$inittabscriptpath = "/etc/rc.ncftpd";
		$inittabscript_redirect = ' >/dev/console 2>&1';
		$runlevels = "2";
	} elsif ($SYS eq "solaris") {
		@rc_links = (
			"/etc/rc2.d/S75ncftpd",
		);
		$initscriptpath = "/etc/init.d/NcFTPd";
	} elsif ($SYS =~ /^irix/) {
		@rc_links = (
			"/etc/rc2.d/S65ncftpd",
		);
		$initscriptpath = "/etc/init.d/NcFTPd";
	} elsif ($SYS eq "tru64unix") {
		@rc_links = (
			"/sbin/rc0.d/K62ncftpd",
			"/sbin/rc2.d/K62ncftpd",
			"/sbin/rc3.d/S62ncftpd",
		);
		$initscriptpath = "/sbin/init.d/NcFTPd";
	} elsif ($SYS =~ /(openbsd|netbsd|bsdos|sunos)/) {
		$initscriptpath = "/etc/rc.ncftpd";
		$rc_local = "/etc/rc.local";
	} elsif ($SYS eq "hpux") {
		@rc_links = (
			"/sbin/rc1.d/K517ncftpd",
			"/sbin/rc2.d/S517ncftpd",
		);
		$initscriptpath = "/sbin/init.d/NcFTPd";
	} elsif (-f "/etc/inittab") {
		$inittab_preferred = 1;
		Log(0, "Warning: Unknown system type ($SYS)!  Startup script was not installed.\n");
		Log(0, "         NcFTPd will be run from the /etc/inittab instead.\n");
	} elsif (-f "/etc/rc.local") {
		$initscriptpath = "/etc/rc.ncftpd";
		$rc_local = "/etc/rc.local";
		Log(0, "Warning: Unknown system type ($SYS)!  Startup script was not installed.\n");
		Log(0, "         NcFTPd will be run from the /etc/rc.local instead.\n");
	} else {
		Log(0, "ERROR: Unknown system type ($SYS)!  Startup script was not installed.\n");
		Log(0, "       You will need to start NcFTPd manually and arrange for it to be\n");
		Log(0, "       started when the system boots.\n");
		return;
	}

	if ($inittab_preferred) {
		$inittabscriptpath = "$confdir/ncftpd.inittab.sh";
		if ($initscriptpath ne "") {
			Log(1, "Using $inittabscriptpath instead of $initscriptpath.\n");
		}
	}

	if ($inittabscriptpath ne "") {
		SafeRenameWithNcVersion($inittabscriptpath);
		if (! open(SCRIPT, ">$inittabscriptpath")) {
			Exit(1, "Could not create $inittabscriptpath: $!\n");
		} else {
			print SCRIPT @inittabscript;
			close(SCRIPT);
		}
		chmod(0755, $inittabscriptpath);
		Log(0, "Installed $inittabscriptpath.\n");
		UninstallLog("Remove(\"$inittabscriptpath\");\n");
		push (@scripts_to_fix, $inittabscriptpath);

		unlink("$sbindir/restart_ncftpd");
		if (open(RESTARTSCRIPT, ">$sbindir/restart_ncftpd")) {
			print RESTARTSCRIPT "#!/bin/sh\n";
			print RESTARTSCRIPT "NCFTPD_STOP_FILE=\"$confdir/stop\"\n";
			print RESTARTSCRIPT "NCFTPD_STOP_SCRIPT=\`grep \'^pid-file\' \"$new_general_cf\" \| cut -d= -f2\`\n";
			print RESTARTSCRIPT "es=1\n";
			print RESTARTSCRIPT "if [ -f \"\$NCFTPD_STOP_FILE\" ] ; then\n";
			print RESTARTSCRIPT "		echo \'Restarting NcFTPd.\'\n";
			print RESTARTSCRIPT "		/bin/rm \"\$NCFTPD_STOP_FILE\"\n";
			print RESTARTSCRIPT "		es=\$?\n";
			print RESTARTSCRIPT "fi\n";
			print RESTARTSCRIPT "if [ -x \"\$NCFTPD_STOP_SCRIPT\" ] ; then\n";
			print RESTARTSCRIPT "		echo \'Restarting NcFTPd.\'\n";
			print RESTARTSCRIPT "		/bin/sh \"\$NCFTPD_STOP_SCRIPT\"\n";
			print RESTARTSCRIPT "		es=0\n";
			print RESTARTSCRIPT "fi\n";
			print RESTARTSCRIPT "exit \$es\n";
			close(RESTARTSCRIPT);
			chmod(0700, "$sbindir/restart_ncftpd");
			Log(0, "Created $sbindir/restart_ncftpd.\n");
			UninstallLog("Remove(\"$sbindir/restart_ncftpd\");\n");
		}

		unlink("$sbindir/stop_ncftpd");
		if (open(STOPSCRIPT, ">$sbindir/stop_ncftpd")) {
			print STOPSCRIPT "#!/bin/sh\n";
			print STOPSCRIPT "NCFTPD_STOP_FILE=\"$confdir/stop\"\n";
			print STOPSCRIPT "NCFTPD_STOP_SCRIPT=\`grep \'^pid-file\' \"$new_general_cf\" \| cut -d= -f2\`\n";
			print STOPSCRIPT "date > \"\$NCFTPD_STOP_FILE\"\n";
			print STOPSCRIPT "if [ -x \"\$NCFTPD_STOP_SCRIPT\" ] ; then\n";
			print STOPSCRIPT "		echo \'Stopping NcFTPd.\'\n";
			print STOPSCRIPT "		/bin/sh \"\$NCFTPD_STOP_SCRIPT\"\n";
			print STOPSCRIPT "		exit 0\n";
			print STOPSCRIPT "fi\n";
			print STOPSCRIPT "exit 1\n";
			close(STOPSCRIPT);
			chmod(0700, "$sbindir/stop_ncftpd");
			Log(0, "Created $sbindir/stop_ncftpd.\n");
			UninstallLog("Remove(\"$sbindir/stop_ncftpd\");\n");
		}

		#
		# Now add our entry to the /etc/inittab.
		# When we "init q", NcFTPd will start up.
		#
		if (open(ETCINITTAB, ">>/etc/inittab")) {
			print ETCINITTAB "#\n# Added by install_ncftpd.pl\n#\n";
			print ETCINITTAB "nc:$runlevels:respawn:$inittabscriptpath$inittabscript_redirect\n";
			close(ETCINITTAB);
			Log(1, "Added entry to /etc/inittab.\n");
			UninstallLog("RemoveInittabEntry(\"nc:$runlevels:respawn:$inittabscriptpath$inittabscript_redirect\");\n");
		}
		$inittab_startup_script = $inittabscriptpath;
	} elsif ($initscriptpath ne "") {
		SafeRenameWithNcVersion($initscriptpath);
		if (! open(SCRIPT, ">$initscriptpath")) {
			Exit(1, "Could not create $initscriptpath: $!\n");
		} else {
			print SCRIPT @initscript;
			close(SCRIPT);
		}
		chmod(0755, $initscriptpath);
		Log(1, "Installed $initscriptpath.\n");
		UninstallLog("Remove(\"$initscriptpath\");\n");
		push (@scripts_to_fix, $initscriptpath);

		if (($use_systemd) || ($use_upstart) || ($use_launchd)) {
			@rc_links = ();
		}

		push(@rc_links, "$sbindir/restart_ncftpd");
		push(@rc_links, "$sbindir/stop_ncftpd");
		for $rc_link (@rc_links) {
			Symlink($initscriptpath, $rc_link, "overwrite");
		}
		$rc_startup_script = $initscriptpath;

		if ($insserv ne "") {
			Log(1, "Running $insserv to activate startup script in rc.d directories.\n");
			if (RunProgramOrExit($insserv, "NcFTPd")) {
				# Assumes it works...
				UninstallLog("system(\"$insserv\", \"-r\", \"NcFTPd\");\n");
			}
		}

		if ($use_systemd) {
			UninstallLog("system(\"$systemctl\", \"daemon-reload\");\n");
			InstallFile("extra/ncftpd.service", $systemd_services_dir, 0644, 0, 0, 1);
			RunProgramOrExit("$systemctl", "daemon-reload");
			if (! RunProgramOrExit("$systemctl", "--quiet", "enable", "ncftpd.service")) {
				Exit(1, "Could not enable ncftpd.service in systemd.\n");
			}
			UninstallLog("system(\"$systemctl\", \"disable\", \"ncftpd.service\");\n");
			# Installed service file, but do not start up just yet.
		} elsif ($use_upstart) {
			UninstallLog("system(\"$initctl\", \"reload-configuration\");\n");
			InstallFile("extra/ncftpd.upstart.conf", ">$upstart_jobs_dir/ncftpd.conf", 0644, 0, 0, 1);
			if (! RunProgramOrExit("$initctl", "--quiet", "reload-configuration")) {
				Exit(1, "Could not enable $upstart_jobs_dir/ncftpd.conf in upstart.\n");
			}
			# InstallFile does: UninstallLog("Remove(\"$upstart_jobs_dir/ncftpd.conf\");\n");
			# Installed service file, but do not start up just yet.
		} elsif ($use_launchd) {
			InstallFile("extra/com.ncftp.ncftpd.plist", $launchd_jobs_dir, 0644, 0, 0, 1);
			if (! RunProgramOrExit("$launchctl", "load", "-w", "$launchd_jobs_dir/com.ncftp.ncftpd.plist")) {
				Exit(1, "Could not load com.ncftp.ncftpd.plist in launchd.\n");
			}
			UninstallLog("system(\"$launchctl\", \"unload\", \"-w\", \"$launchd_jobs_dir/com.ncftp.ncftpd.plist\");\n");
			# Installed plist file, but do not start up just yet.
		}
	}

	if (($rc_local ne "") && ($rc_startup_script ne "")) {
		if (open(RCLOCAL, "<$rc_local")) {
			while (defined($line = <RCLOCAL>)) {
				if ($line =~ /^[^#]*$rc_startup_script/) {
					$rc_local_already++;
					Log(1, "Already found $rc_startup_script in $rc_local.\n");
					last;
				}
			}
			close(RCLOCAL);
		}
		if (($rc_local_already == 0) && (open(RCLOCAL, ">>$rc_local"))) {
			if (($SYS eq "freebsd") && (! -s $rc_local)) {
				print RCLOCAL "#!/bin/sh\n\n. /etc/rc.conf\n\n";
			}
			print RCLOCAL "#\n# Added by install_ncftpd.pl\n#\n";
			print RCLOCAL "$rc_startup_script start\n";
			close(RCLOCAL);
			chmod(0755, $rc_local);
			Log(1, "Added entry to $rc_local.\n");
			UninstallLog("RemoveRcLocalEntry(\"$rc_local\", \"$rc_startup_script start\");\n");
		}
	}

	if ($bash ne "") {
		foreach my $script (@scripts_to_fix) {
			if (-e $script) {
				FixScript($script, "#!${bash}");
			}
		}
	}
}	# InstallStartupScript




sub StartNcFTPd
{
	if ($use_systemd) {
		Log(0, "Starting NcFTPd.\n");
		if (! RunProgramOrExit("$systemctl", "start", "ncftpd.service")) {
			Exit(1, "Could not start ncftpd.service via Systemd ($systemctl).\n");
		} else {
			UninstallLog("system(\"$systemctl\", \"stop\", \"ncftpd.service\");\n");
		}
	} elsif ($use_upstart) {
		Log(0, "Starting NcFTPd.\n");
		if (! RunProgramOrExit("$initctl", "start", "ncftpd")) {
			Exit(1, "Could not start ncftpd via Upstart ($initctl).\n");
		} else {
			UninstallLog("system(\"$initctl\", \"stop\", \"ncftpd\");\n");
		}
	} elsif ($use_launchd) {
		Log(0, "Starting NcFTPd.\n");
		if (! RunProgramOrExit("$launchctl", "start", "com.ncftp.ncftpd")) {
			Exit(1, "Could not start com.ncftp.ncftpd via launchd.\n");
		} else {
			UninstallLog("system(\"$launchctl\", \"stop\", \"com.ncftp.ncftpd\");\n");
		}
	} elsif ($rc_startup_script ne "") {
		Log(1, "  /bin/sh $rc_startup_script start\n");
		# This will echo "Starting NcFTPd:"
		RunProgramOrExit("/bin/sh", $rc_startup_script, "start");
		UninstallLog("KillFTPd(\"ncftpd\");\n");
	} elsif ($inittab_startup_script ne "") {
		my ($init) = SearchPath("init", "/sbin:/usr/sbin:/usr/bin:/bin");
		$init = "init" unless ($init);
		Log(0, "Running \"$init q\" to reload modified /etc/inittab.\n");
		RunProgramOrExit($init, "q");
		Log(0, "Starting NcFTPd.\n");
	}
	sleep(3);
}	# StartNcFTPd




sub NcFTPdShouldNowBeRunning
{
	if (! WaitForFTPServerToStart()) {
		Log(0, "\nERROR: NcFTPd did not startup.\n\n");
		if (open(ERRLOG, "<$startup_error_log")) {
			my ($line);
			while (defined($line = <ERRLOG>)) {
				Log(0, $line);
			}
		}
		Exit(1);
	}

	Log(0, "\nCONGRATULATIONS!  NcFTPd has been successfully installed.\n");
	Log(0, "Your next step is to customize your installation by editing:\n\n");
	Log(0, "  $new_general_cf\n");
	Log(0, "  $new_domain_cf\n");
	Log(0, "\nBe sure to run $sbindir/restart_ncftpd after doing that.\n");
}	# NcFTPdShouldNowBeRunning




sub Main
{
	umask(077);
	OpenLog();
#	OpenPager();
	SetOSVar();
	Getopts();
	SetOSDefaults();
	if (($> != 0) && (! $force_install)) {
		Exit(1, "You need to be Superuser to install NcFTPd.  You are UID ", $>, ".\n");
	}

	VerifyOSPrerequisites();
	SetPATHEnvironmentVar( VerifyLocation() );
	FindPerl();
	Log(0, "Starting installation of NcFTPd Server.\n");
	VerifyPackageContents();
	VerifyFreeSpace();
	LookupFTPServiceName();
	CheckStartupSystem();
	AbortIfSomethingWeDontKnowHowToDisableIsUsingPort21();
	KillAnyRunningFTPd();
	CreateEtcShells();
	CreateEtcFtpusers();
	AddFTPGroup();
	AddFTPUser();
	CreateFTPHome();
	IsThereAnFTPServerServing(0);
	DisableInetdHandlingOfFTP();
	DisableExistingFTPServersRunningFromInittab();	# Includes NcFTPd in /etc/inittab.
	DisableExistingNcFTPd();
	CheckIfSomeOtherFTPServerIsStillRunning();
	DisableRcScripts();
	NoteExistingNcFTPdVersion();
	LookupWebUser();
	MkInstallDirs();
	FixScript("extra/pgrep.pl", "#!${perl} -w");
	FixReportScripts();
	RemoveFtpFromEtcFtpusers();
	InstallConfigs();
	InstallPrograms();
	InstallStartupScript();
	StartNcFTPd();
	NcFTPdShouldNowBeRunning();
	Exit(0);
}	# Main

Main();
