/* Text manipulation primitives for XEmacs.
   Copyright (C) 1995 Sun Microsystems, Inc.
   Copyright (C) 1995, 1996, 2000, 2001, 2002, 2003, 2004, 2005, 2010
   Ben Wing.
   Copyright (C) 1999 Martin Buchholz.

This file is part of XEmacs.

XEmacs is free software: you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation, either version 3 of the License, or (at your
option) any later version.

XEmacs is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License
along with XEmacs.  If not, see <http://www.gnu.org/licenses/>. */

/* Synched up with: Not in FSF. */

/* Authorship:
 */

#include <config.h>
#include "lisp.h"

#include "buffer.h"
#include "casetab.h"
#include "charset.h"
#include "file-coding.h"
#include "lstream.h"
#include "profile.h"


/************************************************************************/
/*                            long comments                             */
/************************************************************************/

/* NB: Everything below was written by Ben Wing except as otherwise noted. */

/************************************************************************/
/*                                                                      */
/*                                                                      */
/*               Part A: More carefully-written documentation           */
/*                                                                      */
/*                                                                      */
/************************************************************************/

/* Authorship: Ben Wing


   ==========================================================================
                         7. Handling non-default formats
   ==========================================================================

   We support, at least to some extent, formats other than the default
   variable-width format, for speed; all of these alternative formats are
   fixed-width.  Currently we only handle these non-default formats in
   buffers, because access to their text is strictly controlled and thus
   the details of the format mostly compartmentalized.  The only really
   tricky part is the search code -- the regex, Boyer-Moore, and
   simple-search algorithms in search.c and regex.c.  All other code that
   knows directly about the buffer representation is the basic code to
   modify or retrieve the buffer text.

   Supporting fixed-width formats in Lisp strings is harder, but possible
   -- FSF currently does this, for example.  In this case, however,
   probably only 8-bit-fixed is reasonable for Lisp strings -- getting
   non-ASCII-compatible fixed-width formats to work is much, much harder
   because a lot of code assumes that strings are ASCII-compatible
   (i.e. ASCII + other characters represented exclusively using high-bit
   bytes) and a lot of code mixes Lisp strings and non-Lisp strings freely.

   The different possible fixed-width formats are 8-bit fixed, 16-bit
   fixed, and 32-bit fixed.  The latter can represent all possible
   characters, but at a substantial memory penalty.  The other two can
   represent only a subset of the possible characters.  How these subsets
   are defined can be simple or very tricky.

   Currently we support only the default format and the 8-bit fixed format,
   and in the latter, we only allow these to be the first 256 characters in
   an Ichar (ASCII and Latin 1).
   
   One reasonable approach for 8-bit fixed is to allow the upper half to
   represent any 1-byte charset, which is specified on a per-buffer basis.
   This should work fairly well in practice since most documents are in
   only one foreign language (possibly with some English mixed in).  I
   think FSF does something like this; or at least, they have something
   called nonascii-translation-table and use it when converting from
   8-bit-fixed text ("unibyte text") to default text ("multibyte text").
   With 16-bit fixed, you could do something like assign chunks of the 64K
   worth of characters to charsets as they're encountered in documents.
   This should work well with most Asian documents.

   If/when we switch to using Unicode internally, we might have formats more
   like this:

   -- UTF-8 or some extension as the default format.  Perl uses an
   extension that handles 64-bit chars and requires as much as 13 bytes per
   char, vs. the standard of 31-bit chars and 6 bytes max.  UTF-8 has the
   same basic properties as our own variable-width format (see text.c,
   Internal String Encoding) and so most code would not need to be changed.

   -- UTF-16 as a "pseudo-fixed" format (i.e. 16-bit fixed plus surrogates
   for representing characters not in the BMP, aka >= 65536).  The vast
   majority of documents will have no surrogates in them so byte/char
   conversion will be very fast.

   -- an 8-bit fixed format, like currently.
   
   -- possibly, UCS-4 as a 32-bit fixed format.

   The fixed-width formats essentially treat the buffer as an array of
   8-bit, 16-bit or 32-bit integers.  This means that how they are stored
   in memory (in particular, big-endian or little-endian) depends on the
   native format of the machine's processor.  It also means we have to
   worry a bit about alignment (basically, we just need to keep the gap an
   integral size of the character size, and get things aligned properly
   when converting the buffer between formats).

   ==========================================================================
                     8. Using UTF-16 as the default text format
   ==========================================================================

   NOTE: The Eistring API is (or should be) Mule-correct even without
   an ASCII-compatible internal representation.

   #### Currently, the assumption that text units are one byte in size is
   embedded throughout XEmacs, and `Ibyte *' is used where `Itext *' should
   be.  The way to fix this is to (among other things)

   (a) review all places referencing `Ibyte' and `Ibyte *', change them to
       use Itext, and fix up the code.
   (b) change XSTRING_DATA to be of type Itext *
   (c) review all uses of XSTRING_DATA
   (d) eliminate XSTRING_LENGTH, splitting it into XSTRING_BYTE_LENGTH and
       XSTRING_TEXT_LENGTH and reviewing all places referencing this
   (e) make similar changes to other API's that refer to the "length" of
       something, such as qxestrlen() and eilen()
   (f) review all use of `CIbyte *'.  Currently this is usually a way of
       passing literal ASCII text strings in places that want internal text.
       Either create separate _ascii() and _itext() versions of the
       functions taking CIbyte *, or make use of something like the
       WEXTTEXT() macro, which will generate wide strings as appropriate.
   (g) review all uses of Bytecount and see which ones should be Textcount.
   (h) put in error-checking code that will be tripped as often as possible
       when doing anything with internal text, and check to see that ASCII
       text has not mistakenly filtered in.  This should be fairly easy as
       ASCII text will generally be entirely spaces and letters whereas every
       second byte of Unicode text will generally be a null byte.  Either we
       abort if the second bytes are entirely letters and numbers, or,
       perhaps better, do the equivalent of a non-MULE build, where we should
       be dealing entirely with 8-bit characters, and assert that the high
       bytes of each pair are null.
   (i) review places where xmalloc() is called.  If we convert each use of
       xmalloc() to instead be xnew_array() or some other typed routine,
       then we will find every place that allocates space for Itext and
       assumes it is based on one-byte units.
   (j) encourage the use of ITEXT_ZTERM_SIZE instead of '+ 1' whenever we
       are adding space for a zero-terminator, to emphasize what we are
       doing and make sure the calculations are correct.  Similarly for
       EXTTEXT_ZTERM_SIZE.
   (k) Note that the qxestr*() functions, among other things, will need to
       be rewritten.

   Note that this is a lot of work, and is not high on the list of priorities
   currently.

   ==========================================================================
                                9. Miscellaneous
   ==========================================================================

   A. Unicode Support

   Unicode support is very desirable.  Currrently we know how to handle
   externally-encoded Unicode data in various encodings -- UTF-16, UTF-8,
   etc.  However, we really need to represent Unicode characters internally
   as-is, rather than converting to some language-specific character set.
   For efficiency, we should represent Unicode characters using 3 bytes
   rather than 4.  This means we need to find charset ID's for Unicode.
   Given that there are 65,536 characters in Unicode and we can attach
   96x96 = 9,216 characters per leading byte, we need eight leading bytes
   for Unicode.  We currently have four free (0x9A - 0x9D), and with a
   little bit of rearranging we can get five: ASCII doesn't really need to
   take up a leading byte. (We could just as well use 0x7F, with a little
   change to the functions that assume that 0x80 is the lowest leading
   byte.) This means we still need to dump three leading bytes and move
   them into private space.  The CNS charsets are good candidates since
   they are rarely used, and JAPANESE_JISX0208_1978 is becoming less and
   less used and could also be dumped.

   B. Composite Characters
      
   Composite characters are characters constructed by overstriking two
   or more regular characters.

   1) The old Mule implementation involves storing composite characters
      in a buffer as a tag followed by all of the actual characters
      used to make up the composite character.  I think this is a bad
      idea; it greatly complicates code that wants to handle strings
      one character at a time because it has to deal with the possibility
      of great big ungainly characters.  It's much more reasonable to
      simply store an index into a table of composite characters.

   2) The current implementation only allows for 16,384 separate
      composite characters over the lifetime of the XEmacs process.
      This could become a potential problem if the user
      edited lots of different files that use composite characters.
      Due to FSF bogosity, increasing the number of allowable
      composite characters under Mule would decrease the number
      of possible faces that can exist.  Mule already has shrunk
      this to 2048, and further shrinkage would become uncomfortable.
      No such problems exist in XEmacs.

      Composite characters could be represented as 0x8D C1 C2 C3, where
      each C[1-3] is in the range 0xA0 - 0xFF.  This allows for slightly
      under 2^20 (one million) composite characters over the XEmacs process
      lifetime.  Mule characters are already 21 bits, which should
      more-or-less be enough to fit them, but maybe we might need to
      increase the size to 22 bits.  Or you could use 0x8D C1 C2 C3 C4,
      allowing for about 85 million (slightly over 2^26) composite
      characters.

   ==========================================================================
                               10. Internal API's
   ==========================================================================

   All of these are documented in more detail in text.h.

   Basic internal-format API's
   
   These are simple functions and macros to convert between text
   representation and characters, move forward and back in text, etc.
   
   The DFC API
   
   This is for conversion between internal and external text.  Note that
   there is also the "new DFC" API, which *returns* a pointer to the
   converted text (in alloca space), rather than storing it into a
   variable.
   
   The Eistring API
   
   (This API is currently under-used) When doing simple things with
   internal text, the basic internal-format API's are enough.  But to do
   things like delete or replace a substring, concatenate various strings,
   etc. is difficult to do cleanly because of the allocation issues.
   The Eistring API is designed to deal with this, and provides a clean
   way of modifying and building up internal text. (Note that the former
   lack of this API has meant that some code uses Lisp strings to do
   similar manipulations, resulting in excess garbage and increased
   garbage collection.)
   
   NOTE: The Eistring API is (or should be) Mule-correct even without
   an ASCII-compatible internal representation.

   ==========================================================================
                      11. Other Sources of Documentation
   ==========================================================================

   man/lispref/mule.texi

      another intro to characters, encodings, etc; #### Merge with the
      above info
      
      documentation of ISO-2022
      
      The charset and coding-system Lisp API's
      
      The CCL conversion language for writing encoding conversions
      
      The Latin-Unity package for unifying Latin charsets

   man/internals/internals.texi (the Internals manual)

      "Coding for Mule" -- how to write Mule-aware code
      
      "Modules for Internationalization"
      
      "The Text in a Buffer" -- more about the different ways of
      viewing buffer positions; #### Merge with the above info
      
      "MULE Character Sets and Encodings" -- yet another intro
      to characters, encodings, etc; #### Merge with the
      above info; also some documentation of Japanese EUC and JIS7,
      and CCL internals

   text.h -- info about specific XEmacs-C API's for handling internal and
             external text

   intl-win32.c -- Windows-specific I18N information 

   lisp.h -- some info appears alongside the definitions of the basic
             character-related types

   unicode.c -- documentation about Unicode translation tables
*/


/************************************************************************/
/*                                                                      */
/*                                                                      */
/*               Part B: Random proposals for work to be done           */
/*                                                                      */
/*                                                                      */
/************************************************************************/


/*


   ==========================================================================
                   - Mule design issues (ben)
   ==========================================================================

   circa 1999
   
   Here is a more detailed list of Mule-related projects that we will be
   working on.  They are more or less ordered according to how we will
   proceed, but it's not exact.  In particular, there will probably be
   time overlap among adjacent projects.
   
   @enumerate
   @item
   Modify the internal/external conversion macros to allow for
   MS Windows support.
   
   @item
   Modify the buffer macros to allow for more than one internal
   representation, e.g. fixed width and variable width.
   
   @item
   Review the existing Mule code, especially the lisp code, for code
   quality issues and improve the cleanliness of it.  Also work on
   creating a specification for the Mule API.
   
   @item
   Write some more automated mule tests.
   
   @item
   Integrate Tomohiko's UTF-2000 code, fixing it up so that nothing is
   broken when the UTF-2000 configure option is not enabled.
   
   @item
   Fix up the MS Windows code to be Mule-correct, so that you can
   compile with Mule support under MS windows and have a working
   XEmacs, at least just with Latin-1.
   
   @item
   Implement a scheme to guarantee no corruption of files, even with
   an incorrect coding system - in particular, guarantee no corruption
   of binary files.
   
   @item
   Make the text property support in XEmacs robust with respect to
   string and text operations, so that the `no corruption' support in
   the previous entry works properly, even if a lot of cutting and
   pasting is done.
   
   @item
   Improve the handling of auto-detection so that, when there is any
   possibility at all of mistake, the user is informed of the detected
   encoding and given the choice of choosing other possibilities.
   
   @item
   Improve the support for different language environments in XEmacs,
   for example, the priority of coding systems used in auto-detection
   should properly reflect the language environment.  This probably
   necessitates rethinking the current `coding system priority'
   scheme.
   
   @item
   Do quality work to improve the existing UTF-2000 implementation.
   
   @item
   Implement preliminary support for 8-bit fixed width
   representation.  First, we will only implement 7-bit support, and
   will fall back to variable width as soon as any non-ASCII
   character is encountered.  Then we will improve the support to
   handle an arbitrary character set in the upper half of the 8-bit space.
   
   @item
   Investigate any remaining hurdles to making --with-mule be the
   default configure option.
   @end enumerate
   
   ==========================================================================
                   - Mule design issues (stephen)
   ==========================================================================

   What I see as Mule priorities (in rough benefit order, I am not taking
   account of difficulty, nor the fact that some - eg 8 & 10 - will
   probably come as packages):
   
   @enumerate
   @item
   Fix the autodetect problem (by making the coding priority list
   user-configurable, as short as he likes, even null, with "binary"
   as the default).
   @item
   Document the language environments and other Mule "APIs" as
   implemented (since there is no real design spec).  Check to see 
   how and where they are broken.
   @item
   Make the Mule menu useful to non-ISO-2022-literate folks.
   @item
   Redo the lstreams stuff to make it easy and robust to "pipeline",
   eg, libz | gnupg | jis2mule.
   @item
   Make Custom Mule-aware.  (This probably depends on a sensible
   fonts model.)
   @item
   Implement the "literal byte stream" memory feature.
   @item
   Study the FSF implementation of Mule for background for 7 & 8.
   @item
   Identify desirable Mule features (eg, i18n-ized messages as above, 
   collating tables by language environment, etc).  (New features
   might have priority as high as 9.)
   @item
   Specify Mule UIs, APIs, etc, and design and (re)implement them.
   @item
   Implement the 8-bit-wide buffer optimization.
   @item
   Move the internal encoding to UTF-32 (subject to Olivier's caveats 
   regarding compose characters), with the variable-width char
   buffers using UTF-8.
   @item
   Implement the 16- and 32-bit-wide buffer optimizations.
   @end enumerate
   
   ==========================================================================
                   - Mule design issues "short term" (ben)
   ==========================================================================

   @enumerate
   @item
   Finish changes in fixup/directory, get in CVS.
   
   (Test with and without "quick-build", to see if really faster)
   (need autoconf)
   
   @item
   Finish up Windows/Mule changes.  Outline of this elsewhere;  Do
   *minimal* effort.
   
   @item
   Continue work on Windows stability, e.g. go through existing notes
   on Windows Mule-ization + extract all info.
   
   @item
   Get Unicode translation tables integrated.
   
   Finish UCS2/UTF16 coding system.
   
   @item
   Make sure coding system priority list is language-environment specific.
   
   @item
   Consider moving language selection Menu up to be parallel with Mule menu.
   
   @item
   Check to make sure we grok the default locale at startup under
   Windows and understand the Windows locales.  Finish implementation
   of mswindows-multibyte and make sure it groks all the locales.
   
   @item
   Do the above as best as we can without using Unicode tables.
   
   @item
   Start tagging all text with a language text property,
   indicating the current language environment when the text was input.
   
   @item
   Make sure we correctly accept input of non-ASCII chars
   (probably already do!)
   
   @item
   Implement active language/keyboard switching under Windows.
   
   @item
   Look into implementing support for "MS IME" protocol (Microsoft
   fancy built-in Asian input methods).
   
   @item
   Redo implementation of mswindows-multibyte and internal display to
   entirely use translation to/from Unicode for increased accuracy.
   
   @item
   Implement buf<->char improvements from FSF.  Also implement
   my string byte<->char optimization structure.
   
   @item
   Integrate all Mule DOCS from 20.6 or 21.0.  Try to add sections
   for what we've added.
   
   @item
   Implement 8-bit fixed width optimizations.  Then work on 16-bit.
   @end enumerate
   
   ==========================================================================
                   - Mule design issues (more) (ben)
   ==========================================================================

   Get minimal Mule for Windows working using Ikeyama's patches.  At
   first, rely on his conversion of internal -> external
   locale-specific but very soon (as soon as we get translation
   tables) can switch to using Unicode versions of display funs, which
   will allow many more charsets to be handled and in a more
   consistent fashion.

   i.e. to convert an internal string to an external format, at first
   we use our own knowledge of the Microsoft locale file formats but
   an alternative is to convert to Unicode and use Microsoft's
   convert-Unicode-to-locale encoding functions.  This gains us a
   great deal of generality, since in practice all charset caching
   points can be wrapped into Unicode caching points.

   This requires adding UCS2 support, which I'm doing.  This support
   would let us convert internal -> Unicode, which is exactly what we
   want.

   At first, though, I would do the UCS2 support, but leave the
   existing way of doing things in redisplay.  Meanwhile, I'd go
   through and fix up the places in the code that assume we are
   dealing with unibytes.

   After this, the font problems will be fixed , we should have a
   pretty well working XEmacs + MULE under Windows.  The only real
   other work is the clipboard code, which should be straightforward.

   ==========================================================================
                   - Mule design discussion
   ==========================================================================

   Ben
   
   April 11, 2000
   
   Well yes, this was the whole point of my "no lossage" proposal of being
   able to undo any coding-system transformation on a buffer.  The idea was
   to figure out which transformations were definitely reversable, and for
   all the others, cache the original text in a text property.  This way, you
   could probably still do a fairly good job at constructing a good reversal
   even after you've gone into the text and added, deleted, and rearranged
   some things.
   
   But you could implement it much more simply and usefully by just
   determining, for any text being decoded into mule-internal, can we go back
   and read the source again?  If not, remember the entire file (GNUS
   message, etc) in text properties.  Then, implement the UI interface (like
   Netscape's) on top of that.  This way, you have something that at least
   works, but it might be inefficient.  All we would need to do is work on
   making the
   underlying implementation more efficient.
   
   Are you interested in doing this?  It would be a huge win for users.
   Hrvoje Niksic wrote:
   
   > Ben Wing <ben@666.com> writes:
   >
   > > let me know exactly what "rethink" functionality you want and i'll
   > > come up with an interface.  perhaps you just want something like
   > > netscape's encoding menu, where if you switch encodings, it reloads
   > > and reencodes?
   >
   > It might be a bit more complex than that.  In many cases, it's hard or
   > impossible to meaningfully "reload" -- for instance, this
   > functionality should be available while editing a Gnus message, as
   > well as while visiting a file.
   >
   > For the special case of Latin-N <-> Latin-M conversion, things could
   > be done easily -- to convert from N to M, you only need to convert
   > internal representation back to N, and then convert it forth to M.
   
   --------------------------------------------------------------------------
   April 11, 2000
   
   Well yes, this was the whole point of my "no lossage" proposal of being
   able to undo any coding-system transformation on a buffer.  The idea was
   to figure out which transformations were definitely reversable, and for
   all the others, cache the original text in a text property.  This way, you
   could probably still do a fairly good job at constructing a good reversal
   even after you've gone into the text and added, deleted, and rearranged
   some things.
   
   But you could implement it much more simply and usefully by just
   determining, for any text being decoded into mule-internal, can we go back
   and read the source again?  If not, remember the entire file (GNUS
   message, etc) in text properties.  Then, implement the UI interface (like
   Netscape's) on top of that.  This way, you have something that at least
   works, but it might be inefficient.  All we would need to do is work on
   making the
   underlying implementation more efficient.
   
   Are you interested in doing this?  It would be a huge win for users.
   Hrvoje Niksic wrote:
   
   > Ben Wing <ben@666.com> writes:
   >
   > > let me know exactly what "rethink" functionality you want and i'll
   > > come up with an interface.  perhaps you just want something like
   > > netscape's encoding menu, where if you switch encodings, it reloads
   > > and reencodes?
   >
   > It might be a bit more complex than that.  In many cases, it's hard or
   > impossible to meaningfully "reload" -- for instance, this
   > functionality should be available while editing a Gnus message, as
   > well as while visiting a file.
   >
   > For the special case of Latin-N <-> Latin-M conversion, things could
   > be done easily -- to convert from N to M, you only need to convert
   > internal representation back to N, and then convert it forth to M.
   

   ==========================================================================
   - Redoing translation macros [old]
   ==========================================================================

  Currently the translation macros (the macros with names such as
  GET_C_STRING_CTEXT_DATA_ALLOCA) have names that are difficult to parse
  or remember, and are not all that general.  In the process of
  reviewing the Windows code so that it could be muleized, I discovered
  that these macros need to be extended in various ways to allow for
  the Windows code to be easily muleized.
  
  Since the macros needed to be changed anyways, I figured it would be a
  good time to redo them properly.  I propose new macros which have
  names like this:
  
  @itemize @bullet
  @item
  <A>_TO_EXTERNAL_FORMAT_<B>
  @item
  <A>_TO_EXTERNAL_FORMAT_<B>_1
  @item
  <C>_TO_INTERNAL_FORMAT_<D>
  @item
  <C>_TO_INTERNAL_FORMAT_<D>_1
  @end itemize
  
  A and C represent the source of the data, and B and D represent the
  sink of the data.
  
  All of these macros call either the functions
  convert_to_external_format or convert_to_internal_format internally,
  with some massaging of the arguments.
  
  All of these macros take the following arguments:
  
  @itemize @bullet
  @item
  First, one or two arguments indicating the source of the data.
  @item
  Second, an argument indicating the coding system. (In order to avoid
  an excessive number of macros, we no longer provide separate macros
  for specific coding systems.)
  @item
  Third, one or two arguments indicating the sink of the data.
  @item
  Fourth, optionally, arguments indicating the error behavior and the
  warning class (these arguments are only present in the _1 versions
  of the macros).  The other, shorter named macros are trivial
  interfaces onto these macros with the error behavior being
  ERROR_ME_WARN, with the warning class being Vstandard_warning_class.
  @end itemize
  
  <A> can be one of the following:
  @itemize @bullet
  @item
  LISP (which means a Lisp string) Takes one argument, a Lisp Object.
  @item
  LSTREAM (which indicates an lstream) Takes one argument, an
  lstream.  The data is read from the lstream until EOF is reached.
  @item
  DATA (which indicates a raw memory area) Takes two arguments, a
  pointer and a length in bytes.
  (You must never use this if the source of the data is a Lisp string,
  because of the possibility of relocation during garbage collection.)
  @end itemize
  
  <B> can be one of the following:
  @itemize @bullet
  @item
  ALLOCA (which means that the resulting data is stored in alloca()ed
  memory.  Two arguments should be specified, a pointer and a length,
  which should be lvalues.)
  @item
  MALLOC (which means that the resulting data is stored in malloc()ed
  memory.  Two arguments should be specified, a pointer and a
  length.  The memory must be free()d by the caller.
  @item
  OPAQUE (which means the resulting data is stored in an opaque Lisp
  Object.  This takes one argument, a lvalue Lisp Object.
  @item
  LSTREAM. The data is written to an lstream.
  @end itemize
  
  <C> can be one of the :
  @itemize @bullet
  @item
  DATA
  @item
  LSTREAM
  @end itemize
  (just like <A> above)
  
  <D> can be one of
  @itemize @bullet
  @item
  ALLOCA
  @item
  MALLOC
  @item
  LISP This means a Lisp String.
  @item
  BUFFER The resulting data is inserted into a buffer at the buffer's
  value of point.
  @item
  LSTREAM The data is written to the lstream.
  @end itemize
  
  
  Note that I have eliminated the FORMAT argument of previous macros,
  and replaced it with a coding system.  This was made possible by
  coding system aliases.  In place of old `format's, we use a `virtual
  coding system', which is aliased to the actual coding system.
  
  The value of the coding system argument can be anything that is legal
  input to get_coding_system, i.e. a symbol or a coding system object.

   ==========================================================================
   - creation of generic macros for accessing internally formatted data [old]
   ==========================================================================

 I have a design; it's all written down (I did it in Tsukuba), and I just have
 to have it transcribed.  It's higher level than the macros, though; it's Lisp
 primitives that I'm designing.
 
 As for the design of the macros, don't worry so much about all files having to
 get included (which is inevitable with macros), but about how the files are
 separated.  Your design might go like this:
 
 @enumerate
 @item
 you have generic macro interfaces, which specify a particular
 behavior but not an implementation.  these generic macros have
 complementary versions for buffers and for strings (and the buffer
 or string is an argument to all of the macros), and do such things
 as convert between byte and char indices, retrieve the character at
 a particular byte or char index, increment or decrement a byte
 index to the beginning of the next or previous character, indicate
 the number of bytes occupied by the character at a particular byte
 or character index, etc.  These are similar to what's already out
 there except that they confound buffers and strings and that they
 can also work with actual char *'s, which I think is a really bad
 idea because it encourages code to "assume" that the representation
 is ASCII compatible, which is might not be (e.g. 16-bit fixed
 width).  In fact, one thing I'm planning on doing is redefining
 Bufbyte as a struct, for debugging purposes, to catch all places
 that cavalierly compare them with ASCII char's.  Note also that I
 really want to rename Bufpos and Bytind, which are confusing and
 wrong in that they also apply to strings. They should be Bytepos
 and Charpos, or something like that, to go along with Bytecount and
 Charcount. Similarly, Bufbyte is similarly a misnomer and should be
 Intbyte -- a byte in the internal string representation (any of the
 internal representations) of a string or buffer.  Corresponding to
 this is Extbyte (which we already have), a byte in any external
 string representation.  We also have Extcount, which makes sense,
 and we might possibly want Extcharcount, the number of characters
 in an external string representation; but that gets sticky in modal
 encodings, and it's not clear how useful it would be.
 
 @item
 for all generic macro interfaces, there are specific versions of
 each of them for each possible representation (pure ASCII in the
 non-Mule world, Mule standard, UTF-8, 8-bit fixed, 16-bit fixed,
 32-bit fixed, etc.; there may well be more than one possible 16-bit
 fixed version, as well). Each representation has a corresponding
 prefix, e.g. MULE_ or FIXED16_ or whatever, which is prefixed onto
 the generic macro names.  The resulting macros perform the
 operation defined for the macro, but assume, and only work
 correctly with, text in the corresponding representation.
 
 @item
 The definition of the generic versions merely conditionalizes on
 the appropriate things (i.e. bit flags in the buffer or string
 object) and calls the appropriate representation-specific version.
 There may be more than one definition (protected by ifdefs, of
 course), or one definition that amalgamated out of many ifdef'ed
 sections.
 
 @item
 You should probably put each different representation in its own
 header file, e.g. charset-mule.h or charset-fixed16.h or
 charset-ascii.h or whatever.  Then put the main macros into
 charset.h, and conditionalize in this file appropriately to include
 the other ones.  That way, code that actually needs to play around
 with internal-format text at this level can include "charset.h"
 (certainly a much better place than buffer.h), and everyone else
 uses higher-level routines.  The representation-specific macros
 should not normally be used *directly* at all; they are invoked
 automatically from the generic macros.  However, code that needs to
 be highly, highly optimized might choose to take a loop and write
 two versions of it, one for each representation, to avoid the
 per-loop-iteration cost of a comparison. Until the macro interface
 is rock stable and solid, we should strongly discourage such
 nanosecond optimizations.
 @end enumerate
 
   ==========================================================================
                   - UTF-16 compatible representation
   ==========================================================================

 NOTE: One possible default internal representation that was compatible
 with UTF16 but allowed all possible chars in UCS4 would be to take a
 more-or-less unused range of 2048 chars (not from the private area
 because Microsoft actually uses up most or all of it with EUDC chars).
 Let's say we picked A400 - ABFF.  Then, we'd have:
 
 0000 - FFFF    Simple chars
 
 D[8-B]xx D[C-F]xx  Surrogate char, represents 1M chars
 
 A[4-B]xx D[C-F]xx D[C-F]xx   Surrogate char, represents 2G chars
 
 This is exactly the same number of chars as UCS-4 handles, and it follows the
 same property as UTF8 and Mule-internal:
 
 @enumerate
 @item
 There are two disjoint groupings of units, one representing leading units
 and one representing non-leading units.
 @item
 Given a leading unit, you immediately know how many units follow to make
 up a valid char, irrespective of any other context.
 @end enumerate
 
 Note that A4xx is actually currently assigned to Yi.  Since this is an
 internal representation, we could just move these elsewhere.
 
 An alternative is to pick two disjoint ranges, e.g. 2D00 - 2DFF and
 A500 - ABFF.
 
   ==========================================================================
                        New API for char->font mapping
   ==========================================================================
 - ; supersedes charset-registry and CCL;
   supports all windows systems; powerful enough for Unicode; etc.
 
   (charset-font-mapping charset)
 
 font-mapping-specifier  string
 
 char-font-mapping-table
 
   char-table, specifier; elements of char table are either strings (which
   specify a registry or comparable font property, or vectors of a string
   (same) followed by keyword-value pairs (optional).  The only allowable
   keyword currently is :ccl-program, which specifies a CCL program to map
   the characters into font indices.  Other keywords may be added
   e.g. allowing Elisp fragments instead of CCL programs, also allowed is
   [inherit], which inherits from the next less-specific char-table in the
   specifier.
 
   The preferred interface onto this mapping (which should be portable
   across Emacsen) is
 
   (set-char-font-mapping key value &optional locale tag-set how-to-add)
 
   where key is a char, range or charset (as for put-char-table), value is
   as above, and the other arguments are standard for specifiers.  This
   automatically creates a char table in the locale, as necessary (all
   elements default to [inherit]).  On GNU Emacs, some specifiers arguments
   may be unimplemented.
 
  (char-font-mapping key value &optional locale)
 works vaguely like get-specifier?   But does inheritance processing.
 locale should clearly default here to current-buffer
 
 #### should get-specifier as well?  Would make it work most like
 #### buffer-local variables.
 
 NB.  set-charset-registry and set-charset-ccl-program are obsoleted.
 
   ==========================================================================
                 Implementing fixed-width 8,16,32 bit buffer optimizations
   ==========================================================================

 Add set-buffer-optimization (buffer &rest keywords) for
 controlling these things.
 
 Also, put in hack so that correct arglist can be retrieved by
 Lisp code.
 
 Look at the way keyword primitives are currently handled; make
 sure it works and is documented, etc.
 
 Implement 8-bit fixed width optimization.  Take the things that
 know about the actual implementation and put them in a single
 file, in essence creating an abstraction layer to allow
 pluggable internal representations.  Implement a fairly general
 scheme for mapping between character codes in the 8 bits or 16
 bits representation and on actual charset characters.  As part of
 set-buffer-optimization, you can specify a list of character sets
 to be used in the 8 bit to 16 bit, etc. world.  You can also
 request that the buffer be in 8, 16, etc. if possible.
 
 -> set defaults wrt this.
 -> perhaps this should be just buffer properties.
 -> this brings up the idea of default properties on an object.
 -> Implement default-put, default-get, etc.
 
 What happens when a character not assigned in the range gets
 added?  Then, must convert to variable width of some sort.
 
 Note: at first, possibly we just convert whole hog to get things
 right.  Then we'd have to poy alternative to characters that got
 added + deleted that were unassigned in the fixed width.  When
 this goes to zero and there's been enough time (heuristics), we
 go back to fixed.
 
 Side note:  We could dynamically build up the set of assigned
 chars as they go.  Conceivably this could even go down to the
 single char level: Just keep a big array of mapping from 16 bit
 values to chars, and add empty time, a char has been encountered
 that wasn't there before.  Problem need inverse mapping.
 
 -> Possibility; chars are actual objects, not just numbers.
 Then you could keep track of such info in the chars itself.
 *Think about this.*
 
 Eventually, we might consider allowing mixed fixed-width,
 variable-width buffer encodings.  Then, we use range tables to
 indicate which sections are fixed and which variable and INC_CHAR does
 something like this: binary search to find the current range, which
 indicates whether it's fixed or variable, and tells us what the
 increment is.  We can cache this info and use it next time to speed
 up.
 
 -> We will then have two partially shared range tables - one for
 overall fixed width vs. variable width, and possibly one containing
 this same info, but partitioning the variable width in one.  Maybe
 need fancier nested range table model.
 
   ==========================================================================
        Expansion of display table and case mapping table support for all
                           chars, not just ASCII/Latin1.
   ==========================================================================

   ==========================================================================
       Improved flexibility for display tables, and evaluation of its
      features to make sure it meshes with and complements the char<->font
                       mapping API mentioned earlier
   ==========================================================================

   ==========================================================================
                              String access speedup:
   ==========================================================================

  For strings larger than some size in bytes (10?), keep extra fields of
  info: length in chars, and a (char, byte) pair in the middle to speed
  up sequential access.
  
  (Better idea: do this for any size string, but only if it contains
  non-ASCII chars.  Then if info is missing, we know string is
  ASCII-only.)
  
  Use a string-extra-info object, replacing string property slot and
  containing fields for string mod tick, string extents, string props,
  and string char length, and cached (char,byte) pair.
  string-extra-info (or string-auxiliary?) objects could be in frob
  blocks, esp. if creating frob blocks is easy + worth it.

- Caching of char<->byte conversions in strings - should make nearly
  all operations on strings O(N)

   ==========================================================================
                    Improvements in buffer char<->byte mapping
   ==========================================================================

  - Range table implementation - especially when there are few runs of
    different widths, e.g. recently converted from fixed-width
    optimization to variable width

  Range Tables to speed up Bufpos <-> Bytind caching
  ==================================================
  
  This describes an alternative implementation using ranges.  We
  maintain a range table of all spans of characters of a fixed width.
  Updating this table could take time if there are a large number of
  spans; but constant factors of operations should be quick.  This method really wins
  when you have 8-bit buffers just converted to variable width, where
  there will be few spans.  More specifically, lookup in this range
  table is O(log N) and can be done with simple binary search, which is
  very fast.  If we maintain the ranges using a gap array, updating this
  table will be fast for local operations, which is most of the time.
  
  We will also provide (at first, at least) a Lisp function to set the
  caching mechanism explicitly - either range tables or the existing
  implementation.  Eventually, we want to improve things, to the point
  where we automatically pick the right caching for the situation and
  have more caching schemes implemented.

   ==========================================================================
                        - Robustify Text Properties
   ==========================================================================

   ==========================================================================
           Support for unified internal representation, e.g. Unicode
   ==========================================================================

   Start tagging all text with a language text property,
   indicating the current language environment when the text was input.
   (needs "Robustify Text Properties")

   ==========================================================================
                          - Generalized Coding Systems
   ==========================================================================

  - Lisp API for Defining Coding Systems

  User-defined coding systems.
  
  (define-coding-system-type 'type
    :encode-function fun
    :decode-function fun
    :detect-function fun
    :buffering (number = at least this many chars
                line   = buffer up to end of line
                regexp = buffer until this regexp is found in match
                source data.  match data will be appropriate when fun is
                called
  
  encode fun is called as
  
  (encode instream outstream)
  
  should read data from instream and write converted result onto
  outstream.  Can leave some data stuff in stream, it will reappear
  next time.  Generally, there is a finite amount of data in instream
  and further attempts to read lead to would-block errors or retvals.
  Can use instream properties to record state.  May use read-stream
  functionality to read everything into a vector or string.
  
  ->Need vectors + string exposed to resizing of Lisp implementation
    where necessary.
  
   ==========================================================================
     Support Windows Active Kbd Switching, Far East IME API (done already?)
   ==========================================================================

   ==========================================================================
              - UI/design changes for Coding System Pipelining
   ==========================================================================

  ------------------------------------------------------------------
                            CODING-SYSTEM CHAINS
  ------------------------------------------------------------------

  sjt sez:

  There should be no elementary coding systems in the Lisp API, only
  chains.  Chains should be declared, not computed, as a sequence of coding
  formats.  (Probably the internal representation can be a vector for
  efficiency but programmers would probably rather work with lists.)  A
  stream has a token type.  Most streams are octet streams.  Text is a
  stream of characters (in _internal_ format; a file on disk is not text!)
  An octet-stream has no implicit semantics, so its format must always be
  specified.  The only type currently having semantics is characters.  This
  means that the chain [euc-jp -> internal -> shift_jis) may be specified
  (euc-jp, shift_jis), and if no euc-jp -> shift_jis converter is
  available, then the chain is automatically constructed.  (N.B.  I f we
  have fixed width buffers in the future, then we could have ASCII -> 8-bit
  char -> 16-bit char -> ISO-2022-JP (with escape sequences).

  EOL handling is a char <-> char coding.  It should not be part of another
  coding system except as a convenience for users.  For text coding,
  automatically insert EOL handlers between char <-> octet boundaries.

  ------------------------------------------------------------------
                            ABOUT DETECTION
  ------------------------------------------------------------------


  ------------------------------------------------------------------
     EFFICIENCY OF CODING CONVERSION WITH MULTIPLE COPIES/CHAINS
  ------------------------------------------------------------------

   A comment in encode_decode_coding_region():

   The chain of streams looks like this:

     [BUFFER] <----- (( read from/send to loop ))
                     ------> [CHAR->BYTE i.e. ENCODE AS BINARY if source is
                              in bytes]
		             ------> [ENCODE/DECODE AS SPECIFIED]
			             ------> [BYTE->CHAR i.e. DECODE AS BINARY
                                              if sink is in bytes]
					     ------> [AUTODETECT EOL if
					              we're decoding and
						      coding system calls
						      for this]
			                              ------> [BUFFER]

    sjt (?) responds:

     Of course, this is just horrible.  BYTE<->CHAR should only be available
     to I/O routines.  It should not be visible to Mule proper.

     A comment on the implementation.  Hrvoje and Kyle worry about the
     inefficiency of repeated copying among buffers that chained coding
     systems entail.  But this may not be as time inefficient as it appears
     in the Mule ("house rules") context.  The issue is how do you do chain
     coding systems without copying?  In theory you could have

     IChar external_to_raw (ExtChar *cp, State *s);
     IChar decode_utf16 (IChar c, State *s);
     IChar decode_crlf (ExtChar *cp, State *s);

     typedef Ichar (*Converter[]) (Ichar, State*);

     Converter utf16[2] = { &decode_utf16, &decode_crlf };

     void convert (ExtChar *inbuf, IChar *outbuf, Converter cvtr)
     {
       int i;
       ExtChar c;
       State s;

       while (c = external_to_raw (*inbuf++, &s))
	 {
	   for (i = 0; i < sizeof(cvtr)/sizeof(Converter); ++i)
	     if (s.ready)
	       c = (*cvtr[i]) (c, &s);
	 }
       if (s.ready)
         *outbuf++ = c;
     }

     But this is a lot of function calls; what Ben is doing is basically
     reducing this to one call per buffer-full.  The only way to avoid this
     is to hardcode all the "interesting" coding systems, maybe using
     inline or macros to give structure.  But this is still a huge amount
     of work, and code.

     One advantage to the call-per-char approach is that we might be able
     to do something about the marker/extent destruction that coding
     normally entails.

    ben sez:

     it should be possible to preserve the markers/extents without
     switching completely to one-call-per-char -- we could at least do one
     call per "run", where a run is more or less the maximal stretch of
     text not overlapping any markers or extent boundaries. (It's a bit
     more complicated if we want to properly support the different extent
     begins/ends; in some cases we might have to pump a single character
     adjacent to where two extents meet.) The "stateless" way that I wrote
     all of the conversion routines may be a real hassle but it allows
     something like this to work without too much problem -- pump in one
     run at a time into one end of the chain, do a flush after each
     iteration, and stick what comes out the other end in its place.

  ------------------------------------------------------------------
                              ABOUT FORMATS
  ------------------------------------------------------------------
  
  when calling make-coding-system, the name can be a cons of (format1 .
  format2), specifying that it decodes format1->format2 and encodes the other
  way.  if only one name is given, that is assumed to be format1, and the
  other is either `external' or `internal' depending on the end type.
  normally the user when decoding gives the decoding order in formats, but
  can leave off the last one, `internal', which is assumed.  a multichain
  might look like gzip|multibyte|unicode, using the coding systems named
  `gzip', `(unicode . multibyte)' and `unicode'.  the way this actually works
  is by searching for gzip->multibyte; if not found, look for gzip->external
  or gzip->internal. (In general we automatically do conversion between
  internal and external as necessary: thus gzip|crlf does the expected, and
  maps to gzip->external, external->internal, crlf->internal, which when
  fully specified would be gzip|external:external|internal:crlf|internal --
  see below.)  To forcibly fit together two converters that have explicitly
  specified and incompatible names (say you have unicode->multibyte and
  iso8859-1->ebcdic and you know that the multibyte and iso8859-1 in this
  case are compatible), you can force-cast using :, like this:
  ebcdic|iso8859-1:multibyte|unicode. (again, if you force-cast between
  internal and external formats, the conversion happens automatically.)
  
  --------------------------------------------------------------------------
  ABOUT PDUMP, UNICODE, AND RUNNING XEMACS FROM A DIRECTORY WITH WEIRD CHARS
  --------------------------------------------------------------------------

-- there's the problem that XEmacs can't be run in a directory with
   non-ASCII/Latin-1 chars in it, since it will be doing Unicode
   processing before we've had a chance to load the tables.  In fact,
   even finding the tables in such a situation is problematic using
   the normal commands.  my idea is to eventually load the stuff
   extremely extremely early, at the same time as the pdump data gets
   loaded.  in fact, the unicode table data (stored in an efficient
   binary format) can even be stuck into the pdump file (which would
   mean as a resource to the executable, for windows).  we'd need to
   extend pdump a bit: to allow for attaching extra data to the pdump
   file. (something like pdump_attach_extra_data (addr, length)
   returns a number of some sort, an index into the file, which you
   can then retrieve with pdump_load_extra_data(), which returns an
   addr (mmap()ed or loaded), and later you pdump_unload_extra_data()
   when finished.  we'd probably also need
   pdump_attach_extra_data_append(), which appends data to the data
   just written out with pdump_attach_extra_data().  this way,
   multiple tables in memory can be written out into one contiguous
   table. (we'd use the tar-like trick of allowing new blocks to be
   written without going back to change the old blocks -- we just rely
   on the end of file/end of memory.) this same mechanism could be
   extracted out of pdump and used to handle the non-pdump situation
   (or alternatively, we could just dump either the memory image of
   the tables themselves or the compressed binary version).  in the
   case of extra unicode tables not known about at compile time that
   get loaded before dumping, we either just dump them into the image
   (pdump and all) or extract them into the compressed binary format,
   free the original tables, and treat them like all other tables.


   ==========================================================================
        - Generalized language appropriate word wrapping (requires
                 layout-exposing API defined in BIDI section)
   ==========================================================================

   ==========================================================================
                            - Make Custom Mule-aware
   ==========================================================================

   ==========================================================================
                         - Composite character support
   ==========================================================================

   ==========================================================================
                 - Language appropriate sorting and searching
   ==========================================================================

   ==========================================================================
                    - Glyph shaping for Arabic and Devanagari
   ==========================================================================

-  (needs to be handled mostly
  at C level, as part of layout; luckily it's entirely local in its
  changes, as this is not hard)


   ==========================================================================
    Consider moving language selection Menu up to be parallel with Mule menu
   ==========================================================================

*/


/************************************************************************/
/*                              declarations                            */
/************************************************************************/

Eistring the_eistring_zero_init, the_eistring_malloc_zero_init;

#ifdef MULE

#define MAX_CHARBPOS_GAP_SIZE_3 (65535/3)
#define MAX_BYTEBPOS_GAP_SIZE_3 (3 * MAX_CHARBPOS_GAP_SIZE_3)


short three_to_one_table[1 + MAX_BYTEBPOS_GAP_SIZE_3];

/* Table of number of bytes in the string representation of a character
   indexed by the first byte of that representation.
*/

#ifdef UNICODE_INTERNAL
const Bytecount rep_bytes_by_first_byte[256] =
{ /* 0x00 - 0x7f are for straight ASCII */
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
  /* 0x80 - 0xbf are not valid first bytes. A bare 0xDEADBEEF or
     DEADBEEF_CONSTANT without the cast runs foul of the C++ compiler. */
  (Bytecount) (DEADBEEF_CONSTANT), (Bytecount) (DEADBEEF_CONSTANT),
  (Bytecount) (DEADBEEF_CONSTANT), (Bytecount) (DEADBEEF_CONSTANT), 
  (Bytecount) (DEADBEEF_CONSTANT), (Bytecount) (DEADBEEF_CONSTANT),
  (Bytecount) (DEADBEEF_CONSTANT), (Bytecount) (DEADBEEF_CONSTANT),
  (Bytecount) (DEADBEEF_CONSTANT), (Bytecount) (DEADBEEF_CONSTANT),
  (Bytecount) (DEADBEEF_CONSTANT), (Bytecount) (DEADBEEF_CONSTANT),
  (Bytecount) (DEADBEEF_CONSTANT), (Bytecount) (DEADBEEF_CONSTANT),
  (Bytecount) (DEADBEEF_CONSTANT), (Bytecount) (DEADBEEF_CONSTANT),
  (Bytecount) (DEADBEEF_CONSTANT), (Bytecount) (DEADBEEF_CONSTANT),
  (Bytecount) (DEADBEEF_CONSTANT), (Bytecount) (DEADBEEF_CONSTANT),
  (Bytecount) (DEADBEEF_CONSTANT), (Bytecount) (DEADBEEF_CONSTANT),
  (Bytecount) (DEADBEEF_CONSTANT), (Bytecount) (DEADBEEF_CONSTANT),
  (Bytecount) (DEADBEEF_CONSTANT), (Bytecount) (DEADBEEF_CONSTANT),
  (Bytecount) (DEADBEEF_CONSTANT), (Bytecount) (DEADBEEF_CONSTANT),
  (Bytecount) (DEADBEEF_CONSTANT), (Bytecount) (DEADBEEF_CONSTANT),
  (Bytecount) (DEADBEEF_CONSTANT), (Bytecount) (DEADBEEF_CONSTANT),
  (Bytecount) (DEADBEEF_CONSTANT), (Bytecount) (DEADBEEF_CONSTANT),
  (Bytecount) (DEADBEEF_CONSTANT), (Bytecount) (DEADBEEF_CONSTANT),
  (Bytecount) (DEADBEEF_CONSTANT), (Bytecount) (DEADBEEF_CONSTANT),
  (Bytecount) (DEADBEEF_CONSTANT), (Bytecount) (DEADBEEF_CONSTANT),
  (Bytecount) (DEADBEEF_CONSTANT), (Bytecount) (DEADBEEF_CONSTANT),
  (Bytecount) (DEADBEEF_CONSTANT), (Bytecount) (DEADBEEF_CONSTANT),
  (Bytecount) (DEADBEEF_CONSTANT), (Bytecount) (DEADBEEF_CONSTANT),
  (Bytecount) (DEADBEEF_CONSTANT), (Bytecount) (DEADBEEF_CONSTANT),
  (Bytecount) (DEADBEEF_CONSTANT), (Bytecount) (DEADBEEF_CONSTANT),
  (Bytecount) (DEADBEEF_CONSTANT), (Bytecount) (DEADBEEF_CONSTANT),
  (Bytecount) (DEADBEEF_CONSTANT), (Bytecount) (DEADBEEF_CONSTANT),
  (Bytecount) (DEADBEEF_CONSTANT), (Bytecount) (DEADBEEF_CONSTANT),
  (Bytecount) (DEADBEEF_CONSTANT), (Bytecount) (DEADBEEF_CONSTANT),
  (Bytecount) (DEADBEEF_CONSTANT), (Bytecount) (DEADBEEF_CONSTANT),
  (Bytecount) (DEADBEEF_CONSTANT), (Bytecount) (DEADBEEF_CONSTANT),
  (Bytecount) (DEADBEEF_CONSTANT), (Bytecount) (DEADBEEF_CONSTANT),
  /* 0xc0 - 0xdf for 2-byte sequences */
  2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
  2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
  /* 0xe0 - 0xef for 3-byte sequences */
  3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3,
  /* 0xf0 - 0xf7 for 4-byte sequences;
     0xf8 - 0xfb for 5-byte sequences;
     0xfc - 0xfd for 6-byte sequences;
     0xfe, 0xff not allowed
   */
  4, 4, 4, 4, 4, 4, 4, 4, 5, 5, 5, 5, 6, 6,
  (Bytecount) (DEADBEEF_CONSTANT), (Bytecount) (DEADBEEF_CONSTANT)
};
#else
/* #### Maybe this table should be derived programmatically, at least
   the parts from 0x80 - 0x9D. */
const Bytecount rep_bytes_by_first_byte[0xA0] =
{ /* 0x00 - 0x7f are for straight ASCII */
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
  /* 0x80 - 0x8c/8d are for Dimension-1 official charsets */
  2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,
#ifndef ENABLE_COMPOSITE_CHARS
  2,
#else
  3,
#endif
  /* 0x8d/8e - 0x9d are for Dimension-2 official charsets */
  3, 3,
  /* 0x9e is for Dimension-1 private charsets */
  /* 0x9f is for Dimension-2 private charsets */
  3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 4
};
#endif /* UNICODE_INTERNAL */


#ifdef UNICODE_INTERNAL
/* For UTF-8 conversion */
unsigned int utf8_offsets_by_rep_bytes[7] = {0, 0, 0x3080, 0xE2080, 0x03C82080,
					     0xFA082080U, 0x82082080U};
#endif /* UNICODE_INTERNAL */

#ifdef ENABLE_COMPOSITE_CHARS

/* Hash tables for composite chars.  One maps string representing
   composed chars to their equivalent chars; one goes the
   other way. */
Lisp_Object Vcomposite_char_char2string_hash_table;
Lisp_Object Vcomposite_char_string2char_hash_table;

static int composite_char_row_next;
static int composite_char_col_next;

#endif /* ENABLE_COMPOSITE_CHARS */
#endif /* MULE */

/* For UTF-8 conversion */

int firstbyte_mask[7] = {0, 0, 0xC0, 0xE0, 0xF0, 0xF8, 0xFC};

Lisp_Object QSin_char_byte_conversion, QSin_byte_char_conversion;
Lisp_Object QSin_internal_external_conversion, QSin_external_internal_conversion;

Lisp_Object /* Qfail, Qsubstitute, */ Quse_private;

Fixnum Vchar_code_limit;

/************************************************************************/
/*                       Basic Ichar functions                          */
/************************************************************************/

#ifdef MULE

/* Convert a non-ASCII Mule character C into a one-character Mule-encoded
   string in STR.  Returns the number of bytes stored.
   Do not call this directly.  Use the macro set_itext_ichar() instead.
 */

Bytecount
non_ascii_set_itext_ichar (Ibyte *str, Ichar c)
{
#ifdef UNICODE_INTERNAL
  /* #### This code is duplicated in encode_unicode_to_dynarr in unicode.c.
     There should be a better way. */
  REGISTER Bytecount bytes;

  text_checking_assert (c >= 0x80);
  ASSERT_VALID_ICHAR (c);
  if (c <= 0x7ff) bytes = 2;
  else if (c <= 0xffff) bytes = 3;
  else if (c <= 0x1fffff) bytes = 4;
  else if (c <= 0x3ffffff) bytes = 5;
  else bytes = 6;
    
  str += bytes;
  switch (bytes)
    {
    case 6:*--str = (c | 0x80) & 0xBF; c >>= 6;
    case 5:*--str = (c | 0x80) & 0xBF; c >>= 6;
    case 4:*--str = (c | 0x80) & 0xBF; c >>= 6;
    case 3:*--str = (c | 0x80) & 0xBF; c >>= 6;
    case 2:*--str = (c | 0x80) & 0xBF; c >>= 6;
    case 1:*--str = c | firstbyte_mask[bytes];
    }

  ASSERT_VALID_ITEXT (str);
  return bytes;
#else
  Lisp_Object charset;
  int c1, c2;
  Bytecount bytes;

  old_mule_non_ascii_ichar_to_charset_codepoint_raw (c, &charset, &c1, &c2);
  bytes = old_mule_non_ascii_charset_codepoint_to_itext_raw (charset, c1, c2,
							     str);
  text_checking_assert (bytes > 0);
  return bytes;
#endif /* UNICODE_INTERNAL */
}

/* Return the first character from a Mule-encoded string in STR,
   assuming it's non-ASCII.  Do not call this directly.
   Use the macro itext_ichar() instead. */

Ichar
non_ascii_itext_ichar (const Ibyte *str)
{
#ifdef UNICODE_INTERNAL
  /* #### Conversion from UTF8 also happens in unicode_convert in unicode.c.
     The algorithm is very different, however (this algorithm is the optimized
     kind from Appendix A of the Unicode Standard, V 2.0).  The other situation
     would be tricky to convert using this algorithm since (a) it gets
     arbitrarily buffered data, which may cross character boundaries; (b)
     there's no guarantee the UTF8 is correct (maybe we misguessed and we're
     reading totally random bytes and trying to interpret them as UTF8). */
  unsigned int ch = 0;
  int bytes = rep_bytes_by_first_byte (*str);
  Ichar ich;

  /* The ASCII case should already have been filtered out. */
  text_checking_assert (!byte_ascii_p (*str));
  ASSERT_VALID_ITEXT (str);

  switch (bytes)
    {
    case 6: ch += *str++; ch <<= 6;
    case 5: ch += *str++; ch <<= 6;
    case 4: ch += *str++; ch <<= 6;
    case 3: ch += *str++; ch <<= 6;
    case 2: ch += *str++; ch <<= 6;
    case 1: ch += *str;
    }

  ich = (Ichar) (ch - utf8_offsets_by_rep_bytes[bytes]);
  ASSERT_VALID_ICHAR (ich);
  return ich;
#else /* not UNICODE_INTERNAL */
  Lisp_Object charset;
  int c1, c2;
  Ichar ich;

  old_mule_non_ascii_itext_to_charset_codepoint_raw (str, &charset, &c1, &c2);
  ich = old_mule_non_ascii_charset_codepoint_to_ichar_raw (charset, c1, c2);
  ASSERT_VALID_ICHAR (ich);
  return ich;
#endif /* UNICODE_INTERNAL */
}

/* Convert a charset codepoint (guaranteed not to be ASCII) into a
   character in the internal string representation.  Return number
   of bytes written out.  FAIL controls what happens when the charset
   codepoint cannot be converted to Unicode. */
Bytecount
non_ascii_charset_codepoint_to_itext (Lisp_Object charset, int c1, int c2,
				      Ibyte *ptr, enum converr fail)
{
  Ichar ch;

  text_checking_assert (!EQ (charset, Vcharset_ascii));
  ch = charset_codepoint_to_ichar (charset, c1, c2, fail);

  if (ch < 0)
    return 0;
  /* We can't rely on the converted character being non-ASCII.  For
     example, JISX0208 codepoint (33, 64) == Unicode 0x5C (ASCII
     backslash). */
  return set_itext_ichar (ptr, ch);
}

#ifndef UNICODE_INTERNAL

/* Return whether CH corresponds to a valid Ichar. Do not call this
   directly.  Use the macro valid_ichar_p() instead. */

Boolint
old_mule_non_ascii_valid_ichar_p (EMACS_INT ch)
{
  int f1, f2, f3;

  /* Must have only lowest 21 bits set */
  if (ch & ~(CHAR_CODE_LIMIT - 1))
    return 0;

  f1 = ichar_field1 (ch);
  f2 = ichar_field2 (ch);
  f3 = ichar_field3 (ch);

  if (f1 == 0)
    {
      /* dimension-1 char */
      Lisp_Object charset =
	charset_by_encodable_id (f2 + FIELD2_TO_CHARSET_ID);
      /* charset exists and of the correct dimension */
      if (NILP (charset) || XCHARSET_DIMENSION (charset) == 2)
	return 0;
      /* octet not out of range */
      if (f3 < 0x20)
	return 0;
      /* check range as per size (94 or 96) of charset */
      return ((f3 > 0x20 && f3 < 0x7f) || XCHARSET_CHARS (charset, 1) == 96);
    }
  else
    {
      /* dimension-2 char */
      Lisp_Object charset =
	charset_by_encodable_id (f1 <= MAX_ICHAR_FIELD1_OFFICIAL ?
				 f1 + FIELD1_TO_OFFICIAL_CHARSET_ID :
				 f1 + FIELD1_TO_PRIVATE_CHARSET_ID);
      /* charset exists and of the correct dimension */
      if (NILP (charset) || XCHARSET_DIMENSION (charset) == 1)
	return 0;
      /* octets not out of range */
      if (f2 < 0x20 || f3 < 0x20)
	return 0;
#ifdef ENABLE_COMPOSITE_CHARS
      if (EQ (charset, Vcharset_composite))
	{
	  if (UNBOUNDP (Fgethash (make_fixnum (ch),
				  Vcomposite_char_char2string_hash_table,
				  Qunbound)))
	    return 0;
	  return 1;
	}
#endif /* ENABLE_COMPOSITE_CHARS */
      /* check range as per size (94x94 or 96x96) of charset */
      return
	(((f2 != 0x20 && f2 != 0x7F) || XCHARSET_CHARS (charset, 0) == 96) &&
	 ((f3 != 0x20 && f3 != 0x7F) || XCHARSET_CHARS (charset, 1) == 96));
    }
}

/* Separate an Ichar into its components.  The charset of character C is
   stored in CHARSET, and the position-codes of C are stored in C1 and C2.
   A dimension-1 character has a C1 of 0.  */

void
old_mule_non_ascii_ichar_to_charset_codepoint_raw (Ichar c,
						   Lisp_Object *charset,
						   int *c1, int *c2)
{
  text_checking_assert (c >= 0x80);
  ASSERT_VALID_ICHAR (c);
  if (c <= 0x9F)
    {
      *charset = Vcharset_control_1;
      *c1 = 0;
      *c2 = (int) c;
    }
  else
    {
      *charset = charset_by_encodable_id (old_mule_ichar_charset_id (c));
      *c1 = XCHARSET_DIMENSION (*charset) == 1 ? 0 : ichar_field2 (c);
      *c2 = ichar_field3 (c);
      if (XCHARSET_OFFSET (*charset, 0) >= 128)
	*c1 += 128;
      if (XCHARSET_OFFSET (*charset, 1) >= 128)
	*c2 += 128;
    }
  ASSERT_VALID_CHARSET_CODEPOINT (*charset, *c1, *c2);
}

/* Convert a character in the internal string representation (guaranteed
   not to be ASCII) into a charset codepoint. */
void
old_mule_non_ascii_itext_to_charset_codepoint_raw (const Ibyte *ptr,
						   Lisp_Object *charset,
						   int *c1, int *c2)
{
  int id;

  text_checking_assert (!byte_ascii_p (*ptr));
  ASSERT_VALID_ITEXT (ptr);

  if (*ptr == LEAD_BYTE_PRIVATE_1)
    id = byte_id_to_private_charset_id (*++ptr, 1);
  else if (*ptr == LEAD_BYTE_PRIVATE_2)
    id = byte_id_to_private_charset_id (*++ptr, 2);
  else
    id = byte_id_to_official_charset_id (*ptr);

  *charset = charset_by_encodable_id (id);
  if (XCHARSET_DIMENSION (*charset) == 2)
    *c1 = *++ptr & 0x7F;
  else
    *c1 = 0;
  *c2 = *++ptr & 0x7F;
  if (XCHARSET_OFFSET (*charset, 0) >= 128)
    *c1 += 128;
  if (XCHARSET_OFFSET (*charset, 1) >= 128)
    *c2 += 128;
  if (EQ (*charset, Vcharset_control_1))
    *c2 -= 0x20;

  ASSERT_VALID_CHARSET_CODEPOINT (*charset, *c1, *c2);
}

Ichar
old_mule_non_ascii_charset_codepoint_to_ichar_raw (Lisp_Object charset,
						   int c1, int c2)
{
  Ichar retval;

  text_checking_assert (!EQ (charset, Vcharset_ascii));
  ASSERT_VALID_CHARSET_CODEPOINT (charset, c1, c2);
  if (EQ (charset, Vcharset_control_1))
    retval = c2;
  else
    {
      int id = XCHARSET_ID (charset);
      if (id > MAX_ENCODABLE_CHARSET_ID)
	return -1;
      c1 &= 127;
      c2 &= 127;
      if (XCHARSET_DIMENSION (charset) == 1)
	retval = ((id - FIELD2_TO_CHARSET_ID) << 7) | (c2);
      else
	retval =
	  ((id - (id >= MIN_PRIVATE_CHARSET_ID ?
		  FIELD1_TO_PRIVATE_CHARSET_ID :
		  FIELD1_TO_OFFICIAL_CHARSET_ID)) << 14) |
	  ((c1) << 7) | (c2);
    }

  ASSERT_VALID_ICHAR (retval);
  return retval;
}

/* Convert a charset codepoint (guaranteed not to be ASCII) into a
   character in the old-Mule internal string representation.  Return number
   of bytes written out. */

Bytecount
old_mule_non_ascii_charset_codepoint_to_itext_raw (Lisp_Object charset,
						   int c1, int c2, Ibyte *ptr)
{
  Ibyte *p = ptr;
  int id = XCHARSET_ID (charset);

  text_checking_assert (!EQ (charset, Vcharset_ascii));
  ASSERT_VALID_CHARSET_CODEPOINT (charset, c1, c2);

  /* We are only called as a result of breaking a character into charset
     and octets; so non-encodable charsets that cannot form a character
     should never occur */
  text_checking_assert (id <= MAX_ENCODABLE_CHARSET_ID);

  if (EQ (charset, Vcharset_control_1))
    c2 += 0x20;

  if (id >= MIN_PRIVATE_CHARSET_ID)
    {
    *p++ = (XCHARSET_DIMENSION (charset) == 1 ? LEAD_BYTE_PRIVATE_1 :
	    LEAD_BYTE_PRIVATE_2);
    *p++ = private_charset_id_to_byte_id (id, XCHARSET_DIMENSION (charset));
    }
  else
    *p++ = official_charset_id_to_byte_id (id);
  if (XCHARSET_DIMENSION (charset) == 2)
    *p++ = c1 | 0x80;
  *p++ = c2 | 0x80;

  ASSERT_VALID_ITEXT (ptr);
  return (p - ptr);
}

int
old_mule_ichar_columns (Ichar c)
{
  return XCHARSET_COLUMNS (charset_by_encodable_id
			   (old_mule_ichar_charset_id (c)));
}

int
old_mule_ichar_to_unicode (Ichar chr, enum converr fail)
{
  ASSERT_VALID_ICHAR (chr);
  text_checking_assert (chr >= 256); /* The inline function should handle
                                        ASCII, control-1, Latin 1. */
  {
    int c1, c2;
    Lisp_Object charset;

    old_mule_non_ascii_ichar_to_charset_codepoint_raw (chr, &charset, &c1,
						       &c2);
    return charset_codepoint_to_unicode (charset, c1, c2, fail);
  }
}

/* @@#### This is a big kludge.  It could fail, e.g., if the predicate in
   question triggers a recursive call to old_mule_unicode_to_ichar().
   It would also fail if we parallelized XEmacs (not bloody likely).
   A better fix is to change the predicate to take an extra "void *" argument
   and add an extra void * argument to all functions that take a predicate
   argument (yuck).  Alternatively, make the predicate argument a structure
   and have it contain the predicate and extra argument.  That way there's
   only one argument, a structure. */
static charset_pred omu2i_kludge_predicate;

static int
omu2i_predicate (Lisp_Object charset)
{
  return (old_mule_charset_encodable (charset) &&
	  omu2i_kludge_predicate (charset));
}

/* Convert a Unicode codepoint to an Ichar.  Return value will correspond
   to FAIL -- possibly (Ichar) -1, a substituted character, or something
   else. */

Ichar
old_mule_unicode_to_ichar (int code, Lisp_Object precarray,
			   charset_pred predicate, enum converr fail)
{
  Lisp_Object charset;
  int c1, c2;
  charset_pred mypred;

  /* This shortcut depends on the representation of an Ichar, see text.c.
     It could potentially be extended to 0x80 to 0x9F, if we bothered to
     keep track of whether there are any Unicode mappings in the range
     0x80 to 0x9F that override Control-1, similar to what we do below
     for ASCII.

     Note that it may _not_ be extended to U+00A0 to U+00FF (many ISO 8859
     coded character sets have points that map into that region, so this
     function is many-valued).
  */
  if (code <= 0x7F && !XPRECEDENCE_ARRAY (precarray)->has_overriding_ascii
      && (!predicate || (*predicate) (Vcharset_ascii)))
    return (Ichar) code;

  /* Convert to a charset codepoint, but ignore charsets that can't be
     encoded into text */
  if (!predicate)
    mypred = old_mule_charset_encodable;
  else
    {
      mypred = omu2i_predicate;
      omu2i_kludge_predicate = predicate;
    }
  unicode_to_charset_codepoint_raw (code, precarray, mypred,
				    &charset, &c1, &c2);
  if (NILP (charset))
    HANDLE_ICHAR_ERROR ("Can't convert Unicode codepoint to character",
			make_fixnum (code), fail);
  return charset_codepoint_to_ichar (charset, c1, c2, CONVERR_FAIL);
}

static Ichar
old_mule_round_up_to_valid_ichar (int charpos)
{
  int i, dim;

  /* Within a particular dimension (1 or 2), charset ID order corresponds
     to character order.  Furthermore, all dimension-1 characters are less
     than all dimension-2 characters.  This is not the case with charset
     ID's, however.  So we have to loop twice over the range of encodable
     ID's, once per dimension.  We could shorten things somewhat by using
     more specific knowledge about exactly which ID's are assigned to which
     dimensions, but that would be more fragile, and it's unlikely that
     this way takes a significant amount of time as there are only 225 or
     so possible encodable ID's. */
  for (dim = 1; dim <= 2; dim++)
    {
      for (i = MIN_ENCODABLE_CHARSET_ID; i <= MAX_ENCODABLE_CHARSET_ID; i++)
	{
	  Lisp_Object charset = charset_by_encodable_id (i);
	  if (!NILP (charset) && XCHARSET_DIMENSION (charset) == dim)
	    {
	      int l1, l2, h1, h2;
	      Ichar minchar, maxchar;
	      get_charset_limits (charset, &l1, &l2, &h1, &h2);
	      minchar = charset_codepoint_to_ichar (charset, l1, l2,
						    CONVERR_ABORT);
	      maxchar = charset_codepoint_to_ichar (charset, h1, h2,
						    CONVERR_ABORT);
	      /* Either we are between charsets, or in a gap within a
		 charset. */
	      if (charpos < minchar)
		/* We are between charsets */
		return minchar;
	      if (charpos < maxchar)
		{
		  /* We are in a gap.  The gaps aren't more than 34 characters
		     wide, so just move up till we find the end of the gap. */
		  while (!valid_ichar_p (charpos))
		    charpos++;
		  return charpos;
		}
	    }
	}
    }

  return -1;
}

static Ichar
old_mule_round_down_to_valid_ichar (int charpos)
{
  int i, dim;

  for (dim = 2; dim >= 1; dim--)
    {
      for (i = MAX_ENCODABLE_CHARSET_ID; i >= MIN_ENCODABLE_CHARSET_ID; i--)
	{
	  Lisp_Object charset = charset_by_encodable_id (i);
	  if (!NILP (charset) && XCHARSET_DIMENSION (charset) == dim)
	    {
	      int l1, l2, h1, h2;
	      Ichar minchar, maxchar;
	      get_charset_limits (charset, &l1, &l2, &h1, &h2);
	      minchar = charset_codepoint_to_ichar (charset, l1, l2,
						    CONVERR_ABORT);
	      maxchar = charset_codepoint_to_ichar (charset, h1, h2,
						    CONVERR_ABORT);
	      if (charpos > maxchar)
		return maxchar;
	      if (charpos > minchar)
		{
		  while (!valid_ichar_p (charpos))
		    charpos--;
		  return charpos;
		}
	    }
	}
    }

  return -1;
}

#else /* if defined (UNICODE_INTERNAL) */

int
unicode_internal_handle_bad_ichar_to_unicode (Ichar chr, enum converr fail)
{
  ASSERT_VALID_ICHAR (chr);
  text_checking_assert (fail != CONVERR_USE_PRIVATE);
  text_checking_assert (!valid_unicode_codepoint_p (chr,
                                                    UNICODE_OFFICIAL_ONLY));
  switch (fail)
    {
    case CONVERR_FAIL:
      return -1;
    case CONVERR_ERROR:
      if (chr > UNICODE_OFFICIAL_MAX)
        {
          args_out_of_range_3 (make_fixnum (chr), Qzero,
                               make_fixnum (UNICODE_OFFICIAL_MAX));
        }
      if (chr >= FIRST_UTF_16_SURROGATE)
        {
          args_out_of_range_3 (make_fixnum (chr), Qzero,
                               make_fixnum (FIRST_UTF_16_SURROGATE));
        }
      /* FALLTHROUGH */
    case CONVERR_ABORT:
    case CONVERR_USE_PRIVATE:
    default:
      ABORT ();
    case CONVERR_SUCCEED:
    case CONVERR_SUBSTITUTE:
      return 0xFFFD;
    }
}

#endif /* defined (UNICODE_INTERNAL) */

#endif /* MULE */

/* Take a possibly invalid Ichar value (must be >= 0) and move upwards as
   necessary until we find the first valid Ichar.  Return -1 if we're above
   all valid Ichars. */

Ichar
round_up_to_valid_ichar (int charpos)
{
  text_checking_assert (charpos >= 0);
  if (valid_ichar_p (charpos))
    return (Ichar) charpos;
#ifdef UNICODE_INTERNAL
  if (valid_unicode_surrogate (charpos))
    return (Ichar) (LAST_UTF_16_SURROGATE + 1);
  text_checking_assert (charpos >= CHAR_CODE_LIMIT);
  return -1;
#elif defined (MULE)
  return old_mule_round_up_to_valid_ichar (charpos);
#else
  return -1;
#endif
}

/* Take a possibly invalid Ichar value (must be >= 0) and move downwards as
   necessary until we find the first valid Ichar. */

Ichar
round_down_to_valid_ichar (int charpos)
{
  text_checking_assert (charpos >= 0);
  if (valid_ichar_p (charpos))
    return (Ichar) charpos;
#ifdef UNICODE_INTERNAL
  if (charpos >= CHAR_CODE_LIMIT)
    return CHAR_CODE_LIMIT - 1;
  text_checking_assert (valid_unicode_surrogate (charpos));
  return (Ichar) (FIRST_UTF_16_SURROGATE - 1);
#elif defined (MULE)
  return old_mule_round_down_to_valid_ichar (charpos);
#else
  return 255;
#endif
}

/****************************************************************************/
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*              Everything above here knows about the specifics of          */
/*              the internal character and text formats.  Nothing           */
/*              below or anywhere else knows, except text.h.                */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/****************************************************************************/


/************************************************************************/
/*                          qxestr***() functions                       */
/************************************************************************/

/* Most are inline functions in lisp.h */

/* strcasecmp() implementation from BSD */
static Ibyte strcasecmp_charmap[] = {
        0000, 0001, 0002, 0003, 0004, 0005, 0006, 0007,
        0010, 0011, 0012, 0013, 0014, 0015, 0016, 0017,
        0020, 0021, 0022, 0023, 0024, 0025, 0026, 0027,
        0030, 0031, 0032, 0033, 0034, 0035, 0036, 0037,
        0040, 0041, 0042, 0043, 0044, 0045, 0046, 0047,
        0050, 0051, 0052, 0053, 0054, 0055, 0056, 0057,
        0060, 0061, 0062, 0063, 0064, 0065, 0066, 0067,
        0070, 0071, 0072, 0073, 0074, 0075, 0076, 0077,
        0100, 0141, 0142, 0143, 0144, 0145, 0146, 0147,
        0150, 0151, 0152, 0153, 0154, 0155, 0156, 0157,
        0160, 0161, 0162, 0163, 0164, 0165, 0166, 0167,
        0170, 0171, 0172, 0133, 0134, 0135, 0136, 0137,
        0140, 0141, 0142, 0143, 0144, 0145, 0146, 0147,
        0150, 0151, 0152, 0153, 0154, 0155, 0156, 0157,
        0160, 0161, 0162, 0163, 0164, 0165, 0166, 0167,
        0170, 0171, 0172, 0173, 0174, 0175, 0176, 0177,
        0200, 0201, 0202, 0203, 0204, 0205, 0206, 0207,
        0210, 0211, 0212, 0213, 0214, 0215, 0216, 0217,
        0220, 0221, 0222, 0223, 0224, 0225, 0226, 0227,
        0230, 0231, 0232, 0233, 0234, 0235, 0236, 0237,
        0240, 0241, 0242, 0243, 0244, 0245, 0246, 0247,
        0250, 0251, 0252, 0253, 0254, 0255, 0256, 0257,
        0260, 0261, 0262, 0263, 0264, 0265, 0266, 0267,
        0270, 0271, 0272, 0273, 0274, 0275, 0276, 0277,
        0300, 0301, 0302, 0303, 0304, 0305, 0306, 0307,
        0310, 0311, 0312, 0313, 0314, 0315, 0316, 0317,
        0320, 0321, 0322, 0323, 0324, 0325, 0326, 0327,
        0330, 0331, 0332, 0333, 0334, 0335, 0336, 0337,
        0340, 0341, 0342, 0343, 0344, 0345, 0346, 0347,
        0350, 0351, 0352, 0353, 0354, 0355, 0356, 0357,
        0360, 0361, 0362, 0363, 0364, 0365, 0366, 0367,
        0370, 0371, 0372, 0373, 0374, 0375, 0376, 0377
};

/* A version that works like generic strcasecmp() -- only collapsing
   case in ASCII A-Z/a-z.  This is safe on Mule strings due to the
   current representation.

   This version was written by some Berkeley coder, favoring
   nanosecond improvements over clarity.  In all other versions below,
   we use symmetrical algorithms that may sacrifice a few machine
   cycles but are MUCH MUCH clearer, which counts a lot more.
*/

int
qxestrcasecmp (const Ibyte *s1, const Ibyte *s2)
{
  Ibyte *cm = strcasecmp_charmap;

  while (cm[*s1] == cm[*s2++])
    if (*s1++ == '\0')
      return (0);

  return (cm[*s1] - cm[*--s2]);
}

int
ascii_strcasecmp (const Ascbyte *s1, const Ascbyte *s2)
{
  return qxestrcasecmp ((const Ibyte *) s1, (const Ibyte *) s2);
}

int
qxestrcasecmp_ascii (const Ibyte *s1, const Ascbyte *s2)
{
  return qxestrcasecmp (s1, (const Ibyte *) s2);
}

/* An internationalized version that collapses case in a general fashion.
 */

int
qxestrcasecmp_i18n (const Ibyte *s1, const Ibyte *s2)
{
  while (*s1 && *s2)
    {
      if (CANONCASE (0, itext_ichar (s1)) !=
	  CANONCASE (0, itext_ichar (s2)))
	break;
      INC_IBYTEPTR (s1);
      INC_IBYTEPTR (s2);
    }

  return (CANONCASE (0, itext_ichar (s1)) -
	  CANONCASE (0, itext_ichar (s2)));
}

/* The only difference between these next two and
   qxememcasecmp()/qxememcasecmp_i18n() is that these two will stop if
   both strings are equal and less than LEN in length, while
   the mem...() versions would would run off the end. */

int
qxestrncasecmp (const Ibyte *s1, const Ibyte *s2, Bytecount len)
{
  Ibyte *cm = strcasecmp_charmap;

  while (len--)
    {
      int diff = cm[*s1] - cm[*s2];
      if (diff != 0)
	return diff;
      if (!*s1)
	return 0;
      s1++, s2++;
    }

  return 0;
}

int
ascii_strncasecmp (const Ascbyte *s1, const Ascbyte *s2, Bytecount len)
{
  return qxestrncasecmp ((const Ibyte *) s1, (const Ibyte *) s2, len);
}

int
qxestrncasecmp_ascii (const Ibyte *s1, const Ascbyte *s2, Bytecount len)
{
  return qxestrncasecmp (s1, (const Ibyte *) s2, len);
}

/* Compare LEN_FROM_S1 worth of characters from S1 with the same number of
   characters from S2, case insensitive.  NOTE: Downcasing can convert
   characters from one length in bytes to another, so reversing S1 and S2
   is *NOT* a symmetric operations!  You must choose a length that agrees
   with S1. */

int
qxestrncasecmp_i18n (const Ibyte *s1, const Ibyte *s2,
		     Bytecount len_from_s1)
{
  while (len_from_s1 > 0)
    {
      const Ibyte *old_s1 = s1;
      int diff = (CANONCASE (0, itext_ichar (s1)) -
		  CANONCASE (0, itext_ichar (s2)));
      if (diff != 0)
	return diff;
      if (!*s1)
	return 0;
      INC_IBYTEPTR (s1);
      INC_IBYTEPTR (s2);
      len_from_s1 -= s1 - old_s1;
    }

  return 0;
}

int
qxememcmp (const Ibyte *s1, const Ibyte *s2, Bytecount len)
{
  return memcmp (s1, s2, len);
}

int
qxememcmp4 (const Ibyte *s1, Bytecount len1,
	    const Ibyte *s2, Bytecount len2)
{
  int retval = qxememcmp (s1, s2, min (len1, len2));
  if (retval)
    return retval;
  return len1 - len2;
}

int
qxememcasecmp (const Ibyte *s1, const Ibyte *s2, Bytecount len)
{
  Ibyte *cm = strcasecmp_charmap;

  while (len--)
    {
      int diff = cm[*s1] - cm[*s2];
      if (diff != 0)
	return diff;
      s1++, s2++;
    }

  return 0;
}

int
qxememcasecmp4 (const Ibyte *s1, Bytecount len1,
		const Ibyte *s2, Bytecount len2)
{
  int retval = qxememcasecmp (s1, s2, min (len1, len2));
  if (retval)
    return retval;
  return len1 - len2;
}

/* Do a character-by-character comparison, returning "which is greater" by
   comparing the Ichar values. (#### Should have option to compare Unicode
   points) */

int
qxetextcmp (const Ibyte *s1, Bytecount len1,
	    const Ibyte *s2, Bytecount len2)
{
  while (len1 > 0 && len2 > 0)
    {
      const Ibyte *old_s1 = s1;
      const Ibyte *old_s2 = s2;
      int diff = itext_ichar (s1) - itext_ichar (s2);
      if (diff != 0)
	return diff;
      INC_IBYTEPTR (s1);
      INC_IBYTEPTR (s2);
      len1 -= s1 - old_s1;
      len2 -= s2 - old_s2;
    }

  assert (len1 >= 0 && len2 >= 0);
  return len1 - len2;
}

int
qxetextcmp_matching (const Ibyte *s1, Bytecount len1,
		     const Ibyte *s2, Bytecount len2,
		     Charcount *matching)
{
  *matching = 0;
  while (len1 > 0 && len2 > 0)
    {
      const Ibyte *old_s1 = s1;
      const Ibyte *old_s2 = s2;
      int diff = itext_ichar (s1) - itext_ichar (s2);
      if (diff != 0)
	return diff;
      INC_IBYTEPTR (s1);
      INC_IBYTEPTR (s2);
      len1 -= s1 - old_s1;
      len2 -= s2 - old_s2;
      (*matching)++;
    }

  assert (len1 >= 0 && len2 >= 0);
  return len1 - len2;
}

/* Do a character-by-character comparison, returning "which is greater" by
   comparing the Ichar values, case insensitively (by downcasing both
   first). (#### Should have option to compare Unicode points)

   In this case, both lengths must be specified becaused downcasing can
   convert characters from one length in bytes to another; therefore, two
   blocks of text of different length might be equal.  If both compare
   equal up to the limit in length of one but not the other, the longer one
   is "greater". */

int
qxetextcasecmp (const Ibyte *s1, Bytecount len1,
		const Ibyte *s2, Bytecount len2)
{
  while (len1 > 0 && len2 > 0)
    {
      const Ibyte *old_s1 = s1;
      const Ibyte *old_s2 = s2;
      int diff = (CANONCASE (0, itext_ichar (s1)) -
		  CANONCASE (0, itext_ichar (s2)));
      if (diff != 0)
	return diff;
      INC_IBYTEPTR (s1);
      INC_IBYTEPTR (s2);
      len1 -= s1 - old_s1;
      len2 -= s2 - old_s2;
    }

  assert (len1 >= 0 && len2 >= 0);
  return len1 - len2;
}

/* Like qxetextcasecmp() but also return number of characters at
   beginning that match. */

int
qxetextcasecmp_matching (const Ibyte *s1, Bytecount len1,
			 const Ibyte *s2, Bytecount len2,
			 Charcount *matching)
{
  *matching = 0;
  while (len1 > 0 && len2 > 0)
    {
      const Ibyte *old_s1 = s1;
      const Ibyte *old_s2 = s2;
      int diff = (CANONCASE (0, itext_ichar (s1)) -
		  CANONCASE (0, itext_ichar (s2)));
      if (diff != 0)
	return diff;
      INC_IBYTEPTR (s1);
      INC_IBYTEPTR (s2);
      len1 -= s1 - old_s1;
      len2 -= s2 - old_s2;
      (*matching)++;
    }

  assert (len1 >= 0 && len2 >= 0);
  return len1 - len2;
}

int
lisp_strcasecmp_ascii (Lisp_Object s1, Lisp_Object s2)
{
  Ibyte *cm = strcasecmp_charmap;
  Ibyte *p1 = XSTRING_DATA (s1);
  Ibyte *p2 = XSTRING_DATA (s2);
  Ibyte *e1 = p1 + XSTRING_LENGTH (s1);
  Ibyte *e2 = p2 + XSTRING_LENGTH (s2);

  /* again, we use a symmetric algorithm and favor clarity over
     nanosecond improvements. */
  while (1)
    {
      /* if we reached the end of either string, compare lengths.
	 do NOT compare the final null byte against anything, in case
	 the other string also has a null byte at that position. */
      if (p1 == e1 || p2 == e2)
	return e1 - e2;
      if (cm[*p1] != cm[*p2])
	return cm[*p1] - cm[*p2];
      p1++, p2++;
    }
}

int
lisp_strcasecmp_i18n (Lisp_Object s1, Lisp_Object s2)
{
  return qxetextcasecmp (XSTRING_DATA (s1), XSTRING_LENGTH (s1),
			 XSTRING_DATA (s2), XSTRING_LENGTH (s2));
}

/* Compare a wide string with an ASCII string */

int
wcscmp_ascii (const wchar_t *s1, const Ascbyte *s2)
{
  while (*s1 && *s2)
    {
      if (*s1 != (wchar_t) *s2)
       break;
      s1++, s2++;
    }

  return *s1 - *s2;
}

int
wcsncmp_ascii (const wchar_t *s1, const Ascbyte *s2, Charcount len)
{
  while (len--)
    {
      int diff = *s1 - *s2;
      if (diff != 0)
	return diff;
      if (!*s1)
	return 0;
      s1++, s2++;
    }

  return 0;
}


/************************************************************************/
/*               conversion between textual representations             */
/************************************************************************/

Charcount
convert_ibyte_string_into_ichar_string (const Ibyte *str, Bytecount len,
					Ichar *arr)
{
  const Ibyte *strend = str + len;
  Charcount newlen = 0;
  while (str < strend)
    {
      Ichar ch = itext_ichar (str);
      arr[newlen++] = ch;
      INC_IBYTEPTR (str);
    }
  return newlen;
}

#define COPY_TEXT_BETWEEN_FORMATS(srcfmt, dstfmt)			\
do									\
{									\
  if (dst)								\
    {									\
      Ibyte *dstend = dst + dstlen;					\
      Ibyte *dstp = dst;						\
      const Ibyte *srcend = src + srclen;				\
      const Ibyte *srcp = src;						\
									\
      while (srcp < srcend)						\
	{								\
	  Ichar ch = itext_ichar_fmt (srcp, srcfmt, srcobj);		\
	  Bytecount len = ichar_len_fmt (ch, dstfmt);			\
									\
	    if (dstp + len <= dstend)					\
	      {								\
		(void) set_itext_ichar_fmt (dstp, ch, dstfmt, dstobj);	\
		dstp += len;						\
	      }								\
	    else							\
	      break;							\
	  INC_IBYTEPTR_FMT (srcp, srcfmt);				\
	}								\
      text_checking_assert (srcp <= srcend);				\
      if (src_used)							\
	*src_used = srcp - src;						\
      return dstp - dst;						\
    }									\
  else									\
    {									\
      const Ibyte *srcend = src + srclen;				\
      const Ibyte *srcp = src;						\
      Bytecount total = 0;						\
									\
      while (srcp < srcend)						\
	{								\
	  total += ichar_len_fmt (itext_ichar_fmt (srcp, srcfmt,	\
						       srcobj), dstfmt); \
	  INC_IBYTEPTR_FMT (srcp, srcfmt);				\
	}								\
      text_checking_assert (srcp == srcend);				\
      if (src_used)							\
	*src_used = srcp - src;						\
      return total;							\
    }									\
}									\
while (0)

/* Copy as much text from SRC/SRCLEN to DST/DSTLEN as will fit, converting
   from SRCFMT/SRCOBJ to DSTFMT/DSTOBJ.  Return number of bytes stored into
   DST as return value, and number of bytes copied from SRC through
   SRC_USED (if not NULL).  If DST is NULL, don't actually store anything
   and just return the size needed to store all the text.  Will not copy
   partial characters into DST. */

Bytecount
copy_text_between_formats (const Ibyte *src, Bytecount srclen,
			   Internal_Format srcfmt,
			   Lisp_Object USED_IF_MULE (srcobj),
			   Ibyte *dst, Bytecount dstlen,
			   Internal_Format dstfmt,
			   Lisp_Object USED_IF_MULE (dstobj),
			   Bytecount *src_used)
{
  if (srcfmt == dstfmt &&
      objects_have_same_internal_representation (srcobj, dstobj))
    {
      if (dst)
	{
	  srclen = min (srclen, dstlen);
	  srclen = validate_ibyte_string_backward (src, srclen);
	  memcpy (dst, src, srclen);
	  if (src_used)
	    *src_used = srclen;
	  return srclen;
	}
      else
	return srclen;
    }
  /* Everything before the final else statement is an optimization.
     The inner loops inside COPY_TEXT_BETWEEN_FORMATS() have a number
     of calls to *_fmt(), each of which has a switch statement in it.
     By using constants as the FMT argument, these switch statements
     will be optimized out of existence. */
#define ELSE_FORMATS(fmt1, fmt2)		\
  else if (srcfmt == fmt1 && dstfmt == fmt2)	\
    COPY_TEXT_BETWEEN_FORMATS (fmt1, fmt2)
  ELSE_FORMATS (FORMAT_DEFAULT, FORMAT_8_BIT_FIXED);
  ELSE_FORMATS (FORMAT_8_BIT_FIXED, FORMAT_DEFAULT);
  ELSE_FORMATS (FORMAT_DEFAULT, FORMAT_32_BIT_FIXED);
  ELSE_FORMATS (FORMAT_32_BIT_FIXED, FORMAT_DEFAULT);
  else
    COPY_TEXT_BETWEEN_FORMATS (srcfmt, dstfmt);
#undef ELSE_FORMATS
}

/* Copy as much buffer text in BUF, starting at POS, of length LEN, as will
   fit into DST/DSTLEN, converting to DSTFMT.  Return number of bytes
   stored into DST as return value, and number of bytes copied from BUF
   through SRC_USED (if not NULL).  If DST is NULL, don't actually store
   anything and just return the size needed to store all the text. */

Bytecount
copy_buffer_text_out (struct buffer *buf, Bytebpos pos,
		      Bytecount len, Ibyte *dst, Bytecount dstlen,
		      Internal_Format dstfmt, Lisp_Object dstobj,
		      Bytecount *src_used)
{
  Bytecount dst_used = 0;
  if (src_used)
    *src_used = 0;

  {
    BUFFER_TEXT_LOOP (buf, pos, len, runptr, runlen)
      {
	Bytecount the_src_used = -1, the_dst_used;
	
	the_dst_used = copy_text_between_formats (runptr, runlen,
						  BUF_FORMAT (buf),
						  wrap_buffer (buf),
						  dst, dstlen, dstfmt,
						  dstobj, &the_src_used);
	dst_used += the_dst_used;
	if (src_used)
          {
            text_checking_assert (the_src_used >= 0);
            *src_used += the_src_used;
          }
	if (dst)
	  {
	    dst += the_dst_used;
	    dstlen -= the_dst_used;
	    /* Stop if we didn't use all of the source text.  Also stop
	       if the destination is full.  We need the first test because
	       there might be a couple bytes left in the destination, but
	       not enough to fit a full character.  The first test will in
	       fact catch the vast majority of cases where the destination
	       is empty, too -- but in case the destination holds *exactly*
	       the run length, we put in the second check. (It shouldn't
	       really matter though -- next time through we'll just get a
	       0.) */
	    if (the_src_used < runlen || !dstlen)
	      break;
	  }
      }
  }

  return dst_used;
}


/************************************************************************/
/*                    charset properties of strings                     */
/************************************************************************/

/* Add OBJ to DYNARR unless it's already present in DYNARR */

static void
add_to_dynarr_unless_present (Lisp_Object obj, Lisp_Object_dynarr *dynarr)
{
  int i;

  for (i = 0; i < Dynarr_length (dynarr); i++)
    {
      if (EQ (Dynarr_at (dynarr, i), obj))
	return;
    }
  Dynarr_add (dynarr, obj);
}

/* Find the charsets in an array of Ichars, using BUF's Unicode-precedence
   list when under Unicode-internal to decide which charset to pick.  Add
   the charsets found to CHARSETS, but don't add any duplicates. */

void
find_charsets_in_ibyte_string (Lisp_Object_dynarr *charsets,
			       const Ibyte *str, Bytecount len,
			       struct buffer *buf)
{
  const Ibyte *endp = str + len;
  Lisp_Object prev_charset = Qunbound;

  /* #### SJT doesn't like this. */
  if (len == 0)
    {
      Dynarr_add (charsets, Vcharset_ascii);
      return;
    }

  while (str < endp)
    {
      Ibyte i0 = *str;

      if (byte_ascii_p (i0))
        {
	  add_to_dynarr_unless_present (Vcharset_ascii, charsets);
          str = skip_ascii (str, endp);
          continue;
        }
      else
        {
          Lisp_Object charset;
          int c1, c2;

          buffer_ichar_to_charset_codepoint (itext_ichar (str), buf, &charset,
                                             &c1, &c2, CONVERR_FAIL);
          if (!NILP (charset) && !EQ (charset, prev_charset))
            {
              prev_charset = charset;
              add_to_dynarr_unless_present (charset, charsets);
            }

          INC_IBYTEPTR (str);
        }
    }
}


/* Find the charsets in a region of a buffer.  Add the
   charsets found to CHARSETS, but don't add any duplicates. */

void
find_charsets_in_buffer (Lisp_Object_dynarr *charsets,
			 struct buffer *buf, Charbpos pos, Charcount len)
{
  Lisp_Object prev_charset = Qunbound;
  Charbpos stop = pos + len;

  while (pos < stop)
    {
      Ichar ich = BUF_FETCH_CHAR (buf, pos);
      Lisp_Object charset;
      int c1, c2;
      
      buffer_ichar_to_charset_codepoint (ich, buf, &charset,
					 &c1, &c2, CONVERR_FAIL);
      if (!NILP (charset) && !EQ (charset, prev_charset))
	{
	  prev_charset = charset;
	  add_to_dynarr_unless_present (charset, charsets);
	}

      ++pos;
    }
}

int
ibyte_string_displayed_columns (const Ibyte *str, Bytecount len)
{
  int cols = 0;
  const Ibyte *end = str + len;

  while (str < end)
    {
      cols += ichar_columns (itext_ichar (str));
      INC_IBYTEPTR (str);
    }

  return cols;
}


/***************************************************************************/
/*                     Eistring helper functions                           */
/***************************************************************************/

int
eistr_casefiddle_1 (Ibyte *olddata, Bytecount len, Ibyte *newdata,
		    int downp)
{
  Ibyte *endp = olddata + len;
  Ibyte *newp = newdata;
  int changedp = 0;

  while (olddata < endp)
    {
      Ichar c = itext_ichar (olddata);
      Ichar newc;

      if (downp)
	newc = DOWNCASE (0, c);
      else
	newc = UPCASE (0, c);

      if (c != newc)
	changedp = 1;

      newp += set_itext_ichar (newp, newc);
      INC_IBYTEPTR (olddata);
    }

  *newp = '\0';

  return changedp ? newp - newdata : 0;
}

Bytecount
eifind_large_enough_buffer (Bytecount oldbufsize, Bytecount needed_size)
{
  while (oldbufsize < needed_size)
    {
      oldbufsize = oldbufsize * 3 / 2;
      oldbufsize = max (oldbufsize, 32);
    }

  return oldbufsize;
}

void
eito_malloc_1 (Eistring *ei)
{
  if (ei->mallocp_)
    return;
  ei->mallocp_ = 1;
  if (ei->data_)
    {
      Ibyte *newdata;

      ei->max_size_allocated_ =
	eifind_large_enough_buffer (0, ei->bytelen_ + 1);
      newdata = xnew_ibytes (ei->max_size_allocated_);
      memcpy (newdata, ei->data_, ei->bytelen_ + 1);
      ei->data_ = newdata;
    }

  if (ei->extdata_)
    {
      Extbyte *newdata = xnew_extbytes (ei->extlen_ + 2);

      memcpy (newdata, ei->extdata_, ei->extlen_);
      /* Double null-terminate in case of Unicode data */
      newdata[ei->extlen_] = '\0';
      newdata[ei->extlen_ + 1] = '\0';
      ei->extdata_ = newdata;
    }
}  

int
eicmp_1 (Eistring *ei, Bytecount off, Charcount charoff,
	 Bytecount len, Charcount charlen, const Ibyte *data,
	 const Eistring *ei2, int is_ascii, int fold_case)
{
  assert ((data == 0) != (ei == 0)); 
  assert ((is_ascii != 0) == (data != 0));
  assert (fold_case >= 0 && fold_case <= 2);
  assert ((off < 0) != (charoff < 0));

  if (off < 0)
    {
      off = charcount_to_bytecount (ei->data_, charoff);
      if (charlen < 0)
	len = -1;
      else
	len = charcount_to_bytecount (ei->data_ + off, charlen);
    }
  if (len < 0)
    len = ei->bytelen_ - off;

  assert (off >= 0 && off <= ei->bytelen_);
  assert (len >= 0 && off + len <= ei->bytelen_);

  {
    Bytecount dstlen;
    const Ibyte *src = ei->data_, *dst;

    if (data)
      {
	dst = data;
	dstlen = qxestrlen (data);
      }
    else
      {
	dst = ei2->data_;
	dstlen = ei2->bytelen_;
      }

    if (is_ascii)
      ASSERT_ASCTEXT_ASCII_LEN ((Ascbyte *) dst, dstlen);

    return (fold_case == 0 ? qxememcmp4 (src, len, dst, dstlen) :
	    fold_case == 1 ? qxememcasecmp4 (src, len, dst, dstlen) :
	    qxetextcasecmp (src, len, dst, dstlen));
  }
}

Ibyte *
eicpyout_malloc_fmt (Eistring *eistr, Bytecount *len_out, Internal_Format fmt,
		     Lisp_Object UNUSED (object))
{
  Ibyte *ptr;

  assert (fmt == FORMAT_DEFAULT);
  ptr = xnew_array (Ibyte, eistr->bytelen_ + 1);
  if (len_out)
    *len_out = eistr->bytelen_;
  memcpy (ptr, eistr->data_, eistr->bytelen_ + 1);
  return ptr;
}


/************************************************************************/
/*                    Charcount/Bytecount conversion                    */
/************************************************************************/

#ifdef MULE

/* Function equivalents of bytecount_to_charcount/charcount_to_bytecount.
   These work on strings of all sizes but are more efficient than a simple
   loop on large strings and probably less efficient on sufficiently small
   strings. */

Charcount
bytecount_to_charcount_fun (const Ibyte *ptr, Bytecount len)
{
  Charcount count = 0;
  const Ibyte *end = ptr + len;
  while (1)
    {
      const Ibyte *newptr = skip_ascii (ptr, end);
      count += newptr - ptr;
      ptr = newptr;
      if (ptr == end)
	break;
      {
	/* Optimize for successive characters from the same charset */
	/* This still sort of works in the Unicode world -- 0 - 7F take one
	   byte, 80 - 7FF take two bytes, 800 - FFFF take three bytes.
	   Latin, Greek, Cyrillic, Hebrew and Arabic scripts end up
	   (mostly) in the two-byte ranges, and Indian and CJK scripts in
	   the three-byte ranges. */
	Ibyte lead_byte = *ptr;
	int bytes = rep_bytes_by_first_byte (lead_byte);
	while (ptr < end && *ptr == lead_byte)
	  ptr += bytes, count++;
      }
    }

  /* Bomb out if the specified substring ends in the middle
     of a character.  Note that we might have already gotten
     a core dump above from an invalid reference, but at least
     we will get no farther than here.

     This also catches len < 0. */
  text_checking_assert (ptr == end);

  return count;
}

/* Return the character count of an lstream or coding buffer of
   internal-format text, counting partial characters at the beginning of the
   buffer as whole characters, and *not* counting partial characters at the
   end of the buffer. The result of this function is subtracted from the
   character count given by the coding system character tell methods, and we
   need to treat each buffer in the same way to avoid double-counting. */

Charcount
buffered_bytecount_to_charcount (const Ibyte *bufptr, Bytecount len)
{
  Boolint partial_first = 0;
  Bytecount impartial;

  if (valid_ibyteptr_p (bufptr))
    {
      if (rep_bytes_by_first_byte (*bufptr) > len)
        {
          /* This is a partial last character. Return 0, avoid treating it
             as a partial first character, since that would lead to it being
             counted twice. */
          return (Charcount) 0;
        }
    }
  else
    {
      const Ibyte *newstart = bufptr, *limit = newstart + len;

      /* Our consumer has the start of a partial character, we have the
         rest. */
      while (newstart < limit && !valid_ibyteptr_p (newstart))
        {
          newstart++;
        }
                  
      partial_first = 1;
      bufptr = newstart;
      len = limit - newstart;
    }

  if (len && valid_ibyteptr_p (bufptr))
    {
      /* There's at least one valid starting char in the string,
         validate_ibyte_string_backward won't run off the begining. */
      impartial = validate_ibyte_string_backward (bufptr, len);
    }
  else
    {
      impartial = 0;
    }

  return (Charcount) partial_first + bytecount_to_charcount (bufptr,
                                                             impartial);
}

Bytecount
charcount_to_bytecount_fun (const Ibyte *ptr, Charcount len)
{
  const Ibyte *newptr = ptr;
  while (1)
    {
      const Ibyte *newnewptr = skip_ascii (newptr, newptr + len);
      len -= newnewptr - newptr;
      newptr = newnewptr;
      if (!len)
	break;
      {
	/* Optimize for successive characters from the same charset */
	/* This still sort of works in the Unicode world -- see
	   comment in bytecount_to_charcount_fun(). */
	Ibyte lead_byte = *newptr;
	int bytes = rep_bytes_by_first_byte (lead_byte);
	while (len > 0 && *newptr == lead_byte)
	  newptr += bytes, len--;
      }
    }
  return newptr - ptr;
}

/* Function equivalent of charcount_to_bytecount_down.  This works on strings
   of all sizes but is more efficient than a simple loop on large strings
   and probably less efficient on sufficiently small strings. */

Bytecount
charcount_to_bytecount_down_fun (const Ibyte *ptr, Charcount len)
{
  const Ibyte *newptr = ptr;
  while (1)
    {
      const Ibyte *newnewptr = skip_ascii_down (newptr, newptr - len);
      len -= newptr - newnewptr;
      newptr = newnewptr;
      /* Skip over all non-ASCII chars, counting the length and
	 stopping if it's zero */
      while (len && !byte_ascii_p (*(newptr - 1)))
	if (ibyte_first_byte_p (*--newptr))
	  len--;
      if (!len)
	break;
    }
  text_checking_assert (ptr - newptr >= 0);
  return ptr - newptr;
}

/* The next two functions are the actual meat behind the
   charbpos-to-bytebpos and bytebpos-to-charbpos conversions.  Currently
   the method they use is fairly unsophisticated; see buffer.h.

   Note that charbpos_to_bytebpos_func() is probably the most-called
   function in all of XEmacs.  Therefore, it must be FAST FAST FAST.
   This is the reason why so much of the code is duplicated.

   Similar considerations apply to bytebpos_to_charbpos_func(), although
   less so because the function is not called so often.
 */

/*

Info on Byte-Char conversion:

  (Info-goto-node "(internals)Byte-Char Position Conversion")
*/

#ifdef OLD_BYTE_CHAR
static int not_very_random_number;
#endif /* OLD_BYTE_CHAR */

#define OLD_LOOP

/* If we are this many characters away from any known position, cache the
   new position in the buffer's char-byte cache. */
#define FAR_AWAY_DISTANCE 5000

/* Converting between character positions and byte positions.  */

/* There are several places in the buffer where we know
   the correspondence: BEG, BEGV, PT, GPT, ZV and Z,
   and everywhere there is a marker.  So we find the one of these places
   that is closest to the specified position, and scan from there.  */

/* This macro is a subroutine of charbpos_to_bytebpos_func.
   Note that it is desirable that BYTEPOS is not evaluated
   except when we really want its value.  */

#define CONSIDER(CHARPOS, BYTEPOS)					\
do									\
{									\
  Charbpos this_charpos = (CHARPOS);					\
  int changed = 0;							\
									\
  if (this_charpos == x)						\
    {									\
      retval = (BYTEPOS);						\
      goto done;							\
    }									\
  else if (this_charpos > x)						\
    {									\
      if (this_charpos < best_above)					\
	{								\
	  best_above = this_charpos;					\
	  best_above_byte = (BYTEPOS);					\
	  changed = 1;							\
	}								\
    }									\
  else if (this_charpos > best_below)					\
    {									\
      best_below = this_charpos;					\
      best_below_byte = (BYTEPOS);					\
      changed = 1;							\
    }									\
									\
  if (changed)								\
    {									\
      if (best_above - best_below == best_above_byte - best_below_byte)	\
        {								\
	  retval = best_below_byte + (x - best_below);			\
          goto done;							\
	}								\
    }									\
}									\
while (0)


Bytebpos
charbpos_to_bytebpos_func (struct buffer *buf, Charbpos x)
{
#ifdef OLD_BYTE_CHAR
  Charbpos bufmin;
  Charbpos bufmax;
  Bytebpos bytmin;
  Bytebpos bytmax;
  int size;
  int forward_p;
  int diff_so_far;
  int add_to_cache = 0;
#endif /* OLD_BYTE_CHAR */

  Charbpos best_above, best_below;
  Bytebpos best_above_byte, best_below_byte;
  int i;
  struct buffer_text *t;
  Bytebpos retval;

  PROFILE_DECLARE ();

  PROFILE_RECORD_ENTERING_SECTION (QSin_char_byte_conversion);

  best_above = BUF_Z (buf);
  best_above_byte = BYTE_BUF_Z (buf);

  /* In this case, we simply have all one-byte characters.  But this should
     have been intercepted before, in charbpos_to_bytebpos(). */
  text_checking_assert (best_above != best_above_byte);

  best_below = BUF_BEG (buf);
  best_below_byte = BYTE_BUF_BEG (buf);

  /* We find in best_above and best_above_byte
     the closest known point above CHARPOS,
     and in best_below and best_below_byte
     the closest known point below CHARPOS,
     
     If at any point we can tell that the space between those
     two best approximations is all single-byte,
     we interpolate the result immediately.  */

  CONSIDER (BUF_PT (buf), BYTE_BUF_PT (buf));
  CONSIDER (BUF_GPT (buf), BYTE_BUF_GPT (buf));
  CONSIDER (BUF_BEGV (buf), BYTE_BUF_BEGV (buf));
  CONSIDER (BUF_ZV (buf), BYTE_BUF_ZV (buf));

  t = buf->text;
  CONSIDER (t->cached_charpos, t->cached_bytepos);

  /* Check the most recently entered positions first */

  for (i = t->next_cache_pos - 1; i >= 0; i--)
    {
      CONSIDER (t->mule_charbpos_cache[i], t->mule_bytebpos_cache[i]);

      /* If we are down to a range of 50 chars,
	 don't bother checking any other markers;
	 scan the intervening chars directly now.  */
      if (best_above - best_below < 50)
	break;
    }

  /* We get here if we did not exactly hit one of the known places.
     We have one known above and one known below.
     Scan, counting characters, from whichever one is closer.  */

  if (x - best_below < best_above - x)
    {
      int record = x - best_below > FAR_AWAY_DISTANCE;

#ifdef OLD_LOOP /* old code */
      while (best_below != x)
	{
	  best_below++;
	  INC_BYTEBPOS (buf, best_below_byte);
	}
#else
      text_checking_assert (BUF_FORMAT (buf) == FORMAT_DEFAULT);
      /* The gap should not occur between best_below and x, or we will be
	 screwed in using charcount_to_bytecount().  It should not be exactly
	 at x either, because we already should have caught that. */
      text_checking_assert
	(BUF_CEILING_OF_IGNORE_ACCESSIBLE (buf, best_below) > x);

      /* Using charcount_to_bytecount() is potentially a lot faster than a
	 simple loop using INC_BYTEBPOS() because (a) the checks for gap
	 and buffer format are factored out instead of getting checked
	 every time; (b) the checking goes 4 or 8 bytes at a time in ASCII
	 text.
      */
      best_below_byte +=
	charcount_to_bytecount
	(BYTE_BUF_BYTE_ADDRESS (buf, best_below_byte), x - best_below);
      best_below = x;
#endif /* 0 */

      /* If this position is quite far from the nearest known position,
	 cache the correspondence.

	 NB FSF does this: "... by creating a marker here.
	 It will last until the next GC."
      */

      if (record)
	{
	  /* If we have run out of positions to record, discard some of the
	     old ones.  I used to use a circular buffer, which avoids the
	     need to block-move any memory.  But it makes it more difficult
	     to keep track of which positions haven't been used -- commonly
	     we haven't yet filled out anywhere near the whole set of
	     positions and don't want to check them all.  We should not be
	     recording that often, and block-moving is extremely fast in
	     any case. --ben */
	  if (t->next_cache_pos == NUM_CACHED_POSITIONS)
	    {
	      memmove (t->mule_charbpos_cache,
		       t->mule_charbpos_cache + NUM_MOVED_POSITIONS,
		       sizeof (Charbpos) *
		       (NUM_CACHED_POSITIONS - NUM_MOVED_POSITIONS));
	      memmove (t->mule_bytebpos_cache,
		       t->mule_bytebpos_cache + NUM_MOVED_POSITIONS,
		       sizeof (Bytebpos) *
		       (NUM_CACHED_POSITIONS - NUM_MOVED_POSITIONS));
	      t->next_cache_pos -= NUM_MOVED_POSITIONS;
	    }
	  t->mule_charbpos_cache[t->next_cache_pos] = best_below;
	  t->mule_bytebpos_cache[t->next_cache_pos] = best_below_byte;
	  t->next_cache_pos++;
	}

      t->cached_charpos = best_below;
      t->cached_bytepos = best_below_byte;

      retval = best_below_byte;
      text_checking_assert (best_below_byte >= best_below);
      goto done;
    }
  else
    {
      int record = best_above - x > FAR_AWAY_DISTANCE;

#ifdef OLD_LOOP
      while (best_above != x)
	{
	  best_above--;
	  DEC_BYTEBPOS (buf, best_above_byte);
	}
#else
      text_checking_assert (BUF_FORMAT (buf) == FORMAT_DEFAULT);
      /* The gap should not occur between best_above and x, or we will be
	 screwed in using charcount_to_bytecount_down().  It should not be
	 exactly at x either, because we already should have caught
	 that. */
      text_checking_assert
	(BUF_FLOOR_OF_IGNORE_ACCESSIBLE (buf, best_above) < x);

      /* Using charcount_to_bytecount_down() is potentially a lot faster
	 than a simple loop using DEC_BYTEBPOS(); see above. */
      best_above_byte -=
	charcount_to_bytecount_down
	/* BYTE_BUF_BYTE_ADDRESS will return a value on the high side of the
	   gap if we are at the gap, which is the wrong side.  So do the
	   following trick instead. */
	(BYTE_BUF_BYTE_ADDRESS_BEFORE (buf, best_above_byte) + 1,
	 best_above - x);
      best_above = x;
#endif /* SLEDGEHAMMER_CHECK_TEXT */


      /* If this position is quite far from the nearest known position,
	 cache the correspondence.

	 NB FSF does this: "... by creating a marker here.
	 It will last until the next GC."
      */
      if (record)
	{
	  if (t->next_cache_pos == NUM_CACHED_POSITIONS)
	    {
	      memmove (t->mule_charbpos_cache,
		       t->mule_charbpos_cache + NUM_MOVED_POSITIONS,
		       sizeof (Charbpos) *
		       (NUM_CACHED_POSITIONS - NUM_MOVED_POSITIONS));
	      memmove (t->mule_bytebpos_cache,
		       t->mule_bytebpos_cache + NUM_MOVED_POSITIONS,
		       sizeof (Bytebpos) *
		       (NUM_CACHED_POSITIONS - NUM_MOVED_POSITIONS));
	      t->next_cache_pos -= NUM_MOVED_POSITIONS;
	    }
	  t->mule_charbpos_cache[t->next_cache_pos] = best_above;
	  t->mule_bytebpos_cache[t->next_cache_pos] = best_above_byte;
	  t->next_cache_pos++;
	}

      t->cached_charpos = best_above;
      t->cached_bytepos = best_above_byte;

      retval = best_above_byte;
      text_checking_assert (best_above_byte >= best_above);
      goto done;
    }

#ifdef OLD_BYTE_CHAR

  bufmin = buf->text->mule_bufmin;
  bufmax = buf->text->mule_bufmax;
  bytmin = buf->text->mule_bytmin;
  bytmax = buf->text->mule_bytmax;
  size = (1 << buf->text->mule_shifter) + !!buf->text->mule_three_p;

  /* The basic idea here is that we shift the "known region" up or down
     until it overlaps the specified position.  We do this by moving
     the upper bound of the known region up one character at a time,
     and moving the lower bound of the known region up as necessary
     when the size of the character just seen changes.

     We optimize this, however, by first shifting the known region to
     one of the cached points if it's close by. (We don't check BEG or
     Z, even though they're cached; most of the time these will be the
     same as BEGV and ZV, and when they're not, they're not likely
     to be used.) */

  if (x > bufmax)
    {
      Charbpos diffmax = x - bufmax;
      Charbpos diffpt = x - BUF_PT (buf);
      Charbpos diffzv = BUF_ZV (buf) - x;
      /* #### This value could stand some more exploration. */
      Charcount heuristic_hack = (bufmax - bufmin) >> 2;

      /* Check if the position is closer to PT or ZV than to the
	 end of the known region. */

      if (diffpt < 0)
	diffpt = -diffpt;
      if (diffzv < 0)
	diffzv = -diffzv;

      /* But also implement a heuristic that favors the known region
	 over PT or ZV.  The reason for this is that switching to
	 PT or ZV will wipe out the knowledge in the known region,
	 which might be annoying if the known region is large and
	 PT or ZV is not that much closer than the end of the known
	 region. */

      diffzv += heuristic_hack;
      diffpt += heuristic_hack;
      if (diffpt < diffmax && diffpt <= diffzv)
	{
	  bufmax = bufmin = BUF_PT (buf);
	  bytmax = bytmin = BYTE_BUF_PT (buf);
	  /* We set the size to 1 even though it doesn't really
	     matter because the new known region contains no
	     characters.  We do this because this is the most
	     likely size of the characters around the new known
	     region, and we avoid potential yuckiness that is
	     done when size == 3. */
	  size = 1;
	}
      if (diffzv < diffmax)
	{
	  bufmax = bufmin = BUF_ZV (buf);
	  bytmax = bytmin = BYTE_BUF_ZV (buf);
	  size = 1;
	}
    }
#ifdef ERROR_CHECK_TEXT
  else if (x >= bufmin)
    ABORT ();
#endif
  else
    {
      Charbpos diffmin = bufmin - x;
      Charbpos diffpt = BUF_PT (buf) - x;
      Charbpos diffbegv = x - BUF_BEGV (buf);
      /* #### This value could stand some more exploration. */
      Charcount heuristic_hack = (bufmax - bufmin) >> 2;

      if (diffpt < 0)
	diffpt = -diffpt;
      if (diffbegv < 0)
	diffbegv = -diffbegv;

      /* But also implement a heuristic that favors the known region --
	 see above. */

      diffbegv += heuristic_hack;
      diffpt += heuristic_hack;

      if (diffpt < diffmin && diffpt <= diffbegv)
	{
	  bufmax = bufmin = BUF_PT (buf);
	  bytmax = bytmin = BYTE_BUF_PT (buf);
	  /* We set the size to 1 even though it doesn't really
	     matter because the new known region contains no
	     characters.  We do this because this is the most
	     likely size of the characters around the new known
	     region, and we avoid potential yuckiness that is
	     done when size == 3. */
	  size = 1;
	}
      if (diffbegv < diffmin)
	{
	  bufmax = bufmin = BUF_BEGV (buf);
	  bytmax = bytmin = BYTE_BUF_BEGV (buf);
	  size = 1;
	}
    }

  diff_so_far = x > bufmax ? x - bufmax : bufmin - x;
  if (diff_so_far > 50)
    {
      /* If we have to move more than a certain amount, then look
	 into our cache. */
      int minval = INT_MAX;
      int found = 0;
      int i;

      add_to_cache = 1;
      /* I considered keeping the positions ordered.  This would speed
	 up this loop, but updating the cache would take longer, so
	 it doesn't seem like it would really matter. */
      for (i = 0; i < NUM_CACHED_POSITIONS; i++)
	{
	  int diff = buf->text->mule_charbpos_cache[i] - x;

	  if (diff < 0)
	    diff = -diff;
	  if (diff < minval)
	    {
	      minval = diff;
	      found = i;
	    }
	}

      if (minval < diff_so_far)
	{
	  bufmax = bufmin = buf->text->mule_charbpos_cache[found];
	  bytmax = bytmin = buf->text->mule_bytebpos_cache[found];
	  size = 1;
	}
    }

  /* It's conceivable that the caching above could lead to X being
     the same as one of the range edges. */
  if (x >= bufmax)
    {
      Bytebpos newmax;
      Bytecount newsize;

      forward_p = 1;
      while (x > bufmax)
	{
	  newmax = bytmax;

	  INC_BYTEBPOS (buf, newmax);
	  newsize = newmax - bytmax;
	  if (newsize != size)
	    {
	      bufmin = bufmax;
	      bytmin = bytmax;
	      size = newsize;
	    }
	  bytmax = newmax;
	  bufmax++;
	}
      retval = bytmax;

      /* #### Should go past the found location to reduce the number
	 of times that this function is called */
    }
  else /* x < bufmin */
    {
      Bytebpos newmin;
      Bytecount newsize;

      forward_p = 0;
      while (x < bufmin)
	{
	  newmin = bytmin;

	  DEC_BYTEBPOS (buf, newmin);
	  newsize = bytmin - newmin;
	  if (newsize != size)
	    {
	      bufmax = bufmin;
	      bytmax = bytmin;
	      size = newsize;
	    }
	  bytmin = newmin;
	  bufmin--;
	}
      retval = bytmin;

      /* #### Should go past the found location to reduce the number
	 of times that this function is called
         */
    }

  /* If size is three, than we have to max sure that the range we
     discovered isn't too large, because we use a fixed-length
     table to divide by 3. */

  if (size == 3)
    {
      int gap = bytmax - bytmin;
      buf->text->mule_three_p = 1;
      buf->text->mule_shifter = 1;

      if (gap > MAX_BYTEBPOS_GAP_SIZE_3)
	{
	  if (forward_p)
	    {
	      bytmin = bytmax - MAX_BYTEBPOS_GAP_SIZE_3;
	      bufmin = bufmax - MAX_CHARBPOS_GAP_SIZE_3;
	    }
	  else
	    {
	      bytmax = bytmin + MAX_BYTEBPOS_GAP_SIZE_3;
	      bufmax = bufmin + MAX_CHARBPOS_GAP_SIZE_3;
	    }
	}
    }
  else
    {
      buf->text->mule_three_p = 0;
      if (size == 4)
	buf->text->mule_shifter = 2;
      else
	buf->text->mule_shifter = size - 1;
    }

  buf->text->mule_bufmin = bufmin;
  buf->text->mule_bufmax = bufmax;
  buf->text->mule_bytmin = bytmin;
  buf->text->mule_bytmax = bytmax;
  
  if (add_to_cache)
    {
      int replace_loc;

      /* We throw away a "random" cached value and replace it with
	 the new value.  It doesn't actually have to be very random
	 at all, just evenly distributed.

	 #### It would be better to use a least-recently-used algorithm
	 or something that tries to space things out, but I'm not sure
	 it's worth it to go to the trouble of maintaining that. */
      not_very_random_number += 621;
      replace_loc = not_very_random_number & 15;
      buf->text->mule_charbpos_cache[replace_loc] = x;
      buf->text->mule_bytebpos_cache[replace_loc] = retval;
    }

#endif /* OLD_BYTE_CHAR */

done:
  PROFILE_RECORD_EXITING_SECTION (QSin_char_byte_conversion);

  return retval;
}

#undef CONSIDER

/* bytepos_to_charpos returns the char position corresponding to BYTEPOS.  */

/* This macro is a subroutine of bytebpos_to_charbpos_func.
   It is used when BYTEPOS is actually the byte position.  */

#define CONSIDER(BYTEPOS, CHARPOS)					\
do									\
{									\
  Bytebpos this_bytepos = (BYTEPOS);					\
  int changed = 0;							\
									\
  if (this_bytepos == x)						\
    {									\
      retval = (CHARPOS);						\
      goto done;							\
    }									\
  else if (this_bytepos > x)						\
    {									\
      if (this_bytepos < best_above_byte)				\
	{								\
	  best_above = (CHARPOS);					\
	  best_above_byte = this_bytepos;				\
	  changed = 1;							\
	}								\
    }									\
  else if (this_bytepos > best_below_byte)				\
    {									\
      best_below = (CHARPOS);						\
      best_below_byte = this_bytepos;					\
      changed = 1;							\
    }									\
									\
  if (changed)								\
    {									\
      if (best_above - best_below == best_above_byte - best_below_byte)	\
	{								\
	  retval = best_below + (x - best_below_byte);			\
	  goto done;							\
	}								\
    }									\
}									\
while (0)

/* The logic in this function is almost identical to the logic in
   the previous function. */

Charbpos
bytebpos_to_charbpos_func (struct buffer *buf, Bytebpos x)
{
#ifdef OLD_BYTE_CHAR
  Charbpos bufmin;
  Charbpos bufmax;
  Bytebpos bytmin;
  Bytebpos bytmax;
  int size;
  int forward_p;
  int diff_so_far;
  int add_to_cache = 0;
#endif /* OLD_BYTE_CHAR */

  Charbpos best_above, best_above_byte;
  Bytebpos best_below, best_below_byte;
  int i;
  struct buffer_text *t;
  Charbpos retval;

  PROFILE_DECLARE ();

  PROFILE_RECORD_ENTERING_SECTION (QSin_byte_char_conversion);

  best_above = BUF_Z (buf);
  best_above_byte = BYTE_BUF_Z (buf);

  /* In this case, we simply have all one-byte characters.  But this should
     have been intercepted before, in bytebpos_to_charbpos(). */
  text_checking_assert (best_above != best_above_byte);

  best_below = BUF_BEG (buf);
  best_below_byte = BYTE_BUF_BEG (buf);

  CONSIDER (BYTE_BUF_PT (buf), BUF_PT (buf));
  CONSIDER (BYTE_BUF_GPT (buf), BUF_GPT (buf));
  CONSIDER (BYTE_BUF_BEGV (buf), BUF_BEGV (buf));
  CONSIDER (BYTE_BUF_ZV (buf), BUF_ZV (buf));

  t = buf->text;
  CONSIDER (t->cached_bytepos, t->cached_charpos);

  /* Check the most recently entered positions first */

  for (i = t->next_cache_pos - 1; i >= 0; i--)
    {
      CONSIDER (t->mule_bytebpos_cache[i], t->mule_charbpos_cache[i]);

      /* If we are down to a range of 50 chars,
	 don't bother checking any other markers;
	 scan the intervening chars directly now.  */
      if (best_above - best_below < 50)
	break;
    }

  /* We get here if we did not exactly hit one of the known places.
     We have one known above and one known below.
     Scan, counting characters, from whichever one is closer.  */

  if (x - best_below_byte < best_above_byte - x)
    {
      int record = x - best_below_byte > 5000;

#ifdef OLD_LOOP /* old code */
      while (best_below_byte < x)
	{
	  best_below++;
	  INC_BYTEBPOS (buf, best_below_byte);
	}
#else
      text_checking_assert (BUF_FORMAT (buf) == FORMAT_DEFAULT);
      /* The gap should not occur between best_below and x, or we will be
	 screwed in using charcount_to_bytecount().  It should not be exactly
	 at x either, because we already should have caught that. */
      text_checking_assert
	(BYTE_BUF_CEILING_OF_IGNORE_ACCESSIBLE (buf, best_below_byte) > x);

      /* Using bytecount_to_charcount() is potentially a lot faster than
	 a simple loop above using INC_BYTEBPOS(); see above.
      */
      best_below +=
	bytecount_to_charcount
	(BYTE_BUF_BYTE_ADDRESS (buf, best_below_byte), x - best_below_byte);
      best_below_byte = x;
#endif

      /* If this position is quite far from the nearest known position,
	 cache the correspondence.

	 NB FSF does this: "... by creating a marker here.
	 It will last until the next GC."
      */

      if (record)
	{
	  if (t->next_cache_pos == NUM_CACHED_POSITIONS)
	    {
	      memmove (t->mule_charbpos_cache,
		       t->mule_charbpos_cache + NUM_MOVED_POSITIONS,
		       sizeof (Charbpos) *
		       (NUM_CACHED_POSITIONS - NUM_MOVED_POSITIONS));
	      memmove (t->mule_bytebpos_cache,
		       t->mule_bytebpos_cache + NUM_MOVED_POSITIONS,
		       sizeof (Bytebpos) *
		       (NUM_CACHED_POSITIONS - NUM_MOVED_POSITIONS));
	      t->next_cache_pos -= NUM_MOVED_POSITIONS;
	    }
	  t->mule_charbpos_cache[t->next_cache_pos] = best_below;
	  t->mule_bytebpos_cache[t->next_cache_pos] = best_below_byte;
	  t->next_cache_pos++;
	}


      t->cached_charpos = best_below;
      t->cached_bytepos = best_below_byte;

      retval = best_below;
      text_checking_assert (best_below_byte >= best_below);
      goto done;
    }
  else
    {
      int record = best_above_byte - x > 5000;

#ifdef OLD_LOOP /* old code */
      while (best_above_byte > x)
	{
	  best_above--;
	  DEC_BYTEBPOS (buf, best_above_byte);
	}
#else
      text_checking_assert (BUF_FORMAT (buf) == FORMAT_DEFAULT);
      /* The gap should not occur between best_above and x, or we will be
	 screwed in using bytecount_to_charcount_down().  It should not be
	 exactly at x either, because we already should have caught
	 that. */
      text_checking_assert
	(BYTE_BUF_FLOOR_OF_IGNORE_ACCESSIBLE (buf, best_above_byte) < x);

      /* Using bytecount_to_charcount_down() is potentially a lot faster
	 than a simple loop using INC_BYTEBPOS(); see above. */
      best_above -=
	bytecount_to_charcount_down
	/* BYTE_BUF_BYTE_ADDRESS will return a value on the high side of the
	   gap if we are at the gap, which is the wrong side.  So do the
	   following trick instead. */
	(BYTE_BUF_BYTE_ADDRESS_BEFORE (buf, best_above_byte) + 1,
	best_above_byte - x);
      best_above_byte = x;
#endif


      /* If this position is quite far from the nearest known position,
	 cache the correspondence.

	 NB FSF does this: "... by creating a marker here.
	 It will last until the next GC."
      */
      if (record)
	{
	  if (t->next_cache_pos == NUM_CACHED_POSITIONS)
	    {
	      memmove (t->mule_charbpos_cache,
		       t->mule_charbpos_cache + NUM_MOVED_POSITIONS,
		       sizeof (Charbpos) *
		       (NUM_CACHED_POSITIONS - NUM_MOVED_POSITIONS));
	      memmove (t->mule_bytebpos_cache,
		       t->mule_bytebpos_cache + NUM_MOVED_POSITIONS,
		       sizeof (Bytebpos) *
		       (NUM_CACHED_POSITIONS - NUM_MOVED_POSITIONS));
	      t->next_cache_pos -= NUM_MOVED_POSITIONS;
	    }
	  t->mule_charbpos_cache[t->next_cache_pos] = best_above;
	  t->mule_bytebpos_cache[t->next_cache_pos] = best_above_byte;
	  t->next_cache_pos++;
	}

      t->cached_charpos = best_above;
      t->cached_bytepos = best_above_byte;

      retval = best_above;
      text_checking_assert (best_above_byte >= best_above);
      goto done;
    }

#ifdef OLD_BYTE_CHAR

  bufmin = buf->text->mule_bufmin;
  bufmax = buf->text->mule_bufmax;
  bytmin = buf->text->mule_bytmin;
  bytmax = buf->text->mule_bytmax;
  size = (1 << buf->text->mule_shifter) + !!buf->text->mule_three_p;

  /* The basic idea here is that we shift the "known region" up or down
     until it overlaps the specified position.  We do this by moving
     the upper bound of the known region up one character at a time,
     and moving the lower bound of the known region up as necessary
     when the size of the character just seen changes.

     We optimize this, however, by first shifting the known region to
     one of the cached points if it's close by. (We don't check BYTE_BEG or
     BYTE_Z, even though they're cached; most of the time these will be the
     same as BYTE_BEGV and BYTE_ZV, and when they're not, they're not likely
     to be used.) */

  if (x > bytmax)
    {
      Bytebpos diffmax = x - bytmax;
      Bytebpos diffpt = x - BYTE_BUF_PT (buf);
      Bytebpos diffzv = BYTE_BUF_ZV (buf) - x;
      /* #### This value could stand some more exploration. */
      Bytecount heuristic_hack = (bytmax - bytmin) >> 2;

      /* Check if the position is closer to PT or ZV than to the
	 end of the known region. */

      if (diffpt < 0)
	diffpt = -diffpt;
      if (diffzv < 0)
	diffzv = -diffzv;

      /* But also implement a heuristic that favors the known region
	 over BYTE_PT or BYTE_ZV.  The reason for this is that switching to
	 BYTE_PT or BYTE_ZV will wipe out the knowledge in the known region,
	 which might be annoying if the known region is large and
	 BYTE_PT or BYTE_ZV is not that much closer than the end of the known
	 region. */

      diffzv += heuristic_hack;
      diffpt += heuristic_hack;
      if (diffpt < diffmax && diffpt <= diffzv)
	{
	  bufmax = bufmin = BUF_PT (buf);
	  bytmax = bytmin = BYTE_BUF_PT (buf);
	  /* We set the size to 1 even though it doesn't really
	     matter because the new known region contains no
	     characters.  We do this because this is the most
	     likely size of the characters around the new known
	     region, and we avoid potential yuckiness that is
	     done when size == 3. */
	  size = 1;
	}
      if (diffzv < diffmax)
	{
	  bufmax = bufmin = BUF_ZV (buf);
	  bytmax = bytmin = BYTE_BUF_ZV (buf);
	  size = 1;
	}
    }
#ifdef ERROR_CHECK_TEXT
  else if (x >= bytmin)
    ABORT ();
#endif
  else
    {
      Bytebpos diffmin = bytmin - x;
      Bytebpos diffpt = BYTE_BUF_PT (buf) - x;
      Bytebpos diffbegv = x - BYTE_BUF_BEGV (buf);
      /* #### This value could stand some more exploration. */
      Bytecount heuristic_hack = (bytmax - bytmin) >> 2;

      if (diffpt < 0)
	diffpt = -diffpt;
      if (diffbegv < 0)
	diffbegv = -diffbegv;

      /* But also implement a heuristic that favors the known region --
	 see above. */

      diffbegv += heuristic_hack;
      diffpt += heuristic_hack;

      if (diffpt < diffmin && diffpt <= diffbegv)
	{
	  bufmax = bufmin = BUF_PT (buf);
	  bytmax = bytmin = BYTE_BUF_PT (buf);
	  /* We set the size to 1 even though it doesn't really
	     matter because the new known region contains no
	     characters.  We do this because this is the most
	     likely size of the characters around the new known
	     region, and we avoid potential yuckiness that is
	     done when size == 3. */
	  size = 1;
	}
      if (diffbegv < diffmin)
	{
	  bufmax = bufmin = BUF_BEGV (buf);
	  bytmax = bytmin = BYTE_BUF_BEGV (buf);
	  size = 1;
	}
    }

  diff_so_far = x > bytmax ? x - bytmax : bytmin - x;
  if (diff_so_far > 50)
    {
      /* If we have to move more than a certain amount, then look
	 into our cache. */
      int minval = INT_MAX;
      int found = 0;
      int i;

      add_to_cache = 1;
      /* I considered keeping the positions ordered.  This would speed
	 up this loop, but updating the cache would take longer, so
	 it doesn't seem like it would really matter. */
      for (i = 0; i < NUM_CACHED_POSITIONS; i++)
	{
	  int diff = buf->text->mule_bytebpos_cache[i] - x;

	  if (diff < 0)
	    diff = -diff;
	  if (diff < minval)
	    {
	      minval = diff;
	      found = i;
	    }
	}

      if (minval < diff_so_far)
	{
	  bufmax = bufmin = buf->text->mule_charbpos_cache[found];
	  bytmax = bytmin = buf->text->mule_bytebpos_cache[found];
	  size = 1;
	}
    }

  /* It's conceivable that the caching above could lead to X being
     the same as one of the range edges. */
  if (x >= bytmax)
    {
      Bytebpos newmax;
      Bytecount newsize;

      forward_p = 1;
      while (x > bytmax)
	{
	  newmax = bytmax;

	  INC_BYTEBPOS (buf, newmax);
	  newsize = newmax - bytmax;
	  if (newsize != size)
	    {
	      bufmin = bufmax;
	      bytmin = bytmax;
	      size = newsize;
	    }
	  bytmax = newmax;
	  bufmax++;
	}
      retval = bufmax;

      /* #### Should go past the found location to reduce the number
	 of times that this function is called */
    }
  else /* x <= bytmin */
    {
      Bytebpos newmin;
      Bytecount newsize;

      forward_p = 0;
      while (x < bytmin)
	{
	  newmin = bytmin;

	  DEC_BYTEBPOS (buf, newmin);
	  newsize = bytmin - newmin;
	  if (newsize != size)
	    {
	      bufmax = bufmin;
	      bytmax = bytmin;
	      size = newsize;
	    }
	  bytmin = newmin;
	  bufmin--;
	}
      retval = bufmin;

      /* #### Should go past the found location to reduce the number
	 of times that this function is called
         */
    }

  /* If size is three, than we have to max sure that the range we
     discovered isn't too large, because we use a fixed-length
     table to divide by 3. */

  if (size == 3)
    {
      int gap = bytmax - bytmin;
      buf->text->mule_three_p = 1;
      buf->text->mule_shifter = 1;

      if (gap > MAX_BYTEBPOS_GAP_SIZE_3)
	{
	  if (forward_p)
	    {
	      bytmin = bytmax - MAX_BYTEBPOS_GAP_SIZE_3;
	      bufmin = bufmax - MAX_CHARBPOS_GAP_SIZE_3;
	    }
	  else
	    {
	      bytmax = bytmin + MAX_BYTEBPOS_GAP_SIZE_3;
	      bufmax = bufmin + MAX_CHARBPOS_GAP_SIZE_3;
	    }
	}
    }
  else
    {
      buf->text->mule_three_p = 0;
      if (size == 4)
	buf->text->mule_shifter = 2;
      else
	buf->text->mule_shifter = size - 1;
    }

  buf->text->mule_bufmin = bufmin;
  buf->text->mule_bufmax = bufmax;
  buf->text->mule_bytmin = bytmin;
  buf->text->mule_bytmax = bytmax;

  if (add_to_cache)
    {
      int replace_loc;

      /* We throw away a "random" cached value and replace it with
	 the new value.  It doesn't actually have to be very random
	 at all, just evenly distributed.

	 #### It would be better to use a least-recently-used algorithm
	 or something that tries to space things out, but I'm not sure
	 it's worth it to go to the trouble of maintaining that. */
      not_very_random_number += 621;
      replace_loc = not_very_random_number & 15;
      buf->text->mule_charbpos_cache[replace_loc] = retval;
      buf->text->mule_bytebpos_cache[replace_loc] = x;
    }
#endif /* OLD_BYTE_CHAR */

done:
  PROFILE_RECORD_EXITING_SECTION (QSin_byte_char_conversion);

  return retval;
}

/* Text of length BYTELENGTH and CHARLENGTH (in different units)
   was inserted at charbpos START. */

void
buffer_mule_signal_inserted_region (struct buffer *buf, Charbpos start,
				    Bytecount bytelength,
				    Charcount charlength)
{
#ifdef OLD_BYTE_CHAR
  int size = (1 << buf->text->mule_shifter) + !!buf->text->mule_three_p;
#endif /* OLD_BYTE_CHAR */
  int i;

  /* Adjust the cache of known positions. */
  for (i = 0; i < buf->text->next_cache_pos; i++)
    {

      if (buf->text->mule_charbpos_cache[i] > start)
	{
	  buf->text->mule_charbpos_cache[i] += charlength;
	  buf->text->mule_bytebpos_cache[i] += bytelength;
	}
    }

  /* Adjust the special cached position. */

  if (buf->text->cached_charpos > start)
    {
      buf->text->cached_charpos += charlength;
      buf->text->cached_bytepos += bytelength;
    }

#ifdef OLD_BYTE_CHAR
  if (start >= buf->text->mule_bufmax)
    return;

  /* The insertion is either before the known region, in which case
     it shoves it forward; or within the known region, in which case
     it shoves the end forward. (But it may make the known region
     inconsistent, so we may have to shorten it.) */

  if (start <= buf->text->mule_bufmin)
    {
      buf->text->mule_bufmin += charlength;
      buf->text->mule_bufmax += charlength;
      buf->text->mule_bytmin += bytelength;
      buf->text->mule_bytmax += bytelength;
    }
  else
    {
      Charbpos end = start + charlength;
      /* the insertion point divides the known region in two.
	 Keep the longer half, at least, and expand into the
	 inserted chunk as much as possible. */

      if (start - buf->text->mule_bufmin > buf->text->mule_bufmax - start)
	{
	  Bytebpos bytestart = (buf->text->mule_bytmin
			      + size * (start - buf->text->mule_bufmin));
	  Bytebpos bytenew;

	  while (start < end)
	    {
	      bytenew = bytestart;
	      INC_BYTEBPOS (buf, bytenew);
	      if (bytenew - bytestart != size)
		break;
	      start++;
              bytestart = bytenew;
	    }
	  if (start != end)
	    {
	      buf->text->mule_bufmax = start;
	      buf->text->mule_bytmax = bytestart;
	    }
	  else
	    {
	      buf->text->mule_bufmax += charlength;
	      buf->text->mule_bytmax += bytelength;
	    }
	}
      else
	{
	  Bytebpos byteend = (buf->text->mule_bytmin
			    + size * (start - buf->text->mule_bufmin)
			    + bytelength);
	  Bytebpos bytenew;

	  buf->text->mule_bufmax += charlength;
	  buf->text->mule_bytmax += bytelength;

	  while (end > start)
	    {
	      bytenew = byteend;
	      DEC_BYTEBPOS (buf, bytenew);
	      if (byteend - bytenew != size)
		break;
	      end--;
              byteend = bytenew;
	    }
	  if (start != end)
	    {
	      buf->text->mule_bufmin = end;
	      buf->text->mule_bytmin = byteend;
	    }
	}
    }
#endif /* OLD_BYTE_CHAR */
}

/* Text from START to END (equivalent in Bytebpos's: from BYTE_START to
   BYTE_END) was deleted. */

void
buffer_mule_signal_deleted_region (struct buffer *buf, Charbpos start,
				   Charbpos end, Bytebpos byte_start,
				   Bytebpos byte_end)
{
  int i;

  /* Adjust the cache of known positions. */
  for (i = 0; i < buf->text->next_cache_pos; i++)
    {
      /* After the end; gets shoved backward */
      if (buf->text->mule_charbpos_cache[i] > end)
	{
	  buf->text->mule_charbpos_cache[i] -= end - start;
	  buf->text->mule_bytebpos_cache[i] -= byte_end - byte_start;
	}
      /* In the range; moves to start of range */
      else if (buf->text->mule_charbpos_cache[i] > start)
	{
	  buf->text->mule_charbpos_cache[i] = start;
	  buf->text->mule_bytebpos_cache[i] = byte_start;
	}
    }

  /* Adjust the special cached position. */

  /* After the end; gets shoved backward */
  if (buf->text->cached_charpos > end)
    {
      buf->text->cached_charpos -= end - start;
      buf->text->cached_bytepos -= byte_end - byte_start;
    }
  /* In the range; moves to start of range */
  else if (buf->text->cached_charpos > start)
    {
      buf->text->cached_charpos = start;
      buf->text->cached_bytepos = byte_start;
    }

#ifdef OLD_BYTE_CHAR
  /* We don't care about any text after the end of the known region. */

  end = min (end, buf->text->mule_bufmax);
  byte_end = min (byte_end, buf->text->mule_bytmax);
  if (start >= end)
    return;

  /* The end of the known region offsets by the total amount of deletion,
     since it's all before it. */

  buf->text->mule_bufmax -= end - start;
  buf->text->mule_bytmax -= byte_end - byte_start;

  /* Now we don't care about any text after the start of the known region. */

  end = min (end, buf->text->mule_bufmin);
  byte_end = min (byte_end, buf->text->mule_bytmin);
  if (start < end)
    {
      buf->text->mule_bufmin -= end - start;
      buf->text->mule_bytmin -= byte_end - byte_start;
    }
#endif /* OLD_BYTE_CHAR */
}

#endif /* MULE */


/************************************************************************/
/*                verifying buffer and string positions                 */
/************************************************************************/

/* Functions below are tagged with either _byte or _char indicating
   whether they return byte or character positions.  For a buffer,
   a character position is a "Charbpos" and a byte position is a "Bytebpos".
   For strings, these are sometimes typed using "Charcount" and
   "Bytecount". */

/* Flags for the functions below are:

   GB_ALLOW_PAST_ACCESSIBLE

     Allow positions to range over the entire buffer (BUF_BEG to BUF_Z),
     rather than just the accessible portion (BUF_BEGV to BUF_ZV).
     For strings, this flag has no effect.

   GB_COERCE_RANGE

     If the position is outside the allowable range, return the lower
     or upper bound of the range, whichever is closer to the specified
     position.

   GB_NO_ERROR_IF_BAD

     If the position is outside the allowable range, return -1.

   GB_NEGATIVE_FROM_END

     If a value is negative, treat it as an offset from the end.
     Only applies to strings.

   GB_ALLOW_NIL

     If this flag is supplied to get_buffer_pos_{byte,char}, a nil position
     returns the value of point.

     When supplied to a function that returns ranges, either or both positions
     can be nil. If FROM is nil, FROM_OUT will contain the lower bound of the
     allowed range.  If TO is nil, TO_OUT will contain the upper bound of the
     allowed range.

     If this flag is supplied to get_string_pos_{byte,char} a nil position
     will provoke an error.

   The following additional flags apply only to the functions
   that return ranges:

   GB_CHECK_ORDER

     FROM must contain the lower bound and TO the upper bound
     of the range.  If the positions are reversed, an error is
     signalled.

   GB_NEED_CHAR_BEFORE, GB_NEED_CHAR_AFTER

     The character before or after POS must be in the accessible portion
     of the buffer (BUF_BEGV to BUF_ZV).

   The following is a combination flag:

   GB_HISTORICAL_STRING_BEHAVIOR

     Equivalent to (GB_NEGATIVE_FROM_END | GB_ALLOW_NIL).
 */

/* Return a buffer position stored in a Lisp_Object.  Full
   error-checking is done on the position.  Flags can be specified to
   control the behavior of out-of-range values.  The default behavior
   is to require that the position is within the accessible part of
   the buffer (BEGV and ZV), and to signal an error if the position is
   out of range.

*/

Charbpos
get_buffer_pos_char (struct buffer *b, Lisp_Object pos, unsigned int flags)
{
  /* Does not GC */
  Charbpos ind;
  Charbpos min_allowed, max_allowed;

  if (NILP (pos) && (flags & GB_ALLOW_NIL))
    {
      ind = BUF_PT (b);
    }
  else
    {
      CHECK_FIXNUM_COERCE_MARKER (pos);
      ind = XFIXNUM (pos);
    }

  min_allowed = flags & GB_ALLOW_PAST_ACCESSIBLE ? BUF_BEG (b) : BUF_BEGV (b);
  if (flags & GB_NEED_CHAR_BEFORE) min_allowed += 1;
  max_allowed = flags & GB_ALLOW_PAST_ACCESSIBLE ? BUF_Z   (b) : BUF_ZV   (b);
  if (flags & GB_NEED_CHAR_AFTER) max_allowed -= 1;

  if (ind < min_allowed || ind > max_allowed)
    {
      if (flags & GB_COERCE_RANGE)
	ind = ind < min_allowed ? min_allowed : max_allowed;
      else if (flags & GB_NO_ERROR_IF_BAD)
	ind = -1;
      else
	{
	  Lisp_Object buffer = wrap_buffer (b);

	  args_out_of_range (buffer, pos);
	}
    }

  return ind;
}

Bytebpos
get_buffer_pos_byte (struct buffer *b, Lisp_Object pos, unsigned int flags)
{
  if (NILP (pos) && (flags & GB_ALLOW_NIL))
    {
      pos = b->point_marker;
    }

  if (MARKERP (pos) && XMARKER (pos)->buffer == b)
    {
      /* Does not GC */
      Bytebpos ind;
      Bytebpos min_allowed, max_allowed;

      ind = marker_byte_position (pos);
      min_allowed
        = flags & GB_ALLOW_PAST_ACCESSIBLE ? BYTE_BUF_BEG (b)
        : BYTE_BUF_BEGV (b);
      max_allowed
        = flags & GB_ALLOW_PAST_ACCESSIBLE ? BYTE_BUF_Z (b) : BYTE_BUF_ZV (b);

      if (ind < min_allowed || ind > max_allowed)
        {
          if (flags & GB_COERCE_RANGE)
            ind = ind < min_allowed ? min_allowed : max_allowed;
          else if (flags & GB_NO_ERROR_IF_BAD)
            ind = -1;
          else
            {
              args_out_of_range (wrap_buffer (b), pos);
            }
        }

      return ind;
    }
  else
    {
      Charbpos bpos = get_buffer_pos_char (b, pos, flags);
      if (bpos < 0) /* could happen with GB_NO_ERROR_IF_BAD */
        return -1;
      return charbpos_to_bytebpos (b, bpos);
    }
}

void
get_buffer_pos_both (struct buffer *b, Lisp_Object pos, unsigned int flags,
                     Charbpos *cpos_out, Bytebpos *bpos_out)
{
  /* Does not GC */
  Bytebpos byte_ind = -1, byte_min_allowed, byte_max_allowed;
  Charbpos char_ind = -1, char_min_allowed, char_max_allowed;

  if (flags & GB_ALLOW_PAST_ACCESSIBLE)
    {
      byte_min_allowed = BYTE_BUF_BEG (b);
      byte_max_allowed = BYTE_BUF_Z (b);
      char_min_allowed = BUF_BEG (b);
      char_max_allowed = BUF_Z (b);
    }
  else
    {
      byte_min_allowed = BYTE_BUF_BEGV (b);
      byte_max_allowed = BYTE_BUF_ZV (b);
      char_min_allowed = BUF_BEGV (b);
      char_max_allowed = BUF_ZV (b);
    }

  if (NILP (pos) && (flags & GB_ALLOW_NIL))
    {
      pos = b->point_marker;
      char_ind = BUF_PT (b);
    }

  if (MARKERP (pos) && XMARKER (pos)->buffer == b)
    {
      byte_ind = marker_byte_position (pos);

      if (byte_ind < byte_min_allowed || byte_ind > byte_max_allowed)
        {
          if (flags & GB_COERCE_RANGE)
            {
              if (byte_ind < byte_min_allowed)
                {
                  *cpos_out = char_min_allowed;
                  *bpos_out = byte_min_allowed;
                  return;
                }

              *cpos_out = char_max_allowed;
              *bpos_out = byte_max_allowed;
              return;
            }
          else if (flags & GB_NO_ERROR_IF_BAD)
            {
              *cpos_out = -1;
              *bpos_out = -1;
              return;
            }

          args_out_of_range (wrap_buffer (b), pos);
        }

      *bpos_out = byte_ind;
      *cpos_out
        = char_ind < 0 ? bytebpos_to_charbpos (b, byte_ind) : char_ind;
      return;
    }

  CHECK_FIXNUM_COERCE_MARKER (pos);

  char_ind = XFIXNUM (pos);

  if (char_ind < char_min_allowed || char_ind > char_max_allowed)
    {
      if (flags & GB_COERCE_RANGE)
	{
	  if (char_ind < char_min_allowed)
	    {
	      *cpos_out = char_min_allowed;
	      *bpos_out = byte_min_allowed;
	      return;
	    }

	  *cpos_out = char_max_allowed;
	  *bpos_out = byte_max_allowed;
	  return;
	}
      else if (flags & GB_NO_ERROR_IF_BAD)
        {
          *cpos_out = -1;
          *bpos_out = -1;
          return;
        }

      args_out_of_range (wrap_buffer (b), pos);
    }

  *cpos_out = char_ind;
  *bpos_out = charbpos_to_bytebpos (b, char_ind);
  return;
}

/* Return a pair of buffer positions representing a range of text,
   taken from a pair of Lisp_Objects.  Full error-checking is
   done on the positions.  Flags can be specified to control the
   behavior of out-of-range values.  The default behavior is to
   allow the range bounds to be specified in either order
   (however, FROM_OUT will always be the lower bound of the range
   and TO_OUT the upper bound),to require that the positions
   are within the accessible part of the buffer (BEGV and ZV),
   and to signal an error if the positions are out of range.
*/

void
get_buffer_range_char (struct buffer *b, Lisp_Object from, Lisp_Object to,
		       Charbpos *from_out, Charbpos *to_out,
		       unsigned int flags)
{
  /* Does not GC */
  Charbpos min_allowed, max_allowed;

  min_allowed = (flags & GB_ALLOW_PAST_ACCESSIBLE) ?
    BUF_BEG (b) : BUF_BEGV (b);
  max_allowed = (flags & GB_ALLOW_PAST_ACCESSIBLE) ?
    BUF_Z (b) : BUF_ZV (b);

  if (NILP (from) && (flags & GB_ALLOW_NIL))
    *from_out = min_allowed;
  else
    *from_out = get_buffer_pos_char (b, from, flags | GB_NO_ERROR_IF_BAD);

  if (NILP (to) && (flags & GB_ALLOW_NIL))
    *to_out = max_allowed;
  else
    *to_out = get_buffer_pos_char (b, to, flags | GB_NO_ERROR_IF_BAD);

  if ((*from_out < 0 || *to_out < 0) && !(flags & GB_NO_ERROR_IF_BAD))
    {
      Lisp_Object buffer = wrap_buffer (b);

      args_out_of_range_3 (buffer, from, to);
    }

  if (*from_out >= 0 && *to_out >= 0 && *from_out > *to_out)
    {
      if (flags & GB_CHECK_ORDER)
	invalid_argument_2 ("start greater than end", from, to);
      else
	{
	  Charbpos temp = *from_out;
	  *from_out = *to_out;
	  *to_out = temp;
	}
    }
}

void
get_buffer_range_byte (struct buffer *b, Lisp_Object from, Lisp_Object to,
		       Bytebpos *from_out, Bytebpos *to_out,
		       unsigned int flags)
{
  if (((MARKERP (from) && XMARKER (from)->buffer == b) || NILP (from))
      && ((MARKERP (to) && XMARKER (to)->buffer == b) || NILP (to)))
    {
      /* Does not GC */
      Bytebpos min_allowed, max_allowed;

      min_allowed = (flags & GB_ALLOW_PAST_ACCESSIBLE) ?
        BYTE_BUF_BEG (b) : BYTE_BUF_BEGV (b);
      max_allowed = (flags & GB_ALLOW_PAST_ACCESSIBLE) ?
        BYTE_BUF_Z (b) : BYTE_BUF_ZV (b);

      if (NILP (from) && (flags & GB_ALLOW_NIL))
        *from_out = min_allowed;
      else
        *from_out = get_buffer_pos_byte (b, from, flags | GB_NO_ERROR_IF_BAD);

      if (NILP (to) && (flags & GB_ALLOW_NIL))
        *to_out = max_allowed;
      else
        *to_out = get_buffer_pos_byte (b, to, flags | GB_NO_ERROR_IF_BAD);

      if ((*from_out < 0 || *to_out < 0) && !(flags & GB_NO_ERROR_IF_BAD))
        {
          args_out_of_range_3 (wrap_buffer (b), from, to);
        }

      if (*from_out >= 0 && *to_out >= 0 && *from_out > *to_out)
        {
          if (flags & GB_CHECK_ORDER)
            invalid_argument_2 ("start greater than end", from, to);
          else
            {
              Bytebpos temp = *from_out;
              *from_out = *to_out;
              *to_out = temp;
            }
        }
    }
  else
    {
      Charbpos s, e;

      get_buffer_range_char (b, from, to, &s, &e, flags);
      if (s >= 0)
        *from_out = charbpos_to_bytebpos (b, s);
      else /* could happen with GB_NO_ERROR_IF_BAD */
        *from_out = -1;
      if (e >= 0)
        *to_out = charbpos_to_bytebpos (b, e);
      else
        *to_out = -1;
    }
}

static Charcount
get_string_pos_char_1 (Lisp_Object string, Lisp_Object pos, unsigned int flags,
		       Charcount known_length)
{
  Charcount ccpos;
  Charcount min_allowed = (flags & GB_NEED_CHAR_BEFORE) ? 1 : 0;
  Charcount max_allowed = (flags & GB_NEED_CHAR_AFTER) ? known_length -1 : known_length;

  /* Computation of KNOWN_LENGTH is potentially expensive so we pass
     it in. */
  CHECK_FIXNUM (pos);
  ccpos = XFIXNUM (pos);
  if (ccpos < 0 && flags & GB_NEGATIVE_FROM_END)
    ccpos += max_allowed;

  if (ccpos < min_allowed || ccpos > max_allowed)
    {
      if (flags & GB_COERCE_RANGE)
	ccpos = ccpos < min_allowed ? min_allowed : max_allowed;
      else if (flags & GB_NO_ERROR_IF_BAD)
	ccpos = -1;
      else
	args_out_of_range (string, pos);
    }

  return ccpos;
}

Charcount
get_string_pos_char (Lisp_Object string, Lisp_Object pos, unsigned int flags)
{
  return get_string_pos_char_1 (string, pos, flags,
				string_char_length (string));
}

static Bytecount
get_string_pos_byte_1 (Lisp_Object string, Lisp_Object pos, unsigned int flags,
                       Charcount char_length)
{
  Charcount ccpos;
  Boolint need_char_before = !!(flags & GB_NEED_CHAR_BEFORE);
  Boolint need_char_after = !!(flags & GB_NEED_CHAR_AFTER);
  Charcount min_cpos_allowed = need_char_before ? 1 : 0;
  Charcount max_cpos_allowed = char_length < 0 ? MOST_POSITIVE_FIXNUM + 1
    : char_length - need_char_after;
  Bytecount bpos = -1;

  CHECK_FIXNUM (pos);
  ccpos = XFIXNUM (pos);

  if (ccpos < 0 && flags & GB_NEGATIVE_FROM_END)
    {
      if (char_length < 0)
        {
          if (ccpos > -6 &&
              /* These are rare, and considerably complicate the below if
                 non-zero: */
              !need_char_after && !need_char_before)
            {
              Charcount counter = ccpos;
              bpos = XSTRING_LENGTH (string);

              while (counter && bpos > 0)
                {
                  bpos = prev_string_index (string, bpos);
                  counter++;
                }

              if (counter == 0)
                {
                  return bpos;
                }
              /* Fallthrough, do the error checking: */
            }
          char_length = string_char_length (string);
        }
      max_cpos_allowed = char_length - need_char_after;
      ccpos += char_length;
    }

  if (ccpos < min_cpos_allowed)
    {
      if (flags & GB_COERCE_RANGE)
        {
          /* Zero and the length of the character at XSTRING_DATA (string)
             (which will be one with zero termination and a zero length
             string) are always valid return values for this, the difficulty
             below with GB_NEED_CHAR_AFTER doesn't arise with
             GB_NEED_CHAR_BEFORE. */
          bpos =
            need_char_before ? itext_ichar_len (XSTRING_DATA (string)) : 0;
          return bpos;
        }
      else if (flags & GB_NO_ERROR_IF_BAD)
        {
          return -1;
        }
      else
        {
          args_out_of_range (string, pos);
        }
    }

  if (char_length < 0 && 
      /* XSTRING_LENGTH (string) / MAX_ICHAR_LEN is a lower bound on the
         character length of the string, if ccpos is greater than this we need
         to actually check the string's character length. */
      ccpos > ((XSTRING_LENGTH (string) / MAX_ICHAR_LEN) -
               (MAX_ICHAR_LEN * need_char_after)))
    {
      char_length = string_char_length (string);
      max_cpos_allowed = char_length - need_char_after;
    }

  if (ccpos > max_cpos_allowed)
    {
      if (flags & GB_COERCE_RANGE
          /* A negative MAX_CPOS_ALLOWED can happen with a zero-length string
             and GB_NEED_CHAR_AFTER */
          && max_cpos_allowed >= 0)
        {
          return need_char_after ?
            prev_string_index (string, XSTRING_LENGTH (string)) :
            XSTRING_LENGTH (string);
        }
      else if (flags & GB_NO_ERROR_IF_BAD)
        {
          return -1;
        }
      else
        {
          args_out_of_range (string, pos);
        }
    }

  return string_index_char_to_byte (string, ccpos);
}

Bytecount
get_string_pos_byte (Lisp_Object string, Lisp_Object pos, unsigned int flags)
{
  return get_string_pos_byte_1 (string, pos, flags, -1);
}

void
get_string_range_char (Lisp_Object string, Lisp_Object from, Lisp_Object to,
		       Charcount *from_out, Charcount *to_out,
		       unsigned int flags)
{
  Charcount min_allowed = 0;
  Charcount max_allowed = string_char_length (string);

  if (NILP (from) && (flags & GB_ALLOW_NIL))
    *from_out = min_allowed;
  else
    *from_out = get_string_pos_char_1 (string, from,
				       flags | GB_NO_ERROR_IF_BAD,
				       max_allowed);

  if (NILP (to) && (flags & GB_ALLOW_NIL))
    *to_out = max_allowed;
  else
    *to_out = get_string_pos_char_1 (string, to,
				     flags | GB_NO_ERROR_IF_BAD,
				     max_allowed);

  if ((*from_out < 0 || *to_out < 0) && !(flags & GB_NO_ERROR_IF_BAD))
    args_out_of_range_3 (string, from, to);

  if (*from_out >= 0 && *to_out >= 0 && *from_out > *to_out)
    {
      if (flags & GB_CHECK_ORDER)
	invalid_argument_2 ("start greater than end", from, to);
      else
	{
	  Charbpos temp = *from_out;
	  *from_out = *to_out;
	  *to_out = temp;
	}
    }
}

void
get_string_range_byte (Lisp_Object string, Lisp_Object from, Lisp_Object to,
		       Bytecount *from_out, Bytecount *to_out,
		       unsigned int flags)
{
  Charcount char_length = -1;

  if (NILP (from) && (flags & GB_ALLOW_NIL))
    *from_out = 0;
  else
    {
      Charcount cfrom;
      CHECK_FIXNUM (from);

      cfrom = XFIXNUM (from);

      if ((cfrom < 0 && flags & GB_NEGATIVE_FROM_END)
          /* XSTRING_LENGTH (string) / MAX_ICHAR_LEN is a lower bound on the
             character length of the string, if cfrom is greater than this we
             need to actually check the string's character length. */
          || cfrom > (XSTRING_LENGTH (string) / MAX_ICHAR_LEN))
        {
          char_length = string_char_length (string);
        }

      *from_out = get_string_pos_byte_1 (string, from,
                                         flags | GB_NO_ERROR_IF_BAD,
                                         char_length);
    }

  if (NILP (to) && (flags & GB_ALLOW_NIL))
    {
      *to_out = XSTRING_LENGTH (string);
    }
  else
    {
      *to_out = get_string_pos_byte_1 (string, to,
                                       flags | GB_NO_ERROR_IF_BAD,
                                       char_length);
    }

  if ((*from_out < 0 || *to_out < 0) && !(flags & GB_NO_ERROR_IF_BAD))
    args_out_of_range_3 (string, from, to);

  if (*from_out >= 0 && *to_out >= 0 && *from_out > *to_out)
    {
      if (flags & GB_CHECK_ORDER)
	invalid_argument_2 ("start greater than end", from, to);
      else
	{
	  Bytebpos temp = *from_out;
	  *from_out = *to_out;
	  *to_out = temp;
	}
    }
}

Charxpos
get_buffer_or_string_pos_char (Lisp_Object object, Lisp_Object pos,
			       unsigned int flags)
{
  return STRINGP (object) ?
    get_string_pos_char (object, pos, flags) :
    get_buffer_pos_char (XBUFFER (object), pos, flags);
}

Bytexpos
get_buffer_or_string_pos_byte (Lisp_Object object, Lisp_Object pos,
			       unsigned int flags)
{
  return STRINGP (object) ?
    get_string_pos_byte (object, pos, flags) :
    get_buffer_pos_byte (XBUFFER (object), pos, flags);
}

void
get_buffer_or_string_range_char (Lisp_Object object, Lisp_Object from,
				 Lisp_Object to, Charxpos *from_out,
				 Charxpos *to_out, unsigned int flags)
{
  if (STRINGP (object))
    get_string_range_char (object, from, to, from_out, to_out, flags);
  else
    get_buffer_range_char (XBUFFER (object), from, to, from_out, to_out,
			   flags);
}

void
get_buffer_or_string_range_byte (Lisp_Object object, Lisp_Object from,
				 Lisp_Object to, Bytexpos *from_out,
				 Bytexpos *to_out, unsigned int flags)
{
  if (STRINGP (object))
    get_string_range_byte (object, from, to, from_out, to_out, flags);
  else
    get_buffer_range_byte (XBUFFER (object), from, to, from_out, to_out,
			   flags);
}

Charxpos
buffer_or_string_accessible_begin_char (Lisp_Object object)
{
  return STRINGP (object) ? 0 : BUF_BEGV (XBUFFER (object));
}

Charxpos
buffer_or_string_accessible_end_char (Lisp_Object object)
{
  return STRINGP (object) ?
    string_char_length (object) : BUF_ZV (XBUFFER (object));
}

Bytexpos
buffer_or_string_accessible_begin_byte (Lisp_Object object)
{
  return STRINGP (object) ? 0 : BYTE_BUF_BEGV (XBUFFER (object));
}

Bytexpos
buffer_or_string_accessible_end_byte (Lisp_Object object)
{
  return STRINGP (object) ?
    XSTRING_LENGTH (object) : BYTE_BUF_ZV (XBUFFER (object));
}

Charxpos
buffer_or_string_absolute_begin_char (Lisp_Object object)
{
  return STRINGP (object) ? 0 : BUF_BEG (XBUFFER (object));
}

Charxpos
buffer_or_string_absolute_end_char (Lisp_Object object)
{
  return STRINGP (object) ?
    string_char_length (object) : BUF_Z (XBUFFER (object));
}

Bytexpos
buffer_or_string_absolute_begin_byte (Lisp_Object object)
{
  return STRINGP (object) ? 0 : BYTE_BUF_BEG (XBUFFER (object));
}

Bytexpos
buffer_or_string_absolute_end_byte (Lisp_Object object)
{
  return STRINGP (object) ?
    XSTRING_LENGTH (object) : BYTE_BUF_Z (XBUFFER (object));
}

Charbpos
charbpos_clip_to_bounds (Charbpos lower, Charbpos num, Charbpos upper)
{
  return (num < lower ? lower :
	  num > upper ? upper :
	  num);
}

Bytebpos
bytebpos_clip_to_bounds (Bytebpos lower, Bytebpos num, Bytebpos upper)
{
  return (num < lower ? lower :
	  num > upper ? upper :
	  num);
}

Charxpos
charxpos_clip_to_bounds (Charxpos lower, Charxpos num, Charxpos upper)
{
  return (num < lower ? lower :
	  num > upper ? upper :
	  num);
}

Bytexpos
bytexpos_clip_to_bounds (Bytexpos lower, Bytexpos num, Bytexpos upper)
{
  return (num < lower ? lower :
	  num > upper ? upper :
	  num);
}

/* These could be implemented in terms of the get_buffer_or_string()
   functions above, but those are complicated and handle lots of weird
   cases stemming from uncertain external input. */

Charxpos
buffer_or_string_clip_to_accessible_char (Lisp_Object object, Charxpos pos)
{
  return (charxpos_clip_to_bounds
	  (pos, buffer_or_string_accessible_begin_char (object),
	   buffer_or_string_accessible_end_char (object)));
}

Bytexpos
buffer_or_string_clip_to_accessible_byte (Lisp_Object object, Bytexpos pos)
{
  return (bytexpos_clip_to_bounds
	  (pos, buffer_or_string_accessible_begin_byte (object),
	   buffer_or_string_accessible_end_byte (object)));
}

Charxpos
buffer_or_string_clip_to_absolute_char (Lisp_Object object, Charxpos pos)
{
  return (charxpos_clip_to_bounds
	  (pos, buffer_or_string_absolute_begin_char (object),
	   buffer_or_string_absolute_end_char (object)));
}

Bytexpos
buffer_or_string_clip_to_absolute_byte (Lisp_Object object, Bytexpos pos)
{
  return (bytexpos_clip_to_bounds
	  (pos, buffer_or_string_absolute_begin_byte (object),
	   buffer_or_string_absolute_end_byte (object)));
}


/************************************************************************/
/*           Implement TO_EXTERNAL_FORMAT, TO_INTERNAL_FORMAT           */
/************************************************************************/

typedef struct
{
  Dynarr_declare (Ibyte_dynarr *);
} Ibyte_dynarr_dynarr;

typedef struct
{
  Dynarr_declare (Extbyte_dynarr *);
} Extbyte_dynarr_dynarr;

static Extbyte_dynarr_dynarr *conversion_out_dynarr_list;
static Ibyte_dynarr_dynarr *conversion_in_dynarr_list;

static int dfc_convert_to_external_format_in_use;
static int dfc_convert_to_internal_format_in_use;

void
dfc_convert_to_external_format (dfc_conversion_type source_type,
				dfc_conversion_data *source,
				Lisp_Object coding_system,
				dfc_conversion_type sink_type,
				dfc_conversion_data *sink)
{
  /* It's guaranteed that many callers are not prepared for GC here,
     esp. given that this code conversion occurs in many very hidden
     places. */
  int count;
  Extbyte_dynarr *conversion_out_dynarr;
  PROFILE_DECLARE ();

  assert (!inhibit_non_essential_conversion_operations);
  PROFILE_RECORD_ENTERING_SECTION (QSin_internal_external_conversion);

  count = begin_gc_forbidden ();

  type_checking_assert
    (((source_type == DFC_TYPE_DATA) ||
      (source_type == DFC_TYPE_LISP_LSTREAM && LSTREAMP (source->lisp_object)) ||
      (source_type == DFC_TYPE_LISP_STRING && STRINGP (source->lisp_object)))
     &&
     ((sink_type == DFC_TYPE_DATA) ||
      (sink_type == DFC_TYPE_LISP_LSTREAM && LSTREAMP (source->lisp_object))));

  if (Dynarr_length (conversion_out_dynarr_list) <=
      dfc_convert_to_external_format_in_use)
    Dynarr_add (conversion_out_dynarr_list, Dynarr_new (Extbyte));
  conversion_out_dynarr = Dynarr_at (conversion_out_dynarr_list,
				     dfc_convert_to_external_format_in_use);
  Dynarr_reset (conversion_out_dynarr);

  internal_bind_int (&dfc_convert_to_external_format_in_use,
		     dfc_convert_to_external_format_in_use + 1);

  coding_system = get_coding_system_for_text_file (coding_system, 0);

  /* Here we optimize in the case where the coding system does no
     conversion. However, we don't want to optimize in case the source
     or sink is an lstream, since writing to an lstream can cause a
     garbage collection, and this could be problematic if the source
     is a lisp string. */
  if (source_type != DFC_TYPE_LISP_LSTREAM &&
      sink_type   != DFC_TYPE_LISP_LSTREAM &&
      coding_system_is_binary (coding_system))
    {
      const Ibyte *ptr;
      Bytecount len;

      if (source_type == DFC_TYPE_LISP_STRING)
	{
	  ptr = XSTRING_DATA   (source->lisp_object);
	  len = XSTRING_LENGTH (source->lisp_object);
	}
      else
	{
	  ptr = (Ibyte *) source->data.ptr;
	  len = source->data.len;
	}

#ifdef MULE
      {
	const Ibyte *end;
	for (end = ptr + len; ptr < end;)
	  {
	    Ibyte c;
	    if (byte_ascii_p (*ptr))
	      c = *ptr;
	    else
	      {
		Ichar ich = non_ascii_itext_ichar (ptr);
		if (ich < 256)
		  c = (Ibyte) ich;
		else
		  /* #### This is just plain unacceptable. */
		  /* untranslatable character */
		  c = CANT_CONVERT_CHAR_WHEN_ENCODING;
	      }

	    Dynarr_add (conversion_out_dynarr, (Extbyte) c);
	    INC_IBYTEPTR (ptr);
	  }
	text_checking_assert (ptr == end);
      }
#else
      Dynarr_add_many (conversion_out_dynarr, ptr, len);
#endif

    }
#ifdef WIN32_ANY
  /* Optimize the common case involving Unicode where only ASCII is involved */
  else if (source_type != DFC_TYPE_LISP_LSTREAM &&
	   sink_type   != DFC_TYPE_LISP_LSTREAM &&
	   dfc_coding_system_is_unicode (coding_system))
    {
      const Ibyte *ptr, *p;
      Bytecount len;
      const Ibyte *end;

      if (source_type == DFC_TYPE_LISP_STRING)
	{
	  ptr = XSTRING_DATA   (source->lisp_object);
	  len = XSTRING_LENGTH (source->lisp_object);
	}
      else
	{
	  ptr = (Ibyte *) source->data.ptr;
	  len = source->data.len;
	}
      end = ptr + len;

      for (p = ptr; p < end; p++)
	{
	  if (!byte_ascii_p (*p))
	    goto the_hard_way;
	}

      for (p = ptr; p < end; p++)
	{
	  Dynarr_add (conversion_out_dynarr, (Extbyte) (*p));
	  Dynarr_add (conversion_out_dynarr, (Extbyte) '\0');
	}
    }
#endif /* WIN32_ANY */
  else
    {
      Lisp_Object streams_to_delete[3];
      int delete_count;
      Lisp_Object instream, outstream;
      Lstream *reader, *writer;

#ifdef WIN32_ANY
    the_hard_way:
#endif /* WIN32_ANY */
      delete_count = 0;
      if (source_type == DFC_TYPE_LISP_LSTREAM)
	instream = source->lisp_object;
      else if (source_type == DFC_TYPE_DATA)
	streams_to_delete[delete_count++] = instream =
	  make_fixed_buffer_input_stream (source->data.ptr, source->data.len);
      else
	{
	  type_checking_assert (source_type == DFC_TYPE_LISP_STRING);
	  streams_to_delete[delete_count++] = instream =
	    /* This will GCPRO the Lisp string */
	    make_lisp_string_input_stream (source->lisp_object, 0, -1);
	}

      if (sink_type == DFC_TYPE_LISP_LSTREAM)
	outstream = sink->lisp_object;
      else
	{
	  type_checking_assert (sink_type == DFC_TYPE_DATA);
	  streams_to_delete[delete_count++] = outstream =
	    make_dynarr_output_stream
	    ((unsigned_char_dynarr *) conversion_out_dynarr);
	}

      streams_to_delete[delete_count++] = outstream =
	make_coding_output_stream (XLSTREAM (outstream), coding_system,
				   CODING_ENCODE, 0);

      reader = XLSTREAM (instream);
      writer = XLSTREAM (outstream);
      /* decoding_stream will gc-protect outstream */
      {
	struct gcpro gcpro1, gcpro2;
	GCPRO2 (instream, outstream);

	while (1)
	  {
	    Bytecount size_in_bytes;
	    char tempbuf[1024]; /* some random amount */

	    size_in_bytes = Lstream_read (reader, tempbuf, sizeof (tempbuf));

	    if (size_in_bytes == 0)
	      break;
	    else if (size_in_bytes < 0)
	      text_conversion_error ("Error converting to external format",
				     Qunbound);

	    if (Lstream_write (writer, tempbuf, size_in_bytes) < 0)
	      text_conversion_error ("Error converting to external format",
				     Qunbound);
	  }

	/* Closing writer will close any stream at the other end of writer. */
	Lstream_close (writer);
	Lstream_close (reader);
	UNGCPRO;
      }

      /* The idea is that this function will create no garbage. */
      while (delete_count)
	Lstream_delete (XLSTREAM (streams_to_delete [--delete_count]));
    }

  unbind_to (count);

  if (sink_type != DFC_TYPE_LISP_LSTREAM)
    {
      sink->data.len = Dynarr_length (conversion_out_dynarr);
      /* double zero-extend because we may be dealing with Unicode data */
      Dynarr_add (conversion_out_dynarr, '\0');
      Dynarr_add (conversion_out_dynarr, '\0');
      sink->data.ptr = Dynarr_begin (conversion_out_dynarr);
    }

  PROFILE_RECORD_EXITING_SECTION (QSin_internal_external_conversion);
}

void
dfc_convert_to_internal_format (dfc_conversion_type source_type,
				dfc_conversion_data *source,
				Lisp_Object coding_system,
				dfc_conversion_type sink_type,
				dfc_conversion_data *sink)
{
  /* It's guaranteed that many callers are not prepared for GC here,
     esp. given that this code conversion occurs in many very hidden
     places. */
  int count;
  Ibyte_dynarr *conversion_in_dynarr;
  Lisp_Object underlying_cs;
  PROFILE_DECLARE ();

  assert (!inhibit_non_essential_conversion_operations);
  PROFILE_RECORD_ENTERING_SECTION (QSin_external_internal_conversion);

  count = begin_gc_forbidden ();

  type_checking_assert
    ((source_type == DFC_TYPE_DATA ||
      source_type == DFC_TYPE_LISP_LSTREAM)
    &&
    (sink_type   == DFC_TYPE_DATA ||
     sink_type   == DFC_TYPE_LISP_LSTREAM));

  if (Dynarr_length (conversion_in_dynarr_list) <=
      dfc_convert_to_internal_format_in_use)
    Dynarr_add (conversion_in_dynarr_list, Dynarr_new (Ibyte));
  conversion_in_dynarr = Dynarr_at (conversion_in_dynarr_list,
				    dfc_convert_to_internal_format_in_use);
  Dynarr_reset (conversion_in_dynarr);

  internal_bind_int (&dfc_convert_to_internal_format_in_use,
		     dfc_convert_to_internal_format_in_use + 1);

  /* The second call does the equivalent of both calls, but we need
     the result after the first call (which wraps just a to-text
     converter) as well as the result after the second call (which
     also wraps an EOL-detection converter). */
  underlying_cs = get_coding_system_for_text_file (coding_system, 0);
  coding_system = get_coding_system_for_text_file (underlying_cs, 1);

  if (source_type != DFC_TYPE_LISP_LSTREAM &&
      sink_type   != DFC_TYPE_LISP_LSTREAM &&
      coding_system_is_binary (underlying_cs))
    {
#ifdef MULE
      const Ibyte *ptr;
      Bytecount len = source->data.len;
      const Ibyte *end;

      /* Make sure no EOL conversion is needed.  With a little work we
	 could handle EOL conversion as well but it may not be needed as an
	 optimization. */
      if (!EQ (coding_system, underlying_cs))
	{
	  for (ptr = (const Ibyte *) source->data.ptr, end = ptr + len;
	       ptr < end; ptr++)
	    {
	      if (*ptr == '\r' || *ptr == '\n')
		goto the_hard_way;
	    }
	}

      for (ptr = (const Ibyte *) source->data.ptr, end = ptr + len;
	   ptr < end; ptr++)
        {
          Ibyte c = *ptr;

	  DECODE_ADD_BINARY_CHAR (c, conversion_in_dynarr);
        }
#else
      Dynarr_add_many (conversion_in_dynarr, source->data.ptr, source->data.len);
#endif
    }
#ifdef WIN32_ANY
  /* Optimize the common case involving Unicode where only ASCII/Latin-1 is
     involved */
  else if (source_type != DFC_TYPE_LISP_LSTREAM &&
	   sink_type   != DFC_TYPE_LISP_LSTREAM &&
	   dfc_coding_system_is_unicode (underlying_cs))
    {
      const Ibyte *ptr;
      Bytecount len = source->data.len;
      const Ibyte *end;

      if (len & 1)
	goto the_hard_way;

      /* Make sure only ASCII/Latin-1 is involved */
      for (ptr = (const Ibyte *) source->data.ptr + 1, end = ptr + len;
	   ptr < end; ptr += 2)
	{
	  if (*ptr)
	    goto the_hard_way;
	}

      /* Make sure no EOL conversion is needed.  With a little work we
	 could handle EOL conversion as well but it may not be needed as an
	 optimization. */
      if (!EQ (coding_system, underlying_cs))
	{
	  for (ptr = (const Ibyte *) source->data.ptr, end = ptr + len;
	       ptr < end; ptr += 2)
	    {
	      if (*ptr == '\r' || *ptr == '\n')
		goto the_hard_way;
	    }
	}

      for (ptr = (const Ibyte *) source->data.ptr, end = ptr + len;
	   ptr < end; ptr += 2)
	{
          Ibyte c = *ptr;

	  DECODE_ADD_BINARY_CHAR (c, conversion_in_dynarr);
        }
    }
#endif /* WIN32_ANY */
  else
    {
      Lisp_Object streams_to_delete[3];
      int delete_count;
      Lisp_Object instream, outstream;
      Lstream *reader, *writer;

#if defined (WIN32_ANY) || defined (MULE)
    the_hard_way:
#endif
      delete_count = 0;
      if (source_type == DFC_TYPE_LISP_LSTREAM)
	instream = source->lisp_object;
      else
	{
	  type_checking_assert (source_type == DFC_TYPE_DATA);
	  streams_to_delete[delete_count++] = instream =
	    make_fixed_buffer_input_stream (source->data.ptr, source->data.len);
	}

      if (sink_type == DFC_TYPE_LISP_LSTREAM)
	outstream = sink->lisp_object;
      else
	{
	  type_checking_assert (sink_type == DFC_TYPE_DATA);
	  streams_to_delete[delete_count++] = outstream =
	    make_dynarr_output_stream
	    ((unsigned_char_dynarr *) conversion_in_dynarr);
	}

      streams_to_delete[delete_count++] = outstream =
	make_coding_output_stream (XLSTREAM (outstream), coding_system,
				   CODING_DECODE, 0);

      reader = XLSTREAM (instream);
      writer = XLSTREAM (outstream);
      {
	struct gcpro gcpro1, gcpro2;
	/* outstream will gc-protect its sink stream, if necessary */
	GCPRO2 (instream, outstream);

	while (1)
	  {
	    Bytecount size_in_bytes;
	    char tempbuf[1024]; /* some random amount */

	    size_in_bytes = Lstream_read (reader, tempbuf, sizeof (tempbuf));

	    if (size_in_bytes == 0)
	      break;
	    else if (size_in_bytes < 0)
	      text_conversion_error ("Error converting to internal format",
				     Qunbound);

	    if (Lstream_write (writer, tempbuf, size_in_bytes) < 0)
	      text_conversion_error ("Error converting to internal format",
				     Qunbound);
	  }

	/* Closing writer will close any stream at the other end of writer. */
	Lstream_close (writer);
	Lstream_close (reader);
	UNGCPRO;
      }

      /* The idea is that this function will create no garbage. */
      while (delete_count)
	Lstream_delete (XLSTREAM (streams_to_delete [--delete_count]));
    }

  unbind_to (count);

  if (sink_type != DFC_TYPE_LISP_LSTREAM)
    {
      sink->data.len = Dynarr_length (conversion_in_dynarr);
      Dynarr_add (conversion_in_dynarr, '\0'); /* remember to NUL-terminate! */
      /* The macros don't currently distinguish between internal and
	 external sinks, and allocate and copy two extra bytes in both
	 cases.  So we add a second zero, just like for external data
	 (in that case, because we may be converting to Unicode). */
      Dynarr_add (conversion_in_dynarr, '\0');
      sink->data.ptr = Dynarr_begin (conversion_in_dynarr);
    }

  PROFILE_RECORD_EXITING_SECTION (QSin_external_internal_conversion);
}

/* ----------------------------------------------------------------------- */
/*                         Alloca-conversion helpers                       */
/* ----------------------------------------------------------------------- */

/* For alloca(), things are trickier because the calling function needs to
   allocate.  This means that the caller needs to do the following:

   (a) invoke us to do the conversion, remember the data and return the size.
   (b) alloca() the proper size.
   (c) invoke us again to copy the data.

   We need to handle the possibility of two or more invocations of the
   converter in the same expression.  In such cases it's conceivable that
   the evaluation of the sub-expressions will be overlapping (e.g. one size
   function called, then the other one called, then the copy functions
   called).  To handle this, we keep a list of active data, indexed by the
   src expression. (We use the stringize operator to avoid evaluating the
   expression multiple times.) If the caller uses the exact same src
   expression twice in two converter calls in the same subexpression, we
   will lose, but at least we can check for this and ABORT().  We could
   conceivably try to index on other parameters as well, but there is not
   really any point. */

alloca_convert_vals_dynarr *active_alloca_convert;

int
find_pos_of_existing_active_alloca_convert (const char *srctext)
{
  alloca_convert_vals *vals = NULL;
  int i;

  if (!active_alloca_convert)
    active_alloca_convert = Dynarr_new (alloca_convert_vals);

  for (i = 0; i < Dynarr_length (active_alloca_convert); i++)
    {
      vals = Dynarr_atp (active_alloca_convert, i);
      /* On my system, two different occurrences of the same stringized
	 argument always point to the same string.  However, on someone
	 else's system, that wasn't the case.  We check for equality
	 first, since it seems systems work my way more than the other
	 way. */
      if (vals->srctext == srctext || !strcmp (vals->srctext, srctext))
	return i;
    }

  return -1;
}

/* ----------------------------------------------------------------------- */
/* New-style DFC converters (data is returned rather than stored into var) */
/* ----------------------------------------------------------------------- */

/* We handle here the cases where SRC is a Lisp_Object, internal data
   (sized or unsized), or external data (sized or unsized), and return type
   is unsized alloca() or malloc() data.  If the return type is a
   Lisp_Object, use build_extstring() for unsized external data,
   make_extstring() for sized external data.  If the return type needs to
   be sized data, use the *_TO_SIZED_*() macros, and for other more
   complicated cases, use the original TO_*_FORMAT() macros. */

static void
new_dfc_convert_now_damn_it (const void *src, Bytecount src_size,
			     enum new_dfc_src_type type,
			     void **dst, Bytecount *dst_size,
			     Lisp_Object codesys)
{
  /* #### In the case of alloca(), it would be a bit more efficient, for
     small strings, to use static Dynarr's like are used internally in
     TO_*_FORMAT(), or some other way of avoiding malloc() followed by
     free().  I doubt it really matters, though. */

  switch (type)
    {
    case DFC_EXTERNAL:
      TO_INTERNAL_FORMAT (C_STRING, src,
			  MALLOC, (*dst, *dst_size), codesys);
      break;

    case DFC_SIZED_EXTERNAL:
      TO_INTERNAL_FORMAT (DATA, (src, src_size),
			  MALLOC, (*dst, *dst_size), codesys);
      break;

    case DFC_INTERNAL:
      TO_EXTERNAL_FORMAT (C_STRING, src,
			  MALLOC, (*dst, *dst_size), codesys);
      break;

    case DFC_SIZED_INTERNAL:
      TO_EXTERNAL_FORMAT (DATA, (src, src_size),
			  MALLOC, (*dst, *dst_size), codesys);
      break;

    case DFC_LISP_STRING:
      TO_EXTERNAL_FORMAT (LISP_STRING, GET_LISP_FROM_VOID (src),
			  MALLOC, (*dst, *dst_size), codesys);
      break;

    default:
      ABORT ();
    }

  /* The size is always + 2 because we have double zero-termination at the
     end of all data (for Unicode-correctness). */
  *dst_size += 2;
}

Bytecount
new_dfc_convert_size (const char *srctext, const void *src,
		      Bytecount src_size, enum new_dfc_src_type type,
		      Lisp_Object codesys)
{
  alloca_convert_vals vals;

  int i = find_pos_of_existing_active_alloca_convert (srctext);
  assert (i < 0);

  vals.srctext = srctext;

  new_dfc_convert_now_damn_it (src, src_size, type, &vals.dst, &vals.dst_size,
			       codesys);

  Dynarr_add (active_alloca_convert, vals);
  return vals.dst_size;
}

Bytecount
new_dfc_get_existing_size (const char *srctext)
{
  alloca_convert_vals *vals;
  int i = find_pos_of_existing_active_alloca_convert (srctext);

  assert (i >= 0);
  vals = Dynarr_atp (active_alloca_convert, i);
  return vals->dst_size;
}

void *
new_dfc_convert_copy_data (const char *srctext, void *alloca_data)
{
  alloca_convert_vals *vals;
  int i = find_pos_of_existing_active_alloca_convert (srctext);

  assert (i >= 0);
  vals = Dynarr_atp (active_alloca_convert, i);
  assert (alloca_data);
  memcpy (alloca_data, vals->dst, vals->dst_size);
  xfree (vals->dst);
  Dynarr_delete (active_alloca_convert, i);
  return alloca_data;
}

void *
new_dfc_convert_malloc (const void *src, Bytecount src_size,
			enum new_dfc_src_type type, Lisp_Object codesys)
{
  void *dst;
  Bytecount dst_size;

  new_dfc_convert_now_damn_it (src, src_size, type, &dst, &dst_size, codesys);
  return dst;
}


/************************************************************************/
/*                        streams of Ichars                             */
/************************************************************************/

#ifdef MULE

/* Treat a stream as a stream of Ichar's rather than a stream of bytes.
   The functions below are not meant to be called directly; use
   the macros in insdel.h. */

Ichar
Lstream_get_ichar_1 (Lstream *stream, int ch)
{
  Ibyte str[MAX_ICHAR_LEN];
  Ibyte *strptr = str;
  Bytecount bytes;

  str[0] = (Ibyte) ch;

  for (bytes = rep_bytes_by_first_byte (ch) - 1; bytes; bytes--)
    {
      int c = Lstream_getc (stream);
      text_checking_assert (c >= 0);
      *++strptr = (Ibyte) c;
    }
  return itext_ichar (str);
}

int
Lstream_fput_ichar (Lstream *stream, Ichar ch)
{
  Ibyte str[MAX_ICHAR_LEN];
  Bytecount len = set_itext_ichar (str, ch);
  return Lstream_write (stream, str, len) >= 0 ? 0 : -1;
}

void
Lstream_funget_ichar (Lstream *stream, Ichar ch)
{
  Ibyte str[MAX_ICHAR_LEN];
  Bytecount len = set_itext_ichar (str, ch);
  Lstream_unread (stream, str, len);
}

#endif /* MULE */


/************************************************************************/
/*              Lisp primitives for working with characters             */
/************************************************************************/

/* Internally, dimension-1 charset codepoints are "little-endian", stored
   as (0, C2), whereas externally they are "big-endian", stored as (C1, 0). */
void
internal_to_external_charset_codepoint (Lisp_Object charset,
					int int_c1, int int_c2,
					int *ext_c1, int *ext_c2,
					int munge_codepoints)
{
  ASSERT_VALID_CHARSET_CODEPOINT (charset, int_c1, int_c2);
  if (XCHARSET_DIMENSION (charset) == 1)
    {
      *ext_c1 = int_c2;
      *ext_c2 = 0;
    }
  else
    {
      *ext_c1 = int_c1;
      *ext_c2 = int_c2;
    }

  if (munge_codepoints)
    {
      /* Bogus bogus bogus.  Munge the codepoints to match the old way of
	 doing things, where all charset codepoints were coerced to be in
	 the range of 32-127 and ascii and control-1 were handled specially.
      */
      if (EQ (charset, Vcharset_control_1))
	{
	  text_checking_assert (*ext_c2 == 0);
	  text_checking_assert (*ext_c1 >= 128 && *ext_c1 <= 159);
	  *ext_c1 -= 128;
	}
#ifdef MULE
      else if (get_charset_iso2022_type (charset) >= 0)
	{
	  *ext_c1 &= 127;
	  *ext_c2 &= 127;
	}
#endif /* MULE */
    }
}

#ifdef MULE

static int
check_coerce_octet (Lisp_Object charset, Lisp_Object arg, int low, int high,
		    int munge_codepoints)
{
  EMACS_INT retval;
  if (munge_codepoints)
    {
      CHECK_FIXNUM (arg);
      retval = XFIXNUM (arg);

      if (EQ (charset, Vcharset_control_1) && retval >= 0 && retval < 32)
	retval += 128;
      else if (get_charset_iso2022_type (charset) >= 0)
	{
	  /* It is useful (and safe, according to Olivier Galibert) to strip
	     the 8th bit off ARG1 and ARG2 because it allows programmers to
	     write (make-char 'latin-iso8859-2 CODE) where code is the actual
	     Latin 2 code of the character.  */

	  if (high < 128 && retval - 128 >= low && retval - 128 <= high)
	    retval -= 128;
	  if (low >= 128 && retval + 128 >= low && retval + 128 <= high)
	    retval += 128;
	}
      check_integer_range (make_fixnum (retval),
			   make_fixnum (low),
			   make_fixnum (high));
    }
  else
    {
      check_integer_range (arg, make_fixnum (low), make_fixnum (high));
      retval = XFIXNUM (arg);
    }

  return (int) retval;
}

#endif /* MULE */

/******************** Helper functions for basic conversions ***************/


/* Validate an external charset codepoint and convert to internal form.
   This involves reversing the octets for dimension-1 charsets.
   See comment at internal_to_external_charset_codepoint().
   */
Lisp_Object
get_external_charset_codepoint (Lisp_Object charset,
				Lisp_Object arg1, Lisp_Object arg2,
				int *a1, int *a2, int munge_codepoints)
{
#ifdef MULE
  int low1, low2, high1, high2;

  charset = Fget_charset (charset);
  get_charset_limits (charset, &low1, &low2, &high1, &high2);

  if (XCHARSET_DIMENSION (charset) == 1)
    {
      if (!NILP (arg2))
	invalid_argument
	  ("Charset is of dimension one; second octet must be nil", arg2);
      /* Dimension-1 internal and external codepoints are handled
	 differently, see above. */
      *a1 = 0;
      *a2 = check_coerce_octet (charset, arg1, low2, high2, munge_codepoints);
    }
  else
    {
      *a1 = check_coerce_octet (charset, arg1, low1, high1, munge_codepoints);
      *a2 = check_coerce_octet (charset, arg2, low2, high2, munge_codepoints);
    }
  return charset;
#else /* not MULE */
  CHECK_SYMBOL (charset);
  CHECK_NIL (arg2);
  *a1 = 0;
  if (EQ (charset, Vcharset_ascii))
    {
      check_integer_range (arg1, Qzero, make_fixnum (255));
    }
  else if (munge_codepoints)
    {
      EMACS_INT a1val;

      /* If munge checkpoints, we are very free with what we allow, just
	 to make sure we're backward-compatible with various versions of
	 XEmacs and GNU Emacs, and coerce to the proper range. */
      check_integer_range (arg1, Qzero, make_fixnum (255));

      a1val = XFIXNUM (arg1);

      if (EQ (charset, Vcharset_control_1))
	a1val = (a1val & 0x1F) + 128;
      else
	a1val = (a1val & 0x7F) + 128;

      arg1 = make_fixnum (a1val);
    }
  else
    {
      if (EQ (charset, Vcharset_control_1))
        check_integer_range (arg1, make_fixnum (128), make_fixnum (159));
      else
        check_integer_range (arg1, make_fixnum (160), make_fixnum (255));
    }

  *a2 = XFIXNUM (arg1);
  return Vcharset_ascii;
#endif /* (not) MULE */
}

/* Like ichar_to_charset_codepoint(..., CONVERR_FAIL) but takes a
   BUFFER_OR_PRECEDENCE_LIST list and converts it into an internal
   precedence array. */
static void
external_char_to_charset_codepoint (Lisp_Object lispch,
				    Lisp_Object buffer_or_precedence_list,
				    Lisp_Object *charset, int *c1, int *c2,
				    enum converr fail)
{
  Lisp_Object bopa =
    decode_buffer_or_precedence_list (buffer_or_precedence_list);
  Ichar ch;
  CHECK_CHAR_COERCE_INT (lispch);
  ch = XCHAR (lispch);

  if (BUFFERP (bopa))
    buffer_ichar_to_charset_codepoint (ch, XBUFFER (bopa),
				       charset, c1, c2, fail);
  else
    ichar_to_charset_codepoint (ch, bopa, charset, c1, c2, fail);
  if (NILP (*charset))
    return;
}

/* Convert an Lisp integer into a Unicode codepoint, and convert a
   BUFFER_OR_PRECEDENCE_LIST into a precedence array. */
static int
get_external_unicode_codepoint (Lisp_Object unicode,
				Lisp_Object buffer_or_precedence_list,
				Lisp_Object *buffer_or_precarray_out)
{
  int code = decode_unicode (unicode, UNICODE_ALLOW_PRIVATE);
  Lisp_Object bopa =
    decode_buffer_or_precedence_list (buffer_or_precedence_list);

  if (buffer_or_precarray_out)
    *buffer_or_precarray_out = bopa;
  return code;
}

enum converr
decode_handle_error (Lisp_Object err, int allow_use_private)
{
  CHECK_SYMBOL (err);
  if (NILP (err) || EQ (err, Qfail))
    return CONVERR_FAIL;
  if (EQ (err, Qerror))
    return CONVERR_ERROR;
  if (EQ (err, Qsucceed))
    return CONVERR_SUCCEED;
  if (EQ (err, Qsubstitute))
    return CONVERR_SUBSTITUTE;
  if (allow_use_private)
    {
      if (EQ (err, Quse_private))
	return CONVERR_USE_PRIVATE;
      invalid_constant
	("Must be nil, `fail', `error', `succeed', `substitute', or `use-private'", err);
    }
  else
    invalid_constant
      ("Must be nil, `fail', `error', `succeed', or `substitute'", err);
  /* Not reached */
}

DEFUN ("make-char", Fmake_char, 1, 4, 0, /*
Make a character from a charset codepoint (a charset and one or two octets).

This attempts to generate a character from a particular codepoint in a
coded character set such as ISO 8859-1 or GB-2312.  See also
`unicode-to-char', used for generating a character from a Unicode
codepoint.

CHARSET is a symbol naming a "charset", e.g. `latin-iso8859-1'. "Charsets"
are coded character sets, i.e. sets of characters indexed by one or two
octets (integers in the range 0-255, also known as "position codes").
Charsets are typically intended to be sufficient to encode text in a
particular nation's language.  See `make-charset' for more information on
charsets.

OCTET1 and OCTET2 are the octets (i.e. indices) specifying the particular
character to select.  OCTET2 is either required or disallowed, depending on
whether CHARSET is of one or two dimensions (see `make-charset').

HANDLE-ERROR is a symbol controlling error behavior stemming from inability
to translate.  Currently, this happens only with Unicode-internal (see
below):

`use-private' or `nil'	Encode using private Unicode space
`fail'                  Return nil
`error'                 Signal an error
`succeed'               Same as `substitute'
`substitute'            Substitute the Unicode replacement char (0xFFFD)

Each octet should be in the range corresponding to the offset and size
for that dimension, as defined in the charset.  For a typical one-dimensional
charset of size 96, such as ISO-8859-1 (aka Latin-1), the range will be
\[160, 255].  For a typical two-dimensional charset of size 94x94, such as
GB-2312 (Simplified Chinese), JISX-0208 (Japanese), or KS-5601 (Korean),
the range will be [33, 126] in each dimension.  Big5 and certain other
large charsets have a range of [33, 126] in the first dimension and
\[64, 253] or something similar in the second dimension.  Other charsets
may have other dimensions; e.g. ASCII has the range [0, 127] and Control-1
\(i.e. high-bit control characters) has the range [128, 159].

Note that the ranges as used for 94x94 charsets are 32 more in each dimension
than the "ku-ten" representation often seen for these charsets, with a range
of [1, 94] in each dimension.

For compatibility purposes, if the allowed values in a particular dimension
are entirely in the range [0, 127] or [128, 255], octet values offset by
128 are allowed and will be converted appropriately.  Hence, either
\(make-char 'latin-iso8859-2 185) or (make-char 'latin-iso8859-2 57) will
return the Latin 2 character s with caron.  This behavior should not be
relied upon, and is present only in functions that existed prior to XEmacs
21.5 (e.g. `make-char' and `split-char').  It is not present in new
functions in version 21.5 (e.g. `char-to-charset-codepoint').

When two octets are required, the order is "standard" -- the same as
appears in ISO-2022 encodings, reference tables, etc.

\(Note the following non-obvious result: Computerized translation
tables often encode the two octets as the high and low bytes,
respectively, of a hex short, while when there's only one octet, it
goes in the low byte.  When decoding such a value, you need to treat
the two cases differently when calling make-char: One is (make-char
CHARSET HIGH LOW), the other is (make-char CHARSET LOW).)

As an example, the Japanese character for "kawa" (stream), looks something
like this:

   |     |
   |  |  |
   |  |  |
   |  |  |
  /      |

It appears in the Unicode Standard (version 2.0) on page 7-287 with the
following values (see also page 7-4):

U 5DDD     (Unicode)
G 0-2008   (GB 2312-80)
J 0-3278   (JIS X 0208-1990)
K 0-8425   (KS C 5601-1987)
B A474     (Big Five)
C 1-4455   (CNS 11643-1986 (1st plane))
A 213C34   (ANSI Z39.64-1989)

These are equivalent to:

\(make-char 'chinese-gb2312 52 40)
\(make-char 'japanese-jisx0208 64 110)
\(make-char 'korean-ksc5601 116 57)
\(make-char 'chinese-cns11643-1 76 87)
\(decode-big5-char '(164 . 116))

All codes above are two decimal numbers except for Big Five and ANSI
Z39.64, which we don't support.  We add 32 to each of the decimal numbers.
Big Five is split in a rather hackish fashion into two charsets,
`chinese-big5-1' and `chinese-big5-2', due to its excessive size (94x157),
with the first codepoint in the range 0xA1 to 0xFE and the second in the range
0x40 to 0x7E or 0xA1 to 0xFE.  `decode-big5-char' is used to generate the char
from its codes, and `encode-big5-char' extracts the codes.

Note that there are three different internal formats for characters:

1. ("non-Mule", configured `--with-mule=no')
   An integer between 0 and 255.  All characters are taken to be in charset
   `ascii'.  No charset object exists corresponding to this name (see
   `get-charset').

2. ("Unicode-internal", configured `--with-mule' and `--with-unicode-internal')
   An integer representing a Unicode codepoint.

3. ("old-Mule", configured `--with-mule' and `--with-unicode-internal=no')
   An integer that internally encodes a national character set
   (e.g. ISO-8859-1 or GB-2312) and associated codepoint.  This is the old
   Mule representation.  This is a flawed representation because what is
   the same character from a logical standpoint can have multiple
   representations. (This is a particular problem with accented Latin
   characters.)

Note that all three representations more or less agree in encoding ASCII in
the range 0-127 and ISO-8859-1 in the range 128-255. ("More or less"
because representation #1 does not really care what the actual significance
of the characters is.  Under X Windows at least, XEmacs without Mule
support could be made to support various character sets with appropriate
font settings.)

XEmacs tries to hide the internal representation of characters as much as
possible, but it is not completely possible to hide the difference between
representations #2 and #3 because of the explicit encoding of a charset or
lack thereof in the character.  Conversion from Unicode codepoints to
charset codepoints is a one-to-many operation, and requires a charset
precedence list to determine which of many charsets to choose from.  This
means:

-- In a Unicode-internal world, conversion from a character to a charset
   codepoint (`char-charset', `char-octet', `split-char') uses a charset
   precedence list.

-- In an old Mule-internal world, conversion from a Unicode codepoint to
   a character (`make-char') uses a charset precedence list.

In all cases, the precedence list is optional; when not specified, a
default list is used.  See `unicode-to-char' for more information on
charset precedence lists.

When compiled non-Mule, this function does not do much, but it's provided
for compatibility.   CHARSET should be the symbol `ascii', OCTET1 should
be in the range 0-255, and OCTET2 should not be present.  The resulting
character is equivalent to what would be returned by calling `int-to-char'.
For compatibility, if CHARSET is `control-1', OCTET1 is first coerced
\(using logical AND and OR) to the range 128-159, and for other values of
CHARSET, OCTET1 is coerced to the range 128-255 by setting the high bit.
This behavior should not be relied upon.

Note that there are various ways of representing a character:

1. A character object, as returned by `make-char' or `unicode-to-char'.

2. An integer, the raw internal representation of a character.  With
   Unicode-internal, this will simply be a Unicode codepoint.  With old-Mule,
   this will be a 21-bit number in which the charset and octets have been
   encoded in different bit fields.

3. A charset codepoint, consisting of a symbol naming a charset
   (alternatively, a charset object as returned by `get-charset') and
   one or two octets, typically specifying a character in a national
   character set.

4. A Unicode codepoint, an integer between 0 and 0x10FFFF.  Unicode is a
   standard that attempts to encompass all the characters required to
   encoded text in any human language, past or present, as well as certain
   other language-like systems (e.g. mathematical symbols or musical
   notation).
    
The following functions deal with the various representations of a
character:

-- `make-char' and `charset-codepoint-to-char' generate a
   character from a charset codepoint.
-- `char-to-charset-codepoint' converts a character into a charset
   codepoint.  See also the compatibility functions `split-char',
   `char-charset' and `char-octet'.
-- `unicode-to-char' and `char-to-unicode' convert between a character
   and a Unicode codepoint.
-- `int-to-char' and `char-to-int' convert between a character and its
   internal integer representation.
-- `charset-codepoint-to-unicode' and `unicode-to-charset-codepoint'
   convert directly between charset codepoints and Unicode codepoints,
   regardless of the particular representation of a character.

Functions that involve converting from a Unicode codepoint to a charset
codepoint, regardless of whether these codepoints are encapsulated as a
character object, involve a charset precedence list, which is an optional
argument.  The following functions make use of a charset precedence list:

-- Under Unicode-internal, `char-to-charset-codepoint', along with
   `split-char', `char-charset' and `char-octet'.
-- Under old-Mule, `unicode-to-char'.
-- In all circumstances, `unicode-to-charset-codepoint'.
*/
       (charset, octet1, octet2, handle_error))
{
  enum converr fail;
  int a1, a2;
  Ichar ch;

  if (NILP (handle_error))
    {
      handle_error = Quse_private;
    }

  fail = decode_handle_error (handle_error, 1);
  charset = get_external_charset_codepoint (charset, octet1, octet2,
					    &a1, &a2, 1);
  ch = charset_codepoint_to_ichar (charset, a1, a2, fail);
  if (ch < 0)
    return Qnil;
  return make_char (ch);
}

DEFUN ("char-to-unicode", Fchar_to_unicode, 1, 2, 0, /*
Convert character to Unicode codepoint.
When there is no international support (i.e. the 'mule feature is not
present), this function simply does `char-to-int'.

HANDLE-ERROR controls error behavior:

nil or `fail'	Return nil
`error'		Signal an error
`succeed'	Same as `substitute'
`substitute'	Substitute the Unicode replacement char (0xFFFD)
`use-private'	Encode using private Unicode space
*/
       (character, handle_error))
{
  enum converr fail = decode_handle_error (handle_error, 1);
  INT_32_BIT result;

  CHECK_CHAR_COERCE_INT (character);

  result = ichar_to_unicode (XCHAR (character), fail);

  if (result < 0)
    {
      return Qnil;
    }

  return make_fixnum (result);
}

DEFUN ("unicode-to-char", Funicode_to_char, 1, 3, 0, /*
Convert Unicode codepoint to char.

Attempts to generate a character from a particular Unicode codepoint, which
should be a non-negative integer.  When the new Unicode-internal
representation is used, the conversion is quite direct and there are no
errors.

See `make-char' for more information about characters.

When the old Mule-internal representation is used, there are multiple
possible return values; the particular value returned will reflect the
given charset precedence list, or the default precedence.  The codepoint
will be converted according to BUFFER-OR-PRECEDENCE-LIST.  This is either a
Unicode precedence list, a buffer (use that buffer's precedence list, see
`set-buffer-unicode-precedence-list'), or nil (use the current buffer's
precedence list).

A Unicode precedence list is a list of charsets or charset tags, used to
convert Unicode codepoints to charset codepoints.  These are searched in
order for a translation matching a given Unicode character.  Charset tags
are tags that can match multiple charsets and generally correspond to
classes of charsets. (See `define-charset-tag'.)

The actual charset ordering used for converting Unicode codepoints is
determined by concatenating the buffer-specific Unicode precedence list
\(see `set-buffer-unicode-precedence-list'), the default precedence list
\(see `set-default-unicode-precedence-list'), and the list of all charsets;
converting tags to their corresponding charsets using
`charset-tag-to-charset-list'; and removing any duplicates.

When there is no international support \(i.e. the `mule' feature is not
present, caused by configuring `--with-mule=no'), this function simply does
`int-to-char' and ignores the PRECEDENCE-LIST argument. (Redisplay will
work on the sjt-xft branch, but not with server-side X11 fonts as is the
default.)

Under old-Mule, if the UNICODE codepoint would not otherwise be converted
to an XEmacs character, and a buffer or nil was given as the argument to
BUFFER-OR-PRECEDENCE LIST, a new XEmacs character will be created for it in
one of the `jit-ucs-charset' Mule character sets, and that character will
be returned. (More specifically, this will happen when the list of charsets
to be consulted includes `jit-ucs-charset-0', which will be the case for
buffer-local Unicode precedence lists, since they list all existing
charsets in some order.)

HANDLE-ERROR controls error behavior:

nil or `fail'	Return nil
`error'		Signal an error
`succeed'	Same as `substitute'
`substitute'	Substitute a '?' character
*/
       (unicode, buffer_or_precedence_list, handle_error))
{
  Lisp_Object bopa;
  int c = get_external_unicode_codepoint (unicode, buffer_or_precedence_list,
					  &bopa);
  enum converr fail = decode_handle_error (handle_error, 0);
  Ichar ret;

  if (BUFFERP (bopa))
    ret = buffer_unicode_to_ichar (c, XBUFFER (bopa), fail);
  else
    ret = unicode_to_ichar (c, bopa, fail);

  if (ret == -1)
    return Qnil;
  return make_char (ret);
}

DEFUN ("char-to-charset-codepoint", Fchar_to_charset_codepoint, 1, 3, 0, /*
Return a charset codepoint corresponding to character CH.
A charset codepoint is a list of a charset symbol (typically describing a
national character set) and one or two octets, indexing the particular
character in the character set (see `make-char').

Use this function in place of `split-char'.

When a Unicode internal representation is used (--with-unicode-internal
option to configure), CH will be converted according to
BUFFER-OR-PRECEDENCE-LIST.  This is either a Unicode precedence list \(see
`unicode-to-char'), a buffer (use that buffer's precedence list, see
`set-buffer-unicode-precedence-list'), or nil (use the current buffer's
precedence list).  The returned charset is determined by searching the list
of charsets specified by the Unicode precedence list, in order, for the
given character.  The return value will be nil if the character is not
found in any of the charsets in the precedence list.

HANDLE-ERROR controls error behavior:

nil or `fail'	Return nil
`error'		Signal an error
`succeed'	Same as `substitute'
`substitute'	Substitute a '?' character
*/
       (ch, buffer_or_precedence_list, handle_error))
{
  Lisp_Object charset;
  int c1, c2;
  enum converr fail = decode_handle_error (handle_error, 0);

  external_char_to_charset_codepoint (ch, buffer_or_precedence_list,
				      &charset, &c1, &c2, fail);

  if (NILP (charset))
    return Qnil;

  if (XCHARSET_DIMENSION (charset) == 2)
    return list3 (XCHARSET_NAME (charset), make_fixnum (c1), make_fixnum (c2));
  else
    /* See comment at internal_to_external_charset_codepoint(). */
    return list2 (XCHARSET_NAME (charset), make_fixnum (c2));
}


DEFUN ("charset-codepoint-to-unicode", Fcharset_codepoint_to_unicode,
       2, 4, 0, /*
Convert a charset codepoint to a Unicode codepoint.
A charset codepoint is a list of a charset symbol (typically describing a
national character set) and one or two octets, indexing the particular
character in the character set (see `make-char').

HANDLE-ERROR controls error behavior:

nil or `fail'	Return nil
`error'		Signal an error
`succeed'	Same as `substitute'
`substitute'	Substitute the Unicode replacement char (0xFFFD)
`use-private'	Encode using private Unicode space
*/
       (charset, arg1, arg2, handle_error))
{
  int a1, a2;
  enum converr err = decode_handle_error (handle_error, 1);
  int code;

  charset = get_external_charset_codepoint (charset, arg1, arg2, &a1, &a2, 0);
  code = charset_codepoint_to_unicode (charset, a1, a2, err);
  if (code == -1)
    return Qnil;
  return make_fixnum (code);
}

DEFUN ("unicode-to-charset-codepoint", Funicode_to_charset_codepoint,
       1, 3, 0, /*
Convert a Unicode codepoint to a charset codepoint.
A charset codepoint is a list of a charset symbol (typically describing a
national character set) and one or two octets, indexing the particular
character in the character set (see `make-char').

When a Unicode internal representation is used (--with-unicode-internal
option to configure), CH will be converted according to
BUFFER-OR-PRECEDENCE-LIST.  This is either a Unicode precedence list \(see
`unicode-to-char'), a buffer (use that buffer's precedence list, see
`set-buffer-unicode-precedence-list'), or nil (use the current buffer's
precedence list).  The returned charset is determined by searching the list
of charsets specified by the Unicode precedence list, in order, for the
given codepoint.  The return value will be nil if the codepoint is not
found in any of the charsets in the precedence list.

HANDLE-ERROR controls error behavior:

nil or `fail'	Return nil
`error'		Signal an error
`succeed'	Same as `substitute'
`substitute'	Substitute a '?' character
*/
       (code, buffer_or_precedence_list, handle_error))
{
  Lisp_Object bopa;
  int c = get_external_unicode_codepoint (code, buffer_or_precedence_list,
					  &bopa);
  enum converr fail = decode_handle_error (handle_error, 0);
  Lisp_Object charset;
  int a1, a2;

  if (BUFFERP (bopa))
    buffer_unicode_to_charset_codepoint (c, XBUFFER (bopa),
					 &charset, &a1, &a2, fail);
  else
    unicode_to_charset_codepoint (c, bopa, &charset, &a1, &a2, fail);
  
  if (NILP (charset))
    return Qnil;

  if (XCHARSET_DIMENSION (charset) == 2)
    return list3 (XCHARSET_NAME (charset), make_fixnum (a1), make_fixnum (a2));
  else
    /* See comment at internal_to_external_charset_codepoint(). */
    return list2 (XCHARSET_NAME (charset), make_fixnum (a2));
}

DEFUN ("char-charset", Fchar_charset, 1, 3, 0, /*
Convert character CH to a charset codepoint and return the charset.
The returned value is a symbol naming a charset (typically, a national
character set); see `make-char'.

This function is exactly equivalent to the expression
\(first (char-to-charset-codepoint CH PRECEDENCE-LIST HANDLE-ERROR)).

When a Unicode internal representation is used (--with-unicode-internal
option to configure), CH will be converted according to
BUFFER-OR-PRECEDENCE-LIST.  This is either a Unicode precedence list \(see
`unicode-to-char'), a buffer (use that buffer's precedence list, see
`set-buffer-unicode-precedence-list'), or nil (use the current buffer's
precedence list).  The returned charset is determined by searching the list
of charsets specified by the Unicode precedence list, in order, for the
given character.  The return value will be nil if the character is not
found in any of the charsets in the precedence list.

HANDLE-ERROR controls error behavior:

nil or `fail'	Return nil
`error'		Signal an error
`succeed'	Same as `substitute'
`substitute'	Substitute a '?' character
*/
       (ch, buffer_or_precedence_list, handle_error))
{
  Lisp_Object charset;
  int c1, c2;
  enum converr fail = decode_handle_error (handle_error, 0);

  external_char_to_charset_codepoint (ch, buffer_or_precedence_list,
				      &charset, &c1, &c2, fail);

  if (NILP (charset))
    return Qnil;

  return XCHARSET_NAME (charset);
}

DEFUN ("char-octet", Fchar_octet, 1, 4, 0, /*
Return the octet numbered N (should be 0 or 1) of char CH.
N defaults to 0 if omitted.

This function is for compatibility; consider using `char-to-charset-codepoint'
instead.  See `char-to-charset-codepoint' for the semantics of
BUFFER-OR-PRECEDENCE-LIST and HANDLE-ERROR.

This function is not very useful when a Unicode internal representation is
used (--with-unicode-internal option to configure). Specifically, this
function is more or less equivalent to (nth (1+ N) (split-char CH)), but
returns 0 instead of nil when N=1 and the discovered charset of the character
has only one dimension.
*/
       (ch, n, buffer_or_precedence_list, handle_error))
{
  Lisp_Object charset;
  int c1, c2;
  enum converr fail = decode_handle_error (handle_error, 0);

  external_char_to_charset_codepoint (ch, buffer_or_precedence_list,
				      &charset, &c1, &c2, fail);

  if (NILP (charset))
    return Qnil;

  internal_to_external_charset_codepoint (charset, c1, c2, &c1, &c2, 1);

  if (NILP (n) || EQ (n, Qzero))
    return make_fixnum (c1);
  else if (EQ (n, Qone))
    return make_fixnum (c2);
  else
    invalid_constant ("Octet number must be 0 or 1", n);
}

DEFUN ("split-char", Fsplit_char, 1, 3, 0, /*
Return list of charset and one or two position-codes of char CH.

This function is for compatibility; consider using `char-to-charset-codepoint'
instead.

This function is like `char-to-charset-codepoint' but its return
value is hacked up for compatibility purposes: If the returned charset of
the character is ISO-2022 compatible, the position codes will be coerced into
the range [0, 127], even if they should be in the range [128, 255].

When a Unicode internal representation is used (--with-unicode-internal
option to configure), CH will be converted according to
BUFFER-OR-PRECEDENCE-LIST.  This is either a Unicode precedence list \(see
`unicode-to-char'), a buffer (use that buffer's precedence list, see
`set-buffer-unicode-precedence-list'), or nil (use the current buffer's
precedence list).  The returned charset is determined by searching the list
of charsets specified by the Unicode precedence list, in order, for the
given character.  The return value will be nil if the character is not
found in any of the charsets in the precedence list.

HANDLE-ERROR controls error behavior:

nil or `fail'	Return nil
`error'		Signal an error
`succeed'	Same as `substitute'
`substitute'	Substitute a '?' character
*/
       (ch, buffer_or_precedence_list, handle_error))
{
  Lisp_Object charset;
  int c1, c2;
  enum converr fail = decode_handle_error (handle_error, 0);

  external_char_to_charset_codepoint (ch, buffer_or_precedence_list,
				      &charset, &c1, &c2, fail);

  if (NILP (charset))
    return Qnil;

  internal_to_external_charset_codepoint (charset, c1, c2, &c1, &c2, 1);

  if (XCHARSET_DIMENSION (charset) == 2)
    return list3 (XCHARSET_NAME (charset), make_fixnum (c1), make_fixnum (c2));
  else
    /* See comment at internal_to_external_charset_codepoint(). */
    return list2 (XCHARSET_NAME (charset), make_fixnum (c1));
}

DEFUN ("char-width", Fchar_width, 1, 1, 0, /*
Return number of columns a CHARACTER occupies when displayed.
*/
       (character))
{
  CHECK_CHAR_COERCE_INT (character);

  /* #### Consider having this take a graphical context--a buffer or frame or
     #### whatever. Ditto with #'string-width.  */
  return make_fixnum (ichar_columns (XCHAR (character)));
}

/************************************************************************/
/*                     composite character functions                    */
/************************************************************************/

#ifdef ENABLE_COMPOSITE_CHARS

Ichar
lookup_composite_char (Ibyte *str, int len)
{
  Lisp_Object lispstr = make_string (str, len);
  Lisp_Object ch = Fgethash (lispstr,
			     Vcomposite_char_string2char_hash_table,
			     Qunbound);
  Ichar emch;

  if (UNBOUNDP (ch))
    {
      if (composite_char_row_next >= 128)
	invalid_operation ("No more composite chars available", lispstr);
      emch = charset_codepoint_to_ichar (Vcharset_composite,
					 composite_char_row_next,
					 composite_char_col_next,
					 CONVERR_SUCCEED);
      Fputhash (make_char (emch), lispstr,
	        Vcomposite_char_char2string_hash_table);
      Fputhash (lispstr, make_char (emch),
		Vcomposite_char_string2char_hash_table);
      composite_char_col_next++;
      if (composite_char_col_next >= 128)
	{
	  composite_char_col_next = 32;
	  composite_char_row_next++;
	}
    }
  else
    emch = XCHAR (ch);
  return emch;
}

Lisp_Object
composite_char_string (Ichar ch)
{
  Lisp_Object str = Fgethash (make_char (ch),
			      Vcomposite_char_char2string_hash_table,
			      Qunbound);
  assert (!UNBOUNDP (str));
  return str;
}

DEFUN ("make-composite-char", Fmake_composite_char, 1, 1, 0, /*
Convert a string into a single composite character.
The character is the result of overstriking all the characters in
the string.
*/
       (string))
{
  CHECK_STRING (string);
  return make_char (lookup_composite_char (XSTRING_DATA (string),
					   XSTRING_LENGTH (string)));
}

DEFUN ("composite-char-string", Fcomposite_char_string, 1, 1, 0, /*
Return a string of the characters comprising a composite character.
*/
       (ch))
{
  Ichar emch;

  CHECK_CHAR (ch);
  emch = XCHAR (ch);
  if (!EQ (ichar_charset (emch), Vcharset_composite))
    invalid_argument ("Must be composite char", ch);
  return composite_char_string (emch);
}
#endif /* ENABLE_COMPOSITE_CHARS */


/************************************************************************/
/*                            initialization                            */
/************************************************************************/

void
reinit_eistring_early (void)
{
  the_eistring_malloc_zero_init = the_eistring_zero_init;
  the_eistring_malloc_zero_init.mallocp_ = 1;
}

void
init_eistring_once_early (void)
{
  reinit_eistring_early ();
}

void
syms_of_text (void)
{
  DEFSUBR (Fmake_char);
  DEFSUBR (Fchar_to_unicode);
  DEFSUBR (Funicode_to_char);
  DEFSUBR (Fchar_to_charset_codepoint);
  DEFSUBR (Fcharset_codepoint_to_unicode);
  DEFSUBR (Funicode_to_charset_codepoint);
  DEFSUBR (Fchar_charset);
  DEFSUBR (Fchar_octet);
  DEFSUBR (Fsplit_char);
  DEFSUBR (Fchar_width);

  /* Qfail, Qsubstitute, Qsucceed in general.c */
  DEFSYMBOL (Quse_private);

#ifdef ENABLE_COMPOSITE_CHARS
  DEFSUBR (Fmake_composite_char);
  DEFSUBR (Fcomposite_char_string);
#endif
}

void
reinit_vars_of_text (void)
{
  conversion_in_dynarr_list = Dynarr_new2 (Ibyte_dynarr_dynarr,
					   Ibyte_dynarr *);
  conversion_out_dynarr_list = Dynarr_new2 (Extbyte_dynarr_dynarr,
					    Extbyte_dynarr *);

#ifdef MULE
  {
    int i;

    for (i = 0; i <= MAX_BYTEBPOS_GAP_SIZE_3; i++)
      three_to_one_table[i] = i / 3;
  }
#endif /* MULE */
}

void
vars_of_text (void)
{
  QSin_char_byte_conversion = build_defer_string ("(in char-byte conversion)");
  staticpro (&QSin_char_byte_conversion);
  QSin_byte_char_conversion = build_defer_string ("(in byte-char conversion)");
  staticpro (&QSin_byte_char_conversion);
  QSin_internal_external_conversion =
    build_defer_string ("(in internal-external conversion)");
  staticpro (&QSin_internal_external_conversion);
  QSin_external_internal_conversion =
    build_defer_string ("(in external-internal conversion)");
  staticpro (&QSin_external_internal_conversion);

  DEFVAR_CONST_INT ("char-code-limit", &Vchar_code_limit /*
Exclusive upper bound on the values return by `char-int'.

Note that not every fixnum with a value below `char-code-limit' has an
associated character; check with `char-int-p' if necessary.
*/);
  Vchar_code_limit = CHAR_CODE_LIMIT;

#ifdef ENABLE_COMPOSITE_CHARS
  /* #### not dumped properly */
  composite_char_row_next = 32;
  composite_char_col_next = 32;

  Vcomposite_char_string2char_hash_table =
    make_lisp_hash_table (500, HASH_TABLE_NON_WEAK, Qequal);
  Vcomposite_char_char2string_hash_table =
    make_lisp_hash_table (500, HASH_TABLE_NON_WEAK, Qeq);
  staticpro (&Vcomposite_char_string2char_hash_table);
  staticpro (&Vcomposite_char_char2string_hash_table);
#endif /* ENABLE_COMPOSITE_CHARS */
}
