/*
 * Copyright (c)  2000
 * SWsoft  company
 *
 * This material is provided "as is", with absolutely no warranty expressed
 * or implied. Any use is at your own risk.
 *
 * Permission to use or copy this software for any purpose is hereby granted 
 * without fee, provided the above notices are retained on all copies.
 * Permission to modify the code and to distribute modified code is granted,
 * provided the above notices are retained, and a notice that the code was
 * modified is included with the above copyright notice.
 *
 */

//--------------------------------------------------------------------
// MySQL OLE DB Provider 
// Functionality: minimum
// Release: 0.1
//
// @doc 
//
// @module ROWSET.H | CRowset base object and contained interface
// definitions
//
#ifndef _ROWSET_H_
#define _ROWSET_H_

#include "bitarray.h"
#include "extbuff.h"
#include "hashtbl.h"

#define	ADVISELIMIT		5L

// Forward declarations ------------------------------------------------------
class CCommand;
class CImpIRowset;
class CImpIRowsetChange;
class CImpIColumnsInfo;
class CImpIAccessor;
class CImpIRowsetInfo;
class CImpIConvertType;
class CImpIConnectionPointContainer;
class CImpIEnumConnections;
class CImpIEnumConnectionPoints;
class CImpIConnectionPoint;
class CImpIRowsetLocate;
class CImpIRowsetScroll;
class CImpIRowsetFind;
class CImpIRowsetIdentity;
class CImpIRowsetIndex;
class CBtrieve;
struct SWSTINDEXtag;

typedef CImpIRowset*				PIMPIROWSET;
typedef CImpIRowsetChange*			PIMPIROWSETCHANGE;
typedef CImpIColumnsInfo*			PIMPICOLUMNSINFO;
typedef CImpIAccessor *				PIMPIACCESSOR;
typedef CImpIRowsetInfo*			PIMPIROWSETINFO;
typedef CImpIConvertType*			PIMPICONVERTTYPE;
typedef	CImpIConnectionPointContainer* PIMPICONNECTIONPOINTCONTAINER;
typedef	CImpIEnumConnections*		PIMPIENUMCONNECTIONS;
typedef	CImpIEnumConnectionPoints*	PIMPIENUMCONNECTIONPOINTS;
typedef	CImpIConnectionPoint*		PIMPICONNECTIONPOINT;
typedef CImpIRowsetLocate*			PIMPIROWSETLOCATE;
typedef CImpIRowsetScroll*			PIMPIROWSETSCROLL;
typedef CImpIRowsetFind*			PIMPIROWSETFIND;
typedef	CImpIRowsetIdentity*		PIMPIROWSETIDENTITY;
typedef	CImpIRowsetIndex*			PIMPIROWSETINDEX;

// General Helper Function
HRESULT GetInternalTypeFromSwstType(SWORD swBDataType, UDWORD cbColDef, BOOL fIsSigned, DWORD* pdwdbType);

struct	SLOTREF
{
	ULONG	NextSlot;
	BYTE	Status;
	HROW	hRow;
};

enum ADVANCED_INTERFACES
{
	ROWSET_NONE		= 0,
	ROWSET_FIND		= 1,
	ROWSET_SCROLL	= 2,
	ROWSET_CHANGE	= 4,
	ROWSET_INDEX	= 8,
	ROWSET_ALL		= ROWSET_FIND | ROWSET_SCROLL | ROWSET_CHANGE | ROWSET_INDEX
};

// Classes -------------------------------------------------------------------

//----------------------------------------------------------------------------
// @class CRowset | Rowset object. Containing class for all interfaces on the Rowset 
// Object
//
class CRowset : public IUnknown //@base public | IUnknown
{
//	Contained interfaces are friends
friend class CImpIColumnsInfo;
friend class CImpIRowset;
friend class CImpIRowsetChange;
friend class CImpIAccessor;
friend class CImpIRowsetInfo;
friend class CImpIConvertType;
friend class CImpIConnectionPointContainer;
friend class CImpIConnectionPoint;
friend class CImpIEnumConnectionPoints;
friend class CImpIEnumConnections;
friend class CImpIRowsetLocate;
friend class CImpIRowsetFind;
friend class CImpIRowsetScroll;
friend class CImpIRowsetIdentity;
friend class CImpIRowsetIndex;

//Command object and ICommand interface are friends as well
friend class CImpICommand;
friend class CCommand;

	START_CLASS(); 

private: //@access private
	//@cmember Builds DBCOLINFO structures
	HRESULT GatherColumnInfo(void);
	//@cmember Creates Helper Classes 
	HRESULT CreateHelperFunctions(void);
	//@cmember Returns the Buffer Pointer for the specified row
	ROWBUFF* GetRowBuff(ULONG iRow);
	//@member
	HRESULT	OnFieldChange
			(
				HROW			hRow,
				ULONG			cColumns,
				ULONG			rgColumns[],
				DBREASON		eReason,
				DBEVENTPHASE	ePhase,
				BOOL			fCantDeny
			);
	//@member
	HRESULT	OnRowChange
			(
				ULONG			cRows,
				const HROW		rghRows[],
				DBREASON		eReason,
				DBEVENTPHASE	ePhase,
				BOOL			fCantDeny
			);
	//@member
	HRESULT	OnRowsetChange
			(
				DBREASON		eReason,
				DBEVENTPHASE	ePhase,
				BOOL			fCantDeny
			);

	bool CheckReason
			(
			DBREASON		eReason,
			DBEVENTPHASE	ePhase
			);

	//@member Returns row handle corresponds to a bookmark
	HROW	Bookmark2Row(const ULONG* pBookmark);
	//@member Returns bookmark of a row handle
	ULONG*	Row2Bookmark(HROW hRow);
	//@member
	ULONG	GetFreeSlot();
	//@member
	void	ReleaseSlot(ULONG iSlot);
	//@member
	BOOL	IsSlotLimitReached();
	// New data from accessor --> to row
	HRESULT ApplyAccessor(HACCESSOR hAccessor, BYTE* pRow, void* pData, ULONG ulMaxBind = ~(ULONG)0);

protected: //Moving operations
	// Skip or move function
	typedef HRESULT (CData::*GoPendingFunction) ( LONG nRows, BOOL bSaveCurrentPosition );
	// Move or skip some rows
	HRESULT GoPending( LONG nMove, GoPendingFunction func );

public: //Moving operations
	//Skip with position correction
	HRESULT Skip(LONG nMove)
		{ return GoPending( nMove, m_pData->Skip ); }
	//Move with position correction
	HRESULT MovePending(LONG nMove)
		{ return GoPending( nMove, m_pData->MovePending );}
	//Move with position correction
	HRESULT GetFetchedData(LONG nPosAfterLastMove, DWORD* pdwBmk);
	//Move to bookmark
	HRESULT MoveToBookmark(void *pBookmark, ULONG cbBookmark);
	//Initialize position
	void InitPosition();

protected:
	//Parent command
	CCommand* m_pCommand;
	//If position is before first row
	bool m_bStartPosition;
	//Previous move direction
	int m_nPreviousMove;

	bool m_bEOF, m_bBOF;

	bool m_bSkipRow;


protected: //@access protected

	//Provider is inside notification process
	bool							m_bNotReentrant;
	//@cmember Session number
	ULONG							m_cNRowset;						
	//@cmember Data Manipulation Class
	CData*							m_pData;
	//@cmember
	bool							m_fBookmarkCol;
	//@cmember Count of Columns in Result Set
	ULONG							m_cCols;
	//@cmember ColumnInfo array
	DBCOLUMNINFO*					m_rgdbcolinfo;     
	//@cmember column offsets in SwstProv's buffer
	ULONG*							m_rgdwDataOffsets;	
	//@cmember array of accessor ptrs
	LPEXTBUFFER     				m_pExtBufferAccessor;
	//@cmember internal buffer structure
	PLSTSLOT        				m_pIBuffer;         
	//@cmember bit array to mark active rows
	LPBITARRAY						m_prowbitsIBuffer;	
	//@cmember size of row in the buffer
	ULONG           				m_cbRowSize;        
	//@cmember points to the first buffered row 
	BYTE*							m_rgbRowData;      
	//@cmember index of the first available rowbuffer
	ULONG           				m_irowMin;          
	//@cmember current # of rows in the buffer
	ULONG           				m_cRows;
	//@cmember position in the resultset
	ULONG           				m_irowFilePos;
	//@cmember Start of the rowset
	ULONG							m_irowLastFilePos;
	//@cmember status word for the entire cursor
	UDWORD          				m_dwStatus;         
	//@cmember RefCount of all outstanding row handles
	ULONG           				m_ulRowRefCount;	
    //@member Utility object to manage properties
    PCUTILPROP                      m_pUtilProp;
	//@cmember Connections pointers for IRowsetNotify
	// IUnknown*					m_Connections[ADVISELIMIT];
	IRowsetNotify*					m_Connections[ADVISELIMIT];
	//@cmember
	ULONG							m_FirstFreeSlot;
	//@cmember
	SLOTREF							m_SlotRef[ACTIVE_ROWS_LIMIT];
	//@cmember
	BOOL							m_LastDirection;

	// Interface and OLE Variables

	//@cmember Reference count
	ULONG							m_cRef;							
	//@cmember Controlling IUnknown
	LPUNKNOWN						m_pUnkOuter;					
	//@cmember Contained IColumnsInfo
	PIMPICOLUMNSINFO				m_pIColumnsInfo;				
	//@cmember Contained IRowset
	PIMPIROWSET						m_pIRowset;
	//@cmember Contained IRowsetChange
	PIMPIROWSETCHANGE				m_pIRowsetChange;				
	//@cmember Contained IAccessor
	PIMPIACCESSOR					m_pIAccessor;
	//@cmember Contained IRowsetInfo
	PIMPIROWSETINFO					m_pIRowsetInfo;
	//@cmember Contained IConvertType
	PIMPICONVERTTYPE				m_pIConvertType;
	//@cmember Contained IConnectionPointcontainer
	PIMPICONNECTIONPOINTCONTAINER	m_pIConnectionPointContainer;
	//@cmember Contained IRowsetLocate
	PIMPIROWSETLOCATE				m_pIRowsetLocate;
	//@cmember Contained IRowsetScroll
	PIMPIROWSETSCROLL				m_pIRowsetScroll;
	//@cmember Contained IRowsetFind
	PIMPIROWSETFIND					m_pIRowsetFind;
	//@cmember Contained IRowsetIdentity
	PIMPIROWSETIDENTITY				m_pIRowsetIdentity;
	//@cmember Contained IConnectionPoint for IRowsetNotify
	PIMPICONNECTIONPOINT			m_pIRowsetNotifyConnectionPoint;
	//@cmember Contained IRowsetIndex
	PIMPIROWSETINDEX				m_pIRowsetIndex;

	// MaxRecords property of ADO
	LONG	m_cMaxRows;
	// Fetched rows (limited by m_cMaxRows)
	LONG	m_cTotRows;

public: //@access public
	//@cmember Constructor
	CRowset(LPUNKNOWN);
	//@cmember Destructor
	~CRowset(void);

	//@member
	bool	IfRowMarkedAsDeleted(HROW hRow);
	//@cmember Mark the row as deleted
	HRESULT MarkAsDeleted(ULONG* pulRowBmkStatus);
	//@cmember Determines if the row has already been deleted
	HRESULT IsDeleted(ULONG ulRowBmkStatus);
	//@cmember Returns slot number from HROW value. Returns UNVALID_SLOT_NUMBER if hRow is bad.
	ULONG GetSlotNumber( HROW hRow);

	//@cmember Intitialization Routine
	BOOL Init(ULONG ulRowset, CData* pData, UINT enumAdvanced, CCommand* pCommand = NULL);
	
	//	Object's base IUnknown
	//@cmember Request an Interface
	STDMETHODIMP			QueryInterface(REFIID, LPVOID *);
	//@cmember Increments the Reference count
	STDMETHODIMP_(ULONG)	AddRef(void);
	//@cmember Decrements the Reference count
	STDMETHODIMP_(ULONG)	Release(void);

	//Back pointer to a session object. 
	PCDBSESSION             m_pSession;  
	//Pointer to creator object
	IUnknown*				m_pCreator;

	FINISH_CLASS(); 

};

typedef CRowset *PCROWSET;


//----------------------------------------------------------------------------
// @class CImpIRowset | Contained IRowset class
//
class CImpIRowset : public IRowset		//@base public | IRowset
{
	START_CLASS(); 

friend class	CImpIRowsetLocate;
friend class	CImpIRowsetScroll;
friend class	CImpIRowsetFind;

private: //@access private
	DEBUGCODE(ULONG m_cRef);										
	CRowset		*m_pObj;											
	LPUNKNOWN	m_pUnkOuter;

public: //@access public
	CImpIRowset( CRowset *pObj, IUnknown *pUnkOuter )					
	{																	
		CLEAR_CONSTRUCT( CImpIRowset );
		
		DEBUGCODE(m_cRef = 0L);											
		m_pObj		= pObj;												
		m_pUnkOuter	= pUnkOuter;										
	}																	
	
	~CImpIRowset()														
	{																	
	}


	STDMETHODIMP_(ULONG)	AddRef(void)									
	{																	
		DEBUGCODE( ++m_cRef );											
		return m_pUnkOuter->AddRef();									
	}
																		
	STDMETHODIMP_(ULONG)	Release(void)									
	{																	
		ASSERT( m_cRef > 0 );											
		DEBUGCODE( --m_cRef );											
		return m_pUnkOuter->Release();									
	}
																		
	STDMETHODIMP			QueryInterface(REFIID riid, LPVOID *ppv)		
	{																	
		return m_pUnkOuter->QueryInterface(riid, ppv);					
	}


	//	IRowset members
	//@cmember GetData Method
	STDMETHODIMP			GetData(HROW, HACCESSOR, void*);
	//@cmember GetNextRows Method
	STDMETHODIMP			GetNextRows(HCHAPTER, LONG, LONG, ULONG*, HROW**);
    //cmember ReleaseRows method
	STDMETHODIMP			ReleaseRows(ULONG, const HROW[], DBROWOPTIONS[], ULONG[], DBROWSTATUS[]);
    //@cmember RestartPosition method
	STDMETHODIMP			RestartPosition(HCHAPTER);
    //@cmember AddRefRows method
    STDMETHODIMP            AddRefRows(ULONG, const HROW[], ULONG[], DBROWSTATUS[]);

protected:
	//@cmember Helper: GetNextRows Method 
	STDMETHODIMP			GetNextRows(bool, HCHAPTER, LONG, LONG, ULONG*, HROW**);

	FINISH_CLASS(); 
};


//----------------------------------------------------------------------------
// @class CImpIRowsetChange | Contained IRowsetChange class
//
class CImpIRowsetChange : public IRowsetChange	//@base public | IRowsetChange
{
	START_CLASS(); 

	private: //@access private
		DEBUGCODE(ULONG m_cRef);										
		CRowset		*m_pObj;											
		LPUNKNOWN	m_pUnkOuter;

	public: //@access public		
		CImpIRowsetChange( CRowset *pObj, IUnknown *pUnkOuter )					
		{																	
			CLEAR_CONSTRUCT( CImpIRowsetChange );
			
			DEBUGCODE(m_cRef = 0L);											
			m_pObj		= pObj;												
			m_pUnkOuter	= pUnkOuter;										
		}																	
	
		~CImpIRowsetChange()														
		{																	
		}

		STDMETHODIMP_(ULONG)	AddRef(void)									
		{																	
			DEBUGCODE( ++m_cRef );											
			return m_pUnkOuter->AddRef();									
		}
																			
		STDMETHODIMP_(ULONG)	Release(void)									
		{																	
			ASSERT( m_cRef > 0 );											
			DEBUGCODE( --m_cRef );											
			return m_pUnkOuter->Release();									
		}
																			
		STDMETHODIMP			QueryInterface(REFIID riid, LPVOID *ppv)		
		{																	
			return m_pUnkOuter->QueryInterface(riid, ppv);					
		}


		//	IRowsetChange members
		//@cmember SetData Method
	    STDMETHODIMP	SetData(HROW, HACCESSOR, void*);
	    STDMETHODIMP	DeleteRows(HCHAPTER, ULONG, const HROW[], DBROWSTATUS[]);
		STDMETHODIMP    InsertRow
					    (
						    HCHAPTER	hReserved,
							HACCESSOR	hAccessor,
							void*		pData,
							HROW*		phRow
						);
		FINISH_CLASS(); 
};


//----------------------------------------------------------------------------
// @class CImpIColumnsInfo | Contained IColumnsInfo class
//
class CImpIColumnsInfo : public IColumnsInfo 		//@base public | IColumnsInfo
{
	START_CLASS(); 

	private: //@access private
		DEBUGCODE(ULONG m_cRef);										

		LPUNKNOWN		m_pUnkOuter;

		//@cmember	pointer to column count
		ULONG*			m_pcCols;
		//@cmember	pointer to bookmark flag
		bool*			m_pbBookmark;
		//@cmember	pointer to column info structures
		DBCOLUMNINFO**	m_ppColInfo;

		//@cmember	pointer to Command (used for returning special results)
		CCommand*		m_pCommand;


	public: //@access public
		CImpIColumnsInfo(ULONG* pcCols, bool* pbBookmark, DBCOLUMNINFO** ppColInfo, CCommand* pCommand, IUnknown *pUnkOuter )					
		{																	
			CLEAR_CONSTRUCT( CImpIColumnsInfo );
			
			DEBUGCODE(m_cRef = 0L);											
			m_pUnkOuter	= pUnkOuter;										

			m_pcCols		= pcCols;
			m_pbBookmark	= pbBookmark;
			m_ppColInfo		= ppColInfo;
			m_pCommand		= pCommand;
		}																	
		
		~CImpIColumnsInfo()														
		{																	
		}


		STDMETHODIMP_(ULONG)	AddRef(void)									
		{																	
			DEBUGCODE( ++m_cRef );											
			return m_pUnkOuter->AddRef();									
		}
																			
		STDMETHODIMP_(ULONG)	Release(void)									
		{																	
			ASSERT( m_cRef > 0 );											
			DEBUGCODE( --m_cRef );											
			return m_pUnkOuter->Release();									
		}
																			
		STDMETHODIMP			QueryInterface(REFIID riid, LPVOID *ppv)		
		{																	
			return m_pUnkOuter->QueryInterface(riid, ppv);					
		}


		//	IColumnsInfo members
		//@cmember GetColumnInfo method
	    STDMETHODIMP			GetColumnInfo(ULONG*, DBCOLUMNINFO**, WCHAR**);
		//@cmember MapColumnIDs
		STDMETHODIMP			MapColumnIDs(ULONG, const DBID[], ULONG[]);

		FINISH_CLASS(); 
};

//----------------------------------------------------------------------------
// @class CImpIAccessor | Contained IAccessor class
//
class CImpIAccessor : public IAccessor 		//@base public | IAccessor
{
	START_CLASS(); 

friend class CImpICommand;

private: //@access private
	DEBUGCODE(ULONG m_cRef);										
	//@cmember Rowset that is used for validating accessor (may be NULL)
	CRowset*	m_pRowset;	
	//@cmember Unknown outer pointer
	LPUNKNOWN	m_pUnkOuter;
	//@cmember array of accessor ptrs
	LPEXTBUFFER m_pExtBufferAccessor;


public: //@access public
	CImpIAccessor(LPEXTBUFFER pExtBufferAccessor, CRowset *pRowset, IUnknown *pUnkOuter )					
	{																	
		CLEAR_CONSTRUCT( CImpIAccessor );

		DEBUGCODE(m_cRef = 0L);											
		
		m_pExtBufferAccessor	= pExtBufferAccessor;
		m_pRowset				= pRowset;
		m_pUnkOuter				= pUnkOuter;										
	}																	

	~CImpIAccessor()														
	{																	
	}

	
	STDMETHODIMP_(ULONG)	AddRef(void)									
	{																	
		DEBUGCODE( ++m_cRef );											
		return m_pUnkOuter->AddRef();									
	}
																		
	STDMETHODIMP_(ULONG)	Release(void)									
	{																	
		ASSERT( m_cRef > 0 );											
		DEBUGCODE( --m_cRef );											
		return m_pUnkOuter->Release();									
	}
																		
	STDMETHODIMP			QueryInterface(REFIID riid, LPVOID *ppv)		
	{																	
		return m_pUnkOuter->QueryInterface(riid, ppv);					
	}


	//@cmember Increment Reference count on accessor
	STDMETHODIMP			AddRefAccessor(HACCESSOR hAccessor, ULONG* pcRefCounts);
	//@cmember CreateAccessor Method
	STDMETHODIMP			CreateAccessor(DBACCESSORFLAGS, ULONG, const DBBINDING[], ULONG, HACCESSOR*, DBBINDSTATUS[]);
	//@cmember GetBindings Method
	STDMETHODIMP			GetBindings(HACCESSOR, DBACCESSORFLAGS*, ULONG*, DBBINDING**);
	//@cmember ReleaseAccessor Method
	STDMETHODIMP			ReleaseAccessor(HACCESSOR, ULONG*);

	FINISH_CLASS(); 
};

//----------------------------------------------------------------------------
// @class CImpIRowsetInfo | Contained IRowsetInfo class
//
class CImpIRowsetInfo : public IRowsetInfo 		//@base public | IRowsetInfo
{
	START_CLASS(); 

	private: //@access private
		DEBUGCODE(ULONG m_cRef);										
		CRowset		*m_pObj;											
		LPUNKNOWN	m_pUnkOuter;

	public: //@access public
 		DEFINE_DEFAULT_IUNKNOWN_CTOR_DTOR(CRowset, CImpIRowsetInfo);
		CImpIRowsetInfo( CRowset *pObj, IUnknown *pUnkOuter )					
		{																	
			CLEAR_CONSTRUCT( CImpIRowsetInfo );
			
			DEBUGCODE(m_cRef = 0L);											
			m_pObj		= pObj;												
			m_pUnkOuter	= pUnkOuter;										
		}																	
	
		~CImpIRowsetInfo()														
		{																	
		}


		STDMETHODIMP_(ULONG)	AddRef(void)									
		{																	
			DEBUGCODE( ++m_cRef );											
			return m_pUnkOuter->AddRef();									
		}
																			
		STDMETHODIMP_(ULONG)	Release(void)									
		{																	
			ASSERT( m_cRef > 0 );											
			DEBUGCODE( --m_cRef );											
			return m_pUnkOuter->Release();									
		}
																			
		STDMETHODIMP			QueryInterface(REFIID riid, LPVOID *ppv)		
		{																	
			return m_pUnkOuter->QueryInterface(riid, ppv);					
		}


		//@cmember GetReferencedRowset
		STDMETHODIMP			GetReferencedRowset
			(
				ULONG		iOrdinal, 
				REFIID		rrid,
				IUnknown**	ppReferencedRowset
			);

		//@cmember GetProperties
		STDMETHODIMP			GetProperties
		    (
			    const ULONG			cPropertySets,
			    const DBPROPIDSET	rgPropertySets[],
			    ULONG*              pcProperties,
			    DBPROPSET**			prgProperties
		    );

		//@cmember GetSpecification Method
		STDMETHODIMP			GetSpecification(REFIID, IUnknown**);
	
		FINISH_CLASS(); 
};


//----------------------------------------------------------------------------
// @class CImpIConvertType | Contained IConvertType class
//
class CImpIConvertType : public IConvertType 		//@base public | IConvertType
{
		START_CLASS(); 

	private: //@access private
		DEBUGCODE(ULONG m_cRef);											
		LPUNKNOWN	m_pUnkOuter;

	public: //@access public
		CImpIConvertType(IUnknown *pUnkOuter)					
		{																	
			CLEAR_CONSTRUCT( CImpIConvertType );
			
			DEBUGCODE(m_cRef = 0L);											
			m_pUnkOuter	= pUnkOuter;										
		}																	
	
		~CImpIConvertType()														
		{																	
		}

		
		STDMETHODIMP_(ULONG)	AddRef(void)									
		{																	
			DEBUGCODE( ++m_cRef );											
			return m_pUnkOuter->AddRef();									
		}
																			
		STDMETHODIMP_(ULONG)	Release(void)									
		{																	
			ASSERT( m_cRef > 0 );											
			DEBUGCODE( --m_cRef );											
			return m_pUnkOuter->Release();									
		}
																			
		STDMETHODIMP			QueryInterface(REFIID riid, LPVOID *ppv)		
		{																	
			return m_pUnkOuter->QueryInterface(riid, ppv);					
		}


		//@cmember CanConvert
		STDMETHODIMP CImpIConvertType::CanConvert
			(
			DBTYPE			wFromType,		//@parm IN | src type
			DBTYPE			wToType,		//@parm IN | dst type
			DBCONVERTFLAGS	dwConvertFlags	//@parm IN | conversion flags
			);
	
		FINISH_CLASS(); 
};

//----------------------------------------------------------------------------
// @class CImpIConnectionPointContainer | Contained IConnectionPointContainer class
//
class	CImpIConnectionPointContainer : public IConnectionPointContainer  //@base public | IConnectionPointContainer
{
	START_CLASS(); 

	friend class	CRowset;
	friend class	CImpIEnumConnectionPoints;
	friend class	CImpIConnectionPoint;

	private: //@access private
		DEBUGCODE(ULONG m_cRef);										
		CRowset		*m_pObj;											
		LPUNKNOWN	m_pUnkOuter;
	
	public: //@access public
		CImpIConnectionPointContainer( CRowset *pObj, IUnknown *pUnkOuter )					
		{																	
			CLEAR_CONSTRUCT( CImpIConnectionPointContainer );
			
			DEBUGCODE(m_cRef = 0L);											
			m_pObj		= pObj;												
			m_pUnkOuter	= pUnkOuter;										
		}																	
	
		~CImpIConnectionPointContainer()														
		{																	
		}


		STDMETHODIMP_(ULONG)	AddRef(void)									
		{																	
			DEBUGCODE( ++m_cRef );											
			return m_pUnkOuter->AddRef();									
		}
																			
		STDMETHODIMP_(ULONG)	Release(void)									
		{																	
			ASSERT( m_cRef > 0 );											
			DEBUGCODE( --m_cRef );											
			return m_pUnkOuter->Release();									
		}
																			
		STDMETHODIMP			QueryInterface(REFIID riid, LPVOID *ppv)		
		{																	
			return m_pUnkOuter->QueryInterface(riid, ppv);					
		}


		//@cmember
		// Returns a pointer to the IConnectionPoint interface of a connection point
		// for a specified IID, if that IID describes a supported outgoing interface.
		STDMETHODIMP FindConnectionPoint( REFIID riid, IConnectionPoint **ppCP );

		//@cmember
		// Creates an enumerator object to iterate through all the connection
		// points supported in the connectable object, one connection point per outgoing IID.
		STDMETHODIMP EnumConnectionPoints( IEnumConnectionPoints **ppEnum );

		FINISH_CLASS(); 
};

//----------------------------------------------------------------------------
// @class CImpIEnumConnectionPoints | Contained IEnumConnectionPoints class
//
class	CImpIEnumConnectionPoints : public IEnumConnectionPoints	//@base public | IEnumConnectionPoints
{
	START_CLASS(); 

	friend	class	CImpIConnectionPointContainer;

	private: //@access private

		//@cmember
		ULONG							m_cRef;
		//@cmember
		PIMPICONNECTIONPOINT			m_pCurrentCP;
		//@cmember
		PIMPICONNECTIONPOINTCONTAINER	m_pParentContainer;
		//@cmember
		CRowset*	m_pObj;

	public: //@access public

		//@cmember Constructor for this class
		CImpIEnumConnectionPoints(CRowset* pObj)
		{
			CLEAR_CONSTRUCT( CImpIEnumConnectionPoints );

			//m_cRef(0), m_pCurrentCP(0), 
			m_pObj = pObj;
		}

// IUnknown interfaces

		//@cmember
		STDMETHODIMP_(ULONG)	AddRef(void)
		{
			++m_cRef; 
			return S_OK;
		}

		//@cmember
		STDMETHODIMP_(ULONG)	Release(void)
		{									
			ASSERT( m_cRef > 0 );

			if (!--m_cRef)
				delete this;

			return S_OK;
		}

		//@cmember
		STDMETHODIMP			QueryInterface(REFIID riid, LPVOID *ppv)
		{
			if (riid != IID_IUnknown && riid != IID_IEnumConnectionPoints)
				return E_NOINTERFACE;

			*ppv = this;
			return S_OK;
		}

		//@cmember
		STDMETHODIMP			Next(ULONG cConnections,IConnectionPoint **rgpcn,ULONG *pcFetched);
		//@cmember
		STDMETHODIMP			Skip(ULONG cConnections);
		//@cmember
		STDMETHODIMP			Reset(void);
		//@cmember
		STDMETHODIMP			Clone(IEnumConnectionPoints **ppEnum);
	
		FINISH_CLASS(); 
};

//----------------------------------------------------------------------------
// @class CImpIConnectionPoint | Contained IConnectionPoint class
//
class	CImpIConnectionPoint : public IConnectionPoint		//@base public | IConnectionPoint
{
	START_CLASS(); 

	friend class	CImpIConnectionPointContainer;
	friend class	CImpIEnumConnectionPoints;

	private: //@access private
		DEBUGCODE(ULONG m_cRef);										
		//@cmember
		CRowset*	m_pObj;
		//@cmember
		IID			m_iid;
		//@cmember
		PIMPICONNECTIONPOINTCONTAINER	m_pParentContainer;

	public: //@access public

		CImpIConnectionPoint
			(
			CRowset* pObj, 
			REFIID riid,
			PIMPICONNECTIONPOINTCONTAINER pParentContainer
			)
		{
			CLEAR_CONSTRUCT( CImpIConnectionPoint );
			
			DEBUGCODE(m_cRef = 0L);											
			m_iid		= riid;
			m_pObj		= pObj;												
			m_pParentContainer = pParentContainer;
		}

		// IUnknown interfaces
		STDMETHODIMP_(ULONG)	AddRef(void)
		{
			DEBUGCODE( ++m_cRef );											
			return m_pParentContainer->AddRef();									
		}

		STDMETHODIMP_(ULONG)	Release(void)
		{									
			ASSERT( m_cRef > 0 );
			DEBUGCODE( --m_cRef );											
			return m_pParentContainer->Release();									
		}

		STDMETHODIMP			QueryInterface(REFIID riid, LPVOID *ppv)
		{
			if (ppv == NULL)
				return E_INVALIDARG;

		    //  Place NULL in *ppv in case of failure
		    *ppv = NULL;

			if (riid != IID_IUnknown && riid != IID_IConnectionPoint)
				return E_NOINTERFACE;
			
			// return interface
			*ppv = this;

			//  If we're going to return an interface, AddRef it first
			if (*ppv)
			{
				((LPUNKNOWN) *ppv)->AddRef();
				return S_OK;
			}
			else
				return E_NOINTERFACE;

		return S_OK;
		}

		//@cmember
		// Retrieves the IConnectionPointContainer interface pointer for the parent
		// connectable object.
		STDMETHODIMP GetConnectionPointContainer(IConnectionPointContainer **ppCPC);
		
		//@cmember
		// Returns the IID of the outgoing interface managed by this connection point.
		STDMETHODIMP GetConnectionInterface(IID *pIID);

		//@cmember
		// Establishes a connection between the connection point object and the client's sink.
		STDMETHODIMP Advise( IUnknown *pUnk, DWORD *pdwCookie );

		//@cmember
		// Terminates an advisory connection previously established through
		// IConnectionPoint::Advise. The dwCookie parameter identifies the connection to
		// terminate.
		STDMETHODIMP Unadvise( DWORD dwCookie );

		//@cmember
		// Returns an object to enumerate the current advisory connections for this connection point.
		STDMETHODIMP EnumConnections( IEnumConnections **ppEnum );

		FINISH_CLASS(); 
};

//----------------------------------------------------------------------------
// @class CImpIEnumConnections | Contained IEnumConnections class
//
class	CImpIEnumConnections : public IEnumConnections	//@base public | IEnumConnections
{
	START_CLASS(); 

	friend class	CImpIConnectionPoint;

	private: //@access private

		//@cmember
		DWORD		m_CurrentConnection;
		//@cmember
		ULONG		m_cRef;
		//@cmember
		CRowset*	m_pObj;

	public: //@access public

		//@cmember Constructor for this class
		CImpIEnumConnections(CRowset* pObj)
		{
			CLEAR_CONSTRUCT( CImpIEnumConnections );
			//m_cRef(0), m_CurrentConnection(0) 
			m_pObj = pObj; 
		}

		// IUnknown interfaces

		//@cmember
		STDMETHODIMP_(ULONG)	AddRef(void)
		{
			++m_cRef; 
			return S_OK;
		}

		//@cmember
		STDMETHODIMP_(ULONG)	Release(void)
		{									
			ASSERT( m_cRef > 0 );

			if (!--m_cRef)
				delete this;

			return S_OK;
		}

		//@cmember
		STDMETHODIMP			QueryInterface(REFIID riid, LPVOID *ppv)
		{
			if (riid != IID_IUnknown && riid != IID_IEnumConnections)
				return E_NOINTERFACE;

			if (ppv == NULL)
				return E_INVALIDARG;

			*ppv = this;
			return S_OK;
		}

		//@cmember
		STDMETHODIMP CImpIEnumConnections::Next(ULONG cConnections,CONNECTDATA *rgcd,ULONG *pcFetched);
		//@cmember
		STDMETHODIMP CImpIEnumConnections::Skip(ULONG cConnections);
		//@cmember
		STDMETHODIMP CImpIEnumConnections::Reset(void);
		//@cmember
		STDMETHODIMP CImpIEnumConnections::Clone(IEnumConnections **ppEnum);

		FINISH_CLASS(); 
};


//----------------------------------------------------------------------------
// @class CImpIRowsetLocate | Contained IRowsetLocate class
//
class	CImpIRowsetLocate	:	public IRowsetLocate
{
	START_CLASS(); 

	private: //@access private
		DEBUGCODE(ULONG m_cRef);										
		CRowset		*m_pObj;											
		LPUNKNOWN	m_pUnkOuter;

	public: //@access public
		CImpIRowsetLocate( CRowset *pObj, IUnknown *pUnkOuter )					
		{																	
			CLEAR_CONSTRUCT( CImpIRowsetLocate );

			DEBUGCODE(m_cRef = 0L);											
			m_pObj		= pObj;												
			m_pUnkOuter	= pUnkOuter;										
		}																	
	
		~CImpIRowsetLocate()														
		{																	
		}

		
		STDMETHODIMP_(ULONG)	AddRef(void)									
		{																	
			DEBUGCODE( ++m_cRef );											
			return m_pUnkOuter->AddRef();									
		}
																			
		STDMETHODIMP_(ULONG)	Release(void)									
		{																	
			ASSERT( m_cRef > 0 );											
			DEBUGCODE( --m_cRef );											
			return m_pUnkOuter->Release();									
		}
																			
		STDMETHODIMP			QueryInterface(REFIID riid, LPVOID *ppv)		
		{																	
			return m_pUnkOuter->QueryInterface(riid, ppv);					
		}


	//@cmember Compares two bookmarks.
	STDMETHODIMP	Compare (
						HCHAPTER	hReserved, 
						ULONG		cbBookmark1, 
					const BYTE*		pBookmark1,
						ULONG		cbBookmark2,
					const BYTE*		pBookmark2,
						DBCOMPARE*	pComparison);

	//@cmember Fetches rows starting with the row specified by an offset from a bookmark.
	STDMETHODIMP	GetRowsAt (
						HWATCHREGION	hReserved1,
						HCHAPTER		hReserved2,
						ULONG			cbBookmark,
					const BYTE*			pBookmark,
						LONG			lRowsOffset,
						LONG			cRows,
						ULONG*			pcRowsObtained,
						HROW**			prghRows);

	//@cmember Fetches the rows that match the specified bookmarks.
	STDMETHODIMP	GetRowsByBookmark (
						HCHAPTER		hChapter,
						ULONG			cRows,
					const ULONG			rgcbBookmarks[],
					const BYTE*			rgpBookmarks[],
						HROW			rghRows[],
						DBROWSTATUS		rgRowStatus[]);

	//@cmember Returns hash values for the specified bookmarks.
	STDMETHODIMP	Hash (
						HCHAPTER		hChapter,
						ULONG			cBookmarks,
					const ULONG			rgcbBookmarks[],
					const BYTE*			rgpBookmarks[],
						DWORD			rgHashedValues[],
						DBROWSTATUS		rgBookmarkStatus[]);

	//@cmember GetData Method
	STDMETHODIMP			GetData(HROW, HACCESSOR, void*);
	//@cmember GetNextRows Method
	STDMETHODIMP			GetNextRows(HCHAPTER, LONG, LONG, ULONG*, HROW**);
    //cmember ReleaseRows method
	STDMETHODIMP			ReleaseRows(ULONG, const HROW[], DBROWOPTIONS[], ULONG[], DBROWSTATUS[]);
    //@cmember RestartPosition method
	STDMETHODIMP			RestartPosition(HCHAPTER);
    //@cmember AddRefRows method
    STDMETHODIMP            AddRefRows(ULONG, const HROW[], ULONG[], DBROWSTATUS[]);

	FINISH_CLASS(); 
};


//----------------------------------------------------------------------------
// @class CImpIRowsetScroll | Contained IRowsetScroll class
//
class CImpIRowsetScroll : public IRowsetScroll		//@base public | IRowsetScroll
{
	START_CLASS(); 

	private: //@access private
		DEBUGCODE(ULONG m_cRef);									
		CRowset		*m_pObj;										
		LPUNKNOWN	m_pUnkOuter;

	public: //@access public
		CImpIRowsetScroll( CRowset *pObj, IUnknown *pUnkOuter )					
		{																	
			CLEAR_CONSTRUCT( CImpIRowsetScroll );
			
			DEBUGCODE(m_cRef = 0L);											
			m_pObj		= pObj;												
			m_pUnkOuter	= pUnkOuter;										
		}																	
	
		~CImpIRowsetScroll()														
		{																	
		}


		STDMETHODIMP_(ULONG)	AddRef(void)									
		{																	
			DEBUGCODE( ++m_cRef );											
			return m_pUnkOuter->AddRef();									
		}
																			
		STDMETHODIMP_(ULONG)	Release(void)									
		{																	
			ASSERT( m_cRef > 0 );											
			DEBUGCODE( --m_cRef );											
			return m_pUnkOuter->Release();									
		}
																			
		STDMETHODIMP			QueryInterface(REFIID riid, LPVOID *ppv)		
		{																	
			return m_pUnkOuter->QueryInterface(riid, ppv);					
		}



	//	IRowsetScroll members inherited from IRowset
	//@cmember GetData Method
	STDMETHODIMP			GetData(HROW, HACCESSOR, void*);
	//@cmember GetNextRows Method
	STDMETHODIMP			GetNextRows(HCHAPTER, LONG, LONG, ULONG*, HROW**);
    //cmember ReleaseRows method
	STDMETHODIMP			ReleaseRows(ULONG, const HROW[], DBROWOPTIONS[], ULONG[], DBROWSTATUS[]);
    //@cmember RestartPosition method
	STDMETHODIMP			RestartPosition(HCHAPTER);
    //@cmember AddRefRows method
    STDMETHODIMP            AddRefRows(ULONG, const HROW[], ULONG[], DBROWSTATUS[]);

	//	IRowsetScroll members inherited from IRowsetLocate
	//@cmember Compares two bookmarks.
	STDMETHODIMP	Compare (
						HCHAPTER		hReserved, 
						ULONG			cbBookmark1, 
						const BYTE*		pBookmark1,
						ULONG			cbBookmark2,
						const BYTE*		pBookmark2,
						DBCOMPARE*		pComparison);

	//@cmember Fetches rows starting with the row specified by an offset from a bookmark.
	STDMETHODIMP	GetRowsAt (
						HWATCHREGION	hReserved1,
						HCHAPTER		hReserved2,
						ULONG			cbBookmark,
						const BYTE*		pBookmark,
						LONG			lRowsOffset,
						LONG			cRows,
						ULONG*			pcRowsObtained,
						HROW**			prghRows);

	//@cmember Fetches the rows that match the specified bookmarks.
	STDMETHODIMP	GetRowsByBookmark (
						HCHAPTER		hChapter,
						ULONG			cRows,
						const ULONG		rgcbBookmarks[],
						const BYTE*		rgpBookmarks[],
						HROW			rghRows[],
						DBROWSTATUS		rgRowStatus[]);

	//@cmember Returns hash values for the specified bookmarks.
	STDMETHODIMP	Hash (
						HCHAPTER		hChapter,
						ULONG			cBookmarks,
						const ULONG		rgcbBookmarks[],
						const BYTE*		rgpBookmarks[],
						DWORD			rgHashedValues[],
						DBROWSTATUS		rgBookmarkStatus[]);

		//	"Native" IRowsetScroll members
		//@cmember GetApproximatePosition Method 
		// Gets the approximate position of a row corresponding to a specified bookmark
		STDMETHODIMP GetApproximatePosition (
						HCHAPTER        hChapter,
						ULONG           cbBookmark,
						const BYTE *    pBookmark,
						ULONG *         pulPosition,
						ULONG *         pcRows);

		//@cmember GetRowsAtRatio Method 
		// Fetches rows starting from a fractional position in the rowset
		STDMETHODIMP GetRowsAtRatio (
						HWATCHREGION    hReserved1,
						HCHAPTER        hChapter,
						ULONG           ulNumerator,
						ULONG           ulDenominator,
						LONG            cRows,
						ULONG *         pcRowsObtained,
						HROW **         prghRows);

		FINISH_CLASS(); 
};


//----------------------------------------------------------------------------
// @class CImpIRowsetFind | Contained IRowsetFind class
//
class	CImpIRowsetFind : public IRowsetFind
{
	START_CLASS(); 

	private: //@access private
		DEBUGCODE(ULONG m_cRef);										
		CRowset		*m_pObj;											
		LPUNKNOWN	m_pUnkOuter;

	public: //@access public
		CImpIRowsetFind( CRowset *pObj, IUnknown *pUnkOuter )					
		{																	
			CLEAR_CONSTRUCT( CImpIRowsetFind );
			
			DEBUGCODE(m_cRef = 0L);											
			m_pObj		= pObj;												
			m_pUnkOuter	= pUnkOuter;										
		}																	
	
		~CImpIRowsetFind()														
		{																	
		}

		
		STDMETHODIMP_(ULONG)	AddRef(void)									
		{																	
			DEBUGCODE( ++m_cRef );											
			return m_pUnkOuter->AddRef();									
		}
																			
		STDMETHODIMP_(ULONG)	Release(void)									
		{																	
			ASSERT( m_cRef > 0 );											
			DEBUGCODE( --m_cRef );											
			return m_pUnkOuter->Release();									
		}
																			
		STDMETHODIMP			QueryInterface(REFIID riid, LPVOID *ppv)		
		{																	
			return m_pUnkOuter->QueryInterface(riid, ppv);					
		}


	// @cmember	Begin at the specified bookmark and find the next row
	// matching the specified value.
	STDMETHODIMP FindNextRow (
		    HCHAPTER		hReserved,
			HACCESSOR		hAccessor,
			void *			pFindValue,
			DBCOMPAREOP		CompareOp,
			ULONG			cbBookmark,
			const BYTE *	pBookmark,
			LONG			IRowsOffset,
			LONG			cRows,
			ULONG *			pcRowsObtained,
			HROW **			prghRows);

	FINISH_CLASS(); 
};

//----------------------------------------------------------------------------
// @class CImpIRowsetIdentity| Contained IRowsetIdentity class
//
class	CImpIRowsetIdentity : public IRowsetIdentity
{
	START_CLASS(); 

	private: //@access private
		DEBUGCODE(ULONG m_cRef);										
		CRowset		*m_pObj;											
		LPUNKNOWN	m_pUnkOuter;

	public: //@access public
		CImpIRowsetIdentity( CRowset *pObj, IUnknown *pUnkOuter )					
		{																	
			CLEAR_CONSTRUCT( CImpIRowsetIdentity );

			DEBUGCODE(m_cRef = 0L);											
			m_pObj		= pObj;												
			m_pUnkOuter	= pUnkOuter;										
		}																	
	
		~CImpIRowsetIdentity()														
		{																	
		}

		
		STDMETHODIMP_(ULONG)	AddRef(void)									
		{																	
			DEBUGCODE( ++m_cRef );											
			return m_pUnkOuter->AddRef();									
		}
																			
		STDMETHODIMP_(ULONG)	Release(void)									
		{																	
			ASSERT( m_cRef > 0 );											
			DEBUGCODE( --m_cRef );											
			return m_pUnkOuter->Release();									
		}
																			
		STDMETHODIMP			QueryInterface(REFIID riid, LPVOID *ppv)		
		{																	
			return m_pUnkOuter->QueryInterface(riid, ppv);					
		}


		//@cmember
		STDMETHODIMP IsSameRow (HROW hThisRow, HROW hThatRow);
	
		FINISH_CLASS(); 
};

//----------------------------------------------------------------------------
// @class CImpIRowsetIndex | Contained IRowsetIndex class
//
class CImpIRowsetIndex : public IRowsetIndex 		//@base public | IRowsetIndex
{
	START_CLASS(); 

	private: //@access private
		DEBUGCODE(ULONG m_cRef);										
		CRowset		*m_pObj;											
		LPUNKNOWN	m_pUnkOuter;

	public: //@access public
 		DEFINE_DEFAULT_IUNKNOWN_CTOR_DTOR(CRowset, CImpIRowsetIndex);
		CImpIRowsetIndex( CRowset *pObj, IUnknown *pUnkOuter )					
		{																	
			CLEAR_CONSTRUCT( CImpIRowsetIndex );
			
			DEBUGCODE(m_cRef = 0L);											
			m_pObj		= pObj;												
			m_pUnkOuter	= pUnkOuter;										
		}																	
	
		~CImpIRowsetIndex()														
		{																	
			FreeCurrentIndexInfos();
		}


		STDMETHODIMP_(ULONG)	AddRef(void)									
		{																	
			DEBUGCODE( ++m_cRef );											
			return m_pUnkOuter->AddRef();									
		}
																			
		STDMETHODIMP_(ULONG)	Release(void)									
		{																	
			ASSERT( m_cRef > 0 );											
			DEBUGCODE( --m_cRef );											
			return m_pUnkOuter->Release();									
		}
																			
		STDMETHODIMP			QueryInterface(REFIID riid, LPVOID *ppv)		
		{																	
			return m_pUnkOuter->QueryInterface(riid, ppv);					
		}

// Interface methods
        STDMETHODIMP GetIndexInfo( 
            ULONG			*pcKeyColumns,
            DBINDEXCOLUMNDESC **prgIndexColumnDesc,
            ULONG			*pcIndexProperties,
            DBPROPSET		**prgIndexProperties);
        
        STDMETHODIMP Seek( 
            HACCESSOR hAccessor,
            ULONG cKeyValues,
            void *pData,
            DBSEEK dwSeekOptions);
        
        STDMETHODIMP SetRange( 
            HACCESSOR hAccessor,
            ULONG cStartKeyColumns,
            void *pStartData,
            ULONG cEndKeyColumns,
            void *pEndData,
            DBRANGE dwRangeOptions);

	// Helper methods
	protected:
		// Free descriptions array
		void FreeCurrentIndexInfos();
		// Allocate descriptions array
		BOOL AllocateCurrentIndexInfos();
		// Get info #
		DBINDEXCOLUMNDESC* GetCurrentIndexInfos( int iInfoNumber = 0 );
		// Get index #
		MT_WORD GetCurrentIndexNumber();
		// Get number of index parts
		MT_WORD GetCurrentIndexParts();
		// Get pointer to index segment #
		SWSTINDEXtag* GetCurrentIndexPtr( int iIndexSegment );
		// Compare relative field ordinal in a table and field ID
		bool CompareRelAbs( ULONG iOrdinal, MT_WORD wID );
		// Get  object. No other CData objects more are available for IRowsetIndex
		CBtrieve* GetBtrieve();

	// Helper data members
	protected:		
		// Advanced index segments descriptions
		DBINDEXCOLUMNDESC* m_pIndexColDesc;
		// Index segments descriptions
		DBID*			   m_pColId;
	
		FINISH_CLASS(); 
};


#endif


