/*
 * Copyright (c)  2000
 * SWsoft  company
 *
 * This material is provided "as is", with absolutely no warranty expressed
 * or implied. Any use is at your own risk.
 *
 * Permission to use or copy this software for any purpose is hereby granted 
 * without fee, provided the above notices are retained on all copies.
 * Permission to modify the code and to distribute modified code is granted,
 * provided the above notices are retained, and a notice that the code was
 * modified is included with the above copyright notice.
 *
 */

//--------------------------------------------------------------------
// MySQL OLE DB Provider 
// Functionality: minimum
// Release: 0.1
//
// @doc
//
// @module IROWSET.CPP | IRowset interface implementation
//
//

// Includes ------------------------------------------------------------------

#include "hfiles.h"
#include "headers.h"

// Code ----------------------------------------------------------------------

//  IRowset specific interface methods

// CImpIRowset::GetData --------------------------------------------------
//
// @mfunc Retrieves data from the rowset's cache
//
// @rdesc HRESULT
//      @flag S_OK                   | Method Succeeded
//      @flag DB_S_ERRORSOCCURED     | Could not coerce a column value
//      @flag DB_E_BADACCESSORHANDLE | Invalid Accessor given
//      @flag DB_E_BADROWHANDLE      | Invalid row handle given
//      @flag E_INVALIDARG           | pData was NULL
//      @flag OTHER                  | Other HRESULTs returned by called functions
//
STDMETHODIMP CImpIRowset::GetData
    (
    HROW        hRow,       //@parm IN | Row Handle
    HACCESSOR   hAccessor,  //@parm IN | Accessor to use
    void       *pData       //@parm OUT | Pointer to buffer where data should go.
    )
{
	INTERFACE_METHOD_START("IRowset::GetData");

    ULONG       ulErrorCount;
    PACCESSOR   pAccessor;
    ULONG       icol, ibind;
    ROWBUFF     *pRowBuff;
    COLUMNDATA  *pColumnData;
    DBBINDING   *pBinding;
    ULONG       cBindings;
    DBTYPE      dwSrcType;
    DBTYPE      dwDstType;
    void        *pSrc;
    void        *pDst;
    ULONG       ulSrcLength;
    ULONG       *pulDstLength;
    ULONG       ulDstMaxLength;
    DWORD       dwSrcStatus;
    DWORD       *pdwDstStatus;
    DWORD       dwPart;
    HRESULT     hr;

    // Coerce data for row 'hRow', according to hAccessor.
    // Put in location 'pData'.  Offsets and types are in hAccessor's bindings.
    //
    // Return S_OK if all lossless conversions,
    // return DB_S_ERRORSOCCURED if lossy conversion (truncation, rounding, etc.)
    // Return E_FAIL, etc., if horrible errors.

    // GetItemOfExtBuffer is basically operator[].
    // It takes an index (or handle) (referenced from 1...n),
    // and a ptr for where to write the data.
    //
    // It holds ptrs to a variable-length ACCESSOR struct.
    // So we get the ACCESSOR ptr for the client's accessor handle.

   
	assert( m_pObj->m_pExtBufferAccessor );
    hr = m_pObj->m_pExtBufferAccessor->GetItemOfExtBuffer((ULONG) hAccessor, &pAccessor );
    if (FAILED( hr ) || pAccessor == NULL)
        return DB_E_BADACCESSORHANDLE;

    assert( pAccessor );
    cBindings = pAccessor->bNullAccessor ? 0 : pAccessor->cBindings; // Return no data if it was null accessor
    pBinding  = pAccessor->rgBindings;

    // IsSlotSet returns S_OK    if row is marked.
    //                   S_FALSE if row is not marked.
    // The "mark" means that there is data present in the row.
    // Rows are [1...n], slot marks are [0...n-1].
    //if (m_pObj->m_prowbitsIBuffer->IsSlotSet((ULONG) hRow ) != S_OK)
	if (m_pObj->m_prowbitsIBuffer->IsSlotSet( m_pObj->GetSlotNumber( hRow ) ) != S_OK)
        return DB_E_BADROWHANDLE;

    // Ensure a place to put data.
    if (pData == NULL && !pAccessor->bNullAccessor )
        return E_INVALIDARG;

    pRowBuff = m_pObj->GetRowBuff((ULONG) hRow );

    // Internal error for a 0 reference count on this row,
    // since we depend on the slot-set stuff.
	#ifdef _DEBUG
		if(!(pRowBuff->ulRefCount))
			TRACE("%d", pRowBuff->dwBmk);
		TRACE( "  Bindings total: %d", (int)cBindings );
	#endif

	ulErrorCount = 0;
    for (ibind = 0; ibind < cBindings; ibind++)
        {
        icol = pBinding[ibind].iOrdinal;
        pColumnData = (COLUMNDATA*) (((BYTE*)pRowBuff) + m_pObj->m_rgdwDataOffsets[icol]);
        dwSrcType      = m_pObj->m_rgdbcolinfo[icol].wType;
        pSrc           = &(pColumnData->bData);

        ulSrcLength    = pColumnData->dwLength;
        dwSrcStatus    = pColumnData->dwStatus;
        ulDstMaxLength = pBinding[ibind].cbMaxLen;
        dwDstType      = pBinding[ibind].wType;
        dwPart         = pBinding[ibind].dwPart;

        pDst           = dwPart & DBPART_VALUE ? ((BYTE*) pData + pBinding[ibind].obValue) : NULL;
        pulDstLength   = dwPart & DBPART_LENGTH ? (ULONG *) ((BYTE*) pData + pBinding[ibind].obLength) : NULL;
        pdwDstStatus   = dwPart & DBPART_STATUS ? (ULONG *) ((BYTE*) pData + pBinding[ibind].obStatus) : NULL;

#ifdef _DEBUG	
		char szBuff[ 1024 ]; *szBuff = 0;
		DWORD dwDstStatus, dwDstLength = 0;

		g_pIDataConvert->DataConvert(
                dwSrcType,
                DBTYPE_STR,
                ulSrcLength,
                &dwDstLength,
                pSrc,
                szBuff,
                1024,
                dwSrcStatus,
                &dwDstStatus,
                0,	// bPrecision for conversion to DBNUMERIC
				0,	// bScale for conversion to DBNUMERIC
				DBDATACONVERT_DEFAULT);
		TRACE( "  Column %d, binding %d = [%s]",  (int)icol, (int)ibind, szBuff );
#endif
        
		hr = g_pIDataConvert->DataConvert(
                dwSrcType,
                dwDstType,
                ulSrcLength,
                pulDstLength,
                pSrc,
                pDst,
                ulDstMaxLength,
                dwSrcStatus,
                pdwDstStatus,
                0,	// bPrecision for conversion to DBNUMERIC
				0,	// bScale for conversion to DBNUMERIC
				DBDATACONVERT_DEFAULT);


        if (FAILED( hr ))
		{
            TRACE4( "  Failed to convert binding %ld from %ld to %ld", ibind, dwSrcType, dwDstType );
			return hr;  // fatal error
		}
        if (hr != S_OK)
            ulErrorCount++; // can't coerce
        }

    // We report any lossy conversions with a special status.
    // Note that DB_S_ERRORSOCCURED is a success, rather than failure.
	
	return ulErrorCount ? DB_S_ERRORSOCCURRED : S_OK;

	INTERFACE_METHOD_END();
}

// CImpIRowset::GetNextRows --------------------------------------------------
//
// @mfunc Fetches rows in a sequential style, remembering the previous position
//
//	Real OLEDB method

STDMETHODIMP CImpIRowset::GetNextRows
    (
    HCHAPTER	hReserved,			//@parm IN	| Reserved for future use. Ingored.
    LONG		cRowsToSkip,		//@parm IN	| Rows to skip before reading
    LONG		cRows,				//@parm IN	| Number of rows to fetch
    ULONG		*pcRowsObtained,	//@parm OUT | Number of rows obtained
    HROW		**prghRows			//@parm OUT | Array of Hrows obtained
    )
{
	return GetNextRows(
					true,			//Send all  notifications
					hReserved,
					cRowsToSkip,
					cRows,
					pcRowsObtained,
					prghRows);
}


// CImpIRowset::GetNextRows --------------------------------------------------
//
// @mfunc Fetches rows in a sequential style, remembering the previous position
//
// Internal helper function
//
// @rdesc HRESULT
//      @flag S_OK                      | Method Succeeded
//      @flag DB_S_ENDOFROWSET          | Reached end of rowset
//      @flag DB_E_BADSTARTPOSITION     | Reached end of rowset
//      @flag DB_E_CANTFETCHBACKWARDS   | cRows was negative and we can't fetch backwards
//      @flag DB_E_ROWSNOTRELEASED      | Must release all HROWs before calling GetNextRows
//      @flag E_FAIL                    | Provider-specific error
//      @flag E_INVALIDARG              | pcRowsObtained or prghRows was NULL
//      @flag E_OUTOFMEMORY             | Out of Memory
//      @flag OTHER                     | Other HRESULTs returned by called functions
//

STDMETHODIMP CImpIRowset::GetNextRows
    (
	bool		bAllNotifications,	//@parm IN	| Flag. Send all notifications
    HCHAPTER	hReserved,			//@parm IN	| Reserved for future use. Ingored.
    LONG		cRowsToSkip,		//@parm IN	| Rows to skip before reading
    LONG		cRows,				//@parm IN	| Number of rows to fetch
    ULONG		*pcRowsObtained,	//@parm OUT | Number of rows obtained
    HROW		**prghRows			//@parm OUT | Array of Hrows obtained
    )
{
	INTERFACE_METHOD_START( "IRowset::GetNextRows" );
	
	TRACE3(" Skip %d, Rows %d", cRowsToSkip, cRows);

    PROWBUFF prowbuff, tmpRowBuff;
	ULONG dwBookmarks[MAXBMKS];
	
    ULONG	cFetchedRows = 0;
    ULONG	cSlotAlloc =0;
	ULONG	cBRowsToSkip = 0;				// Rows to skip in Btrive file
	ULONG	cBRows = 0;						// Rows to fetch in Btrive file

    ULONG	irow, krow;
	ULONG	iSlot;

	ULONG	ulPropIndex;					// Number of property
	BOOL	bCanFetchBackwards = TRUE;		// We can support DBPROP_CANFETCHBACKWARDS 
	BOOL	bCanScrollBackwards = TRUE;		// We can support DBPROP_CANSCROLLBACKWARDS 
	BOOL	bCanHoldRows = FALSE;
	BOOL	bRowsetIdentity = FALSE;

	BOOL	bRowFound = FALSE;
	BOOL	bMemoryAlloced = FALSE;

	int		nMove;

	BOOL	bSingleGranularity	= TRUE;
	HROW*	phActivatedRows		= NULL;
	ULONG	ulActivatedRows		= 0;

	HRESULT hr = S_OK;

	// available free slots
	bool blnIfFreeSlotsAvailable = true;
	long lUnRoomedRows = 0;

	if (m_pObj->m_bNotReentrant)
		return DB_E_NOTREENTRANT;

    // Check validity of arguments.
    if (pcRowsObtained == NULL)
        return E_INVALIDARG;

    // init out-params
	*pcRowsObtained = 0;

    // Check validity of arguments.
    if (prghRows == NULL)
        return E_INVALIDARG;

    // No-op case always succeeds.
    if (cRows == 0)
        return S_OK;

	//Send notifications
	if (bAllNotifications)
	{
		hr = m_pObj->OnRowsetChange(DBREASON_ROWSET_FETCHPOSITIONCHANGE, DBEVENTPHASE_OKTODO, FALSE);			
		if (hr != S_OK)
			return hr;

		hr = m_pObj->OnRowsetChange(DBREASON_ROWSET_FETCHPOSITIONCHANGE, DBEVENTPHASE_ABOUTTODO, FALSE);			
		if (hr != S_OK)
			return hr;
	}

    // Look if consumer requested supporting scrolling and fetching backward.
	if (m_pObj->m_pUtilProp->GetPropIndex(DBPROP_CANFETCHBACKWARDS, &ulPropIndex) == TRUE)
		bCanFetchBackwards = m_pObj->m_pUtilProp->m_rgproperties[ulPropIndex].boolVal;

	if (m_pObj->m_pUtilProp->GetPropIndex(DBPROP_CANSCROLLBACKWARDS, &ulPropIndex) == TRUE)
		bCanScrollBackwards = m_pObj->m_pUtilProp->m_rgproperties[ulPropIndex].boolVal;

	if (m_pObj->m_pUtilProp->GetPropIndex(DBPROP_CANHOLDROWS, &ulPropIndex) == TRUE)
		bCanHoldRows = m_pObj->m_pUtilProp->m_rgproperties[ulPropIndex].boolVal;

	if (m_pObj->m_pUtilProp->GetPropIndex(DBPROP_IRowsetIdentity, &ulPropIndex) == TRUE)
		bRowsetIdentity = m_pObj->m_pUtilProp->m_rgproperties[ulPropIndex].boolVal;

	if (m_pObj->m_pUtilProp->GetPropIndex(DBPROP_NOTIFICATIONGRANULARITY, &ulPropIndex) == TRUE)
		bSingleGranularity = m_pObj->m_pUtilProp->m_rgproperties[ulPropIndex].boolVal;

	// Check if already allocated rows must be released
	if (!bCanHoldRows && m_pObj->m_ulRowRefCount != 0)
		return DB_E_ROWSNOTRELEASED;

	// Look if given parameters correspond with se properties
	if ((cRows < 0) && (!bCanFetchBackwards))  
        return DB_E_CANTFETCHBACKWARDS;

	if ((cRowsToSkip < 0) && (!bCanScrollBackwards))  
        return DB_E_CANTSCROLLBACKWARDS;

	//Prepare for moving
	nMove = (cRows > 0) ? 1 : -1;
	cBRows = abs(cRows);

	//We need to workaround case when consumer asks for huge amount of rows
	//otherwise memory allocation may fail
	if (cBRows > ACTIVE_ROWS_LIMIT)
		cBRows = ACTIVE_ROWS_LIMIT;

	//Create temporary HROW array
	if (!bSingleGranularity)
	{
		phActivatedRows = new HROW[cBRows];
		
		if (phActivatedRows == NULL)
			return E_OUTOFMEMORY;
	}

	// Is the cursor fully materialized (end-of-cursor condition)?
	// of course it makes sense only for forward-only rowsets
	if ( (!bCanScrollBackwards) && (!bCanFetchBackwards) && (m_pObj->m_dwStatus & STAT_ENDOFCURSOR))
	{
		hr = DB_S_ENDOFROWSET;
		TRACE("  IRowset::GetNextRows: DB_S_ENDOFROWSET" );
		goto CImpIRowset_GetNextRows_Exit;
	}

	// Skip rows if needed
	hr = m_pObj->Skip(cRowsToSkip);
	if (hr != S_OK)
	{
		if (hr == DB_S_ENDOFROWSET)
			m_pObj->m_dwStatus |= STAT_ENDOFCURSOR;

		goto CImpIRowset_GetNextRows_Exit;
	}

	// Fetch rows if needed
	if (cBRows)
	{
		if (*prghRows == NULL)
		{
			// Here we are allocing memory for all rows. If we will not
			// be able to get all rows then we should realloc this block
			// to through out memory extents.
			*prghRows = (HROW*) g_pIMalloc->Alloc(cBRows * sizeof(HROW));
			if (*prghRows == NULL)
			{
				hr = E_OUTOFMEMORY;
				goto CImpIRowset_GetNextRows_Exit;
			}
			bMemoryAlloced = TRUE;
		}		
	}
	
	// Calculate how many rows to fetch
	if( m_pObj->m_cTotRows + cBRows >= m_pObj->m_cMaxRows )
		cBRows = m_pObj->m_cMaxRows - m_pObj->m_cTotRows;

	// Get data to internal buffer of m_pObj
	hr = m_pObj->MovePending( cBRows * nMove );
	if( FAILED( hr ) )
	{
		TRACE("  IRowset::GetNextRows: MovePending failure" );
		goto CImpIRowset_GetNextRows_Exit;
	}
	
	cFetchedRows = 0;
    for (irow = 0; irow < cBRows; irow++)
	{
		if ( m_pObj->IsSlotLimitReached() )
		{
			// hr = DB_S_ROWLIMITEXCEEDED;
			blnIfFreeSlotsAvailable = false;
			// break;
		}

		if( m_pObj->m_cTotRows + cFetchedRows >= m_pObj->m_cMaxRows )
			blnIfFreeSlotsAvailable = false;

		// Get the Data from the File into the row buffer
		hr =  m_pObj->GetFetchedData( irow, dwBookmarks );
        if (hr != S_OK)
		{
			if (hr == DB_S_ENDOFROWSET)
			{
				m_pObj->m_dwStatus |= STAT_ENDOFCURSOR;
				TRACE("  IRowset::GetNextRows: STAT_ENDOFCURSOR" );
				break;
			}
			
			TRACE("  IRowset::GetNextRows: Move does not want to move" );
			goto CImpIRowset_GetNextRows_Exit;
			
		}

		if (bRowsetIdentity)
		{
			bRowFound = FALSE;

			// finding the same rows in the buffer 
			for ( krow = 0; krow < ACTIVE_ROWS_LIMIT; krow++ )
			{
				if (m_pObj->m_SlotRef[krow].Status == 0)
					continue;

				tmpRowBuff = m_pObj->GetRowBuff(m_pObj->m_SlotRef[krow].hRow);
				// The row thaught to be in buffer only if refcount > 0, otherwise it is "free"
				if ( !memcmp( tmpRowBuff->dwBmks, dwBookmarks, tmpRowBuff->cbBmk ) && 
					 tmpRowBuff->ulRefCount > 0 )
				{
					tmpRowBuff->ulRefCount++;
					m_pObj->m_ulRowRefCount++;

					(*prghRows)[cFetchedRows++] = m_pObj->m_SlotRef[krow].hRow;

					hr = S_OK;
					bRowFound = TRUE;
					break;
				}
			}

			if (bRowFound)
				continue;
		}

		// if we have not free slots any more, 
		if (blnIfFreeSlotsAvailable)
		{
			// Find next free slot
			iSlot = m_pObj->GetFreeSlot();
			assert (iSlot != 0xFFFFFFFF);

			prowbuff = m_pObj->GetRowBuff(m_pObj->m_SlotRef[iSlot].hRow);

			hr = m_pObj->m_pData->GetRow(m_pObj->m_rgdwDataOffsets,	(BYTE*)prowbuff);
			if (hr != S_OK)
			{
				TRACE("  IRowset::GetNextRows: GetRow failed" );
				goto CImpIRowset_GetNextRows_Exit;
			}

			prowbuff->ulRefCount = 1;
			prowbuff->cbBmk = BMKS_SIZE * m_pObj->m_pData->bookmarks();
			prowbuff->dwBmkStatus = BMKS_ROW_IN_BUFFER;
			memcpy( prowbuff->dwBmks, dwBookmarks, prowbuff->cbBmk );

			m_pObj->m_ulRowRefCount++;
			
			(*prghRows)[cFetchedRows++] = m_pObj->m_SlotRef[iSlot].hRow;

			if (bSingleGranularity)
			{
				//Send notification
				m_pObj->OnRowChange
							(
							1, //Count of rows
							&m_pObj->m_SlotRef[iSlot].hRow, //hRow
							DBREASON_ROW_ACTIVATE,
							DBEVENTPHASE_DIDEVENT,
							TRUE);				
			}
			else
			{
				//Store activated HROW to send notification later
				phActivatedRows[ulActivatedRows++] = m_pObj->m_SlotRef[iSlot].hRow;
			}
		}
		else
		{
			lUnRoomedRows++;
			(*prghRows)[cFetchedRows] = 0;
			continue;
		}
	}
	
    *pcRowsObtained = cFetchedRows;
    m_pObj->m_cRows	+= cFetchedRows;
    m_pObj->m_cTotRows	+= cFetchedRows;


	// let us decide what the code to return:
	if (!blnIfFreeSlotsAvailable && lUnRoomedRows > 0)
	{
		if (abs(cRows) > ACTIVE_ROWS_LIMIT)
			hr = DB_S_ROWLIMITEXCEEDED;
		else
			hr = DB_S_STOPLIMITREACHED;

		TRACE("  IRowset::GetNextRows: LIMITEXCEEDED" );
		goto CImpIRowset_GetNextRows_Exit;
	}


CImpIRowset_GetNextRows_Exit:

	// 1'st send OnRowsetChange notification, than OnRowChange
	// Send notification on DBREASON_ROWSET_FETCHPOSITIONCHANGE
	if (bAllNotifications)
	{
		if (FAILED(hr))
			m_pObj->OnRowsetChange(DBREASON_ROWSET_FETCHPOSITIONCHANGE,	DBEVENTPHASE_FAILEDTODO, TRUE);					
		else
			m_pObj->OnRowsetChange(DBREASON_ROWSET_FETCHPOSITIONCHANGE,	DBEVENTPHASE_DIDEVENT, TRUE);			
	}

	// Send notification on DBREASON_ROW_ACTIVATE
	if (!FAILED(hr) && ulActivatedRows > 0 && !bSingleGranularity)
		m_pObj->OnRowChange(ulActivatedRows, phActivatedRows, DBREASON_ROW_ACTIVATE, DBEVENTPHASE_DIDEVENT, TRUE);			

	delete [] phActivatedRows;

	// Free allocated rows on error
	if (bMemoryAlloced && (FAILED(hr) || cFetchedRows == 0))
	{
		g_pIMalloc->Free(*prghRows);
		*prghRows = NULL;
	}

	TRACE2("  IRowset::GetNextRows: hr=0x%x", hr);
	
	return hr;

	INTERFACE_METHOD_END();
}

// CImpIRowset::ReleaseRows ---------------------------------------
//
// @mfunc Releases row handles
//
// @rdesc HRESULT
//      @flag S_OK              | Method Succeeded
//      @flag E_INVALIDARG      | Invalid parameters were specified
//      @flag DB_E_BADROWHANDLE | Row handle was invalid
//
STDMETHODIMP CImpIRowset::ReleaseRows
    (
    ULONG			cRows,				//@parm IN	| Number of rows to release
    const HROW		rghRows[],			//@parm IN	| Array of handles of rows to be released
	DBROWOPTIONS	rgRowOptions[],		//@parm IN	| Additional Options
    ULONG			rgRefCounts[],		//@parm OUT | Array of refcnts for the rows
	DBROWSTATUS     rgRowStatus[]		//@parm OUT | Array of row status
    )
{
	INTERFACE_METHOD_START( "IRowset::ReleaseRows" )

	ULONG   ihRow = 0, i = 0;
    BOOL    fEncounteredError = FALSE;
	long	lFailedHandles = 0, lSuccessfulHandles = 0;
    ROWBUFF *pRowBuff;

	ULONG*	arrReleasedRows = NULL;
	ULONG	cReleasedRows	= 0;

	if (m_pObj->m_bNotReentrant)
		return DB_E_NOTREENTRANT;

    // check params
    if (cRows && !rghRows)
        return E_INVALIDARG;

	//Prepare notification
	ULONG	ulPropIndex;					// Number of property
	BOOL	bSingleGranularity = TRUE;

	//Load granularity information
	if (m_pObj->m_pUtilProp->GetPropIndex(DBPROP_NOTIFICATIONGRANULARITY, &ulPropIndex) == TRUE)
		bSingleGranularity = m_pObj->m_pUtilProp->m_rgproperties[ulPropIndex].boolVal;

	//Create temporary array with row numbers for multi-row singularity
	if (!bSingleGranularity)
	{
		arrReleasedRows = new ULONG[cRows];
		
		if (arrReleasedRows == NULL)
			return E_OUTOFMEMORY;
	}

	// begin release handles
    for (ihRow = 0; ihRow < cRows; ihRow++)
	{
		for ( i = 0; i < ACTIVE_ROWS_LIMIT; i++ )
			if ( rghRows[ihRow] == m_pObj->m_SlotRef[ i ].hRow )
				break;

		if (i == ACTIVE_ROWS_LIMIT)
			goto InvalidRow;


        // Found valid row, so decrement reference counts.
        // (Internal error for refcount to be 0 here, since slot set.)
		pRowBuff = m_pObj->GetRowBuff(m_pObj->m_SlotRef[i].hRow);
		
		if ( pRowBuff->ulRefCount <= 0 )
		{
			// it is a candidate for error
			lFailedHandles++;
			goto InvalidRow;
		}

		// Releasing itself !!!
		--pRowBuff->ulRefCount;
		if (pRowBuff->ulRefCount == 0)
		{
			if (bSingleGranularity)
			{
				//Perform notification
				m_pObj->OnRowChange
							(
							1,
							&m_pObj->m_SlotRef[i].hRow,
							DBREASON_ROW_RELEASE,
							DBEVENTPHASE_DIDEVENT,
							TRUE);

				//Release row
				m_pObj->ReleaseSlot(arrReleasedRows[i]);
				--m_pObj->m_cRows;
				assert(m_pObj->m_cRows >= 0);
			}
			else
			{
				//Real row releasing for multy-row singularity will be done later
				//Now we just store the row number
				arrReleasedRows[cReleasedRows++] = i; 
			}
		}

		// debugging helper
		--m_pObj->m_ulRowRefCount;

		// success
		lSuccessfulHandles++;

		// refcounts
		if (rgRefCounts)
			rgRefCounts[ihRow] = pRowBuff->ulRefCount;

		// status
        if (rgRowStatus)
            rgRowStatus[ihRow] = DBROWSTATUS_S_OK;

        // Loop troughout all row handles     
		continue;

InvalidRow:
		// Invalid row handle was passed
		// refcounts
        if (rgRefCounts)
            rgRefCounts[ihRow] = 0;

		// status
        if (rgRowStatus)
            rgRowStatus[ihRow] = DBROWSTATUS_E_INVALID;
	} //end of for

	//Send notification message for multi-row singularity
	if (!bSingleGranularity && cReleasedRows > 0)
	{
		assert(arrReleasedRows != NULL);

		//Allocate temporary array for storing hRow's
		HROW* phReleasedRows = new HROW[cReleasedRows];
		
		if (phReleasedRows == NULL)
			return E_OUTOFMEMORY;

		//Fill temporary array
		for (i = 0; i < cReleasedRows; i++)
			phReleasedRows[i] = m_pObj->m_SlotRef[i].hRow;

		//Perform notification
		m_pObj->OnRowChange
					(
					cReleasedRows,
					phReleasedRows,
					DBREASON_ROW_RELEASE,
					DBEVENTPHASE_DIDEVENT,
					TRUE
					);
		
		//Release temporary array
		delete [] phReleasedRows;

		//Real row release performed here for multi-row singularity
		for (i = 0; i < cReleasedRows; i++)
		{
			m_pObj->ReleaseSlot(arrReleasedRows[i]);
			--m_pObj->m_cRows;
			assert(m_pObj->m_cRows >= 0);
		}
	}

	//Release array with row numbers
	delete [] arrReleasedRows;

	// analyze and return code	
	
	// Check if all handles released succeessfully
	if (lSuccessfulHandles == cRows)
		return S_OK; 
	
	// Ceck if all handles failed were released with errors	
	if (lSuccessfulHandles == 0)
		return DB_E_ERRORSOCCURRED; 

	// some rows were released with errors
	return DB_S_ERRORSOCCURRED;

	INTERFACE_METHOD_END();
}


// CImpIRowset::RestartPosition ---------------------------------------
//
// @mfunc Repositions the next fetch position to the start of the rowset
//
// - all rows must be released before calling this method
// - it is not expensive to Restart us, because we are from a single table
//
//
// @rdesc HRESULT
//      @flag S_OK                  | Method Succeeded
//      @flag DB_E_ROWSNOTRELEASED  | All HROWs must be released before calling
//
STDMETHODIMP CImpIRowset::RestartPosition
    (
    HCHAPTER    hReserved        //@parm IN | Reserved for future use.  Ignored.
    )
{
	INTERFACE_METHOD_START("IRowset::RestartPosition");

	HRESULT hr;
	ULONG	ulPropIndex;
	BOOL	bCanHoldRows = FALSE;

	if (m_pObj->m_bNotReentrant)
		return DB_E_NOTREENTRANT;

	if (m_pObj->m_pUtilProp->GetPropIndex(DBPROP_CANHOLDROWS, &ulPropIndex))
		bCanHoldRows = m_pObj->m_pUtilProp->m_rgproperties[ulPropIndex].boolVal;
	
	//Send notifications
	hr = m_pObj->OnRowsetChange(DBREASON_ROWSET_FETCHPOSITIONCHANGE, DBEVENTPHASE_OKTODO, FALSE);			
	if (hr != S_OK)
		return hr;

	hr = m_pObj->OnRowsetChange(DBREASON_ROWSET_FETCHPOSITIONCHANGE, DBEVENTPHASE_ABOUTTODO, FALSE);			
	if (hr != S_OK)
		return hr;

	// make sure all rows have been released
	if (!bCanHoldRows)
	{
		if ((m_pObj->m_prowbitsIBuffer)->ArrayEmpty() != S_OK)
			return DB_E_ROWSNOTRELEASED;

	   assert( m_pObj->m_ulRowRefCount == 0 ); // should be true since array was empty
	}

	// set "next fetch" position to the start of the rowset
	m_pObj->InitPosition();
	
	// Send notification on DBREASON_ROWSET_FETCHPOSITIONCHANGE
	m_pObj->OnRowsetChange(DBREASON_ROWSET_FETCHPOSITIONCHANGE, DBEVENTPHASE_DIDEVENT, TRUE);			
	
	// clear "end of cursor" flag
	m_pObj->m_dwStatus &= ~STAT_ENDOFCURSOR;

    return S_OK;
	
	INTERFACE_METHOD_END();
}


// CImpIRowset::AddRefRows --------------------------------------------------
//
// @mfunc Adds a reference count to an existing row handle
//
// @rdesc HResult
//      @flag S_OK              | Method Succeeded
//      @flag E_INVALIDARG      | rghRows was NULL and cRows was not 0
//      @flag DB_E_BADROWHANDLE | An element of rghRows was invalid
STDMETHODIMP  CImpIRowset::AddRefRows
    (
    ULONG           cRows,          // @parm IN     | Number of rows to refcount
    const HROW      rghRows[],      // @parm IN     | Array of row handles to refcount
    ULONG           rgRefCounts[],  // @parm OUT    | Array of refcounts
    DBROWSTATUS     rgRowStatus[]   // @parm OUT    | Array of row status
	)
{
	INTERFACE_METHOD_START( "IRowset::AddRefRows" )

    ULONG   ihRow = 0;
    ROWBUFF *pRowBuff;
    ULONG   cAddRef;
	long	lFailedHandles = 0, lSuccessfulHandles = 0;

    // check params
    if (cRows && !rghRows)
        return E_INVALIDARG;

    cAddRef = 0;

    // for each of the HROWs the caller provided...
    for (ihRow = 0; ihRow < cRows; ihRow++)
    {
        //if (S_OK != (m_pObj->m_prowbitsIBuffer)->IsSlotSet((ULONG) rghRows[ihRow] ))
		if (S_OK != (m_pObj->m_prowbitsIBuffer)->IsSlotSet( m_pObj->GetSlotNumber( rghRows[ihRow]) ))
		{
			// refcounts
			if (rgRefCounts)
				rgRefCounts[ihRow] = 0;
			
			// status
			if (rgRowStatus)
				rgRowStatus[ihRow] = DBROWSTATUS_E_INVALID;
			
			// failed
			lFailedHandles++;
			
            // Loop troughout all row handles even if we hit an error    
            continue;
		}

        // if rfcount == 0, it is an error
        pRowBuff = m_pObj->GetRowBuff((ULONG) rghRows[ihRow] );
		if( pRowBuff->ulRefCount == 0 )
		{
			// refcounts
			if (rgRefCounts)
				rgRefCounts[ihRow] = 0;
			
			// status
			if (rgRowStatus)
				rgRowStatus[ihRow] = DBROWSTATUS_E_INVALID;
			
			// failed
			lFailedHandles++;
			
            // Loop troughout all row handles even if we hit an error    
            continue;			
		}
 
		// hrow can not reference to deleted row
        if (S_OK == m_pObj->IsDeleted((ULONG) (pRowBuff->dwBmkStatus )))
        {
			// refcounts
			if (rgRefCounts)
				rgRefCounts[ihRow] = 0;
			
			// status
			if (rgRowStatus)
				rgRowStatus[ihRow] = DBROWSTATUS_E_DELETED;
			
			// failed
			lFailedHandles++;
			
            // Loop troughout all row handles even if we hit an error    
            continue;			
		}

		assert( m_pObj->m_ulRowRefCount != 0 );
	    ++m_pObj->m_ulRowRefCount;

        // bump total refcounted
        cAddRef++;

		// success
		++pRowBuff->ulRefCount;
  		lSuccessfulHandles++;

        // stuff new refcount into caller's array
        if (rgRefCounts)
            rgRefCounts[ihRow] = pRowBuff->ulRefCount;

        // stuff new refcount into caller's array
        if (rgRowStatus)
            rgRowStatus[ihRow] = DBROWSTATUS_S_OK;
	}

	// analyze and return code
	
	// Check if all handles released succeessfully
	if (lFailedHandles == 0 && lSuccessfulHandles == cRows)
		return S_OK; 

	// Check if all handles failed were released with errors
	if (lFailedHandles == cRows && lSuccessfulHandles == 0)
		return DB_E_ERRORSOCCURRED;

	// Check if some rows were released with errors
	if (lFailedHandles > 0 && lFailedHandles < cRows && lSuccessfulHandles > 0 && lSuccessfulHandles < cRows)
		return DB_S_ERRORSOCCURRED;

	// something unlikely
	return E_FAIL;

	INTERFACE_METHOD_END();
}

