/*
 * Copyright (c)  2000
 * SWsoft  company
 *
 * This material is provided "as is", with absolutely no warranty expressed
 * or implied. Any use is at your own risk.
 *
 * Permission to use or copy this software for any purpose is hereby granted 
 * without fee, provided the above notices are retained on all copies.
 * Permission to modify the code and to distribute modified code is granted,
 * provided the above notices are retained, and a notice that the code was
 * modified is included with the above copyright notice.
 *
 */

//--------------------------------------------------------------------
// MySQL OLE DB Provider 
// Functionality: minimum
// Release: 0.1
//
// @doc
//
// @module DATASRC.CPP | CDataSource object implementation
//
//

// Includes ------------------------------------------------------------------

#include  "hfiles.h"
#include  "headers.h"
#include  "mysqlmeta.h"
#include  "utilprop.h"

#define CONTINUE -1

// Code ----------------------------------------------------------------------

// CDataSource::CDataSource --------------------------------------------------
//
// @mfunc Constructor for this class
//
// @rdesc NONE
//
CDataSource::CDataSource
    (
    LPUNKNOWN pUnkOuter         //@parm IN | Outer Unkown Pointer
    )
{
	CLEAR_CONSTRUCT( CDataSource );
	
    //  Initialize simple member vars
    m_pUnkOuter         = pUnkOuter ? pUnkOuter : this;

	// Increment global object count.
    OBJECT_CONSTRUCTED();

	TRACE( " Data Source Object is constructed!" );

    return;
}


// CDataSource::~CDataSource -------------------------------------------------
//
// @mfunc Destructor for this class
//
// @rdesc NONE
//
CDataSource:: ~CDataSource(void)
{
	ULONG	ulRefCount;

	// Decrement the ref count on the data conversion object
	if( g_pIDataConvert )
	{
		ulRefCount = g_pIDataConvert->Release();

		// Is it gone for good?
		if( !ulRefCount )
			g_pIDataConvert = NULL;
	}

    // Release meta if necessary
	ClearSwstMeta();
	
	// Free properties management object
    delete m_pUtilProp;

    //  Free contained interfaces
    delete m_pIDBInitialize;
    delete m_pIDBProperties;
    delete m_pIDBInfo;
	delete m_pIDBCreateSession;
	delete m_pIPersist;

    // Decrement global object count.
    OBJECT_DESTRUCTED();

	TRACE( " Data Source Object is destructed!" );

    return;
}


static WCHAR* getPart( WCHAR* str, LPCOLESTR wanted )
{
	WCHAR* p = wcsstr( str, wanted );
	if( p != NULL )
		p += wcslen( wanted );
	return p;
}

inline static WCHAR* getCatalogsPart( WCHAR* str )
{
	return getPart( str, L"CATALOGS=" );
}

WCHAR* getOwnersPart( WCHAR* str )
{
	return getPart( str, L"OWNERS=" );
}

static BOOL getNextLexem( WCHAR* buf, WCHAR*& first, int nMax )
{
	if( first == NULL )
		return FALSE;
	
	if( *first==L'\0' || *first==L';' )
		return FALSE;

	int ret = CONTINUE;
	
	for( WCHAR* p = first; *p!=L'\0' && *p!=L',' && *p!=L';'; p++ ) {}
	if( p <= first + nMax && p > first )
	{
		wcscpy0( buf, first, p - first );
		ret = TRUE;
	}

	if( *p == L',' )
		first = p + 1;
	else
		first = p;

	return ret;
}


inline static BOOL getNextOwner( WCHAR* buf, WCHAR*& first )
{
	return getNextLexem( buf, first, MAXOWNERNAME );
}

inline static BOOL getNextCatalog( WCHAR* buf, WCHAR*& first )
{
	return getNextLexem( buf, first, MAXDATASOURCENAME );
}

extern void MakeMySqlConnStr( const char* in, char* out );

// Check and load one dictionary 
HRESULT CDataSource::CheckAndTrackOneDatabase( LPOLESTR pwszName, LPOLESTR pwszOwners )
{
	// check args
	if( pwszName == NULL )
		pwszName = L"";

	if( pwszOwners == NULL )
		pwszOwners = L"";

	// Get current database name from properties
	ULONG dsn;
	if( !m_pUtilProp->GetPropIndex(DBPROP_DATASOURCENAME, &dsn ) )
		return E_FAIL;
	LPOLESTR pwszMySqlString = EXTRACT_MYSQLSTR( m_pUtilProp->m_rgproperties[dsn].pwstrVal );

	// Catalog of loaded data source ?
	if( *pwszName == L'\0' )
		return TrackOneDatabase( m_szDDFPath, m_szDataPath, pwszName, pwszOwners, m_typeOfDataSource, m_iDriverID, pwszMySqlString );
	
	char szName[ MAXDATASOURCENAME+1 ];
	char szDDFPath[ MAXDDFPATH+1 ]; *szDDFPath = '\0';
	char szDataPath[ MAXDDFPATH+1 ]; *szDataPath = '\0';

	// Convert wide string of database name to ANSI string
	if( !WideCharToMultiByte( CP_ACP, 0, pwszName, -1, szName, MAXDATASOURCENAME, NULL, NULL ) )
	{
		TRACE( "  CDataSource::CheckAndTrackOneDatabase: MultiByte error" );
		return E_FAIL;
	}
	
	// Check and get correct path
	DATASOURCE_TYPE type = DATASOURCE_NONE;
	BYTE iDriverID = 0;

	//if( m_pUtilProp->MySqlMode() )
	switch( m_pUtilProp->InternalSqlSupport() )
	{
	case ISS_WRONG:
		return E_UNEXPECTED;
	
	case ISS_MYSQL:
		{
		char szFullInfo[ 1024 ]; 
		WCHAR wszFullInfo[ 1024 ]; 
		
		SQLHENV henv;
		SQLHDBC hdbc;
		SQLSMALLINT cbFullInfo;

		MakeMySqlConnStr( szName, szFullInfo );

		if( SQL_SUCCEEDED( SQLAllocEnv( &henv ) ) )
		{
			if( SQL_SUCCEEDED( SQLAllocConnect( henv, &hdbc ) ) )
			{
				if( SQL_SUCCEEDED( SQLDriverConnect( hdbc, NULL, (BYTE*)szFullInfo, strlen( szFullInfo ), (BYTE*)szFullInfo, MAXSTR(szFullInfo), &cbFullInfo, SQL_DRIVER_NOPROMPT ) ) )
				{
					type = DATASOURCE_MYSQL;
					const char _db[] = ";DB=", _sysdb[] = ";SYSDB=";
					char * db = strstr( szFullInfo, _db );
					char * sysdb = strstr( szFullInfo, _sysdb );

					if( sysdb )
					{
						sysdb += MAXSTR( _sysdb );
						strcpy( szDDFPath, sysdb );
					}
					else if( db )
					{
						db += MAXSTR( _db );
						strcpy( szDDFPath, db );
					}

					if( db || sysdb )
					{
						char * end = strchr( szDDFPath, ';' );
						if( end ) *end = 0;
						strcpy( szDataPath, szDDFPath );					
					}
					
					SQLDisconnect( hdbc );
				}
			
				SQLFreeConnect( hdbc );
			}
			
			SQLFreeEnv( henv );
		}
		
		if( type == DATASOURCE_NONE )
			return E_FAIL;

		A2Wsz( szFullInfo, wszFullInfo );
		
		// Track selected database
		return TrackOneDatabase( szDDFPath, szDataPath, pwszName, pwszOwners, type, iDriverID, wszFullInfo );
		}

	}
	return E_FAIL;
}


// Load distionary of one database
HRESULT CDataSource::TrackOneDatabase( LPSTR pszPath, LPSTR pszDataPath, LPOLESTR pwszName, LPOLESTR pwszOwners, DATASOURCE_TYPE typeOfDataSource, BYTE iDriverID, LPOLESTR pwszMySqlStr )
{
	// check args
	if( pszPath == NULL )
		return E_INVALIDARG;

	if( pwszName == NULL )
		pwszName = L"";
	
	// Ensure that we have initialized m_pSwstMetaHolder
	if( m_pSwstMetaHolder == NULL && FAILED( GetSwstMeta( NULL ) ) )
		return E_FAIL;
	
	DSNINFO dsnInfo;
	GetAuthInfo( &dsnInfo );
	
	HRESULT hr = m_pSwstMetaHolder->LoadMetaData(pszPath, pszDataPath, pwszName, NULL, typeOfDataSource, iDriverID, dsnInfo.m_szServerName, dsnInfo.m_szUser, dsnInfo.m_szPassword, pwszMySqlStr );
	if( FAILED( hr ) )
		return hr;

	WCHAR wszOwner[ MAXOWNERNAME+1 ];
	for( int status=getNextOwner( wszOwner, pwszOwners ); status != FALSE; status=getNextOwner( wszOwner, pwszOwners ) )
		if( status == TRUE )
			m_pSwstMetaHolder->LoadMetaData(pszPath, pszDataPath, pwszName, wszOwner, typeOfDataSource, iDriverID, dsnInfo.m_szServerName, dsnInfo.m_szUser, dsnInfo.m_szPassword, pwszMySqlStr );
		
	return S_OK;
}


// Load distionary of one enum 
HRESULT CDataSource::TrackOneEnum( EnumDatabase* pEnum, LPOLESTR pwszOwners )
{
	// check args
	if( pEnum == NULL )
		return E_INVALIDARG;
	
	// Save all authentificational data
	DSNINFO dsnInfo;
	GetAuthInfo( &dsnInfo );
	
	// Track enum
	WCHAR wszDataSourceName[ MAXDATASOURCENAME+1 ];
	if( SUCCEEDED( pEnum->Init( &dsnInfo ) ) )
	{
		while( SUCCEEDED( pEnum->Next( &dsnInfo ) ) )
			if( MultiByteToWideChar(CP_ACP, MB_PRECOMPOSED, dsnInfo.m_szDataSource,	-1, wszDataSourceName, MAXDATASOURCENAME ) )
				TrackOneDatabase( dsnInfo.m_szDDFPath, dsnInfo.m_szDataPath, wszDataSourceName, pwszOwners, dsnInfo.m_type, dsnInfo.m_iDriverID, NULL );
		
		pEnum->Done();
	}

	return S_OK;
}


// Load all available dictionaries
HRESULT CDataSource::TrackAllEnums( LPOLESTR pwszOwners )
{
	// Save all authentificational data
	DSNINFO dsnInfo;
	GetAuthInfo( &dsnInfo );

	//EnumOfEnumsDataSources enumDataSources( m_pUtilProp ? m_pUtilProp->MySqlMode() : FALSE ); 
	EnumOfEnumsDataSources enumDataSources( m_pUtilProp ? (m_pUtilProp->InternalSqlSupport() == ISS_MYSQL) : FALSE ); // correct
	if( SUCCEEDED( enumDataSources.Init( &dsnInfo ) ) )
	{
		while ( SUCCEEDED( enumDataSources.Next( &dsnInfo ) ) )
		{
			EnumDatabase* pEnum = NULL;
			if( SUCCEEDED( enumDataSources.GetNewObject( dsnInfo.m_type, &pEnum ) ) )
			{
				TrackOneEnum( pEnum, pwszOwners );
				delete pEnum;
			}
		} 
		enumDataSources.Done();
	}

	return S_OK;
}

// This function adds reference to SwstMetaHolder if agr is not NULL!
HRESULT CDataSource::GetSwstMeta(CSwstMetaHolder** ppSwstMetaHolder)
{
	if (m_pSwstMetaHolder == NULL)
	{
		m_pSwstMetaHolder = new CSwstMetaHolder;
		if (m_pSwstMetaHolder == NULL)
			return E_OUTOFMEMORY;

		// Get current database name from properties
		ULONG dsn, prvStr;
		if( !m_pUtilProp->GetPropIndex(DBPROP_DATASOURCENAME, &dsn ) ||
			!m_pUtilProp->GetPropIndex(DBPROP_INIT_PROVIDERSTRING, &prvStr ) )
		{
			delete m_pSwstMetaHolder;
			m_pSwstMetaHolder = NULL;
			return E_FAIL;
		}

		// Begin of substrings with list of owners (, is separator  ; is end mark)
		WCHAR* pwszOwnerPart = getOwnersPart( m_pUtilProp->m_rgproperties[prvStr].pwstrVal );
		// Begin of substrings with list of catalogs (, is separator  ; is end mark)
		WCHAR* pwszCatalogsPart = getCatalogsPart( m_pUtilProp->m_rgproperties[prvStr].pwstrVal );
		
		// Init catalog name as \0 & DataSourceCatalog & MySql ODBC connection string
		WCHAR wszCatalog[ MAXDATASOURCENAME+1+1 ], wszMySqlStr[ MYSQL_MAXDATASOURCENAME+1 ];
		wszCatalog[0] = L'\0';
		wcscpy0( wszCatalog + 1, m_pUtilProp->m_rgproperties[dsn].pwstrVal, MAXDATASOURCENAME );
		wcscpy0( wszMySqlStr, EXTRACT_MYSQLSTR( m_pUtilProp->m_rgproperties[dsn].pwstrVal ), MYSQL_MAXDATASOURCENAME );

		// Load database with initial path and empty CatalogName as #0. Should succeed
		HRESULT hr = TrackOneDatabase( m_szDDFPath, 
										m_szDataPath, 
										wszCatalog,
										pwszOwnerPart,
										m_typeOfDataSource,
										m_iDriverID,
										wszMySqlStr);
		if ( FAILED( hr ) )
		{
			delete m_pSwstMetaHolder;
			m_pSwstMetaHolder = NULL;
			return hr;
		}

		// Load other requested dictionaries. wszCatalog will be overwritten
		for( int status = getNextCatalog( wszCatalog, pwszCatalogsPart ); status != FALSE; status = getNextCatalog( wszCatalog, pwszCatalogsPart ) )
			if( status == TRUE )
			{
				if( !wcsicmp( wszCatalog, L"ALL" ) )
					TrackAllEnums( pwszOwnerPart );
				else if( !wcsicmp( wszCatalog, L"ALLNAMEDDB" ) )
				{
					 EnumNamedDatabases enumNamedDb;
					 TrackOneEnum( &enumNamedDb, pwszOwnerPart );
				}
				else if( !wcsicmp( wszCatalog, L"ALLUSERDSNS" ) )
				{
					 EnumUserDSNs enumUserDSNs;
					 TrackOneEnum( &enumUserDSNs, pwszOwnerPart );
				}
				else if( !wcsicmp( wszCatalog, L"ALLSYSTEMDSNS" ) )
				{
					 EnumSystemDSNs enumSystemDSNs;
					 TrackOneEnum( &enumSystemDSNs, pwszOwnerPart );
				}
				else
				{
#ifdef _DEBUG
					 if( FAILED( CheckAndTrackOneDatabase( wszCatalog, pwszOwnerPart ) ) )
						TRACE( "  CDataSource::CheckAndTrackOneDatabase: Loading error" );
#else
					 CheckAndTrackOneDatabase( wszCatalog, pwszOwnerPart );	
#endif
				}
			}

#ifdef _DEBUG
			int maxNumb = -1;
			m_pSwstMetaHolder->GetMaxNumber(&maxNumb);
			TRACE( " >> Loaded %d dictionaries", maxNumb );
#endif
	}

	// Save result and Add ref if necessary
	if( ppSwstMetaHolder != NULL )
		*ppSwstMetaHolder = m_pSwstMetaHolder;

	return S_OK;
}


void CDataSource::ClearSwstMeta() 
{
	TRACE( "CDataSource::ClearSwstMeta" );
	
	if (m_pSwstMetaHolder != NULL)
	{
		delete m_pSwstMetaHolder; 
		m_pSwstMetaHolder = NULL;
	}
}


// CDataSource::FInit --------------------------------------------------------
//
// @mfunc Initialize the command Object
//
// @rdesc Did the Initialization Succeed
//      @flag  TRUE | Initialization succeeded
//      @flag  FALSE | Initialization failed
//
BOOL CDataSource::FInit( BOOL bMySqlMode )
{
	TRACE( "CDataSource::FInit" );
	
	HRESULT		hr;
    LPUNKNOWN   pIUnknown = (LPUNKNOWN) this;

    if (m_pUnkOuter)
        pIUnknown = m_pUnkOuter;

	// Instantiate the data conversion service object
	if( !g_pIDataConvert )
	{
		hr = CoCreateInstance(CLSID_OLEDB_CONVERSIONLIBRARY,
							  NULL,
							  CLSCTX_INPROC_SERVER,
							  IID_IDataConvert,
							  (void **)&g_pIDataConvert);
		if( FAILED(hr) )
			return FALSE;
	}
	else
		// Already instantiated, increment reference count
		g_pIDataConvert->AddRef();

	// Is MySql data source?
	m_bMySqlMode = bMySqlMode;

    // Allocate properties management object
    m_pUtilProp = new CUtilProp( this );

    //  Allocate contained interface objects
    m_pIDBInitialize    = new CImpIDBInitialize( this, pIUnknown );
    m_pIDBProperties	= new CImpIDBProperties( this, pIUnknown );
	m_pIDBInfo			= new CImpIDBInfo( this, pIUnknown );
    m_pIDBCreateSession = new CImpIDBCreateSession( this, pIUnknown );
    m_pIPersist			= new CImpIPersist( this, pIUnknown );

    return (BOOL) (m_pIDBInitialize &&
				   m_pIDBInfo &&
                   m_pIDBProperties	&&
                   m_pIDBCreateSession &&
                   m_pIPersist);
}


// CDataSource::QueryInterface -----------------------------------------------
//
// @mfunc Returns a pointer to a specified interface. Callers use
// QueryInterface to determine which interfaces the called object
// supports.
//
// @rdesc HRESULT indicating the status of the method
//      @flag S_OK          | Interface is supported and ppvObject is set.
//      @flag E_NOINTERFACE | Interface is not supported by the object
//      @flag E_INVALIDARG  | One or more arguments are invalid.
//
STDMETHODIMP CDataSource::QueryInterface
    (
    REFIID riid,    //@parm IN | Interface ID of the interface being queried for.
    LPVOID * ppv    //@parm OUT | Pointer to interface that was instantiated
    )
{
    TRACE( "CDataSource::QueryInterface" );
	
	// Is the pointer bad?
    if (ppv == NULL)
        return  ( E_INVALIDARG );

    //  Place NULL in *ppv in case of failure
    *ppv = NULL;

    //  This is the non-delegating IUnknown implementation
    if (riid == IID_IUnknown)
        *ppv = (LPVOID) this;
    else if (riid == IID_IDBInitialize)
        *ppv = (LPVOID) m_pIDBInitialize;
	else if (riid == IID_IDBInfo)
		*ppv = (LPVOID) m_pIDBInfo;
	else if (riid == IID_IDBProperties)
        *ppv = (LPVOID) m_pIDBProperties;
    else if (riid == IID_IPersist)
        *ppv = (LPVOID) m_pIPersist;
    else if (riid == IID_IPersistFile)
        *ppv = (LPVOID) m_pIPersist;
	else
	{
		// These are not valid at Uninit state.
		if( riid == IID_IDBCreateSession )
			*ppv = (LPVOID)m_pIDBCreateSession;

		// Special case for uninitialized.
		if( *ppv && !m_fDSOInitialized )
		{
			*ppv = NULL;			
			return  (E_UNEXPECTED);
		}
	}

    //  If we're going to return an interface, AddRef it first
    if (*ppv)
        {
        ((LPUNKNOWN) *ppv)->AddRef();
        return  ( S_OK );
        }
    else
        return  ( E_NOINTERFACE );
}


// CDataSource::AddRef -------------------------------------------------------
//
// @mfunc Increments a persistence count for the object
//
// @rdesc Current reference count
//
STDMETHODIMP_( ULONG ) CDataSource::AddRef(void)
{
    TRACE2( "CDataSource::AddRef (%d)", m_cRef+1 );
	return ++m_cRef;
}


// CDataSource::Release ------------------------------------------------------
//
// @mfunc Decrements a persistence count for the object and if
// persistence count is 0, the object destroys itself.
//
// @rdesc Current reference count
//
STDMETHODIMP_( ULONG ) CDataSource::Release(void)
{
    TRACE2( "CDataSource::Release (%d)", m_cRef-1 );

    if (!--m_cRef)
        {
        delete this;
        return 0;
        }

    return m_cRef;
}


// CDataSource::SetActiveSession
//
// @mfunc MARK SESSION ulSession As Active
//      @flag S_OK		       | 
//      @flag S_FALSE          | 
//
HRESULT CDataSource::SetActiveSession
	(
	ULONG ulSession	//@parm IN | # of Session
	)
{
	m_sbSessMonitor[ulSession] = true;
	return (S_OK);
}


// CDataSource::SetInActiveSession
//
// @mfunc MARK SESSION ulSession As InActive
//      @flag S_OK		       | 
//      @flag S_FALSE          | 
//
HRESULT CDataSource::SetInActiveSession
	(
	ULONG ulSession	//@parm IN | # of Session
	)
{
	m_sbSessMonitor[ulSession] = false;
	return (S_OK);
}


// CDataSource::GetFirstInactiveSession
//
// @mfunc Get first inactive session
//      @flag S_OK		       | 
//      @flag S_FALSE          | 
//
HRESULT CDataSource::GetFirstInactiveSession
	(
	ULONG *ulSession	//@parm IN | # of Session
	)
{
	TRACE( "CDataSource::GetFirstInactiveSession" );
	
	ULONG ul;
	for (ul = 0; ul < NUM_SUPPORTED_SESSIONS_PER_DATASOURCE; ul++)
	{
		if (m_sbSessMonitor[ul] == false)
		{
			*ulSession = ul;
			return (S_OK);
		}
	}

	return (S_FALSE);
}


// CDataSource::CountActiveSessions
//
// @mfunc Count Active Sessions
//      @flag S_OK		       | 
//      @flag S_FALSE          | 
//
HRESULT CDataSource::CountActiveSessions()
{
	TRACE( "CDataSource::CountActiveSessions" );

	ULONG ul;
	ULONG ulSC = 0;
	for (ul = 0; ul < NUM_SUPPORTED_SESSIONS_PER_DATASOURCE; ul++)
	{
		if (m_sbSessMonitor[ul] == true)
		{
			ulSC++;
		}
	}

	m_ulTotalSessions = ulSC;

	return (S_OK);
}


// CDataSource::IfActiveSessionsExist
//
// @mfunc Determines if activge sessions exist
//      @flag true	       | 
//      @flag false        | 
//
bool CDataSource::IfActiveSessionsExist()
{
	TRACE( "CDataSource::IfActiveSessionsExist" );
	CountActiveSessions(); 
	return (m_ulTotalSessions > 0 ? true : false); 
}


// CDataSource::RemoveSession
//
// @mfunc Performs all operations, connected with removing session
//      @flag S_OK		       | 
//      @flag S_FALSE          | 
//
HRESULT	CDataSource::RemoveSession
	(
	ULONG ulSession	//@parm IN | # of Session
	)
{
	TRACE( "CDataSource::RemoveSession" );
	
	HRESULT hr = (S_OK);
	
	// some cautions
	assert(ulSession < NUM_SUPPORTED_SESSIONS_PER_DATASOURCE);

	// mark session # ulSess as inactive
	hr = SetInActiveSession(ulSession);
	if (hr == S_FALSE)
		return E_FAIL;	

	// shortcut for (m_pObj->m_pSessions)[ulSess]
	m_pSessions[ulSession] = NULL;

	// refresh the number of active sessions
	hr = CountActiveSessions();	

	return (hr);
}


//Load authentificational information
void CDataSource::GetAuthInfo( DSNINFO* pDSNINFO )
{
	if( pDSNINFO == NULL )
		return;	// Nothing to do

	ULONG ulIndex;
	if( m_typeOfDataSource != DATASOURCE_DDF && m_pUtilProp->GetPropIndex(DBPROP_INIT_LOCATION, &ulIndex ) )
	{
		W2Asz( m_pUtilProp->m_rgproperties[ulIndex].pwstrVal, pDSNINFO->m_szServerName );
		// check if server is local computer
		if( !stricmp( ".", pDSNINFO->m_szServerName ) )
			*pDSNINFO->m_szServerName = '\0';
		else if( !getenv("USERDOMAIN") || !stricmp( getenv("USERDOMAIN"), pDSNINFO->m_szServerName ) )
			*pDSNINFO->m_szServerName = '\0';
	}
	if( m_pUtilProp->GetPropIndex(DBPROP_AUTH_USERID, &ulIndex ) )
		W2Asz( m_pUtilProp->m_rgproperties[ulIndex].pwstrVal, pDSNINFO->m_szUser );
	if( m_pUtilProp->GetPropIndex(DBPROP_AUTH_PASSWORD, &ulIndex ) )
		W2Asz( m_pUtilProp->m_rgproperties[ulIndex].pwstrVal, pDSNINFO->m_szPassword );
}
