/*
 * Copyright (c)  2000
 * SWsoft  company
 *
 * This material is provided "as is", with absolutely no warranty expressed
 * or implied. Any use is at your own risk.
 *
 * Permission to use or copy this software for any purpose is hereby granted 
 * without fee, provided the above notices are retained on all copies.
 * Permission to modify the code and to distribute modified code is granted,
 * provided the above notices are retained, and a notice that the code was
 * modified is included with the above copyright notice.
 *
 */

//--------------------------------------------------------------------
// MySQL OLE DB Provider 
// Functionality: minimum
// Release: 0.1
//
// @doc
//
// @module CONNPT.CPP | IConnectionPointContainer, IEnumConnectionPoints
// IConnectionPoints and IEnumConnections objects Implementation

// Includes ------------------------------------------------------------------

#include "hfiles.h"
#include "headers.h"
#include "rowset.h"

// Code ----------------------------------------------------------------------

// IConnectionPointContainer specific methods

// CImpIConnectionPointContainer::FindConnectionPoint
//
// @mfunc Returns a pointer to the IConnectionPoint interface of a connection point
//		  for a specified IID, if that IID describes a supported outgoing interface.
//
// @rdesc NONE
//
STDMETHODIMP CImpIConnectionPointContainer::FindConnectionPoint
(
  REFIID riid,				// Requested connection point's interface identifier
  IConnectionPoint **ppCP	// Address of output variable that receives the 
							// IConnectionPoint interface pointer
)
{
	INTERFACE_METHOD_START( "IConnectionPointContainer::FindConnectionPoint" );

	if (ppCP == NULL)
		return E_POINTER;

	if (riid == IID_IRowsetNotify)
	{
		if (m_pObj->m_pIRowsetNotifyConnectionPoint == NULL)
		{
			if (FAILED(m_pObj->m_pIRowsetNotifyConnectionPoint = 
							new CImpIConnectionPoint(m_pObj, riid, this)))
				return E_OUTOFMEMORY;
		}

		*ppCP = m_pObj->m_pIRowsetNotifyConnectionPoint;

		(*ppCP)->AddRef();
		return S_OK;
	}
	
	return CONNECT_E_NOCONNECTION;
	
	INTERFACE_METHOD_END();
}

// CImpIConectionPointContainer::EnumConnectionPoints
//
// @mfunc Creates an enumerator object to iterate through all the connection points
//		  supported in the connectable object, one connection point per outgoing IID.
//
// @rdesc NONE

STDMETHODIMP CImpIConnectionPointContainer::EnumConnectionPoints
(
  IEnumConnectionPoints **ppEnum  // Address of output variable that receives the 
								  // IEnumConnectionPoints interface pointer
)
{	
	INTERFACE_METHOD_START( "IConnectionPointContainer::EnumConnectionPoints" );
	
	if (ppEnum == NULL)
		return E_POINTER;

	if (FAILED((*ppEnum) = new CImpIEnumConnectionPoints(m_pObj)))
		return E_OUTOFMEMORY;
	
	(*ppEnum) -> AddRef();
	
	return S_OK;

	INTERFACE_METHOD_END();
}

// IEnumConnectionPoints specific methods ----------------------------------

// CImpIEnumConnectionPoints::Next
//
// @mfunc 
//
// @rdesc 
//
HRESULT CImpIEnumConnectionPoints::Next
(
  ULONG cConnections,		//[in]Number of IConnectionPoint values 
							// returned in rgpcn array
  IConnectionPoint **rgpcn, //[out]Array to receive enumerated connection 
							// points
  ULONG *pcFetched			//[out]Pointer to the actual number of 
							// connection points returned in rgpcn array
)
{
	INTERFACE_METHOD_START( "IEnumConnectionPoints::Next" );

	if (pcFetched == NULL || rgpcn == NULL)
		return E_POINTER;

// Our Provider has only one connection point. That is for IRowsetNotify interface.

	*pcFetched = 0;
	*rgpcn = NULL;

	if (m_pCurrentCP != NULL)
	{
		m_pCurrentCP = m_pObj->m_pIRowsetNotifyConnectionPoint;
		*rgpcn = m_pObj->m_pIRowsetNotifyConnectionPoint;
		(*rgpcn)->AddRef();
		*pcFetched = 1;
	}
	
	return S_OK;

	INTERFACE_METHOD_END();
}
 
// CImpIEnumConnectionPoints::Skip
//
// @mfunc 
//
// @rdesc 
//
HRESULT CImpIEnumConnectionPoints::Skip
(
  ULONG cConnections  //[in]Number of elements to skip
)
{
	INTERFACE_METHOD_START( "IEnumConnectionPoints::Skip" );
	
	if (cConnections)
		m_pCurrentCP = m_pObj->m_pIRowsetNotifyConnectionPoint;
	
	return S_OK;

	INTERFACE_METHOD_END();
}
 
// CImpIEnumConnectionPoints::Reset
//
// @mfunc
//
// @rdesc
//
HRESULT CImpIEnumConnectionPoints::Reset(void)
{
	INTERFACE_METHOD_START( "IEnumConnectionPoints::Reset" );
	
	m_pCurrentCP = NULL;

	return S_OK;
	
	INTERFACE_METHOD_END();
}
 
// CImpIEnumConnectionPoints::Clone
//
// @mfunc 
//
// @rdesc 
//
HRESULT CImpIEnumConnectionPoints::Clone
(
  IEnumConnectionPoints **ppEnum  //[out]Address of output variable 
                                  // that receives the 
                                  // IEnumConnectionPoints interface 
                                  // pointer
)
{
	INTERFACE_METHOD_START( "IEnumConnectionPoints::Clone" );
	
	if(ppEnum == NULL)
		return E_POINTER;

	if (FAILED((*ppEnum) = new CImpIEnumConnectionPoints(m_pObj)))
		return E_OUTOFMEMORY;
	
	(*(PIMPIENUMCONNECTIONPOINTS*)ppEnum) -> m_pParentContainer = m_pParentContainer;
	(*(PIMPIENUMCONNECTIONPOINTS*)ppEnum) -> m_pCurrentCP = m_pCurrentCP;
	(*ppEnum) -> AddRef();
	
	return S_OK;

	INTERFACE_METHOD_END();
}

// IConnectionPoint specific methods -----------------------------------------

// CImpIConectionPoint::GetConnectionPointContainer
//
// @mfunc Retrieves the IConnectionPointContainer interface pointer for the parent
//		  connectable object.
//
// @rdesc NONE
//
STDMETHODIMP CImpIConnectionPoint::GetConnectionPointContainer
(
	IConnectionPointContainer **ppCPC	// Address of output variable that receives
										// the IConnectionPointContainer interface pointer
)
{
	INTERFACE_METHOD_START( "IConnectionPoint::GetConnectionPointContainer" );
	
	if (ppCPC == NULL)
		return E_POINTER;

	*ppCPC = m_pParentContainer;
	// Don Box's schema
	(*ppCPC)->AddRef();
	
	return S_OK;

	INTERFACE_METHOD_END();
}

// CImpIConnectionPoint::GetConnectionInterface
//
// @mfunc Returns the IID of the outgoing interface managed by this connection point.
//
// @rdesc NONE
//
STDMETHODIMP CImpIConnectionPoint::GetConnectionInterface
(
	IID *pIID  //Pointer to an IID variable
)
{
	INTERFACE_METHOD_START( "IConnectionPoint::GetConnectionInterface" );

	if (pIID == NULL)
		return E_POINTER;

	assert(m_iid == IID_IRowsetNotify);
	*pIID = m_iid;
	
	return S_OK;
	
	INTERFACE_METHOD_END();
}

// CImpIConnectionPoint::Advise
//
// @mfunc Establishes a connection between the connection point object and the client's sink.
//
// @rdesc NONE
//
STDMETHODIMP CImpIConnectionPoint::Advise
(
	IUnknown *pUnk,  //Pointer to the client's advise sink
	DWORD *pdwCookie //Pointer to the connection point identifier used by Unadvise
)
{
	INTERFACE_METHOD_START( "IConnectionPoint::Advise" );

	HRESULT hr = (S_OK);
	DWORD i;

	if ((pUnk == NULL) || (pdwCookie == NULL))
		return E_POINTER;

	// Find free connection point
	for (i = 0; i < ADVISELIMIT; i++)
		if (m_pObj->m_Connections[i] == NULL)
			break;

	if (i == ADVISELIMIT)
		return CONNECT_E_ADVISELIMIT;

	// Don Box recommends more complex call schema
	// m_pObj->m_Connections[i] = pUnk;
	// *pdwCookie = i;
 	hr = pUnk->QueryInterface(IID_IRowsetNotify, (void**)&(m_pObj->m_Connections[i]));
	
	if (hr == E_NOINTERFACE)
		hr = CONNECT_E_NOCONNECTION;

	if (SUCCEEDED(hr))
		*pdwCookie = (i+1); // *pdwCookie > 0
	
	return S_OK;

	INTERFACE_METHOD_END();
}

// CImpIConnectionPoint::Unadvise
//
// @mfunc Terminates an advisory connection previously established through
//		  IConnectionPoint::Advise. The dwCookie parameter identifies the
//		  connectionto terminate.
//
// @rdesc NONE
//
STDMETHODIMP	CImpIConnectionPoint::Unadvise
(
	DWORD dwCookie  //Connection token
)
{
	INTERFACE_METHOD_START( "IConnectionPoint::Unadvise" );
	
	if (dwCookie > ADVISELIMIT)
		return CONNECT_E_NOCONNECTION;	// Wrong cookie
	
	// Release the connection
	// Don Box's calling schema
	// m_pObj->m_Connections[dwCookie] = NULL;
	(m_pObj->m_Connections[dwCookie - 1])->Release();
	m_pObj->m_Connections[dwCookie - 1] = NULL;
	
	return S_OK;

	INTERFACE_METHOD_END();
}

// CImpIConnectionPoint::EnumConnections
//
// @mfunc Returns an object to enumerate the current advisory connections for
//		  this connection point.
//
// @rdesc

STDMETHODIMP CImpIConnectionPoint::EnumConnections
(
  IEnumConnections **ppEnum  // Address of output variable that 
                             // receives the IEnumConnections interface 
                             // pointer
)
{
	INTERFACE_METHOD_START( "IConnectionPoint::EnumConnections" );
	
	if (ppEnum == NULL)
		return E_POINTER;

	if (FAILED((*ppEnum) = new CImpIEnumConnections(m_pObj)))
		return E_OUTOFMEMORY;
	
	(*ppEnum)->AddRef();

	return S_OK;
	
	INTERFACE_METHOD_END();
}

// IEnumConnections specific methods

// CImpIEnumConnections::Next
//
// @mfunc
//
// @desc
//
STDMETHODIMP CImpIEnumConnections::Next
(
  ULONG cConnections,  //[in]Number of CONNECTDATA structures 
                       // returned in rgpcd
  CONNECTDATA *rgcd,   //[out]Array to receive enumerated 
                       // CONNECTDATA structures
  ULONG *pcFetched     //[out]Pointer to actual number of 
                       // CONNECTDATA structures
)
{
	INTERFACE_METHOD_START( "IEnumConnections::Next" );

	if (pcFetched == NULL || rgcd == NULL)
		return E_POINTER;

//	cConnections = (m_CurrentConnection + cConnections <= ADVISELIMIT) ?
//		cConnections : ADVISELIMIT - m_CurrentConnection;

	ULONG i = 0;
	while (i < cConnections && m_CurrentConnection < ADVISELIMIT)
	{
		if (m_pObj->m_Connections[m_CurrentConnection])
		{
			(rgcd[i].pUnk = m_pObj->m_Connections[m_CurrentConnection])->AddRef();
			rgcd[i].dwCookie = m_CurrentConnection;
			i++;
		}
		m_CurrentConnection++;
	}

	*pcFetched = i;

	return S_OK;

	INTERFACE_METHOD_END();
}
 
STDMETHODIMP CImpIEnumConnections::Skip
(
  ULONG cConnections  //[in]Number of elements to skip
)
{
	m_CurrentConnection = ((m_CurrentConnection + cConnections) < ADVISELIMIT) ?
		(m_CurrentConnection + cConnections) : ADVISELIMIT;

	return S_OK;
}
 
STDMETHODIMP CImpIEnumConnections::Reset(void)
{
	m_CurrentConnection = 0;
	return S_OK;
}
 
STDMETHODIMP CImpIEnumConnections::Clone
(
  IEnumConnections **ppEnum  //[out]Address of output variable that 
                             // receives the IEnumConnections interface 
                             // pointer
)
{
	INTERFACE_METHOD_START( "IEnumConnections::Clone" );

	if (ppEnum == NULL)
		return E_POINTER;

	if (FAILED((*ppEnum) = new CImpIEnumConnections(m_pObj)))
		return E_OUTOFMEMORY;

	(*(CImpIEnumConnections **)ppEnum)->m_CurrentConnection = m_CurrentConnection;
	(*ppEnum)->AddRef();

	return S_OK;
	
	INTERFACE_METHOD_END();
}
 