/*
 * Copyright (c)  2000
 * SWsoft  company
 *
 * This material is provided "as is", with absolutely no warranty expressed
 * or implied. Any use is at your own risk.
 *
 * Permission to use or copy this software for any purpose is hereby granted 
 * without fee, provided the above notices are retained on all copies.
 * Permission to modify the code and to distribute modified code is granted,
 * provided the above notices are retained, and a notice that the code was
 * modified is included with the above copyright notice.
 *
 */

//--------------------------------------------------------------------
// MySQL OLE DB Provider 
// Functionality: minimum
// Release: 0.1
//
// @doc
//
// @module COLINFO.CPP | IColumnsInfo interface implementation
//

// Includes ------------------------------------------------------------------

#include "hfiles.h"
#include "headers.h"

// Code ----------------------------------------------------------------------

//  IColumnsInfo specific methods

// CImpIColumnsInfo::GetColumnInfo -------------------------------------------
//
// @mfunc Returns the column metadata needed by most consumers.
//
// @rdesc HRESULT
//      @flag S_OK              | The method succeeded
//      @flag E_OUTOFMEMORY     | Out of memory
//      @flag E_INVALIDARG      | pcColumns or prginfo or ppStringsbuffer was NULL
//
STDMETHODIMP CImpIColumnsInfo::GetColumnInfo
    (
    ULONG*          pcColumns,      //@parm OUT | Number of columns in rowset
    DBCOLUMNINFO**  prgInfo,        //@parm OUT | Array of DBCOLUMNINFO Structures
    WCHAR**         ppStringsBuffer //@parm OUT | Storage for all string values
    )
{
    INTERFACE_METHOD_START( "IColumnsInfo::GetColumnInfo" );

	DWORD			icol;
	ULONG           icolStart;
    DBCOLUMNINFO*   rgdbcolinfo;
    WCHAR*          pstrBuffer;
	ULONG			cTotCols;
	ULONG			cChars;

	// Initialize
	if (pcColumns)
		*pcColumns = 0;
	if (prgInfo)
		*prgInfo = NULL;
	if (ppStringsBuffer)
		*ppStringsBuffer = NULL;

    // Usual argument checking, prescribed by the spec.
    if (pcColumns == NULL || prgInfo == NULL || ppStringsBuffer == NULL)
        return E_INVALIDARG;

	//Checking Command object status
	if (m_pCommand != NULL)
	{
		if (!m_pCommand->IsTextSet())
			return DB_E_NOCOMMAND;

		if (!m_pCommand->IsPrepared())
			return DB_E_NOTPREPARED;

		HRESULT hr = m_pCommand->FillColumnInfo(NULL);
		if (hr != S_OK)
			return hr;
	}
	
	if (*m_pbBookmark)
	{
		icolStart = 0;
		cTotCols = *m_pcCols;
	}
	else
	{
		icolStart   = 1;
		cTotCols = *m_pcCols - 1;
	}

    rgdbcolinfo = (DBCOLUMNINFO *) g_pIMalloc->Alloc( cTotCols*sizeof( DBCOLUMNINFO ));
    if (rgdbcolinfo == NULL)
        return E_OUTOFMEMORY;

    memcpy( rgdbcolinfo, &((*m_ppColInfo)[icolStart]), cTotCols*sizeof( DBCOLUMNINFO ));

	//Count total number of bytes that is needed for string buffer
	DWORD cBytes = 0;
	for (icol = icolStart; icol < cTotCols; icol++)
	{
		cChars = wcslen((*m_ppColInfo)[icol].pwszName) + 1;
		//Temporary storage for length (old bad style)
		rgdbcolinfo[icol].pwszName = (WCHAR*)cChars;
		//Count bytes
		cBytes += cChars * sizeof(WCHAR);
	}

	//Create string buffer
	pstrBuffer = (WCHAR *) g_pIMalloc->Alloc(cBytes);

	//Fill out the string buffer
	WCHAR* pstrTemp = pstrBuffer;

	for (icol = icolStart; icol < cTotCols; icol++)
	{
		cChars = (ULONG)rgdbcolinfo[icol].pwszName;
		wcscpy(pstrTemp, (*m_ppColInfo)[icol].pwszName);
		//pstrTemp[cChars] = 0;
		rgdbcolinfo[icol].pwszName = pstrTemp;
		TRACE2("   %S", rgdbcolinfo[icol].pwszName);
		rgdbcolinfo[icol].columnid.eKind = DBKIND_NAME;
		rgdbcolinfo[icol].columnid.uName.pwszName = rgdbcolinfo[icol].pwszName;
		pstrTemp += cChars;
	}

    *prgInfo     = rgdbcolinfo;
    *ppStringsBuffer = pstrBuffer;
    *pcColumns   = cTotCols;
	
	return S_OK;

    INTERFACE_METHOD_END();
}

// CImpIColumnsInfo::MapColumnIDs --------------------------------------------
//
// @mfunc Returns an array of ordinals of the columns in a rowset that are
// identified by the specified column IDs.
//
// @rdesc HRESULT
//      @flag S_OK                      | The method succeeded
//      @flag E_INVALIDARG              | cColumnIDs was not 0 and rgColumnIDs was NULL,
//                                        rgColumns was NULL
//      @flag DB_E_COLUMNUNAVAILABLE    | An element of rgColumnIDs was invalid
//
STDMETHODIMP CImpIColumnsInfo::MapColumnIDs
    (
    ULONG           cColumnIDs,     //@parm IN | Number of Column IDs to map
    const DBID		rgColumnIDs[],  //@parm IN | Column IDs to map
    ULONG           rgColumns[]     //@parm OUT | Ordinal values
    )
{
	INTERFACE_METHOD_START( "IColumnsInfo::MapColumnIDs" );

	ULONG	ulError = 0;
	
    // If cColumnIDs is 0 return
	if (0 == cColumnIDs)
        return S_OK;

    // Check arguments
    if (cColumnIDs != 0 && NULL == rgColumnIDs)
        return E_INVALIDARG;

    if (NULL == rgColumns)
        return E_INVALIDARG;

	//Checking Command object status
	if (m_pCommand != NULL)
	{
		if (!m_pCommand->IsTextSet())
			return DB_E_NOCOMMAND;

		if (!m_pCommand->IsPrepared())
			return DB_E_NOTPREPARED;

		HRESULT hr = m_pCommand->FillColumnInfo(NULL);
		if (hr != S_OK)
			return hr;
	}

    // Walk the Column ID structs and determine
    // the ordinal value
	ULONG j;
    for (ULONG i = 0; i < cColumnIDs; i++)
    {
		switch (rgColumnIDs[i].eKind)
		{
		case DBKIND_GUID_PROPID:
			if (rgColumnIDs[i].uGuid.guid != GUID_NULL)
			{
				TRACE( "   IColumnsInfo::MapColumnIDs - columns by GUID - failed");
				break; //go to the error trap
			}
			//else go to next case
		case DBKIND_PROPID:
	        if (rgColumnIDs[i].uName.ulPropid < 0  ||
				rgColumnIDs[i].uName.ulPropid > *m_pcCols - 1)
			{
				rgColumns[i] = rgColumnIDs[i].uName.ulPropid + 1;
			}
			TRACE( "   IColumnsInfo::MapColumnIDs - columns by propId");// %d", (int)rgColumnIDs[i].uName.ulPropid );
			continue; //Success

		case DBKIND_GUID_NAME:
			if (rgColumnIDs[i].uGuid.guid != GUID_NULL)
			{
				TRACE( "   IColumnsInfo::MapColumnIDs - columns by GUID - failed");
				break; //go to the error trap
			}
			//else go to next case
		case DBKIND_NAME:
			assert(m_pcCols != NULL);
			assert(m_ppColInfo != NULL);
			
			//Look for name
			bool bFound = false;
			for (j = 0; j < *m_pcCols; j++)
			{
				if (0 == wcsicmp((*m_ppColInfo)[j].pwszName, rgColumnIDs[i].uName.pwszName))
				{
					//We have found it
					rgColumns[i] = j;
					bFound = true;
					break;
				}
			}
		
#ifdef _DEBUG
			if( bFound )
				TRACE( "   IColumnsInfo::MapColumnIDs - column with name %S has ordinal #%d", (*m_ppColInfo)[j].pwszName, j );
			else
				TRACE( "   IColumnsInfo::MapColumnIDs - column with name %S is not found", (*m_ppColInfo)[j].pwszName );
#endif

			if (bFound)
				continue; //Success
			else
				break;	//So we failed to find name
		}

        rgColumns[i] = DB_INVALIDCOLUMN;
        ulError++;
    }
    
	if (ulError == 0)
		return S_OK;

	if (ulError < cColumnIDs)
        return DB_S_ERRORSOCCURRED;

	return DB_E_ERRORSOCCURRED;

	INTERFACE_METHOD_END();
}
