/*
 * Copyright (c)  2000
 * SWsoft  company
 *
 * This material is provided "as is", with absolutely no warranty expressed
 * or implied. Any use is at your own risk.
 *
 * Permission to use or copy this software for any purpose is hereby granted 
 * without fee, provided the above notices are retained on all copies.
 * Permission to modify the code and to distribute modified code is granted,
 * provided the above notices are retained, and a notice that the code was
 * modified is included with the above copyright notice.
 *
 */

//--------------------------------------------------------------------
// MySQL OLE DB Provider 
// Functionality: minimum
// Release: 0.1
//
// @doc
//
// @module ACCESSOR.CPP | CImpIAccessor object implementation
//
//
//

// Includes ------------------------------------------------------------------

#include "hfiles.h"
#include "headers.h"
#include <MSDADC.H>

#define DEFAULT_CBMAXLENGTH 40	// cbMaxLength for binding

// Code ----------------------------------------------------------------------

// IAccessor specific methods

// CImpIAccessor::AddRefAccessor -----------------------------------------
//
// @mfunc Adds a reference count to an existing accessor
//
// @rdesc HRESULT
//      @flag S_OK                      | Method Succeeded
//      @flag E_FAIL                    | Provider specific Error
//
STDMETHODIMP CImpIAccessor::AddRefAccessor
    (
	HACCESSOR	hAccessor,		//@parm IN | Accessor Handle
	ULONG*		pcRefCounts		//@parm OUT | Reference Count
    )
{
    INTERFACE_METHOD_START( "IAccessor::AddRefAccessor" );
	
	// Retrieve our accessor structure from the client's hAccessor,
    // free it, then mark accessor ptr as unused.
    // We do not re-use accessor handles.  This way, we hope
    // to catch more client errors.  (Also, ExtBuffer doesn't
    // maintain a free list, so it doesn't know how to.)

    PACCESSOR   pAccessor;
    HRESULT     hr;

    if( pcRefCounts )
		*pcRefCounts = 0;

	hr = m_pExtBufferAccessor->GetItemOfExtBuffer((ULONG) hAccessor, &pAccessor );
    if (FAILED( hr ) || pAccessor == NULL)
        return DB_E_BADACCESSORHANDLE;

	InterlockedIncrement(&(pAccessor->cRef));

	if( pcRefCounts )
		*pcRefCounts = (ULONG)(pAccessor->cRef);

	return S_OK;

	INTERFACE_METHOD_END();
}


// CImpIAccessor::CreateAccessor -----------------------------------------
//
// @mfunc Creates a set of bindings that can be used to send data
// to or retrieve data from the data cache.
// NOTE:  Currently does not support returning rgStatus[].
//
// @rdesc HRESULT
//      @flag S_OK                      | Method Succeeded
//      @flag E_FAIL                    | Provider specific Error
//      @flag E_INVALIDARG              | pHAccessor was NULL, dwAccessorFlags was
//                                        invalid, or cBindings was not 0 and
//                                        rgBindings was NULL
//      @flag E_OUTOFMEMORY             | Out of Memory
//      @flag DB_E_ERRORSOCCURRED		| dwBindPart in an rgBindings element was invalid, OR
//									    | Column number specified was out of range, OR
//										| Requested coercion is not supported.
//      @flag OTHER                     | Other HRESULTs returned by called functions
//
STDMETHODIMP CImpIAccessor::CreateAccessor
    (
    DBACCESSORFLAGS dwAccessorFlags,
    ULONG           cBindings,      //@parm IN | Number of Bindings
    const DBBINDING rgBindings[],   //@parm IN | Array of DBBINDINGS
    ULONG           cbRowSize,      //@parm IN | Number of bytes in consumer's buffer
    HACCESSOR*      phAccessor,     //@parm OUT | Accessor Handle
	DBBINDSTATUS	rgStatus[]		//@parm OUT	| Binding status
    )
{
	INTERFACE_METHOD_START("IAccessor::CreateAccessor");

    PACCESSOR   pAccessor;
    ULONG       hAccessor;
    ULONG       ibind;
    ULONG       icol;
    HRESULT     hr;
	HRESULT		hrCanConvertF = S_OK;
	HRESULT		hrCanConvertB = S_OK;

    if ((cBindings && !rgBindings) || phAccessor == NULL)
        return E_INVALIDARG;

    // init out params
    *phAccessor = (HACCESSOR) 0;

	//Check reentrantness
	if (m_pRowset != NULL && m_pRowset->m_bNotReentrant)
		return DB_E_NOTREENTRANT;

    // Check null accessor
	if (cBindings == 0 && (m_pRowset == NULL || m_pRowset->m_pIRowsetChange == NULL)
//		&& 0 == (dwAccessorFlags & DBACCESSOR_PARAMETERDATA) // To support null accessors on commands with parameter
		) return DB_E_NULLACCESSORNOTSUPPORTED;

    // Check for the binding types we don't accept.
    if (dwAccessorFlags & DBACCESSOR_PASSBYREF)
		return DB_E_BYREFACCESSORNOTSUPPORTED;
    
    // .. then check for the binding type that is required
    if (0 != (dwAccessorFlags & ~(DBACCESSOR_ROWDATA | DBACCESSOR_PARAMETERDATA | DBACCESSOR_OPTIMIZED)) || 
		0 == (dwAccessorFlags & (DBACCESSOR_ROWDATA | DBACCESSOR_PARAMETERDATA)) ||
		(m_pRowset != NULL && 0 != (dwAccessorFlags & DBACCESSOR_PARAMETERDATA)))
        return DB_E_BADACCESSORFLAGS;

	// Attempt to create null accessor?
	DBBINDING *pBindings = (DBBINDING *)rgBindings;
	bool bNullAccessor = cBindings == 0;
	if( bNullAccessor )
	{
		// Add all columns from rowset
		pBindings = new DBBINDING[ m_pRowset->m_cCols ];
		
		if( pBindings == NULL )
			return E_OUTOFMEMORY;

		// Clear allocated memory
		ZeroMemory( pBindings, m_pRowset->m_cCols * sizeof(DBBINDING) );

		// Set up every binding information based on column infromation
		ULONG dwOffset = 0;
		for( int iCol = 0, iBind = 0; iCol < m_pRowset->m_cCols; iCol++ )
		{
			// Skip columns of type _VECTOR. 
			if (m_pRowset->m_rgdbcolinfo[ iCol ].wType & DBTYPE_VECTOR)
				continue;

			pBindings[ iBind ].bPrecision = m_pRowset->m_rgdbcolinfo[ iCol ].bPrecision;
			pBindings[ iBind ].bScale	  = m_pRowset->m_rgdbcolinfo[ iCol ].bScale;
			pBindings[ iBind ].cbMaxLen	= m_pRowset->m_rgdbcolinfo[ iCol ].ulColumnSize; 
			pBindings[ iBind ].dwMemOwner = DBMEMOWNER_CLIENTOWNED;
			pBindings[ iBind ].dwPart	  = DBPART_VALUE | DBPART_LENGTH | DBPART_STATUS;
			pBindings[ iBind ].eParamIO	  = DBPARAMIO_NOTPARAM;
			pBindings[ iBind ].iOrdinal	  = m_pRowset->m_rgdbcolinfo[ iCol ].iOrdinal;
			pBindings[ iBind ].obLength	  = dwOffset + offsetof(COLUMNDATA,dwLength);
			pBindings[ iBind ].obStatus   = dwOffset + offsetof(COLUMNDATA,dwStatus);
			pBindings[ iBind ].obValue	  = dwOffset + offsetof(COLUMNDATA,bData);
			pBindings[ iBind ].pTypeInfo  = m_pRowset->m_rgdbcolinfo[ iCol ].pTypeInfo;
			pBindings[ iBind ].wType	  = m_pRowset->m_rgdbcolinfo[ iCol ].wType;
			
			dwOffset += pBindings[ iBind ].cbMaxLen + offsetof( COLUMNDATA, bData );
			dwOffset = ROUND_UP( dwOffset, COLUMN_ALIGNVAL );

			iBind++; //  Go to next binding
		}

		// set up bindings count. May differ with iCol
		cBindings = iBind;
	}
	
	// Check on the bindings the user gave us.
	bool bFailed = false;
	for (ibind =0; ibind < cBindings; ibind++)
	{
		icol = pBindings[ibind].iOrdinal;

		// At least one of these valid parts has to be set. In SetData I assume it is the case.
		if (pBindings[ibind].dwPart > DBPART_VALUE + DBPART_LENGTH + DBPART_STATUS ||
			pBindings[ibind].dwPart == 0)
		 {
			TRACE3( "  CreateAccessor failure: binding %d, column %d, _VALUE, _LENGTH, _STATUS not specified", ibind, icol );

			//  Set Bind status to DBBINDSTATUS_BADBINDINFO
			if (rgStatus != NULL)
				rgStatus[ibind] = DBBINDSTATUS_BADBINDINFO;

			bFailed = true;
			continue;
		}

		// ensure that binding type is correct
		if (pBindings[ibind].wType == DBTYPE_EMPTY ||
			pBindings[ibind].wType == DBTYPE_NULL ||
			pBindings[ibind].wType == (DBTYPE_BYREF | DBTYPE_EMPTY) || 
			pBindings[ibind].wType == (DBTYPE_BYREF | DBTYPE_NULL) ||
			pBindings[ibind].wType == (DBTYPE_BYREF | DBTYPE_RESERVED) ||
			((pBindings[ibind].wType & (DBTYPE_BYREF | DBTYPE_ARRAY)) != 0 && 
					(pBindings[ibind].wType & (DBTYPE_BYREF | DBTYPE_ARRAY)) == (DBTYPE_BYREF | DBTYPE_ARRAY)) ||
			((pBindings[ibind].wType & (DBTYPE_BYREF | DBTYPE_VECTOR)) != 0 && 
					(pBindings[ibind].wType & (DBTYPE_BYREF | DBTYPE_VECTOR)) == (DBTYPE_BYREF | DBTYPE_VECTOR)) ||
			((pBindings[ibind].wType & (DBTYPE_VECTOR | DBTYPE_ARRAY)) != 0 && 
					(pBindings[ibind].wType & (DBTYPE_VECTOR | DBTYPE_ARRAY)) == (DBTYPE_VECTOR | DBTYPE_ARRAY)) ||
			(pBindings[ibind].dwMemOwner == DBMEMOWNER_PROVIDEROWNED && 
					!(pBindings[ibind].wType & (DBTYPE_BYREF | DBTYPE_VECTOR | DBTYPE_ARRAY)) &&
					(~DBTYPE_BYREF & pBindings[ibind].wType) != DBTYPE_BSTR) ||
			((~DBTYPE_BYREF & pBindings[ibind].wType) == DBTYPE_IUNKNOWN &&
					m_pRowset != NULL && 
					m_pRowset->m_rgdbcolinfo[icol].wType != DBTYPE_STR))
		{
			TRACE3( "  CreateAccessor failure: binding %d, column %d, bad type info", ibind, icol );

			if (rgStatus != NULL)
				rgStatus[ibind] = DBBINDSTATUS_BADBINDINFO;
			
			bFailed = true;
			continue;
		}

		//Only for parameter accessor
		if (dwAccessorFlags & DBACCESSOR_PARAMETERDATA)
		{
			//Parameter ordinal can not be 0
			if (icol == 0)
			{
				TRACE3( "  CreateAccessor failure: binding %d, column %d, parameter ordinal == 0\n", ibind, icol );

				if (rgStatus != NULL)
					rgStatus[ibind] = DBBINDSTATUS_BADORDINAL;
				
				bFailed = true;
				continue;
			}

			//Check ParamIO values
			if (pBindings[ibind].eParamIO != DBPARAMIO_INPUT)
			{
				TRACE3( "  CreateAccessor failure: binding %d, column %d, invalid eParamIO\n", ibind, icol );

				if (rgStatus != NULL)
					rgStatus[ibind] = DBBINDSTATUS_BADBINDINFO;
				
				bFailed = true;
				continue;
			}

			if (pBindings[ibind].dwMemOwner == DBMEMOWNER_PROVIDEROWNED)
			{
				TRACE3( "  CreateAccessor failure: binding %d, column %d, provided-owned memory flag for parameter0\n", ibind, icol );

				if (rgStatus != NULL)
					rgStatus[ibind] = DBBINDSTATUS_BADBINDINFO;
				
				bFailed = true;
				continue;
			}

			//Check if there is another binding with the same ordinal for input params
			bool bFound = false;
			for (int j = 0; j < ibind; j++)
			{
				if (pBindings[j].iOrdinal == icol)
				{
					bFound = true;
					break;
				}
			}
			
			if (bFound)
			{
				TRACE3( "  CreateAccessor failure: binding %d, column %d, the same ordinal\n", ibind, icol );

				if (rgStatus != NULL)
					rgStatus[ibind] = DBBINDSTATUS_BADBINDINFO;
				
				bFailed = true;
				continue;
			}
		}

		//Check if column is already binded in existing accessors
		//Only for optimized accessors
		if (dwAccessorFlags & DBACCESSOR_OPTIMIZED)
		{
			hAccessor = 1;
			bool bFound = false;
			ULONG hFirst, hLast;
			m_pExtBufferAccessor->GetFirstLastItemH(hFirst, hLast);
			for (hAccessor = hFirst; hAccessor <= hLast; hAccessor++)
			{
				//Extract pointer to accessor
				pAccessor = *(PACCESSOR*)(*m_pExtBufferAccessor)[hAccessor];

				//Check if it is non-empty accessor
				if (pAccessor == NULL)
					continue;

				//Only for optimized accessors
				if (!(pAccessor->dwAccessorFlags & DBACCESSOR_OPTIMIZED))
					continue;

				//Walk through all bindings 
				for (ULONG i = 0; i < pAccessor->cBindings; i++)
				{
					//Check for columns
					if (pAccessor->rgBindings[i].iOrdinal == icol)
					{
						bFound = true;
						break;
					}
				}
			}

			if (bFound)
			{
				TRACE3( "  CreateAccessor failure: binding %d, column %d, column is already bound\n", ibind, icol );
				
				if (rgStatus != NULL)
					rgStatus[ibind] = DBBINDSTATUS_BADBINDINFO;
				
				bFailed = true;
				continue;
			}
		}


		if (m_pRowset != NULL)
		{
			// make sure column number is in range
			if (icol < 0 || icol >= m_pRowset->m_cCols)
			{
				TRACE3( "  CreateAccessor failure: binding %d, bad column number %d\n", ibind, icol );
			
				if (rgStatus != NULL)
					rgStatus[ibind] = DBBINDSTATUS_BADORDINAL;

				bFailed = true;
				continue;
			}

			// Make sure we can do the conversion that is requested
			hrCanConvertF = g_pIDataConvert->CanConvert(pBindings[ibind].wType, m_pRowset->m_rgdbcolinfo[icol].wType); 
			hrCanConvertB = g_pIDataConvert->CanConvert(m_pRowset->m_rgdbcolinfo[icol].wType, pBindings[ibind].wType); 
			if (NOERROR != hrCanConvertF || NOERROR != hrCanConvertB )
			{
				// Note: DBTYPE_UI8 is unsupported by CanConvert
				//  Set Bind status to DBBINDSTATUS_UNSUPPORTEDCONVERSION
				if (rgStatus != NULL)
					rgStatus[ibind] = DBBINDSTATUS_UNSUPPORTEDCONVERSION;
			}
		}
		
		//  successfull binding 
		//  Set Bind status to DBBINDSTATUS_OK
		if (rgStatus != NULL)
			rgStatus[ibind] = DBBINDSTATUS_OK;
	}

	if (bFailed)
	{
		if( bNullAccessor )
			delete pBindings;
		return DB_E_ERRORSOCCURRED;
	}

    // Make a copy of the client's binding array, and the type of binding.
    // Note: accessors with no bindings (cBindings == 0) are legal.
    pAccessor = (ACCESSOR*) new BYTE[sizeof(ACCESSOR) + cBindings * sizeof(DBBINDING)];
    if (pAccessor == NULL)
	{
		if( bNullAccessor )
			delete pBindings;
        return E_OUTOFMEMORY;
	}

    // We store a ptr to the newly created variable-sized ACCESSOR.
    // We have an array of ptrs (to ACCESSOR's).
    // The handle is the index into the array of ptrs.
    // The InsertIntoExtBuffer function appends to the end of the array.
    assert(m_pExtBufferAccessor);
    hr = m_pExtBufferAccessor->InsertIntoExtBuffer(&pAccessor, hAccessor);
    if (FAILED(hr))
    {
		if( bNullAccessor )
			delete pBindings;
        delete [] pAccessor;
        return E_OUTOFMEMORY;
    }
    assert( hAccessor );

    // Copy the client's bindings into the ACCESSOR.
    pAccessor->dwAccessorFlags	= dwAccessorFlags;
    pAccessor->cBindings		= cBindings;
	pAccessor->bNullAccessor	= bNullAccessor;
    pAccessor->cRef				= 1;		// Establish Reference count.
	memcpy( &(pAccessor->rgBindings[0]), &pBindings[0], cBindings*sizeof( DBBINDING ));

    // fill out-param and return
    *phAccessor = (HACCESSOR) hAccessor;

	// Clean up
	if( bNullAccessor )
		delete pBindings;

	return S_OK;
    
	INTERFACE_METHOD_END();
}


// CImpIAccessor::GetBindings --------------------------------------------------
//
// @mfunc Returns the bindings in an accessor
//
// @rdesc HRESULT
//      @flag S_OK                      | Method Succeeded
//      @flag E_INVALIDARG              | pdwAccessorFlags/pcBinding/prgBinding were NULL
//      @flag E_OUTOFMEMORY             | Out of Memory
//      @flag DB_E_BADACCESSORHANDLE    | Invalid Accessor given
//
STDMETHODIMP CImpIAccessor::GetBindings
    (
    HACCESSOR        hAccessor,         //@parm IN | Accessor Handle
    DBACCESSORFLAGS* pdwAccessorFlags,  //@parm OUT | Binding Type flag
    ULONG*           pcBindings,        //@parm OUT | Number of Bindings returned
    DBBINDING**      prgBindings        //@parm OUT | Bindings
    )
{
	INTERFACE_METHOD_START("IAccessor::GetBindings");

    // Retrieve our accessor structure from the client's hAccessor,
    // make a copy of the bindings for the user, then done.
    PACCESSOR   pAccessor;
    ULONG       cBindingSize;
    HRESULT     hr;

    // init out-params	
	if (pdwAccessorFlags != NULL)
		*pdwAccessorFlags = DBACCESSOR_INVALID;

	if (pcBindings != NULL)
		*pcBindings = 0;

    if (prgBindings != NULL)
		*prgBindings = NULL;

    // check parameters
    if (pdwAccessorFlags == NULL || pcBindings == NULL || prgBindings == NULL)
        return E_INVALIDARG;

    // Validate Accessor Handle
    hr = m_pExtBufferAccessor->GetItemOfExtBuffer((ULONG)hAccessor, &pAccessor);
    if (FAILED( hr ) || pAccessor == NULL)
        return DB_E_BADACCESSORHANDLE;

	// Check is the accessor is null one
	if( pAccessor->bNullAccessor )
	{
		// No information will be returned
		*pcBindings = 0;
		*prgBindings = NULL;
	}
	else
	{
		// Allocate and return Array of bindings
		cBindingSize = pAccessor->cBindings * sizeof(DBBINDING);
		if( cBindingSize )
		{
			*prgBindings = (DBBINDING*)g_pIMalloc->Alloc(cBindingSize);
			if (*prgBindings == NULL)
				return E_OUTOFMEMORY;

			memcpy(*prgBindings, pAccessor->rgBindings, cBindingSize);
		}
		else
		   *prgBindings = NULL;	 // For NULL Accessors

		*pdwAccessorFlags = pAccessor->dwAccessorFlags;
		*pcBindings = pAccessor->cBindings;
	}

	return S_OK;
    
	INTERFACE_METHOD_END()
}



// CImpIAccessor::ReleaseAccessor ---------------------------------------
//
// @mfunc Releases an Accessor
//
// @rdesc HRESULT
//      @flag S_OK                      | Method Succeeded
//      @flag DB_E_BADACCESSORHANDLE    | hAccessor was invalid
//
STDMETHODIMP CImpIAccessor::ReleaseAccessor
    (
    HACCESSOR	hAccessor,      //@parm IN | Accessor handle to release
	ULONG*		pcRefCounts		//@parm OUT | Reference Count
    )
{
    INTERFACE_METHOD_START( "IAccessor::ReleaseAccessor" );
	// Retrieve our accessor structure from the client's hAccessor,
    // free it, then mark accessor ptr as unused.
    // We do not re-use accessor handles.  This way, we hope
    // to catch more client errors.  (Also, ExtBuffer doesn't
    // maintain a free list, so it doesn't know how to.)

    PACCESSOR   pAccessor;
    HRESULT     hr;

	if( pcRefCounts )
		*pcRefCounts = 0;

    hr = m_pExtBufferAccessor->GetItemOfExtBuffer((ULONG) hAccessor, &pAccessor );
    if (FAILED( hr ) || pAccessor == NULL)
        return DB_E_BADACCESSORHANDLE;

    // Free the actual structure.
	InterlockedDecrement(&(pAccessor->cRef));
	assert( pAccessor->cRef >= 0 );
	if( pAccessor->cRef <= 0 )
	{
		delete [] pAccessor;
		if( pcRefCounts )
			*pcRefCounts = 0;

	    // Store a null in our array-of-ptrs,
	    // so we know next time that it is invalid.
	    // (operator[] returns a ptr to the space where the ptr is stored.)
	    *(PACCESSOR*) ((*m_pExtBufferAccessor)[(ULONG)hAccessor]) = NULL;
	}
	else
	{
		if( pcRefCounts )
			*pcRefCounts = (ULONG)(pAccessor->cRef);
	}
	
	return S_OK;

    INTERFACE_METHOD_END()
}

