/* Copyright (C) 2006 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

// Transaction.h: interface for the Transaction class.
//
//////////////////////////////////////////////////////////////////////

#if !defined(AFX_TRANSACTION_H__02AD6A4D_A433_11D2_AB5B_0000C01D2301__INCLUDED_)
#define AFX_TRANSACTION_H__02AD6A4D_A433_11D2_AB5B_0000C01D2301__INCLUDED_

#if _MSC_VER >= 1000
#pragma once
#endif // _MSC_VER >= 1000

#include "SyncObject.h"

static const int NO_TRANSACTION = 0;

class RecordVersion;
class Database;
class Transaction;
class Connection;
class Table;
class SavePoint;
class IO;
class DeferredIndex;
class Index;
class Bitmap;

// Transaction States

enum State {
	Active,
	Limbo,
	Committed,
	RolledBack,

	// The following are 'relative states'.  See getRelativeState()
	
	Us,
	CommittedAndOlder,
	CommittedButYounger,
	WasActive,
	Deadlock,
	
	// And the remaining are for transactions pending reuse
	
	Available,
	Initializing
	};

struct TransState {
	Transaction	*transaction;
	TransId		transactionId;
	int			state;
	};

struct Savepoint {
	int			id;
	RecordVersion	**records;
	};

class Transaction
{
public:
	Transaction(Connection *connection, TransId seq);

	State		getRelativeState (Transaction *transaction, TransId transId);
	void		removeRecord (RecordVersion *record);
	void		expungeTransaction (Transaction *transaction);
	void		commitRecords();
	void		releaseDependencies();
	bool		visible (Transaction *transaction);
	void		addRecord (RecordVersion *record);
	void		prepare(int xidLength, const UCHAR *xid);
	void		rollback();
	void		commit();
	void		release();
	void		addRef();
	void		waitForTransaction();
	bool		waitForTransaction (TransId transId);
	void		dropTable(Table* table);
	bool		hasUncommittedRecords(Table* table);
	void		writeComplete(void);
	void		releaseDependency(void);
	void		rollbackSavepoint (int savepointId);
	void		releaseSavepoint(int savepointId);
	int			createSavepoint();
	void		scavengeRecords(int ageGroup);
	void		add(DeferredIndex* deferredIndex);
	void		initialize(Connection* cnct, TransId seq);

	inline bool isActive()
		{
		return state == Active || state == Limbo;
		}

	Connection		*connection;
	Database		*database;
	volatile int	state;
	TransId			transactionId;
	TransId			oldestActive;
	int				curSavePointId;
	Transaction		*next;			// next in database
	Transaction		*prior;			// next in database
	Transaction		*waitingFor;
	SavePoint		*savePoints;
	SavePoint		*freeSavePoints;
	DeferredIndex	*deferredIndexes;
	int				numberStates;
	int				statesAllocated;
	int				isolationLevel;
	int				xidLength;
	UCHAR			*xid;
	TransState		*states;
	bool			commitTriggers;
	bool			systemTransaction;
	bool			hasUpdates;
	bool			writePending;
	bool			scavenged;
	SyncObject		syncActive;
	
	volatile INTERLOCK_TYPE	dependencies;
	volatile INTERLOCK_TYPE	useCount;

protected:
	RecordVersion	*records;
	RecordVersion	**recordPtr;

	virtual ~Transaction();
public:
	bool isXidEqual(int testLength, const UCHAR* test);
};

#endif // !defined(AFX_TRANSACTION_H__02AD6A4D_A433_11D2_AB5B_0000C01D2301__INCLUDED_)
